/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2013-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::cyclicACMIFvPatchgpuField

Group
    grpCoupledBoundaryConditions

Description
    This boundary condition enforces a cyclic condition between a pair of
    boundaries, whereby communication between the patches is performed using
    an arbitrarily coupled mesh interface (ACMI) interpolation.

Usage
    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            cyclicACMI;
    }
    \endverbatim

See also
    Foam::AMIInterpolation

SourceFiles
    cyclicACMIFvPatchgpuField.C

\*---------------------------------------------------------------------------*/

#ifndef cyclicACMIFvPatchgpuField_H
#define cyclicACMIFvPatchgpuField_H

#include "coupledFvPatchgpuField.H"
#include "cyclicACMILduInterfacegpuField.H"
#include "cyclicACMIgpuFvPatch.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                   Class cyclicACMIFvPatchgpuField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class cyclicACMIFvPatchgpuField
:
    virtual public cyclicACMILduInterfacegpuField,
    public coupledFvPatchgpuField<Type>
{
    // Private data

        //- Local reference cast into the cyclic patch
        const cyclicACMIgpuFvPatch& cyclicACMIPatch_;


    // Private Member Functions

        //- Return neighbour side field given internal fields
        template<class Type2>
        tmp<gpuField<Type2>> neighbourSideField
        (
            const gpuField<Type2>&
        ) const;

         //- Return new matrix coeffs
        tmp<gpuField<scalar>> coeffs
        (
            gpufvMatrix<Type>& matrix,
            const gpuField<scalar>&,
            const label
        ) const;



public:

    //- Runtime type information
    TypeName(cyclicACMIgpuFvPatch::typeName_());


    // Constructors

        //- Construct from patch and internal field
        cyclicACMIFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        cyclicACMIFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given cyclicACMIFvPatchgpuField onto a new patch
        cyclicACMIFvPatchgpuField
        (
            const cyclicACMIFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        cyclicACMIFvPatchgpuField(const cyclicACMIFvPatchgpuField<Type>&);

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new cyclicACMIFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        cyclicACMIFvPatchgpuField
        (
            const cyclicACMIFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new cyclicACMIFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member functions

        // Access

            //- Return local reference cast into the cyclic AMI patch
            const cyclicACMIgpuFvPatch& cyclicACMIPatch() const
            {
                return cyclicACMIPatch_;
            }


        // Evaluation functions

            //- Return true if coupled. Note that the underlying patch
            //  is not coupled() - the points don't align
            virtual bool coupled() const;

            //- Return true if this patch field fixes a value
            //  Needed to check if a level has to be specified while solving
            //  Poisson equations
            virtual bool fixesValue() const
            {
                const scalarField& mask =
                    cyclicACMIPatch_.cyclicACMIPatch().mask();

                if (gMax(mask) > 1e-5)
                {
                    // regions connected
                    return false;
                }
                else
                {
                    // fully separated
                    return nonOverlapPatchField().fixesValue();
                }
            }


            //- Return neighbour coupled internal cell data
            virtual tmp<gpuField<Type>> patchNeighbourField() const;

            //- Return reference to neighbour patchField
            const cyclicACMIFvPatchgpuField<Type>& neighbourPatchField() const;

            //- Return reference to non-overlapping patchField
            const fvPatchgpuField<Type>& nonOverlapPatchField() const;

            //- Update result field based on interface functionality

            virtual void updateInterfaceMatrix
            (
                scalargpuField& result,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const scalargpuField& psiInternal,
                const scalargpuField& coeffs,
                const direction cmpt,
                const Pstream::commsTypes commsType
            ) const;

            //- Update result field based on interface functionality
            virtual void updateInterfaceMatrix
            (
                gpuField<Type>&,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const gpuField<Type>&,
                const scalargpuField&,
                const Pstream::commsTypes commsType
            ) const;

            //- Manipulate matrix
            virtual void manipulateMatrix(gpufvMatrix<Type>& matrix);

            //- Manipulate matrix
            virtual void manipulateMatrix
            (
                gpufvMatrix<Type>& m,
                const label iMatrix,
                const direction cmpt
            );

            //- Update the coefficients associated with the patch field
            virtual void updateCoeffs();


        // Cyclic AMI coupled interface functions

            //- Does the patch field perform the transformation
            virtual bool doTransform() const
            {
                return
                    !(cyclicACMIPatch_.parallel() || pTraits<Type>::rank == 0);
            }

            //- Return face transformation tensor

            virtual const tensorgpuField& gpuForwardT() const
            {
                return cyclicACMIPatch_.gpuForwardT();
            }

            //- Return neighbour-cell transformation tensor

            virtual const tensorgpuField& gpuReverseT() const
            {
                return cyclicACMIPatch_.gpuReverseT();
            }

            //- Return rank of component for transform
            virtual int rank() const
            {
                return pTraits<Type>::rank;
            }


        // I-O

            //- Write
            virtual void write(Ostream& os) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "cyclicACMIFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
