/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fixedValueFvPatchgpuField

Group
    grpGenericBoundaryConditions

Description
    This boundary condition supplies a fixed value constraint, and is the base
    class for a number of other boundary conditions.

Usage
    \table
        Property     | Description             | Required    | Default value
        value        | Patch face values       | yes         |
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            fixedValue;
        value           uniform 0;  // Example for scalar field usage
    }
    \endverbatim

SourceFiles
    fixedValueFvPatchgpuField.C

\*---------------------------------------------------------------------------*/

#ifndef fixedValueFvPatchgpuField_H
#define fixedValueFvPatchgpuField_H

#include "fvPatchgpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                   Class fixedValueFvPatchgpuField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class fixedValueFvPatchgpuField
:
    public fvPatchgpuField<Type>
{

public:

    //- Runtime type information
    TypeName("fixedValue");


    // Constructors

        //- Construct from patch and internal field
        fixedValueFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and value
        fixedValueFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const Type& value
        );

        //- Construct from patch, internal field and dictionary
        fixedValueFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&,
            const bool valueRequired=true
        );

        //- Construct by mapping the given fixedValueFvPatchgpuField<Type>
        //  onto a new patch
        fixedValueFvPatchgpuField
        (
            const fixedValueFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        fixedValueFvPatchgpuField
        (
            const fixedValueFvPatchgpuField<Type>&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new fixedValueFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        fixedValueFvPatchgpuField
        (
            const fixedValueFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new fixedValueFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member functions

        // Attributes

            //- Return true if this patch field fixes a value.
            //  Needed to check if a level has to be specified while solving
            //  Poissons equations.
            virtual bool fixesValue() const
            {
                return true;
            }

            //- Return false: this patch field is not altered by assignment
            virtual bool assignable() const
            {
                return false;
            }


        // Evaluation functions

            //- Return the matrix diagonal coefficients corresponding to the
            //  evaluation of the value of this patchField with given weights
            virtual tmp<gpuField<Type>> valueInternalCoeffs
            (
                const tmp<scalargpuField>&
            ) const;

            //- Return the matrix source coefficients corresponding to the
            //  evaluation of the value of this patchField with given weights
            virtual tmp<gpuField<Type>> valueBoundaryCoeffs
            (
                const tmp<scalargpuField>&
            ) const;

            //- Return the matrix diagonal coefficients corresponding to the
            //  evaluation of the gradient of this patchField
            virtual tmp<gpuField<Type>> gradientInternalCoeffs() const;

            //- Return the matrix source coefficients corresponding to the
            //  evaluation of the gradient of this patchField
            virtual tmp<gpuField<Type>> gradientBoundaryCoeffs() const;


        //- Write
        virtual void write(Ostream&) const;


    // Member operators

        virtual void operator=(const UList<Type>&) {}
        virtual void operator=(const gpuList<Type>&) {}

        virtual void operator=(const fvPatchgpuField<Type>&) {}
        virtual void operator+=(const fvPatchgpuField<Type>&) {}
        virtual void operator-=(const fvPatchgpuField<Type>&) {}
        virtual void operator*=(const fvPatchgpuField<scalar>&) {}
        virtual void operator/=(const fvPatchgpuField<scalar>&) {}

        virtual void operator+=(const gpuField<Type>&) {}
        virtual void operator-=(const gpuField<Type>&) {}

        virtual void operator*=(const gpuField<scalar>&) {}
        virtual void operator/=(const gpuField<scalar>&) {}

        virtual void operator=(const Type&) {}
        virtual void operator+=(const Type&) {}
        virtual void operator-=(const Type&) {}
        virtual void operator*=(const scalar) {}
        virtual void operator/=(const scalar) {}
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "fixedValueFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
