/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpugaussLaplacianScheme.H"
#include "gpusurfaceInterpolate.H"
#include "gpufvcDiv.H"
#include "gpufvcGrad.H"
#include "gpufvMatrices.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Type, class GType>
tmp<gpufvMatrix<Type>>
gpugaussLaplacianScheme<Type, GType>::fvmLaplacianUncorrected
(
    const surfaceScalargpuField& gammaMagSf,
    const surfaceScalargpuField& deltaCoeffs,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            deltaCoeffs.dimensions()*gammaMagSf.dimensions()*vf.dimensions()
        )
    );
    gpufvMatrix<Type>& fvm = tfvm.ref();

    fvm.gpuUpper() = deltaCoeffs.primitiveField()*gammaMagSf.primitiveField();
    fvm.negSumDiag();

    forAll(vf.boundaryField(), patchi)
    {
        const fvPatchgpuField<Type>& pvf = vf.boundaryField()[patchi];
        const fvsPatchScalargpuField& pGamma = gammaMagSf.boundaryField()[patchi];
        const fvsPatchScalargpuField& pDeltaCoeffs =
            deltaCoeffs.boundaryField()[patchi];

        if (pvf.coupled())
        {
            fvm.internalCoeffs()[patchi] =
                pGamma*pvf.gradientInternalCoeffs(pDeltaCoeffs);
            fvm.boundaryCoeffs()[patchi] =
               -pGamma*pvf.gradientBoundaryCoeffs(pDeltaCoeffs);
        }
        else
        {
            fvm.internalCoeffs()[patchi] = pGamma*pvf.gradientInternalCoeffs();
            fvm.boundaryCoeffs()[patchi] = -pGamma*pvf.gradientBoundaryCoeffs();
        }
    }

    return tfvm;
}


template<class Type, class GType>
tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
gpugaussLaplacianScheme<Type, GType>::gammaSnGradCorr
(
    const surfaceVectorgpuField& SfGammaCorr,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const gpufvMesh& mesh = this->mesh();

    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tgammaSnGradCorr
    (
        new GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>
        (
            IOobject
            (
                "gammaSnGradCorr("+vf.name()+')',
                vf.instance(),
                mesh.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE
            ),
            mesh,
            SfGammaCorr.dimensions()
           *vf.dimensions()*mesh.deltaCoeffs().dimensions()
        )
    );
    tgammaSnGradCorr.ref().oriented() = SfGammaCorr.oriented();

    for (direction cmpt = 0; cmpt < pTraits<Type>::nComponents; cmpt++)
    {
        tgammaSnGradCorr.ref().replace
        (
            cmpt,
            fvc::dotInterpolate(SfGammaCorr, fvc::grad(vf.component(cmpt)))
        );
    }

    return tgammaSnGradCorr;
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Type, class GType>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpugaussLaplacianScheme<Type, GType>::fvcLaplacian
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const gpufvMesh& mesh = this->mesh();

    tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> tLaplacian
    (
        fvc::div(this->tsnGradScheme_().snGrad(vf)*mesh.magSf())
    );

    tLaplacian.ref().rename("laplacian(" + vf.name() + ')');

    return tLaplacian;
}


template<class Type, class GType>
tmp<gpufvMatrix<Type>>
gpugaussLaplacianScheme<Type, GType>::fvmLaplacian
(
    const GeometricgpuField<GType, fvsPatchgpuField, gpusurfaceMesh>& gamma,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const gpufvMesh& mesh = this->mesh();

    const surfaceVectorgpuField Sn(mesh.Sf()/mesh.magSf());

    const surfaceVectorgpuField SfGamma(mesh.Sf() & gamma);
    const GeometricgpuField<scalar, fvsPatchgpuField, gpusurfaceMesh> SfGammaSn
    (
        SfGamma & Sn
    );
    const surfaceVectorgpuField SfGammaCorr(SfGamma - SfGammaSn*Sn);

    tmp<gpufvMatrix<Type>> tfvm = fvmLaplacianUncorrected
    (
        SfGammaSn,
        this->tsnGradScheme_().deltaCoeffs(vf),
        vf
    );
    gpufvMatrix<Type>& fvm = tfvm.ref();

    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tfaceFluxCorrection
        = gammaSnGradCorr(SfGammaCorr, vf);

    if (this->tsnGradScheme_().corrected())
    {
        tfaceFluxCorrection.ref() +=
            SfGammaSn*this->tsnGradScheme_().correction(vf);
    }

    fvm.source() -= mesh.V()*fvc::div(tfaceFluxCorrection())().primitiveField();

    if (mesh.hostmesh().fluxRequired(vf.name()))
    {
        fvm.faceFluxCorrectionPtr() = tfaceFluxCorrection.ptr();
    }

    return tfvm;
}


template<class Type, class GType>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpugaussLaplacianScheme<Type, GType>::fvcLaplacian
(
    const GeometricgpuField<GType, fvsPatchgpuField, gpusurfaceMesh>& gamma,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const gpufvMesh& mesh = this->mesh();

    const surfaceVectorgpuField Sn(mesh.Sf()/mesh.magSf());
    const surfaceVectorgpuField SfGamma(mesh.Sf() & gamma);
    const GeometricgpuField<scalar, fvsPatchgpuField, gpusurfaceMesh> SfGammaSn
    (
        SfGamma & Sn
    );
    const surfaceVectorgpuField SfGammaCorr(SfGamma - SfGammaSn*Sn);

    tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> tLaplacian
    (
        fvc::div
        (
            SfGammaSn*this->tsnGradScheme_().snGrad(vf)
          + gammaSnGradCorr(SfGammaCorr, vf)
        )
    );

    tLaplacian.ref().rename
    (
        "laplacian(" + gamma.name() + ',' + vf.name() + ')'
    );

    return tLaplacian;
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// ************************************************************************* //
