/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::gradScheme

Description
    Abstract base class for gradient schemes.

SourceFiles
    gradScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpugradScheme_H
#define gpugradScheme_H

#include "tmp.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class gpufvMesh;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                           Class gradScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpugradScheme
:
    public refCount
{
    // Private Data

        //- Reference to mesh
        const gpufvMesh& mesh_;


    // Private Member Functions

        //- No copy construct
        gpugradScheme(const gpugradScheme&) = delete;

        //- No copy assignment
        void operator=(const gpugradScheme&) = delete;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpugradScheme,
            Istream,
            (const gpufvMesh& mesh, Istream& schemeData),
            (mesh, schemeData)
        );


    // Constructors

        //- Construct from mesh
        gpugradScheme(const gpufvMesh& mesh)
        :
            mesh_(mesh)
        {}


    // Selectors

        //- Return a pointer to a new gradScheme created on freestore
        static tmp<gpugradScheme<Type>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpugradScheme() = default;


    // Member Functions

        //- Return const reference to mesh
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        //- Calculate and return the grad of the given field.
        //  Used by grad either to recalculate the cached gradient when it is
        //  out of date with respect to the field or when it is not cached.
        virtual tmp
        <
            GeometricgpuField
            <typename outerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>
        > calcGrad
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const word& name
        ) const = 0;

        //- Calculate and return the grad of the given field
        //- which may have been cached
        tmp
        <
            GeometricgpuField
            <typename outerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>
        > grad
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const word& name
        ) const;

        //- Calculate and return the grad of the given field
        //- with the default name
        //- which may have been cached
        tmp
        <
            GeometricgpuField
            <typename outerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>
        > grad
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const;

        //- Calculate and return the grad of the given field
        //- with the default name
        //- which may have been cached
        tmp
        <
            GeometricgpuField
            <typename outerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>
        > grad
        (
            const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>&
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeFvgpuGradTypeScheme(SS, Type)                                         \
    defineNamedTemplateTypeNameAndDebug(Foam::fv::SS<Foam::Type>, 0);          \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            gpugradScheme<Type>::addIstreamConstructorToTable<SS<Type>>           \
                add##SS##Type##IstreamConstructorToTable_;                     \
        }                                                                      \
    }


#define makeFvgpuGradScheme(SS)                                                   \
                                                                               \
makeFvgpuGradTypeScheme(SS, scalar)                                               \
makeFvgpuGradTypeScheme(SS, vector)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpugradScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
