/*---------------------------------------------------------------------------* \
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016, 2019 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "nutkWallFunctionFvPatchScalargpuField.H"
#include "turbulenceModel.H"
#include "fvPatchgpuFieldMapper.H"
#include "volgpuFields.H"
#include "wallgpuFvPatch.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * * //
namespace Foam
{
    struct nutkWallFunctionFvPatchScalarFieldcalcNutFunctor
    {
         const label blend_;
         const scalar nutVis_;
         const scalar n_; 
         const scalar yPlusLam_;
         const scalar Cmu25_;
         const scalar kappa_;
         const scalar E_;
	
	nutkWallFunctionFvPatchScalarFieldcalcNutFunctor
	(
            label blend,
            scalar nutVis,
            scalar n,
            scalar yPlusLam,
            scalar Cmu25,
            scalar kappa,
            scalar E
	):
            blend_(blend),
            nutVis_(nutVis),
            n_(n),
            yPlusLam_(yPlusLam),
            Cmu25_(Cmu25),
	    kappa_(kappa),
	    E_(E)
	{}
	
	__host__ __device__
	scalar operator () (const thrust::tuple<scalar,scalar,scalar>& t)
	{
	    scalar y = thrust::get<0>(t);
	    scalar k = thrust::get<1>(t);
	    scalar nuw = thrust::get<2>(t);
			
	    scalar yPlus = Cmu25_*y*sqrt(k)/nuw;
	
            scalar nutLog = nuw*(yPlus*kappa_/log(max(E_*yPlus, 1 + 1e-4)) - 1.0);

            switch(blend_)
            {
                case 0:
                {
                    if (yPlus > yPlusLam_)
                    {
                        return nutLog;
                    }
                    else
                    {
                        return nutVis_;
                    }
                }
                
                case 1:
                {
                    return max(nutVis_, nutLog);
                }

                case 2:
                {
                    return pow
                    (
                        pow(nutVis_, n_) + pow(nutLog, n_),
                        1.0/n_
                    );
                }

                case 3:
                {
                    const scalar Gamma = 0.01*pow4(yPlus)/(1.0 + 5.0*yPlus);
                    const scalar invGamma = 1.0/(Gamma + ROOTVSMALL);

                    return nutVis_*exp(-Gamma) + nutLog*exp(-invGamma);
                }
            }

	}	
    };

    struct nutkWallFunctionFvPatchScalarFieldyPlusFunctor
    {
        const scalar yPlusLam_;
	const scalar Cmu25_;

	nutkWallFunctionFvPatchScalarFieldyPlusFunctor
        (
	    scalar yPlusLam,
	    scalar Cmu25
	):
	    yPlusLam_(yPlusLam),
	    Cmu25_(Cmu25)
	{}
	
	__host__ __device__
	scalar operator () (const thrust::tuple<scalar,scalar,scalar,scalar,scalar>& t)
	{
	    scalar y = thrust::get<0>(t);
	    scalar k = thrust::get<1>(t);
	    scalar nuw = thrust::get<2>(t);
	    scalar nuEff = thrust::get<3>(t);
	    scalar magGradUw = thrust::get<4>(t);
			
	    scalar yPlus = Cmu25_*y*sqrt(k)/nuw;
	
	    if (yPlusLam_ > yPlus)
	    {
		return y*sqrt(nuEff*magGradUw)/nuw;
	    }
	    else
	    {
	        return yPlus;
	    }
	}
    };
}

Foam::tmp<Foam::scalargpuField> Foam::nutkWallFunctionFvPatchScalargpuField::
calcNut() const
{
    const label patchi = patch().index();

    const turbulenceModel& turbModel = db().lookupObject<turbulenceModel>
    (
        IOobject::groupName
        (
            turbulenceModel::propertiesName,
            internalField().group()
        )
    );

    const scalargpuField& y = turbModel.y()[patchi];
    const tmp<volScalargpuField> tk = turbModel.k();
    const volScalargpuField& k = tk();
    const tmp<scalargpuField> tnuw = turbModel.nu(patchi);
    const scalargpuField& nuw = tnuw();

    const scalar Cmu25 = pow025(Cmu_);

    tmp<scalargpuField> tnutw(new scalargpuField(patch().size(), Zero));
    scalargpuField& nutw = tnutw.ref();

    const scalar nutVis = 0.0;
    label blending;

    /*switch (blending_)
    {
        case blendingType::STEPWISE:
        {
            blending = 0;
            break;
        }

        case blendingType::MAX:
        {
            blending = 1;
            break;
        }

        case blendingType::BINOMIAL:
        {
            blending = 2;
            break;
        }

        case blendingType::EXPONENTIAL:
        {
            blending = 3;
            break;
        }
    }*/

    thrust::transform
    (
        thrust::make_zip_iterator(thrust::make_tuple
        (
            y.begin(),
            thrust::make_permutation_iterator
            (
                k.begin(),
                patch().gpuFaceCells().begin()
            ),
            nuw.begin()
        )),
        thrust::make_zip_iterator(thrust::make_tuple
        (
            y.end(),
            thrust::make_permutation_iterator
            (
                k.begin(),
                patch().gpuFaceCells().end()
            ),
            nuw.end()
        )),
        nutw.begin(),
        nutkWallFunctionFvPatchScalarFieldcalcNutFunctor(blending_,nutVis,n_,yPlusLam_,Cmu25,kappa_,E_)
    );

    /*forAll(nutw, facei)
    {
        const label celli = patch().faceCells()[facei];

        const scalar yPlus = Cmu25*y[facei]*sqrt(k[celli])/nuw[facei];

        // Viscous sublayer contribution
        const scalar nutVis = 0.0;

        // Inertial sublayer contribution
        const scalar nutLog =
            nuw[facei]*(yPlus*kappa_/log(max(E_*yPlus, 1 + 1e-4)) - 1.0);

        nutw[facei] = blend(nutVis, nutLog, yPlus);
    }*/

    return tnutw;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::nutkWallFunctionFvPatchScalargpuField::nutkWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    nutWallFunctionFvPatchScalargpuField(p, iF)
{}


Foam::nutkWallFunctionFvPatchScalargpuField::nutkWallFunctionFvPatchScalargpuField
(
    const nutkWallFunctionFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    nutWallFunctionFvPatchScalargpuField(ptf, p, iF, mapper)
{}


Foam::nutkWallFunctionFvPatchScalargpuField::nutkWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    nutWallFunctionFvPatchScalargpuField(p, iF, dict)
{}


Foam::nutkWallFunctionFvPatchScalargpuField::nutkWallFunctionFvPatchScalargpuField
(
    const nutkWallFunctionFvPatchScalargpuField& wfpsf
)
:
    nutWallFunctionFvPatchScalargpuField(wfpsf)
{}


Foam::nutkWallFunctionFvPatchScalargpuField::nutkWallFunctionFvPatchScalargpuField
(
    const nutkWallFunctionFvPatchScalargpuField& wfpsf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    nutWallFunctionFvPatchScalargpuField(wfpsf, iF)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

Foam::tmp<Foam::scalargpuField> Foam::nutkWallFunctionFvPatchScalargpuField::
yPlus() const
{
    const label patchi = patch().index();

    const auto& turbModel = db().lookupObject<turbulenceModel>
    (
        IOobject::groupName
        (
            turbulenceModel::propertiesName,
            internalField().group()
        )
    );

    const scalargpuField& y = turbModel.y()[patchi];

    tmp<volScalargpuField> tk = turbModel.k();
    const volScalargpuField& k = tk();
    tmp<scalargpuField> tkwc = k.boundaryField()[patchi].patchInternalField();
    const scalargpuField& kwc = tkwc();

    tmp<scalargpuField> tnuw = turbModel.nu(patchi);
    const scalargpuField& nuw = tnuw();

    tmp<scalargpuField> tnuEff = turbModel.nuEff(patchi);
    const scalargpuField& nuEff = tnuEff();

    const fvPatchVectorgpuField& Uw = U(turbModel).boundaryField()[patchi];
    const scalargpuField magGradUw(mag(Uw.snGrad()));

    const scalar Cmu25 = pow025(Cmu_);

    auto tyPlus = tmp<scalargpuField>::New(patch().size(), Zero);
    auto& yPlus = tyPlus.ref();

    thrust::transform
    (
        thrust::make_zip_iterator(thrust::make_tuple
        (
            y.begin(),
            kwc.begin(),
            nuw.begin(),
            nuEff.begin(),
            magGradUw.begin()
        )),
        thrust::make_zip_iterator(thrust::make_tuple
        (
            y.end(),
            kwc.end(),
            nuw.end(),
            nuEff.end(),
            magGradUw.end()
        )),
        yPlus.begin(),
        nutkWallFunctionFvPatchScalarFieldyPlusFunctor(yPlusLam_,Cmu25)
    );
/*    forAll(yPlus, facei)
    {
        // inertial sublayer
        yPlus[facei] = Cmu25*y[facei]*sqrt(kwc[facei])/nuw[facei];

        if (yPlusLam_ > yPlus[facei])
        {
            // viscous sublayer
            yPlus[facei] =
                y[facei]*sqrt(nuEff[facei]*magGradUw[facei])/nuw[facei];
        }
    }*/

    return tyPlus;
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        nutkWallFunctionFvPatchScalargpuField
    );
}


// ************************************************************************* //
