/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2015-2022 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include <migraphx/layout_nhwc.hpp>
#include <migraphx/module.hpp>
#include <migraphx/instruction.hpp>
#include <migraphx/iterator_for.hpp>
#include <migraphx/permutation.hpp>
#include <migraphx/functional.hpp>
#include <migraphx/ranges.hpp>
#include <migraphx/make_op.hpp>
#include <migraphx/eliminate_contiguous.hpp>
#include <migraphx/dead_code_elimination.hpp>
#include <migraphx/pass_manager.hpp>

namespace migraphx {
inline namespace MIGRAPHX_INLINE_NS {

// template <class Predicate>
// std::vector<instruction_ref> find_lasts(const module& m, Predicate pred)
// {
//     std::vector<instruction_ref> result;
//     fix([&](auto self, auto ins) {
//         if(pred(ins))
//         {
//             result.push_back(ins);
//             return;
//         }
//         for(auto input : ins->inputs())
//             self(input);
//     })(std::prev(m.end()));
//     return result;
// }

// std::unordered_set<instruction_ref> preserve_output_layout(module& m)
// {
//     std::unordered_set<instruction_ref> result;
//     std::vector<instruction_ref> outputs =
//         find_lasts(m, [](auto ins) { return ins->get_shape().lens().size() == 4; });
//     for(auto output : outputs)
//     {
//         auto permutation = find_permutation(output->get_shape());
//         auto layout      = m.insert_instruction(
//             std::next(output), make_op("layout", {{"permutation", permutation}}), output);
//         result.insert(m.replace_instruction(output, layout));
//     }
//     return result;
// }

void transform_convolutions(module& m, bool skip_elim_contiguous)
{
    for(auto ins : iterator_for(m))
    {
        if(ins->name() != "convolution")
            continue;
        if(ins->get_shape().lens().size() != 4)
            continue;
        auto v = ins->get_operator().to_value();
        if(v.at("group").to<int>() > 1)
            continue;
        auto args = ins->inputs();
        if(skip_elim_contiguous)
        {
            for(auto i = 0; i < args.size(); i++)
            {

                if(args[i]->name() != "layout" and args[i]->get_shape().standard())
                {
                    args[i] = m.insert_instruction(
                        ins, make_op("layout", {{"permutation", {0, 2, 3, 1}}}), args[i]);
                }
            }
        }
        else
            std::transform(args.begin(), args.end(), args.begin(), [&](auto& i) {
                return m.insert_instruction(
                    ins, make_op("layout", {{"permutation", {0, 2, 3, 1}}}), i);
            });
        auto conv = m.insert_instruction(ins, ins->get_operator(), args);
        // m.debug_print(conv);
        // auto c    = conv;
        // auto nchw = m.insert_instruction(ins, make_op("layout", {{"permutation", {0, 1, 2, 3}}}), conv);
        // m.debug_print();
        // if(not skip_elim_contiguous)
        //     c = m.insert_instruction(ins, make_op("contiguous"), conv);
        m.replace_instruction(ins, conv);
    }
}

void insert_contiguous(module& m)
{
    for(auto ins : iterator_for(m))
    {
        if(ins->name() != "reshape" and ins->name() != "pooling")
            continue;
        auto c = m.insert_instruction(ins, make_op("contiguous"), ins->inputs().front());
        auto reshape = m.insert_instruction(ins, ins->get_operator(), c);
        m.replace_instruction(ins, reshape);
    }
    std::cout << "after" << std::endl;
    // m.debug_print();
}

// void remove_layout(module& m, const std::unordered_set<instruction_ref>& output_layouts)
// {
//     for(auto ins : iterator_for(m))
//     {
//         if(ins->name() != "layout")
//             continue;
//         if(ins->get_shape() != ins->inputs().front()->get_shape())
//             continue;
//         if(contains(output_layouts, ins))
//             continue;
//         m.replace_instruction(ins, ins->inputs().front());
//     }
// }

void layout_nhwc::apply(module_pass_manager& mpm) const
{
    // std::unordered_set<instruction_ref> output_layouts =
    // preserve_output_layout(mpm.get_module());
    insert_contiguous(mpm.get_module());
    mpm.run_pass(dead_code_elimination{});
    mpm.get_module().debug_print();
    transform_convolutions(mpm.get_module(), this->skip_elim_contiguous);
    mpm.run_pass(dead_code_elimination{});
    if(not this->skip_elim_contiguous)
        mpm.run_pass(eliminate_contiguous{"contiguous"});
    mpm.run_pass(dead_code_elimination{});
    // remove_layout(mpm.get_module(), output_layouts);
    // mpm.run_pass(dead_code_elimination{});
}

} // namespace MIGRAPHX_INLINE_NS
} // namespace migraphx
