/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2015-2023 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include <cstddef>
#include <limits>
#include <iterator>
#include <unordered_map>
#include <unordered_set>

#include <migraphx/env.hpp>
#include <migraphx/algorithm.hpp>
#include <migraphx/stringutils.hpp>
#include <migraphx/generate_root_modules.hpp>
#include <migraphx/pass_manager.hpp>
#include <migraphx/dead_code_elimination.hpp>
#include <migraphx/instruction.hpp>
#include <migraphx/program.hpp>
#include <migraphx/make_op.hpp>
#include <migraphx/iterator_for.hpp>
#include <migraphx/ranges.hpp>

MIGRAPHX_DECLARE_ENV_VAR(MIGRAPHX_DEBUG_ROOT_GENERATOR)
namespace migraphx {
inline namespace MIGRAPHX_INLINE_NS {

// copied from fuse_pointwise.cpp
static literal get_scalar(instruction_ref ins)
{
    if(ins->name() == "contiguous")
        return get_scalar(ins->inputs().front());
    const auto& s = ins->get_shape();
    if(s.elements() != 1 and not(s.scalar()))
        return {};
    if(not ins->can_eval())
        return {};
    auto e = ins->eval();
    literal r{};
    // needed for bool as visit_at invokes as() which promotes bool to int8
    // Without this we'll break type checks for logical ops that are fused.
    if(e.get_shape().type() == shape::bool_type)
    {
        r = literal{e.at<bool>()};
    }
    else
    {
        e.visit_at([&](auto x) { r = literal{x}; });
    }
    return r;
}

/*
Given target assignments (tass) for the instructions, generate run_on_target modules subgraphs
automatically. Input graph should be uncompiled migraphx program. target assignments (tass) map
should have a map of instruction to target_id. Instructions that are not inside tass map are
considered to be targeted for the "Ref" by default. params, literals and other builtins shouldn't be
part of the tass, only compute and reshaper instructions should be part of tass. Copy, sync and
alloc instructions would be generated by compiler at later stage, so those shouldn't be considered.
(TODO): CustomOps may require special handling.

Step 1:
Identify subgraph boundaries:
(a) Boundaries can happen when any output of a node doesn't have same target
assignment as the node itself.
(b) Boundaries can happen when any output of any node doesn't have all its inputs with same target
assignment as the node itself.

Ref is used for instructions that do not have assignments.
For example graphs like following:
1.  Ref --> Target X --> Ref
2.  Ref --> Target X --> Target Y
3.  Target X --> Target Y --> Target Z , in this case target X and target Z can be same
4.  Target X --> "@return"
5.  Target X --> Ref --> "@return"
6.  When there is a fork in graph :
         Ref
          |
    -------------
    |           |
    |           |
Target X       Ref

7. When there is merge in a graph :

    Target X        Ref
       |             |
       ---------------
             |
         Target X

Each of those identified regions could have futher nested sub modules which needs to be handled
separately.

Step 2:
Collect parameters and return instructions for the subgraphs identified in Step 1.

Step 3:
Create modules using information collected in step 2 and insert run_on_target instructions.
*/

struct auto_gen_root_modules
{

    auto_gen_root_modules(migraphx::program& p, const target_assignments& target_assignments)
        : tass(target_assignments)
    {
        auto* mm = p.get_main_module();
        find_subgraphs(mm, p);
        dead_code_elimination{}.apply(p);
    }

    void update_tid_counter(std::size_t tid)
    {
        if(tid_counter.find(tid) != tid_counter.end())
        {
            tid_counter[tid]++;
        }
        else
        {
            tid_counter[tid] = 0;
        }
    }

    bool is_different_subgraph(migraphx::instruction_ref ins, std::optional<std::size_t> tid)
    {
        if(tass.find(ins) == tass.end())
        {
            return tid.has_value();
        }
        return tass.at(ins) != tid.value_or(std::numeric_limits<std::size_t>::max());
    }

    bool is_merge_node(migraphx::instruction_ref ins, std::optional<std::size_t> tid)
    {
        const auto inputs = ins->inputs();
        if(inputs.size() == 1)
        {
            return false;
        }
        if(std::any_of(inputs.begin(), inputs.end(), [&](auto input_ins) {
               if((skip_ins.find(input_ins) != skip_ins.end()) or
                  (tass.find(input_ins) != tass.end() and
                   tass.at(ins) != tid.value_or(std::numeric_limits<std::size_t>::max())))
               {
                   return true;
               }
               return false;
           }))
            return true;
        return false;
    }

    bool is_fork_node(migraphx::instruction_ref ins, std::optional<std::size_t> tid)
    {
        const auto outputs = ins->outputs();
        if(outputs.size() == 1)
        {
            return false;
        }
        if(std::any_of(outputs.begin(), outputs.end(), [&](auto output_ins) {
               if(tass.find(output_ins) != tass.end() and
                  tass.at(output_ins) != tid.value_or(std::numeric_limits<std::size_t>::max()) and
                  output_ins->name() != "@return")
               {
                   return true;
               }
               return false;
           }))
            return true;
        return false;
    }

    void find_subgraphs(migraphx::module_ref mm, migraphx::program& p)
    {
        // sort the graph in reverse post order DFS order
        mm->sort();
        if(enabled(MIGRAPHX_DEBUG_ROOT_GENERATOR{}))
        {
            std::cout << "sorted module: \n";
            mm->debug_print();
        }
        bool fork_node                         = false;
        std::optional<std::size_t> current_tid = nullopt;
        for(auto ins : iterator_for(*mm))
        {
            if(enabled(MIGRAPHX_DEBUG_ROOT_GENERATOR{}))
            {
                std::cout << "looking at instruction: \n";
                ins->debug_print();
            }
            if(fork_node)
            {
                assert(current_tid.has_value());
                generate_run_on_target_modules(mm, p, ins, current_tid.value());
                if(not same_tid_ins_vec.empty())
                {
                    current_tid = nullopt;
                    same_tid_ins_set.erase(ins);
                    same_tid_ins_vec.pop_back();
                }
                fork_node = false;
            }
            // skip all params, literal and builtins other than return, skip "run_on_target_mod"
            // ins
            if((starts_with(ins->name(), "@") and ins->name() != "@return") or
               skip_ins.count(ins) != 0)
            {
                continue;
            }
            if(not current_tid.has_value())
            {
                if(tass.find(ins) != tass.end())
                {
                    current_tid = std::make_optional<std::size_t>(tass.at(ins));
                    update_tid_counter(current_tid.value());
                    same_tid_ins_vec.push_back(ins);
                    same_tid_ins_set.insert(ins);
                    fork_node = is_fork_node(ins, current_tid);
                }
            }
            else
            {
                if(ins->name() == "@return" or is_different_subgraph(ins, current_tid) or
                   is_merge_node(ins, current_tid))
                {
                    generate_run_on_target_modules(mm, p, ins, current_tid.value());
                }
                else if(tass.at(ins) == current_tid.value())
                {
                    same_tid_ins_vec.push_back(ins);
                    same_tid_ins_set.insert(ins);
                }
                else
                {
                    MIGRAPHX_THROW("GenerateRootModules: this case shouldn't occur");
                }
                fork_node = is_fork_node(ins, current_tid);
            }

            if(not ins->module_inputs().empty())
            {
                std::vector<instruction_ref> same_tid_ins_vec_copy        = {};
                std::unordered_set<instruction_ref> same_tid_ins_set_copy = {};
                std::swap(same_tid_ins_set_copy, same_tid_ins_set);
                std::swap(same_tid_ins_vec_copy, same_tid_ins_vec);
                for(auto* sub_mod : ins->module_inputs())
                {
                    find_subgraphs(sub_mod, p);
                }
                std::swap(same_tid_ins_set_copy, same_tid_ins_set);
                std::swap(same_tid_ins_vec_copy, same_tid_ins_vec);
                mm->replace_instruction(
                    ins, ins->get_operator(), ins->inputs(), ins->module_inputs());
            }
        }
        assert(same_tid_ins_set.empty() and same_tid_ins_vec.empty());
    }

    void generate_run_on_target_modules(migraphx::module_ref mm,
                                        migraphx::program& p,
                                        migraphx::instruction_ref ins,
                                        std::size_t& current_tid)
    {
        assert(same_tid_ins_vec.size() == same_tid_ins_set.size());
        if(same_tid_ins_vec.empty())
        {
            assert(current_tid == std::numeric_limits<std::size_t>::max());
            return;
        }
        // gather all parameters
        std::unordered_set<instruction_ref> params;
        // gather all return values
        std::unordered_set<instruction_ref> return_ins;
        for(auto tins : iterator_for(same_tid_ins_vec))
        {
            auto inputs  = (*tins)->inputs();
            auto outputs = (*tins)->outputs();
            transform_if(
                inputs.cbegin(),
                inputs.cend(),
                std::inserter(params, params.end()),
                [&](auto in_param) {
                    return (params.count(in_param) == 0 and same_tid_ins_set.count(in_param) == 0);
                },
                [&](auto in_param) { return in_param; });
            if(std::any_of(outputs.begin(), outputs.end(), [&](const auto out_ins) {
                   return same_tid_ins_set.count(out_ins) == 0;
               }))
            {
                return_ins.insert(*tins);
            }
        }
        if(enabled(MIGRAPHX_DEBUG_ROOT_GENERATOR{}))
        {
            std::cout << "params ins: \n";
            for(auto tmp : iterator_for(params))
            {
                (*tmp)->debug_print();
            }
            std::cout << "return ins: \n";
            for(auto tmp : iterator_for(return_ins))
            {
                (*tmp)->debug_print();
            }
        }

        auto* tmod = p.create_module("target_mod_" + std::to_string(current_tid) + "_" +
                                     std::to_string(tid_counter[current_tid]));
        update_tid_counter(current_tid);
        std::unordered_map<instruction_ref, instruction_ref> params_map;
        std::size_t param_counter = 0;
        std::vector<instruction_ref> rot_ins_params;
        for(auto pins : iterator_for(params))
        {
            auto scalar = get_scalar(*pins);
            if(scalar.empty())
            {
                params_map[*pins] = tmod->add_parameter("param:" + std::to_string(param_counter++),
                                                        (*pins)->get_shape());
                rot_ins_params.push_back(*pins);
            }
            else
            {
                params_map[*pins] = tmod->add_literal(scalar);
            }
        }

        // TODO: what if params_map is empty ?
        assert(not params_map.empty());
        for(auto tins : iterator_for(same_tid_ins_vec))
        {
            auto inputs = (*tins)->inputs();
            std::vector<instruction_ref> new_inputs;
            std::transform(inputs.begin(),
                           inputs.end(),
                           std::back_inserter(new_inputs),
                           [&](auto input_ins) { return params_map.at(input_ins); });
            auto tmod_tins = tmod->add_instruction(
                (*tins)->get_operator(), new_inputs, (*tins)->module_inputs());
            // add parameter to params map so that it can be looked up by other insturctions
            params_map[*tins] = tmod_tins;
        }

        std::vector<instruction_ref> rins;
        std::unordered_map<instruction_ref, std::size_t> return_ins_idx_map;
        for(auto ritr : iterator_for(return_ins))
        {
            rins.push_back(params_map.at(*ritr));
            return_ins_idx_map[*ritr] = std::distance(ritr, return_ins.begin());
        }
        tmod->add_return(rins);

        if(enabled(MIGRAPHX_DEBUG_ROOT_GENERATOR{}))
        {
            std::cout << "Created target module: " << tmod->name() << "\n";
            tmod->debug_print();
        }

        // add run_on_target ins
        auto rot_ins = mm->insert_instruction(
            ins, make_op("run_on_target", {{"target_id", current_tid}}), rot_ins_params, {tmod});
        skip_ins.insert(rot_ins);

        // fetch return instructions from tuple
        for(auto mm_rins : iterator_for(return_ins))
        {
            auto tuple_elem_ins = mm->insert_instruction(
                ins,
                make_op("get_tuple_elem", {{"index", return_ins_idx_map.at(*mm_rins)}}),
                rot_ins);
            skip_ins.insert(tuple_elem_ins);
            // replace returns from tmod in main module
            mm->replace_instruction(*mm_rins, tuple_elem_ins);
        }
        dead_code_elimination{}.apply(*mm);
        // update current_tid
        same_tid_ins_set.clear();
        same_tid_ins_vec.clear();
        if(tass.find(ins) != tass.end())
        {
            current_tid = tass.at(ins);
            update_tid_counter(current_tid);
            same_tid_ins_set.insert(ins);
            same_tid_ins_vec.push_back(ins);
        }
        else
        {
            current_tid = std::numeric_limits<std::size_t>::max();
        }
        if(enabled(MIGRAPHX_DEBUG_ROOT_GENERATOR{}))
        {
            std::cout << "Main module after creation of target module: " << tmod->name() << "\n";
            mm->debug_print();
        }
    }

    private:
    const target_assignments tass;
    std::unordered_map<std::size_t, std::size_t> tid_counter;
    std::unordered_set<instruction_ref> skip_ins;
    std::vector<instruction_ref> same_tid_ins_vec;
    std::unordered_set<instruction_ref> same_tid_ins_set;
};

void generate_root_modules(migraphx::program& p, const target_assignments& tass)
{
    auto_gen_root_modules(p, tass);
}

} // namespace MIGRAPHX_INLINE_NS
} // namespace migraphx
