# Copyright 2020 The HuggingFace Datasets Authors and the current dataset script contributor.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""GPT-3 Arithmetic Test Dataset."""


import json

import datasets


_CITATION = """\
@inproceedings{NEURIPS2020_1457c0d6,
    author = {Brown, Tom and Mann, Benjamin and Ryder, Nick and Subbiah, Melanie and Kaplan, Jared D and Dhariwal, Prafulla and Neelakantan, Arvind and Shyam, Pranav and Sastry, Girish and Askell, Amanda and Agarwal, Sandhini and Herbert-Voss, Ariel and Krueger, Gretchen and Henighan, Tom and Child, Rewon and Ramesh, Aditya and Ziegler, Daniel and Wu, Jeffrey and Winter, Clemens and Hesse, Chris and Chen, Mark and Sigler, Eric and Litwin, Mateusz and Gray, Scott and Chess, Benjamin and Clark, Jack and Berner, Christopher and McCandlish, Sam and Radford, Alec and Sutskever, Ilya and Amodei, Dario},
    booktitle = {Advances in Neural Information Processing Systems},
    editor = {H. Larochelle and M. Ranzato and R. Hadsell and M. F. Balcan and H. Lin},
    pages = {1877--1901},
    publisher = {Curran Associates, Inc.},
    title = {Language Models are Few-Shot Learners},
    url = {https://proceedings.neurips.cc/paper/2020/file/1457c0d6bfcb4967418bfb8ac142f64a-Paper.pdf},
    volume = {33},
    year = {2020}
}
"""

_DESCRIPTION = """\
A small battery of 10 tests that involve asking language models a simple arithmetic
problem in natural language.
"""

_HOMEPAGE = "https://github.com/openai/gpt-3/tree/master/data"

# TODO: Add the licence for the dataset here if you can find it
_LICENSE = ""


class ArithmeticConfig(datasets.BuilderConfig):
    """BuilderConfig for GPT3 Arithmetic Test Dataset."""

    def __init__(self, url, features, **kwargs):
        """BuilderConfig for GPT3 Arithmetic dataset.

        Args:
        url: *string*, the url to the specific subset of the GPT3 Arithmetic dataset.
        features: *list[string]*, list of the features that will appear in the
            feature dict.
        """
        # Version history:
        super().__init__(version=datasets.Version("0.0.1"), **kwargs)
        self.url = url
        self.features = features


class Arithmetic(datasets.GeneratorBasedBuilder):
    """A small battery of 10 tests involving simple arithmetic problems."""

    BUILDER_CONFIGS = [
        ArithmeticConfig(
            name="arithmetic_2da",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/two_digit_addition.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="2-digit addition",
        ),
        ArithmeticConfig(
            name="arithmetic_2ds",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/two_digit_subtraction.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="2-digit subtraction",
        ),
        ArithmeticConfig(
            name="arithmetic_3da",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/three_digit_addition.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="3-digit addition",
        ),
        ArithmeticConfig(
            name="arithmetic_3ds",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/three_digit_subtraction.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="3-digit subtraction",
        ),
        ArithmeticConfig(
            name="arithmetic_4da",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/four_digit_addition.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="4-digit addition",
        ),
        ArithmeticConfig(
            name="arithmetic_4ds",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/four_digit_subtraction.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="4-digit subtraction",
        ),
        ArithmeticConfig(
            name="arithmetic_5da",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/five_digit_addition.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="5-digit addition",
        ),
        ArithmeticConfig(
            name="arithmetic_5ds",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/five_digit_subtraction.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="5-digit subtraction",
        ),
        ArithmeticConfig(
            name="arithmetic_2dm",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/two_digit_multiplication.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="2-digit multiplication",
        ),
        ArithmeticConfig(
            name="arithmetic_1dc",
            url="https://raw.githubusercontent.com/openai/gpt-3/master/data/single_digit_three_ops.jsonl",
            features=datasets.Features({"context": datasets.Value("string"), "completion": datasets.Value("string")}),
            description="Single digit 3 operations",
        ),
    ]

    def _info(self):
        return datasets.DatasetInfo(
            description=f"{_DESCRIPTION}\n{self.config.description}",
            features=self.config.features,
            homepage=_HOMEPAGE,
            license=_LICENSE,
            citation=_CITATION,
        )

    def _split_generators(self, dl_manager):
        urls = self.config.url
        data_dir = dl_manager.download_and_extract(urls)
        return [
            datasets.SplitGenerator(
                name=datasets.Split.VALIDATION,
                # These kwargs will be passed to _generate_examples
                gen_kwargs={
                    "filepath": data_dir,
                    "split": datasets.Split.VALIDATION,
                },
            ),
        ]

    # method parameters are unpacked from `gen_kwargs` as given in `_split_generators`
    def _generate_examples(self, filepath, split):
        with open(filepath, encoding="utf-8") as f:
            for key, row in enumerate(f):
                data = json.loads(row)
                context = data['context'].strip() \
                    .replace('\n\n', '\n') \
                    .replace('Q:', 'Question:') \
                    .replace('A:', 'Answer:')
                completion = data['completion']
                yield key, {'context': context, 'completion': completion}
