/*
 *  Copyright 2008-2013 NVIDIA Corporation
 *  Modifications Copyright© 2019 Advanced Micro Devices, Inc. All rights reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/*! \file thrust/system/hip/error.h
 *  \brief HIP-specific error reporting
 */

#pragma once

#include <thrust/detail/config.h>
#include <thrust/detail/type_traits.h>
#include <thrust/system/error_code.h>
#include <thrust/system/cuda/detail/guarded_driver_types.h>

THRUST_NAMESPACE_BEGIN

namespace system
{

namespace cuda_cub
{

/*! \addtogroup system
 *  \{
 */

// To construct an error_code after a HIP Runtime error:
//
//   error_code(::cudaGetLastError(), cuda_category())

// XXX N3000 prefers enum class errc { ... }
namespace errc
{

/*! \p errc_t enumerates the kinds of HIP Runtime errors.
 */
//TODO：cudaErrorTbd,cudaErrorLaunchTimeOut增加到cudamocker的runtime中，目前cudaErrorLaunchTimeOut改为cudaErrorLaunchFailure，cudaErrorTbd改为cudaErrorUnknown
enum errc_t
{
    // from hip/include/driver_types.h
    // mirror their order
    success                            = cudaSuccess,
    missing_configuration              = cudaErrorMissingConfiguration,
    memory_allocation                  = cudaErrorMemoryAllocation,
    initialization_error               = cudaErrorInitializationError,
    launch_failure                     = cudaErrorLaunchFailure,
    prior_launch_failure               = cudaErrorPriorLaunchFailure,
    launch_timeout                     = cudaErrorLaunchFailure,
    launch_out_of_resources            = cudaErrorLaunchOutOfResources,
    invalid_device_function            = cudaErrorInvalidDeviceFunction,
    invalid_configuration              = cudaErrorInvalidConfiguration,
    invalid_device                     = cudaErrorInvalidDevice,
    invalid_value                      = cudaErrorInvalidValue,
    invalid_pitch_value                = cudaErrorUnknown,
    invalid_symbol                     = cudaErrorInvalidSymbol,
    map_buffer_object_failed           = cudaErrorMapBufferObjectFailed,
    unmap_buffer_object_failed         = cudaErrorUnknown,
    invalid_host_pointer               = cudaErrorUnknown,
    invalid_device_pointer             = cudaErrorInvalidDevicePointer,
    invalid_texture                    = cudaErrorUnknown,
    invalid_texture_binding            = cudaErrorUnknown,
    invalid_channel_descriptor         = cudaErrorUnknown,
    invalid_memcpy_direction           = cudaErrorInvalidMemcpyDirection,
    address_of_constant_error          = cudaErrorUnknown,
    texture_fetch_failed               = cudaErrorUnknown,
    texture_not_bound                  = cudaErrorUnknown,
    synchronization_error              = cudaErrorUnknown,
    invalid_filter_setting             = cudaErrorUnknown,
    invalid_norm_setting               = cudaErrorUnknown,
    mixed_device_execution             = cudaErrorUnknown,
    cuda_runtime_unloading             = cudaErrorUnknown,
    unknown                            = cudaErrorUnknown,
    not_yet_implemented                = cudaErrorUnknown,
    memory_value_too_large             = cudaErrorUnknown,
    invalid_resource_handle            = cudaErrorInvalidResourceHandle,
    not_ready                          = cudaErrorNotReady,
    insufficient_driver                = cudaErrorUnknown,
    set_on_active_process_error        = cudaErrorSetOnActiveProcess,
    no_device                          = cudaErrorNoDevice,
    ecc_uncorrectable                  = cudaErrorUnknown,
    shared_object_symbol_not_found     = cudaErrorSharedObjectSymbolNotFound,
    shared_object_init_failed          = cudaErrorSharedObjectInitFailed,
    unsupported_limit                  = cudaErrorUnsupportedLimit,
    duplicate_variable_name            = cudaErrorUnknown,
    duplicate_texture_name             = cudaErrorUnknown,
    duplicate_surface_name             = cudaErrorUnknown,
    devices_unavailable                = cudaErrorUnknown,
    invalid_kernel_image               = cudaErrorUnknown,
    no_kernel_image_for_device         = cudaErrorUnknown,
    incompatible_driver_context        = cudaErrorUnknown,
    peer_access_already_enabled        = cudaErrorPeerAccessAlreadyEnabled,
    peer_access_not_enabled            = cudaErrorPeerAccessNotEnabled,
    device_already_in_use              = cudaErrorUnknown,
    profiler_disabled                  = cudaErrorUnknown,
    assert_triggered                   = cudaErrorUnknown,
    too_many_peers                     = cudaErrorUnknown,
    host_memory_already_registered     = cudaErrorHostMemoryAlreadyRegistered,
    host_memory_not_registered         = cudaErrorHostMemoryNotRegistered,
    operating_system_error             = cudaErrorUnknown,
    peer_access_unsupported            = cudaErrorUnknown,
    launch_max_depth_exceeded          = cudaErrorUnknown,
    launch_file_scoped_texture_used    = cudaErrorUnknown,
    launch_file_scoped_surface_used    = cudaErrorUnknown,
    sync_depth_exceeded                = cudaErrorUnknown,
    attempted_operation_not_permitted  = cudaErrorUnknown,
    attempted_operation_not_supported  = cudaErrorUnknown,
    startup_failure                    = cudaErrorUnknown
}; // end errc_t


} // end namespace errc

} // end namespace cuda_cub

/*! \return A reference to an object of a type derived from class \p thrust::error_category.
 *  \note The object's \p equivalent virtual functions shall behave as specified
 *        for the class \p thrust::error_category. The object's \p name virtual function shall
 *        return a pointer to the string <tt>"hip"</tt>. The object's
 *        \p default_error_condition virtual function shall behave as follows:
 *
 *        If the argument <tt>ev</tt> corresponds to a HIP error value, the function
 *        shall return <tt>error_condition(ev,cuda_category())</tt>.
 *        Otherwise, the function shall return <tt>system_category.default_error_condition(ev)</tt>.
 */
inline const error_category &cuda_category(void);


// XXX N3000 prefers is_error_code_enum<hip::errc>

/*! Specialization of \p is_error_code_enum for \p hip::errc::errc_t
 */
template<> struct is_error_code_enum<cuda_cub::errc::errc_t> : thrust::detail::true_type {};


/*! \return <tt>error_code(static_cast<int>(e), hip::error_category())</tt>
 */
inline error_code make_error_code(cuda_cub::errc::errc_t e);


/*! \return <tt>error_condition(static_cast<int>(e), hip::error_category())</tt>.
 */
inline error_condition make_error_condition(cuda_cub::errc::errc_t e);

/*! \} // end system
 */


} // end system

namespace system {
namespace cuda {
namespace errc {
using system::cuda_cub::errc::errc_t;
} // namespace errc
} // namespace cuda
} // namespace system

namespace cuda
{
// XXX replace with using system::hip_errc upon c++0x
namespace errc = system::cuda::errc;
} // end hip

using system::cuda_category;

THRUST_NAMESPACE_END

#include <thrust/system/cuda/detail/error.inl>
