


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ==============================================================================
! hipfort: FORTRAN Interfaces for GPU kernels
! ==============================================================================
! Copyright (c) 2020-2022 Advanced Micro Devices, Inc. All rights reserved.
! [MITx11 License]
! 
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
! IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
! AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
! OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
! THE SOFTWARE.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
          
           
module cudafor_cusparse
  use cudafor_cusparse_enums
  implicit none

 
  !>  \ingroup aux_module
  !>   \brief Create a hipsparse handle
  !> 
  !>   \details
  !>   \p hipsparseCreate creates the hipSPARSE library context. It must be
  !>   initialized before any other hipSPARSE API function is invoked and must be passed to
  !>   all subsequent library function calls. The handle should be destroyed at the end
  !>   using hipsparseDestroy().
  interface cusparseCreate
    function cusparseCreate_(handle) bind(c, name="cusparseCreate")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreate_
      type(c_ptr) :: handle
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a hipsparse handle
  !> 
  !>   \details
  !>   \p hipsparseDestroy destroys the hipSPARSE library context and releases all
  !>   resources used by the hipSPARSE library.
  interface cusparseDestroy
    function cusparseDestroy_(handle) bind(c, name="cusparseDestroy")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDestroy_
      type(c_ptr),value :: handle
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Get hipSPARSE version
  !> 
  !>   \details
  !>   \p hipsparseGetVersion gets the hipSPARSE library version number.
  !>   - patch = version % 100
  !>   - minor = version 100 % 1000
  !>   - major = version 100000
  interface cusparseGetVersion
    function cusparseGetVersion_(handle,version) bind(c, &
&name="cusparseGetVersion")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseGetVersion_
      type(c_ptr),value :: handle
      type(c_ptr),value :: version
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Get hipSPARSE git revision
  !> 
  !>   \details
  !>   \p hipsparseGetGitRevision gets the hipSPARSE library git commit revision (SHA-1).
  interface cusparseGetGitRevision
    function cusparseGetGitRevision_(handle,rev) bind(c, &
&name="cusparseGetGitRevision")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseGetGitRevision_
      type(c_ptr),value :: handle
      type(c_ptr),value :: rev
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Specify user defined HIP stream
  !> 
  !>   \details
  !>   \p hipsparseSetStream specifies the stream to be used by the hipSPARSE library
  !>   context and all subsequent function calls.
  interface cusparseSetStream
    function cusparseSetStream_(handle,streamId) bind(c, &
&name="cusparseSetStream")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSetStream_
      type(c_ptr),value :: handle
      type(c_ptr),value :: streamId
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Get current stream from library context
  !> 
  !>   \details
  !>   \p hipsparseGetStream gets the hipSPARSE library context stream which is currently
  !>   used for all subsequent function calls.
  interface cusparseGetStream
    function cusparseGetStream_(handle,streamId) bind(c, &
&name="cusparseGetStream")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseGetStream_
      type(c_ptr),value :: handle
      type(c_ptr) :: streamId
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Specify pointer mode
  !> 
  !>   \details
  !>   \p hipsparseSetPointerMode specifies the pointer mode to be used by the hipSPARSE
  !>   library context and all subsequent function calls. By default, all values are passed
  !>   by reference on the host. Valid pointer modes are \ref HIPSPARSE_POINTER_MODE_HOST
  !>   or \p HIPSPARSE_POINTER_MODE_DEVICE.
  interface cusparseSetPointerMode
    function cusparseSetPointerMode_(handle,mode) bind(c, &
&name="cusparseSetPointerMode")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSetPointerMode_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_POINTER_MODE_HOST)),value :: mode
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Get current pointer mode from library context
  !> 
  !>   \details
  !>   \p hipsparseGetPointerMode gets the hipSPARSE library context pointer mode which
  !>   is currently used for all subsequent function calls.
  interface cusparseGetPointerMode
    function cusparseGetPointerMode_(handle,mode) bind(c, &
&name="cusparseGetPointerMode")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseGetPointerMode_
      type(c_ptr),value :: handle
      type(c_ptr),value :: mode
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a matrix descriptor
  !>   \details
  !>   \p hipsparseCreateMatDescr creates a matrix descriptor. It initializes
  !>   \ref hipsparseMatrixType_t to \ref HIPSPARSE_MATRIX_TYPE_GENERAL and
  !>   \ref hipsparseIndexBase_t to \ref HIPSPARSE_INDEX_BASE_ZERO. It should be destroyed
  !>   at the end using hipsparseDestroyMatDescr().
  interface cusparseCreateMatDescr
    function cusparseCreateMatDescr_(descrA) bind(c, &
&name="cusparseCreateMatDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateMatDescr_
      type(c_ptr) :: descrA
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a matrix descriptor
  !> 
  !>   \details
  !>   \p hipsparseDestroyMatDescr destroys a matrix descriptor and releases all
  !>   resources used by the descriptor.
  interface cusparseDestroyMatDescr
    function cusparseDestroyMatDescr_(descrA) bind(c, &
&name="cusparseDestroyMatDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDestroyMatDescr_
      type(c_ptr),value :: descrA
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Copy a matrix descriptor
  !>   \details
  !>   \p hipsparseCopyMatDescr copies a matrix descriptor. Both, source and destination
  !>   matrix descriptors must be initialized prior to calling \p hipsparseCopyMatDescr.
  interface cusparseCopyMatDescr
    function cusparseCopyMatDescr_(dest,src) bind(c, &
&name="cusparseCopyMatDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCopyMatDescr_
      type(c_ptr),value :: dest
      type(c_ptr),value :: src
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Specify the matrix type of a matrix descriptor
  !> 
  !>   \details
  !>   \p hipsparseSetMatType sets the matrix type of a matrix descriptor. Valid
  !>   matrix types are \ref HIPSPARSE_MATRIX_TYPE_GENERAL,
  !>   \ref HIPSPARSE_MATRIX_TYPE_SYMMETRIC, \ref HIPSPARSE_MATRIX_TYPE_HERMITIAN or
  !>   \ref HIPSPARSE_MATRIX_TYPE_TRIANGULAR.
  interface cusparseSetMatType
    function cusparseSetMatType_(descrA,myType) bind(c, &
&name="cusparseSetMatType")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSetMatType_
      type(c_ptr),value :: descrA
      integer(kind(CUSPARSE_MATRIX_TYPE_GENERAL)),value :: myType
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Specify the matrix fill mode of a matrix descriptor
  !> 
  !>   \details
  !>   \p hipsparseSetMatFillMode sets the matrix fill mode of a matrix descriptor.
  !>   Valid fill modes are \ref HIPSPARSE_FILL_MODE_LOWER or
  !>   \ref HIPSPARSE_FILL_MODE_UPPER.
  interface cusparseSetMatFillMode
    function cusparseSetMatFillMode_(descrA,fillMode) bind(c, &
&name="cusparseSetMatFillMode")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSetMatFillMode_
      type(c_ptr),value :: descrA
      integer(kind(CUSPARSE_FILL_MODE_LOWER)),value :: fillMode
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Specify the matrix diagonal type of a matrix descriptor
  !> 
  !>   \details
  !>   \p hipsparseSetMatDiagType sets the matrix diagonal type of a matrix
  !>   descriptor. Valid diagonal types are \ref HIPSPARSE_DIAG_TYPE_UNIT or
  !>   \ref HIPSPARSE_DIAG_TYPE_NON_UNIT.
  interface cusparseSetMatDiagType
    function cusparseSetMatDiagType_(descrA,diagType) bind(c, &
&name="cusparseSetMatDiagType")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSetMatDiagType_
      type(c_ptr),value :: descrA
      integer(kind(CUSPARSE_DIAG_TYPE_NON_UNIT)),value :: diagType
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Specify the index base of a matrix descriptor
  !> 
  !>   \details
  !>   \p hipsparseSetMatIndexBase sets the index base of a matrix descriptor. Valid
  !>   options are \ref HIPSPARSE_INDEX_BASE_ZERO or \ref HIPSPARSE_INDEX_BASE_ONE.
  interface cusparseSetMatIndexBase
    function cusparseSetMatIndexBase_(descrA,base) bind(c, &
&name="cusparseSetMatIndexBase")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSetMatIndexBase_
      type(c_ptr),value :: descrA
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: base
    end function

  end interface
  
  interface cusparseCreateHybMat
    function cusparseCreateHybMat_(hybA) bind(c, &
&name="cusparseCreateHybMat")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateHybMat_
      type(c_ptr) :: hybA
    end function

  end interface
  
  interface cusparseDestroyHybMat
    function cusparseDestroyHybMat_(hybA) bind(c, &
&name="cusparseDestroyHybMat")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDestroyHybMat_
      type(c_ptr),value :: hybA
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a bsrsv2 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateBsrsv2Info creates a structure that holds the bsrsv2 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyBsrsv2Info().
  interface cusparseCreateBsrsv2Info
    function cusparseCreateBsrsv2Info_(myInfo) bind(c, &
&name="cusparseCreateBsrsv2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateBsrsv2Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a bsrsv2 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyBsrsv2Info destroys a bsrsv2 info structure.
  interface cusparseDestroyBsrsv2Info
    function cusparseDestroyBsrsv2Info_(myInfo) bind(c, &
&name="cusparseDestroyBsrsv2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyBsrsv2Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a bsrsm2 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateBsrsm2Info creates a structure that holds the bsrsm2 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyBsrsm2Info().
  interface cusparseCreateBsrsm2Info
    function cusparseCreateBsrsm2Info_(myInfo) bind(c, &
&name="cusparseCreateBsrsm2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateBsrsm2Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a bsrsm2 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyBsrsm2Info destroys a bsrsm2 info structure.
  interface cusparseDestroyBsrsm2Info
    function cusparseDestroyBsrsm2Info_(myInfo) bind(c, &
&name="cusparseDestroyBsrsm2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyBsrsm2Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a bsrilu02 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateBsrilu02Info creates a structure that holds the bsrilu02 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyBsrilu02Info().
  interface cusparseCreateBsrilu02Info
    function cusparseCreateBsrilu02Info_(myInfo) bind(c, &
&name="cusparseCreateBsrilu02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateBsrilu02Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a bsrilu02 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyBsrilu02Info destroys a bsrilu02 info structure.
  interface cusparseDestroyBsrilu02Info
    function cusparseDestroyBsrilu02Info_(myInfo) bind(c, &
&name="cusparseDestroyBsrilu02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyBsrilu02Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a bsric02 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateBsric02Info creates a structure that holds the bsric02 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyBsric02Info().
  interface cusparseCreateBsric02Info
    function cusparseCreateBsric02Info_(myInfo) bind(c, &
&name="cusparseCreateBsric02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateBsric02Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a bsric02 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyBsric02Info destroys a bsric02 info structure.
  interface cusparseDestroyBsric02Info
    function cusparseDestroyBsric02Info_(myInfo) bind(c, &
&name="cusparseDestroyBsric02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyBsric02Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a csrsv2 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateCsrsv2Info creates a structure that holds the csrsv2 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyCsrsv2Info().
  interface cusparseCreateCsrsv2Info
    function cusparseCreateCsrsv2Info_(myInfo) bind(c, &
&name="cusparseCreateCsrsv2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateCsrsv2Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a csrsv2 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyCsrsv2Info destroys a csrsv2 info structure.
  interface cusparseDestroyCsrsv2Info
    function cusparseDestroyCsrsv2Info_(myInfo) bind(c, &
&name="cusparseDestroyCsrsv2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyCsrsv2Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a csrsm2 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateCsrsm2Info creates a structure that holds the csrsm2 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyCsrsm2Info().
  interface cusparseCreateCsrsm2Info
    function cusparseCreateCsrsm2Info_(myInfo) bind(c, &
&name="cusparseCreateCsrsm2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateCsrsm2Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a csrsm2 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyCsrsm2Info destroys a csrsm2 info structure.
  interface cusparseDestroyCsrsm2Info
    function cusparseDestroyCsrsm2Info_(myInfo) bind(c, &
&name="cusparseDestroyCsrsm2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyCsrsm2Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a csrilu02 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateCsrilu02Info creates a structure that holds the csrilu02 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyCsrilu02Info().
  interface cusparseCreateCsrilu02Info
    function cusparseCreateCsrilu02Info_(myInfo) bind(c, &
&name="cusparseCreateCsrilu02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateCsrilu02Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a csrilu02 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyCsrilu02Info destroys a csrilu02 info structure.
  interface cusparseDestroyCsrilu02Info
    function cusparseDestroyCsrilu02Info_(myInfo) bind(c, &
&name="cusparseDestroyCsrilu02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyCsrilu02Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a csric02 info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateCsric02Info creates a structure that holds the csric02 info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyCsric02Info().
  interface cusparseCreateCsric02Info
    function cusparseCreateCsric02Info_(myInfo) bind(c, &
&name="cusparseCreateCsric02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateCsric02Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a csric02 info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyCsric02Info destroys a csric02 info structure.
  interface cusparseDestroyCsric02Info
    function cusparseDestroyCsric02Info_(myInfo) bind(c, &
&name="cusparseDestroyCsric02Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyCsric02Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a csru2csr info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateCsru2csrInfo creates a structure that holds the csru2csr info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyCsru2csrInfo().
  interface cusparseCreateCsru2csrInfo
    function cusparseCreateCsru2csrInfo_(myInfo) bind(c, &
&name="cusparseCreateCsru2csrInfo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateCsru2csrInfo_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a csru2csr info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyCsru2csrInfo destroys a csru2csr info structure.
  interface cusparseDestroyCsru2csrInfo
    function cusparseDestroyCsru2csrInfo_(myInfo) bind(c, &
&name="cusparseDestroyCsru2csrInfo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyCsru2csrInfo_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a color info structure
  !> 
  !>   \details
  !>   \p hipsparseCreateColorInfo creates a structure that holds the color info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyColorInfo().
  interface cusparseCreateColorInfo
    function cusparseCreateColorInfo_(myInfo) bind(c, &
&name="cusparseCreateColorInfo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateColorInfo_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a color info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyColorInfo destroys a color info structure.
  interface cusparseDestroyColorInfo
    function cusparseDestroyColorInfo_(myInfo) bind(c, &
&name="cusparseDestroyColorInfo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyColorInfo_
      type(c_ptr),value :: myInfo
    end function

  end interface
  
  interface cusparseCreateCsrgemm2Info
    function cusparseCreateCsrgemm2Info_(myInfo) bind(c, &
&name="cusparseCreateCsrgemm2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateCsrgemm2Info_
      type(c_ptr) :: myInfo
    end function

  end interface
  
  interface cusparseDestroyCsrgemm2Info
    function cusparseDestroyCsrgemm2Info_(myInfo) bind(c, &
&name="cusparseDestroyCsrgemm2Info")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyCsrgemm2Info_
      type(c_ptr),value :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Create a prune info structure
  !> 
  !>   \details
  !>   \p hipsparseCreatePruneInfo creates a structure that holds the prune info data
  !>   that is gathered during the analysis routines available. It should be destroyed
  !>   at the end using hipsparseDestroyPruneInfo().
  interface cusparseCreatePruneInfo
    function cusparseCreatePruneInfo_(myInfo) bind(c, &
&name="cusparseCreatePruneInfo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreatePruneInfo_
      type(c_ptr) :: myInfo
    end function

  end interface
  !>  \ingroup aux_module
  !>   \brief Destroy a prune info structure
  !> 
  !>   \details
  !>   \p hipsparseDestroyPruneInfo destroys a prune info structure.
  interface cusparseDestroyPruneInfo
    function cusparseDestroyPruneInfo_(myInfo) bind(c, &
&name="cusparseDestroyPruneInfo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDestroyPruneInfo_
      type(c_ptr),value :: myInfo
    end function

  end interface
  
  interface cusparseSaxpyi
    function cusparseSaxpyi_(handle,nnz,alpha,xVal,xInd,y,idxBase) &
&bind(c, name="cusparseSaxpyi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSaxpyi_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseSaxpyi_rank_0,&
      cusparseSaxpyi_rank_1
  end interface
  
  interface cusparseDaxpyi
    function cusparseDaxpyi_(handle,nnz,alpha,xVal,xInd,y,idxBase) &
&bind(c, name="cusparseDaxpyi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDaxpyi_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseDaxpyi_rank_0,&
      cusparseDaxpyi_rank_1
  end interface
  
  interface cusparseCaxpyi
    function cusparseCaxpyi_(handle,nnz,alpha,xVal,xInd,y,idxBase) &
&bind(c, name="cusparseCaxpyi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCaxpyi_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseCaxpyi_rank_0,&
      cusparseCaxpyi_rank_1
  end interface
  
  interface cusparseZaxpyi
    function cusparseZaxpyi_(handle,nnz,alpha,xVal,xInd,y,idxBase) &
&bind(c, name="cusparseZaxpyi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZaxpyi_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseZaxpyi_rank_0,&
      cusparseZaxpyi_rank_1
  end interface
  
  interface cusparseSdoti
    function cusparseSdoti_(handle,nnz,xVal,xInd,y,myResult,idxBase) &
&bind(c, name="cusparseSdoti")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSdoti_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      type(c_ptr),value :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseSdoti_rank_0,&
      cusparseSdoti_rank_1
  end interface
  
  interface cusparseDdoti
    function cusparseDdoti_(handle,nnz,xVal,xInd,y,myResult,idxBase) &
&bind(c, name="cusparseDdoti")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDdoti_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      type(c_ptr),value :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseDdoti_rank_0,&
      cusparseDdoti_rank_1
  end interface
  
  interface cusparseCdoti
    function cusparseCdoti_(handle,nnz,xVal,xInd,y,myResult,idxBase) &
&bind(c, name="cusparseCdoti")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdoti_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      type(c_ptr),value :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseCdoti_rank_0,&
      cusparseCdoti_rank_1
  end interface
  
  interface cusparseZdoti
    function cusparseZdoti_(handle,nnz,xVal,xInd,y,myResult,idxBase) &
&bind(c, name="cusparseZdoti")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdoti_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      type(c_ptr),value :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseZdoti_rank_0,&
      cusparseZdoti_rank_1
  end interface
  
  interface cusparseCdotci
    function cusparseCdotci_(handle,nnz,xVal,xInd,y,myResult,idxBase) &
&bind(c, name="cusparseCdotci")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdotci_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      type(c_ptr),value :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseCdotci_rank_0,&
      cusparseCdotci_rank_1
  end interface
  
  interface cusparseZdotci
    function cusparseZdotci_(handle,nnz,xVal,xInd,y,myResult,idxBase) &
&bind(c, name="cusparseZdotci")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdotci_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      type(c_ptr),value :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseZdotci_rank_0,&
      cusparseZdotci_rank_1
  end interface
  
  interface cusparseSgthr
    function cusparseSgthr_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseSgthr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgthr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseSgthr_rank_0,&
      cusparseSgthr_rank_1
  end interface
  
  interface cusparseDgthr
    function cusparseDgthr_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseDgthr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgthr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseDgthr_rank_0,&
      cusparseDgthr_rank_1
  end interface
  
  interface cusparseCgthr
    function cusparseCgthr_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseCgthr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgthr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseCgthr_rank_0,&
      cusparseCgthr_rank_1
  end interface
  
  interface cusparseZgthr
    function cusparseZgthr_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseZgthr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgthr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseZgthr_rank_0,&
      cusparseZgthr_rank_1
  end interface
  
  interface cusparseSgthrz
    function cusparseSgthrz_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseSgthrz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgthrz_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseSgthrz_rank_0,&
      cusparseSgthrz_rank_1
  end interface
  
  interface cusparseDgthrz
    function cusparseDgthrz_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseDgthrz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgthrz_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseDgthrz_rank_0,&
      cusparseDgthrz_rank_1
  end interface
  
  interface cusparseCgthrz
    function cusparseCgthrz_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseCgthrz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgthrz_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseCgthrz_rank_0,&
      cusparseCgthrz_rank_1
  end interface
  
  interface cusparseZgthrz
    function cusparseZgthrz_(handle,nnz,y,xVal,xInd,idxBase) bind(c, &
&name="cusparseZgthrz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgthrz_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: y
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseZgthrz_rank_0,&
      cusparseZgthrz_rank_1
  end interface
  
  interface cusparseSroti
    function cusparseSroti_(handle,nnz,xVal,xInd,y,c,s,idxBase) bind(c,&
& name="cusparseSroti")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSroti_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      real(c_float) :: c
      real(c_float) :: s
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseSroti_rank_0,&
      cusparseSroti_rank_1
  end interface
  
  interface cusparseDroti
    function cusparseDroti_(handle,nnz,xVal,xInd,y,c,s,idxBase) bind(c,&
& name="cusparseDroti")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDroti_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      real(c_double) :: c
      real(c_double) :: s
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseDroti_rank_0,&
      cusparseDroti_rank_1
  end interface
  
  interface cusparseSsctr
    function cusparseSsctr_(handle,nnz,xVal,xInd,y,idxBase) bind(c, &
&name="cusparseSsctr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSsctr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseSsctr_rank_0,&
      cusparseSsctr_rank_1
  end interface
  
  interface cusparseDsctr
    function cusparseDsctr_(handle,nnz,xVal,xInd,y,idxBase) bind(c, &
&name="cusparseDsctr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDsctr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseDsctr_rank_0,&
      cusparseDsctr_rank_1
  end interface
  
  interface cusparseCsctr
    function cusparseCsctr_(handle,nnz,xVal,xInd,y,idxBase) bind(c, &
&name="cusparseCsctr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCsctr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseCsctr_rank_0,&
      cusparseCsctr_rank_1
  end interface
  
  interface cusparseZsctr
    function cusparseZsctr_(handle,nnz,xVal,xInd,y,idxBase) bind(c, &
&name="cusparseZsctr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZsctr_
      type(c_ptr),value :: handle
      integer(c_int),value :: nnz
      type(c_ptr),value :: xVal
      type(c_ptr),value :: xInd
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseZsctr_rank_0,&
      cusparseZsctr_rank_1
  end interface
  
  interface cusparseScsrmv
    function cusparseScsrmv_(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y) bind(c, &
&name="cusparseScsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: x
      real(c_float) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseScsrmv_rank_0,&
      cusparseScsrmv_rank_1
  end interface
  
  interface cusparseDcsrmv
    function cusparseDcsrmv_(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y) bind(c, &
&name="cusparseDcsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: x
      real(c_double) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseDcsrmv_rank_0,&
      cusparseDcsrmv_rank_1
  end interface
  
  interface cusparseCcsrmv
    function cusparseCcsrmv_(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y) bind(c, &
&name="cusparseCcsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: x
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseCcsrmv_rank_0,&
      cusparseCcsrmv_rank_1
  end interface
  
  interface cusparseZcsrmv
    function cusparseZcsrmv_(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y) bind(c, &
&name="cusparseZcsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: x
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseZcsrmv_rank_0,&
      cusparseZcsrmv_rank_1
  end interface
  !>  \ingroup level2_module
  !>   \brief Sparse triangular solve using CSR storage format
  !> 
  !>   \details
  !>   \p hipsparseXcsrsv2_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseScsrsv2_solve(),
  !>   hipsparseDcsrsv2_solve(), hipsparseCcsrsv2_solve() or hipsparseZcsrsv2_solve()
  !>   computation. The first zero pivot \f$j\f$ at \f$A_{j,j}\f$ is stored in \p position,
  !>   using same index base as the CSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note \p hipsparseXcsrsv2_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXcsrsv2_zeroPivot
    function cusparseXcsrsv2_zeroPivot_(handle,myInfo,position) bind(c,&
& name="cusparseXcsrsv2_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrsv2_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseScsrsv2_bufferSize
    function cusparseScsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseScsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseScsrsv2_bufferSize_rank_0,&
      cusparseScsrsv2_bufferSize_rank_1
  end interface
  
  interface cusparseDcsrsv2_bufferSize
    function cusparseDcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseDcsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDcsrsv2_bufferSize_rank_0,&
      cusparseDcsrsv2_bufferSize_rank_1
  end interface
  
  interface cusparseCcsrsv2_bufferSize
    function cusparseCcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseCcsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCcsrsv2_bufferSize_rank_0,&
      cusparseCcsrsv2_bufferSize_rank_1
  end interface
  
  interface cusparseZcsrsv2_bufferSize
    function cusparseZcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseZcsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZcsrsv2_bufferSize_rank_0,&
      cusparseZcsrsv2_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseScsrsv2_bufferSizeExt
    function cusparseScsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseScsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseScsrsv2_bufferSizeExt_rank_0,&
      cusparseScsrsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDcsrsv2_bufferSizeExt
    function cusparseDcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseDcsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseDcsrsv2_bufferSizeExt_rank_0,&
      cusparseDcsrsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCcsrsv2_bufferSizeExt
    function cusparseCcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseCcsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseCcsrsv2_bufferSizeExt_rank_0,&
      cusparseCcsrsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZcsrsv2_bufferSizeExt
    function cusparseZcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseZcsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseZcsrsv2_bufferSizeExt_rank_0,&
      cusparseZcsrsv2_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseScsrsv2_analysis
    function cusparseScsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseScsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrsv2_analysis_rank_0,&
      cusparseScsrsv2_analysis_rank_1
  end interface
  
  interface cusparseDcsrsv2_analysis
    function cusparseDcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseDcsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrsv2_analysis_rank_0,&
      cusparseDcsrsv2_analysis_rank_1
  end interface
  
  interface cusparseCcsrsv2_analysis
    function cusparseCcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseCcsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrsv2_analysis_rank_0,&
      cusparseCcsrsv2_analysis_rank_1
  end interface
  
  interface cusparseZcsrsv2_analysis
    function cusparseZcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseZcsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrsv2_analysis_rank_0,&
      cusparseZcsrsv2_analysis_rank_1
  end interface
  !> @{
  interface cusparseScsrsv2_solve
    function cusparseScsrsv2_solve_(handle,transA,m,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,policy,&
&pBuffer) bind(c, name="cusparseScsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrsv2_solve_rank_0,&
      cusparseScsrsv2_solve_rank_1
  end interface
  
  interface cusparseDcsrsv2_solve
    function cusparseDcsrsv2_solve_(handle,transA,m,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,policy,&
&pBuffer) bind(c, name="cusparseDcsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrsv2_solve_rank_0,&
      cusparseDcsrsv2_solve_rank_1
  end interface
  
  interface cusparseCcsrsv2_solve
    function cusparseCcsrsv2_solve_(handle,transA,m,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,policy,&
&pBuffer) bind(c, name="cusparseCcsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrsv2_solve_rank_0,&
      cusparseCcsrsv2_solve_rank_1
  end interface
  
  interface cusparseZcsrsv2_solve
    function cusparseZcsrsv2_solve_(handle,transA,m,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,policy,&
&pBuffer) bind(c, name="cusparseZcsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrsv2_solve_rank_0,&
      cusparseZcsrsv2_solve_rank_1
  end interface
  
  interface cusparseShybmv
    function cusparseShybmv_(handle,transA,alpha,descrA,hybA,x,beta,y) &
&bind(c, name="cusparseShybmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseShybmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: x
      real(c_float) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseShybmv_rank_0,&
      cusparseShybmv_rank_1
  end interface
  
  interface cusparseDhybmv
    function cusparseDhybmv_(handle,transA,alpha,descrA,hybA,x,beta,y) &
&bind(c, name="cusparseDhybmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDhybmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: x
      real(c_double) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseDhybmv_rank_0,&
      cusparseDhybmv_rank_1
  end interface
  
  interface cusparseChybmv
    function cusparseChybmv_(handle,transA,alpha,descrA,hybA,x,beta,y) &
&bind(c, name="cusparseChybmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseChybmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: x
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseChybmv_rank_0,&
      cusparseChybmv_rank_1
  end interface
  
  interface cusparseZhybmv
    function cusparseZhybmv_(handle,transA,alpha,descrA,hybA,x,beta,y) &
&bind(c, name="cusparseZhybmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZhybmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: x
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseZhybmv_rank_0,&
      cusparseZhybmv_rank_1
  end interface
  !> @{
  interface cusparseSbsrmv
    function cusparseSbsrmv_(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y) bind(c, name="cusparseSbsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      real(c_float) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseSbsrmv_rank_0,&
      cusparseSbsrmv_rank_1
  end interface
  
  interface cusparseDbsrmv
    function cusparseDbsrmv_(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y) bind(c, name="cusparseDbsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      real(c_double) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseDbsrmv_rank_0,&
      cusparseDbsrmv_rank_1
  end interface
  
  interface cusparseCbsrmv
    function cusparseCbsrmv_(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y) bind(c, name="cusparseCbsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseCbsrmv_rank_0,&
      cusparseCbsrmv_rank_1
  end interface
  
  interface cusparseZbsrmv
    function cusparseZbsrmv_(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y) bind(c, name="cusparseZbsrmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseZbsrmv_rank_0,&
      cusparseZbsrmv_rank_1
  end interface
  !> @{
  interface cusparseSbsrxmv
    function cusparseSbsrxmv_(handle,dir,trans,sizeOfMask,mb,nb,nnzb,&
&alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,blockDim,&
&x,beta,y) bind(c, name="cusparseSbsrxmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrxmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: trans
      integer(c_int),value :: sizeOfMask
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      real(c_float) :: alpha
      type(c_ptr),value :: descr
      type(c_ptr),value :: bsrVal
      type(c_ptr),value :: bsrMaskPtr
      type(c_ptr),value :: bsrRowPtr
      type(c_ptr),value :: bsrEndPtr
      type(c_ptr),value :: bsrColInd
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      real(c_float) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseSbsrxmv_rank_0,&
      cusparseSbsrxmv_rank_1
  end interface
  
  interface cusparseDbsrxmv
    function cusparseDbsrxmv_(handle,dir,trans,sizeOfMask,mb,nb,nnzb,&
&alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,blockDim,&
&x,beta,y) bind(c, name="cusparseDbsrxmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrxmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: trans
      integer(c_int),value :: sizeOfMask
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      real(c_double) :: alpha
      type(c_ptr),value :: descr
      type(c_ptr),value :: bsrVal
      type(c_ptr),value :: bsrMaskPtr
      type(c_ptr),value :: bsrRowPtr
      type(c_ptr),value :: bsrEndPtr
      type(c_ptr),value :: bsrColInd
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      real(c_double) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseDbsrxmv_rank_0,&
      cusparseDbsrxmv_rank_1
  end interface
  
  interface cusparseCbsrxmv
    function cusparseCbsrxmv_(handle,dir,trans,sizeOfMask,mb,nb,nnzb,&
&alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,blockDim,&
&x,beta,y) bind(c, name="cusparseCbsrxmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrxmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: trans
      integer(c_int),value :: sizeOfMask
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descr
      type(c_ptr),value :: bsrVal
      type(c_ptr),value :: bsrMaskPtr
      type(c_ptr),value :: bsrRowPtr
      type(c_ptr),value :: bsrEndPtr
      type(c_ptr),value :: bsrColInd
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseCbsrxmv_rank_0,&
      cusparseCbsrxmv_rank_1
  end interface
  
  interface cusparseZbsrxmv
    function cusparseZbsrxmv_(handle,dir,trans,sizeOfMask,mb,nb,nnzb,&
&alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,blockDim,&
&x,beta,y) bind(c, name="cusparseZbsrxmv")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrxmv_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: trans
      integer(c_int),value :: sizeOfMask
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descr
      type(c_ptr),value :: bsrVal
      type(c_ptr),value :: bsrMaskPtr
      type(c_ptr),value :: bsrRowPtr
      type(c_ptr),value :: bsrEndPtr
      type(c_ptr),value :: bsrColInd
      integer(c_int),value :: blockDim
      type(c_ptr),value :: x
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
    end function

    module procedure &
      cusparseZbsrxmv_rank_0,&
      cusparseZbsrxmv_rank_1
  end interface
  !>  \ingroup level2_module
  !>   \brief Sparse triangular solve using BSR storage format
  !> 
  !>   \details
  !>   \p hipsparseXbsrsv2_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseXbsrsv2_analysis() or
  !>   hipsparseXbsrsv2_solve() computation. The first zero pivot \f$j\f$ at \f$A_{j,j}\f$
  !>   is stored in \p position, using same index base as the BSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note \p hipsparseXbsrsv2_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXbsrsv2_zeroPivot
    function cusparseXbsrsv2_zeroPivot_(handle,myInfo,position) bind(c,&
& name="cusparseXbsrsv2_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXbsrsv2_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseSbsrsv2_bufferSize
    function cusparseSbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes) bind(c, name="cusparseSbsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseSbsrsv2_bufferSize_rank_0,&
      cusparseSbsrsv2_bufferSize_rank_1
  end interface
  
  interface cusparseDbsrsv2_bufferSize
    function cusparseDbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes) bind(c, name="cusparseDbsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDbsrsv2_bufferSize_rank_0,&
      cusparseDbsrsv2_bufferSize_rank_1
  end interface
  
  interface cusparseCbsrsv2_bufferSize
    function cusparseCbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes) bind(c, name="cusparseCbsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCbsrsv2_bufferSize_rank_0,&
      cusparseCbsrsv2_bufferSize_rank_1
  end interface
  
  interface cusparseZbsrsv2_bufferSize
    function cusparseZbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes) bind(c, name="cusparseZbsrsv2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZbsrsv2_bufferSize_rank_0,&
      cusparseZbsrsv2_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseSbsrsv2_bufferSizeExt
    function cusparseSbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSize) bind(c, name="cusparseSbsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseSbsrsv2_bufferSizeExt_rank_0,&
      cusparseSbsrsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDbsrsv2_bufferSizeExt
    function cusparseDbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSize) bind(c, name="cusparseDbsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseDbsrsv2_bufferSizeExt_rank_0,&
      cusparseDbsrsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCbsrsv2_bufferSizeExt
    function cusparseCbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSize) bind(c, name="cusparseCbsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseCbsrsv2_bufferSizeExt_rank_0,&
      cusparseCbsrsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZbsrsv2_bufferSizeExt
    function cusparseZbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSize) bind(c, name="cusparseZbsrsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseZbsrsv2_bufferSizeExt_rank_0,&
      cusparseZbsrsv2_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSbsrsv2_analysis
    function cusparseSbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer) bind(c, name="cusparseSbsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsrsv2_analysis_rank_0,&
      cusparseSbsrsv2_analysis_rank_1
  end interface
  
  interface cusparseDbsrsv2_analysis
    function cusparseDbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer) bind(c, name="cusparseDbsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsrsv2_analysis_rank_0,&
      cusparseDbsrsv2_analysis_rank_1
  end interface
  
  interface cusparseCbsrsv2_analysis
    function cusparseCbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer) bind(c, name="cusparseCbsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsrsv2_analysis_rank_0,&
      cusparseCbsrsv2_analysis_rank_1
  end interface
  
  interface cusparseZbsrsv2_analysis
    function cusparseZbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer) bind(c, name="cusparseZbsrsv2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsrsv2_analysis_rank_0,&
      cusparseZbsrsv2_analysis_rank_1
  end interface
  !> @{
  interface cusparseSbsrsv2_solve
    function cusparseSbsrsv2_solve_(handle,dirA,transA,mb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer) bind(c, name="cusparseSbsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsrsv2_solve_rank_0,&
      cusparseSbsrsv2_solve_rank_1
  end interface
  
  interface cusparseDbsrsv2_solve
    function cusparseDbsrsv2_solve_(handle,dirA,transA,mb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer) bind(c, name="cusparseDbsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsrsv2_solve_rank_0,&
      cusparseDbsrsv2_solve_rank_1
  end interface
  
  interface cusparseCbsrsv2_solve
    function cusparseCbsrsv2_solve_(handle,dirA,transA,mb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer) bind(c, name="cusparseCbsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsrsv2_solve_rank_0,&
      cusparseCbsrsv2_solve_rank_1
  end interface
  
  interface cusparseZbsrsv2_solve
    function cusparseZbsrsv2_solve_(handle,dirA,transA,mb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer) bind(c, name="cusparseZbsrsv2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrsv2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: f
      type(c_ptr),value :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsrsv2_solve_rank_0,&
      cusparseZbsrsv2_solve_rank_1
  end interface
  !> @{
  interface cusparseSgemvi_bufferSize
    function cusparseSgemvi_bufferSize_(handle,transA,m,n,nnz,&
&pBufferSize) bind(c, name="cusparseSgemvi_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgemvi_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      integer(c_int) :: pBufferSize
    end function

  end interface
  
  interface cusparseDgemvi_bufferSize
    function cusparseDgemvi_bufferSize_(handle,transA,m,n,nnz,&
&pBufferSize) bind(c, name="cusparseDgemvi_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgemvi_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      integer(c_int) :: pBufferSize
    end function

  end interface
  
  interface cusparseCgemvi_bufferSize
    function cusparseCgemvi_bufferSize_(handle,transA,m,n,nnz,&
&pBufferSize) bind(c, name="cusparseCgemvi_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgemvi_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      integer(c_int) :: pBufferSize
    end function

  end interface
  
  interface cusparseZgemvi_bufferSize
    function cusparseZgemvi_bufferSize_(handle,transA,m,n,nnz,&
&pBufferSize) bind(c, name="cusparseZgemvi_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgemvi_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      integer(c_int) :: pBufferSize
    end function

  end interface
  !> @{
  interface cusparseSgemvi
    function cusparseSgemvi_(handle,transA,m,n,alpha,A,lda,nnz,x,xInd,&
&beta,y,idxBase,pBuffer) bind(c, name="cusparseSgemvi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemvi_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int),value :: nnz
      type(c_ptr),value :: x
      type(c_ptr),value :: xInd
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSgemvi_full_rank,&
      cusparseSgemvi_rank_0,&
      cusparseSgemvi_rank_1
  end interface
  
  interface cusparseDgemvi
    function cusparseDgemvi_(handle,transA,m,n,alpha,A,lda,nnz,x,xInd,&
&beta,y,idxBase,pBuffer) bind(c, name="cusparseDgemvi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemvi_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int),value :: nnz
      type(c_ptr),value :: x
      type(c_ptr),value :: xInd
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDgemvi_full_rank,&
      cusparseDgemvi_rank_0,&
      cusparseDgemvi_rank_1
  end interface
  
  interface cusparseCgemvi
    function cusparseCgemvi_(handle,transA,m,n,alpha,A,lda,nnz,x,xInd,&
&beta,y,idxBase,pBuffer) bind(c, name="cusparseCgemvi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemvi_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int),value :: nnz
      type(c_ptr),value :: x
      type(c_ptr),value :: xInd
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCgemvi_full_rank,&
      cusparseCgemvi_rank_0,&
      cusparseCgemvi_rank_1
  end interface
  
  interface cusparseZgemvi
    function cusparseZgemvi_(handle,transA,m,n,alpha,A,lda,nnz,x,xInd,&
&beta,y,idxBase,pBuffer) bind(c, name="cusparseZgemvi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemvi_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int),value :: nnz
      type(c_ptr),value :: x
      type(c_ptr),value :: xInd
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZgemvi_full_rank,&
      cusparseZgemvi_rank_0,&
      cusparseZgemvi_rank_1
  end interface
  !> @{
  interface cusparseSbsrmm
    function cusparseSbsrmm_(handle,dirA,transA,transB,mb,n,kb,nnzb,&
&alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseSbsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: mb
      integer(c_int),value :: n
      integer(c_int),value :: kb
      integer(c_int),value :: nnzb
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseSbsrmm_full_rank,&
      cusparseSbsrmm_rank_0,&
      cusparseSbsrmm_rank_1
  end interface
  
  interface cusparseDbsrmm
    function cusparseDbsrmm_(handle,dirA,transA,transB,mb,n,kb,nnzb,&
&alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseDbsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: mb
      integer(c_int),value :: n
      integer(c_int),value :: kb
      integer(c_int),value :: nnzb
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseDbsrmm_full_rank,&
      cusparseDbsrmm_rank_0,&
      cusparseDbsrmm_rank_1
  end interface
  
  interface cusparseCbsrmm
    function cusparseCbsrmm_(handle,dirA,transA,transB,mb,n,kb,nnzb,&
&alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseCbsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: mb
      integer(c_int),value :: n
      integer(c_int),value :: kb
      integer(c_int),value :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseCbsrmm_full_rank,&
      cusparseCbsrmm_rank_0,&
      cusparseCbsrmm_rank_1
  end interface
  
  interface cusparseZbsrmm
    function cusparseZbsrmm_(handle,dirA,transA,transB,mb,n,kb,nnzb,&
&alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseZbsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: mb
      integer(c_int),value :: n
      integer(c_int),value :: kb
      integer(c_int),value :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseZbsrmm_full_rank,&
      cusparseZbsrmm_rank_0,&
      cusparseZbsrmm_rank_1
  end interface
  
  interface cusparseScsrmm
    function cusparseScsrmm_(handle,transA,m,n,k,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseScsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseScsrmm_full_rank,&
      cusparseScsrmm_rank_0,&
      cusparseScsrmm_rank_1
  end interface
  
  interface cusparseDcsrmm
    function cusparseDcsrmm_(handle,transA,m,n,k,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseDcsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseDcsrmm_full_rank,&
      cusparseDcsrmm_rank_0,&
      cusparseDcsrmm_rank_1
  end interface
  
  interface cusparseCcsrmm
    function cusparseCcsrmm_(handle,transA,m,n,k,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseCcsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseCcsrmm_full_rank,&
      cusparseCcsrmm_rank_0,&
      cusparseCcsrmm_rank_1
  end interface
  
  interface cusparseZcsrmm
    function cusparseZcsrmm_(handle,transA,m,n,k,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,ldc) &
&bind(c, name="cusparseZcsrmm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseZcsrmm_full_rank,&
      cusparseZcsrmm_rank_0,&
      cusparseZcsrmm_rank_1
  end interface
  
  interface cusparseScsrmm2
    function cusparseScsrmm2_(handle,transA,transB,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc) bind(c, name="cusparseScsrmm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmm2_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseScsrmm2_full_rank,&
      cusparseScsrmm2_rank_0,&
      cusparseScsrmm2_rank_1
  end interface
  
  interface cusparseDcsrmm2
    function cusparseDcsrmm2_(handle,transA,transB,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc) bind(c, name="cusparseDcsrmm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmm2_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseDcsrmm2_full_rank,&
      cusparseDcsrmm2_rank_0,&
      cusparseDcsrmm2_rank_1
  end interface
  
  interface cusparseCcsrmm2
    function cusparseCcsrmm2_(handle,transA,transB,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc) bind(c, name="cusparseCcsrmm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmm2_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseCcsrmm2_full_rank,&
      cusparseCcsrmm2_rank_0,&
      cusparseCcsrmm2_rank_1
  end interface
  
  interface cusparseZcsrmm2
    function cusparseZcsrmm2_(handle,transA,transB,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc) bind(c, name="cusparseZcsrmm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmm2_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseZcsrmm2_full_rank,&
      cusparseZcsrmm2_rank_0,&
      cusparseZcsrmm2_rank_1
  end interface
  !>  \ingroup level3_module
  !>   \brief Sparse triangular system solve using BSR storage format
  !> 
  !>   \details
  !>   \p hipsparseXbsrsm2_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseXbsrsm2_analysis() or
  !>   hipsparseXbsrsm2_solve() computation. The first zero pivot \f$j\f$ at \f$A_{j,j}\f$
  !>   is stored in \p position, using same index base as the BSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note \p hipsparseXbsrsm2_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXbsrsm2_zeroPivot
    function cusparseXbsrsm2_zeroPivot_(handle,myInfo,position) bind(c,&
& name="cusparseXbsrsm2_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXbsrsm2_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseSbsrsm2_bufferSize
    function cusparseSbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseSbsrsm2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseSbsrsm2_bufferSize_rank_0,&
      cusparseSbsrsm2_bufferSize_rank_1
  end interface
  
  interface cusparseDbsrsm2_bufferSize
    function cusparseDbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseDbsrsm2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDbsrsm2_bufferSize_rank_0,&
      cusparseDbsrsm2_bufferSize_rank_1
  end interface
  
  interface cusparseCbsrsm2_bufferSize
    function cusparseCbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseCbsrsm2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCbsrsm2_bufferSize_rank_0,&
      cusparseCbsrsm2_bufferSize_rank_1
  end interface
  
  interface cusparseZbsrsm2_bufferSize
    function cusparseZbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseZbsrsm2_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZbsrsm2_bufferSize_rank_0,&
      cusparseZbsrsm2_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseSbsrsm2_analysis
    function cusparseSbsrsm2_analysis_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseSbsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsrsm2_analysis_rank_0,&
      cusparseSbsrsm2_analysis_rank_1
  end interface
  
  interface cusparseDbsrsm2_analysis
    function cusparseDbsrsm2_analysis_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseDbsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsrsm2_analysis_rank_0,&
      cusparseDbsrsm2_analysis_rank_1
  end interface
  
  interface cusparseCbsrsm2_analysis
    function cusparseCbsrsm2_analysis_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseCbsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsrsm2_analysis_rank_0,&
      cusparseCbsrsm2_analysis_rank_1
  end interface
  
  interface cusparseZbsrsm2_analysis
    function cusparseZbsrsm2_analysis_(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseZbsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsrsm2_analysis_rank_0,&
      cusparseZbsrsm2_analysis_rank_1
  end interface
  !> @{
  interface cusparseSbsrsm2_solve
    function cusparseSbsrsm2_solve_(handle,dirA,transA,transX,mb,nrhs,&
&nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer) bind(c, &
&name="cusparseSbsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrsm2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: X
      integer(c_int),value :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsrsm2_solve_full_rank,&
      cusparseSbsrsm2_solve_rank_0,&
      cusparseSbsrsm2_solve_rank_1
  end interface
  
  interface cusparseDbsrsm2_solve
    function cusparseDbsrsm2_solve_(handle,dirA,transA,transX,mb,nrhs,&
&nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer) bind(c, &
&name="cusparseDbsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrsm2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: X
      integer(c_int),value :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsrsm2_solve_full_rank,&
      cusparseDbsrsm2_solve_rank_0,&
      cusparseDbsrsm2_solve_rank_1
  end interface
  
  interface cusparseCbsrsm2_solve
    function cusparseCbsrsm2_solve_(handle,dirA,transA,transX,mb,nrhs,&
&nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer) bind(c, &
&name="cusparseCbsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrsm2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: X
      integer(c_int),value :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsrsm2_solve_full_rank,&
      cusparseCbsrsm2_solve_rank_0,&
      cusparseCbsrsm2_solve_rank_1
  end interface
  
  interface cusparseZbsrsm2_solve
    function cusparseZbsrsm2_solve_(handle,dirA,transA,transX,mb,nrhs,&
&nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer) bind(c, &
&name="cusparseZbsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrsm2_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transX
      integer(c_int),value :: mb
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: X
      integer(c_int),value :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsrsm2_solve_full_rank,&
      cusparseZbsrsm2_solve_rank_0,&
      cusparseZbsrsm2_solve_rank_1
  end interface
  !>  \ingroup level3_module
  !>   \brief Sparse triangular system solve using CSR storage format
  !> 
  !>   \details
  !>   \p hipsparseXcsrsm2_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseXcsrsm2_analysis() or
  !>   hipsparseXcsrsm2_solve() computation. The first zero pivot \f$j\f$ at \f$A_{j,j}\f$
  !>   is stored in \p position, using same index base as the CSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note \p hipsparseXcsrsm2_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXcsrsm2_zeroPivot
    function cusparseXcsrsm2_zeroPivot_(handle,myInfo,position) bind(c,&
& name="cusparseXcsrsm2_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrsm2_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseScsrsm2_bufferSizeExt
    function cusparseScsrsm2_bufferSizeExt_(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize) bind(c, &
&name="cusparseScsrsm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseScsrsm2_bufferSizeExt_full_rank,&
      cusparseScsrsm2_bufferSizeExt_rank_0,&
      cusparseScsrsm2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDcsrsm2_bufferSizeExt
    function cusparseDcsrsm2_bufferSizeExt_(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize) bind(c, &
&name="cusparseDcsrsm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseDcsrsm2_bufferSizeExt_full_rank,&
      cusparseDcsrsm2_bufferSizeExt_rank_0,&
      cusparseDcsrsm2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCcsrsm2_bufferSizeExt
    function cusparseCcsrsm2_bufferSizeExt_(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize) bind(c, &
&name="cusparseCcsrsm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseCcsrsm2_bufferSizeExt_full_rank,&
      cusparseCcsrsm2_bufferSizeExt_rank_0,&
      cusparseCcsrsm2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZcsrsm2_bufferSizeExt
    function cusparseZcsrsm2_bufferSizeExt_(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize) bind(c, &
&name="cusparseZcsrsm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseZcsrsm2_bufferSizeExt_full_rank,&
      cusparseZcsrsm2_bufferSizeExt_rank_0,&
      cusparseZcsrsm2_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseScsrsm2_analysis
    function cusparseScsrsm2_analysis_(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer) bind(c, name="cusparseScsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrsm2_analysis_full_rank,&
      cusparseScsrsm2_analysis_rank_0,&
      cusparseScsrsm2_analysis_rank_1
  end interface
  
  interface cusparseDcsrsm2_analysis
    function cusparseDcsrsm2_analysis_(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer) bind(c, name="cusparseDcsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrsm2_analysis_full_rank,&
      cusparseDcsrsm2_analysis_rank_0,&
      cusparseDcsrsm2_analysis_rank_1
  end interface
  
  interface cusparseCcsrsm2_analysis
    function cusparseCcsrsm2_analysis_(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer) bind(c, name="cusparseCcsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrsm2_analysis_full_rank,&
      cusparseCcsrsm2_analysis_rank_0,&
      cusparseCcsrsm2_analysis_rank_1
  end interface
  
  interface cusparseZcsrsm2_analysis
    function cusparseZcsrsm2_analysis_(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer) bind(c, name="cusparseZcsrsm2_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrsm2_analysis_full_rank,&
      cusparseZcsrsm2_analysis_rank_0,&
      cusparseZcsrsm2_analysis_rank_1
  end interface
  !> @{
  interface cusparseScsrsm2_solve
    function cusparseScsrsm2_solve_(handle,algo,transA,transB,m,nrhs,&
&nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,&
&ldb,myInfo,policy,pBuffer) bind(c, name="cusparseScsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrsm2_solve_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrsm2_solve_full_rank,&
      cusparseScsrsm2_solve_rank_0,&
      cusparseScsrsm2_solve_rank_1
  end interface
  
  interface cusparseDcsrsm2_solve
    function cusparseDcsrsm2_solve_(handle,algo,transA,transB,m,nrhs,&
&nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,&
&ldb,myInfo,policy,pBuffer) bind(c, name="cusparseDcsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrsm2_solve_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrsm2_solve_full_rank,&
      cusparseDcsrsm2_solve_rank_0,&
      cusparseDcsrsm2_solve_rank_1
  end interface
  
  interface cusparseCcsrsm2_solve
    function cusparseCcsrsm2_solve_(handle,algo,transA,transB,m,nrhs,&
&nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,&
&ldb,myInfo,policy,pBuffer) bind(c, name="cusparseCcsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrsm2_solve_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrsm2_solve_full_rank,&
      cusparseCcsrsm2_solve_rank_0,&
      cusparseCcsrsm2_solve_rank_1
  end interface
  
  interface cusparseZcsrsm2_solve
    function cusparseZcsrsm2_solve_(handle,algo,transA,transB,m,nrhs,&
&nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,&
&ldb,myInfo,policy,pBuffer) bind(c, name="cusparseZcsrsm2_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrsm2_solve_
      type(c_ptr),value :: handle
      integer(c_int),value :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: nrhs
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrsm2_solve_full_rank,&
      cusparseZcsrsm2_solve_rank_0,&
      cusparseZcsrsm2_solve_rank_1
  end interface
  
  interface cusparseSgemmi
    function cusparseSgemmi_(handle,m,n,k,nnz,alpha,A,lda,cscValB,&
&cscColPtrB,cscRowIndB,beta,C,ldc) bind(c, name="cusparseSgemmi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemmi_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: cscValB
      type(c_ptr),value :: cscColPtrB
      type(c_ptr),value :: cscRowIndB
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseSgemmi_full_rank,&
      cusparseSgemmi_rank_0,&
      cusparseSgemmi_rank_1
  end interface
  
  interface cusparseDgemmi
    function cusparseDgemmi_(handle,m,n,k,nnz,alpha,A,lda,cscValB,&
&cscColPtrB,cscRowIndB,beta,C,ldc) bind(c, name="cusparseDgemmi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemmi_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: cscValB
      type(c_ptr),value :: cscColPtrB
      type(c_ptr),value :: cscRowIndB
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseDgemmi_full_rank,&
      cusparseDgemmi_rank_0,&
      cusparseDgemmi_rank_1
  end interface
  
  interface cusparseCgemmi
    function cusparseCgemmi_(handle,m,n,k,nnz,alpha,A,lda,cscValB,&
&cscColPtrB,cscRowIndB,beta,C,ldc) bind(c, name="cusparseCgemmi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemmi_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: cscValB
      type(c_ptr),value :: cscColPtrB
      type(c_ptr),value :: cscRowIndB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseCgemmi_full_rank,&
      cusparseCgemmi_rank_0,&
      cusparseCgemmi_rank_1
  end interface
  
  interface cusparseZgemmi
    function cusparseZgemmi_(handle,m,n,k,nnz,alpha,A,lda,cscValB,&
&cscColPtrB,cscRowIndB,beta,C,ldc) bind(c, name="cusparseZgemmi")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemmi_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      integer(c_int),value :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: cscValB
      type(c_ptr),value :: cscColPtrB
      type(c_ptr),value :: cscRowIndB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cusparseZgemmi_full_rank,&
      cusparseZgemmi_rank_0,&
      cusparseZgemmi_rank_1
  end interface
  
  interface cusparseXcsrgeamNnz
    function cusparseXcsrgeamNnz_(handle,m,n,descrA,nnzA,csrRowPtrA,&
&csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,descrC,csrRowPtrC,&
&nnzTotalDevHostPtr) bind(c, name="cusparseXcsrgeamNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsrgeamNnz_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
    end function

    module procedure &
      cusparseXcsrgeamNnz_rank_0,&
      cusparseXcsrgeamNnz_rank_1
  end interface
  
  interface cusparseScsrgeam
    function cusparseScsrgeam_(handle,m,n,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,&
&descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseScsrgeam")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgeam_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseScsrgeam_rank_0,&
      cusparseScsrgeam_rank_1
  end interface
  
  interface cusparseDcsrgeam
    function cusparseDcsrgeam_(handle,m,n,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,&
&descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseDcsrgeam")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgeam_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseDcsrgeam_rank_0,&
      cusparseDcsrgeam_rank_1
  end interface
  
  interface cusparseCcsrgeam
    function cusparseCcsrgeam_(handle,m,n,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,&
&descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseCcsrgeam")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgeam_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      complex(c_float_complex) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseCcsrgeam_rank_0,&
      cusparseCcsrgeam_rank_1
  end interface
  
  interface cusparseZcsrgeam
    function cusparseZcsrgeam_(handle,m,n,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,&
&descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseZcsrgeam")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgeam_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      complex(c_double_complex) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseZcsrgeam_rank_0,&
      cusparseZcsrgeam_rank_1
  end interface
  !> @{
  interface cusparseScsrgeam2_bufferSizeExt
    function cusparseScsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,&
&nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes) bind(c, &
&name="cusparseScsrgeam2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrgeam2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      real(c_float) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseScsrgeam2_bufferSizeExt_rank_0,&
      cusparseScsrgeam2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDcsrgeam2_bufferSizeExt
    function cusparseDcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,&
&nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes) bind(c, &
&name="cusparseDcsrgeam2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrgeam2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      real(c_double) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDcsrgeam2_bufferSizeExt_rank_0,&
      cusparseDcsrgeam2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCcsrgeam2_bufferSizeExt
    function cusparseCcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,&
&nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes) bind(c, &
&name="cusparseCcsrgeam2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrgeam2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      complex(c_float_complex) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCcsrgeam2_bufferSizeExt_rank_0,&
      cusparseCcsrgeam2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZcsrgeam2_bufferSizeExt
    function cusparseZcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,&
&nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes) bind(c, &
&name="cusparseZcsrgeam2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrgeam2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      complex(c_double_complex) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZcsrgeam2_bufferSizeExt_rank_0,&
      cusparseZcsrgeam2_bufferSizeExt_rank_1
  end interface
  !>  \ingroup extra_module
  !>   \brief Sparse matrix sparse matrix addition using CSR storage format
  !> 
  !>   \details
  !>   \p hipsparseXcsrgeam2Nnz computes the total CSR non-zero elements and the CSR row
  !>   offsets, that point to the start of every row of the sparse CSR matrix, of the
  !>   resulting matrix C. It is assumed that \p csr_row_ptr_C has been allocated with
  !>   size \p m + 1.
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  !>   \note
  !>   Currently, only \ref HIPSPARSE_MATRIX_TYPE_GENERAL is supported.
  interface cusparseXcsrgeam2Nnz
    function cusparseXcsrgeam2Nnz_(handle,m,n,descrA,nnzA,&
&csrSortedRowPtrA,csrSortedColIndA,descrB,nnzB,csrSortedRowPtrB,&
&csrSortedColIndB,descrC,csrSortedRowPtrC,nnzTotalDevHostPtr,&
&workspace) bind(c, name="cusparseXcsrgeam2Nnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsrgeam2Nnz_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: workspace
    end function

    module procedure &
      cusparseXcsrgeam2Nnz_rank_0,&
      cusparseXcsrgeam2Nnz_rank_1
  end interface
  !> @{
  interface cusparseScsrgeam2
    function cusparseScsrgeam2_(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer) bind(c, &
&name="cusparseScsrgeam2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgeam2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      real(c_float) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrgeam2_rank_0,&
      cusparseScsrgeam2_rank_1
  end interface
  
  interface cusparseDcsrgeam2
    function cusparseDcsrgeam2_(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer) bind(c, &
&name="cusparseDcsrgeam2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgeam2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      real(c_double) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrgeam2_rank_0,&
      cusparseDcsrgeam2_rank_1
  end interface
  
  interface cusparseCcsrgeam2
    function cusparseCcsrgeam2_(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer) bind(c, &
&name="cusparseCcsrgeam2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgeam2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      complex(c_float_complex) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrgeam2_rank_0,&
      cusparseCcsrgeam2_rank_1
  end interface
  
  interface cusparseZcsrgeam2
    function cusparseZcsrgeam2_(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer) bind(c, &
&name="cusparseZcsrgeam2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgeam2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      complex(c_double_complex) :: beta
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrSortedValB
      type(c_ptr),value :: csrSortedRowPtrB
      type(c_ptr),value :: csrSortedColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrSortedValC
      type(c_ptr),value :: csrSortedRowPtrC
      type(c_ptr),value :: csrSortedColIndC
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrgeam2_rank_0,&
      cusparseZcsrgeam2_rank_1
  end interface
  
  interface cusparseXcsrgemmNnz
    function cusparseXcsrgemmNnz_(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,descrC,&
&csrRowPtrC,nnzTotalDevHostPtr) bind(c, name="cusparseXcsrgemmNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsrgemmNnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
    end function

    module procedure &
      cusparseXcsrgemmNnz_rank_0,&
      cusparseXcsrgemmNnz_rank_1
  end interface
  
  interface cusparseScsrgemm
    function cusparseScsrgemm_(handle,transA,transB,m,n,k,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseScsrgemm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgemm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseScsrgemm_rank_0,&
      cusparseScsrgemm_rank_1
  end interface
  
  interface cusparseDcsrgemm
    function cusparseDcsrgemm_(handle,transA,transB,m,n,k,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseDcsrgemm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgemm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseDcsrgemm_rank_0,&
      cusparseDcsrgemm_rank_1
  end interface
  
  interface cusparseCcsrgemm
    function cusparseCcsrgemm_(handle,transA,transB,m,n,k,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseCcsrgemm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgemm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseCcsrgemm_rank_0,&
      cusparseCcsrgemm_rank_1
  end interface
  
  interface cusparseZcsrgemm
    function cusparseZcsrgemm_(handle,transA,transB,m,n,k,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC) bind(c, &
&name="cusparseZcsrgemm")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgemm_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: transB
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseZcsrgemm_rank_0,&
      cusparseZcsrgemm_rank_1
  end interface
  
  interface cusparseScsrgemm2_bufferSizeExt
    function cusparseScsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseScsrgemm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrgemm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      real(c_float) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseScsrgemm2_bufferSizeExt_rank_0,&
      cusparseScsrgemm2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDcsrgemm2_bufferSizeExt
    function cusparseDcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseDcsrgemm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrgemm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      real(c_double) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDcsrgemm2_bufferSizeExt_rank_0,&
      cusparseDcsrgemm2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCcsrgemm2_bufferSizeExt
    function cusparseCcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseCcsrgemm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrgemm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCcsrgemm2_bufferSizeExt_rank_0,&
      cusparseCcsrgemm2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZcsrgemm2_bufferSizeExt
    function cusparseZcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseZcsrgemm2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrgemm2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZcsrgemm2_bufferSizeExt_rank_0,&
      cusparseZcsrgemm2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseXcsrgemm2Nnz
    function cusparseXcsrgemm2Nnz_(handle,m,n,k,descrA,nnzA,csrRowPtrA,&
&csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,descrD,nnzD,csrRowPtrD,&
&csrColIndD,descrC,csrRowPtrC,nnzTotalDevHostPtr,myInfo,pBuffer) &
&bind(c, name="cusparseXcsrgemm2Nnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsrgemm2Nnz_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseXcsrgemm2Nnz_rank_0,&
      cusparseXcsrgemm2Nnz_rank_1
  end interface
  
  interface cusparseScsrgemm2
    function cusparseScsrgemm2_(handle,m,n,k,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,beta,&
&descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,csrValC,csrRowPtrC,&
&csrColIndC,myInfo,pBuffer) bind(c, name="cusparseScsrgemm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgemm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      real(c_float) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrValD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrgemm2_rank_0,&
      cusparseScsrgemm2_rank_1
  end interface
  
  interface cusparseDcsrgemm2
    function cusparseDcsrgemm2_(handle,m,n,k,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,beta,&
&descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,csrValC,csrRowPtrC,&
&csrColIndC,myInfo,pBuffer) bind(c, name="cusparseDcsrgemm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgemm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      real(c_double) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrValD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrgemm2_rank_0,&
      cusparseDcsrgemm2_rank_1
  end interface
  
  interface cusparseCcsrgemm2
    function cusparseCcsrgemm2_(handle,m,n,k,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,beta,&
&descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,csrValC,csrRowPtrC,&
&csrColIndC,myInfo,pBuffer) bind(c, name="cusparseCcsrgemm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgemm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrValD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrgemm2_rank_0,&
      cusparseCcsrgemm2_rank_1
  end interface
  
  interface cusparseZcsrgemm2
    function cusparseZcsrgemm2_(handle,m,n,k,alpha,descrA,nnzA,csrValA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,csrColIndB,beta,&
&descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,csrValC,csrRowPtrC,&
&csrColIndC,myInfo,pBuffer) bind(c, name="cusparseZcsrgemm2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgemm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: descrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: descrB
      integer(c_int),value :: nnzB
      type(c_ptr),value :: csrValB
      type(c_ptr),value :: csrRowPtrB
      type(c_ptr),value :: csrColIndB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: descrD
      integer(c_int),value :: nnzD
      type(c_ptr),value :: csrValD
      type(c_ptr),value :: csrRowPtrD
      type(c_ptr),value :: csrColIndD
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrgemm2_rank_0,&
      cusparseZcsrgemm2_rank_1
  end interface
  !>  \ingroup precond_module
  !>   \brief Incomplete LU factorization with 0 fill-ins and no pivoting using BSR storage
  !>   format
  !> 
  !>   \details
  !>   \p hipsparseXbsrilu02_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseXbsrilu02_analysis() or
  !>   hipsparseXbsrilu02() computation. The first zero pivot \f$j\f$ at \f$A_{j,j}\f$ is
  !>   stored in \p position, using same index base as the BSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note
  !>   If a zero pivot is found, \p position \f$=j\f$ means that either the diagonal block
  !>   \f$A_{j,j}\f$ is missing (structural zero) or the diagonal block \f$A_{j,j}\f$ is not
  !>   invertible (numerical zero).
  !> 
  !>   \note \p hipsparseXbsrilu02_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXbsrilu02_zeroPivot
    function cusparseXbsrilu02_zeroPivot_(handle,myInfo,position) &
&bind(c, name="cusparseXbsrilu02_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXbsrilu02_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseSbsrilu02_numericBoost
    function cusparseSbsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseSbsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      real(c_float) :: boost_val
    end function

  end interface
  
  interface cusparseDbsrilu02_numericBoost
    function cusparseDbsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseDbsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      real(c_double) :: boost_val
    end function

  end interface
  
  interface cusparseCbsrilu02_numericBoost
    function cusparseCbsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseCbsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      complex(c_float_complex) :: boost_val
    end function

  end interface
  
  interface cusparseZbsrilu02_numericBoost
    function cusparseZbsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseZbsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      complex(c_double_complex) :: boost_val
    end function

  end interface
  !> @{
  interface cusparseSbsrilu02_bufferSize
    function cusparseSbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseSbsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseSbsrilu02_bufferSize_rank_0,&
      cusparseSbsrilu02_bufferSize_rank_1
  end interface
  
  interface cusparseDbsrilu02_bufferSize
    function cusparseDbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseDbsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDbsrilu02_bufferSize_rank_0,&
      cusparseDbsrilu02_bufferSize_rank_1
  end interface
  
  interface cusparseCbsrilu02_bufferSize
    function cusparseCbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseCbsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCbsrilu02_bufferSize_rank_0,&
      cusparseCbsrilu02_bufferSize_rank_1
  end interface
  
  interface cusparseZbsrilu02_bufferSize
    function cusparseZbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseZbsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZbsrilu02_bufferSize_rank_0,&
      cusparseZbsrilu02_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseSbsrilu02_analysis
    function cusparseSbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseSbsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsrilu02_analysis_rank_0,&
      cusparseSbsrilu02_analysis_rank_1
  end interface
  
  interface cusparseDbsrilu02_analysis
    function cusparseDbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseDbsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsrilu02_analysis_rank_0,&
      cusparseDbsrilu02_analysis_rank_1
  end interface
  
  interface cusparseCbsrilu02_analysis
    function cusparseCbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseCbsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsrilu02_analysis_rank_0,&
      cusparseCbsrilu02_analysis_rank_1
  end interface
  
  interface cusparseZbsrilu02_analysis
    function cusparseZbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseZbsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsrilu02_analysis_rank_0,&
      cusparseZbsrilu02_analysis_rank_1
  end interface
  !> @{
  interface cusparseSbsrilu02
    function cusparseSbsrilu02_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseSbsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrilu02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA_valM
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsrilu02_rank_0,&
      cusparseSbsrilu02_rank_1
  end interface
  
  interface cusparseDbsrilu02
    function cusparseDbsrilu02_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseDbsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrilu02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA_valM
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsrilu02_rank_0,&
      cusparseDbsrilu02_rank_1
  end interface
  
  interface cusparseCbsrilu02
    function cusparseCbsrilu02_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseCbsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrilu02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA_valM
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsrilu02_rank_0,&
      cusparseCbsrilu02_rank_1
  end interface
  
  interface cusparseZbsrilu02
    function cusparseZbsrilu02_(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer) bind(c, name="cusparseZbsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrilu02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrSortedValA_valM
      type(c_ptr),value :: bsrSortedRowPtrA
      type(c_ptr),value :: bsrSortedColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsrilu02_rank_0,&
      cusparseZbsrilu02_rank_1
  end interface
  !>  \ingroup precond_module
  !>   \brief Incomplete LU factorization with 0 fill-ins and no pivoting using CSR
  !>   storage format
  !> 
  !>   \details
  !>   \p hipsparseXcsrilu02_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseXcsrilu02() computation.
  !>   The first zero pivot \f$j\f$ at \f$A_{j,j}\f$ is stored in \p position, using same
  !>   index base as the CSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note \p hipsparseXcsrilu02_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXcsrilu02_zeroPivot
    function cusparseXcsrilu02_zeroPivot_(handle,myInfo,position) &
&bind(c, name="cusparseXcsrilu02_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrilu02_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseScsrilu02_numericBoost
    function cusparseScsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseScsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      real(c_float) :: boost_val
    end function

  end interface
  
  interface cusparseDcsrilu02_numericBoost
    function cusparseDcsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseDcsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      real(c_double) :: boost_val
    end function

  end interface
  
  interface cusparseCcsrilu02_numericBoost
    function cusparseCcsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseCcsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      complex(c_float_complex) :: boost_val
    end function

  end interface
  
  interface cusparseZcsrilu02_numericBoost
    function cusparseZcsrilu02_numericBoost_(handle,myInfo,&
&enable_boost,tol,boost_val) bind(c, &
&name="cusparseZcsrilu02_numericBoost")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_numericBoost_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int),value :: enable_boost
      real(c_double) :: tol
      complex(c_double_complex) :: boost_val
    end function

  end interface
  !> @{
  interface cusparseScsrilu02_bufferSize
    function cusparseScsrilu02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseScsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseScsrilu02_bufferSize_rank_0,&
      cusparseScsrilu02_bufferSize_rank_1
  end interface
  
  interface cusparseDcsrilu02_bufferSize
    function cusparseDcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseDcsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDcsrilu02_bufferSize_rank_0,&
      cusparseDcsrilu02_bufferSize_rank_1
  end interface
  
  interface cusparseCcsrilu02_bufferSize
    function cusparseCcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseCcsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCcsrilu02_bufferSize_rank_0,&
      cusparseCcsrilu02_bufferSize_rank_1
  end interface
  
  interface cusparseZcsrilu02_bufferSize
    function cusparseZcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseZcsrilu02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZcsrilu02_bufferSize_rank_0,&
      cusparseZcsrilu02_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseScsrilu02_bufferSizeExt
    function cusparseScsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseScsrilu02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseScsrilu02_bufferSizeExt_rank_0,&
      cusparseScsrilu02_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDcsrilu02_bufferSizeExt
    function cusparseDcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseDcsrilu02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseDcsrilu02_bufferSizeExt_rank_0,&
      cusparseDcsrilu02_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCcsrilu02_bufferSizeExt
    function cusparseCcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseCcsrilu02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseCcsrilu02_bufferSizeExt_rank_0,&
      cusparseCcsrilu02_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZcsrilu02_bufferSizeExt
    function cusparseZcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseZcsrilu02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseZcsrilu02_bufferSizeExt_rank_0,&
      cusparseZcsrilu02_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseScsrilu02_analysis
    function cusparseScsrilu02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseScsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrilu02_analysis_rank_0,&
      cusparseScsrilu02_analysis_rank_1
  end interface
  
  interface cusparseDcsrilu02_analysis
    function cusparseDcsrilu02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseDcsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrilu02_analysis_rank_0,&
      cusparseDcsrilu02_analysis_rank_1
  end interface
  
  interface cusparseCcsrilu02_analysis
    function cusparseCcsrilu02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseCcsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrilu02_analysis_rank_0,&
      cusparseCcsrilu02_analysis_rank_1
  end interface
  
  interface cusparseZcsrilu02_analysis
    function cusparseZcsrilu02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseZcsrilu02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrilu02_analysis_rank_0,&
      cusparseZcsrilu02_analysis_rank_1
  end interface
  !> @{
  interface cusparseScsrilu02
    function cusparseScsrilu02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseScsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrilu02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsrilu02_rank_0,&
      cusparseScsrilu02_rank_1
  end interface
  
  interface cusparseDcsrilu02
    function cusparseDcsrilu02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseDcsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrilu02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsrilu02_rank_0,&
      cusparseDcsrilu02_rank_1
  end interface
  
  interface cusparseCcsrilu02
    function cusparseCcsrilu02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseCcsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrilu02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsrilu02_rank_0,&
      cusparseCcsrilu02_rank_1
  end interface
  
  interface cusparseZcsrilu02
    function cusparseZcsrilu02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseZcsrilu02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrilu02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsrilu02_rank_0,&
      cusparseZcsrilu02_rank_1
  end interface
  !>  \ingroup precond_module
  !>   \brief Incomplete Cholesky factorization with 0 fill-ins and no pivoting using BSR
  !>   storage format
  !> 
  !>   \details
  !>   \p hipsparseXbsric02_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseXbsric02_analysis() or
  !>   hipsparseXbsric02() computation. The first zero pivot \f$j\f$ at \f$A_{j,j}\f$ is
  !>   stored in \p position, using same index base as the BSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note
  !>   If a zero pivot is found, \p position=j means that either the diagonal block \p A(j,j)
  !>   is missing (structural zero) or the diagonal block \p A(j,j) is not positive definite
  !>   (numerical zero).
  !> 
  !>   \note \p hipsparseXbsric02_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXbsric02_zeroPivot
    function cusparseXbsric02_zeroPivot_(handle,myInfo,position) &
&bind(c, name="cusparseXbsric02_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXbsric02_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseSbsric02_bufferSize
    function cusparseSbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseSbsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseSbsric02_bufferSize_rank_0,&
      cusparseSbsric02_bufferSize_rank_1
  end interface
  
  interface cusparseDbsric02_bufferSize
    function cusparseDbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseDbsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDbsric02_bufferSize_rank_0,&
      cusparseDbsric02_bufferSize_rank_1
  end interface
  
  interface cusparseCbsric02_bufferSize
    function cusparseCbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseCbsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCbsric02_bufferSize_rank_0,&
      cusparseCbsric02_bufferSize_rank_1
  end interface
  
  interface cusparseZbsric02_bufferSize
    function cusparseZbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,pBufferSizeInBytes) &
&bind(c, name="cusparseZbsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZbsric02_bufferSize_rank_0,&
      cusparseZbsric02_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseSbsric02_analysis
    function cusparseSbsric02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseSbsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsric02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsric02_analysis_rank_0,&
      cusparseSbsric02_analysis_rank_1
  end interface
  
  interface cusparseDbsric02_analysis
    function cusparseDbsric02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseDbsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsric02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsric02_analysis_rank_0,&
      cusparseDbsric02_analysis_rank_1
  end interface
  
  interface cusparseCbsric02_analysis
    function cusparseCbsric02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseCbsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsric02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsric02_analysis_rank_0,&
      cusparseCbsric02_analysis_rank_1
  end interface
  
  interface cusparseZbsric02_analysis
    function cusparseZbsric02_analysis_(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseZbsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsric02_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsric02_analysis_rank_0,&
      cusparseZbsric02_analysis_rank_1
  end interface
  !> @{
  interface cusparseSbsric02
    function cusparseSbsric02_(handle,dirA,mb,nnzb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseSbsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsric02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSbsric02_rank_0,&
      cusparseSbsric02_rank_1
  end interface
  
  interface cusparseDbsric02
    function cusparseDbsric02_(handle,dirA,mb,nnzb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseDbsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsric02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDbsric02_rank_0,&
      cusparseDbsric02_rank_1
  end interface
  
  interface cusparseCbsric02
    function cusparseCbsric02_(handle,dirA,mb,nnzb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseCbsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsric02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCbsric02_rank_0,&
      cusparseCbsric02_rank_1
  end interface
  
  interface cusparseZbsric02
    function cusparseZbsric02_(handle,dirA,mb,nnzb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer) bind(c, &
&name="cusparseZbsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsric02_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZbsric02_rank_0,&
      cusparseZbsric02_rank_1
  end interface
  !>  \ingroup precond_module
  !>   \brief Incomplete Cholesky factorization with 0 fill-ins and no pivoting using CSR
  !>   storage format
  !> 
  !>   \details
  !>   \p hipsparseXcsric02_zeroPivot returns \ref HIPSPARSE_STATUS_ZERO_PIVOT, if either a
  !>   structural or numerical zero has been found during hipsparseXcsric02_analysis() or
  !>   hipsparseXcsric02() computation. The first zero pivot \f$j\f$ at \f$A_{j,j}\f$
  !>   is stored in \p position, using same index base as the CSR matrix.
  !> 
  !>   \p position can be in host or device memory. If no zero pivot has been found,
  !>   \p position is set to -1 and \ref HIPSPARSE_STATUS_SUCCESS is returned instead.
  !> 
  !>   \note \p hipsparseXcsric02_zeroPivot is a blocking function. It might influence
  !>   performance negatively.
  interface cusparseXcsric02_zeroPivot
    function cusparseXcsric02_zeroPivot_(handle,myInfo,position) &
&bind(c, name="cusparseXcsric02_zeroPivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsric02_zeroPivot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: myInfo
      integer(c_int) :: position
    end function

  end interface
  !> @{
  interface cusparseScsric02_bufferSize
    function cusparseScsric02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseScsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseScsric02_bufferSize_rank_0,&
      cusparseScsric02_bufferSize_rank_1
  end interface
  
  interface cusparseDcsric02_bufferSize
    function cusparseDcsric02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseDcsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDcsric02_bufferSize_rank_0,&
      cusparseDcsric02_bufferSize_rank_1
  end interface
  
  interface cusparseCcsric02_bufferSize
    function cusparseCcsric02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseCcsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCcsric02_bufferSize_rank_0,&
      cusparseCcsric02_bufferSize_rank_1
  end interface
  
  interface cusparseZcsric02_bufferSize
    function cusparseZcsric02_bufferSize_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes) bind(c, name="cusparseZcsric02_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_int) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZcsric02_bufferSize_rank_0,&
      cusparseZcsric02_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseScsric02_bufferSizeExt
    function cusparseScsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseScsric02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseScsric02_bufferSizeExt_rank_0,&
      cusparseScsric02_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDcsric02_bufferSizeExt
    function cusparseDcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseDcsric02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseDcsric02_bufferSizeExt_rank_0,&
      cusparseDcsric02_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCcsric02_bufferSizeExt
    function cusparseCcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseCcsric02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseCcsric02_bufferSizeExt_rank_0,&
      cusparseCcsric02_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZcsric02_bufferSizeExt
    function cusparseZcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize) &
&bind(c, name="cusparseZcsric02_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSize
    end function

    module procedure &
      cusparseZcsric02_bufferSizeExt_rank_0,&
      cusparseZcsric02_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseScsric02_analysis
    function cusparseScsric02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseScsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsric02_analysis_rank_0,&
      cusparseScsric02_analysis_rank_1
  end interface
  
  interface cusparseDcsric02_analysis
    function cusparseDcsric02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseDcsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsric02_analysis_rank_0,&
      cusparseDcsric02_analysis_rank_1
  end interface
  
  interface cusparseCcsric02_analysis
    function cusparseCcsric02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseCcsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsric02_analysis_rank_0,&
      cusparseCcsric02_analysis_rank_1
  end interface
  
  interface cusparseZcsric02_analysis
    function cusparseZcsric02_analysis_(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer) bind(c, name="cusparseZcsric02_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_analysis_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsric02_analysis_rank_0,&
      cusparseZcsric02_analysis_rank_1
  end interface
  !> @{
  interface cusparseScsric02
    function cusparseScsric02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseScsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsric02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsric02_rank_0,&
      cusparseScsric02_rank_1
  end interface
  
  interface cusparseDcsric02
    function cusparseDcsric02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseDcsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsric02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsric02_rank_0,&
      cusparseDcsric02_rank_1
  end interface
  
  interface cusparseCcsric02
    function cusparseCcsric02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseCcsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsric02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsric02_rank_0,&
      cusparseCcsric02_rank_1
  end interface
  
  interface cusparseZcsric02
    function cusparseZcsric02_(handle,m,nnz,descrA,csrSortedValA_valM,&
&csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,pBuffer) bind(c, &
&name="cusparseZcsric02")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsric02_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA_valM
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)),value :: policy
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsric02_rank_0,&
      cusparseZcsric02_rank_1
  end interface
  !> @{
  interface cusparseSgtsv2_bufferSizeExt
    function cusparseSgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,B,ldb,&
&pBufferSizeInBytes) bind(c, name="cusparseSgtsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseSgtsv2_bufferSizeExt_full_rank,&
      cusparseSgtsv2_bufferSizeExt_rank_0,&
      cusparseSgtsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDgtsv2_bufferSizeExt
    function cusparseDgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,B,db,&
&pBufferSizeInBytes) bind(c, name="cusparseDgtsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: db
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDgtsv2_bufferSizeExt_full_rank,&
      cusparseDgtsv2_bufferSizeExt_rank_0,&
      cusparseDgtsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCgtsv2_bufferSizeExt
    function cusparseCgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,B,ldb,&
&pBufferSizeInBytes) bind(c, name="cusparseCgtsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCgtsv2_bufferSizeExt_full_rank,&
      cusparseCgtsv2_bufferSizeExt_rank_0,&
      cusparseCgtsv2_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZgtsv2_bufferSizeExt
    function cusparseZgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,B,ldb,&
&pBufferSizeInBytes) bind(c, name="cusparseZgtsv2_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZgtsv2_bufferSizeExt_full_rank,&
      cusparseZgtsv2_bufferSizeExt_rank_0,&
      cusparseZgtsv2_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSgtsv2
    function cusparseSgtsv2_(handle,m,n,dl,d,du,B,ldb,pBuffer) bind(c, &
&name="cusparseSgtsv2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgtsv2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSgtsv2_full_rank,&
      cusparseSgtsv2_rank_0,&
      cusparseSgtsv2_rank_1
  end interface
  
  interface cusparseDgtsv2
    function cusparseDgtsv2_(handle,m,n,dl,d,du,B,ldb,pBuffer) bind(c, &
&name="cusparseDgtsv2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgtsv2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDgtsv2_full_rank,&
      cusparseDgtsv2_rank_0,&
      cusparseDgtsv2_rank_1
  end interface
  
  interface cusparseCgtsv2
    function cusparseCgtsv2_(handle,m,n,dl,d,du,B,ldb,pBuffer) bind(c, &
&name="cusparseCgtsv2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgtsv2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCgtsv2_full_rank,&
      cusparseCgtsv2_rank_0,&
      cusparseCgtsv2_rank_1
  end interface
  
  interface cusparseZgtsv2
    function cusparseZgtsv2_(handle,m,n,dl,d,du,B,ldb,pBuffer) bind(c, &
&name="cusparseZgtsv2")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgtsv2_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZgtsv2_full_rank,&
      cusparseZgtsv2_rank_0,&
      cusparseZgtsv2_rank_1
  end interface
  !> @{
  interface cusparseSgtsv2_nopivot_bufferSizeExt
    function cusparseSgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,&
&B,ldb,pBufferSizeInBytes) bind(c, &
&name="cusparseSgtsv2_nopivot_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_nopivot_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseSgtsv2_nopivot_bufferSizeExt_full_rank,&
      cusparseSgtsv2_nopivot_bufferSizeExt_rank_0,&
      cusparseSgtsv2_nopivot_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDgtsv2_nopivot_bufferSizeExt
    function cusparseDgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,&
&B,db,pBufferSizeInBytes) bind(c, &
&name="cusparseDgtsv2_nopivot_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_nopivot_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: db
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDgtsv2_nopivot_bufferSizeExt_full_rank,&
      cusparseDgtsv2_nopivot_bufferSizeExt_rank_0,&
      cusparseDgtsv2_nopivot_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCgtsv2_nopivot_bufferSizeExt
    function cusparseCgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,&
&B,ldb,pBufferSizeInBytes) bind(c, &
&name="cusparseCgtsv2_nopivot_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_nopivot_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCgtsv2_nopivot_bufferSizeExt_full_rank,&
      cusparseCgtsv2_nopivot_bufferSizeExt_rank_0,&
      cusparseCgtsv2_nopivot_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZgtsv2_nopivot_bufferSizeExt
    function cusparseZgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,&
&B,ldb,pBufferSizeInBytes) bind(c, &
&name="cusparseZgtsv2_nopivot_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_nopivot_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZgtsv2_nopivot_bufferSizeExt_full_rank,&
      cusparseZgtsv2_nopivot_bufferSizeExt_rank_0,&
      cusparseZgtsv2_nopivot_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSgtsv2_nopivot
    function cusparseSgtsv2_nopivot_(handle,m,n,dl,d,du,B,ldb,pBuffer) &
&bind(c, name="cusparseSgtsv2_nopivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgtsv2_nopivot_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSgtsv2_nopivot_full_rank,&
      cusparseSgtsv2_nopivot_rank_0,&
      cusparseSgtsv2_nopivot_rank_1
  end interface
  
  interface cusparseDgtsv2_nopivot
    function cusparseDgtsv2_nopivot_(handle,m,n,dl,d,du,B,ldb,pBuffer) &
&bind(c, name="cusparseDgtsv2_nopivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgtsv2_nopivot_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDgtsv2_nopivot_full_rank,&
      cusparseDgtsv2_nopivot_rank_0,&
      cusparseDgtsv2_nopivot_rank_1
  end interface
  
  interface cusparseCgtsv2_nopivot
    function cusparseCgtsv2_nopivot_(handle,m,n,dl,d,du,B,ldb,pBuffer) &
&bind(c, name="cusparseCgtsv2_nopivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgtsv2_nopivot_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCgtsv2_nopivot_full_rank,&
      cusparseCgtsv2_nopivot_rank_0,&
      cusparseCgtsv2_nopivot_rank_1
  end interface
  
  interface cusparseZgtsv2_nopivot
    function cusparseZgtsv2_nopivot_(handle,m,n,dl,d,du,B,ldb,pBuffer) &
&bind(c, name="cusparseZgtsv2_nopivot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgtsv2_nopivot_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZgtsv2_nopivot_full_rank,&
      cusparseZgtsv2_nopivot_rank_0,&
      cusparseZgtsv2_nopivot_rank_1
  end interface
  !> @{
  interface cusparseSgtsv2StridedBatch_bufferSizeExt
    function cusparseSgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,&
&du,x,batchCount,batchStride,pBufferSizeInBytes) bind(c, &
&name="cusparseSgtsv2StridedBatch_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2StridedBatch_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseSgtsv2StridedBatch_bufferSizeExt_rank_0,&
      cusparseSgtsv2StridedBatch_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDgtsv2StridedBatch_bufferSizeExt
    function cusparseDgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,&
&du,x,batchCount,batchStride,pBufferSizeInBytes) bind(c, &
&name="cusparseDgtsv2StridedBatch_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2StridedBatch_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDgtsv2StridedBatch_bufferSizeExt_rank_0,&
      cusparseDgtsv2StridedBatch_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCgtsv2StridedBatch_bufferSizeExt
    function cusparseCgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,&
&du,x,batchCount,batchStride,pBufferSizeInBytes) bind(c, &
&name="cusparseCgtsv2StridedBatch_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2StridedBatch_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCgtsv2StridedBatch_bufferSizeExt_rank_0,&
      cusparseCgtsv2StridedBatch_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZgtsv2StridedBatch_bufferSizeExt
    function cusparseZgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,&
&du,x,batchCount,batchStride,pBufferSizeInBytes) bind(c, &
&name="cusparseZgtsv2StridedBatch_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2StridedBatch_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZgtsv2StridedBatch_bufferSizeExt_rank_0,&
      cusparseZgtsv2StridedBatch_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSgtsv2StridedBatch
    function cusparseSgtsv2StridedBatch_(handle,m,dl,d,du,x,batchCount,&
&batchStride,pBuffer) bind(c, name="cusparseSgtsv2StridedBatch")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2StridedBatch_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseSgtsv2StridedBatch_rank_0,&
      cusparseSgtsv2StridedBatch_rank_1
  end interface
  
  interface cusparseDgtsv2StridedBatch
    function cusparseDgtsv2StridedBatch_(handle,m,dl,d,du,x,batchCount,&
&batchStride,pBuffer) bind(c, name="cusparseDgtsv2StridedBatch")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2StridedBatch_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDgtsv2StridedBatch_rank_0,&
      cusparseDgtsv2StridedBatch_rank_1
  end interface
  
  interface cusparseCgtsv2StridedBatch
    function cusparseCgtsv2StridedBatch_(handle,m,dl,d,du,x,batchCount,&
&batchStride,pBuffer) bind(c, name="cusparseCgtsv2StridedBatch")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2StridedBatch_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCgtsv2StridedBatch_rank_0,&
      cusparseCgtsv2StridedBatch_rank_1
  end interface
  
  interface cusparseZgtsv2StridedBatch
    function cusparseZgtsv2StridedBatch_(handle,m,dl,d,du,x,batchCount,&
&batchStride,pBuffer) bind(c, name="cusparseZgtsv2StridedBatch")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2StridedBatch_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: dl
      type(c_ptr),value :: d
      type(c_ptr),value :: du
      type(c_ptr),value :: x
      integer(c_int),value :: batchCount
      integer(c_int),value :: batchStride
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZgtsv2StridedBatch_rank_0,&
      cusparseZgtsv2StridedBatch_rank_1
  end interface
  !> @{
  interface cusparseSnnz
    function cusparseSnnz_(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr) bind(c, name="cusparseSnnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSnnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
    end function

    module procedure &
      cusparseSnnz_full_rank,&
      cusparseSnnz_rank_0,&
      cusparseSnnz_rank_1
  end interface
  
  interface cusparseDnnz
    function cusparseDnnz_(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr) bind(c, name="cusparseDnnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
    end function

    module procedure &
      cusparseDnnz_full_rank,&
      cusparseDnnz_rank_0,&
      cusparseDnnz_rank_1
  end interface
  
  interface cusparseCnnz
    function cusparseCnnz_(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr) bind(c, name="cusparseCnnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCnnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
    end function

    module procedure &
      cusparseCnnz_full_rank,&
      cusparseCnnz_rank_0,&
      cusparseCnnz_rank_1
  end interface
  
  interface cusparseZnnz
    function cusparseZnnz_(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr) bind(c, name="cusparseZnnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZnnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
    end function

    module procedure &
      cusparseZnnz_full_rank,&
      cusparseZnnz_rank_0,&
      cusparseZnnz_rank_1
  end interface
  !> @{
  interface cusparseSdense2csr
    function cusparseSdense2csr_(handle,m,n,descr,A,ld,nnz_per_rows,&
&csr_val,csr_row_ptr,csr_col_ind) bind(c, name="cusparseSdense2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSdense2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_rows
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
    end function

    module procedure &
      cusparseSdense2csr_full_rank,&
      cusparseSdense2csr_rank_0,&
      cusparseSdense2csr_rank_1
  end interface
  
  interface cusparseDdense2csr
    function cusparseDdense2csr_(handle,m,n,descr,A,ld,nnz_per_rows,&
&csr_val,csr_row_ptr,csr_col_ind) bind(c, name="cusparseDdense2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDdense2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_rows
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
    end function

    module procedure &
      cusparseDdense2csr_full_rank,&
      cusparseDdense2csr_rank_0,&
      cusparseDdense2csr_rank_1
  end interface
  
  interface cusparseCdense2csr
    function cusparseCdense2csr_(handle,m,n,descr,A,ld,nnz_per_rows,&
&csr_val,csr_row_ptr,csr_col_ind) bind(c, name="cusparseCdense2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdense2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_rows
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
    end function

    module procedure &
      cusparseCdense2csr_full_rank,&
      cusparseCdense2csr_rank_0,&
      cusparseCdense2csr_rank_1
  end interface
  
  interface cusparseZdense2csr
    function cusparseZdense2csr_(handle,m,n,descr,A,ld,nnz_per_rows,&
&csr_val,csr_row_ptr,csr_col_ind) bind(c, name="cusparseZdense2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdense2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_rows
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
    end function

    module procedure &
      cusparseZdense2csr_full_rank,&
      cusparseZdense2csr_rank_0,&
      cusparseZdense2csr_rank_1
  end interface
  !> @{
  interface cusparseSpruneDense2csr_bufferSize
    function cusparseSpruneDense2csr_bufferSize_(handle,m,n,A,lda,&
&threshold,descr,csrVal,csrRowPtr,csrColInd,bufferSize) bind(c, &
&name="cusparseSpruneDense2csr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csr_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneDense2csr_bufferSize_full_rank,&
      cusparseSpruneDense2csr_bufferSize_rank_0,&
      cusparseSpruneDense2csr_bufferSize_rank_1
  end interface
  
  interface cusparseDpruneDense2csr_bufferSize
    function cusparseDpruneDense2csr_bufferSize_(handle,m,n,A,lda,&
&threshold,descr,csrVal,csrRowPtr,csrColInd,bufferSize) bind(c, &
&name="cusparseDpruneDense2csr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneDense2csr_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneDense2csr_bufferSize_full_rank,&
      cusparseDpruneDense2csr_bufferSize_rank_0,&
      cusparseDpruneDense2csr_bufferSize_rank_1
  end interface
  
  interface cusparseSpruneDense2csr_bufferSizeExt
    function cusparseSpruneDense2csr_bufferSizeExt_(handle,m,n,A,lda,&
&threshold,descr,csrVal,csrRowPtr,csrColInd,bufferSize) bind(c, &
&name="cusparseSpruneDense2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneDense2csr_bufferSizeExt_full_rank,&
      cusparseSpruneDense2csr_bufferSizeExt_rank_0,&
      cusparseSpruneDense2csr_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDpruneDense2csr_bufferSizeExt
    function cusparseDpruneDense2csr_bufferSizeExt_(handle,m,n,A,lda,&
&threshold,descr,csrVal,csrRowPtr,csrColInd,bufferSize) bind(c, &
&name="cusparseDpruneDense2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneDense2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneDense2csr_bufferSizeExt_full_rank,&
      cusparseDpruneDense2csr_bufferSizeExt_rank_0,&
      cusparseDpruneDense2csr_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSpruneDense2csrNnz
    function cusparseSpruneDense2csrNnz_(handle,m,n,A,lda,threshold,&
&descr,csrRowPtr,nnzTotalDevHostPtr,buffer) bind(c, &
&name="cusparseSpruneDense2csrNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csrNnz_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrRowPtr
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneDense2csrNnz_full_rank,&
      cusparseSpruneDense2csrNnz_rank_0,&
      cusparseSpruneDense2csrNnz_rank_1
  end interface
  
  interface cusparseDpruneDense2csrNnz
    function cusparseDpruneDense2csrNnz_(handle,m,n,A,lda,threshold,&
&descr,csrRowPtr,nnzTotalDevHostPtr,buffer) bind(c, &
&name="cusparseDpruneDense2csrNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneDense2csrNnz_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrRowPtr
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneDense2csrNnz_full_rank,&
      cusparseDpruneDense2csrNnz_rank_0,&
      cusparseDpruneDense2csrNnz_rank_1
  end interface
  !> @{
  interface cusparseSpruneDense2csr
    function cusparseSpruneDense2csr_(handle,m,n,A,lda,threshold,descr,&
&csrVal,csrRowPtr,csrColInd,buffer) bind(c, &
&name="cusparseSpruneDense2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpruneDense2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneDense2csr_full_rank,&
      cusparseSpruneDense2csr_rank_0,&
      cusparseSpruneDense2csr_rank_1
  end interface
  
  interface cusparseDpruneDense2csr
    function cusparseDpruneDense2csr_(handle,m,n,A,lda,threshold,descr,&
&csrVal,csrRowPtr,csrColInd,buffer) bind(c, &
&name="cusparseDpruneDense2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDpruneDense2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double) :: threshold
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneDense2csr_full_rank,&
      cusparseDpruneDense2csr_rank_0,&
      cusparseDpruneDense2csr_rank_1
  end interface
  !> @{
  interface cusparseSpruneDense2csrByPercentage_bufferSize
    function cusparseSpruneDense2csrByPercentage_bufferSize_(handle,m,&
&n,A,lda,percentage,descr,csrVal,csrRowPtr,csrColInd,myInfo,&
&bufferSize) bind(c, &
&name="cusparseSpruneDense2csrByPercentage_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csrByPercentage_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneDense2csrByPercentage_bufferSize_full_rank,&
      cusparseSpruneDense2csrByPercentage_bufferSize_rank_0,&
      cusparseSpruneDense2csrByPercentage_bufferSize_rank_1
  end interface
  
  interface cusparseDpruneDense2csrByPercentage_bufferSize
    function cusparseDpruneDense2csrByPercentage_bufferSize_(handle,m,&
&n,A,lda,percentage,descr,csrVal,csrRowPtr,csrColInd,myInfo,&
&bufferSize) bind(c, &
&name="cusparseDpruneDense2csrByPercentage_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneDense2csrByPercentage_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneDense2csrByPercentage_bufferSize_full_rank,&
      cusparseDpruneDense2csrByPercentage_bufferSize_rank_0,&
      cusparseDpruneDense2csrByPercentage_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseSpruneDense2csrByPercentage_bufferSizeExt
    function cusparseSpruneDense2csrByPercentage_bufferSizeExt_(handle,&
&m,n,A,lda,percentage,descr,csrVal,csrRowPtr,csrColInd,myInfo,&
&bufferSize) bind(c, &
&name="cusparseSpruneDense2csrByPercentage_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csrByPercentage_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneDense2csrByPercentage_bufferSizeExt_full_rank,&
      cusparseSpruneDense2csrByPercentage_bufferSizeExt_rank_0,&
      cusparseSpruneDense2csrByPercentage_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDpruneDense2csrByPercentage_bufferSizeExt
    function cusparseDpruneDense2csrByPercentage_bufferSizeExt_(handle,&
&m,n,A,lda,percentage,descr,csrVal,csrRowPtr,csrColInd,myInfo,&
&bufferSize) bind(c, &
&name="cusparseDpruneDense2csrByPercentage_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneDense2csrByPercentage_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneDense2csrByPercentage_bufferSizeExt_full_rank,&
      cusparseDpruneDense2csrByPercentage_bufferSizeExt_rank_0,&
      cusparseDpruneDense2csrByPercentage_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSpruneDense2csrNnzByPercentage
    function cusparseSpruneDense2csrNnzByPercentage_(handle,m,n,A,lda,&
&percentage,descr,csrRowPtr,nnzTotalDevHostPtr,myInfo,buffer) bind(c, &
&name="cusparseSpruneDense2csrNnzByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csrNnzByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrRowPtr
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneDense2csrNnzByPercentage_full_rank,&
      cusparseSpruneDense2csrNnzByPercentage_rank_0,&
      cusparseSpruneDense2csrNnzByPercentage_rank_1
  end interface
  
  interface cusparseDpruneDense2csrNnzByPercentage
    function cusparseDpruneDense2csrNnzByPercentage_(handle,m,n,A,lda,&
&percentage,descr,csrRowPtr,nnzTotalDevHostPtr,myInfo,buffer) bind(c, &
&name="cusparseDpruneDense2csrNnzByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneDense2csrNnzByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrRowPtr
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneDense2csrNnzByPercentage_full_rank,&
      cusparseDpruneDense2csrNnzByPercentage_rank_0,&
      cusparseDpruneDense2csrNnzByPercentage_rank_1
  end interface
  !> @{
  interface cusparseSpruneDense2csrByPercentage
    function cusparseSpruneDense2csrByPercentage_(handle,m,n,A,lda,&
&percentage,descr,csrVal,csrRowPtr,csrColInd,myInfo,buffer) bind(c, &
&name="cusparseSpruneDense2csrByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csrByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneDense2csrByPercentage_full_rank,&
      cusparseSpruneDense2csrByPercentage_rank_0,&
      cusparseSpruneDense2csrByPercentage_rank_1
  end interface
  
  interface cusparseDpruneDense2csrByPercentage
    function cusparseDpruneDense2csrByPercentage_(handle,m,n,A,lda,&
&percentage,descr,csrVal,csrRowPtr,csrColInd,myInfo,buffer) bind(c, &
&name="cusparseDpruneDense2csrByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneDense2csrByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double),value :: percentage
      type(c_ptr),value :: descr
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneDense2csrByPercentage_full_rank,&
      cusparseDpruneDense2csrByPercentage_rank_0,&
      cusparseDpruneDense2csrByPercentage_rank_1
  end interface
  !> @{
  interface cusparseSdense2csc
    function cusparseSdense2csc_(handle,m,n,descr,A,ld,nnz_per_columns,&
&csc_val,csc_row_ind,csc_col_ptr) bind(c, name="cusparseSdense2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSdense2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_columns
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
    end function

    module procedure &
      cusparseSdense2csc_full_rank,&
      cusparseSdense2csc_rank_0,&
      cusparseSdense2csc_rank_1
  end interface
  
  interface cusparseDdense2csc
    function cusparseDdense2csc_(handle,m,n,descr,A,ld,nnz_per_columns,&
&csc_val,csc_row_ind,csc_col_ptr) bind(c, name="cusparseDdense2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDdense2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_columns
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
    end function

    module procedure &
      cusparseDdense2csc_full_rank,&
      cusparseDdense2csc_rank_0,&
      cusparseDdense2csc_rank_1
  end interface
  
  interface cusparseCdense2csc
    function cusparseCdense2csc_(handle,m,n,descr,A,ld,nnz_per_columns,&
&csc_val,csc_row_ind,csc_col_ptr) bind(c, name="cusparseCdense2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdense2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_columns
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
    end function

    module procedure &
      cusparseCdense2csc_full_rank,&
      cusparseCdense2csc_rank_0,&
      cusparseCdense2csc_rank_1
  end interface
  
  interface cusparseZdense2csc
    function cusparseZdense2csc_(handle,m,n,descr,A,ld,nnz_per_columns,&
&csc_val,csc_row_ind,csc_col_ptr) bind(c, name="cusparseZdense2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdense2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
      type(c_ptr),value :: nnz_per_columns
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
    end function

    module procedure &
      cusparseZdense2csc_full_rank,&
      cusparseZdense2csc_rank_0,&
      cusparseZdense2csc_rank_1
  end interface
  !> @{
  interface cusparseScsr2dense
    function cusparseScsr2dense_(handle,m,n,descr,csr_val,csr_row_ptr,&
&csr_col_ind,A,ld) bind(c, name="cusparseScsr2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsr2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseScsr2dense_full_rank,&
      cusparseScsr2dense_rank_0,&
      cusparseScsr2dense_rank_1
  end interface
  
  interface cusparseDcsr2dense
    function cusparseDcsr2dense_(handle,m,n,descr,csr_val,csr_row_ptr,&
&csr_col_ind,A,ld) bind(c, name="cusparseDcsr2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsr2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseDcsr2dense_full_rank,&
      cusparseDcsr2dense_rank_0,&
      cusparseDcsr2dense_rank_1
  end interface
  
  interface cusparseCcsr2dense
    function cusparseCcsr2dense_(handle,m,n,descr,csr_val,csr_row_ptr,&
&csr_col_ind,A,ld) bind(c, name="cusparseCcsr2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsr2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseCcsr2dense_full_rank,&
      cusparseCcsr2dense_rank_0,&
      cusparseCcsr2dense_rank_1
  end interface
  
  interface cusparseZcsr2dense
    function cusparseZcsr2dense_(handle,m,n,descr,csr_val,csr_row_ptr,&
&csr_col_ind,A,ld) bind(c, name="cusparseZcsr2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsr2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseZcsr2dense_full_rank,&
      cusparseZcsr2dense_rank_0,&
      cusparseZcsr2dense_rank_1
  end interface
  !> @{
  interface cusparseScsc2dense
    function cusparseScsc2dense_(handle,m,n,descr,csc_val,csc_row_ind,&
&csc_col_ptr,A,ld) bind(c, name="cusparseScsc2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsc2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseScsc2dense_full_rank,&
      cusparseScsc2dense_rank_0,&
      cusparseScsc2dense_rank_1
  end interface
  
  interface cusparseDcsc2dense
    function cusparseDcsc2dense_(handle,m,n,descr,csc_val,csc_row_ind,&
&csc_col_ptr,A,ld) bind(c, name="cusparseDcsc2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsc2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseDcsc2dense_full_rank,&
      cusparseDcsc2dense_rank_0,&
      cusparseDcsc2dense_rank_1
  end interface
  
  interface cusparseCcsc2dense
    function cusparseCcsc2dense_(handle,m,n,descr,csc_val,csc_row_ind,&
&csc_col_ptr,A,ld) bind(c, name="cusparseCcsc2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsc2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseCcsc2dense_full_rank,&
      cusparseCcsc2dense_rank_0,&
      cusparseCcsc2dense_rank_1
  end interface
  
  interface cusparseZcsc2dense
    function cusparseZcsc2dense_(handle,m,n,descr,csc_val,csc_row_ind,&
&csc_col_ptr,A,ld) bind(c, name="cusparseZcsc2dense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsc2dense_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descr
      type(c_ptr),value :: csc_val
      type(c_ptr),value :: csc_row_ind
      type(c_ptr),value :: csc_col_ptr
      type(c_ptr),value :: A
      integer(c_int),value :: ld
    end function

    module procedure &
      cusparseZcsc2dense_full_rank,&
      cusparseZcsc2dense_rank_0,&
      cusparseZcsc2dense_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief
  !>   This function computes the number of nonzero block columns per row and the total number of nonzero blocks in a sparse
  !>   BSR matrix given a sparse CSR matrix as input.
  !> 
  !>   \details
  !>   The routine does support asynchronous execution if the pointer mode is set to device.
  interface cusparseXcsr2bsrNnz
    function cusparseXcsr2bsrNnz_(handle,dirA,m,n,descrA,csrRowPtrA,&
&csrColIndA,blockDim,descrC,bsrRowPtrC,bsrNnzb) bind(c, &
&name="cusparseXcsr2bsrNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsr2bsrNnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrNnzb
    end function

    module procedure &
      cusparseXcsr2bsrNnz_rank_0,&
      cusparseXcsr2bsrNnz_rank_1
  end interface
  !> @{
  interface cusparseSnnz_compress
    function cusparseSnnz_compress_(handle,m,descrA,csrValA,csrRowPtrA,&
&nnzPerRow,nnzC,tol) bind(c, name="cusparseSnnz_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSnnz_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: nnzC
      real(c_float),value :: tol
    end function

    module procedure &
      cusparseSnnz_compress_rank_0,&
      cusparseSnnz_compress_rank_1
  end interface
  
  interface cusparseDnnz_compress
    function cusparseDnnz_compress_(handle,m,descrA,csrValA,csrRowPtrA,&
&nnzPerRow,nnzC,tol) bind(c, name="cusparseDnnz_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnnz_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: nnzC
      real(c_double),value :: tol
    end function

    module procedure &
      cusparseDnnz_compress_rank_0,&
      cusparseDnnz_compress_rank_1
  end interface
  
  interface cusparseCnnz_compress
    function cusparseCnnz_compress_(handle,m,descrA,csrValA,csrRowPtrA,&
&nnzPerRow,nnzC,tol) bind(c, name="cusparseCnnz_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCnnz_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: nnzC
      complex(c_float_complex),value :: tol
    end function

    module procedure &
      cusparseCnnz_compress_rank_0,&
      cusparseCnnz_compress_rank_1
  end interface
  
  interface cusparseZnnz_compress
    function cusparseZnnz_compress_(handle,m,descrA,csrValA,csrRowPtrA,&
&nnzPerRow,nnzC,tol) bind(c, name="cusparseZnnz_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZnnz_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: nnzC
      complex(c_double_complex),value :: tol
    end function

    module procedure &
      cusparseZnnz_compress_rank_0,&
      cusparseZnnz_compress_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Convert a sparse CSR matrix into a sparse COO matrix
  !> 
  !>   \details
  !>   \p hipsparseXcsr2coo converts the CSR array containing the row offsets, that point
  !>   to the start of every row, into a COO array of row indices.
  !> 
  !>   \note
  !>   It can also be used to convert a CSC array containing the column offsets into a COO
  !>   array of column indices.
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  interface cusparseXcsr2coo
    function cusparseXcsr2coo_(handle,csrRowPtr,nnz,m,cooRowInd,&
&idxBase) bind(c, name="cusparseXcsr2coo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsr2coo_
      type(c_ptr),value :: handle
      type(c_ptr),value :: csrRowPtr
      integer(c_int),value :: nnz
      integer(c_int),value :: m
      type(c_ptr),value :: cooRowInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseXcsr2coo_rank_0,&
      cusparseXcsr2coo_rank_1
  end interface
  
  interface cusparseScsr2csc
    function cusparseScsr2csc_(handle,m,n,nnz,csrSortedVal,&
&csrSortedRowPtr,csrSortedColInd,cscSortedVal,cscSortedRowInd,&
&cscSortedColPtr,copyValues,idxBase) bind(c, name="cusparseScsr2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsr2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrSortedVal
      type(c_ptr),value :: csrSortedRowPtr
      type(c_ptr),value :: csrSortedColInd
      type(c_ptr),value :: cscSortedVal
      type(c_ptr),value :: cscSortedRowInd
      type(c_ptr),value :: cscSortedColPtr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copyValues
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseScsr2csc_rank_0,&
      cusparseScsr2csc_rank_1
  end interface
  
  interface cusparseDcsr2csc
    function cusparseDcsr2csc_(handle,m,n,nnz,csrSortedVal,&
&csrSortedRowPtr,csrSortedColInd,cscSortedVal,cscSortedRowInd,&
&cscSortedColPtr,copyValues,idxBase) bind(c, name="cusparseDcsr2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsr2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrSortedVal
      type(c_ptr),value :: csrSortedRowPtr
      type(c_ptr),value :: csrSortedColInd
      type(c_ptr),value :: cscSortedVal
      type(c_ptr),value :: cscSortedRowInd
      type(c_ptr),value :: cscSortedColPtr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copyValues
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseDcsr2csc_rank_0,&
      cusparseDcsr2csc_rank_1
  end interface
  
  interface cusparseCcsr2csc
    function cusparseCcsr2csc_(handle,m,n,nnz,csrSortedVal,&
&csrSortedRowPtr,csrSortedColInd,cscSortedVal,cscSortedRowInd,&
&cscSortedColPtr,copyValues,idxBase) bind(c, name="cusparseCcsr2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsr2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrSortedVal
      type(c_ptr),value :: csrSortedRowPtr
      type(c_ptr),value :: csrSortedColInd
      type(c_ptr),value :: cscSortedVal
      type(c_ptr),value :: cscSortedRowInd
      type(c_ptr),value :: cscSortedColPtr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copyValues
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseCcsr2csc_rank_0,&
      cusparseCcsr2csc_rank_1
  end interface
  
  interface cusparseZcsr2csc
    function cusparseZcsr2csc_(handle,m,n,nnz,csrSortedVal,&
&csrSortedRowPtr,csrSortedColInd,cscSortedVal,cscSortedRowInd,&
&cscSortedColPtr,copyValues,idxBase) bind(c, name="cusparseZcsr2csc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsr2csc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrSortedVal
      type(c_ptr),value :: csrSortedRowPtr
      type(c_ptr),value :: csrSortedColInd
      type(c_ptr),value :: cscSortedVal
      type(c_ptr),value :: cscSortedRowInd
      type(c_ptr),value :: cscSortedColPtr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copyValues
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseZcsr2csc_rank_0,&
      cusparseZcsr2csc_rank_1
  end interface
  
  interface cusparseScsr2hyb
    function cusparseScsr2hyb_(handle,m,n,descrA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA,hybA,userEllWidth,partitionType) &
&bind(c, name="cusparseScsr2hyb")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsr2hyb_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: hybA
      integer(c_int),value :: userEllWidth
      integer(kind(CUSPARSE_HYB_PARTITION_AUTO)),value :: partitionType
    end function

    module procedure &
      cusparseScsr2hyb_rank_0,&
      cusparseScsr2hyb_rank_1
  end interface
  
  interface cusparseDcsr2hyb
    function cusparseDcsr2hyb_(handle,m,n,descrA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA,hybA,userEllWidth,partitionType) &
&bind(c, name="cusparseDcsr2hyb")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsr2hyb_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: hybA
      integer(c_int),value :: userEllWidth
      integer(kind(CUSPARSE_HYB_PARTITION_AUTO)),value :: partitionType
    end function

    module procedure &
      cusparseDcsr2hyb_rank_0,&
      cusparseDcsr2hyb_rank_1
  end interface
  
  interface cusparseCcsr2hyb
    function cusparseCcsr2hyb_(handle,m,n,descrA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA,hybA,userEllWidth,partitionType) &
&bind(c, name="cusparseCcsr2hyb")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsr2hyb_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: hybA
      integer(c_int),value :: userEllWidth
      integer(kind(CUSPARSE_HYB_PARTITION_AUTO)),value :: partitionType
    end function

    module procedure &
      cusparseCcsr2hyb_rank_0,&
      cusparseCcsr2hyb_rank_1
  end interface
  
  interface cusparseZcsr2hyb
    function cusparseZcsr2hyb_(handle,m,n,descrA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA,hybA,userEllWidth,partitionType) &
&bind(c, name="cusparseZcsr2hyb")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsr2hyb_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
      type(c_ptr),value :: hybA
      integer(c_int),value :: userEllWidth
      integer(kind(CUSPARSE_HYB_PARTITION_AUTO)),value :: partitionType
    end function

    module procedure &
      cusparseZcsr2hyb_rank_0,&
      cusparseZcsr2hyb_rank_1
  end interface
  !> @{
  interface cusparseSgebsr2gebsc_bufferSize
    function cusparseSgebsr2gebsc_bufferSize_(handle,mb,nb,nnzb,&
&bsr_val,bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseSgebsr2gebsc_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgebsr2gebsc_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

  end interface
  
  interface cusparseDgebsr2gebsc_bufferSize
    function cusparseDgebsr2gebsc_bufferSize_(handle,mb,nb,nnzb,&
&bsr_val,bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseDgebsr2gebsc_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgebsr2gebsc_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

  end interface
  
  interface cusparseCgebsr2gebsc_bufferSize
    function cusparseCgebsr2gebsc_bufferSize_(handle,mb,nb,nnzb,&
&bsr_val,bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseCgebsr2gebsc_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgebsr2gebsc_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

  end interface
  
  interface cusparseZgebsr2gebsc_bufferSize
    function cusparseZgebsr2gebsc_bufferSize_(handle,mb,nb,nnzb,&
&bsr_val,bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseZgebsr2gebsc_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgebsr2gebsc_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

  end interface
  !> @{
  interface cusparseSgebsr2gebsc
    function cusparseSgebsr2gebsc_(handle,mb,nb,nnzb,bsr_val,&
&bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,bsc_val,&
&bsc_row_ind,bsc_col_ptr,copy_values,idx_base,temp_buffer) bind(c, &
&name="cusparseSgebsr2gebsc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgebsr2gebsc_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: bsc_val
      type(c_ptr),value :: bsc_row_ind
      type(c_ptr),value :: bsc_col_ptr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copy_values
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idx_base
      type(c_ptr),value :: temp_buffer
    end function

  end interface
  
  interface cusparseDgebsr2gebsc
    function cusparseDgebsr2gebsc_(handle,mb,nb,nnzb,bsr_val,&
&bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,bsc_val,&
&bsc_row_ind,bsc_col_ptr,copy_values,idx_base,temp_buffer) bind(c, &
&name="cusparseDgebsr2gebsc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgebsr2gebsc_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: bsc_val
      type(c_ptr),value :: bsc_row_ind
      type(c_ptr),value :: bsc_col_ptr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copy_values
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idx_base
      type(c_ptr),value :: temp_buffer
    end function

  end interface
  
  interface cusparseCgebsr2gebsc
    function cusparseCgebsr2gebsc_(handle,mb,nb,nnzb,bsr_val,&
&bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,bsc_val,&
&bsc_row_ind,bsc_col_ptr,copy_values,idx_base,temp_buffer) bind(c, &
&name="cusparseCgebsr2gebsc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgebsr2gebsc_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: bsc_val
      type(c_ptr),value :: bsc_row_ind
      type(c_ptr),value :: bsc_col_ptr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copy_values
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idx_base
      type(c_ptr),value :: temp_buffer
    end function

  end interface
  
  interface cusparseZgebsr2gebsc
    function cusparseZgebsr2gebsc_(handle,mb,nb,nnzb,bsr_val,&
&bsr_row_ptr,bsr_col_ind,row_block_dim,col_block_dim,bsc_val,&
&bsc_row_ind,bsc_col_ptr,copy_values,idx_base,temp_buffer) bind(c, &
&name="cusparseZgebsr2gebsc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgebsr2gebsc_
      type(c_ptr),value :: handle
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: bsc_val
      type(c_ptr),value :: bsc_row_ind
      type(c_ptr),value :: bsc_col_ptr
      integer(kind(CUSPARSE_ACTION_SYMBOLIC)),value :: copy_values
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idx_base
      type(c_ptr),value :: temp_buffer
    end function

  end interface
  !> @{
  interface cusparseScsr2gebsr_bufferSize
    function cusparseScsr2gebsr_bufferSize_(handle,dir,m,n,csr_descr,&
&csr_val,csr_row_ptr,csr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseScsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

    module procedure &
      cusparseScsr2gebsr_bufferSize_rank_0,&
      cusparseScsr2gebsr_bufferSize_rank_1
  end interface
  
  interface cusparseDcsr2gebsr_bufferSize
    function cusparseDcsr2gebsr_bufferSize_(handle,dir,m,n,csr_descr,&
&csr_val,csr_row_ptr,csr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseDcsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

    module procedure &
      cusparseDcsr2gebsr_bufferSize_rank_0,&
      cusparseDcsr2gebsr_bufferSize_rank_1
  end interface
  
  interface cusparseCcsr2gebsr_bufferSize
    function cusparseCcsr2gebsr_bufferSize_(handle,dir,m,n,csr_descr,&
&csr_val,csr_row_ptr,csr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseCcsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

    module procedure &
      cusparseCcsr2gebsr_bufferSize_rank_0,&
      cusparseCcsr2gebsr_bufferSize_rank_1
  end interface
  
  interface cusparseZcsr2gebsr_bufferSize
    function cusparseZcsr2gebsr_bufferSize_(handle,dir,m,n,csr_descr,&
&csr_val,csr_row_ptr,csr_col_ind,row_block_dim,col_block_dim,&
&p_buffer_size) bind(c, name="cusparseZcsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer_size
    end function

    module procedure &
      cusparseZcsr2gebsr_bufferSize_rank_0,&
      cusparseZcsr2gebsr_bufferSize_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief
  !>   This function computes the number of nonzero block columns per row and the total number of nonzero blocks in a sparse
  !>   GEneral BSR matrix given a sparse CSR matrix as input.
  !>
  interface cusparseXcsr2gebsrNnz
    function cusparseXcsr2gebsrNnz_(handle,dir,m,n,csr_descr,&
&csr_row_ptr,csr_col_ind,bsr_descr,bsr_row_ptr,row_block_dim,&
&col_block_dim,bsr_nnz_devhost,p_buffer) bind(c, &
&name="cusparseXcsr2gebsrNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsr2gebsrNnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: bsr_descr
      type(c_ptr),value :: bsr_row_ptr
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: bsr_nnz_devhost
      type(c_ptr),value :: p_buffer
    end function

    module procedure &
      cusparseXcsr2gebsrNnz_rank_0,&
      cusparseXcsr2gebsrNnz_rank_1
  end interface
  !> @{
  interface cusparseScsr2gebsr
    function cusparseScsr2gebsr_(handle,dir,m,n,csr_descr,csr_val,&
&csr_row_ptr,csr_col_ind,bsr_descr,bsr_val,bsr_row_ptr,bsr_col_ind,&
&row_block_dim,col_block_dim,p_buffer) bind(c, &
&name="cusparseScsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: bsr_descr
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer
    end function

    module procedure &
      cusparseScsr2gebsr_rank_0,&
      cusparseScsr2gebsr_rank_1
  end interface
  
  interface cusparseDcsr2gebsr
    function cusparseDcsr2gebsr_(handle,dir,m,n,csr_descr,csr_val,&
&csr_row_ptr,csr_col_ind,bsr_descr,bsr_val,bsr_row_ptr,bsr_col_ind,&
&row_block_dim,col_block_dim,p_buffer) bind(c, &
&name="cusparseDcsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: bsr_descr
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer
    end function

    module procedure &
      cusparseDcsr2gebsr_rank_0,&
      cusparseDcsr2gebsr_rank_1
  end interface
  
  interface cusparseCcsr2gebsr
    function cusparseCcsr2gebsr_(handle,dir,m,n,csr_descr,csr_val,&
&csr_row_ptr,csr_col_ind,bsr_descr,bsr_val,bsr_row_ptr,bsr_col_ind,&
&row_block_dim,col_block_dim,p_buffer) bind(c, &
&name="cusparseCcsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: bsr_descr
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer
    end function

    module procedure &
      cusparseCcsr2gebsr_rank_0,&
      cusparseCcsr2gebsr_rank_1
  end interface
  
  interface cusparseZcsr2gebsr
    function cusparseZcsr2gebsr_(handle,dir,m,n,csr_descr,csr_val,&
&csr_row_ptr,csr_col_ind,bsr_descr,bsr_val,bsr_row_ptr,bsr_col_ind,&
&row_block_dim,col_block_dim,p_buffer) bind(c, &
&name="cusparseZcsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dir
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: csr_descr
      type(c_ptr),value :: csr_val
      type(c_ptr),value :: csr_row_ptr
      type(c_ptr),value :: csr_col_ind
      type(c_ptr),value :: bsr_descr
      type(c_ptr),value :: bsr_val
      type(c_ptr),value :: bsr_row_ptr
      type(c_ptr),value :: bsr_col_ind
      integer(c_int),value :: row_block_dim
      integer(c_int),value :: col_block_dim
      type(c_ptr),value :: p_buffer
    end function

    module procedure &
      cusparseZcsr2gebsr_rank_0,&
      cusparseZcsr2gebsr_rank_1
  end interface
  !> @{
  interface cusparseScsr2bsr
    function cusparseScsr2bsr_(handle,dirA,m,n,descrA,csrValA,&
&csrRowPtrA,csrColIndA,blockDim,descrC,bsrValC,bsrRowPtrC,bsrColIndC) &
&bind(c, name="cusparseScsr2bsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsr2bsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
    end function

    module procedure &
      cusparseScsr2bsr_rank_0,&
      cusparseScsr2bsr_rank_1
  end interface
  
  interface cusparseDcsr2bsr
    function cusparseDcsr2bsr_(handle,dirA,m,n,descrA,csrValA,&
&csrRowPtrA,csrColIndA,blockDim,descrC,bsrValC,bsrRowPtrC,bsrColIndC) &
&bind(c, name="cusparseDcsr2bsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsr2bsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
    end function

    module procedure &
      cusparseDcsr2bsr_rank_0,&
      cusparseDcsr2bsr_rank_1
  end interface
  
  interface cusparseCcsr2bsr
    function cusparseCcsr2bsr_(handle,dirA,m,n,descrA,csrValA,&
&csrRowPtrA,csrColIndA,blockDim,descrC,bsrValC,bsrRowPtrC,bsrColIndC) &
&bind(c, name="cusparseCcsr2bsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsr2bsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
    end function

    module procedure &
      cusparseCcsr2bsr_rank_0,&
      cusparseCcsr2bsr_rank_1
  end interface
  
  interface cusparseZcsr2bsr
    function cusparseZcsr2bsr_(handle,dirA,m,n,descrA,csrValA,&
&csrRowPtrA,csrColIndA,blockDim,descrC,bsrValC,bsrRowPtrC,bsrColIndC) &
&bind(c, name="cusparseZcsr2bsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsr2bsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
    end function

    module procedure &
      cusparseZcsr2bsr_rank_0,&
      cusparseZcsr2bsr_rank_1
  end interface
  !> @{
  interface cusparseSbsr2csr
    function cusparseSbsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,descrC,csrValC,csrRowPtrC,csrColIndC) &
&bind(c, name="cusparseSbsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseSbsr2csr_rank_0,&
      cusparseSbsr2csr_rank_1
  end interface
  
  interface cusparseDbsr2csr
    function cusparseDbsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,descrC,csrValC,csrRowPtrC,csrColIndC) &
&bind(c, name="cusparseDbsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseDbsr2csr_rank_0,&
      cusparseDbsr2csr_rank_1
  end interface
  
  interface cusparseCbsr2csr
    function cusparseCbsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,descrC,csrValC,csrRowPtrC,csrColIndC) &
&bind(c, name="cusparseCbsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseCbsr2csr_rank_0,&
      cusparseCbsr2csr_rank_1
  end interface
  
  interface cusparseZbsr2csr
    function cusparseZbsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,blockDim,descrC,csrValC,csrRowPtrC,csrColIndC) &
&bind(c, name="cusparseZbsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: blockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseZbsr2csr_rank_0,&
      cusparseZbsr2csr_rank_1
  end interface
  !> @{
  interface cusparseSgebsr2csr
    function cusparseSgebsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,rowBlockDim,colBlockDim,descrC,csrValC,&
&csrRowPtrC,csrColIndC) bind(c, name="cusparseSgebsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgebsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDim
      integer(c_int),value :: colBlockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseSgebsr2csr_rank_0,&
      cusparseSgebsr2csr_rank_1
  end interface
  
  interface cusparseDgebsr2csr
    function cusparseDgebsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,rowBlockDim,colBlockDim,descrC,csrValC,&
&csrRowPtrC,csrColIndC) bind(c, name="cusparseDgebsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgebsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDim
      integer(c_int),value :: colBlockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseDgebsr2csr_rank_0,&
      cusparseDgebsr2csr_rank_1
  end interface
  
  interface cusparseCgebsr2csr
    function cusparseCgebsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,rowBlockDim,colBlockDim,descrC,csrValC,&
&csrRowPtrC,csrColIndC) bind(c, name="cusparseCgebsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgebsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDim
      integer(c_int),value :: colBlockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseCgebsr2csr_rank_0,&
      cusparseCgebsr2csr_rank_1
  end interface
  
  interface cusparseZgebsr2csr
    function cusparseZgebsr2csr_(handle,dirA,mb,nb,descrA,bsrValA,&
&bsrRowPtrA,bsrColIndA,rowBlockDim,colBlockDim,descrC,csrValC,&
&csrRowPtrC,csrColIndC) bind(c, name="cusparseZgebsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgebsr2csr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDim
      integer(c_int),value :: colBlockDim
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
    end function

    module procedure &
      cusparseZgebsr2csr_rank_0,&
      cusparseZgebsr2csr_rank_1
  end interface
  !> @{
  interface cusparseScsr2csr_compress
    function cusparseScsr2csr_compress_(handle,m,n,descrA,csrValA,&
&csrColIndA,csrRowPtrA,nnzA,nnzPerRow,csrValC,csrColIndC,csrRowPtrC,&
&tol) bind(c, name="cusparseScsr2csr_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsr2csr_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: csrRowPtrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: csrRowPtrC
      real(c_float),value :: tol
    end function

    module procedure &
      cusparseScsr2csr_compress_rank_0,&
      cusparseScsr2csr_compress_rank_1
  end interface
  
  interface cusparseDcsr2csr_compress
    function cusparseDcsr2csr_compress_(handle,m,n,descrA,csrValA,&
&csrColIndA,csrRowPtrA,nnzA,nnzPerRow,csrValC,csrColIndC,csrRowPtrC,&
&tol) bind(c, name="cusparseDcsr2csr_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsr2csr_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: csrRowPtrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: csrRowPtrC
      real(c_double),value :: tol
    end function

    module procedure &
      cusparseDcsr2csr_compress_rank_0,&
      cusparseDcsr2csr_compress_rank_1
  end interface
  
  interface cusparseCcsr2csr_compress
    function cusparseCcsr2csr_compress_(handle,m,n,descrA,csrValA,&
&csrColIndA,csrRowPtrA,nnzA,nnzPerRow,csrValC,csrColIndC,csrRowPtrC,&
&tol) bind(c, name="cusparseCcsr2csr_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsr2csr_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: csrRowPtrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: csrRowPtrC
      complex(c_float_complex),value :: tol
    end function

    module procedure &
      cusparseCcsr2csr_compress_rank_0,&
      cusparseCcsr2csr_compress_rank_1
  end interface
  
  interface cusparseZcsr2csr_compress
    function cusparseZcsr2csr_compress_(handle,m,n,descrA,csrValA,&
&csrColIndA,csrRowPtrA,nnzA,nnzPerRow,csrValC,csrColIndC,csrRowPtrC,&
&tol) bind(c, name="cusparseZcsr2csr_compress")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsr2csr_compress_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: csrRowPtrA
      integer(c_int),value :: nnzA
      type(c_ptr),value :: nnzPerRow
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: csrRowPtrC
      complex(c_double_complex),value :: tol
    end function

    module procedure &
      cusparseZcsr2csr_compress_rank_0,&
      cusparseZcsr2csr_compress_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csr_bufferSize
    function cusparseSpruneCsr2csr_bufferSize_(handle,m,n,nnzA,descrA,&
&csrValA,csrRowPtrA,csrColIndA,threshold,descrC,csrValC,csrRowPtrC,&
&csrColIndC,bufferSize) bind(c, &
&name="cusparseSpruneCsr2csr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneCsr2csr_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneCsr2csr_bufferSize_rank_0,&
      cusparseSpruneCsr2csr_bufferSize_rank_1
  end interface
  
  interface cusparseDpruneCsr2csr_bufferSize
    function cusparseDpruneCsr2csr_bufferSize_(handle,m,n,nnzA,descrA,&
&csrValA,csrRowPtrA,csrColIndA,threshold,descrC,csrValC,csrRowPtrC,&
&csrColIndC,bufferSize) bind(c, &
&name="cusparseDpruneCsr2csr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneCsr2csr_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneCsr2csr_bufferSize_rank_0,&
      cusparseDpruneCsr2csr_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csr_bufferSizeExt
    function cusparseSpruneCsr2csr_bufferSizeExt_(handle,m,n,nnzA,&
&descrA,csrValA,csrRowPtrA,csrColIndA,threshold,descrC,csrValC,&
&csrRowPtrC,csrColIndC,bufferSize) bind(c, &
&name="cusparseSpruneCsr2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneCsr2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneCsr2csr_bufferSizeExt_rank_0,&
      cusparseSpruneCsr2csr_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDpruneCsr2csr_bufferSizeExt
    function cusparseDpruneCsr2csr_bufferSizeExt_(handle,m,n,nnzA,&
&descrA,csrValA,csrRowPtrA,csrColIndA,threshold,descrC,csrValC,&
&csrRowPtrC,csrColIndC,bufferSize) bind(c, &
&name="cusparseDpruneCsr2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneCsr2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneCsr2csr_bufferSizeExt_rank_0,&
      cusparseDpruneCsr2csr_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csrNnz
    function cusparseSpruneCsr2csrNnz_(handle,m,n,nnzA,descrA,csrValA,&
&csrRowPtrA,csrColIndA,threshold,descrC,csrRowPtrC,nnzTotalDevHostPtr,&
&buffer) bind(c, name="cusparseSpruneCsr2csrNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneCsr2csrNnz_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneCsr2csrNnz_rank_0,&
      cusparseSpruneCsr2csrNnz_rank_1
  end interface
  
  interface cusparseDpruneCsr2csrNnz
    function cusparseDpruneCsr2csrNnz_(handle,m,n,nnzA,descrA,csrValA,&
&csrRowPtrA,csrColIndA,threshold,descrC,csrRowPtrC,nnzTotalDevHostPtr,&
&buffer) bind(c, name="cusparseDpruneCsr2csrNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneCsr2csrNnz_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneCsr2csrNnz_rank_0,&
      cusparseDpruneCsr2csrNnz_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csr
    function cusparseSpruneCsr2csr_(handle,m,n,nnzA,descrA,csrValA,&
&csrRowPtrA,csrColIndA,threshold,descrC,csrValC,csrRowPtrC,csrColIndC,&
&buffer) bind(c, name="cusparseSpruneCsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpruneCsr2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneCsr2csr_rank_0,&
      cusparseSpruneCsr2csr_rank_1
  end interface
  
  interface cusparseDpruneCsr2csr
    function cusparseDpruneCsr2csr_(handle,m,n,nnzA,descrA,csrValA,&
&csrRowPtrA,csrColIndA,threshold,descrC,csrValC,csrRowPtrC,csrColIndC,&
&buffer) bind(c, name="cusparseDpruneCsr2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDpruneCsr2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double) :: threshold
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneCsr2csr_rank_0,&
      cusparseDpruneCsr2csr_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csrByPercentage_bufferSize
    function cusparseSpruneCsr2csrByPercentage_bufferSize_(handle,m,n,&
&nnzA,descrA,csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrValC,&
&csrRowPtrC,csrColIndC,myInfo,bufferSize) bind(c, &
&name="cusparseSpruneCsr2csrByPercentage_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneCsr2csrByPercentage_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneCsr2csrByPercentage_bufferSize_rank_0,&
      cusparseSpruneCsr2csrByPercentage_bufferSize_rank_1
  end interface
  
  interface cusparseDpruneCsr2csrByPercentage_bufferSize
    function cusparseDpruneCsr2csrByPercentage_bufferSize_(handle,m,n,&
&nnzA,descrA,csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrValC,&
&csrRowPtrC,csrColIndC,myInfo,bufferSize) bind(c, &
&name="cusparseDpruneCsr2csrByPercentage_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneCsr2csrByPercentage_bufferSize_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneCsr2csrByPercentage_bufferSize_rank_0,&
      cusparseDpruneCsr2csrByPercentage_bufferSize_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csrByPercentage_bufferSizeExt
    function cusparseSpruneCsr2csrByPercentage_bufferSizeExt_(handle,m,&
&n,nnzA,descrA,csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrValC,&
&csrRowPtrC,csrColIndC,myInfo,bufferSize) bind(c, &
&name="cusparseSpruneCsr2csrByPercentage_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneCsr2csrByPercentage_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSpruneCsr2csrByPercentage_bufferSizeExt_rank_0,&
      cusparseSpruneCsr2csrByPercentage_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDpruneCsr2csrByPercentage_bufferSizeExt
    function cusparseDpruneCsr2csrByPercentage_bufferSizeExt_(handle,m,&
&n,nnzA,descrA,csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrValC,&
&csrRowPtrC,csrColIndC,myInfo,bufferSize) bind(c, &
&name="cusparseDpruneCsr2csrByPercentage_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneCsr2csrByPercentage_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDpruneCsr2csrByPercentage_bufferSizeExt_rank_0,&
      cusparseDpruneCsr2csrByPercentage_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csrNnzByPercentage
    function cusparseSpruneCsr2csrNnzByPercentage_(handle,m,n,nnzA,&
&descrA,csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrRowPtrC,&
&nnzTotalDevHostPtr,myInfo,buffer) bind(c, &
&name="cusparseSpruneCsr2csrNnzByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneCsr2csrNnzByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneCsr2csrNnzByPercentage_rank_0,&
      cusparseSpruneCsr2csrNnzByPercentage_rank_1
  end interface
  
  interface cusparseDpruneCsr2csrNnzByPercentage
    function cusparseDpruneCsr2csrNnzByPercentage_(handle,m,n,nnzA,&
&descrA,csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrRowPtrC,&
&nnzTotalDevHostPtr,myInfo,buffer) bind(c, &
&name="cusparseDpruneCsr2csrNnzByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneCsr2csrNnzByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneCsr2csrNnzByPercentage_rank_0,&
      cusparseDpruneCsr2csrNnzByPercentage_rank_1
  end interface
  !> @{
  interface cusparseSpruneCsr2csrByPercentage
    function cusparseSpruneCsr2csrByPercentage_(handle,m,n,nnzA,descrA,&
&csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrValC,csrRowPtrC,&
&csrColIndC,myInfo,buffer) bind(c, &
&name="cusparseSpruneCsr2csrByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneCsr2csrByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_float),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSpruneCsr2csrByPercentage_rank_0,&
      cusparseSpruneCsr2csrByPercentage_rank_1
  end interface
  
  interface cusparseDpruneCsr2csrByPercentage
    function cusparseDpruneCsr2csrByPercentage_(handle,m,n,nnzA,descrA,&
&csrValA,csrRowPtrA,csrColIndA,percentage,descrC,csrValC,csrRowPtrC,&
&csrColIndC,myInfo,buffer) bind(c, &
&name="cusparseDpruneCsr2csrByPercentage")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDpruneCsr2csrByPercentage_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnzA
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      real(c_double),value :: percentage
      type(c_ptr),value :: descrC
      type(c_ptr),value :: csrValC
      type(c_ptr),value :: csrRowPtrC
      type(c_ptr),value :: csrColIndC
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDpruneCsr2csrByPercentage_rank_0,&
      cusparseDpruneCsr2csrByPercentage_rank_1
  end interface
  
  interface cusparseShyb2csr
    function cusparseShyb2csr_(handle,descrA,hybA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA) bind(c, name="cusparseShyb2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseShyb2csr_
      type(c_ptr),value :: handle
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
    end function

    module procedure &
      cusparseShyb2csr_rank_0,&
      cusparseShyb2csr_rank_1
  end interface
  
  interface cusparseDhyb2csr
    function cusparseDhyb2csr_(handle,descrA,hybA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA) bind(c, name="cusparseDhyb2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDhyb2csr_
      type(c_ptr),value :: handle
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
    end function

    module procedure &
      cusparseDhyb2csr_rank_0,&
      cusparseDhyb2csr_rank_1
  end interface
  
  interface cusparseChyb2csr
    function cusparseChyb2csr_(handle,descrA,hybA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA) bind(c, name="cusparseChyb2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseChyb2csr_
      type(c_ptr),value :: handle
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
    end function

    module procedure &
      cusparseChyb2csr_rank_0,&
      cusparseChyb2csr_rank_1
  end interface
  
  interface cusparseZhyb2csr
    function cusparseZhyb2csr_(handle,descrA,hybA,csrSortedValA,&
&csrSortedRowPtrA,csrSortedColIndA) bind(c, name="cusparseZhyb2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZhyb2csr_
      type(c_ptr),value :: handle
      type(c_ptr),value :: descrA
      type(c_ptr),value :: hybA
      type(c_ptr),value :: csrSortedValA
      type(c_ptr),value :: csrSortedRowPtrA
      type(c_ptr),value :: csrSortedColIndA
    end function

    module procedure &
      cusparseZhyb2csr_rank_0,&
      cusparseZhyb2csr_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Convert a sparse COO matrix into a sparse CSR matrix
  !> 
  !>   \details
  !>   \p hipsparseXcoo2csr converts the COO array containing the row indices into a
  !>   CSR array of row offsets, that point to the start of every row.
  !>   It is assumed that the COO row index array is sorted.
  !> 
  !>   \note It can also be used, to convert a COO array containing the column indices into
  !>   a CSC array of column offsets, that point to the start of every column. Then, it is
  !>   assumed that the COO column index array is sorted, instead.
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  interface cusparseXcoo2csr
    function cusparseXcoo2csr_(handle,cooRowInd,nnz,m,csrRowPtr,&
&idxBase) bind(c, name="cusparseXcoo2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcoo2csr_
      type(c_ptr),value :: handle
      type(c_ptr),value :: cooRowInd
      integer(c_int),value :: nnz
      integer(c_int),value :: m
      type(c_ptr),value :: csrRowPtr
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
    end function

    module procedure &
      cusparseXcoo2csr_rank_0,&
      cusparseXcoo2csr_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Create the identity map
  !> 
  !>   \details
  !>   \p hipsparseCreateIdentityPermutation stores the identity map in \p p, such that
  !>   \f$p = 0:1:(n-1)\f$.
  !> 
  !>   \code{.c}
  !>       for(i = 0; i < n; ++i)
  !>       {
  !>           p[i] = i;
  !>       }
  !>   \endcode
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  interface cusparseCreateIdentityPermutation
    function cusparseCreateIdentityPermutation_(handle,n,p) bind(c, &
&name="cusparseCreateIdentityPermutation")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateIdentityPermutation_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: p
    end function

    module procedure &
      cusparseCreateIdentityPermutation_rank_0,&
      cusparseCreateIdentityPermutation_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Sort a sparse CSR matrix
  !> 
  !>   \details
  !>   \p hipsparseXcsrsort_bufferSizeExt returns the size of the temporary storage buffer
  !>   required by hipsparseXcsrsort(). The temporary storage buffer must be allocated by
  !>   the user.
  interface cusparseXcsrsort_bufferSizeExt
    function cusparseXcsrsort_bufferSizeExt_(handle,m,n,nnz,csrRowPtr,&
&csrColInd,pBufferSizeInBytes) bind(c, &
&name="cusparseXcsrsort_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrsort_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseXcsrsort_bufferSizeExt_rank_0,&
      cusparseXcsrsort_bufferSizeExt_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Sort a sparse CSR matrix
  !> 
  !>   \details
  !>   \p hipsparseXcsrsort sorts a matrix in CSR format. The sorted permutation vector
  !>   \p perm can be used to obtain sorted \p csr_val array. In this case, \p perm must be
  !>   initialized as the identity permutation, see hipsparseCreateIdentityPermutation().
  !> 
  !>   \p hipsparseXcsrsort requires extra temporary storage buffer that has to be allocated by
  !>   the user. Storage buffer size can be determined by hipsparseXcsrsort_bufferSizeExt().
  !> 
  !>   \note
  !>   \p perm can be \p NULL if a sorted permutation vector is not required.
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  interface cusparseXcsrsort
    function cusparseXcsrsort_(handle,m,n,nnz,descrA,csrRowPtr,&
&csrColInd,P,pBuffer) bind(c, name="cusparseXcsrsort")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcsrsort_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: P
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseXcsrsort_full_rank,&
      cusparseXcsrsort_rank_0,&
      cusparseXcsrsort_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Sort a sparse CSC matrix
  !> 
  !>   \details
  !>   \p hipsparseXcscsort_bufferSizeExt returns the size of the temporary storage buffer
  !>   required by hipsparseXcscsort(). The temporary storage buffer must be allocated by
  !>   the user.
  interface cusparseXcscsort_bufferSizeExt
    function cusparseXcscsort_bufferSizeExt_(handle,m,n,nnz,cscColPtr,&
&cscRowInd,pBufferSizeInBytes) bind(c, &
&name="cusparseXcscsort_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcscsort_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: cscColPtr
      type(c_ptr),value :: cscRowInd
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseXcscsort_bufferSizeExt_rank_0,&
      cusparseXcscsort_bufferSizeExt_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Sort a sparse CSC matrix
  !> 
  !>   \details
  !>   \p hipsparseXcscsort sorts a matrix in CSC format. The sorted permutation vector
  !>   \p perm can be used to obtain sorted \p csc_val array. In this case, \p perm must be
  !>   initialized as the identity permutation, see hipsparseCreateIdentityPermutation().
  !> 
  !>   \p hipsparseXcscsort requires extra temporary storage buffer that has to be allocated by
  !>   the user. Storage buffer size can be determined by hipsparseXcscsort_bufferSizeExt().
  !> 
  !>   \note
  !>   \p perm can be \p NULL if a sorted permutation vector is not required.
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  interface cusparseXcscsort
    function cusparseXcscsort_(handle,m,n,nnz,descrA,cscColPtr,&
&cscRowInd,P,pBuffer) bind(c, name="cusparseXcscsort")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcscsort_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: cscColPtr
      type(c_ptr),value :: cscRowInd
      type(c_ptr),value :: P
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseXcscsort_full_rank,&
      cusparseXcscsort_rank_0,&
      cusparseXcscsort_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Sort a sparse COO matrix
  !> 
  !>   \details
  !>   \p hipsparseXcoosort_bufferSizeExt returns the size of the temporary storage buffer
  !>   required by hipsparseXcoosort(). The temporary storage buffer must be allocated by
  !>   the user.
  interface cusparseXcoosort_bufferSizeExt
    function cusparseXcoosort_bufferSizeExt_(handle,m,n,nnz,cooRows,&
&cooCols,pBufferSizeInBytes) bind(c, &
&name="cusparseXcoosort_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcoosort_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: cooRows
      type(c_ptr),value :: cooCols
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseXcoosort_bufferSizeExt_rank_0,&
      cusparseXcoosort_bufferSizeExt_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Sort a sparse COO matrix by row
  !> 
  !>   \details
  !>   \p hipsparseXcoosortByRow sorts a matrix in COO format by row. The sorted
  !>   permutation vector \p perm can be used to obtain sorted \p coo_val array. In this
  !>   case, \p perm must be initialized as the identity permutation, see
  !>   hipsparseCreateIdentityPermutation().
  !> 
  !>   \p hipsparseXcoosortByRow requires extra temporary storage buffer that has to be
  !>   allocated by the user. Storage buffer size can be determined by
  !>   hipsparseXcoosort_bufferSizeExt().
  !> 
  !>   \note
  !>   \p perm can be \p NULL if a sorted permutation vector is not required.
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  interface cusparseXcoosortByRow
    function cusparseXcoosortByRow_(handle,m,n,nnz,cooRows,cooCols,P,&
&pBuffer) bind(c, name="cusparseXcoosortByRow")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXcoosortByRow_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: cooRows
      type(c_ptr),value :: cooCols
      type(c_ptr),value :: P
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseXcoosortByRow_full_rank,&
      cusparseXcoosortByRow_rank_0,&
      cusparseXcoosortByRow_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief Sort a sparse COO matrix by column
  !> 
  !>   \details
  !>   \p hipsparseXcoosortByColumn sorts a matrix in COO format by column. The sorted
  !>   permutation vector \p perm can be used to obtain sorted \p coo_val array. In this
  !>   case, \p perm must be initialized as the identity permutation, see
  !>   hipsparseCreateIdentityPermutation().
  !> 
  !>   \p hipsparseXcoosortByColumn requires extra temporary storage buffer that has to be
  !>   allocated by the user. Storage buffer size can be determined by
  !>   hipsparseXcoosort_bufferSizeExt().
  !> 
  !>   \note
  !>   \p perm can be \p NULL if a sorted permutation vector is not required.
  !> 
  !>   \note
  !>   This function is non blocking and executed asynchronously with respect to the host.
  !>   It may return before the actual computation has finished.
  interface cusparseXcoosortByColumn
    function cusparseXcoosortByColumn_(handle,m,n,nnz,cooRows,cooCols,&
&P,pBuffer) bind(c, name="cusparseXcoosortByColumn")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcoosortByColumn_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: cooRows
      type(c_ptr),value :: cooCols
      type(c_ptr),value :: P
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseXcoosortByColumn_full_rank,&
      cusparseXcoosortByColumn_rank_0,&
      cusparseXcoosortByColumn_rank_1
  end interface
  !> @{
  interface cusparseSgebsr2gebsr_bufferSize
    function cusparseSgebsr2gebsr_bufferSize_(handle,dirA,mb,nb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,&
&rowBlockDimC,colBlockDimC,bufferSize) bind(c, &
&name="cusparseSgebsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgebsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSgebsr2gebsr_bufferSize_rank_0,&
      cusparseSgebsr2gebsr_bufferSize_rank_1
  end interface
  
  interface cusparseDgebsr2gebsr_bufferSize
    function cusparseDgebsr2gebsr_bufferSize_(handle,dirA,mb,nb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,&
&rowBlockDimC,colBlockDimC,bufferSize) bind(c, &
&name="cusparseDgebsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgebsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDgebsr2gebsr_bufferSize_rank_0,&
      cusparseDgebsr2gebsr_bufferSize_rank_1
  end interface
  
  interface cusparseCgebsr2gebsr_bufferSize
    function cusparseCgebsr2gebsr_bufferSize_(handle,dirA,mb,nb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,&
&rowBlockDimC,colBlockDimC,bufferSize) bind(c, &
&name="cusparseCgebsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgebsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseCgebsr2gebsr_bufferSize_rank_0,&
      cusparseCgebsr2gebsr_bufferSize_rank_1
  end interface
  
  interface cusparseZgebsr2gebsr_bufferSize
    function cusparseZgebsr2gebsr_bufferSize_(handle,dirA,mb,nb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,&
&rowBlockDimC,colBlockDimC,bufferSize) bind(c, &
&name="cusparseZgebsr2gebsr_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgebsr2gebsr_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseZgebsr2gebsr_bufferSize_rank_0,&
      cusparseZgebsr2gebsr_bufferSize_rank_1
  end interface
  !>  \ingroup conv_module
  !>   \brief This function is used when converting a general BSR sparse matrix \p A to another general BSR sparse matrix \p C.
  !>   Specifically, this function determines the number of non-zero blocks that will exist in \p C (stored using either a host
  !>   or device pointer), and computes the row pointer array for \p C.
  !> 
  !>   \details
  !>   The routine does support asynchronous execution.
  interface cusparseXgebsr2gebsrNnz
    function cusparseXgebsr2gebsrNnz_(handle,dirA,mb,nb,nnzb,descrA,&
&bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,descrC,bsrRowPtrC,&
&rowBlockDimC,colBlockDimC,nnzTotalDevHostPtr,buffer) bind(c, &
&name="cusparseXgebsr2gebsrNnz")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseXgebsr2gebsrNnz_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrRowPtrC
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseXgebsr2gebsrNnz_rank_0,&
      cusparseXgebsr2gebsrNnz_rank_1
  end interface
  !> @{
  interface cusparseSgebsr2gebsr
    function cusparseSgebsr2gebsr_(handle,dirA,mb,nb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,descrC,&
&bsrValC,bsrRowPtrC,bsrColIndC,rowBlockDimC,colBlockDimC,buffer) &
&bind(c, name="cusparseSgebsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgebsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseSgebsr2gebsr_rank_0,&
      cusparseSgebsr2gebsr_rank_1
  end interface
  
  interface cusparseDgebsr2gebsr
    function cusparseDgebsr2gebsr_(handle,dirA,mb,nb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,descrC,&
&bsrValC,bsrRowPtrC,bsrColIndC,rowBlockDimC,colBlockDimC,buffer) &
&bind(c, name="cusparseDgebsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgebsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseDgebsr2gebsr_rank_0,&
      cusparseDgebsr2gebsr_rank_1
  end interface
  
  interface cusparseCgebsr2gebsr
    function cusparseCgebsr2gebsr_(handle,dirA,mb,nb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,descrC,&
&bsrValC,bsrRowPtrC,bsrColIndC,rowBlockDimC,colBlockDimC,buffer) &
&bind(c, name="cusparseCgebsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgebsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseCgebsr2gebsr_rank_0,&
      cusparseCgebsr2gebsr_rank_1
  end interface
  
  interface cusparseZgebsr2gebsr
    function cusparseZgebsr2gebsr_(handle,dirA,mb,nb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,rowBlockDimA,colBlockDimA,descrC,&
&bsrValC,bsrRowPtrC,bsrColIndC,rowBlockDimC,colBlockDimC,buffer) &
&bind(c, name="cusparseZgebsr2gebsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgebsr2gebsr_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)),value :: dirA
      integer(c_int),value :: mb
      integer(c_int),value :: nb
      integer(c_int),value :: nnzb
      type(c_ptr),value :: descrA
      type(c_ptr),value :: bsrValA
      type(c_ptr),value :: bsrRowPtrA
      type(c_ptr),value :: bsrColIndA
      integer(c_int),value :: rowBlockDimA
      integer(c_int),value :: colBlockDimA
      type(c_ptr),value :: descrC
      type(c_ptr),value :: bsrValC
      type(c_ptr),value :: bsrRowPtrC
      type(c_ptr),value :: bsrColIndC
      integer(c_int),value :: rowBlockDimC
      integer(c_int),value :: colBlockDimC
      type(c_ptr),value :: buffer
    end function

    module procedure &
      cusparseZgebsr2gebsr_rank_0,&
      cusparseZgebsr2gebsr_rank_1
  end interface
  !> @{
  interface cusparseScsru2csr_bufferSizeExt
    function cusparseScsru2csr_bufferSizeExt_(handle,m,n,nnz,csrVal,&
&csrRowPtr,csrColInd,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseScsru2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsru2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseScsru2csr_bufferSizeExt_rank_0,&
      cusparseScsru2csr_bufferSizeExt_rank_1
  end interface
  
  interface cusparseDcsru2csr_bufferSizeExt
    function cusparseDcsru2csr_bufferSizeExt_(handle,m,n,nnz,csrVal,&
&csrRowPtr,csrColInd,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseDcsru2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsru2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseDcsru2csr_bufferSizeExt_rank_0,&
      cusparseDcsru2csr_bufferSizeExt_rank_1
  end interface
  
  interface cusparseCcsru2csr_bufferSizeExt
    function cusparseCcsru2csr_bufferSizeExt_(handle,m,n,nnz,csrVal,&
&csrRowPtr,csrColInd,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseCcsru2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsru2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseCcsru2csr_bufferSizeExt_rank_0,&
      cusparseCcsru2csr_bufferSizeExt_rank_1
  end interface
  
  interface cusparseZcsru2csr_bufferSizeExt
    function cusparseZcsru2csr_bufferSizeExt_(handle,m,n,nnz,csrVal,&
&csrRowPtr,csrColInd,myInfo,pBufferSizeInBytes) bind(c, &
&name="cusparseZcsru2csr_bufferSizeExt")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsru2csr_bufferSizeExt_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
    end function

    module procedure &
      cusparseZcsru2csr_bufferSizeExt_rank_0,&
      cusparseZcsru2csr_bufferSizeExt_rank_1
  end interface
  !> @{
  interface cusparseScsru2csr
    function cusparseScsru2csr_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseScsru2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsru2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsru2csr_rank_0,&
      cusparseScsru2csr_rank_1
  end interface
  
  interface cusparseDcsru2csr
    function cusparseDcsru2csr_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseDcsru2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsru2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsru2csr_rank_0,&
      cusparseDcsru2csr_rank_1
  end interface
  
  interface cusparseCcsru2csr
    function cusparseCcsru2csr_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseCcsru2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsru2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsru2csr_rank_0,&
      cusparseCcsru2csr_rank_1
  end interface
  
  interface cusparseZcsru2csr
    function cusparseZcsru2csr_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseZcsru2csr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsru2csr_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsru2csr_rank_0,&
      cusparseZcsru2csr_rank_1
  end interface
  !> @{
  interface cusparseScsr2csru
    function cusparseScsr2csru_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseScsr2csru")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsr2csru_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseScsr2csru_rank_0,&
      cusparseScsr2csru_rank_1
  end interface
  
  interface cusparseDcsr2csru
    function cusparseDcsr2csru_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseDcsr2csru")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsr2csru_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseDcsr2csru_rank_0,&
      cusparseDcsr2csru_rank_1
  end interface
  
  interface cusparseCcsr2csru
    function cusparseCcsr2csru_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseCcsr2csru")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsr2csru_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseCcsr2csru_rank_0,&
      cusparseCcsr2csru_rank_1
  end interface
  
  interface cusparseZcsr2csru
    function cusparseZcsr2csru_(handle,m,n,nnz,descrA,csrVal,csrRowPtr,&
&csrColInd,myInfo,pBuffer) bind(c, name="cusparseZcsr2csru")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsr2csru_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrVal
      type(c_ptr),value :: csrRowPtr
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: myInfo
      type(c_ptr),value :: pBuffer
    end function

    module procedure &
      cusparseZcsr2csru_rank_0,&
      cusparseZcsr2csru_rank_1
  end interface
  !> @{
  interface cusparseScsrcolor
    function cusparseScsrcolor_(handle,m,nnz,descrA,csrValA,csrRowPtrA,&
&csrColIndA,fractionToColor,ncolors,coloring,reordering,myInfo) bind(c,&
& name="cusparseScsrcolor")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrcolor_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: fractionToColor
      integer(c_int) :: ncolors
      integer(c_int) :: coloring
      integer(c_int) :: reordering
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cusparseScsrcolor_rank_0,&
      cusparseScsrcolor_rank_1
  end interface
  
  interface cusparseDcsrcolor
    function cusparseDcsrcolor_(handle,m,nnz,descrA,csrValA,csrRowPtrA,&
&csrColIndA,fractionToColor,ncolors,coloring,reordering,myInfo) bind(c,&
& name="cusparseDcsrcolor")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrcolor_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: fractionToColor
      integer(c_int) :: ncolors
      integer(c_int) :: coloring
      integer(c_int) :: reordering
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cusparseDcsrcolor_rank_0,&
      cusparseDcsrcolor_rank_1
  end interface
  
  interface cusparseCcsrcolor
    function cusparseCcsrcolor_(handle,m,nnz,descrA,csrValA,csrRowPtrA,&
&csrColIndA,fractionToColor,ncolors,coloring,reordering,myInfo) bind(c,&
& name="cusparseCcsrcolor")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrcolor_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: fractionToColor
      integer(c_int) :: ncolors
      integer(c_int) :: coloring
      integer(c_int) :: reordering
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cusparseCcsrcolor_rank_0,&
      cusparseCcsrcolor_rank_1
  end interface
  
  interface cusparseZcsrcolor
    function cusparseZcsrcolor_(handle,m,nnz,descrA,csrValA,csrRowPtrA,&
&csrColIndA,fractionToColor,ncolors,coloring,reordering,myInfo) bind(c,&
& name="cusparseZcsrcolor")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrcolor_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: nnz
      type(c_ptr),value :: descrA
      type(c_ptr),value :: csrValA
      type(c_ptr),value :: csrRowPtrA
      type(c_ptr),value :: csrColIndA
      type(c_ptr),value :: fractionToColor
      integer(c_int) :: ncolors
      integer(c_int) :: coloring
      integer(c_int) :: reordering
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cusparseZcsrcolor_rank_0,&
      cusparseZcsrcolor_rank_1
  end interface
  
  interface cusparseCreateSpVec
    function cusparseCreateSpVec_(spVecDescr,mySize,nnz,indices,values,&
&idxType,idxBase,valueType) bind(c, name="cusparseCreateSpVec")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateSpVec_
      type(c_ptr) :: spVecDescr
      integer(c_int64_t),value :: mySize
      integer(c_int64_t),value :: nnz
      type(c_ptr),value :: indices
      type(c_ptr),value :: values
      integer(kind(CUSPARSE_INDEX_16U)),value :: idxType
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      integer(kind(CU_R_16F)),value :: valueType
    end function

  end interface
  
  interface cusparseDestroySpVec
    function cusparseDestroySpVec_(spVecDescr) bind(c, &
&name="cusparseDestroySpVec")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDestroySpVec_
      type(c_ptr),value :: spVecDescr
    end function

  end interface
  
  interface cusparseSpVecGet
    function cusparseSpVecGet_(spVecDescr,mySize,nnz,indices,values,&
&idxType,idxBase,valueType) bind(c, name="cusparseSpVecGet")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpVecGet_
      type(c_ptr),value :: spVecDescr
      type(c_ptr),value :: mySize
      type(c_ptr),value :: nnz
      type(c_ptr) :: indices
      type(c_ptr) :: values
      type(c_ptr),value :: idxType
      type(c_ptr),value :: idxBase
      type(c_ptr),value :: valueType
    end function

  end interface
  
  interface cusparseSpVecGetIndexBase
    function cusparseSpVecGetIndexBase_(spVecDescr,idxBase) bind(c, &
&name="cusparseSpVecGetIndexBase")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpVecGetIndexBase_
      type(c_ptr),value :: spVecDescr
      type(c_ptr),value :: idxBase
    end function

  end interface
  
  interface cusparseSpVecGetValues
    function cusparseSpVecGetValues_(spVecDescr,values) bind(c, &
&name="cusparseSpVecGetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpVecGetValues_
      type(c_ptr),value :: spVecDescr
      type(c_ptr) :: values
    end function

  end interface
  
  interface cusparseSpVecSetValues
    function cusparseSpVecSetValues_(spVecDescr,values) bind(c, &
&name="cusparseSpVecSetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpVecSetValues_
      type(c_ptr),value :: spVecDescr
      type(c_ptr),value :: values
    end function

  end interface
  
  interface cusparseCreateCoo
    function cusparseCreateCoo_(spMatDescr,rows,cols,nnz,cooRowInd,&
&cooColInd,cooValues,cooIdxType,idxBase,valueType) bind(c, &
&name="cusparseCreateCoo")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateCoo_
      type(c_ptr) :: spMatDescr
      integer(c_int64_t),value :: rows
      integer(c_int64_t),value :: cols
      integer(c_int64_t),value :: nnz
      type(c_ptr),value :: cooRowInd
      type(c_ptr),value :: cooColInd
      type(c_ptr),value :: cooValues
      integer(kind(CUSPARSE_INDEX_16U)),value :: cooIdxType
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      integer(kind(CU_R_16F)),value :: valueType
    end function

  end interface
  
  interface cusparseCreateCooAoS
    function cusparseCreateCooAoS_(spMatDescr,rows,cols,nnz,cooInd,&
&cooValues,cooIdxType,idxBase,valueType) bind(c, &
&name="cusparseCreateCooAoS")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateCooAoS_
      type(c_ptr) :: spMatDescr
      integer(c_int64_t),value :: rows
      integer(c_int64_t),value :: cols
      integer(c_int64_t),value :: nnz
      type(c_ptr),value :: cooInd
      type(c_ptr),value :: cooValues
      integer(kind(CUSPARSE_INDEX_16U)),value :: cooIdxType
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      integer(kind(CU_R_16F)),value :: valueType
    end function

  end interface
  
  interface cusparseCreateCsr
    function cusparseCreateCsr_(spMatDescr,rows,cols,nnz,csrRowOffsets,&
&csrColInd,csrValues,csrRowOffsetsType,csrColIndType,idxBase,&
&valueType) bind(c, name="cusparseCreateCsr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateCsr_
      type(c_ptr) :: spMatDescr
      integer(c_int64_t),value :: rows
      integer(c_int64_t),value :: cols
      integer(c_int64_t),value :: nnz
      type(c_ptr),value :: csrRowOffsets
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: csrValues
      integer(kind(CUSPARSE_INDEX_16U)),value :: csrRowOffsetsType
      integer(kind(CUSPARSE_INDEX_16U)),value :: csrColIndType
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      integer(kind(CU_R_16F)),value :: valueType
    end function

  end interface
  
  interface cusparseCreateCsc
    function cusparseCreateCsc_(spMatDescr,rows,cols,nnz,cscColOffsets,&
&cscRowInd,cscValues,cscColOffsetsType,cscRowIndType,idxBase,&
&valueType) bind(c, name="cusparseCreateCsc")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateCsc_
      type(c_ptr) :: spMatDescr
      integer(c_int64_t),value :: rows
      integer(c_int64_t),value :: cols
      integer(c_int64_t),value :: nnz
      type(c_ptr),value :: cscColOffsets
      type(c_ptr),value :: cscRowInd
      type(c_ptr),value :: cscValues
      integer(kind(CUSPARSE_INDEX_16U)),value :: cscColOffsetsType
      integer(kind(CUSPARSE_INDEX_16U)),value :: cscRowIndType
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      integer(kind(CU_R_16F)),value :: valueType
    end function

  end interface
  
  interface cusparseCreateBlockedEll
    function cusparseCreateBlockedEll_(spMatDescr,rows,cols,&
&ellBlockSize,ellCols,ellColInd,ellValue,ellIdxType,idxBase,valueType) &
&bind(c, name="cusparseCreateBlockedEll")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCreateBlockedEll_
      type(c_ptr) :: spMatDescr
      integer(c_int64_t),value :: rows
      integer(c_int64_t),value :: cols
      integer(c_int64_t),value :: ellBlockSize
      integer(c_int64_t),value :: ellCols
      type(c_ptr),value :: ellColInd
      type(c_ptr),value :: ellValue
      integer(kind(CUSPARSE_INDEX_16U)),value :: ellIdxType
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)),value :: idxBase
      integer(kind(CU_R_16F)),value :: valueType
    end function

  end interface
  
  interface cusparseDestroySpMat
    function cusparseDestroySpMat_(spMatDescr) bind(c, &
&name="cusparseDestroySpMat")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDestroySpMat_
      type(c_ptr),value :: spMatDescr
    end function

  end interface
  
  interface cusparseCooGet
    function cusparseCooGet_(spMatDescr,rows,cols,nnz,cooRowInd,&
&cooColInd,cooValues,idxType,idxBase,valueType) bind(c, &
&name="cusparseCooGet")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCooGet_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: rows
      type(c_ptr),value :: cols
      type(c_ptr),value :: nnz
      type(c_ptr) :: cooRowInd
      type(c_ptr) :: cooColInd
      type(c_ptr) :: cooValues
      type(c_ptr),value :: idxType
      type(c_ptr),value :: idxBase
      type(c_ptr),value :: valueType
    end function

  end interface
  
  interface cusparseCooAoSGet
    function cusparseCooAoSGet_(spMatDescr,rows,cols,nnz,cooInd,&
&cooValues,idxType,idxBase,valueType) bind(c, &
&name="cusparseCooAoSGet")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCooAoSGet_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: rows
      type(c_ptr),value :: cols
      type(c_ptr),value :: nnz
      type(c_ptr) :: cooInd
      type(c_ptr) :: cooValues
      type(c_ptr),value :: idxType
      type(c_ptr),value :: idxBase
      type(c_ptr),value :: valueType
    end function

  end interface
  
  interface cusparseCsrGet
    function cusparseCsrGet_(spMatDescr,rows,cols,nnz,csrRowOffsets,&
&csrColInd,csrValues,csrRowOffsetsType,csrColIndType,idxBase,&
&valueType) bind(c, name="cusparseCsrGet")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCsrGet_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: rows
      type(c_ptr),value :: cols
      type(c_ptr),value :: nnz
      type(c_ptr) :: csrRowOffsets
      type(c_ptr) :: csrColInd
      type(c_ptr) :: csrValues
      type(c_ptr),value :: csrRowOffsetsType
      type(c_ptr),value :: csrColIndType
      type(c_ptr),value :: idxBase
      type(c_ptr),value :: valueType
    end function

  end interface
  
  interface cusparseBlockedEllGet
    function cusparseBlockedEllGet_(spMatDescr,rows,cols,ellBlockSize,&
&ellCols,ellColInd,ellValue,ellIdxType,idxBase,valueType) bind(c, &
&name="cusparseBlockedEllGet")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseBlockedEllGet_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: rows
      type(c_ptr),value :: cols
      type(c_ptr),value :: ellBlockSize
      type(c_ptr),value :: ellCols
      type(c_ptr) :: ellColInd
      type(c_ptr) :: ellValue
      type(c_ptr),value :: ellIdxType
      type(c_ptr),value :: idxBase
      type(c_ptr),value :: valueType
    end function

  end interface
  
  interface cusparseCsrSetPointers
    function cusparseCsrSetPointers_(spMatDescr,csrRowOffsets,&
&csrColInd,csrValues) bind(c, name="cusparseCsrSetPointers")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCsrSetPointers_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: csrRowOffsets
      type(c_ptr),value :: csrColInd
      type(c_ptr),value :: csrValues
    end function

  end interface
  
  interface cusparseCscSetPointers
    function cusparseCscSetPointers_(spMatDescr,cscColOffsets,&
&cscRowInd,cscValues) bind(c, name="cusparseCscSetPointers")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCscSetPointers_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: cscColOffsets
      type(c_ptr),value :: cscRowInd
      type(c_ptr),value :: cscValues
    end function

  end interface
  
  interface cusparseCooSetPointers
    function cusparseCooSetPointers_(spMatDescr,cooRowInd,cooColInd,&
&cooValues) bind(c, name="cusparseCooSetPointers")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCooSetPointers_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: cooRowInd
      type(c_ptr),value :: cooColInd
      type(c_ptr),value :: cooValues
    end function

  end interface
  
  interface cusparseSpMatGetSize
    function cusparseSpMatGetSize_(spMatDescr,rows,cols,nnz) bind(c, &
&name="cusparseSpMatGetSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMatGetSize_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: rows
      type(c_ptr),value :: cols
      type(c_ptr),value :: nnz
    end function

  end interface
  
  interface cusparseSpMatGetFormat
    function cusparseSpMatGetFormat_(spMatDescr,myFormat) bind(c, &
&name="cusparseSpMatGetFormat")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMatGetFormat_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: myFormat
    end function

  end interface
  
  interface cusparseSpMatGetIndexBase
    function cusparseSpMatGetIndexBase_(spMatDescr,idxBase) bind(c, &
&name="cusparseSpMatGetIndexBase")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpMatGetIndexBase_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: idxBase
    end function

  end interface
  
  interface cusparseSpMatGetValues
    function cusparseSpMatGetValues_(spMatDescr,values) bind(c, &
&name="cusparseSpMatGetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMatGetValues_
      type(c_ptr),value :: spMatDescr
      type(c_ptr) :: values
    end function

  end interface
  
  interface cusparseSpMatSetValues
    function cusparseSpMatSetValues_(spMatDescr,values) bind(c, &
&name="cusparseSpMatSetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMatSetValues_
      type(c_ptr),value :: spMatDescr
      type(c_ptr),value :: values
    end function

  end interface
  
  interface cusparseCreateDnVec
    function cusparseCreateDnVec_(dnVecDescr,mySize,values,valueType) &
&bind(c, name="cusparseCreateDnVec")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateDnVec_
      type(c_ptr) :: dnVecDescr
      integer(c_int64_t),value :: mySize
      type(c_ptr),value :: values
      integer(kind(CU_R_16F)),value :: valueType
    end function

  end interface
  
  interface cusparseDestroyDnVec
    function cusparseDestroyDnVec_(dnVecDescr) bind(c, &
&name="cusparseDestroyDnVec")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDestroyDnVec_
      type(c_ptr),value :: dnVecDescr
    end function

  end interface
  
  interface cusparseDnVecGet
    function cusparseDnVecGet_(dnVecDescr,mySize,values,valueType) &
&bind(c, name="cusparseDnVecGet")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnVecGet_
      type(c_ptr),value :: dnVecDescr
      type(c_ptr),value :: mySize
      type(c_ptr) :: values
      type(c_ptr),value :: valueType
    end function

  end interface
  
  interface cusparseDnVecGetValues
    function cusparseDnVecGetValues_(dnVecDescr,values) bind(c, &
&name="cusparseDnVecGetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnVecGetValues_
      type(c_ptr),value :: dnVecDescr
      type(c_ptr) :: values
    end function

  end interface
  
  interface cusparseDnVecSetValues
    function cusparseDnVecSetValues_(dnVecDescr,values) bind(c, &
&name="cusparseDnVecSetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnVecSetValues_
      type(c_ptr),value :: dnVecDescr
      type(c_ptr),value :: values
    end function

  end interface
  
  interface cusparseCreateDnMat
    function cusparseCreateDnMat_(dnMatDescr,rows,cols,ld,values,&
&valueType,order) bind(c, name="cusparseCreateDnMat")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCreateDnMat_
      type(c_ptr) :: dnMatDescr
      integer(c_int64_t),value :: rows
      integer(c_int64_t),value :: cols
      integer(c_int64_t),value :: ld
      type(c_ptr),value :: values
      integer(kind(CU_R_16F)),value :: valueType
      integer(kind(CUSPARSE_ORDER_COLUMN)),value :: order
    end function

  end interface
  
  interface cusparseDestroyDnMat
    function cusparseDestroyDnMat_(dnMatDescr) bind(c, &
&name="cusparseDestroyDnMat")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDestroyDnMat_
      type(c_ptr),value :: dnMatDescr
    end function

  end interface
  
  interface cusparseDnMatGet
    function cusparseDnMatGet_(dnMatDescr,rows,cols,ld,values,&
&valueType,order) bind(c, name="cusparseDnMatGet")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnMatGet_
      type(c_ptr),value :: dnMatDescr
      type(c_ptr),value :: rows
      type(c_ptr),value :: cols
      type(c_ptr),value :: ld
      type(c_ptr) :: values
      type(c_ptr),value :: valueType
      type(c_ptr),value :: order
    end function

  end interface
  
  interface cusparseDnMatGetValues
    function cusparseDnMatGetValues_(dnMatDescr,values) bind(c, &
&name="cusparseDnMatGetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnMatGetValues_
      type(c_ptr),value :: dnMatDescr
      type(c_ptr) :: values
    end function

  end interface
  
  interface cusparseDnMatSetValues
    function cusparseDnMatSetValues_(dnMatDescr,values) bind(c, &
&name="cusparseDnMatSetValues")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnMatSetValues_
      type(c_ptr),value :: dnMatDescr
      type(c_ptr),value :: values
    end function

  end interface
  
  interface cusparseSpMatGetAttribute
    function cusparseSpMatGetAttribute_(spMatDescr,attribute,myData,&
&dataSize) bind(c, name="cusparseSpMatGetAttribute")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpMatGetAttribute_
      type(c_ptr),value :: spMatDescr
      integer(kind(CUSPARSE_SPMAT_FILL_MODE)),value :: attribute
      type(c_ptr),value :: myData
      integer(c_size_t),value :: dataSize
    end function

  end interface
  
  interface cusparseSpMatSetAttribute
    function cusparseSpMatSetAttribute_(spMatDescr,attribute,myData,&
&dataSize) bind(c, name="cusparseSpMatSetAttribute")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpMatSetAttribute_
      type(c_ptr),value :: spMatDescr
      integer(kind(CUSPARSE_SPMAT_FILL_MODE)),value :: attribute
      type(c_ptr),value :: myData
      integer(c_size_t),value :: dataSize
    end function

  end interface
  
  interface cusparseAxpby
    function cusparseAxpby_(handle,alpha,vecX,beta,vecY) bind(c, &
&name="cusparseAxpby")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseAxpby_
      type(c_ptr),value :: handle
      type(c_ptr),value :: alpha
      type(c_ptr),value :: vecX
      type(c_ptr),value :: beta
      type(c_ptr),value :: vecY
    end function

  end interface
  
  interface cusparseGather
    function cusparseGather_(handle,vecY,vecX) bind(c, &
&name="cusparseGather")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseGather_
      type(c_ptr),value :: handle
      type(c_ptr),value :: vecY
      type(c_ptr),value :: vecX
    end function

  end interface
  
  interface cusparseScatter
    function cusparseScatter_(handle,vecX,vecY) bind(c, &
&name="cusparseScatter")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScatter_
      type(c_ptr),value :: handle
      type(c_ptr),value :: vecX
      type(c_ptr),value :: vecY
    end function

  end interface
  
  interface cusparseRot
    function cusparseRot_(handle,c_coeff,s_coeff,vecX,vecY) bind(c, &
&name="cusparseRot")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseRot_
      type(c_ptr),value :: handle
      type(c_ptr),value :: c_coeff
      type(c_ptr),value :: s_coeff
      type(c_ptr),value :: vecX
      type(c_ptr),value :: vecY
    end function

  end interface
  
  interface cusparseSparseToDense_bufferSize
    function cusparseSparseToDense_bufferSize_(handle,matA,matB,alg,&
&bufferSize) bind(c, name="cusparseSparseToDense_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSparseToDense_bufferSize_
      type(c_ptr),value :: handle
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      integer(kind(CUSPARSE_SPARSETODENSE_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseSparseToDense_bufferSize_rank_0,&
      cusparseSparseToDense_bufferSize_rank_1
  end interface
  
  interface cusparseSparseToDense
    function cusparseSparseToDense_(handle,matA,matB,alg,&
&externalBuffer) bind(c, name="cusparseSparseToDense")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSparseToDense_
      type(c_ptr),value :: handle
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      integer(kind(CUSPARSE_SPARSETODENSE_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  
  interface cusparseDenseToSparse_bufferSize
    function cusparseDenseToSparse_bufferSize_(handle,matA,matB,alg,&
&bufferSize) bind(c, name="cusparseDenseToSparse_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDenseToSparse_bufferSize_
      type(c_ptr),value :: handle
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      integer(kind(CUSPARSE_DENSETOSPARSE_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: bufferSize
    end function

    module procedure &
      cusparseDenseToSparse_bufferSize_rank_0,&
      cusparseDenseToSparse_bufferSize_rank_1
  end interface
  
  interface cusparseDenseToSparse_analysis
    function cusparseDenseToSparse_analysis_(handle,matA,matB,alg,&
&externalBuffer) bind(c, name="cusparseDenseToSparse_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDenseToSparse_analysis_
      type(c_ptr),value :: handle
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      integer(kind(CUSPARSE_DENSETOSPARSE_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  
  interface cusparseDenseToSparse_convert
    function cusparseDenseToSparse_convert_(handle,matA,matB,alg,&
&externalBuffer) bind(c, name="cusparseDenseToSparse_convert")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDenseToSparse_convert_
      type(c_ptr),value :: handle
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      integer(kind(CUSPARSE_DENSETOSPARSE_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  
  interface cusparseSpVV_bufferSize
    function cusparseSpVV_bufferSize_(handle,opX,vecX,vecY,myResult,&
&computeType,bufferSize) bind(c, name="cusparseSpVV_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpVV_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opX
      type(c_ptr),value :: vecX
      type(c_ptr),value :: vecY
      type(c_ptr),value :: myResult
      integer(kind(CU_R_16F)),value :: computeType
      type(c_ptr),value :: bufferSize
    end function

  end interface
  
  interface cusparseSpVV
    function cusparseSpVV_(handle,opX,vecX,vecY,myResult,computeType,&
&externalBuffer) bind(c, name="cusparseSpVV")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpVV_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opX
      type(c_ptr),value :: vecX
      type(c_ptr),value :: vecY
      type(c_ptr),value :: myResult
      integer(kind(CU_R_16F)),value :: computeType
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  
  interface cusparseSpMV_bufferSize
    function cusparseSpMV_bufferSize_(handle,opA,alpha,matA,vecX,beta,&
&vecY,computeType,alg,bufferSize) bind(c, &
&name="cusparseSpMV_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMV_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: vecX
      type(c_ptr),value :: beta
      type(c_ptr),value :: vecY
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_MV_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: bufferSize
    end function

  end interface
  
  interface cusparseSpMV
    function cusparseSpMV_(handle,opA,alpha,matA,vecX,beta,vecY,&
&computeType,alg,externalBuffer) bind(c, name="cusparseSpMV")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMV_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: vecX
      type(c_ptr),value :: beta
      type(c_ptr),value :: vecY
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_MV_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  
  interface cusparseSpMM_bufferSize
    function cusparseSpMM_bufferSize_(handle,opA,opB,alpha,matA,matB,&
&beta,matC,computeType,alg,bufferSize) bind(c, &
&name="cusparseSpMM_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMM_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: beta
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_MM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: bufferSize
    end function

  end interface
  !>  Description: Preprocess step of the sparse matrix multiplication with a dense matrix.
  interface cusparseSpMM_preprocess
    function cusparseSpMM_preprocess_(handle,opA,opB,alpha,matA,matB,&
&beta,matC,computeType,alg,externalBuffer) bind(c, &
&name="cusparseSpMM_preprocess")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMM_preprocess_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: beta
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_MM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  !>  Description: Compute the sparse matrix multiplication with a dense matrix
  interface cusparseSpMM
    function cusparseSpMM_(handle,opA,opB,alpha,matA,matB,beta,matC,&
&computeType,alg,externalBuffer) bind(c, name="cusparseSpMM")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpMM_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: beta
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_MM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  
  interface cusparseSpGEMM_createDescr
    function cusparseSpGEMM_createDescr_(descr) bind(c, &
&name="cusparseSpGEMM_createDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpGEMM_createDescr_
      type(c_ptr) :: descr
    end function

  end interface
  
  interface cusparseSpGEMM_destroyDescr
    function cusparseSpGEMM_destroyDescr_(descr) bind(c, &
&name="cusparseSpGEMM_destroyDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpGEMM_destroyDescr_
      type(c_ptr),value :: descr
    end function

  end interface
  
  interface cusparseSpGEMM_workEstimation
    function cusparseSpGEMM_workEstimation_(handle,opA,opB,alpha,matA,&
&matB,beta,matC,computeType,alg,spgemmDescr,bufferSize1,&
&externalBuffer1) bind(c, name="cusparseSpGEMM_workEstimation")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpGEMM_workEstimation_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: beta
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPGEMM_DEFAULT)),value :: alg
      type(c_ptr),value :: spgemmDescr
      type(c_ptr),value :: bufferSize1
      type(c_ptr),value :: externalBuffer1
    end function

  end interface
  
  interface cusparseSpGEMM_compute
    function cusparseSpGEMM_compute_(handle,opA,opB,alpha,matA,matB,&
&beta,matC,computeType,alg,spgemmDescr,bufferSize2,externalBuffer2) &
&bind(c, name="cusparseSpGEMM_compute")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpGEMM_compute_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: beta
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPGEMM_DEFAULT)),value :: alg
      type(c_ptr),value :: spgemmDescr
      type(c_ptr),value :: bufferSize2
      type(c_ptr),value :: externalBuffer2
    end function

  end interface
  
  interface cusparseSpGEMM_copy
    function cusparseSpGEMM_copy_(handle,opA,opB,alpha,matA,matB,beta,&
&matC,computeType,alg,spgemmDescr) bind(c, name="cusparseSpGEMM_copy")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpGEMM_copy_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: beta
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPGEMM_DEFAULT)),value :: alg
      type(c_ptr),value :: spgemmDescr
    end function

  end interface
  
  interface cusparseSDDMM
    function cusparseSDDMM_(handle,opA,opB,alpha,A,B,beta,C,&
&computeType,alg,tempBuffer) bind(c, name="cusparseSDDMM")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSDDMM_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: A
      type(c_ptr),value :: B
      type(c_ptr),value :: beta
      type(c_ptr),value :: C
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SDDMM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: tempBuffer
    end function

  end interface
  
  interface cusparseSDDMM_bufferSize
    function cusparseSDDMM_bufferSize_(handle,opA,opB,alpha,A,B,beta,C,&
&computeType,alg,bufferSize) bind(c, name="cusparseSDDMM_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSDDMM_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: A
      type(c_ptr),value :: B
      type(c_ptr),value :: beta
      type(c_ptr),value :: C
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SDDMM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: bufferSize
    end function

  end interface
  
  interface cusparseSDDMM_preprocess
    function cusparseSDDMM_preprocess_(handle,opA,opB,alpha,A,B,beta,C,&
&computeType,alg,tempBuffer) bind(c, name="cusparseSDDMM_preprocess")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSDDMM_preprocess_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: A
      type(c_ptr),value :: B
      type(c_ptr),value :: beta
      type(c_ptr),value :: C
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SDDMM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: tempBuffer
    end function

  end interface
  
  interface cusparseSpSV_createDescr
    function cusparseSpSV_createDescr_(descr) bind(c, &
&name="cusparseSpSV_createDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpSV_createDescr_
      type(c_ptr) :: descr
    end function

  end interface
  
  interface cusparseSpSV_destroyDescr
    function cusparseSpSV_destroyDescr_(descr) bind(c, &
&name="cusparseSpSV_destroyDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpSV_destroyDescr_
      type(c_ptr),value :: descr
    end function

  end interface
  
  interface cusparseSpSV_bufferSize
    function cusparseSpSV_bufferSize_(handle,opA,alpha,matA,x,y,&
&computeType,alg,spsvDescr,bufferSize) bind(c, &
&name="cusparseSpSV_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpSV_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: x
      type(c_ptr),value :: y
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPSV_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: spsvDescr
      type(c_ptr),value :: bufferSize
    end function

  end interface
  !>  Description: Analysis step of solution of triangular linear system op(A) * Y = alpha * X,
  !> where A is a sparse matrix in CSR storage format, x and Y are dense vectors.
  interface cusparseSpSV_analysis
    function cusparseSpSV_analysis_(handle,opA,alpha,matA,x,y,&
&computeType,alg,spsvDescr,externalBuffer) bind(c, &
&name="cusparseSpSV_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpSV_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: x
      type(c_ptr),value :: y
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPSV_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: spsvDescr
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  !>  Description: Solve step of solution of triangular linear system op(A) * Y = alpha * X,
  !> where A is a sparse matrix in CSR storage format, x and Y are dense vectors.
  interface cusparseSpSV_solve
    function cusparseSpSV_solve_(handle,opA,alpha,matA,x,y,computeType,&
&alg,spsvDescr,externalBuffer) bind(c, name="cusparseSpSV_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpSV_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: x
      type(c_ptr),value :: y
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPSV_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: spsvDescr
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  
  interface cusparseSpSM_createDescr
    function cusparseSpSM_createDescr_(descr) bind(c, &
&name="cusparseSpSM_createDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpSM_createDescr_
      type(c_ptr) :: descr
    end function

  end interface
  
  interface cusparseSpSM_destroyDescr
    function cusparseSpSM_destroyDescr_(descr) bind(c, &
&name="cusparseSpSM_destroyDescr")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpSM_destroyDescr_
      type(c_ptr),value :: descr
    end function

  end interface
  
  interface cusparseSpSM_bufferSize
    function cusparseSpSM_bufferSize_(handle,opA,opB,alpha,matA,matB,&
&matC,computeType,alg,spsmDescr,bufferSize) bind(c, &
&name="cusparseSpSM_bufferSize")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpSM_bufferSize_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPSM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: spsmDescr
      type(c_ptr),value :: bufferSize
    end function

  end interface
  !>  Description: Analysis step of solution of triangular linear system op(A) * C = alpha * op(B),
  !> where A is a sparse matrix in CSR storage format, B and C are dense vectors.
  interface cusparseSpSM_analysis
    function cusparseSpSM_analysis_(handle,opA,opB,alpha,matA,matB,&
&matC,computeType,alg,spsmDescr,externalBuffer) bind(c, &
&name="cusparseSpSM_analysis")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpSM_analysis_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPSM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: spsmDescr
      type(c_ptr),value :: externalBuffer
    end function

  end interface
  !>  Description: Solve step of solution of triangular linear system op(A) * C = alpha * op(B),
  !> where A is a sparse matrix in CSR storage format, B and C are dense vectors.
  interface cusparseSpSM_solve
    function cusparseSpSM_solve_(handle,opA,opB,alpha,matA,matB,matC,&
&computeType,alg,spsmDescr,externalBuffer) bind(c, &
&name="cusparseSpSM_solve")
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSpSM_solve_
      type(c_ptr),value :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)),value :: opB
      type(c_ptr),value :: alpha
      type(c_ptr),value :: matA
      type(c_ptr),value :: matB
      type(c_ptr),value :: matC
      integer(kind(CU_R_16F)),value :: computeType
      integer(kind(CUSPARSE_SPSM_ALG_DEFAULT)),value :: alg
      type(c_ptr),value :: spsmDescr
      type(c_ptr),value :: externalBuffer
    end function

  end interface

  contains
    function cusparseSaxpyi_rank_0(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSaxpyi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float) :: alpha
      real(c_float),target :: xVal
      integer(c_int),target :: xInd
      real(c_float),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSaxpyi_rank_0 = cusparseSaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseSaxpyi_rank_1(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSaxpyi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float) :: alpha
      real(c_float),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_float),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSaxpyi_rank_1 = cusparseSaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseDaxpyi_rank_0(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDaxpyi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double) :: alpha
      real(c_double),target :: xVal
      integer(c_int),target :: xInd
      real(c_double),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDaxpyi_rank_0 = cusparseDaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseDaxpyi_rank_1(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDaxpyi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double) :: alpha
      real(c_double),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_double),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDaxpyi_rank_1 = cusparseDaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseCaxpyi_rank_0(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCaxpyi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_float_complex),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCaxpyi_rank_0 = cusparseCaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseCaxpyi_rank_1(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCaxpyi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_float_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCaxpyi_rank_1 = cusparseCaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseZaxpyi_rank_0(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZaxpyi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_double_complex),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZaxpyi_rank_0 = cusparseZaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseZaxpyi_rank_1(handle,nnz,alpha,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZaxpyi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_double_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZaxpyi_rank_1 = cusparseZaxpyi_(handle,nnz,alpha,&
&c_loc(xVal),c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseSdoti_rank_0(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSdoti_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target :: xVal
      integer(c_int),target :: xInd
      real(c_float),target :: y
      real(c_float),target :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSdoti_rank_0 = cusparseSdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseSdoti_rank_1(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSdoti_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_float),target,dimension(:) :: y
      real(c_float),target,dimension(:) :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSdoti_rank_1 = cusparseSdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseDdoti_rank_0(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDdoti_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target :: xVal
      integer(c_int),target :: xInd
      real(c_double),target :: y
      real(c_double),target :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDdoti_rank_0 = cusparseDdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseDdoti_rank_1(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDdoti_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_double),target,dimension(:) :: y
      real(c_double),target,dimension(:) :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDdoti_rank_1 = cusparseDdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseCdoti_rank_0(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdoti_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_float_complex),target :: y
      complex(c_float_complex),target :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCdoti_rank_0 = cusparseCdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseCdoti_rank_1(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdoti_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_float_complex),target,dimension(:) :: y
      complex(c_float_complex),target,dimension(:) :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCdoti_rank_1 = cusparseCdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseZdoti_rank_0(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdoti_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_double_complex),target :: y
      complex(c_double_complex),target :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZdoti_rank_0 = cusparseZdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseZdoti_rank_1(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdoti_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_double_complex),target,dimension(:) :: y
      complex(c_double_complex),target,dimension(:) :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZdoti_rank_1 = cusparseZdoti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseCdotci_rank_0(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdotci_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_float_complex),target :: y
      complex(c_float_complex),target :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCdotci_rank_0 = cusparseCdotci_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseCdotci_rank_1(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCdotci_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_float_complex),target,dimension(:) :: y
      complex(c_float_complex),target,dimension(:) :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCdotci_rank_1 = cusparseCdotci_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseZdotci_rank_0(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdotci_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_double_complex),target :: y
      complex(c_double_complex),target :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZdotci_rank_0 = cusparseZdotci_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseZdotci_rank_1(handle,nnz,xVal,xInd,y,myResult,&
&idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZdotci_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_double_complex),target,dimension(:) :: y
      complex(c_double_complex),target,dimension(:) :: myResult
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZdotci_rank_1 = cusparseZdotci_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c_loc(myResult),idxBase)
    end function

    function cusparseSgthr_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgthr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target :: y
      real(c_float),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSgthr_rank_0 = cusparseSgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseSgthr_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgthr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target,dimension(:) :: y
      real(c_float),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSgthr_rank_1 = cusparseSgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseDgthr_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgthr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target :: y
      real(c_double),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDgthr_rank_0 = cusparseDgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseDgthr_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgthr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target,dimension(:) :: y
      real(c_double),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDgthr_rank_1 = cusparseDgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseCgthr_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgthr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target :: y
      complex(c_float_complex),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCgthr_rank_0 = cusparseCgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseCgthr_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgthr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target,dimension(:) :: y
      complex(c_float_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCgthr_rank_1 = cusparseCgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseZgthr_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgthr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target :: y
      complex(c_double_complex),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZgthr_rank_0 = cusparseZgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseZgthr_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgthr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target,dimension(:) :: y
      complex(c_double_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZgthr_rank_1 = cusparseZgthr_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseSgthrz_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgthrz_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target :: y
      real(c_float),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSgthrz_rank_0 = cusparseSgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseSgthrz_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgthrz_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target,dimension(:) :: y
      real(c_float),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSgthrz_rank_1 = cusparseSgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseDgthrz_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgthrz_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target :: y
      real(c_double),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDgthrz_rank_0 = cusparseDgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseDgthrz_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgthrz_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target,dimension(:) :: y
      real(c_double),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDgthrz_rank_1 = cusparseDgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseCgthrz_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgthrz_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target :: y
      complex(c_float_complex),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCgthrz_rank_0 = cusparseCgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseCgthrz_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgthrz_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target,dimension(:) :: y
      complex(c_float_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCgthrz_rank_1 = cusparseCgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseZgthrz_rank_0(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgthrz_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target :: y
      complex(c_double_complex),target :: xVal
      integer(c_int),target :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZgthrz_rank_0 = cusparseZgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseZgthrz_rank_1(handle,nnz,y,xVal,xInd,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgthrz_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target,dimension(:) :: y
      complex(c_double_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZgthrz_rank_1 = cusparseZgthrz_(handle,nnz,c_loc(y),&
&c_loc(xVal),c_loc(xInd),idxBase)
    end function

    function cusparseSroti_rank_0(handle,nnz,xVal,xInd,y,c,s,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSroti_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target :: xVal
      integer(c_int),target :: xInd
      real(c_float),target :: y
      real(c_float) :: c
      real(c_float) :: s
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSroti_rank_0 = cusparseSroti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c,s,idxBase)
    end function

    function cusparseSroti_rank_1(handle,nnz,xVal,xInd,y,c,s,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSroti_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_float),target,dimension(:) :: y
      real(c_float) :: c
      real(c_float) :: s
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSroti_rank_1 = cusparseSroti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c,s,idxBase)
    end function

    function cusparseDroti_rank_0(handle,nnz,xVal,xInd,y,c,s,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDroti_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target :: xVal
      integer(c_int),target :: xInd
      real(c_double),target :: y
      real(c_double) :: c
      real(c_double) :: s
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDroti_rank_0 = cusparseDroti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c,s,idxBase)
    end function

    function cusparseDroti_rank_1(handle,nnz,xVal,xInd,y,c,s,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDroti_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_double),target,dimension(:) :: y
      real(c_double) :: c
      real(c_double) :: s
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDroti_rank_1 = cusparseDroti_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),c,s,idxBase)
    end function

    function cusparseSsctr_rank_0(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSsctr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target :: xVal
      integer(c_int),target :: xInd
      real(c_float),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSsctr_rank_0 = cusparseSsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseSsctr_rank_1(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSsctr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_float),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_float),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseSsctr_rank_1 = cusparseSsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseDsctr_rank_0(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDsctr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target :: xVal
      integer(c_int),target :: xInd
      real(c_double),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDsctr_rank_0 = cusparseDsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseDsctr_rank_1(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDsctr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      real(c_double),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      real(c_double),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseDsctr_rank_1 = cusparseDsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseCsctr_rank_0(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCsctr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_float_complex),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCsctr_rank_0 = cusparseCsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseCsctr_rank_1(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCsctr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_float_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_float_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseCsctr_rank_1 = cusparseCsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseZsctr_rank_0(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZsctr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target :: xVal
      integer(c_int),target :: xInd
      complex(c_double_complex),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZsctr_rank_0 = cusparseZsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseZsctr_rank_1(handle,nnz,xVal,xInd,y,idxBase)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZsctr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: nnz
      complex(c_double_complex),target,dimension(:) :: xVal
      integer(c_int),target,dimension(:) :: xInd
      complex(c_double_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      !
      cusparseZsctr_rank_1 = cusparseZsctr_(handle,nnz,c_loc(xVal),&
&c_loc(xInd),c_loc(y),idxBase)
    end function

    function cusparseScsrmv_rank_0(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float),target :: x
      real(c_float) :: beta
      real(c_float),target :: y
      !
      cusparseScsrmv_rank_0 = cusparseScsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseScsrmv_rank_1(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:) :: x
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: y
      !
      cusparseScsrmv_rank_1 = cusparseScsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseDcsrmv_rank_0(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double),target :: x
      real(c_double) :: beta
      real(c_double),target :: y
      !
      cusparseDcsrmv_rank_0 = cusparseDcsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseDcsrmv_rank_1(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:) :: x
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: y
      !
      cusparseDcsrmv_rank_1 = cusparseDcsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseCcsrmv_rank_0(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex),target :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: y
      !
      cusparseCcsrmv_rank_0 = cusparseCcsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseCcsrmv_rank_1(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:) :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: y
      !
      cusparseCcsrmv_rank_1 = cusparseCcsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseZcsrmv_rank_0(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex),target :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: y
      !
      cusparseZcsrmv_rank_0 = cusparseZcsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseZcsrmv_rank_1(handle,transA,m,n,nnz,alpha,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:) :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: y
      !
      cusparseZcsrmv_rank_1 = cusparseZcsrmv_(handle,transA,m,n,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(x),beta,c_loc(y))
    end function

    function cusparseScsrsv2_bufferSize_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseScsrsv2_bufferSize_rank_0 = &
&cusparseScsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseScsrsv2_bufferSize_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseScsrsv2_bufferSize_rank_1 = &
&cusparseScsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseDcsrsv2_bufferSize_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDcsrsv2_bufferSize_rank_0 = &
&cusparseDcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseDcsrsv2_bufferSize_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDcsrsv2_bufferSize_rank_1 = &
&cusparseDcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseCcsrsv2_bufferSize_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCcsrsv2_bufferSize_rank_0 = &
&cusparseCcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseCcsrsv2_bufferSize_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCcsrsv2_bufferSize_rank_1 = &
&cusparseCcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseZcsrsv2_bufferSize_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZcsrsv2_bufferSize_rank_0 = &
&cusparseZcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseZcsrsv2_bufferSize_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZcsrsv2_bufferSize_rank_1 = &
&cusparseZcsrsv2_bufferSize_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseScsrsv2_bufferSizeExt_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsrsv2_bufferSizeExt_rank_0 = &
&cusparseScsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseScsrsv2_bufferSizeExt_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsrsv2_bufferSizeExt_rank_1 = &
&cusparseScsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseDcsrsv2_bufferSizeExt_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsrsv2_bufferSizeExt_rank_0 = &
&cusparseDcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseDcsrsv2_bufferSizeExt_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsrsv2_bufferSizeExt_rank_1 = &
&cusparseDcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseCcsrsv2_bufferSizeExt_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsrsv2_bufferSizeExt_rank_0 = &
&cusparseCcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseCcsrsv2_bufferSizeExt_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsrsv2_bufferSizeExt_rank_1 = &
&cusparseCcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseZcsrsv2_bufferSizeExt_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsrsv2_bufferSizeExt_rank_0 = &
&cusparseZcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseZcsrsv2_bufferSizeExt_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsrsv2_bufferSizeExt_rank_1 = &
&cusparseZcsrsv2_bufferSizeExt_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseScsrsv2_analysis_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsv2_analysis_rank_0 = &
&cusparseScsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseScsrsv2_analysis_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsv2_analysis_rank_1 = &
&cusparseScsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseDcsrsv2_analysis_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsv2_analysis_rank_0 = &
&cusparseDcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseDcsrsv2_analysis_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsv2_analysis_rank_1 = &
&cusparseDcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseCcsrsv2_analysis_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsv2_analysis_rank_0 = &
&cusparseCcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseCcsrsv2_analysis_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsv2_analysis_rank_1 = &
&cusparseCcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseZcsrsv2_analysis_rank_0(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsv2_analysis_rank_0 = &
&cusparseZcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseZcsrsv2_analysis_rank_1(handle,transA,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsv2_analysis_rank_1 = &
&cusparseZcsrsv2_analysis_(handle,transA,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,policy,pBuffer)
    end function

    function cusparseScsrsv2_solve_rank_0(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      real(c_float),target :: f
      real(c_float),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsv2_solve_rank_0 = cusparseScsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseScsrsv2_solve_rank_1(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      real(c_float),target,dimension(:) :: f
      real(c_float),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsv2_solve_rank_1 = cusparseScsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseDcsrsv2_solve_rank_0(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      real(c_double),target :: f
      real(c_double),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsv2_solve_rank_0 = cusparseDcsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseDcsrsv2_solve_rank_1(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      real(c_double),target,dimension(:) :: f
      real(c_double),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsv2_solve_rank_1 = cusparseDcsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseCcsrsv2_solve_rank_0(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      complex(c_float_complex),target :: f
      complex(c_float_complex),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsv2_solve_rank_0 = cusparseCcsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseCcsrsv2_solve_rank_1(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      complex(c_float_complex),target,dimension(:) :: f
      complex(c_float_complex),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsv2_solve_rank_1 = cusparseCcsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseZcsrsv2_solve_rank_0(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      complex(c_double_complex),target :: f
      complex(c_double_complex),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsv2_solve_rank_0 = cusparseZcsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseZcsrsv2_solve_rank_1(handle,transA,m,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,f,x,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      complex(c_double_complex),target,dimension(:) :: f
      complex(c_double_complex),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsv2_solve_rank_1 = cusparseZcsrsv2_solve_(handle,&
&transA,m,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,c_loc(f),&
&c_loc(x),policy,pBuffer)
    end function

    function cusparseShybmv_rank_0(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseShybmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      real(c_float),target :: x
      real(c_float) :: beta
      real(c_float),target :: y
      !
      cusparseShybmv_rank_0 = cusparseShybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseShybmv_rank_1(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseShybmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      real(c_float),target,dimension(:) :: x
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: y
      !
      cusparseShybmv_rank_1 = cusparseShybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseDhybmv_rank_0(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDhybmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      real(c_double),target :: x
      real(c_double) :: beta
      real(c_double),target :: y
      !
      cusparseDhybmv_rank_0 = cusparseDhybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseDhybmv_rank_1(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDhybmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      real(c_double),target,dimension(:) :: x
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: y
      !
      cusparseDhybmv_rank_1 = cusparseDhybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseChybmv_rank_0(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseChybmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      complex(c_float_complex),target :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: y
      !
      cusparseChybmv_rank_0 = cusparseChybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseChybmv_rank_1(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseChybmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      complex(c_float_complex),target,dimension(:) :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: y
      !
      cusparseChybmv_rank_1 = cusparseChybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseZhybmv_rank_0(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZhybmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      complex(c_double_complex),target :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: y
      !
      cusparseZhybmv_rank_0 = cusparseZhybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseZhybmv_rank_1(handle,transA,alpha,descrA,hybA,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZhybmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      type(c_ptr) :: hybA
      complex(c_double_complex),target,dimension(:) :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: y
      !
      cusparseZhybmv_rank_1 = cusparseZhybmv_(handle,transA,alpha,&
&descrA,hybA,c_loc(x),beta,c_loc(y))
    end function

    function cusparseSbsrmv_rank_0(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      real(c_float),target :: x
      real(c_float) :: beta
      real(c_float),target :: y
      !
      cusparseSbsrmv_rank_0 = cusparseSbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseSbsrmv_rank_1(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      real(c_float),target,dimension(:) :: x
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: y
      !
      cusparseSbsrmv_rank_1 = cusparseSbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseDbsrmv_rank_0(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      real(c_double),target :: x
      real(c_double) :: beta
      real(c_double),target :: y
      !
      cusparseDbsrmv_rank_0 = cusparseDbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseDbsrmv_rank_1(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      real(c_double),target,dimension(:) :: x
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: y
      !
      cusparseDbsrmv_rank_1 = cusparseDbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseCbsrmv_rank_0(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      complex(c_float_complex),target :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: y
      !
      cusparseCbsrmv_rank_0 = cusparseCbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseCbsrmv_rank_1(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      complex(c_float_complex),target,dimension(:) :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: y
      !
      cusparseCbsrmv_rank_1 = cusparseCbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseZbsrmv_rank_0(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      complex(c_double_complex),target :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: y
      !
      cusparseZbsrmv_rank_0 = cusparseZbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseZbsrmv_rank_1(handle,dirA,transA,mb,nb,nnzb,alpha,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,x,&
&beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      complex(c_double_complex),target,dimension(:) :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: y
      !
      cusparseZbsrmv_rank_1 = cusparseZbsrmv_(handle,dirA,transA,mb,nb,&
&nnzb,alpha,descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseSbsrxmv_rank_0(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrxmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      real(c_float),target :: x
      real(c_float) :: beta
      real(c_float),target :: y
      !
      cusparseSbsrxmv_rank_0 = cusparseSbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseSbsrxmv_rank_1(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrxmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      real(c_float),target,dimension(:) :: x
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: y
      !
      cusparseSbsrxmv_rank_1 = cusparseSbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseDbsrxmv_rank_0(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrxmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      real(c_double),target :: x
      real(c_double) :: beta
      real(c_double),target :: y
      !
      cusparseDbsrxmv_rank_0 = cusparseDbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseDbsrxmv_rank_1(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrxmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      real(c_double),target,dimension(:) :: x
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: y
      !
      cusparseDbsrxmv_rank_1 = cusparseDbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseCbsrxmv_rank_0(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrxmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      complex(c_float_complex),target :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: y
      !
      cusparseCbsrxmv_rank_0 = cusparseCbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseCbsrxmv_rank_1(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrxmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      complex(c_float_complex),target,dimension(:) :: x
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: y
      !
      cusparseCbsrxmv_rank_1 = cusparseCbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseZbsrxmv_rank_0(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrxmv_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      complex(c_double_complex),target :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: y
      !
      cusparseZbsrxmv_rank_0 = cusparseZbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseZbsrxmv_rank_1(handle,dir,trans,sizeOfMask,mb,nb,&
&nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,bsrEndPtr,bsrColInd,&
&blockDim,x,beta,y)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrxmv_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dir
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: trans
      integer(c_int) :: sizeOfMask
      integer(c_int) :: mb
      integer(c_int) :: nb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descr
      type(c_ptr) :: bsrVal
      type(c_ptr) :: bsrMaskPtr
      type(c_ptr) :: bsrRowPtr
      type(c_ptr) :: bsrEndPtr
      type(c_ptr) :: bsrColInd
      integer(c_int) :: blockDim
      complex(c_double_complex),target,dimension(:) :: x
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: y
      !
      cusparseZbsrxmv_rank_1 = cusparseZbsrxmv_(handle,dir,trans,&
&sizeOfMask,mb,nb,nnzb,alpha,descr,bsrVal,bsrMaskPtr,bsrRowPtr,&
&bsrEndPtr,bsrColInd,blockDim,c_loc(x),beta,c_loc(y))
    end function

    function cusparseSbsrsv2_bufferSize_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsrsv2_bufferSize_rank_0 = &
&cusparseSbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseSbsrsv2_bufferSize_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsrsv2_bufferSize_rank_1 = &
&cusparseSbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseDbsrsv2_bufferSize_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsrsv2_bufferSize_rank_0 = &
&cusparseDbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseDbsrsv2_bufferSize_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsrsv2_bufferSize_rank_1 = &
&cusparseDbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseCbsrsv2_bufferSize_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsrsv2_bufferSize_rank_0 = &
&cusparseCbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseCbsrsv2_bufferSize_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsrsv2_bufferSize_rank_1 = &
&cusparseCbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseZbsrsv2_bufferSize_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsrsv2_bufferSize_rank_0 = &
&cusparseZbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseZbsrsv2_bufferSize_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsrsv2_bufferSize_rank_1 = &
&cusparseZbsrsv2_bufferSize_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseSbsrsv2_bufferSizeExt_rank_0(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseSbsrsv2_bufferSizeExt_rank_0 = &
&cusparseSbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseSbsrsv2_bufferSizeExt_rank_1(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseSbsrsv2_bufferSizeExt_rank_1 = &
&cusparseSbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseDbsrsv2_bufferSizeExt_rank_0(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDbsrsv2_bufferSizeExt_rank_0 = &
&cusparseDbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseDbsrsv2_bufferSizeExt_rank_1(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDbsrsv2_bufferSizeExt_rank_1 = &
&cusparseDbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseCbsrsv2_bufferSizeExt_rank_0(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCbsrsv2_bufferSizeExt_rank_0 = &
&cusparseCbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseCbsrsv2_bufferSizeExt_rank_1(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCbsrsv2_bufferSizeExt_rank_1 = &
&cusparseCbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseZbsrsv2_bufferSizeExt_rank_0(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZbsrsv2_bufferSizeExt_rank_0 = &
&cusparseZbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseZbsrsv2_bufferSizeExt_rank_1(handle,dirA,transA,&
&mb,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZbsrsv2_bufferSizeExt_rank_1 = &
&cusparseZbsrsv2_bufferSizeExt_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSize)
    end function

    function cusparseSbsrsv2_analysis_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsv2_analysis_rank_0 = &
&cusparseSbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsrsv2_analysis_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsv2_analysis_rank_1 = &
&cusparseSbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrsv2_analysis_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsv2_analysis_rank_0 = &
&cusparseDbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrsv2_analysis_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsv2_analysis_rank_1 = &
&cusparseDbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrsv2_analysis_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsv2_analysis_rank_0 = &
&cusparseCbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrsv2_analysis_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsv2_analysis_rank_1 = &
&cusparseCbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrsv2_analysis_rank_0(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsv2_analysis_rank_0 = &
&cusparseZbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrsv2_analysis_rank_1(handle,dirA,transA,mb,&
&nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsv2_analysis_rank_1 = &
&cusparseZbsrsv2_analysis_(handle,dirA,transA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsrsv2_solve_rank_0(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_float),target :: f
      real(c_float),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsv2_solve_rank_0 = cusparseSbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseSbsrsv2_solve_rank_1(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_float),target,dimension(:) :: f
      real(c_float),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsv2_solve_rank_1 = cusparseSbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseDbsrsv2_solve_rank_0(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_double),target :: f
      real(c_double),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsv2_solve_rank_0 = cusparseDbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseDbsrsv2_solve_rank_1(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_double),target,dimension(:) :: f
      real(c_double),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsv2_solve_rank_1 = cusparseDbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseCbsrsv2_solve_rank_0(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_float_complex),target :: f
      complex(c_float_complex),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsv2_solve_rank_0 = cusparseCbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseCbsrsv2_solve_rank_1(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_float_complex),target,dimension(:) :: f
      complex(c_float_complex),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsv2_solve_rank_1 = cusparseCbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseZbsrsv2_solve_rank_0(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_double_complex),target :: f
      complex(c_double_complex),target :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsv2_solve_rank_0 = cusparseZbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseZbsrsv2_solve_rank_1(handle,dirA,transA,mb,nnzb,&
&alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,f,x,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsv2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_double_complex),target,dimension(:) :: f
      complex(c_double_complex),target,dimension(:) :: x
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsv2_solve_rank_1 = cusparseZbsrsv2_solve_(handle,&
&dirA,transA,mb,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(f),c_loc(x),policy,pBuffer)
    end function

    function cusparseSgemvi_full_rank(handle,transA,m,n,alpha,A,lda,&
&nnz,x,xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemvi_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      real(c_float),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseSgemvi_full_rank = cusparseSgemvi_(handle,transA,m,n,&
&alpha,c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,&
&pBuffer)
    end function

    function cusparseSgemvi_rank_0(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemvi_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      real(c_float),target :: x
      integer(c_int),target :: xInd
      real(c_float) :: beta
      real(c_float),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseSgemvi_rank_0 = cusparseSgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseSgemvi_rank_1(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemvi_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      real(c_float),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseSgemvi_rank_1 = cusparseSgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseDgemvi_full_rank(handle,transA,m,n,alpha,A,lda,&
&nnz,x,xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemvi_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      real(c_double),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseDgemvi_full_rank = cusparseDgemvi_(handle,transA,m,n,&
&alpha,c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,&
&pBuffer)
    end function

    function cusparseDgemvi_rank_0(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemvi_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      real(c_double),target :: x
      integer(c_int),target :: xInd
      real(c_double) :: beta
      real(c_double),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseDgemvi_rank_0 = cusparseDgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseDgemvi_rank_1(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemvi_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      real(c_double),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseDgemvi_rank_1 = cusparseDgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseCgemvi_full_rank(handle,transA,m,n,alpha,A,lda,&
&nnz,x,xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemvi_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseCgemvi_full_rank = cusparseCgemvi_(handle,transA,m,n,&
&alpha,c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,&
&pBuffer)
    end function

    function cusparseCgemvi_rank_0(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemvi_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      complex(c_float_complex),target :: x
      integer(c_int),target :: xInd
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseCgemvi_rank_0 = cusparseCgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseCgemvi_rank_1(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemvi_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseCgemvi_rank_1 = cusparseCgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseZgemvi_full_rank(handle,transA,m,n,alpha,A,lda,&
&nnz,x,xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemvi_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseZgemvi_full_rank = cusparseZgemvi_(handle,transA,m,n,&
&alpha,c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,&
&pBuffer)
    end function

    function cusparseZgemvi_rank_0(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemvi_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      complex(c_double_complex),target :: x
      integer(c_int),target :: xInd
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseZgemvi_rank_0 = cusparseZgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseZgemvi_rank_1(handle,transA,m,n,alpha,A,lda,nnz,x,&
&xInd,beta,y,idxBase,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemvi_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int) :: nnz
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int),target,dimension(:) :: xInd
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: y
      integer(kind(CUSPARSE_INDEX_BASE_ZERO)) :: idxBase
      type(c_ptr) :: pBuffer
      !
      cusparseZgemvi_rank_1 = cusparseZgemvi_(handle,transA,m,n,alpha,&
&c_loc(A),lda,nnz,c_loc(x),c_loc(xInd),beta,c_loc(y),idxBase,pBuffer)
    end function

    function cusparseSbsrmm_full_rank(handle,dirA,transA,transB,mb,n,&
&kb,nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseSbsrmm_full_rank = cusparseSbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseSbsrmm_rank_0(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      real(c_float),target :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target :: C
      integer(c_int) :: ldc
      !
      cusparseSbsrmm_rank_0 = cusparseSbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseSbsrmm_rank_1(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseSbsrmm_rank_1 = cusparseSbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDbsrmm_full_rank(handle,dirA,transA,transB,mb,n,&
&kb,nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseDbsrmm_full_rank = cusparseDbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDbsrmm_rank_0(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      real(c_double),target :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target :: C
      integer(c_int) :: ldc
      !
      cusparseDbsrmm_rank_0 = cusparseDbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDbsrmm_rank_1(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseDbsrmm_rank_1 = cusparseDbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCbsrmm_full_rank(handle,dirA,transA,transB,mb,n,&
&kb,nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseCbsrmm_full_rank = cusparseCbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCbsrmm_rank_0(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseCbsrmm_rank_0 = cusparseCbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCbsrmm_rank_1(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseCbsrmm_rank_1 = cusparseCbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZbsrmm_full_rank(handle,dirA,transA,transB,mb,n,&
&kb,nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseZbsrmm_full_rank = cusparseZbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZbsrmm_rank_0(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseZbsrmm_rank_0 = cusparseZbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZbsrmm_rank_1(handle,dirA,transA,transB,mb,n,kb,&
&nnzb,alpha,descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: mb
      integer(c_int) :: n
      integer(c_int) :: kb
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseZbsrmm_rank_1 = cusparseZbsrmm_(handle,dirA,transA,&
&transB,mb,n,kb,nnzb,alpha,descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),&
&c_loc(bsrColIndA),blockDim,c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseScsrmm_full_rank(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseScsrmm_full_rank = cusparseScsrmm_(handle,transA,m,n,k,&
&nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseScsrmm_rank_0(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float),target :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target :: C
      integer(c_int) :: ldc
      !
      cusparseScsrmm_rank_0 = cusparseScsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseScsrmm_rank_1(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseScsrmm_rank_1 = cusparseScsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDcsrmm_full_rank(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseDcsrmm_full_rank = cusparseDcsrmm_(handle,transA,m,n,k,&
&nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDcsrmm_rank_0(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double),target :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target :: C
      integer(c_int) :: ldc
      !
      cusparseDcsrmm_rank_0 = cusparseDcsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDcsrmm_rank_1(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseDcsrmm_rank_1 = cusparseDcsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCcsrmm_full_rank(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseCcsrmm_full_rank = cusparseCcsrmm_(handle,transA,m,n,k,&
&nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCcsrmm_rank_0(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseCcsrmm_rank_0 = cusparseCcsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCcsrmm_rank_1(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseCcsrmm_rank_1 = cusparseCcsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZcsrmm_full_rank(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmm_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseZcsrmm_full_rank = cusparseZcsrmm_(handle,transA,m,n,k,&
&nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZcsrmm_rank_0(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseZcsrmm_rank_0 = cusparseZcsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZcsrmm_rank_1(handle,transA,m,n,k,nnz,alpha,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,beta,C,&
&ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseZcsrmm_rank_1 = cusparseZcsrmm_(handle,transA,m,n,k,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseScsrmm2_full_rank(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrmm2_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseScsrmm2_full_rank = cusparseScsrmm2_(handle,transA,&
&transB,m,n,k,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,beta,&
&c_loc(C),ldc)
    end function

    function cusparseScsrmm2_rank_0(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmm2_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float),target :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target :: C
      integer(c_int) :: ldc
      !
      cusparseScsrmm2_rank_0 = cusparseScsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseScsrmm2_rank_1(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrmm2_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseScsrmm2_rank_1 = cusparseScsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDcsrmm2_full_rank(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrmm2_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseDcsrmm2_full_rank = cusparseDcsrmm2_(handle,transA,&
&transB,m,n,k,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,beta,&
&c_loc(C),ldc)
    end function

    function cusparseDcsrmm2_rank_0(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmm2_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double),target :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target :: C
      integer(c_int) :: ldc
      !
      cusparseDcsrmm2_rank_0 = cusparseDcsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseDcsrmm2_rank_1(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrmm2_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseDcsrmm2_rank_1 = cusparseDcsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCcsrmm2_full_rank(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrmm2_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseCcsrmm2_full_rank = cusparseCcsrmm2_(handle,transA,&
&transB,m,n,k,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,beta,&
&c_loc(C),ldc)
    end function

    function cusparseCcsrmm2_rank_0(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmm2_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseCcsrmm2_rank_0 = cusparseCcsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseCcsrmm2_rank_1(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrmm2_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseCcsrmm2_rank_1 = cusparseCcsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZcsrmm2_full_rank(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrmm2_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseZcsrmm2_full_rank = cusparseZcsrmm2_(handle,transA,&
&transB,m,n,k,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,beta,&
&c_loc(C),ldc)
    end function

    function cusparseZcsrmm2_rank_0(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmm2_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseZcsrmm2_rank_0 = cusparseZcsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseZcsrmm2_rank_1(handle,transA,transB,m,n,k,nnz,&
&alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,B,ldb,&
&beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrmm2_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseZcsrmm2_rank_1 = cusparseZcsrmm2_(handle,transA,transB,m,&
&n,k,nnz,alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,beta,c_loc(C),ldc)
    end function

    function cusparseSbsrsm2_bufferSize_rank_0(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsrsm2_bufferSize_rank_0 = &
&cusparseSbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseSbsrsm2_bufferSize_rank_1(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsrsm2_bufferSize_rank_1 = &
&cusparseSbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseDbsrsm2_bufferSize_rank_0(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsrsm2_bufferSize_rank_0 = &
&cusparseDbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseDbsrsm2_bufferSize_rank_1(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsrsm2_bufferSize_rank_1 = &
&cusparseDbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseCbsrsm2_bufferSize_rank_0(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsrsm2_bufferSize_rank_0 = &
&cusparseCbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseCbsrsm2_bufferSize_rank_1(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsrsm2_bufferSize_rank_1 = &
&cusparseCbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseZbsrsm2_bufferSize_rank_0(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsrsm2_bufferSize_rank_0 = &
&cusparseZbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseZbsrsm2_bufferSize_rank_1(handle,dirA,transA,&
&transX,mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsrsm2_bufferSize_rank_1 = &
&cusparseZbsrsm2_bufferSize_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseSbsrsm2_analysis_rank_0(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsm2_analysis_rank_0 = &
&cusparseSbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsrsm2_analysis_rank_1(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsm2_analysis_rank_1 = &
&cusparseSbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrsm2_analysis_rank_0(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsm2_analysis_rank_0 = &
&cusparseDbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrsm2_analysis_rank_1(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsm2_analysis_rank_1 = &
&cusparseDbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrsm2_analysis_rank_0(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsm2_analysis_rank_0 = &
&cusparseCbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrsm2_analysis_rank_1(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsm2_analysis_rank_1 = &
&cusparseCbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrsm2_analysis_rank_0(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsm2_analysis_rank_0 = &
&cusparseZbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrsm2_analysis_rank_1(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,&
&blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsm2_analysis_rank_1 = &
&cusparseZbsrsm2_analysis_(handle,dirA,transA,transX,mb,nrhs,nnzb,&
&descrA,c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsrsm2_solve_full_rank(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsm2_solve_full_rank = cusparseSbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseSbsrsm2_solve_rank_0(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_float),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsm2_solve_rank_0 = cusparseSbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseSbsrsm2_solve_rank_1(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrsm2_solve_rank_1 = cusparseSbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseDbsrsm2_solve_full_rank(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsm2_solve_full_rank = cusparseDbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseDbsrsm2_solve_rank_0(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_double),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsm2_solve_rank_0 = cusparseDbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseDbsrsm2_solve_rank_1(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrsm2_solve_rank_1 = cusparseDbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseCbsrsm2_solve_full_rank(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsm2_solve_full_rank = cusparseCbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseCbsrsm2_solve_rank_0(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsm2_solve_rank_0 = cusparseCbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseCbsrsm2_solve_rank_1(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrsm2_solve_rank_1 = cusparseCbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseZbsrsm2_solve_full_rank(handle,dirA,transA,transX,&
&mb,nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsm2_solve_full_rank = cusparseZbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseZbsrsm2_solve_rank_0(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsm2_solve_rank_0 = cusparseZbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseZbsrsm2_solve_rank_1(handle,dirA,transA,transX,mb,&
&nrhs,nnzb,alpha,descrA,bsrSortedValA,bsrSortedRowPtrA,&
&bsrSortedColIndA,blockDim,myInfo,B,ldb,X,ldx,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transX
      integer(c_int) :: mb
      integer(c_int) :: nrhs
      integer(c_int) :: nnzb
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: X
      integer(c_int) :: ldx
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrsm2_solve_rank_1 = cusparseZbsrsm2_solve_(handle,&
&dirA,transA,transX,mb,nrhs,nnzb,alpha,descrA,c_loc(bsrSortedValA),&
&c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),blockDim,myInfo,&
&c_loc(B),ldb,X,ldx,policy,pBuffer)
    end function

    function cusparseScsrsm2_bufferSizeExt_full_rank(handle,algo,&
&transA,transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsrsm2_bufferSizeExt_full_rank = &
&cusparseScsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseScsrsm2_bufferSizeExt_rank_0(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsrsm2_bufferSizeExt_rank_0 = &
&cusparseScsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseScsrsm2_bufferSizeExt_rank_1(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsrsm2_bufferSizeExt_rank_1 = &
&cusparseScsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseDcsrsm2_bufferSizeExt_full_rank(handle,algo,&
&transA,transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsrsm2_bufferSizeExt_full_rank = &
&cusparseDcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseDcsrsm2_bufferSizeExt_rank_0(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsrsm2_bufferSizeExt_rank_0 = &
&cusparseDcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseDcsrsm2_bufferSizeExt_rank_1(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsrsm2_bufferSizeExt_rank_1 = &
&cusparseDcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseCcsrsm2_bufferSizeExt_full_rank(handle,algo,&
&transA,transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsrsm2_bufferSizeExt_full_rank = &
&cusparseCcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseCcsrsm2_bufferSizeExt_rank_0(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsrsm2_bufferSizeExt_rank_0 = &
&cusparseCcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseCcsrsm2_bufferSizeExt_rank_1(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsrsm2_bufferSizeExt_rank_1 = &
&cusparseCcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseZcsrsm2_bufferSizeExt_full_rank(handle,algo,&
&transA,transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsrsm2_bufferSizeExt_full_rank = &
&cusparseZcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseZcsrsm2_bufferSizeExt_rank_0(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsrsm2_bufferSizeExt_rank_0 = &
&cusparseZcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseZcsrsm2_bufferSizeExt_rank_1(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsrsm2_bufferSizeExt_rank_1 = &
&cusparseZcsrsm2_bufferSizeExt_(handle,algo,transA,transB,m,nrhs,nnz,&
&alpha,descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBufferSize)
    end function

    function cusparseScsrsm2_analysis_full_rank(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_analysis_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsm2_analysis_full_rank = &
&cusparseScsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseScsrsm2_analysis_rank_0(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsm2_analysis_rank_0 = &
&cusparseScsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseScsrsm2_analysis_rank_1(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsm2_analysis_rank_1 = &
&cusparseScsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseDcsrsm2_analysis_full_rank(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_analysis_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsm2_analysis_full_rank = &
&cusparseDcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseDcsrsm2_analysis_rank_0(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsm2_analysis_rank_0 = &
&cusparseDcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseDcsrsm2_analysis_rank_1(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsm2_analysis_rank_1 = &
&cusparseDcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseCcsrsm2_analysis_full_rank(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_analysis_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsm2_analysis_full_rank = &
&cusparseCcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseCcsrsm2_analysis_rank_0(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsm2_analysis_rank_0 = &
&cusparseCcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseCcsrsm2_analysis_rank_1(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsm2_analysis_rank_1 = &
&cusparseCcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseZcsrsm2_analysis_full_rank(handle,algo,transA,&
&transB,m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_analysis_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsm2_analysis_full_rank = &
&cusparseZcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseZcsrsm2_analysis_rank_0(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsm2_analysis_rank_0 = &
&cusparseZcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseZcsrsm2_analysis_rank_1(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsm2_analysis_rank_1 = &
&cusparseZcsrsm2_analysis_(handle,algo,transA,transB,m,nrhs,nnz,alpha,&
&descrA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,policy,pBuffer)
    end function

    function cusparseScsrsm2_solve_full_rank(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsm2_solve_full_rank = cusparseScsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseScsrsm2_solve_rank_0(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsm2_solve_rank_0 = cusparseScsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseScsrsm2_solve_rank_1(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrsm2_solve_rank_1 = cusparseScsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseDcsrsm2_solve_full_rank(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsm2_solve_full_rank = cusparseDcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseDcsrsm2_solve_rank_0(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsm2_solve_rank_0 = cusparseDcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseDcsrsm2_solve_rank_1(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrsm2_solve_rank_1 = cusparseDcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseCcsrsm2_solve_full_rank(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsm2_solve_full_rank = cusparseCcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseCcsrsm2_solve_rank_0(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsm2_solve_rank_0 = cusparseCcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseCcsrsm2_solve_rank_1(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrsm2_solve_rank_1 = cusparseCcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseZcsrsm2_solve_full_rank(handle,algo,transA,transB,&
&m,nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,&
&csrSortedColIndA,B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_solve_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsm2_solve_full_rank = cusparseZcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseZcsrsm2_solve_rank_0(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_solve_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsm2_solve_rank_0 = cusparseZcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseZcsrsm2_solve_rank_1(handle,algo,transA,transB,m,&
&nrhs,nnz,alpha,descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,&
&B,ldb,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrsm2_solve_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: algo
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: nrhs
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrsm2_solve_rank_1 = cusparseZcsrsm2_solve_(handle,&
&algo,transA,transB,m,nrhs,nnz,alpha,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),c_loc(B),ldb,myInfo,&
&policy,pBuffer)
    end function

    function cusparseSgemmi_full_rank(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemmi_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      real(c_float),target,dimension(:,:) :: A
      integer(c_int) :: lda
      real(c_float),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      real(c_float) :: beta
      real(c_float),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseSgemmi_full_rank = cusparseSgemmi_(handle,m,n,k,nnz,&
&alpha,c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),&
&beta,c_loc(C),ldc)
    end function

    function cusparseSgemmi_rank_0(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemmi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      real(c_float),target :: A
      integer(c_int) :: lda
      real(c_float),target :: cscValB
      integer(c_int),target :: cscColPtrB
      integer(c_int),target :: cscRowIndB
      real(c_float) :: beta
      real(c_float),target :: C
      integer(c_int) :: ldc
      !
      cusparseSgemmi_rank_0 = cusparseSgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseSgemmi_rank_1(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgemmi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_float) :: alpha
      real(c_float),target,dimension(:) :: A
      integer(c_int) :: lda
      real(c_float),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      real(c_float) :: beta
      real(c_float),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseSgemmi_rank_1 = cusparseSgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseDgemmi_full_rank(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemmi_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      real(c_double),target,dimension(:,:) :: A
      integer(c_int) :: lda
      real(c_double),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      real(c_double) :: beta
      real(c_double),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseDgemmi_full_rank = cusparseDgemmi_(handle,m,n,k,nnz,&
&alpha,c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),&
&beta,c_loc(C),ldc)
    end function

    function cusparseDgemmi_rank_0(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemmi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      real(c_double),target :: A
      integer(c_int) :: lda
      real(c_double),target :: cscValB
      integer(c_int),target :: cscColPtrB
      integer(c_int),target :: cscRowIndB
      real(c_double) :: beta
      real(c_double),target :: C
      integer(c_int) :: ldc
      !
      cusparseDgemmi_rank_0 = cusparseDgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseDgemmi_rank_1(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgemmi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      real(c_double) :: alpha
      real(c_double),target,dimension(:) :: A
      integer(c_int) :: lda
      real(c_double),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      real(c_double) :: beta
      real(c_double),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseDgemmi_rank_1 = cusparseDgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseCgemmi_full_rank(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemmi_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:,:) :: A
      integer(c_int) :: lda
      complex(c_float_complex),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseCgemmi_full_rank = cusparseCgemmi_(handle,m,n,k,nnz,&
&alpha,c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),&
&beta,c_loc(C),ldc)
    end function

    function cusparseCgemmi_rank_0(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemmi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target :: A
      integer(c_int) :: lda
      complex(c_float_complex),target :: cscValB
      integer(c_int),target :: cscColPtrB
      integer(c_int),target :: cscRowIndB
      complex(c_float_complex) :: beta
      complex(c_float_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseCgemmi_rank_0 = cusparseCgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseCgemmi_rank_1(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgemmi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:) :: A
      integer(c_int) :: lda
      complex(c_float_complex),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      complex(c_float_complex) :: beta
      complex(c_float_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseCgemmi_rank_1 = cusparseCgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseZgemmi_full_rank(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemmi_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:,:) :: A
      integer(c_int) :: lda
      complex(c_double_complex),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:,:) :: C
      integer(c_int) :: ldc
      !
      cusparseZgemmi_full_rank = cusparseZgemmi_(handle,m,n,k,nnz,&
&alpha,c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),&
&beta,c_loc(C),ldc)
    end function

    function cusparseZgemmi_rank_0(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemmi_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target :: A
      integer(c_int) :: lda
      complex(c_double_complex),target :: cscValB
      integer(c_int),target :: cscColPtrB
      integer(c_int),target :: cscRowIndB
      complex(c_double_complex) :: beta
      complex(c_double_complex),target :: C
      integer(c_int) :: ldc
      !
      cusparseZgemmi_rank_0 = cusparseZgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseZgemmi_rank_1(handle,m,n,k,nnz,alpha,A,lda,&
&cscValB,cscColPtrB,cscRowIndB,beta,C,ldc)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgemmi_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      integer(c_int) :: nnz
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:) :: A
      integer(c_int) :: lda
      complex(c_double_complex),target,dimension(:) :: cscValB
      integer(c_int),target,dimension(:) :: cscColPtrB
      integer(c_int),target,dimension(:) :: cscRowIndB
      complex(c_double_complex) :: beta
      complex(c_double_complex),target,dimension(:) :: C
      integer(c_int) :: ldc
      !
      cusparseZgemmi_rank_1 = cusparseZgemmi_(handle,m,n,k,nnz,alpha,&
&c_loc(A),lda,c_loc(cscValB),c_loc(cscColPtrB),c_loc(cscRowIndB),beta,&
&c_loc(C),ldc)
    end function

    function cusparseXcsrgeamNnz_rank_0(handle,m,n,descrA,nnzA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,descrC,&
&csrRowPtrC,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgeamNnz_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      integer(c_int),target :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseXcsrgeamNnz_rank_0 = cusparseXcsrgeamNnz_(handle,m,n,&
&descrA,nnzA,c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,&
&c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,c_loc(csrRowPtrC),&
&nnzTotalDevHostPtr)
    end function

    function cusparseXcsrgeamNnz_rank_1(handle,m,n,descrA,nnzA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,descrC,&
&csrRowPtrC,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgeamNnz_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseXcsrgeamNnz_rank_1 = cusparseXcsrgeamNnz_(handle,m,n,&
&descrA,nnzA,c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,&
&c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,c_loc(csrRowPtrC),&
&nnzTotalDevHostPtr)
    end function

    function cusparseScsrgeam_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgeam_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      real(c_float) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      real(c_float),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseScsrgeam_rank_0 = cusparseScsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseScsrgeam_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgeam_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      real(c_float) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      real(c_float),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseScsrgeam_rank_1 = cusparseScsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseDcsrgeam_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgeam_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      real(c_double) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      real(c_double),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseDcsrgeam_rank_0 = cusparseDcsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseDcsrgeam_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgeam_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      real(c_double) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      real(c_double),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseDcsrgeam_rank_1 = cusparseDcsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseCcsrgeam_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgeam_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseCcsrgeam_rank_0 = cusparseCcsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseCcsrgeam_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgeam_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseCcsrgeam_rank_1 = cusparseCcsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseZcsrgeam_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgeam_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseZcsrgeam_rank_0 = cusparseZcsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseZcsrgeam_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,beta,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgeam_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseZcsrgeam_rank_1 = cusparseZcsrgeam_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),beta,&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseScsrgeam2_bufferSizeExt_rank_0(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrgeam2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_float),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseScsrgeam2_bufferSizeExt_rank_0 = &
&cusparseScsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseScsrgeam2_bufferSizeExt_rank_1(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrgeam2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_float),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseScsrgeam2_bufferSizeExt_rank_1 = &
&cusparseScsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseDcsrgeam2_bufferSizeExt_rank_0(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrgeam2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_double),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDcsrgeam2_bufferSizeExt_rank_0 = &
&cusparseDcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseDcsrgeam2_bufferSizeExt_rank_1(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrgeam2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_double),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDcsrgeam2_bufferSizeExt_rank_1 = &
&cusparseDcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseCcsrgeam2_bufferSizeExt_rank_0(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrgeam2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCcsrgeam2_bufferSizeExt_rank_0 = &
&cusparseCcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseCcsrgeam2_bufferSizeExt_rank_1(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrgeam2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCcsrgeam2_bufferSizeExt_rank_1 = &
&cusparseCcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseZcsrgeam2_bufferSizeExt_rank_0(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrgeam2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZcsrgeam2_bufferSizeExt_rank_0 = &
&cusparseZcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseZcsrgeam2_bufferSizeExt_rank_1(handle,m,n,alpha,&
&descrA,nnzA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,&
&descrB,nnzB,csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,&
&csrSortedValC,csrSortedRowPtrC,csrSortedColIndC,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrgeam2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZcsrgeam2_bufferSizeExt_rank_1 = &
&cusparseZcsrgeam2_bufferSizeExt_(handle,m,n,alpha,descrA,nnzA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&beta,descrB,nnzB,c_loc(csrSortedValB),c_loc(csrSortedRowPtrB),c_l&
&oc(csrSortedColIndB),descrC,c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),&
&c_loc(csrSortedColIndC),pBufferSizeInBytes)
    end function

    function cusparseXcsrgeam2Nnz_rank_0(handle,m,n,descrA,nnzA,&
&csrSortedRowPtrA,csrSortedColIndA,descrB,nnzB,csrSortedRowPtrB,&
&csrSortedColIndB,descrC,csrSortedRowPtrC,nnzTotalDevHostPtr,&
&workspace)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgeam2Nnz_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr) :: workspace
      !
      cusparseXcsrgeam2Nnz_rank_0 = cusparseXcsrgeam2Nnz_(handle,m,n,&
&descrA,nnzA,c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),descrB,&
&nnzB,c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedRowPtrC),nnzTotalDevHostPtr,workspace)
    end function

    function cusparseXcsrgeam2Nnz_rank_1(handle,m,n,descrA,nnzA,&
&csrSortedRowPtrA,csrSortedColIndA,descrB,nnzB,csrSortedRowPtrB,&
&csrSortedColIndB,descrC,csrSortedRowPtrC,nnzTotalDevHostPtr,&
&workspace)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgeam2Nnz_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr) :: workspace
      !
      cusparseXcsrgeam2Nnz_rank_1 = cusparseXcsrgeam2Nnz_(handle,m,n,&
&descrA,nnzA,c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),descrB,&
&nnzB,c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedRowPtrC),nnzTotalDevHostPtr,workspace)
    end function

    function cusparseScsrgeam2_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgeam2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_float) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_float),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseScsrgeam2_rank_0 = cusparseScsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseScsrgeam2_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgeam2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_float) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_float),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseScsrgeam2_rank_1 = cusparseScsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseDcsrgeam2_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgeam2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      real(c_double) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_double),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrgeam2_rank_0 = cusparseDcsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseDcsrgeam2_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgeam2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      real(c_double) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      real(c_double),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrgeam2_rank_1 = cusparseDcsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseCcsrgeam2_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgeam2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrgeam2_rank_0 = cusparseCcsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseCcsrgeam2_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgeam2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrgeam2_rank_1 = cusparseCcsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseZcsrgeam2_rank_0(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgeam2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target :: csrSortedValB
      integer(c_int),target :: csrSortedRowPtrB
      integer(c_int),target :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target :: csrSortedValC
      integer(c_int),target :: csrSortedRowPtrC
      integer(c_int),target :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrgeam2_rank_0 = cusparseZcsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseZcsrgeam2_rank_1(handle,m,n,alpha,descrA,nnzA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,beta,descrB,nnzB,&
&csrSortedValB,csrSortedRowPtrB,csrSortedColIndB,descrC,csrSortedValC,&
&csrSortedRowPtrC,csrSortedColIndC,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgeam2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target,dimension(:) :: csrSortedValB
      integer(c_int),target,dimension(:) :: csrSortedRowPtrB
      integer(c_int),target,dimension(:) :: csrSortedColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target,dimension(:) :: csrSortedValC
      integer(c_int),target,dimension(:) :: csrSortedRowPtrC
      integer(c_int),target,dimension(:) :: csrSortedColIndC
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrgeam2_rank_1 = cusparseZcsrgeam2_(handle,m,n,alpha,&
&descrA,nnzA,c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),beta,descrB,nnzB,c_loc(csrSortedValB),&
&c_loc(csrSortedRowPtrB),c_loc(csrSortedColIndB),descrC,&
&c_loc(csrSortedValC),c_loc(csrSortedRowPtrC),c_loc(csrSortedColIndC),pBuffer)
    end function

    function cusparseXcsrgemmNnz_rank_0(handle,transA,transB,m,n,k,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&descrC,csrRowPtrC,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgemmNnz_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      integer(c_int),target :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseXcsrgemmNnz_rank_0 = cusparseXcsrgemmNnz_(handle,transA,&
&transB,m,n,k,descrA,nnzA,c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,c_loc(csrRowPtrC),&
&nnzTotalDevHostPtr)
    end function

    function cusparseXcsrgemmNnz_rank_1(handle,transA,transB,m,n,k,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&descrC,csrRowPtrC,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgemmNnz_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseXcsrgemmNnz_rank_1 = cusparseXcsrgemmNnz_(handle,transA,&
&transB,m,n,k,descrA,nnzA,c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,c_loc(csrRowPtrC),&
&nnzTotalDevHostPtr)
    end function

    function cusparseScsrgemm_rank_0(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgemm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      real(c_float),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseScsrgemm_rank_0 = cusparseScsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseScsrgemm_rank_1(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgemm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      real(c_float),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseScsrgemm_rank_1 = cusparseScsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseDcsrgemm_rank_0(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgemm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      real(c_double),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseDcsrgemm_rank_0 = cusparseDcsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseDcsrgemm_rank_1(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgemm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      real(c_double),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseDcsrgemm_rank_1 = cusparseDcsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseCcsrgemm_rank_0(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgemm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseCcsrgemm_rank_0 = cusparseCcsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseCcsrgemm_rank_1(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgemm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_float_complex),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseCcsrgemm_rank_1 = cusparseCcsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseZcsrgemm_rank_0(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgemm_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      !
      cusparseZcsrgemm_rank_0 = cusparseZcsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseZcsrgemm_rank_1(handle,transA,transB,m,n,k,descrA,&
&nnzA,csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,descrC,csrValC,csrRowPtrC,csrColIndC)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgemm_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transA
      integer(kind(CUSPARSE_OPERATION_NON_TRANSPOSE)) :: transB
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrC
      complex(c_double_complex),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      !
      cusparseZcsrgemm_rank_1 = cusparseZcsrgemm_(handle,transA,transB,&
&m,n,k,descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),&
&descrB,nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC))
    end function

    function cusparseScsrgemm2_bufferSizeExt_rank_0(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrgemm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      real(c_float) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseScsrgemm2_bufferSizeExt_rank_0 = &
&cusparseScsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseScsrgemm2_bufferSizeExt_rank_1(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrgemm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      real(c_float) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseScsrgemm2_bufferSizeExt_rank_1 = &
&cusparseScsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseDcsrgemm2_bufferSizeExt_rank_0(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrgemm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      real(c_double) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDcsrgemm2_bufferSizeExt_rank_0 = &
&cusparseDcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseDcsrgemm2_bufferSizeExt_rank_1(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrgemm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      real(c_double) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDcsrgemm2_bufferSizeExt_rank_1 = &
&cusparseDcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseCcsrgemm2_bufferSizeExt_rank_0(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrgemm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCcsrgemm2_bufferSizeExt_rank_0 = &
&cusparseCcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseCcsrgemm2_bufferSizeExt_rank_1(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrgemm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCcsrgemm2_bufferSizeExt_rank_1 = &
&cusparseCcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseZcsrgemm2_bufferSizeExt_rank_0(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrgemm2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZcsrgemm2_bufferSizeExt_rank_0 = &
&cusparseZcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseZcsrgemm2_bufferSizeExt_rank_1(handle,m,n,k,alpha,&
&descrA,nnzA,csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,&
&beta,descrD,nnzD,csrRowPtrD,csrColIndD,myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrgemm2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZcsrgemm2_bufferSizeExt_rank_1 = &
&cusparseZcsrgemm2_bufferSizeExt_(handle,m,n,k,alpha,descrA,nnzA,&
&c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,c_loc(csrRowPtrB),&
&c_loc(csrColIndB),beta,descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),myInfo,pBufferSizeInBytes)
    end function

    function cusparseXcsrgemm2Nnz_rank_0(handle,m,n,k,descrA,nnzA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,descrD,nnzD,&
&csrRowPtrD,csrColIndD,descrC,csrRowPtrC,nnzTotalDevHostPtr,myInfo,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgemm2Nnz_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: descrC
      integer(c_int),target :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseXcsrgemm2Nnz_rank_0 = cusparseXcsrgemm2Nnz_(handle,m,n,k,&
&descrA,nnzA,c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,&
&c_loc(csrRowPtrB),c_loc(csrColIndB),descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),descrC,c_loc(csrRowPtrC),nnzTotalDevHostPtr,myInfo,&
&pBuffer)
    end function

    function cusparseXcsrgemm2Nnz_rank_1(handle,m,n,k,descrA,nnzA,&
&csrRowPtrA,csrColIndA,descrB,nnzB,csrRowPtrB,csrColIndB,descrD,nnzD,&
&csrRowPtrD,csrColIndD,descrC,csrRowPtrC,nnzTotalDevHostPtr,myInfo,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseXcsrgemm2Nnz_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: descrC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int) :: nnzTotalDevHostPtr
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseXcsrgemm2Nnz_rank_1 = cusparseXcsrgemm2Nnz_(handle,m,n,k,&
&descrA,nnzA,c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,nnzB,&
&c_loc(csrRowPtrB),c_loc(csrColIndB),descrD,nnzD,c_loc(csrRowPtrD),&
&c_loc(csrColIndD),descrC,c_loc(csrRowPtrC),nnzTotalDevHostPtr,myInfo,&
&pBuffer)
    end function

    function cusparseScsrgemm2_rank_0(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgemm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      real(c_float) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      real(c_float),target :: csrValD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: descrC
      real(c_float),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseScsrgemm2_rank_0 = cusparseScsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseScsrgemm2_rank_1(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrgemm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_float) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_float),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_float),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      real(c_float) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      real(c_float),target,dimension(:) :: csrValD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: descrC
      real(c_float),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseScsrgemm2_rank_1 = cusparseScsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseDcsrgemm2_rank_0(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgemm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      real(c_double) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      real(c_double),target :: csrValD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: descrC
      real(c_double),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrgemm2_rank_0 = cusparseDcsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseDcsrgemm2_rank_1(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrgemm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      real(c_double) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      real(c_double),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      real(c_double),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      real(c_double) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      real(c_double),target,dimension(:) :: csrValD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: descrC
      real(c_double),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrgemm2_rank_1 = cusparseDcsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseCcsrgemm2_rank_0(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgemm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      complex(c_float_complex),target :: csrValD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: descrC
      complex(c_float_complex),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrgemm2_rank_0 = cusparseCcsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseCcsrgemm2_rank_1(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrgemm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_float_complex),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_float_complex),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      complex(c_float_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      complex(c_float_complex),target,dimension(:) :: csrValD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: descrC
      complex(c_float_complex),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrgemm2_rank_1 = cusparseCcsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseZcsrgemm2_rank_0(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgemm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target :: csrValA
      integer(c_int),target :: csrRowPtrA
      integer(c_int),target :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target :: csrValB
      integer(c_int),target :: csrRowPtrB
      integer(c_int),target :: csrColIndB
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      complex(c_double_complex),target :: csrValD
      integer(c_int),target :: csrRowPtrD
      integer(c_int),target :: csrColIndD
      type(c_ptr) :: descrC
      complex(c_double_complex),target :: csrValC
      integer(c_int),target :: csrRowPtrC
      integer(c_int),target :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrgemm2_rank_0 = cusparseZcsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseZcsrgemm2_rank_1(handle,m,n,k,alpha,descrA,nnzA,&
&csrValA,csrRowPtrA,csrColIndA,descrB,nnzB,csrValB,csrRowPtrB,&
&csrColIndB,beta,descrD,nnzD,csrValD,csrRowPtrD,csrColIndD,descrC,&
&csrValC,csrRowPtrC,csrColIndC,myInfo,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrgemm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      integer(c_int) :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: descrA
      integer(c_int) :: nnzA
      complex(c_double_complex),target,dimension(:) :: csrValA
      integer(c_int),target,dimension(:) :: csrRowPtrA
      integer(c_int),target,dimension(:) :: csrColIndA
      type(c_ptr) :: descrB
      integer(c_int) :: nnzB
      complex(c_double_complex),target,dimension(:) :: csrValB
      integer(c_int),target,dimension(:) :: csrRowPtrB
      integer(c_int),target,dimension(:) :: csrColIndB
      complex(c_double_complex) :: beta
      type(c_ptr) :: descrD
      integer(c_int) :: nnzD
      complex(c_double_complex),target,dimension(:) :: csrValD
      integer(c_int),target,dimension(:) :: csrRowPtrD
      integer(c_int),target,dimension(:) :: csrColIndD
      type(c_ptr) :: descrC
      complex(c_double_complex),target,dimension(:) :: csrValC
      integer(c_int),target,dimension(:) :: csrRowPtrC
      integer(c_int),target,dimension(:) :: csrColIndC
      type(c_ptr) :: myInfo
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrgemm2_rank_1 = cusparseZcsrgemm2_(handle,m,n,k,alpha,&
&descrA,nnzA,c_loc(csrValA),c_loc(csrRowPtrA),c_loc(csrColIndA),descrB,&
&nnzB,c_loc(csrValB),c_loc(csrRowPtrB),c_loc(csrColIndB),beta,descrD,&
&nnzD,c_loc(csrValD),c_loc(csrRowPtrD),c_loc(csrColIndD),descrC,&
&c_loc(csrValC),c_loc(csrRowPtrC),c_loc(csrColIndC),myInfo,pBuffer)
    end function

    function cusparseSbsrilu02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsrilu02_bufferSize_rank_0 = &
&cusparseSbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseSbsrilu02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsrilu02_bufferSize_rank_1 = &
&cusparseSbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseDbsrilu02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsrilu02_bufferSize_rank_0 = &
&cusparseDbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseDbsrilu02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsrilu02_bufferSize_rank_1 = &
&cusparseDbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseCbsrilu02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsrilu02_bufferSize_rank_0 = &
&cusparseCbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseCbsrilu02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsrilu02_bufferSize_rank_1 = &
&cusparseCbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseZbsrilu02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsrilu02_bufferSize_rank_0 = &
&cusparseZbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseZbsrilu02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsrilu02_bufferSize_rank_1 = &
&cusparseZbsrilu02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,pBufferSizeInBytes)
    end function

    function cusparseSbsrilu02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrilu02_analysis_rank_0 = &
&cusparseSbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsrilu02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrilu02_analysis_rank_1 = &
&cusparseSbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrilu02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrilu02_analysis_rank_0 = &
&cusparseDbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrilu02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrilu02_analysis_rank_1 = &
&cusparseDbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrilu02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrilu02_analysis_rank_0 = &
&cusparseCbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrilu02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrilu02_analysis_rank_1 = &
&cusparseCbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrilu02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrilu02_analysis_rank_0 = &
&cusparseZbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrilu02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrSortedValA,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,&
&myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrSortedValA
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrilu02_analysis_rank_1 = &
&cusparseZbsrilu02_analysis_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrSortedValA),c_loc(bsrSortedRowPtrA),c_loc(bsrSortedColIndA),&
&blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsrilu02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrilu02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrSortedValA_valM
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrilu02_rank_0 = cusparseSbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsrilu02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsrilu02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrSortedValA_valM
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsrilu02_rank_1 = cusparseSbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrilu02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrilu02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrSortedValA_valM
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrilu02_rank_0 = cusparseDbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsrilu02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsrilu02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrSortedValA_valM
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsrilu02_rank_1 = cusparseDbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrilu02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrilu02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrSortedValA_valM
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrilu02_rank_0 = cusparseCbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsrilu02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsrilu02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrSortedValA_valM
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsrilu02_rank_1 = cusparseCbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrilu02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrilu02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrSortedValA_valM
      integer(c_int),target :: bsrSortedRowPtrA
      integer(c_int),target :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrilu02_rank_0 = cusparseZbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsrilu02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrSortedValA_valM,bsrSortedRowPtrA,bsrSortedColIndA,blockDim,myInfo,&
&policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsrilu02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: &
&bsrSortedValA_valM
      integer(c_int),target,dimension(:) :: bsrSortedRowPtrA
      integer(c_int),target,dimension(:) :: bsrSortedColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsrilu02_rank_1 = cusparseZbsrilu02_(handle,dirA,mb,&
&nnzb,descrA,c_loc(bsrSortedValA_valM),c_loc(bsrSortedRowPtrA),&
&c_loc(bsrSortedColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseScsrilu02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseScsrilu02_bufferSize_rank_0 = &
&cusparseScsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseScsrilu02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseScsrilu02_bufferSize_rank_1 = &
&cusparseScsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseDcsrilu02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDcsrilu02_bufferSize_rank_0 = &
&cusparseDcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseDcsrilu02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDcsrilu02_bufferSize_rank_1 = &
&cusparseDcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseCcsrilu02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCcsrilu02_bufferSize_rank_0 = &
&cusparseCcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseCcsrilu02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCcsrilu02_bufferSize_rank_1 = &
&cusparseCcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseZcsrilu02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZcsrilu02_bufferSize_rank_0 = &
&cusparseZcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseZcsrilu02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZcsrilu02_bufferSize_rank_1 = &
&cusparseZcsrilu02_bufferSize_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSizeInBytes)
    end function

    function cusparseScsrilu02_bufferSizeExt_rank_0(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsrilu02_bufferSizeExt_rank_0 = &
&cusparseScsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseScsrilu02_bufferSizeExt_rank_1(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsrilu02_bufferSizeExt_rank_1 = &
&cusparseScsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseDcsrilu02_bufferSizeExt_rank_0(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsrilu02_bufferSizeExt_rank_0 = &
&cusparseDcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseDcsrilu02_bufferSizeExt_rank_1(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsrilu02_bufferSizeExt_rank_1 = &
&cusparseDcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseCcsrilu02_bufferSizeExt_rank_0(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsrilu02_bufferSizeExt_rank_0 = &
&cusparseCcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseCcsrilu02_bufferSizeExt_rank_1(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsrilu02_bufferSizeExt_rank_1 = &
&cusparseCcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseZcsrilu02_bufferSizeExt_rank_0(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsrilu02_bufferSizeExt_rank_0 = &
&cusparseZcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseZcsrilu02_bufferSizeExt_rank_1(handle,m,nnz,&
&descrA,csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsrilu02_bufferSizeExt_rank_1 = &
&cusparseZcsrilu02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseScsrilu02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrilu02_analysis_rank_0 = &
&cusparseScsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseScsrilu02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrilu02_analysis_rank_1 = &
&cusparseScsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseDcsrilu02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrilu02_analysis_rank_0 = &
&cusparseDcsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseDcsrilu02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrilu02_analysis_rank_1 = &
&cusparseDcsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseCcsrilu02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrilu02_analysis_rank_0 = &
&cusparseCcsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseCcsrilu02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrilu02_analysis_rank_1 = &
&cusparseCcsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseZcsrilu02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrilu02_analysis_rank_0 = &
&cusparseZcsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseZcsrilu02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsrilu02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrilu02_analysis_rank_1 = &
&cusparseZcsrilu02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseScsrilu02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrilu02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrilu02_rank_0 = cusparseScsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseScsrilu02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsrilu02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsrilu02_rank_1 = cusparseScsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseDcsrilu02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrilu02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrilu02_rank_0 = cusparseDcsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseDcsrilu02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsrilu02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsrilu02_rank_1 = cusparseDcsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseCcsrilu02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrilu02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrilu02_rank_0 = cusparseCcsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseCcsrilu02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsrilu02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsrilu02_rank_1 = cusparseCcsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseZcsrilu02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrilu02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrilu02_rank_0 = cusparseZcsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseZcsrilu02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsrilu02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: &
&csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsrilu02_rank_1 = cusparseZcsrilu02_(handle,m,nnz,&
&descrA,c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseSbsric02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsric02_bufferSize_rank_0 = &
&cusparseSbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseSbsric02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseSbsric02_bufferSize_rank_1 = &
&cusparseSbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseDbsric02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsric02_bufferSize_rank_0 = &
&cusparseDbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseDbsric02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDbsric02_bufferSize_rank_1 = &
&cusparseDbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseCbsric02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsric02_bufferSize_rank_0 = &
&cusparseCbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseCbsric02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCbsric02_bufferSize_rank_1 = &
&cusparseCbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseZbsric02_bufferSize_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsric02_bufferSize_rank_0 = &
&cusparseZbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseZbsric02_bufferSize_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZbsric02_bufferSize_rank_1 = &
&cusparseZbsric02_bufferSize_(handle,dirA,mb,nnzb,descrA,&
&c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseSbsric02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsric02_analysis_rank_0 = &
&cusparseSbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsric02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSbsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsric02_analysis_rank_1 = &
&cusparseSbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsric02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsric02_analysis_rank_0 = &
&cusparseDbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseDbsric02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDbsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsric02_analysis_rank_1 = &
&cusparseDbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsric02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsric02_analysis_rank_0 = &
&cusparseCbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseCbsric02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCbsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsric02_analysis_rank_1 = &
&cusparseCbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsric02_analysis_rank_0(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsric02_analysis_rank_0 = &
&cusparseZbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseZbsric02_analysis_rank_1(handle,dirA,mb,nnzb,&
&descrA,bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZbsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsric02_analysis_rank_1 = &
&cusparseZbsric02_analysis_(handle,dirA,mb,nnzb,descrA,c_loc(bsrValA),&
&c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,myInfo,policy,pBuffer)
    end function

    function cusparseSbsric02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsric02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsric02_rank_0 = cusparseSbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseSbsric02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSbsric02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseSbsric02_rank_1 = cusparseSbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseDbsric02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsric02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsric02_rank_0 = cusparseDbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseDbsric02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDbsric02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDbsric02_rank_1 = cusparseDbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseCbsric02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsric02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsric02_rank_0 = cusparseCbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseCbsric02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCbsric02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCbsric02_rank_1 = cusparseCbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseZbsric02_rank_0(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsric02_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: bsrValA
      integer(c_int),target :: bsrRowPtrA
      integer(c_int),target :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsric02_rank_0 = cusparseZbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseZbsric02_rank_1(handle,dirA,mb,nnzb,descrA,&
&bsrValA,bsrRowPtrA,bsrColIndA,blockDim,myInfo,policy,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZbsric02_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: mb
      integer(c_int) :: nnzb
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: bsrValA
      integer(c_int),target,dimension(:) :: bsrRowPtrA
      integer(c_int),target,dimension(:) :: bsrColIndA
      integer(c_int) :: blockDim
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZbsric02_rank_1 = cusparseZbsric02_(handle,dirA,mb,nnzb,&
&descrA,c_loc(bsrValA),c_loc(bsrRowPtrA),c_loc(bsrColIndA),blockDim,&
&myInfo,policy,pBuffer)
    end function

    function cusparseScsric02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseScsric02_bufferSize_rank_0 = &
&cusparseScsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseScsric02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseScsric02_bufferSize_rank_1 = &
&cusparseScsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseDcsric02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDcsric02_bufferSize_rank_0 = &
&cusparseDcsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseDcsric02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseDcsric02_bufferSize_rank_1 = &
&cusparseDcsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseCcsric02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCcsric02_bufferSize_rank_0 = &
&cusparseCcsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseCcsric02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseCcsric02_bufferSize_rank_1 = &
&cusparseCcsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseZcsric02_bufferSize_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_bufferSize_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZcsric02_bufferSize_rank_0 = &
&cusparseZcsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseZcsric02_bufferSize_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,&
&pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_bufferSize_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_int) :: pBufferSizeInBytes
      !
      cusparseZcsric02_bufferSize_rank_1 = &
&cusparseZcsric02_bufferSize_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,&
&pBufferSizeInBytes)
    end function

    function cusparseScsric02_bufferSizeExt_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsric02_bufferSizeExt_rank_0 = &
&cusparseScsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseScsric02_bufferSizeExt_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseScsric02_bufferSizeExt_rank_1 = &
&cusparseScsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseDcsric02_bufferSizeExt_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsric02_bufferSizeExt_rank_0 = &
&cusparseDcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseDcsric02_bufferSizeExt_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseDcsric02_bufferSizeExt_rank_1 = &
&cusparseDcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseCcsric02_bufferSizeExt_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsric02_bufferSizeExt_rank_0 = &
&cusparseCcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseCcsric02_bufferSizeExt_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseCcsric02_bufferSizeExt_rank_1 = &
&cusparseCcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseZcsric02_bufferSizeExt_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsric02_bufferSizeExt_rank_0 = &
&cusparseZcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseZcsric02_bufferSizeExt_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,pBufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(c_size_t) :: pBufferSize
      !
      cusparseZcsric02_bufferSizeExt_rank_1 = &
&cusparseZcsric02_bufferSizeExt_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA),c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),&
&myInfo,pBufferSize)
    end function

    function cusparseScsric02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsric02_analysis_rank_0 = &
&cusparseScsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseScsric02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseScsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsric02_analysis_rank_1 = &
&cusparseScsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseDcsric02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsric02_analysis_rank_0 = &
&cusparseDcsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseDcsric02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDcsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsric02_analysis_rank_1 = &
&cusparseDcsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseCcsric02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsric02_analysis_rank_0 = &
&cusparseCcsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseCcsric02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCcsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsric02_analysis_rank_1 = &
&cusparseCcsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseZcsric02_analysis_rank_0(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_analysis_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsric02_analysis_rank_0 = &
&cusparseZcsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseZcsric02_analysis_rank_1(handle,m,nnz,descrA,&
&csrSortedValA,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZcsric02_analysis_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: csrSortedValA
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsric02_analysis_rank_1 = &
&cusparseZcsric02_analysis_(handle,m,nnz,descrA,c_loc(csrSortedValA),&
&c_loc(csrSortedRowPtrA),c_loc(csrSortedColIndA),myInfo,policy,&
&pBuffer)
    end function

    function cusparseScsric02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsric02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsric02_rank_0 = cusparseScsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseScsric02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseScsric02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseScsric02_rank_1 = cusparseScsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseDcsric02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsric02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsric02_rank_0 = cusparseDcsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseDcsric02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDcsric02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseDcsric02_rank_1 = cusparseDcsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseCcsric02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsric02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsric02_rank_0 = cusparseCcsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseCcsric02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCcsric02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseCcsric02_rank_1 = cusparseCcsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseZcsric02_rank_0(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsric02_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: csrSortedValA_valM
      integer(c_int),target :: csrSortedRowPtrA
      integer(c_int),target :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsric02_rank_0 = cusparseZcsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseZcsric02_rank_1(handle,m,nnz,descrA,&
&csrSortedValA_valM,csrSortedRowPtrA,csrSortedColIndA,myInfo,policy,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZcsric02_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: nnz
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: &
&csrSortedValA_valM
      integer(c_int),target,dimension(:) :: csrSortedRowPtrA
      integer(c_int),target,dimension(:) :: csrSortedColIndA
      type(c_ptr) :: myInfo
      integer(kind(CUSPARSE_SOLVE_POLICY_NO_LEVEL)) :: policy
      type(c_ptr) :: pBuffer
      !
      cusparseZcsric02_rank_1 = cusparseZcsric02_(handle,m,nnz,descrA,&
&c_loc(csrSortedValA_valM),c_loc(csrSortedRowPtrA),&
&c_loc(csrSortedColIndA),myInfo,policy,pBuffer)
    end function

    function cusparseSgtsv2_bufferSizeExt_full_rank(handle,m,n,dl,d,du,&
&B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2_bufferSizeExt_full_rank = &
&cusparseSgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseSgtsv2_bufferSizeExt_rank_0(handle,m,n,dl,d,du,B,&
&ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2_bufferSizeExt_rank_0 = &
&cusparseSgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseSgtsv2_bufferSizeExt_rank_1(handle,m,n,dl,d,du,B,&
&ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2_bufferSizeExt_rank_1 = &
&cusparseSgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseDgtsv2_bufferSizeExt_full_rank(handle,m,n,dl,d,du,&
&B,db,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: db
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2_bufferSizeExt_full_rank = &
&cusparseDgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),db,&
&pBufferSizeInBytes)
    end function

    function cusparseDgtsv2_bufferSizeExt_rank_0(handle,m,n,dl,d,du,B,&
&db,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target :: B
      integer(c_int) :: db
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2_bufferSizeExt_rank_0 = &
&cusparseDgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),db,&
&pBufferSizeInBytes)
    end function

    function cusparseDgtsv2_bufferSizeExt_rank_1(handle,m,n,dl,d,du,B,&
&db,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: db
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2_bufferSizeExt_rank_1 = &
&cusparseDgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),db,&
&pBufferSizeInBytes)
    end function

    function cusparseCgtsv2_bufferSizeExt_full_rank(handle,m,n,dl,d,du,&
&B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2_bufferSizeExt_full_rank = &
&cusparseCgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseCgtsv2_bufferSizeExt_rank_0(handle,m,n,dl,d,du,B,&
&ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2_bufferSizeExt_rank_0 = &
&cusparseCgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseCgtsv2_bufferSizeExt_rank_1(handle,m,n,dl,d,du,B,&
&ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2_bufferSizeExt_rank_1 = &
&cusparseCgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseZgtsv2_bufferSizeExt_full_rank(handle,m,n,dl,d,du,&
&B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2_bufferSizeExt_full_rank = &
&cusparseZgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseZgtsv2_bufferSizeExt_rank_0(handle,m,n,dl,d,du,B,&
&ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2_bufferSizeExt_rank_0 = &
&cusparseZgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseZgtsv2_bufferSizeExt_rank_1(handle,m,n,dl,d,du,B,&
&ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2_bufferSizeExt_rank_1 = &
&cusparseZgtsv2_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseSgtsv2_full_rank(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgtsv2_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2_full_rank = cusparseSgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseSgtsv2_rank_0(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgtsv2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2_rank_0 = cusparseSgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseSgtsv2_rank_1(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSgtsv2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2_rank_1 = cusparseSgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseDgtsv2_full_rank(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgtsv2_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2_full_rank = cusparseDgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseDgtsv2_rank_0(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgtsv2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2_rank_0 = cusparseDgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseDgtsv2_rank_1(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDgtsv2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2_rank_1 = cusparseDgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseCgtsv2_full_rank(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgtsv2_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2_full_rank = cusparseCgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseCgtsv2_rank_0(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgtsv2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2_rank_0 = cusparseCgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseCgtsv2_rank_1(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCgtsv2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2_rank_1 = cusparseCgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseZgtsv2_full_rank(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgtsv2_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2_full_rank = cusparseZgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseZgtsv2_rank_0(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgtsv2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2_rank_0 = cusparseZgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseZgtsv2_rank_1(handle,m,n,dl,d,du,B,ldb,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZgtsv2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2_rank_1 = cusparseZgtsv2_(handle,m,n,dl,d,du,&
&c_loc(B),ldb,pBuffer)
    end function

    function cusparseSgtsv2_nopivot_bufferSizeExt_full_rank(handle,m,n,&
&dl,d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_nopivot_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2_nopivot_bufferSizeExt_full_rank = &
&cusparseSgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseSgtsv2_nopivot_bufferSizeExt_rank_0(handle,m,n,dl,&
&d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_nopivot_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2_nopivot_bufferSizeExt_rank_0 = &
&cusparseSgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseSgtsv2_nopivot_bufferSizeExt_rank_1(handle,m,n,dl,&
&d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_nopivot_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2_nopivot_bufferSizeExt_rank_1 = &
&cusparseSgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseDgtsv2_nopivot_bufferSizeExt_full_rank(handle,m,n,&
&dl,d,du,B,db,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_nopivot_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: db
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2_nopivot_bufferSizeExt_full_rank = &
&cusparseDgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),db,&
&pBufferSizeInBytes)
    end function

    function cusparseDgtsv2_nopivot_bufferSizeExt_rank_0(handle,m,n,dl,&
&d,du,B,db,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_nopivot_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target :: B
      integer(c_int) :: db
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2_nopivot_bufferSizeExt_rank_0 = &
&cusparseDgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),db,&
&pBufferSizeInBytes)
    end function

    function cusparseDgtsv2_nopivot_bufferSizeExt_rank_1(handle,m,n,dl,&
&d,du,B,db,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_nopivot_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: db
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2_nopivot_bufferSizeExt_rank_1 = &
&cusparseDgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),db,&
&pBufferSizeInBytes)
    end function

    function cusparseCgtsv2_nopivot_bufferSizeExt_full_rank(handle,m,n,&
&dl,d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_nopivot_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2_nopivot_bufferSizeExt_full_rank = &
&cusparseCgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseCgtsv2_nopivot_bufferSizeExt_rank_0(handle,m,n,dl,&
&d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_nopivot_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2_nopivot_bufferSizeExt_rank_0 = &
&cusparseCgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseCgtsv2_nopivot_bufferSizeExt_rank_1(handle,m,n,dl,&
&d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_nopivot_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2_nopivot_bufferSizeExt_rank_1 = &
&cusparseCgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseZgtsv2_nopivot_bufferSizeExt_full_rank(handle,m,n,&
&dl,d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_nopivot_bufferSizeExt_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2_nopivot_bufferSizeExt_full_rank = &
&cusparseZgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseZgtsv2_nopivot_bufferSizeExt_rank_0(handle,m,n,dl,&
&d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_nopivot_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2_nopivot_bufferSizeExt_rank_0 = &
&cusparseZgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseZgtsv2_nopivot_bufferSizeExt_rank_1(handle,m,n,dl,&
&d,du,B,ldb,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_nopivot_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2_nopivot_bufferSizeExt_rank_1 = &
&cusparseZgtsv2_nopivot_bufferSizeExt_(handle,m,n,dl,d,du,c_loc(B),ldb,&
&pBufferSizeInBytes)
    end function

    function cusparseSgtsv2_nopivot_full_rank(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_nopivot_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2_nopivot_full_rank = &
&cusparseSgtsv2_nopivot_(handle,m,n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseSgtsv2_nopivot_rank_0(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_nopivot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2_nopivot_rank_0 = cusparseSgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseSgtsv2_nopivot_rank_1(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2_nopivot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2_nopivot_rank_1 = cusparseSgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseDgtsv2_nopivot_full_rank(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_nopivot_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2_nopivot_full_rank = &
&cusparseDgtsv2_nopivot_(handle,m,n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseDgtsv2_nopivot_rank_0(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_nopivot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2_nopivot_rank_0 = cusparseDgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseDgtsv2_nopivot_rank_1(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2_nopivot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2_nopivot_rank_1 = cusparseDgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseCgtsv2_nopivot_full_rank(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_nopivot_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2_nopivot_full_rank = &
&cusparseCgtsv2_nopivot_(handle,m,n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseCgtsv2_nopivot_rank_0(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_nopivot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2_nopivot_rank_0 = cusparseCgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseCgtsv2_nopivot_rank_1(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2_nopivot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2_nopivot_rank_1 = cusparseCgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseZgtsv2_nopivot_full_rank(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_nopivot_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:,:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2_nopivot_full_rank = &
&cusparseZgtsv2_nopivot_(handle,m,n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseZgtsv2_nopivot_rank_0(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_nopivot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2_nopivot_rank_0 = cusparseZgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseZgtsv2_nopivot_rank_1(handle,m,n,dl,d,du,B,ldb,&
&pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2_nopivot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:) :: B
      integer(c_int) :: ldb
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2_nopivot_rank_1 = cusparseZgtsv2_nopivot_(handle,m,&
&n,dl,d,du,c_loc(B),ldb,pBuffer)
    end function

    function cusparseSgtsv2StridedBatch_bufferSizeExt_rank_0(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2StridedBatch_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2StridedBatch_bufferSizeExt_rank_0 = &
&cusparseSgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseSgtsv2StridedBatch_bufferSizeExt_rank_1(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2StridedBatch_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseSgtsv2StridedBatch_bufferSizeExt_rank_1 = &
&cusparseSgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseDgtsv2StridedBatch_bufferSizeExt_rank_0(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2StridedBatch_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2StridedBatch_bufferSizeExt_rank_0 = &
&cusparseDgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseDgtsv2StridedBatch_bufferSizeExt_rank_1(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2StridedBatch_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseDgtsv2StridedBatch_bufferSizeExt_rank_1 = &
&cusparseDgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseCgtsv2StridedBatch_bufferSizeExt_rank_0(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2StridedBatch_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2StridedBatch_bufferSizeExt_rank_0 = &
&cusparseCgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseCgtsv2StridedBatch_bufferSizeExt_rank_1(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2StridedBatch_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseCgtsv2StridedBatch_bufferSizeExt_rank_1 = &
&cusparseCgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseZgtsv2StridedBatch_bufferSizeExt_rank_0(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2StridedBatch_bufferSizeExt_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2StridedBatch_bufferSizeExt_rank_0 = &
&cusparseZgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseZgtsv2StridedBatch_bufferSizeExt_rank_1(handle,m,&
&dl,d,du,x,batchCount,batchStride,pBufferSizeInBytes)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2StridedBatch_bufferSizeExt_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      integer(c_size_t) :: pBufferSizeInBytes
      !
      cusparseZgtsv2StridedBatch_bufferSizeExt_rank_1 = &
&cusparseZgtsv2StridedBatch_bufferSizeExt_(handle,m,dl,d,du,c_loc(x),&
&batchCount,batchStride,pBufferSizeInBytes)
    end function

    function cusparseSgtsv2StridedBatch_rank_0(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2StridedBatch_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2StridedBatch_rank_0 = &
&cusparseSgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseSgtsv2StridedBatch_rank_1(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSgtsv2StridedBatch_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseSgtsv2StridedBatch_rank_1 = &
&cusparseSgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseDgtsv2StridedBatch_rank_0(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2StridedBatch_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2StridedBatch_rank_0 = &
&cusparseDgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseDgtsv2StridedBatch_rank_1(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDgtsv2StridedBatch_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseDgtsv2StridedBatch_rank_1 = &
&cusparseDgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseCgtsv2StridedBatch_rank_0(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2StridedBatch_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2StridedBatch_rank_0 = &
&cusparseCgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseCgtsv2StridedBatch_rank_1(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCgtsv2StridedBatch_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseCgtsv2StridedBatch_rank_1 = &
&cusparseCgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseZgtsv2StridedBatch_rank_0(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2StridedBatch_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2StridedBatch_rank_0 = &
&cusparseZgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseZgtsv2StridedBatch_rank_1(handle,m,dl,d,du,x,&
&batchCount,batchStride,pBuffer)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZgtsv2StridedBatch_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      type(c_ptr) :: dl
      type(c_ptr) :: d
      type(c_ptr) :: du
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: batchCount
      integer(c_int) :: batchStride
      type(c_ptr) :: pBuffer
      !
      cusparseZgtsv2StridedBatch_rank_1 = &
&cusparseZgtsv2StridedBatch_(handle,m,dl,d,du,c_loc(x),batchCount,&
&batchStride,pBuffer)
    end function

    function cusparseSnnz_full_rank(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSnnz_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseSnnz_full_rank = cusparseSnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseSnnz_rank_0(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSnnz_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      real(c_float),target :: A
      integer(c_int) :: lda
      integer(c_int),target :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseSnnz_rank_0 = cusparseSnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseSnnz_rank_1(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseSnnz_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      real(c_float),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseSnnz_rank_1 = cusparseSnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseDnnz_full_rank(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnnz_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseDnnz_full_rank = cusparseDnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseDnnz_rank_0(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnnz_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      real(c_double),target :: A
      integer(c_int) :: lda
      integer(c_int),target :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseDnnz_rank_0 = cusparseDnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseDnnz_rank_1(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseDnnz_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      real(c_double),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseDnnz_rank_1 = cusparseDnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseCnnz_full_rank(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCnnz_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseCnnz_full_rank = cusparseCnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseCnnz_rank_0(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCnnz_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      complex(c_float_complex),target :: A
      integer(c_int) :: lda
      integer(c_int),target :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseCnnz_rank_0 = cusparseCnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseCnnz_rank_1(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseCnnz_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      complex(c_float_complex),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseCnnz_rank_1 = cusparseCnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseZnnz_full_rank(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZnnz_full_rank
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:,:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseZnnz_full_rank = cusparseZnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseZnnz_rank_0(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZnnz_rank_0
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      complex(c_double_complex),target :: A
      integer(c_int) :: lda
      integer(c_int),target :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseZnnz_rank_0 = cusparseZnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseZnnz_rank_1(handle,dirA,m,n,descrA,A,lda,&
&nnzPerRowColumn,nnzTotalDevHostPtr)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: cusparseZnnz_rank_1
      type(c_ptr) :: handle
      integer(kind(CUSPARSE_DIRECTION_ROW)) :: dirA
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descrA
      complex(c_double_complex),target,dimension(:) :: A
      integer(c_int) :: lda
      integer(c_int),target,dimension(:) :: nnzPerRowColumn
      integer(c_int) :: nnzTotalDevHostPtr
      !
      cusparseZnnz_rank_1 = cusparseZnnz_(handle,dirA,m,n,descrA,&
&c_loc(A),lda,c_loc(nnzPerRowColumn),nnzTotalDevHostPtr)
    end function

    function cusparseSdense2csr_full_rank(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSdense2csr_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      real(c_float),target,dimension(:,:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      real(c_float),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseSdense2csr_full_rank = cusparseSdense2csr_(handle,m,n,&
&descr,c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),&
&c_loc(csr_row_ptr),c_loc(csr_col_ind))
    end function

    function cusparseSdense2csr_rank_0(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSdense2csr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      real(c_float),target :: A
      integer(c_int) :: ld
      integer(c_int),target :: nnz_per_rows
      real(c_float),target :: csr_val
      integer(c_int),target :: csr_row_ptr
      integer(c_int),target :: csr_col_ind
      !
      cusparseSdense2csr_rank_0 = cusparseSdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseSdense2csr_rank_1(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSdense2csr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      real(c_float),target,dimension(:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      real(c_float),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseSdense2csr_rank_1 = cusparseSdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseDdense2csr_full_rank(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDdense2csr_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      real(c_double),target,dimension(:,:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      real(c_double),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseDdense2csr_full_rank = cusparseDdense2csr_(handle,m,n,&
&descr,c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),&
&c_loc(csr_row_ptr),c_loc(csr_col_ind))
    end function

    function cusparseDdense2csr_rank_0(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDdense2csr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      real(c_double),target :: A
      integer(c_int) :: ld
      integer(c_int),target :: nnz_per_rows
      real(c_double),target :: csr_val
      integer(c_int),target :: csr_row_ptr
      integer(c_int),target :: csr_col_ind
      !
      cusparseDdense2csr_rank_0 = cusparseDdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseDdense2csr_rank_1(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseDdense2csr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      real(c_double),target,dimension(:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      real(c_double),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseDdense2csr_rank_1 = cusparseDdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseCdense2csr_full_rank(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCdense2csr_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      complex(c_float_complex),target,dimension(:,:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      complex(c_float_complex),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseCdense2csr_full_rank = cusparseCdense2csr_(handle,m,n,&
&descr,c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),&
&c_loc(csr_row_ptr),c_loc(csr_col_ind))
    end function

    function cusparseCdense2csr_rank_0(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCdense2csr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      complex(c_float_complex),target :: A
      integer(c_int) :: ld
      integer(c_int),target :: nnz_per_rows
      complex(c_float_complex),target :: csr_val
      integer(c_int),target :: csr_row_ptr
      integer(c_int),target :: csr_col_ind
      !
      cusparseCdense2csr_rank_0 = cusparseCdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseCdense2csr_rank_1(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseCdense2csr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      complex(c_float_complex),target,dimension(:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      complex(c_float_complex),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseCdense2csr_rank_1 = cusparseCdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseZdense2csr_full_rank(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZdense2csr_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      complex(c_double_complex),target,dimension(:,:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      complex(c_double_complex),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseZdense2csr_full_rank = cusparseZdense2csr_(handle,m,n,&
&descr,c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),&
&c_loc(csr_row_ptr),c_loc(csr_col_ind))
    end function

    function cusparseZdense2csr_rank_0(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZdense2csr_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      complex(c_double_complex),target :: A
      integer(c_int) :: ld
      integer(c_int),target :: nnz_per_rows
      complex(c_double_complex),target :: csr_val
      integer(c_int),target :: csr_row_ptr
      integer(c_int),target :: csr_col_ind
      !
      cusparseZdense2csr_rank_0 = cusparseZdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseZdense2csr_rank_1(handle,m,n,descr,A,ld,&
&nnz_per_rows,csr_val,csr_row_ptr,csr_col_ind)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseZdense2csr_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      type(c_ptr) :: descr
      complex(c_double_complex),target,dimension(:) :: A
      integer(c_int) :: ld
      integer(c_int),target,dimension(:) :: nnz_per_rows
      complex(c_double_complex),target,dimension(:) :: csr_val
      integer(c_int),target,dimension(:) :: csr_row_ptr
      integer(c_int),target,dimension(:) :: csr_col_ind
      !
      cusparseZdense2csr_rank_1 = cusparseZdense2csr_(handle,m,n,descr,&
&c_loc(A),ld,c_loc(nnz_per_rows),c_loc(csr_val),c_loc(csr_row_ptr),&
&c_loc(csr_col_ind))
    end function

    function cusparseSpruneDense2csr_bufferSize_full_rank(handle,m,n,A,&
&lda,threshold,descr,csrVal,csrRowPtr,csrColInd,bufferSize)
      use iso_c_binding
      use cudafor_cusparse_enums
      use cudafor_enums
      implicit none
      integer(kind(CUSPARSE_STATUS_SUCCESS)) :: &
&cusparseSpruneDense2csr_bufferSize_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: m
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: A
      integer(c_int) :: lda
      real(c_float) :: threshold
      type(c_ptr) :: descr
      real(c_float),target,dimension(:) :: csrVal
      integer(c_int),target,dimension(:) :: csrRowPtr
      integer(c_int),target,dimension(:) :: csrColInd
      integer(c_size_t),target,dimension(:) :: bufferSize
      !
      cusparseSpruneDense2csr_bufferSize_full_rank = &
&cusparseSpruneDense2csr_bufferSize_(handle,m,n,c_loc(A),lda,threshold,&
&descr,c_loc(csrVal),c_loc(csrRowPtr),c_loc(csrColInd),&
&c_loc(bufferSize))
    end function

    function cusparseSpruneDense