


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ==============================================================================
! hipfort: FORTRAN Interfaces for GPU kernels
! ==============================================================================
! Copyright (c) 2020-2022 Advanced Micro Devices, Inc. All rights reserved.
! [MITx11 License]
! 
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
! IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
! AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
! OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
! THE SOFTWARE.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


module cudafor_cudamalloc

  interface cudaMalloc
    !> 
    !>    @brief Allocate memory on the default accelerator
    !>  
    !>    @param[out] ptr Pointer to the allocated memory
    !>    @param[in]  size Requested memory size
    !>  
    !>    If size is 0, no memory is allocated, ptr returns nullptr, and hipSuccess is returned.
    !>  
    !>    @return #hipSuccess, #hipErrorOutOfMemory, #hipErrorInvalidValue (bad context, null ptr)
    !>  
    !>    @see hipMallocPitch, hipFree, hipMallocArray, hipFreeArray, hipMalloc3D, hipMalloc3DArray,
    !>   hipHostFree, hipHostMalloc
    !>  
    function cudaMalloc_(ptr,mySize) bind(c, name="cudaMalloc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMalloc_
      type(c_ptr) :: ptr
      integer(c_size_t),value :: mySize
    end function

    module procedure cudaMalloc_l_0_source,&
      cudaMalloc_l_1_source,&
      cudaMalloc_l_1_c_int,&
      cudaMalloc_l_1_c_size_t,&
      cudaMalloc_l_2_source,&
      cudaMalloc_l_2_c_int,&
      cudaMalloc_l_2_c_size_t,&
      cudaMalloc_l_3_source,&
      cudaMalloc_l_3_c_int,&
      cudaMalloc_l_3_c_size_t,&
      cudaMalloc_l_4_source,&
      cudaMalloc_l_4_c_int,&
      cudaMalloc_l_4_c_size_t,&
      cudaMalloc_l_5_source,&
      cudaMalloc_l_5_c_int,&
      cudaMalloc_l_5_c_size_t,&
      cudaMalloc_l_6_source,&
      cudaMalloc_l_6_c_int,&
      cudaMalloc_l_6_c_size_t,&
      cudaMalloc_l_7_source,&
      cudaMalloc_l_7_c_int,&
      cudaMalloc_l_7_c_size_t,&
      cudaMalloc_i4_0_source,&
      cudaMalloc_i4_1_source,&
      cudaMalloc_i4_1_c_int,&
      cudaMalloc_i4_1_c_size_t,&
      cudaMalloc_i4_2_source,&
      cudaMalloc_i4_2_c_int,&
      cudaMalloc_i4_2_c_size_t,&
      cudaMalloc_i4_3_source,&
      cudaMalloc_i4_3_c_int,&
      cudaMalloc_i4_3_c_size_t,&
      cudaMalloc_i4_4_source,&
      cudaMalloc_i4_4_c_int,&
      cudaMalloc_i4_4_c_size_t,&
      cudaMalloc_i4_5_source,&
      cudaMalloc_i4_5_c_int,&
      cudaMalloc_i4_5_c_size_t,&
      cudaMalloc_i4_6_source,&
      cudaMalloc_i4_6_c_int,&
      cudaMalloc_i4_6_c_size_t,&
      cudaMalloc_i4_7_source,&
      cudaMalloc_i4_7_c_int,&
      cudaMalloc_i4_7_c_size_t,&
      cudaMalloc_i8_0_source,&
      cudaMalloc_i8_1_source,&
      cudaMalloc_i8_1_c_int,&
      cudaMalloc_i8_1_c_size_t,&
      cudaMalloc_i8_2_source,&
      cudaMalloc_i8_2_c_int,&
      cudaMalloc_i8_2_c_size_t,&
      cudaMalloc_i8_3_source,&
      cudaMalloc_i8_3_c_int,&
      cudaMalloc_i8_3_c_size_t,&
      cudaMalloc_i8_4_source,&
      cudaMalloc_i8_4_c_int,&
      cudaMalloc_i8_4_c_size_t,&
      cudaMalloc_i8_5_source,&
      cudaMalloc_i8_5_c_int,&
      cudaMalloc_i8_5_c_size_t,&
      cudaMalloc_i8_6_source,&
      cudaMalloc_i8_6_c_int,&
      cudaMalloc_i8_6_c_size_t,&
      cudaMalloc_i8_7_source,&
      cudaMalloc_i8_7_c_int,&
      cudaMalloc_i8_7_c_size_t,&
      cudaMalloc_r4_0_source,&
      cudaMalloc_r4_1_source,&
      cudaMalloc_r4_1_c_int,&
      cudaMalloc_r4_1_c_size_t,&
      cudaMalloc_r4_2_source,&
      cudaMalloc_r4_2_c_int,&
      cudaMalloc_r4_2_c_size_t,&
      cudaMalloc_r4_3_source,&
      cudaMalloc_r4_3_c_int,&
      cudaMalloc_r4_3_c_size_t,&
      cudaMalloc_r4_4_source,&
      cudaMalloc_r4_4_c_int,&
      cudaMalloc_r4_4_c_size_t,&
      cudaMalloc_r4_5_source,&
      cudaMalloc_r4_5_c_int,&
      cudaMalloc_r4_5_c_size_t,&
      cudaMalloc_r4_6_source,&
      cudaMalloc_r4_6_c_int,&
      cudaMalloc_r4_6_c_size_t,&
      cudaMalloc_r4_7_source,&
      cudaMalloc_r4_7_c_int,&
      cudaMalloc_r4_7_c_size_t,&
      cudaMalloc_r8_0_source,&
      cudaMalloc_r8_1_source,&
      cudaMalloc_r8_1_c_int,&
      cudaMalloc_r8_1_c_size_t,&
      cudaMalloc_r8_2_source,&
      cudaMalloc_r8_2_c_int,&
      cudaMalloc_r8_2_c_size_t,&
      cudaMalloc_r8_3_source,&
      cudaMalloc_r8_3_c_int,&
      cudaMalloc_r8_3_c_size_t,&
      cudaMalloc_r8_4_source,&
      cudaMalloc_r8_4_c_int,&
      cudaMalloc_r8_4_c_size_t,&
      cudaMalloc_r8_5_source,&
      cudaMalloc_r8_5_c_int,&
      cudaMalloc_r8_5_c_size_t,&
      cudaMalloc_r8_6_source,&
      cudaMalloc_r8_6_c_int,&
      cudaMalloc_r8_6_c_size_t,&
      cudaMalloc_r8_7_source,&
      cudaMalloc_r8_7_c_int,&
      cudaMalloc_r8_7_c_size_t,&
      cudaMalloc_c4_0_source,&
      cudaMalloc_c4_1_source,&
      cudaMalloc_c4_1_c_int,&
      cudaMalloc_c4_1_c_size_t,&
      cudaMalloc_c4_2_source,&
      cudaMalloc_c4_2_c_int,&
      cudaMalloc_c4_2_c_size_t,&
      cudaMalloc_c4_3_source,&
      cudaMalloc_c4_3_c_int,&
      cudaMalloc_c4_3_c_size_t,&
      cudaMalloc_c4_4_source,&
      cudaMalloc_c4_4_c_int,&
      cudaMalloc_c4_4_c_size_t,&
      cudaMalloc_c4_5_source,&
      cudaMalloc_c4_5_c_int,&
      cudaMalloc_c4_5_c_size_t,&
      cudaMalloc_c4_6_source,&
      cudaMalloc_c4_6_c_int,&
      cudaMalloc_c4_6_c_size_t,&
      cudaMalloc_c4_7_source,&
      cudaMalloc_c4_7_c_int,&
      cudaMalloc_c4_7_c_size_t,&
      cudaMalloc_c8_0_source,&
      cudaMalloc_c8_1_source,&
      cudaMalloc_c8_1_c_int,&
      cudaMalloc_c8_1_c_size_t,&
      cudaMalloc_c8_2_source,&
      cudaMalloc_c8_2_c_int,&
      cudaMalloc_c8_2_c_size_t,&
      cudaMalloc_c8_3_source,&
      cudaMalloc_c8_3_c_int,&
      cudaMalloc_c8_3_c_size_t,&
      cudaMalloc_c8_4_source,&
      cudaMalloc_c8_4_c_int,&
      cudaMalloc_c8_4_c_size_t,&
      cudaMalloc_c8_5_source,&
      cudaMalloc_c8_5_c_int,&
      cudaMalloc_c8_5_c_size_t,&
      cudaMalloc_c8_6_source,&
      cudaMalloc_c8_6_c_int,&
      cudaMalloc_c8_6_c_size_t,&
      cudaMalloc_c8_7_source,&
      cudaMalloc_c8_7_c_int,&
      cudaMalloc_c8_7_c_size_t 
  end interface

  !> 
  !>   @brief Allocates memory that will be automatically managed by AMD HMM.
  !>  
  !>   @param [out] dev_ptr - pointer to allocated device memory
  !>   @param [in]  size    - requested allocation size in bytes
  !>   @param [in]  flags   - must be either hipMemAttachGlobal or hipMemAttachHost
  !>                          (defaults to hipMemAttachGlobal)
  !>  
  !>   @returns #hipSuccess, #hipErrorMemoryAllocation, #hipErrorNotSupported, #hipErrorInvalidValue
  !>  
  interface cudaMallocManaged
    function cudaMallocManaged_(dev_ptr,mySize,flags) bind(c, &
&name="cudaMallocManaged")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMallocManaged_
      type(c_ptr) :: dev_ptr
      integer(c_size_t),value :: mySize
      integer(kind=4),value :: flags
    end function

    module procedure cudaMallocManaged_l_0_source,&
      cudaMallocManaged_l_1_source,&
      cudaMallocManaged_l_1_c_int,&
      cudaMallocManaged_l_1_c_size_t,&
      cudaMallocManaged_l_2_source,&
      cudaMallocManaged_l_2_c_int,&
      cudaMallocManaged_l_2_c_size_t,&
      cudaMallocManaged_l_3_source,&
      cudaMallocManaged_l_3_c_int,&
      cudaMallocManaged_l_3_c_size_t,&
      cudaMallocManaged_l_4_source,&
      cudaMallocManaged_l_4_c_int,&
      cudaMallocManaged_l_4_c_size_t,&
      cudaMallocManaged_l_5_source,&
      cudaMallocManaged_l_5_c_int,&
      cudaMallocManaged_l_5_c_size_t,&
      cudaMallocManaged_l_6_source,&
      cudaMallocManaged_l_6_c_int,&
      cudaMallocManaged_l_6_c_size_t,&
      cudaMallocManaged_l_7_source,&
      cudaMallocManaged_l_7_c_int,&
      cudaMallocManaged_l_7_c_size_t,&
      cudaMallocManaged_i4_0_source,&
      cudaMallocManaged_i4_1_source,&
      cudaMallocManaged_i4_1_c_int,&
      cudaMallocManaged_i4_1_c_size_t,&
      cudaMallocManaged_i4_2_source,&
      cudaMallocManaged_i4_2_c_int,&
      cudaMallocManaged_i4_2_c_size_t,&
      cudaMallocManaged_i4_3_source,&
      cudaMallocManaged_i4_3_c_int,&
      cudaMallocManaged_i4_3_c_size_t,&
      cudaMallocManaged_i4_4_source,&
      cudaMallocManaged_i4_4_c_int,&
      cudaMallocManaged_i4_4_c_size_t,&
      cudaMallocManaged_i4_5_source,&
      cudaMallocManaged_i4_5_c_int,&
      cudaMallocManaged_i4_5_c_size_t,&
      cudaMallocManaged_i4_6_source,&
      cudaMallocManaged_i4_6_c_int,&
      cudaMallocManaged_i4_6_c_size_t,&
      cudaMallocManaged_i4_7_source,&
      cudaMallocManaged_i4_7_c_int,&
      cudaMallocManaged_i4_7_c_size_t,&
      cudaMallocManaged_i8_0_source,&
      cudaMallocManaged_i8_1_source,&
      cudaMallocManaged_i8_1_c_int,&
      cudaMallocManaged_i8_1_c_size_t,&
      cudaMallocManaged_i8_2_source,&
      cudaMallocManaged_i8_2_c_int,&
      cudaMallocManaged_i8_2_c_size_t,&
      cudaMallocManaged_i8_3_source,&
      cudaMallocManaged_i8_3_c_int,&
      cudaMallocManaged_i8_3_c_size_t,&
      cudaMallocManaged_i8_4_source,&
      cudaMallocManaged_i8_4_c_int,&
      cudaMallocManaged_i8_4_c_size_t,&
      cudaMallocManaged_i8_5_source,&
      cudaMallocManaged_i8_5_c_int,&
      cudaMallocManaged_i8_5_c_size_t,&
      cudaMallocManaged_i8_6_source,&
      cudaMallocManaged_i8_6_c_int,&
      cudaMallocManaged_i8_6_c_size_t,&
      cudaMallocManaged_i8_7_source,&
      cudaMallocManaged_i8_7_c_int,&
      cudaMallocManaged_i8_7_c_size_t,&
      cudaMallocManaged_r4_0_source,&
      cudaMallocManaged_r4_1_source,&
      cudaMallocManaged_r4_1_c_int,&
      cudaMallocManaged_r4_1_c_size_t,&
      cudaMallocManaged_r4_2_source,&
      cudaMallocManaged_r4_2_c_int,&
      cudaMallocManaged_r4_2_c_size_t,&
      cudaMallocManaged_r4_3_source,&
      cudaMallocManaged_r4_3_c_int,&
      cudaMallocManaged_r4_3_c_size_t,&
      cudaMallocManaged_r4_4_source,&
      cudaMallocManaged_r4_4_c_int,&
      cudaMallocManaged_r4_4_c_size_t,&
      cudaMallocManaged_r4_5_source,&
      cudaMallocManaged_r4_5_c_int,&
      cudaMallocManaged_r4_5_c_size_t,&
      cudaMallocManaged_r4_6_source,&
      cudaMallocManaged_r4_6_c_int,&
      cudaMallocManaged_r4_6_c_size_t,&
      cudaMallocManaged_r4_7_source,&
      cudaMallocManaged_r4_7_c_int,&
      cudaMallocManaged_r4_7_c_size_t,&
      cudaMallocManaged_r8_0_source,&
      cudaMallocManaged_r8_1_source,&
      cudaMallocManaged_r8_1_c_int,&
      cudaMallocManaged_r8_1_c_size_t,&
      cudaMallocManaged_r8_2_source,&
      cudaMallocManaged_r8_2_c_int,&
      cudaMallocManaged_r8_2_c_size_t,&
      cudaMallocManaged_r8_3_source,&
      cudaMallocManaged_r8_3_c_int,&
      cudaMallocManaged_r8_3_c_size_t,&
      cudaMallocManaged_r8_4_source,&
      cudaMallocManaged_r8_4_c_int,&
      cudaMallocManaged_r8_4_c_size_t,&
      cudaMallocManaged_r8_5_source,&
      cudaMallocManaged_r8_5_c_int,&
      cudaMallocManaged_r8_5_c_size_t,&
      cudaMallocManaged_r8_6_source,&
      cudaMallocManaged_r8_6_c_int,&
      cudaMallocManaged_r8_6_c_size_t,&
      cudaMallocManaged_r8_7_source,&
      cudaMallocManaged_r8_7_c_int,&
      cudaMallocManaged_r8_7_c_size_t,&
      cudaMallocManaged_c4_0_source,&
      cudaMallocManaged_c4_1_source,&
      cudaMallocManaged_c4_1_c_int,&
      cudaMallocManaged_c4_1_c_size_t,&
      cudaMallocManaged_c4_2_source,&
      cudaMallocManaged_c4_2_c_int,&
      cudaMallocManaged_c4_2_c_size_t,&
      cudaMallocManaged_c4_3_source,&
      cudaMallocManaged_c4_3_c_int,&
      cudaMallocManaged_c4_3_c_size_t,&
      cudaMallocManaged_c4_4_source,&
      cudaMallocManaged_c4_4_c_int,&
      cudaMallocManaged_c4_4_c_size_t,&
      cudaMallocManaged_c4_5_source,&
      cudaMallocManaged_c4_5_c_int,&
      cudaMallocManaged_c4_5_c_size_t,&
      cudaMallocManaged_c4_6_source,&
      cudaMallocManaged_c4_6_c_int,&
      cudaMallocManaged_c4_6_c_size_t,&
      cudaMallocManaged_c4_7_source,&
      cudaMallocManaged_c4_7_c_int,&
      cudaMallocManaged_c4_7_c_size_t,&
      cudaMallocManaged_c8_0_source,&
      cudaMallocManaged_c8_1_source,&
      cudaMallocManaged_c8_1_c_int,&
      cudaMallocManaged_c8_1_c_size_t,&
      cudaMallocManaged_c8_2_source,&
      cudaMallocManaged_c8_2_c_int,&
      cudaMallocManaged_c8_2_c_size_t,&
      cudaMallocManaged_c8_3_source,&
      cudaMallocManaged_c8_3_c_int,&
      cudaMallocManaged_c8_3_c_size_t,&
      cudaMallocManaged_c8_4_source,&
      cudaMallocManaged_c8_4_c_int,&
      cudaMallocManaged_c8_4_c_size_t,&
      cudaMallocManaged_c8_5_source,&
      cudaMallocManaged_c8_5_c_int,&
      cudaMallocManaged_c8_5_c_size_t,&
      cudaMallocManaged_c8_6_source,&
      cudaMallocManaged_c8_6_c_int,&
      cudaMallocManaged_c8_6_c_size_t,&
      cudaMallocManaged_c8_7_source,&
      cudaMallocManaged_c8_7_c_int,&
      cudaMallocManaged_c8_7_c_size_t 

  end interface
  
  interface cudaHostMalloc
    !> 
    !>    @brief Allocate device accessible page locked host memory
    !>  
    !>    @param[out] ptr Pointer to the allocated host pinned memory
    !>    @param[in]  size Requested memory size
    !>    @param[in]  flags Type of host memory allocation
    !>  
    !>    If size is 0, no memory is allocated, ptr returns nullptr, and hipSuccess is returned.
    !>  
    !>    @return #hipSuccess, #hipErrorOutOfMemory
    !>  
    !>    @see hipSetDeviceFlags, hipHostFree
    !>  
    function cudaHostMalloc_(ptr,mySize,flags) bind(c, &
&name="cudaHostMalloc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaHostMalloc_
      type(c_ptr) :: ptr
      integer(c_size_t),value :: mySize
      integer(kind=4),value :: flags
    end function

    module procedure cudaHostMalloc_l_0_source,&
      cudaHostMalloc_l_1_source,&
      cudaHostMalloc_l_1_c_int,&
      cudaHostMalloc_l_1_c_size_t,&
      cudaHostMalloc_l_2_source,&
      cudaHostMalloc_l_2_c_int,&
      cudaHostMalloc_l_2_c_size_t,&
      cudaHostMalloc_l_3_source,&
      cudaHostMalloc_l_3_c_int,&
      cudaHostMalloc_l_3_c_size_t,&
      cudaHostMalloc_l_4_source,&
      cudaHostMalloc_l_4_c_int,&
      cudaHostMalloc_l_4_c_size_t,&
      cudaHostMalloc_l_5_source,&
      cudaHostMalloc_l_5_c_int,&
      cudaHostMalloc_l_5_c_size_t,&
      cudaHostMalloc_l_6_source,&
      cudaHostMalloc_l_6_c_int,&
      cudaHostMalloc_l_6_c_size_t,&
      cudaHostMalloc_l_7_source,&
      cudaHostMalloc_l_7_c_int,&
      cudaHostMalloc_l_7_c_size_t,&
      cudaHostMalloc_i4_0_source,&
      cudaHostMalloc_i4_1_source,&
      cudaHostMalloc_i4_1_c_int,&
      cudaHostMalloc_i4_1_c_size_t,&
      cudaHostMalloc_i4_2_source,&
      cudaHostMalloc_i4_2_c_int,&
      cudaHostMalloc_i4_2_c_size_t,&
      cudaHostMalloc_i4_3_source,&
      cudaHostMalloc_i4_3_c_int,&
      cudaHostMalloc_i4_3_c_size_t,&
      cudaHostMalloc_i4_4_source,&
      cudaHostMalloc_i4_4_c_int,&
      cudaHostMalloc_i4_4_c_size_t,&
      cudaHostMalloc_i4_5_source,&
      cudaHostMalloc_i4_5_c_int,&
      cudaHostMalloc_i4_5_c_size_t,&
      cudaHostMalloc_i4_6_source,&
      cudaHostMalloc_i4_6_c_int,&
      cudaHostMalloc_i4_6_c_size_t,&
      cudaHostMalloc_i4_7_source,&
      cudaHostMalloc_i4_7_c_int,&
      cudaHostMalloc_i4_7_c_size_t,&
      cudaHostMalloc_i8_0_source,&
      cudaHostMalloc_i8_1_source,&
      cudaHostMalloc_i8_1_c_int,&
      cudaHostMalloc_i8_1_c_size_t,&
      cudaHostMalloc_i8_2_source,&
      cudaHostMalloc_i8_2_c_int,&
      cudaHostMalloc_i8_2_c_size_t,&
      cudaHostMalloc_i8_3_source,&
      cudaHostMalloc_i8_3_c_int,&
      cudaHostMalloc_i8_3_c_size_t,&
      cudaHostMalloc_i8_4_source,&
      cudaHostMalloc_i8_4_c_int,&
      cudaHostMalloc_i8_4_c_size_t,&
      cudaHostMalloc_i8_5_source,&
      cudaHostMalloc_i8_5_c_int,&
      cudaHostMalloc_i8_5_c_size_t,&
      cudaHostMalloc_i8_6_source,&
      cudaHostMalloc_i8_6_c_int,&
      cudaHostMalloc_i8_6_c_size_t,&
      cudaHostMalloc_i8_7_source,&
      cudaHostMalloc_i8_7_c_int,&
      cudaHostMalloc_i8_7_c_size_t,&
      cudaHostMalloc_r4_0_source,&
      cudaHostMalloc_r4_1_source,&
      cudaHostMalloc_r4_1_c_int,&
      cudaHostMalloc_r4_1_c_size_t,&
      cudaHostMalloc_r4_2_source,&
      cudaHostMalloc_r4_2_c_int,&
      cudaHostMalloc_r4_2_c_size_t,&
      cudaHostMalloc_r4_3_source,&
      cudaHostMalloc_r4_3_c_int,&
      cudaHostMalloc_r4_3_c_size_t,&
      cudaHostMalloc_r4_4_source,&
      cudaHostMalloc_r4_4_c_int,&
      cudaHostMalloc_r4_4_c_size_t,&
      cudaHostMalloc_r4_5_source,&
      cudaHostMalloc_r4_5_c_int,&
      cudaHostMalloc_r4_5_c_size_t,&
      cudaHostMalloc_r4_6_source,&
      cudaHostMalloc_r4_6_c_int,&
      cudaHostMalloc_r4_6_c_size_t,&
      cudaHostMalloc_r4_7_source,&
      cudaHostMalloc_r4_7_c_int,&
      cudaHostMalloc_r4_7_c_size_t,&
      cudaHostMalloc_r8_0_source,&
      cudaHostMalloc_r8_1_source,&
      cudaHostMalloc_r8_1_c_int,&
      cudaHostMalloc_r8_1_c_size_t,&
      cudaHostMalloc_r8_2_source,&
      cudaHostMalloc_r8_2_c_int,&
      cudaHostMalloc_r8_2_c_size_t,&
      cudaHostMalloc_r8_3_source,&
      cudaHostMalloc_r8_3_c_int,&
      cudaHostMalloc_r8_3_c_size_t,&
      cudaHostMalloc_r8_4_source,&
      cudaHostMalloc_r8_4_c_int,&
      cudaHostMalloc_r8_4_c_size_t,&
      cudaHostMalloc_r8_5_source,&
      cudaHostMalloc_r8_5_c_int,&
      cudaHostMalloc_r8_5_c_size_t,&
      cudaHostMalloc_r8_6_source,&
      cudaHostMalloc_r8_6_c_int,&
      cudaHostMalloc_r8_6_c_size_t,&
      cudaHostMalloc_r8_7_source,&
      cudaHostMalloc_r8_7_c_int,&
      cudaHostMalloc_r8_7_c_size_t,&
      cudaHostMalloc_c4_0_source,&
      cudaHostMalloc_c4_1_source,&
      cudaHostMalloc_c4_1_c_int,&
      cudaHostMalloc_c4_1_c_size_t,&
      cudaHostMalloc_c4_2_source,&
      cudaHostMalloc_c4_2_c_int,&
      cudaHostMalloc_c4_2_c_size_t,&
      cudaHostMalloc_c4_3_source,&
      cudaHostMalloc_c4_3_c_int,&
      cudaHostMalloc_c4_3_c_size_t,&
      cudaHostMalloc_c4_4_source,&
      cudaHostMalloc_c4_4_c_int,&
      cudaHostMalloc_c4_4_c_size_t,&
      cudaHostMalloc_c4_5_source,&
      cudaHostMalloc_c4_5_c_int,&
      cudaHostMalloc_c4_5_c_size_t,&
      cudaHostMalloc_c4_6_source,&
      cudaHostMalloc_c4_6_c_int,&
      cudaHostMalloc_c4_6_c_size_t,&
      cudaHostMalloc_c4_7_source,&
      cudaHostMalloc_c4_7_c_int,&
      cudaHostMalloc_c4_7_c_size_t,&
      cudaHostMalloc_c8_0_source,&
      cudaHostMalloc_c8_1_source,&
      cudaHostMalloc_c8_1_c_int,&
      cudaHostMalloc_c8_1_c_size_t,&
      cudaHostMalloc_c8_2_source,&
      cudaHostMalloc_c8_2_c_int,&
      cudaHostMalloc_c8_2_c_size_t,&
      cudaHostMalloc_c8_3_source,&
      cudaHostMalloc_c8_3_c_int,&
      cudaHostMalloc_c8_3_c_size_t,&
      cudaHostMalloc_c8_4_source,&
      cudaHostMalloc_c8_4_c_int,&
      cudaHostMalloc_c8_4_c_size_t,&
      cudaHostMalloc_c8_5_source,&
      cudaHostMalloc_c8_5_c_int,&
      cudaHostMalloc_c8_5_c_size_t,&
      cudaHostMalloc_c8_6_source,&
      cudaHostMalloc_c8_6_c_int,&
      cudaHostMalloc_c8_6_c_size_t,&
      cudaHostMalloc_c8_7_source,&
      cudaHostMalloc_c8_7_c_int,&
      cudaHostMalloc_c8_7_c_size_t 
  end interface
  
  interface cudaFree
    !> 
    !>    @brief Free memory allocated by the amd hip memory allocation API.
    !>    This API performs an implicit hipDeviceSynchronize() call.
    !>    If pointer is NULL, the hip runtime is initialized and hipSuccess is returned.
    !>  
    !>    @param[in] ptr Pointer to memory to be freed
    !>    @return #hipSuccess
    !>    @return #hipErrorInvalidDevicePointer (if pointer is invalid, including host pointers allocated
    !>   with hipHostMalloc)
    !>  
    !>    @see hipMalloc, hipMallocPitch, hipMallocArray, hipFreeArray, hipHostFree, hipMalloc3D,
    !>   hipMalloc3DArray, hipHostMalloc
    !>  
    function cudaFree_(ptr) bind(c, name="cudaFree")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFree_
      type(c_ptr),value :: ptr
    end function

    module procedure cudaFree_l_0,&
      cudaFree_l_1,&
      cudaFree_l_2,&
      cudaFree_l_3,&
      cudaFree_l_4,&
      cudaFree_l_5,&
      cudaFree_l_6,&
      cudaFree_l_7,&
      cudaFree_i4_0,&
      cudaFree_i4_1,&
      cudaFree_i4_2,&
      cudaFree_i4_3,&
      cudaFree_i4_4,&
      cudaFree_i4_5,&
      cudaFree_i4_6,&
      cudaFree_i4_7,&
      cudaFree_i8_0,&
      cudaFree_i8_1,&
      cudaFree_i8_2,&
      cudaFree_i8_3,&
      cudaFree_i8_4,&
      cudaFree_i8_5,&
      cudaFree_i8_6,&
      cudaFree_i8_7,&
      cudaFree_r4_0,&
      cudaFree_r4_1,&
      cudaFree_r4_2,&
      cudaFree_r4_3,&
      cudaFree_r4_4,&
      cudaFree_r4_5,&
      cudaFree_r4_6,&
      cudaFree_r4_7,&
      cudaFree_r8_0,&
      cudaFree_r8_1,&
      cudaFree_r8_2,&
      cudaFree_r8_3,&
      cudaFree_r8_4,&
      cudaFree_r8_5,&
      cudaFree_r8_6,&
      cudaFree_r8_7,&
      cudaFree_c4_0,&
      cudaFree_c4_1,&
      cudaFree_c4_2,&
      cudaFree_c4_3,&
      cudaFree_c4_4,&
      cudaFree_c4_5,&
      cudaFree_c4_6,&
      cudaFree_c4_7,&
      cudaFree_c8_0,&
      cudaFree_c8_1,&
      cudaFree_c8_2,&
      cudaFree_c8_3,&
      cudaFree_c8_4,&
      cudaFree_c8_5,&
      cudaFree_c8_6,&
      cudaFree_c8_7 
  end interface

  interface cudaHostFree
    !> 
    !>    @brief Free memory allocated by the amd hip host memory allocation API
    !>    This API performs an implicit hipDeviceSynchronize() call.
    !>    If pointer is NULL, the hip runtime is initialized and hipSuccess is returned.
    !>  
    !>    @param[in] ptr Pointer to memory to be freed
    !>    @return #hipSuccess,
    !>            #hipErrorInvalidValue (if pointer is invalid, including device pointers allocated with
    !>   hipMalloc)
    !>  
    !>    @see hipMalloc, hipMallocPitch, hipFree, hipMallocArray, hipFreeArray, hipMalloc3D,
    !>   hipMalloc3DArray, hipHostMalloc
    !>  
    function cudaHostFree_(ptr) bind(c, name="cudaHostFree")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaHostFree_
      type(c_ptr),value :: ptr
    end function

    module procedure cudaHostFree_l_0,&
      cudaHostFree_l_1,&
      cudaHostFree_l_2,&
      cudaHostFree_l_3,&
      cudaHostFree_l_4,&
      cudaHostFree_l_5,&
      cudaHostFree_l_6,&
      cudaHostFree_l_7,&
      cudaHostFree_i4_0,&
      cudaHostFree_i4_1,&
      cudaHostFree_i4_2,&
      cudaHostFree_i4_3,&
      cudaHostFree_i4_4,&
      cudaHostFree_i4_5,&
      cudaHostFree_i4_6,&
      cudaHostFree_i4_7,&
      cudaHostFree_i8_0,&
      cudaHostFree_i8_1,&
      cudaHostFree_i8_2,&
      cudaHostFree_i8_3,&
      cudaHostFree_i8_4,&
      cudaHostFree_i8_5,&
      cudaHostFree_i8_6,&
      cudaHostFree_i8_7,&
      cudaHostFree_r4_0,&
      cudaHostFree_r4_1,&
      cudaHostFree_r4_2,&
      cudaHostFree_r4_3,&
      cudaHostFree_r4_4,&
      cudaHostFree_r4_5,&
      cudaHostFree_r4_6,&
      cudaHostFree_r4_7,&
      cudaHostFree_r8_0,&
      cudaHostFree_r8_1,&
      cudaHostFree_r8_2,&
      cudaHostFree_r8_3,&
      cudaHostFree_r8_4,&
      cudaHostFree_r8_5,&
      cudaHostFree_r8_6,&
      cudaHostFree_r8_7,&
      cudaHostFree_c4_0,&
      cudaHostFree_c4_1,&
      cudaHostFree_c4_2,&
      cudaHostFree_c4_3,&
      cudaHostFree_c4_4,&
      cudaHostFree_c4_5,&
      cudaHostFree_c4_6,&
      cudaHostFree_c4_7,&
      cudaHostFree_c8_0,&
      cudaHostFree_c8_1,&
      cudaHostFree_c8_2,&
      cudaHostFree_c8_3,&
      cudaHostFree_c8_4,&
      cudaHostFree_c8_5,&
      cudaHostFree_c8_6,&
      cudaHostFree_c8_7 
  end interface

  contains

! scalars
                                                              
    function cudaMalloc_l_0_source(ptr,dsource,source)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,intent(inout) :: ptr
      logical(c_bool),target,intent(in),optional :: dsource,source
         
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMalloc_l_0_source = cudaMalloc_(cptr,1_8)
        cudaMalloc_l_0_source = cudaMemcpy(cptr,c_loc(dsource),1_8,&
&cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMalloc_l_0_source = cudaMalloc_(cptr,1_8)
        cudaMalloc_l_0_source = cudaMemcpy(cptr,c_loc(source),1_8,&
&cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMalloc_l_0_source = cudaMalloc_(cptr,1_8)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMalloc_l_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      logical(c_bool),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      logical(c_bool),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_l_1_source = cudaMalloc_(cptr,size(dsource)*1_8)
        cudaMalloc_l_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMalloc_l_1_source = cudaMalloc_(cptr,size(source)*1_8)
        cudaMalloc_l_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_l_1_source = cudaMalloc_(cptr,size(mold)*1_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_l_1_source = cudaMalloc_(cptr,PRODUCT(dims8)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_l_1_source = cudaMalloc_(cptr,PRODUCT(dims)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_l_1_c_int(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_1_c_int
      !
      cudaMalloc_l_1_c_int = cudaMalloc_(cptr,length1*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_l_1_c_size_t(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_1_c_size_t
      !
      cudaMalloc_l_1_c_size_t = cudaMalloc_(cptr,length1*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_l_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      logical(c_bool),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      logical(c_bool),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_l_2_source = cudaMalloc_(cptr,size(dsource)*1_8)
        cudaMalloc_l_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMalloc_l_2_source = cudaMalloc_(cptr,size(source)*1_8)
        cudaMalloc_l_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_l_2_source = cudaMalloc_(cptr,size(mold)*1_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_l_2_source = cudaMalloc_(cptr,PRODUCT(dims8)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_l_2_source = cudaMalloc_(cptr,PRODUCT(dims)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_l_2_c_int(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_2_c_int
      !
      cudaMalloc_l_2_c_int = cudaMalloc_(cptr,length1*length2*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_l_2_c_size_t(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_2_c_size_t
      !
      cudaMalloc_l_2_c_size_t = cudaMalloc_(cptr,length1*length2*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_l_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      logical(c_bool),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      logical(c_bool),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_l_3_source = cudaMalloc_(cptr,size(dsource)*1_8)
        cudaMalloc_l_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMalloc_l_3_source = cudaMalloc_(cptr,size(source)*1_8)
        cudaMalloc_l_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMalloc_l_3_source = cudaMalloc_(cptr,size(mold)*1_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_l_3_source = cudaMalloc_(cptr,PRODUCT(dims8)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_l_3_source = cudaMalloc_(cptr,PRODUCT(dims)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_l_3_c_int(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_3_c_int
      !
      cudaMalloc_l_3_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_l_3_c_size_t(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_3_c_size_t
      !
      cudaMalloc_l_3_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_l_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      logical(c_bool),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_l_4_source = cudaMalloc_(cptr,size(dsource)*1_8)
        cudaMalloc_l_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMalloc_l_4_source = cudaMalloc_(cptr,size(source)*1_8)
        cudaMalloc_l_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_l_4_source = cudaMalloc_(cptr,size(mold)*1_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_l_4_source = cudaMalloc_(cptr,PRODUCT(dims8)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_l_4_source = cudaMalloc_(cptr,PRODUCT(dims)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_l_4_c_int(ptr,length1,length2,length3,length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_4_c_int
      !
      cudaMalloc_l_4_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_l_4_c_size_t(ptr,length1,length2,length3,&
&length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_4_c_size_t
      !
      cudaMalloc_l_4_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_l_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      logical(c_bool),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_l_5_source = cudaMalloc_(cptr,size(dsource)*1_8)
        cudaMalloc_l_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMalloc_l_5_source = cudaMalloc_(cptr,size(source)*1_8)
        cudaMalloc_l_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_l_5_source = cudaMalloc_(cptr,size(mold)*1_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_l_5_source = cudaMalloc_(cptr,PRODUCT(dims8)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_l_5_source = cudaMalloc_(cptr,PRODUCT(dims)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_l_5_c_int(ptr,length1,length2,length3,length4,&
&length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_5_c_int
      !
      cudaMalloc_l_5_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_l_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_5_c_size_t
      !
      cudaMalloc_l_5_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_l_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      logical(c_bool),target,dimension(:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_l_6_source = cudaMalloc_(cptr,size(dsource)*1_8)
        cudaMalloc_l_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMalloc_l_6_source = cudaMalloc_(cptr,size(source)*1_8)
        cudaMalloc_l_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_l_6_source = cudaMalloc_(cptr,size(mold)*1_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_l_6_source = cudaMalloc_(cptr,PRODUCT(dims8)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_l_6_source = cudaMalloc_(cptr,PRODUCT(dims)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_l_6_c_int(ptr,length1,length2,length3,length4,&
&length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_6_c_int
      !
      cudaMalloc_l_6_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_l_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_6_c_size_t
      !
      cudaMalloc_l_6_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_l_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      logical(c_bool),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_l_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_l_7_source = cudaMalloc_(cptr,size(dsource)*1_8)
        cudaMalloc_l_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMalloc_l_7_source = cudaMalloc_(cptr,size(source)*1_8)
        cudaMalloc_l_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_l_7_source = cudaMalloc_(cptr,size(mold)*1_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMalloc_l_7_source = cudaMalloc_(cptr,PRODUCT(dims8)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_l_7_source = cudaMalloc_(cptr,PRODUCT(dims)*1_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_l_7_c_int(ptr,length1,length2,length3,length4,&
&length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_7_c_int
      !
      cudaMalloc_l_7_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMalloc_l_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_l_7_c_size_t
      !
      cudaMalloc_l_7_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMalloc_i4_0_source(ptr,dsource,source)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,intent(inout) :: ptr
      integer(c_int),target,intent(in),optional :: dsource,source
         
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMalloc_i4_0_source = cudaMalloc_(cptr,4_8)
        cudaMalloc_i4_0_source = cudaMemcpy(cptr,c_loc(dsource),4_8,&
&cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMalloc_i4_0_source = cudaMalloc_(cptr,4_8)
        cudaMalloc_i4_0_source = cudaMemcpy(cptr,c_loc(source),4_8,&
&cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMalloc_i4_0_source = cudaMalloc_(cptr,4_8)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMalloc_i4_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      integer(c_int),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_int),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i4_1_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_i4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i4_1_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_i4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i4_1_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i4_1_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i4_1_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i4_1_c_int(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_1_c_int
      !
      cudaMalloc_i4_1_c_int = cudaMalloc_(cptr,length1*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_i4_1_c_size_t(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_1_c_size_t
      !
      cudaMalloc_i4_1_c_size_t = cudaMalloc_(cptr,length1*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_i4_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      integer(c_int),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_int),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i4_2_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_i4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i4_2_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_i4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i4_2_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i4_2_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i4_2_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i4_2_c_int(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_2_c_int
      !
      cudaMalloc_i4_2_c_int = cudaMalloc_(cptr,length1*length2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_i4_2_c_size_t(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_2_c_size_t
      !
      cudaMalloc_i4_2_c_size_t = cudaMalloc_(cptr,length1*length2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_i4_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      integer(c_int),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_int),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i4_3_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_i4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMalloc_i4_3_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_i4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMalloc_i4_3_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i4_3_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i4_3_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i4_3_c_int(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_3_c_int
      !
      cudaMalloc_i4_3_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_i4_3_c_size_t(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_3_c_size_t
      !
      cudaMalloc_i4_3_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_i4_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      integer(c_int),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_int),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i4_4_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_i4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i4_4_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_i4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i4_4_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i4_4_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i4_4_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i4_4_c_int(ptr,length1,length2,length3,length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_4_c_int
      !
      cudaMalloc_i4_4_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_i4_4_c_size_t(ptr,length1,length2,length3,&
&length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_4_c_size_t
      !
      cudaMalloc_i4_4_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_i4_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      integer(c_int),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      integer(c_int),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i4_5_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_i4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i4_5_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_i4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i4_5_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i4_5_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i4_5_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i4_5_c_int(ptr,length1,length2,length3,length4,&
&length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_5_c_int
      !
      cudaMalloc_i4_5_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_i4_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_5_c_size_t
      !
      cudaMalloc_i4_5_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_i4_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      integer(c_int),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      integer(c_int),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i4_6_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_i4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i4_6_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_i4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i4_6_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i4_6_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i4_6_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i4_6_c_int(ptr,length1,length2,length3,length4,&
&length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_6_c_int
      !
      cudaMalloc_i4_6_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_i4_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_6_c_size_t
      !
      cudaMalloc_i4_6_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_i4_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      integer(c_int),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i4_7_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_i4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i4_7_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_i4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i4_7_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMalloc_i4_7_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i4_7_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i4_7_c_int(ptr,length1,length2,length3,length4,&
&length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_7_c_int
      !
      cudaMalloc_i4_7_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMalloc_i4_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i4_7_c_size_t
      !
      cudaMalloc_i4_7_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMalloc_i8_0_source(ptr,dsource,source)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,intent(inout) :: ptr
      integer(c_long),target,intent(in),optional :: dsource,source
         
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMalloc_i8_0_source = cudaMalloc_(cptr,8_8)
        cudaMalloc_i8_0_source = cudaMemcpy(cptr,c_loc(dsource),8_8,&
&cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMalloc_i8_0_source = cudaMalloc_(cptr,8_8)
        cudaMalloc_i8_0_source = cudaMemcpy(cptr,c_loc(source),8_8,&
&cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMalloc_i8_0_source = cudaMalloc_(cptr,8_8)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMalloc_i8_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      integer(c_long),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_long),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i8_1_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_i8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i8_1_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_i8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i8_1_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i8_1_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i8_1_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i8_1_c_int(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_1_c_int
      !
      cudaMalloc_i8_1_c_int = cudaMalloc_(cptr,length1*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_i8_1_c_size_t(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_1_c_size_t
      !
      cudaMalloc_i8_1_c_size_t = cudaMalloc_(cptr,length1*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_i8_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      integer(c_long),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_long),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i8_2_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_i8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i8_2_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_i8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i8_2_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i8_2_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i8_2_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i8_2_c_int(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_2_c_int
      !
      cudaMalloc_i8_2_c_int = cudaMalloc_(cptr,length1*length2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_i8_2_c_size_t(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_2_c_size_t
      !
      cudaMalloc_i8_2_c_size_t = cudaMalloc_(cptr,length1*length2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_i8_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      integer(c_long),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_long),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i8_3_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_i8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMalloc_i8_3_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_i8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMalloc_i8_3_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i8_3_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i8_3_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i8_3_c_int(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_3_c_int
      !
      cudaMalloc_i8_3_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_i8_3_c_size_t(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_3_c_size_t
      !
      cudaMalloc_i8_3_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_i8_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      integer(c_long),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      integer(c_long),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i8_4_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_i8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i8_4_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_i8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i8_4_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i8_4_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i8_4_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i8_4_c_int(ptr,length1,length2,length3,length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_4_c_int
      !
      cudaMalloc_i8_4_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_i8_4_c_size_t(ptr,length1,length2,length3,&
&length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_4_c_size_t
      !
      cudaMalloc_i8_4_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_i8_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      integer(c_long),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      integer(c_long),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i8_5_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_i8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i8_5_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_i8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i8_5_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i8_5_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i8_5_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i8_5_c_int(ptr,length1,length2,length3,length4,&
&length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_5_c_int
      !
      cudaMalloc_i8_5_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_i8_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_5_c_size_t
      !
      cudaMalloc_i8_5_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_i8_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      integer(c_long),target,dimension(:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      integer(c_long),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i8_6_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_i8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i8_6_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_i8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i8_6_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_i8_6_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i8_6_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i8_6_c_int(ptr,length1,length2,length3,length4,&
&length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_6_c_int
      !
      cudaMalloc_i8_6_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_i8_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_6_c_size_t
      !
      cudaMalloc_i8_6_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_i8_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      integer(c_long),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_i8_7_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_i8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMalloc_i8_7_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_i8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_i8_7_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMalloc_i8_7_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_i8_7_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_i8_7_c_int(ptr,length1,length2,length3,length4,&
&length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_7_c_int
      !
      cudaMalloc_i8_7_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMalloc_i8_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_i8_7_c_size_t
      !
      cudaMalloc_i8_7_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMalloc_r4_0_source(ptr,dsource,source)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,intent(inout) :: ptr
      real(c_float),target,intent(in),optional :: dsource,source
         
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMalloc_r4_0_source = cudaMalloc_(cptr,4_8)
        cudaMalloc_r4_0_source = cudaMemcpy(cptr,c_loc(dsource),4_8,&
&cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMalloc_r4_0_source = cudaMalloc_(cptr,4_8)
        cudaMalloc_r4_0_source = cudaMemcpy(cptr,c_loc(source),4_8,&
&cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMalloc_r4_0_source = cudaMalloc_(cptr,4_8)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMalloc_r4_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      real(c_float),target,dimension(:),intent(in),optional :: dsource,&
&source,mold
         
      ! 
      real(c_float),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r4_1_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_r4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r4_1_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_r4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r4_1_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r4_1_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r4_1_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r4_1_c_int(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_1_c_int
      !
      cudaMalloc_r4_1_c_int = cudaMalloc_(cptr,length1*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_r4_1_c_size_t(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_1_c_size_t
      !
      cudaMalloc_r4_1_c_size_t = cudaMalloc_(cptr,length1*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_r4_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      real(c_float),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_float),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r4_2_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_r4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r4_2_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_r4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r4_2_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r4_2_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r4_2_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r4_2_c_int(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_2_c_int
      !
      cudaMalloc_r4_2_c_int = cudaMalloc_(cptr,length1*length2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_r4_2_c_size_t(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_2_c_size_t
      !
      cudaMalloc_r4_2_c_size_t = cudaMalloc_(cptr,length1*length2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_r4_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      real(c_float),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_float),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r4_3_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_r4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMalloc_r4_3_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_r4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMalloc_r4_3_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r4_3_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r4_3_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r4_3_c_int(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_3_c_int
      !
      cudaMalloc_r4_3_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_r4_3_c_size_t(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_3_c_size_t
      !
      cudaMalloc_r4_3_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_r4_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      real(c_float),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_float),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r4_4_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_r4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r4_4_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_r4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r4_4_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r4_4_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r4_4_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r4_4_c_int(ptr,length1,length2,length3,length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_4_c_int
      !
      cudaMalloc_r4_4_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_r4_4_c_size_t(ptr,length1,length2,length3,&
&length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_4_c_size_t
      !
      cudaMalloc_r4_4_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_r4_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      real(c_float),target,dimension(:,:,:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_float),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r4_5_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_r4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r4_5_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_r4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r4_5_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r4_5_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r4_5_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r4_5_c_int(ptr,length1,length2,length3,length4,&
&length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_5_c_int
      !
      cudaMalloc_r4_5_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_r4_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_5_c_size_t
      !
      cudaMalloc_r4_5_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_r4_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      real(c_float),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      real(c_float),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r4_6_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_r4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r4_6_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_r4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r4_6_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r4_6_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r4_6_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r4_6_c_int(ptr,length1,length2,length3,length4,&
&length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_6_c_int
      !
      cudaMalloc_r4_6_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_r4_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_6_c_size_t
      !
      cudaMalloc_r4_6_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_r4_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      real(c_float),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      real(c_float),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r4_7_source = cudaMalloc_(cptr,size(dsource)*4_8)
        cudaMalloc_r4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r4_7_source = cudaMalloc_(cptr,size(source)*4_8)
        cudaMalloc_r4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r4_7_source = cudaMalloc_(cptr,size(mold)*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMalloc_r4_7_source = cudaMalloc_(cptr,PRODUCT(dims8)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r4_7_source = cudaMalloc_(cptr,PRODUCT(dims)*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r4_7_c_int(ptr,length1,length2,length3,length4,&
&length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_7_c_int
      !
      cudaMalloc_r4_7_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMalloc_r4_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r4_7_c_size_t
      !
      cudaMalloc_r4_7_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMalloc_r8_0_source(ptr,dsource,source)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,intent(inout) :: ptr
      real(c_double),target,intent(in),optional :: dsource,source
         
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMalloc_r8_0_source = cudaMalloc_(cptr,8_8)
        cudaMalloc_r8_0_source = cudaMemcpy(cptr,c_loc(dsource),8_8,&
&cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMalloc_r8_0_source = cudaMalloc_(cptr,8_8)
        cudaMalloc_r8_0_source = cudaMemcpy(cptr,c_loc(source),8_8,&
&cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMalloc_r8_0_source = cudaMalloc_(cptr,8_8)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMalloc_r8_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      real(c_double),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_double),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r8_1_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_r8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r8_1_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_r8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r8_1_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r8_1_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r8_1_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r8_1_c_int(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_1_c_int
      !
      cudaMalloc_r8_1_c_int = cudaMalloc_(cptr,length1*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_r8_1_c_size_t(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_1_c_size_t
      !
      cudaMalloc_r8_1_c_size_t = cudaMalloc_(cptr,length1*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_r8_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      real(c_double),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_double),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r8_2_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_r8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r8_2_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_r8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r8_2_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r8_2_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r8_2_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r8_2_c_int(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_2_c_int
      !
      cudaMalloc_r8_2_c_int = cudaMalloc_(cptr,length1*length2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_r8_2_c_size_t(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_2_c_size_t
      !
      cudaMalloc_r8_2_c_size_t = cudaMalloc_(cptr,length1*length2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_r8_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      real(c_double),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_double),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r8_3_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_r8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMalloc_r8_3_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_r8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMalloc_r8_3_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r8_3_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r8_3_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r8_3_c_int(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_3_c_int
      !
      cudaMalloc_r8_3_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_r8_3_c_size_t(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_3_c_size_t
      !
      cudaMalloc_r8_3_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_r8_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      real(c_double),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
         
      ! 
      real(c_double),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r8_4_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_r8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r8_4_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_r8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r8_4_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r8_4_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r8_4_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r8_4_c_int(ptr,length1,length2,length3,length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_4_c_int
      !
      cudaMalloc_r8_4_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_r8_4_c_size_t(ptr,length1,length2,length3,&
&length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_4_c_size_t
      !
      cudaMalloc_r8_4_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_r8_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      real(c_double),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      real(c_double),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r8_5_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_r8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r8_5_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_r8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r8_5_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r8_5_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r8_5_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r8_5_c_int(ptr,length1,length2,length3,length4,&
&length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_5_c_int
      !
      cudaMalloc_r8_5_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_r8_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_5_c_size_t
      !
      cudaMalloc_r8_5_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_r8_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      real(c_double),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      real(c_double),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r8_6_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_r8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r8_6_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_r8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r8_6_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_r8_6_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r8_6_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r8_6_c_int(ptr,length1,length2,length3,length4,&
&length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_6_c_int
      !
      cudaMalloc_r8_6_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_r8_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_6_c_size_t
      !
      cudaMalloc_r8_6_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_r8_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      real(c_double),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      real(c_double),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_r8_7_source = cudaMalloc_(cptr,size(dsource)*8_8)
        cudaMalloc_r8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMalloc_r8_7_source = cudaMalloc_(cptr,size(source)*8_8)
        cudaMalloc_r8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_r8_7_source = cudaMalloc_(cptr,size(mold)*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMalloc_r8_7_source = cudaMalloc_(cptr,PRODUCT(dims8)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_r8_7_source = cudaMalloc_(cptr,PRODUCT(dims)*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_r8_7_c_int(ptr,length1,length2,length3,length4,&
&length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_7_c_int
      !
      cudaMalloc_r8_7_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMalloc_r8_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_r8_7_c_size_t
      !
      cudaMalloc_r8_7_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMalloc_c4_0_source(ptr,dsource,source)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,intent(inout) :: ptr
      complex(c_float_complex),target,intent(in),optional :: dsource,&
&source
         
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMalloc_c4_0_source = cudaMalloc_(cptr,2*4_8)
        cudaMalloc_c4_0_source = cudaMemcpy(cptr,c_loc(dsource),2*4_8,&
&cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMalloc_c4_0_source = cudaMalloc_(cptr,2*4_8)
        cudaMalloc_c4_0_source = cudaMemcpy(cptr,c_loc(source),2*4_8,&
&cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMalloc_c4_0_source = cudaMalloc_(cptr,2*4_8)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMalloc_c4_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      complex(c_float_complex),target,dimension(:),intent(in),optional &
&:: dsource,source,mold
         
      ! 
      complex(c_float_complex),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c4_1_source = cudaMalloc_(cptr,size(dsource)*2*4_8)
        cudaMalloc_c4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c4_1_source = cudaMalloc_(cptr,size(source)*2*4_8)
        cudaMalloc_c4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c4_1_source = cudaMalloc_(cptr,size(mold)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c4_1_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c4_1_source = cudaMalloc_(cptr,PRODUCT(dims)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c4_1_c_int(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_int) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_1_c_int
      !
      cudaMalloc_c4_1_c_int = cudaMalloc_(cptr,length1*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_c4_1_c_size_t(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_1_c_size_t
      !
      cudaMalloc_c4_1_c_size_t = cudaMalloc_(cptr,length1*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_c4_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      complex(c_float_complex),target,dimension(:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_float_complex),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c4_2_source = cudaMalloc_(cptr,size(dsource)*2*4_8)
        cudaMalloc_c4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c4_2_source = cudaMalloc_(cptr,size(source)*2*4_8)
        cudaMalloc_c4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c4_2_source = cudaMalloc_(cptr,size(mold)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c4_2_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c4_2_source = cudaMalloc_(cptr,PRODUCT(dims)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c4_2_c_int(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_2_c_int
      !
      cudaMalloc_c4_2_c_int = cudaMalloc_(cptr,length1*length2*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_c4_2_c_size_t(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_2_c_size_t
      !
      cudaMalloc_c4_2_c_size_t = cudaMalloc_(cptr,length1*length2*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_c4_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      complex(c_float_complex),target,dimension(:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c4_3_source = cudaMalloc_(cptr,size(dsource)*2*4_8)
        cudaMalloc_c4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMalloc_c4_3_source = cudaMalloc_(cptr,size(source)*2*4_8)
        cudaMalloc_c4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMalloc_c4_3_source = cudaMalloc_(cptr,size(mold)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c4_3_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c4_3_source = cudaMalloc_(cptr,PRODUCT(dims)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c4_3_c_int(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_3_c_int
      !
      cudaMalloc_c4_3_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_c4_3_c_size_t(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_3_c_size_t
      !
      cudaMalloc_c4_3_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_c4_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      complex(c_float_complex),target,dimension(:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c4_4_source = cudaMalloc_(cptr,size(dsource)*2*4_8)
        cudaMalloc_c4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c4_4_source = cudaMalloc_(cptr,size(source)*2*4_8)
        cudaMalloc_c4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c4_4_source = cudaMalloc_(cptr,size(mold)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c4_4_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c4_4_source = cudaMalloc_(cptr,PRODUCT(dims)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c4_4_c_int(ptr,length1,length2,length3,length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_4_c_int
      !
      cudaMalloc_c4_4_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_c4_4_c_size_t(ptr,length1,length2,length3,&
&length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_4_c_size_t
      !
      cudaMalloc_c4_4_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_c4_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      complex(c_float_complex),target,dimension(:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c4_5_source = cudaMalloc_(cptr,size(dsource)*2*4_8)
        cudaMalloc_c4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c4_5_source = cudaMalloc_(cptr,size(source)*2*4_8)
        cudaMalloc_c4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c4_5_source = cudaMalloc_(cptr,size(mold)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c4_5_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c4_5_source = cudaMalloc_(cptr,PRODUCT(dims)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c4_5_c_int(ptr,length1,length2,length3,length4,&
&length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_5_c_int
      !
      cudaMalloc_c4_5_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_c4_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_5_c_size_t
      !
      cudaMalloc_c4_5_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_c4_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      complex(c_float_complex),target,dimension(:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
         
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c4_6_source = cudaMalloc_(cptr,size(dsource)*2*4_8)
        cudaMalloc_c4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c4_6_source = cudaMalloc_(cptr,size(source)*2*4_8)
        cudaMalloc_c4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c4_6_source = cudaMalloc_(cptr,size(mold)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c4_6_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c4_6_source = cudaMalloc_(cptr,PRODUCT(dims)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c4_6_c_int(ptr,length1,length2,length3,length4,&
&length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_6_c_int
      !
      cudaMalloc_c4_6_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_c4_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_6_c_size_t
      !
      cudaMalloc_c4_6_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_c4_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      complex(c_float_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
         
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c4_7_source = cudaMalloc_(cptr,size(dsource)*2*4_8)
        cudaMalloc_c4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c4_7_source = cudaMalloc_(cptr,size(source)*2*4_8)
        cudaMalloc_c4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c4_7_source = cudaMalloc_(cptr,size(mold)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMalloc_c4_7_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c4_7_source = cudaMalloc_(cptr,PRODUCT(dims)*2*4_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c4_7_c_int(ptr,length1,length2,length3,length4,&
&length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_7_c_int
      !
      cudaMalloc_c4_7_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMalloc_c4_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c4_7_c_size_t
      !
      cudaMalloc_c4_7_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMalloc_c8_0_source(ptr,dsource,source)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,intent(inout) :: ptr
      complex(c_double_complex),target,intent(in),optional :: dsource,&
&source
         
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMalloc_c8_0_source = cudaMalloc_(cptr,2*8_8)
        cudaMalloc_c8_0_source = cudaMemcpy(cptr,c_loc(dsource),2*8_8,&
&cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMalloc_c8_0_source = cudaMalloc_(cptr,2*8_8)
        cudaMalloc_c8_0_source = cudaMemcpy(cptr,c_loc(source),2*8_8,&
&cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMalloc_c8_0_source = cudaMalloc_(cptr,2*8_8)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMalloc_c8_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      complex(c_double_complex),target,dimension(:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_double_complex),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c8_1_source = cudaMalloc_(cptr,size(dsource)*2*8_8)
        cudaMalloc_c8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c8_1_source = cudaMalloc_(cptr,size(source)*2*8_8)
        cudaMalloc_c8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c8_1_source = cudaMalloc_(cptr,size(mold)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c8_1_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c8_1_source = cudaMalloc_(cptr,PRODUCT(dims)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c8_1_c_int(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_int) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_1_c_int
      !
      cudaMalloc_c8_1_c_int = cudaMalloc_(cptr,length1*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_c8_1_c_size_t(ptr,length1)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_1_c_size_t
      !
      cudaMalloc_c8_1_c_size_t = cudaMalloc_(cptr,length1*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMalloc_c8_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      complex(c_double_complex),target,dimension(:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_double_complex),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c8_2_source = cudaMalloc_(cptr,size(dsource)*2*8_8)
        cudaMalloc_c8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c8_2_source = cudaMalloc_(cptr,size(source)*2*8_8)
        cudaMalloc_c8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c8_2_source = cudaMalloc_(cptr,size(mold)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c8_2_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c8_2_source = cudaMalloc_(cptr,PRODUCT(dims)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c8_2_c_int(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_2_c_int
      !
      cudaMalloc_c8_2_c_int = cudaMalloc_(cptr,length1*length2*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_c8_2_c_size_t(ptr,length1,length2)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_2_c_size_t
      !
      cudaMalloc_c8_2_c_size_t = cudaMalloc_(cptr,length1*length2*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMalloc_c8_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      complex(c_double_complex),target,dimension(:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c8_3_source = cudaMalloc_(cptr,size(dsource)*2*8_8)
        cudaMalloc_c8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMalloc_c8_3_source = cudaMalloc_(cptr,size(source)*2*8_8)
        cudaMalloc_c8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMalloc_c8_3_source = cudaMalloc_(cptr,size(mold)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c8_3_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c8_3_source = cudaMalloc_(cptr,PRODUCT(dims)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c8_3_c_int(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_3_c_int
      !
      cudaMalloc_c8_3_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_c8_3_c_size_t(ptr,length1,length2,length3)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_3_c_size_t
      !
      cudaMalloc_c8_3_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMalloc_c8_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      complex(c_double_complex),target,dimension(:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c8_4_source = cudaMalloc_(cptr,size(dsource)*2*8_8)
        cudaMalloc_c8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c8_4_source = cudaMalloc_(cptr,size(source)*2*8_8)
        cudaMalloc_c8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c8_4_source = cudaMalloc_(cptr,size(mold)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c8_4_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c8_4_source = cudaMalloc_(cptr,PRODUCT(dims)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c8_4_c_int(ptr,length1,length2,length3,length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_4_c_int
      !
      cudaMalloc_c8_4_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_c8_4_c_size_t(ptr,length1,length2,length3,&
&length4)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_4_c_size_t
      !
      cudaMalloc_c8_4_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMalloc_c8_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      complex(c_double_complex),target,dimension(:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
         
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c8_5_source = cudaMalloc_(cptr,size(dsource)*2*8_8)
        cudaMalloc_c8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c8_5_source = cudaMalloc_(cptr,size(source)*2*8_8)
        cudaMalloc_c8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c8_5_source = cudaMalloc_(cptr,size(mold)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c8_5_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c8_5_source = cudaMalloc_(cptr,PRODUCT(dims)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c8_5_c_int(ptr,length1,length2,length3,length4,&
&length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_5_c_int
      !
      cudaMalloc_c8_5_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_c8_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_5_c_size_t
      !
      cudaMalloc_c8_5_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMalloc_c8_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      complex(c_double_complex),target,dimension(:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
         
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c8_6_source = cudaMalloc_(cptr,size(dsource)*2*8_8)
        cudaMalloc_c8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c8_6_source = cudaMalloc_(cptr,size(source)*2*8_8)
        cudaMalloc_c8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c8_6_source = cudaMalloc_(cptr,size(mold)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMalloc_c8_6_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c8_6_source = cudaMalloc_(cptr,PRODUCT(dims)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c8_6_c_int(ptr,length1,length2,length3,length4,&
&length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_6_c_int
      !
      cudaMalloc_c8_6_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_c8_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_6_c_size_t
      !
      cudaMalloc_c8_6_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMalloc_c8_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      complex(c_double_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
         
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMalloc_c8_7_source = cudaMalloc_(cptr,size(dsource)*2*8_8)
        cudaMalloc_c8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMalloc_c8_7_source = cudaMalloc_(cptr,size(source)*2*8_8)
        cudaMalloc_c8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMalloc_c8_7_source = cudaMalloc_(cptr,size(mold)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMalloc_c8_7_source = cudaMalloc_(cptr,PRODUCT(dims8)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMalloc_c8_7_source = cudaMalloc_(cptr,PRODUCT(dims)*2*8_8)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMalloc_c8_7_c_int(ptr,length1,length2,length3,length4,&
&length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_7_c_int
      !
      cudaMalloc_c8_7_c_int = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMalloc_c8_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
         
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMalloc_c8_7_c_size_t
      !
      cudaMalloc_c8_7_c_size_t = cudaMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMallocManaged_l_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,intent(inout) :: ptr
      logical(c_bool),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMallocManaged (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMallocManaged_l_0_source = cudaMallocManaged_(cptr,1_8,&
&flags)
        cudaMallocManaged_l_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMallocManaged_l_0_source = cudaMallocManaged_(cptr,1_8,&
&flags)
        cudaMallocManaged_l_0_source = cudaMemcpy(cptr,c_loc(source),&
&1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMallocManaged_l_0_source = cudaMallocManaged_(cptr,1_8,&
&flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMallocManaged_l_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      logical(c_bool),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      logical(c_bool),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_l_1_source = cudaMallocManaged_(cptr,&
&size(dsource)*1_8,flags)
        cudaMallocManaged_l_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_l_1_source = cudaMallocManaged_(cptr,&
&size(source)*1_8,flags)
        cudaMallocManaged_l_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_l_1_source = cudaMallocManaged_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_l_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_l_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_l_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_1_c_int
      !
      cudaMallocManaged_l_1_c_int = cudaMallocManaged_(cptr,&
&length1*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_l_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_1_c_size_t
      !
      cudaMallocManaged_l_1_c_size_t = cudaMallocManaged_(cptr,&
&length1*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_l_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      logical(c_bool),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      logical(c_bool),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_l_2_source = cudaMallocManaged_(cptr,&
&size(dsource)*1_8,flags)
        cudaMallocManaged_l_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_l_2_source = cudaMallocManaged_(cptr,&
&size(source)*1_8,flags)
        cudaMallocManaged_l_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_l_2_source = cudaMallocManaged_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_l_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_l_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_l_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_2_c_int
      !
      cudaMallocManaged_l_2_c_int = cudaMallocManaged_(cptr,&
&length1*length2*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_l_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_2_c_size_t
      !
      cudaMallocManaged_l_2_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_l_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      logical(c_bool),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      logical(c_bool),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_l_3_source = cudaMallocManaged_(cptr,&
&size(dsource)*1_8,flags)
        cudaMallocManaged_l_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMallocManaged_l_3_source = cudaMallocManaged_(cptr,&
&size(source)*1_8,flags)
        cudaMallocManaged_l_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMallocManaged_l_3_source = cudaMallocManaged_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_l_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_l_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_l_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_3_c_int
      !
      cudaMallocManaged_l_3_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_l_3_c_size_t(ptr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_3_c_size_t
      !
      cudaMallocManaged_l_3_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_l_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      logical(c_bool),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_l_4_source = cudaMallocManaged_(cptr,&
&size(dsource)*1_8,flags)
        cudaMallocManaged_l_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_l_4_source = cudaMallocManaged_(cptr,&
&size(source)*1_8,flags)
        cudaMallocManaged_l_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_l_4_source = cudaMallocManaged_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_l_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_l_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_l_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_4_c_int
      !
      cudaMallocManaged_l_4_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_l_4_c_size_t(ptr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_4_c_size_t
      !
      cudaMallocManaged_l_4_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_l_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      logical(c_bool),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_l_5_source = cudaMallocManaged_(cptr,&
&size(dsource)*1_8,flags)
        cudaMallocManaged_l_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_l_5_source = cudaMallocManaged_(cptr,&
&size(source)*1_8,flags)
        cudaMallocManaged_l_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_l_5_source = cudaMallocManaged_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_l_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_l_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_l_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_5_c_int
      !
      cudaMallocManaged_l_5_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_l_5_c_size_t(ptr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_5_c_size_t
      !
      cudaMallocManaged_l_5_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_l_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      logical(c_bool),target,dimension(:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_l_6_source = cudaMallocManaged_(cptr,&
&size(dsource)*1_8,flags)
        cudaMallocManaged_l_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_l_6_source = cudaMallocManaged_(cptr,&
&size(source)*1_8,flags)
        cudaMallocManaged_l_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_l_6_source = cudaMallocManaged_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_l_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_l_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_l_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_6_c_int
      !
      cudaMallocManaged_l_6_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_l_6_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_6_c_size_t
      !
      cudaMallocManaged_l_6_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_l_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      logical(c_bool),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_l_7_source = cudaMallocManaged_(cptr,&
&size(dsource)*1_8,flags)
        cudaMallocManaged_l_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_l_7_source = cudaMallocManaged_(cptr,&
&size(source)*1_8,flags)
        cudaMallocManaged_l_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_l_7_source = cudaMallocManaged_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_l_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_l_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_l_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_7_c_int
      !
      cudaMallocManaged_l_7_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMallocManaged_l_7_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_l_7_c_size_t
      !
      cudaMallocManaged_l_7_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMallocManaged_i4_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,intent(inout) :: ptr
      integer(c_int),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMallocManaged (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMallocManaged_i4_0_source = cudaMallocManaged_(cptr,4_8,&
&flags)
        cudaMallocManaged_i4_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMallocManaged_i4_0_source = cudaMallocManaged_(cptr,4_8,&
&flags)
        cudaMallocManaged_i4_0_source = cudaMemcpy(cptr,c_loc(source),&
&4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMallocManaged_i4_0_source = cudaMallocManaged_(cptr,4_8,&
&flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMallocManaged_i4_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      integer(c_int),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_int),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i4_1_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_i4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i4_1_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_i4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i4_1_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i4_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i4_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i4_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_1_c_int
      !
      cudaMallocManaged_i4_1_c_int = cudaMallocManaged_(cptr,&
&length1*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_i4_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_1_c_size_t
      !
      cudaMallocManaged_i4_1_c_size_t = cudaMallocManaged_(cptr,&
&length1*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_i4_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      integer(c_int),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_int),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i4_2_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_i4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i4_2_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_i4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i4_2_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i4_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i4_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i4_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_2_c_int
      !
      cudaMallocManaged_i4_2_c_int = cudaMallocManaged_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_i4_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_2_c_size_t
      !
      cudaMallocManaged_i4_2_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_i4_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      integer(c_int),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_int),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i4_3_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_i4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMallocManaged_i4_3_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_i4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i4_3_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i4_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i4_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i4_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_3_c_int
      !
      cudaMallocManaged_i4_3_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_i4_3_c_size_t(ptr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_3_c_size_t
      !
      cudaMallocManaged_i4_3_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_i4_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      integer(c_int),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_int),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i4_4_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_i4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i4_4_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_i4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i4_4_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i4_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i4_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i4_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_4_c_int
      !
      cudaMallocManaged_i4_4_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_i4_4_c_size_t(ptr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_4_c_size_t
      !
      cudaMallocManaged_i4_4_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_i4_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      integer(c_int),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_int),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i4_5_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_i4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i4_5_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_i4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i4_5_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i4_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i4_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i4_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_5_c_int
      !
      cudaMallocManaged_i4_5_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_i4_5_c_size_t(ptr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_5_c_size_t
      !
      cudaMallocManaged_i4_5_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_i4_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      integer(c_int),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_int),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i4_6_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_i4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i4_6_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_i4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i4_6_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i4_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i4_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i4_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_6_c_int
      !
      cudaMallocManaged_i4_6_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_i4_6_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_6_c_size_t
      !
      cudaMallocManaged_i4_6_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_i4_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      integer(c_int),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i4_7_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_i4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i4_7_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_i4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i4_7_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i4_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i4_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i4_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_7_c_int
      !
      cudaMallocManaged_i4_7_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMallocManaged_i4_7_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i4_7_c_size_t
      !
      cudaMallocManaged_i4_7_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMallocManaged_i8_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,intent(inout) :: ptr
      integer(c_long),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMallocManaged (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMallocManaged_i8_0_source = cudaMallocManaged_(cptr,8_8,&
&flags)
        cudaMallocManaged_i8_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMallocManaged_i8_0_source = cudaMallocManaged_(cptr,8_8,&
&flags)
        cudaMallocManaged_i8_0_source = cudaMemcpy(cptr,c_loc(source),&
&8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMallocManaged_i8_0_source = cudaMallocManaged_(cptr,8_8,&
&flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMallocManaged_i8_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      integer(c_long),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_long),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i8_1_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_i8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i8_1_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_i8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i8_1_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i8_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i8_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i8_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_1_c_int
      !
      cudaMallocManaged_i8_1_c_int = cudaMallocManaged_(cptr,&
&length1*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_i8_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_1_c_size_t
      !
      cudaMallocManaged_i8_1_c_size_t = cudaMallocManaged_(cptr,&
&length1*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_i8_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      integer(c_long),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_long),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i8_2_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_i8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i8_2_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_i8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i8_2_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i8_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i8_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i8_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_2_c_int
      !
      cudaMallocManaged_i8_2_c_int = cudaMallocManaged_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_i8_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_2_c_size_t
      !
      cudaMallocManaged_i8_2_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_i8_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      integer(c_long),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_long),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i8_3_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_i8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMallocManaged_i8_3_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_i8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i8_3_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i8_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i8_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i8_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_3_c_int
      !
      cudaMallocManaged_i8_3_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_i8_3_c_size_t(ptr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_3_c_size_t
      !
      cudaMallocManaged_i8_3_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_i8_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      integer(c_long),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_long),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i8_4_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_i8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i8_4_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_i8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i8_4_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i8_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i8_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i8_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_4_c_int
      !
      cudaMallocManaged_i8_4_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_i8_4_c_size_t(ptr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_4_c_size_t
      !
      cudaMallocManaged_i8_4_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_i8_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      integer(c_long),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_long),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i8_5_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_i8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i8_5_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_i8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i8_5_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i8_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i8_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i8_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_5_c_int
      !
      cudaMallocManaged_i8_5_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_i8_5_c_size_t(ptr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_5_c_size_t
      !
      cudaMallocManaged_i8_5_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_i8_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      integer(c_long),target,dimension(:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_long),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i8_6_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_i8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i8_6_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_i8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i8_6_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i8_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i8_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i8_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_6_c_int
      !
      cudaMallocManaged_i8_6_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_i8_6_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_6_c_size_t
      !
      cudaMallocManaged_i8_6_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_i8_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      integer(c_long),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_i8_7_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_i8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_i8_7_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_i8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_i8_7_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_i8_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_i8_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_i8_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_7_c_int
      !
      cudaMallocManaged_i8_7_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMallocManaged_i8_7_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_i8_7_c_size_t
      !
      cudaMallocManaged_i8_7_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMallocManaged_r4_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,intent(inout) :: ptr
      real(c_float),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMallocManaged (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMallocManaged_r4_0_source = cudaMallocManaged_(cptr,4_8,&
&flags)
        cudaMallocManaged_r4_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMallocManaged_r4_0_source = cudaMallocManaged_(cptr,4_8,&
&flags)
        cudaMallocManaged_r4_0_source = cudaMemcpy(cptr,c_loc(source),&
&4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMallocManaged_r4_0_source = cudaMallocManaged_(cptr,4_8,&
&flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMallocManaged_r4_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      real(c_float),target,dimension(:),intent(in),optional :: dsource,&
&source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_float),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r4_1_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_r4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r4_1_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_r4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r4_1_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r4_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r4_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r4_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_1_c_int
      !
      cudaMallocManaged_r4_1_c_int = cudaMallocManaged_(cptr,&
&length1*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_r4_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_1_c_size_t
      !
      cudaMallocManaged_r4_1_c_size_t = cudaMallocManaged_(cptr,&
&length1*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_r4_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      real(c_float),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_float),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r4_2_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_r4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r4_2_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_r4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r4_2_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r4_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r4_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r4_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_2_c_int
      !
      cudaMallocManaged_r4_2_c_int = cudaMallocManaged_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_r4_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_2_c_size_t
      !
      cudaMallocManaged_r4_2_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_r4_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      real(c_float),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_float),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r4_3_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_r4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMallocManaged_r4_3_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_r4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r4_3_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r4_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r4_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r4_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_3_c_int
      !
      cudaMallocManaged_r4_3_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_r4_3_c_size_t(ptr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_3_c_size_t
      !
      cudaMallocManaged_r4_3_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_r4_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      real(c_float),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_float),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r4_4_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_r4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r4_4_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_r4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r4_4_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r4_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r4_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r4_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_4_c_int
      !
      cudaMallocManaged_r4_4_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_r4_4_c_size_t(ptr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_4_c_size_t
      !
      cudaMallocManaged_r4_4_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_r4_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      real(c_float),target,dimension(:,:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_float),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r4_5_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_r4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r4_5_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_r4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r4_5_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r4_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r4_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r4_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_5_c_int
      !
      cudaMallocManaged_r4_5_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_r4_5_c_size_t(ptr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_5_c_size_t
      !
      cudaMallocManaged_r4_5_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_r4_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      real(c_float),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_float),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r4_6_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_r4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r4_6_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_r4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r4_6_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r4_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r4_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r4_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_6_c_int
      !
      cudaMallocManaged_r4_6_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_r4_6_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_6_c_size_t
      !
      cudaMallocManaged_r4_6_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_r4_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      real(c_float),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_float),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r4_7_source = cudaMallocManaged_(cptr,&
&size(dsource)*4_8,flags)
        cudaMallocManaged_r4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r4_7_source = cudaMallocManaged_(cptr,&
&size(source)*4_8,flags)
        cudaMallocManaged_r4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r4_7_source = cudaMallocManaged_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r4_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r4_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r4_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_7_c_int
      !
      cudaMallocManaged_r4_7_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMallocManaged_r4_7_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r4_7_c_size_t
      !
      cudaMallocManaged_r4_7_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMallocManaged_r8_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,intent(inout) :: ptr
      real(c_double),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMallocManaged (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMallocManaged_r8_0_source = cudaMallocManaged_(cptr,8_8,&
&flags)
        cudaMallocManaged_r8_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMallocManaged_r8_0_source = cudaMallocManaged_(cptr,8_8,&
&flags)
        cudaMallocManaged_r8_0_source = cudaMemcpy(cptr,c_loc(source),&
&8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMallocManaged_r8_0_source = cudaMallocManaged_(cptr,8_8,&
&flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMallocManaged_r8_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      real(c_double),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_double),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r8_1_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_r8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r8_1_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_r8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r8_1_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r8_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r8_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r8_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:), intent(inout) :: ptr
      integer(c_int) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_1_c_int
      !
      cudaMallocManaged_r8_1_c_int = cudaMallocManaged_(cptr,&
&length1*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_r8_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:), intent(inout) :: ptr
      integer(c_size_t) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_1_c_size_t
      !
      cudaMallocManaged_r8_1_c_size_t = cudaMallocManaged_(cptr,&
&length1*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_r8_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      real(c_double),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_double),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r8_2_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_r8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r8_2_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_r8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r8_2_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r8_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r8_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r8_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_2_c_int
      !
      cudaMallocManaged_r8_2_c_int = cudaMallocManaged_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_r8_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_2_c_size_t
      !
      cudaMallocManaged_r8_2_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_r8_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      real(c_double),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_double),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r8_3_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_r8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMallocManaged_r8_3_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_r8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r8_3_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r8_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r8_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r8_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_3_c_int
      !
      cudaMallocManaged_r8_3_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_r8_3_c_size_t(ptr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_3_c_size_t
      !
      cudaMallocManaged_r8_3_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_r8_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      real(c_double),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_double),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r8_4_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_r8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r8_4_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_r8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r8_4_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r8_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r8_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r8_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_4_c_int
      !
      cudaMallocManaged_r8_4_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_r8_4_c_size_t(ptr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_4_c_size_t
      !
      cudaMallocManaged_r8_4_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_r8_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      real(c_double),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_double),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r8_5_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_r8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r8_5_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_r8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r8_5_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r8_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r8_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r8_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_5_c_int
      !
      cudaMallocManaged_r8_5_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_r8_5_c_size_t(ptr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:), intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_5_c_size_t
      !
      cudaMallocManaged_r8_5_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_r8_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      real(c_double),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_double),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r8_6_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_r8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r8_6_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_r8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r8_6_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r8_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r8_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r8_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_6_c_int
      !
      cudaMallocManaged_r8_6_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_r8_6_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_6_c_size_t
      !
      cudaMallocManaged_r8_6_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_r8_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      real(c_double),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      real(c_double),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_r8_7_source = cudaMallocManaged_(cptr,&
&size(dsource)*8_8,flags)
        cudaMallocManaged_r8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_r8_7_source = cudaMallocManaged_(cptr,&
&size(source)*8_8,flags)
        cudaMallocManaged_r8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_r8_7_source = cudaMallocManaged_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_r8_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_r8_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_r8_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_7_c_int
      !
      cudaMallocManaged_r8_7_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMallocManaged_r8_7_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_r8_7_c_size_t
      !
      cudaMallocManaged_r8_7_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMallocManaged_c4_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,intent(inout) :: ptr
      complex(c_float_complex),target,intent(in),optional :: dsource,&
&source
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMallocManaged (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMallocManaged_c4_0_source = cudaMallocManaged_(cptr,2*4_8,&
&flags)
        cudaMallocManaged_c4_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMallocManaged_c4_0_source = cudaMallocManaged_(cptr,2*4_8,&
&flags)
        cudaMallocManaged_c4_0_source = cudaMemcpy(cptr,c_loc(source),&
&2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMallocManaged_c4_0_source = cudaMallocManaged_(cptr,2*4_8,&
&flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMallocManaged_c4_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      complex(c_float_complex),target,dimension(:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_float_complex),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c4_1_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaMallocManaged_c4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c4_1_source = cudaMallocManaged_(cptr,&
&size(source)*2*4_8,flags)
        cudaMallocManaged_c4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c4_1_source = cudaMallocManaged_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c4_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c4_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c4_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_int) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_1_c_int
      !
      cudaMallocManaged_c4_1_c_int = cudaMallocManaged_(cptr,&
&length1*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_c4_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_1_c_size_t
      !
      cudaMallocManaged_c4_1_c_size_t = cudaMallocManaged_(cptr,&
&length1*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_c4_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      complex(c_float_complex),target,dimension(:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_float_complex),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c4_2_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaMallocManaged_c4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c4_2_source = cudaMallocManaged_(cptr,&
&size(source)*2*4_8,flags)
        cudaMallocManaged_c4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c4_2_source = cudaMallocManaged_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c4_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c4_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c4_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_2_c_int
      !
      cudaMallocManaged_c4_2_c_int = cudaMallocManaged_(cptr,&
&length1*length2*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_c4_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_2_c_size_t
      !
      cudaMallocManaged_c4_2_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_c4_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      complex(c_float_complex),target,dimension(:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c4_3_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaMallocManaged_c4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMallocManaged_c4_3_source = cudaMallocManaged_(cptr,&
&size(source)*2*4_8,flags)
        cudaMallocManaged_c4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c4_3_source = cudaMallocManaged_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c4_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c4_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c4_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_3_c_int
      !
      cudaMallocManaged_c4_3_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_c4_3_c_size_t(ptr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_3_c_size_t
      !
      cudaMallocManaged_c4_3_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_c4_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      complex(c_float_complex),target,dimension(:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c4_4_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaMallocManaged_c4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c4_4_source = cudaMallocManaged_(cptr,&
&size(source)*2*4_8,flags)
        cudaMallocManaged_c4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c4_4_source = cudaMallocManaged_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c4_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c4_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c4_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_4_c_int
      !
      cudaMallocManaged_c4_4_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_c4_4_c_size_t(ptr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_4_c_size_t
      !
      cudaMallocManaged_c4_4_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_c4_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      complex(c_float_complex),target,dimension(:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c4_5_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaMallocManaged_c4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c4_5_source = cudaMallocManaged_(cptr,&
&size(source)*2*4_8,flags)
        cudaMallocManaged_c4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c4_5_source = cudaMallocManaged_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c4_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c4_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c4_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_5_c_int
      !
      cudaMallocManaged_c4_5_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_c4_5_c_size_t(ptr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_5_c_size_t
      !
      cudaMallocManaged_c4_5_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_c4_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      complex(c_float_complex),target,dimension(:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c4_6_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaMallocManaged_c4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c4_6_source = cudaMallocManaged_(cptr,&
&size(source)*2*4_8,flags)
        cudaMallocManaged_c4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c4_6_source = cudaMallocManaged_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c4_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c4_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c4_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_6_c_int
      !
      cudaMallocManaged_c4_6_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_c4_6_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_6_c_size_t
      !
      cudaMallocManaged_c4_6_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_c4_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      complex(c_float_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c4_7_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaMallocManaged_c4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c4_7_source = cudaMallocManaged_(cptr,&
&size(source)*2*4_8,flags)
        cudaMallocManaged_c4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c4_7_source = cudaMallocManaged_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c4_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c4_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c4_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_7_c_int
      !
      cudaMallocManaged_c4_7_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMallocManaged_c4_7_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c4_7_c_size_t
      !
      cudaMallocManaged_c4_7_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaMallocManaged_c8_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,intent(inout) :: ptr
      complex(c_double_complex),target,intent(in),optional :: dsource,&
&source
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMallocManaged (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."
    
      if ( present(dsource) ) then
        cudaMallocManaged_c8_0_source = cudaMallocManaged_(cptr,2*8_8,&
&flags)
        cudaMallocManaged_c8_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaMallocManaged_c8_0_source = cudaMallocManaged_(cptr,2*8_8,&
&flags)
        cudaMallocManaged_c8_0_source = cudaMemcpy(cptr,c_loc(source),&
&2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,ptr)
      else
        cudaMallocManaged_c8_0_source = cudaMallocManaged_(cptr,2*8_8,&
&flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaMallocManaged_c8_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      complex(c_double_complex),target,dimension(:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_double_complex),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c8_1_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaMallocManaged_c8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c8_1_source = cudaMallocManaged_(cptr,&
&size(source)*2*8_8,flags)
        cudaMallocManaged_c8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c8_1_source = cudaMallocManaged_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c8_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c8_1_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c8_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_int) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_1_c_int
      !
      cudaMallocManaged_c8_1_c_int = cudaMallocManaged_(cptr,&
&length1*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_c8_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:), intent(inout) :: &
&ptr
      integer(c_size_t) :: length1 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_1_c_size_t
      !
      cudaMallocManaged_c8_1_c_size_t = cudaMallocManaged_(cptr,&
&length1*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaMallocManaged_c8_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      complex(c_double_complex),target,dimension(:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_double_complex),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c8_2_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaMallocManaged_c8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c8_2_source = cudaMallocManaged_(cptr,&
&size(source)*2*8_8,flags)
        cudaMallocManaged_c8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c8_2_source = cudaMallocManaged_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c8_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c8_2_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c8_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_int) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_2_c_int
      !
      cudaMallocManaged_c8_2_c_int = cudaMallocManaged_(cptr,&
&length1*length2*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_c8_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:), intent(inout) &
&:: ptr
      integer(c_size_t) :: length1,length2 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_2_c_size_t
      !
      cudaMallocManaged_c8_2_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaMallocManaged_c8_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      complex(c_double_complex),target,dimension(:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c8_3_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaMallocManaged_c8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaMallocManaged_c8_3_source = cudaMallocManaged_(cptr,&
&size(source)*2*8_8,flags)
        cudaMallocManaged_c8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c8_3_source = cudaMallocManaged_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c8_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c8_3_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c8_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_3_c_int
      !
      cudaMallocManaged_c8_3_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_c8_3_c_size_t(ptr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_3_c_size_t
      !
      cudaMallocManaged_c8_3_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaMallocManaged_c8_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      complex(c_double_complex),target,dimension(:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c8_4_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaMallocManaged_c8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c8_4_source = cudaMallocManaged_(cptr,&
&size(source)*2*8_8,flags)
        cudaMallocManaged_c8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c8_4_source = cudaMallocManaged_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c8_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c8_4_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c8_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_4_c_int
      !
      cudaMallocManaged_c8_4_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_c8_4_c_size_t(ptr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_4_c_size_t
      !
      cudaMallocManaged_c8_4_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaMallocManaged_c8_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      complex(c_double_complex),target,dimension(:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c8_5_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaMallocManaged_c8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c8_5_source = cudaMallocManaged_(cptr,&
&size(source)*2*8_8,flags)
        cudaMallocManaged_c8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c8_5_source = cudaMallocManaged_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c8_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c8_5_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c8_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_5_c_int
      !
      cudaMallocManaged_c8_5_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_c8_5_c_size_t(ptr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_5_c_size_t
      !
      cudaMallocManaged_c8_5_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaMallocManaged_c8_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      complex(c_double_complex),target,dimension(:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c8_6_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaMallocManaged_c8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c8_6_source = cudaMallocManaged_(cptr,&
&size(source)*2*8_8,flags)
        cudaMallocManaged_c8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c8_6_source = cudaMallocManaged_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c8_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c8_6_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c8_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_6_c_int
      !
      cudaMallocManaged_c8_6_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_c8_6_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_6_c_size_t
      !
      cudaMallocManaged_c8_6_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaMallocManaged_c8_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      complex(c_double_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags   
      ! 
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaMalloc: At least one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."

      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaMallocManaged_c8_7_source = cudaMallocManaged_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaMallocManaged_c8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaMallocManaged_c8_7_source = cudaMallocManaged_(cptr,&
&size(source)*2*8_8,flags)
        cudaMallocManaged_c8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToDevice)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaMallocManaged_c8_7_source = cudaMallocManaged_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaMallocManaged_c8_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaMallocManaged_c8_7_source = cudaMallocManaged_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function

                                                              
    function cudaMallocManaged_c8_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_int) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_7_c_int
      !
      cudaMallocManaged_c8_7_c_int = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaMallocManaged_c8_7_c_size_t(ptr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(inout) :: ptr
      integer(c_size_t) :: length1,length2,length3,length4,length5,&
&length6,length7 
      integer(kind=4),intent(in) :: flags   
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaMallocManaged_c8_7_c_size_t
      !
      cudaMallocManaged_c8_7_c_size_t = cudaMallocManaged_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function


! scalars
                                                              
    function cudaHostMalloc_l_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,intent(inout) :: ptr
      logical(c_bool),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."

      if ( present(dsource) ) then
        cudaHostMalloc_l_0_source = cudaHostMalloc_(cptr,1_8,flags)
        cudaHostMalloc_l_0_source = cudaMemcpy(cptr,c_loc(dsource),1_8,&
&cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaHostMalloc_l_0_source = cudaHostMalloc_(cptr,1_8,flags)
        cudaHostMalloc_l_0_source = cudaMemcpy(cptr,c_loc(source),1_8,&
&cudaMemcpyHostToHost)
        call c_f_pointer(cptr,ptr)
      else
        cudaHostMalloc_l_0_source = cudaHostMalloc_(cptr,1_8,flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaHostMalloc_l_1_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      logical(c_bool),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      logical(c_bool),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_l_1_source = cudaHostMalloc_(cptr,&
&size(dsource)*1_8,flags)
        cudaHostMalloc_l_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_l_1_source = cudaHostMalloc_(cptr,&
&size(source)*1_8,flags)
        cudaHostMalloc_l_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_l_1_source = cudaHostMalloc_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_l_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_l_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_l_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:) :: ptr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_1_c_int
      !
      cudaHostMalloc_l_1_c_int = cudaHostMalloc_(cptr,length1*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_l_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:) :: ptr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_1_c_size_t
      !
      cudaHostMalloc_l_1_c_size_t = cudaHostMalloc_(cptr,length1*1_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_l_2_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      logical(c_bool),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      logical(c_bool),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_l_2_source = cudaHostMalloc_(cptr,&
&size(dsource)*1_8,flags)
        cudaHostMalloc_l_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_l_2_source = cudaHostMalloc_(cptr,&
&size(source)*1_8,flags)
        cudaHostMalloc_l_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_l_2_source = cudaHostMalloc_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_l_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_l_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_l_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_2_c_int
      !
      cudaHostMalloc_l_2_c_int = cudaHostMalloc_(cptr,&
&length1*length2*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_l_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_2_c_size_t
      !
      cudaHostMalloc_l_2_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_l_3_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      logical(c_bool),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      logical(c_bool),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_l_3_source = cudaHostMalloc_(cptr,&
&size(dsource)*1_8,flags)
        cudaHostMalloc_l_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaHostMalloc_l_3_source = cudaHostMalloc_(cptr,&
&size(source)*1_8,flags)
        cudaHostMalloc_l_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaHostMalloc_l_3_source = cudaHostMalloc_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_l_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_l_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_l_3_c_int(ptr,length1,length2,length3,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_3_c_int
      !
      cudaHostMalloc_l_3_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_l_3_c_size_t(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_3_c_size_t
      !
      cudaHostMalloc_l_3_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_l_4_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      logical(c_bool),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      logical(c_bool),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_l_4_source = cudaHostMalloc_(cptr,&
&size(dsource)*1_8,flags)
        cudaHostMalloc_l_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_l_4_source = cudaHostMalloc_(cptr,&
&size(source)*1_8,flags)
        cudaHostMalloc_l_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_l_4_source = cudaHostMalloc_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_l_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_l_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_l_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_4_c_int
      !
      cudaHostMalloc_l_4_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_l_4_c_size_t(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_4_c_size_t
      !
      cudaHostMalloc_l_4_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_l_5_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      logical(c_bool),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      logical(c_bool),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_l_5_source = cudaHostMalloc_(cptr,&
&size(dsource)*1_8,flags)
        cudaHostMalloc_l_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_l_5_source = cudaHostMalloc_(cptr,&
&size(source)*1_8,flags)
        cudaHostMalloc_l_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_l_5_source = cudaHostMalloc_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_l_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_l_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_l_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_5_c_int
      !
      cudaHostMalloc_l_5_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_l_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_5_c_size_t
      !
      cudaHostMalloc_l_5_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_l_6_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      logical(c_bool),target,dimension(:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      logical(c_bool),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_l_6_source = cudaHostMalloc_(cptr,&
&size(dsource)*1_8,flags)
        cudaHostMalloc_l_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_l_6_source = cudaHostMalloc_(cptr,&
&size(source)*1_8,flags)
        cudaHostMalloc_l_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_l_6_source = cudaHostMalloc_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_l_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_l_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_l_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_6_c_int
      !
      cudaHostMalloc_l_6_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_l_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_6_c_size_t
      !
      cudaHostMalloc_l_6_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_l_7_source(ptr,dims,dims8,lbounds,lbounds8,&
&dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      logical(c_bool),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_l_7_source = cudaHostMalloc_(cptr,&
&size(dsource)*1_8,flags)
        cudaHostMalloc_l_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*1_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_l_7_source = cudaHostMalloc_(cptr,&
&size(source)*1_8,flags)
        cudaHostMalloc_l_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*1_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_l_7_source = cudaHostMalloc_(cptr,&
&size(mold)*1_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_l_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_l_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*1_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_l_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_7_c_int
      !
      cudaHostMalloc_l_7_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaHostMalloc_l_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_l_7_c_size_t
      !
      cudaHostMalloc_l_7_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaHostMalloc_i4_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,intent(inout) :: ptr
      integer(c_int),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."

      if ( present(dsource) ) then
        cudaHostMalloc_i4_0_source = cudaHostMalloc_(cptr,4_8,flags)
        cudaHostMalloc_i4_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaHostMalloc_i4_0_source = cudaHostMalloc_(cptr,4_8,flags)
        cudaHostMalloc_i4_0_source = cudaMemcpy(cptr,c_loc(source),4_8,&
&cudaMemcpyHostToHost)
        call c_f_pointer(cptr,ptr)
      else
        cudaHostMalloc_i4_0_source = cudaHostMalloc_(cptr,4_8,flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaHostMalloc_i4_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      integer(c_int),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_int),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i4_1_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_i4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i4_1_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_i4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i4_1_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i4_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i4_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i4_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:) :: ptr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_1_c_int
      !
      cudaHostMalloc_i4_1_c_int = cudaHostMalloc_(cptr,length1*4_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_i4_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:) :: ptr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_1_c_size_t
      !
      cudaHostMalloc_i4_1_c_size_t = cudaHostMalloc_(cptr,length1*4_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_i4_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      integer(c_int),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_int),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i4_2_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_i4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i4_2_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_i4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i4_2_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i4_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i4_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i4_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_2_c_int
      !
      cudaHostMalloc_i4_2_c_int = cudaHostMalloc_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_i4_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_2_c_size_t
      !
      cudaHostMalloc_i4_2_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_i4_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      integer(c_int),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_int),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i4_3_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_i4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaHostMalloc_i4_3_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_i4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i4_3_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i4_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i4_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i4_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_3_c_int
      !
      cudaHostMalloc_i4_3_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_i4_3_c_size_t(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_3_c_size_t
      !
      cudaHostMalloc_i4_3_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_i4_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      integer(c_int),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_int),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i4_4_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_i4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i4_4_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_i4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i4_4_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i4_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i4_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i4_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_4_c_int
      !
      cudaHostMalloc_i4_4_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_i4_4_c_size_t(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_4_c_size_t
      !
      cudaHostMalloc_i4_4_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_i4_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      integer(c_int),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_int),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i4_5_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_i4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i4_5_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_i4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i4_5_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i4_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i4_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i4_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_5_c_int
      !
      cudaHostMalloc_i4_5_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_i4_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_5_c_size_t
      !
      cudaHostMalloc_i4_5_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_i4_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      integer(c_int),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_int),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i4_6_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_i4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i4_6_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_i4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i4_6_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i4_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i4_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i4_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_6_c_int
      !
      cudaHostMalloc_i4_6_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_i4_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_6_c_size_t
      !
      cudaHostMalloc_i4_6_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_i4_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      integer(c_int),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i4_7_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_i4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i4_7_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_i4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i4_7_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i4_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i4_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i4_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_7_c_int
      !
      cudaHostMalloc_i4_7_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaHostMalloc_i4_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i4_7_c_size_t
      !
      cudaHostMalloc_i4_7_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaHostMalloc_i8_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,intent(inout) :: ptr
      integer(c_long),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."

      if ( present(dsource) ) then
        cudaHostMalloc_i8_0_source = cudaHostMalloc_(cptr,8_8,flags)
        cudaHostMalloc_i8_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaHostMalloc_i8_0_source = cudaHostMalloc_(cptr,8_8,flags)
        cudaHostMalloc_i8_0_source = cudaMemcpy(cptr,c_loc(source),8_8,&
&cudaMemcpyHostToHost)
        call c_f_pointer(cptr,ptr)
      else
        cudaHostMalloc_i8_0_source = cudaHostMalloc_(cptr,8_8,flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaHostMalloc_i8_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      integer(c_long),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_long),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i8_1_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_i8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i8_1_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_i8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i8_1_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i8_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i8_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i8_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:) :: ptr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_1_c_int
      !
      cudaHostMalloc_i8_1_c_int = cudaHostMalloc_(cptr,length1*8_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_i8_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:) :: ptr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_1_c_size_t
      !
      cudaHostMalloc_i8_1_c_size_t = cudaHostMalloc_(cptr,length1*8_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_i8_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      integer(c_long),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_long),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i8_2_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_i8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i8_2_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_i8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i8_2_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i8_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i8_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i8_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_2_c_int
      !
      cudaHostMalloc_i8_2_c_int = cudaHostMalloc_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_i8_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_2_c_size_t
      !
      cudaHostMalloc_i8_2_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_i8_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      integer(c_long),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_long),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i8_3_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_i8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaHostMalloc_i8_3_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_i8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i8_3_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i8_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i8_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i8_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_3_c_int
      !
      cudaHostMalloc_i8_3_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_i8_3_c_size_t(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_3_c_size_t
      !
      cudaHostMalloc_i8_3_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_i8_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      integer(c_long),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_long),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i8_4_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_i8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i8_4_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_i8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i8_4_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i8_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i8_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i8_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_4_c_int
      !
      cudaHostMalloc_i8_4_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_i8_4_c_size_t(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_4_c_size_t
      !
      cudaHostMalloc_i8_4_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_i8_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      integer(c_long),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_long),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i8_5_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_i8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i8_5_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_i8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i8_5_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i8_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i8_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i8_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_5_c_int
      !
      cudaHostMalloc_i8_5_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_i8_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_5_c_size_t
      !
      cudaHostMalloc_i8_5_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_i8_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      integer(c_long),target,dimension(:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_long),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i8_6_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_i8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i8_6_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_i8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i8_6_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i8_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i8_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i8_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_6_c_int
      !
      cudaHostMalloc_i8_6_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_i8_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_6_c_size_t
      !
      cudaHostMalloc_i8_6_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_i8_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      integer(c_long),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_i8_7_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_i8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_i8_7_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_i8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_i8_7_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_i8_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_i8_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_i8_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_7_c_int
      !
      cudaHostMalloc_i8_7_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaHostMalloc_i8_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_i8_7_c_size_t
      !
      cudaHostMalloc_i8_7_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaHostMalloc_r4_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,intent(inout) :: ptr
      real(c_float),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."

      if ( present(dsource) ) then
        cudaHostMalloc_r4_0_source = cudaHostMalloc_(cptr,4_8,flags)
        cudaHostMalloc_r4_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaHostMalloc_r4_0_source = cudaHostMalloc_(cptr,4_8,flags)
        cudaHostMalloc_r4_0_source = cudaMemcpy(cptr,c_loc(source),4_8,&
&cudaMemcpyHostToHost)
        call c_f_pointer(cptr,ptr)
      else
        cudaHostMalloc_r4_0_source = cudaHostMalloc_(cptr,4_8,flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaHostMalloc_r4_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      real(c_float),target,dimension(:),intent(in),optional :: dsource,&
&source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_float),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r4_1_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_r4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r4_1_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_r4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r4_1_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r4_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r4_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r4_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:) :: ptr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_1_c_int
      !
      cudaHostMalloc_r4_1_c_int = cudaHostMalloc_(cptr,length1*4_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_r4_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:) :: ptr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_1_c_size_t
      !
      cudaHostMalloc_r4_1_c_size_t = cudaHostMalloc_(cptr,length1*4_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_r4_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      real(c_float),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_float),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r4_2_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_r4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r4_2_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_r4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r4_2_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r4_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r4_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r4_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_2_c_int
      !
      cudaHostMalloc_r4_2_c_int = cudaHostMalloc_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_r4_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_2_c_size_t
      !
      cudaHostMalloc_r4_2_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_r4_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      real(c_float),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_float),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r4_3_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_r4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaHostMalloc_r4_3_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_r4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r4_3_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r4_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r4_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r4_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_3_c_int
      !
      cudaHostMalloc_r4_3_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_r4_3_c_size_t(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_3_c_size_t
      !
      cudaHostMalloc_r4_3_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_r4_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      real(c_float),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_float),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r4_4_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_r4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r4_4_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_r4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r4_4_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r4_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r4_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r4_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_4_c_int
      !
      cudaHostMalloc_r4_4_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_r4_4_c_size_t(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_4_c_size_t
      !
      cudaHostMalloc_r4_4_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_r4_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      real(c_float),target,dimension(:,:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_float),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r4_5_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_r4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r4_5_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_r4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r4_5_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r4_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r4_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r4_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_5_c_int
      !
      cudaHostMalloc_r4_5_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_r4_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_5_c_size_t
      !
      cudaHostMalloc_r4_5_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_r4_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      real(c_float),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_float),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r4_6_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_r4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r4_6_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_r4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r4_6_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r4_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r4_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r4_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_6_c_int
      !
      cudaHostMalloc_r4_6_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_r4_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_6_c_size_t
      !
      cudaHostMalloc_r4_6_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_r4_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      real(c_float),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_float),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r4_7_source = cudaHostMalloc_(cptr,&
&size(dsource)*4_8,flags)
        cudaHostMalloc_r4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r4_7_source = cudaHostMalloc_(cptr,&
&size(source)*4_8,flags)
        cudaHostMalloc_r4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r4_7_source = cudaHostMalloc_(cptr,&
&size(mold)*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r4_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r4_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r4_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_7_c_int
      !
      cudaHostMalloc_r4_7_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaHostMalloc_r4_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r4_7_c_size_t
      !
      cudaHostMalloc_r4_7_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaHostMalloc_r8_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,intent(inout) :: ptr
      real(c_double),target,intent(in),optional :: dsource,source
      integer(kind=4),intent(in) :: flags
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."

      if ( present(dsource) ) then
        cudaHostMalloc_r8_0_source = cudaHostMalloc_(cptr,8_8,flags)
        cudaHostMalloc_r8_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaHostMalloc_r8_0_source = cudaHostMalloc_(cptr,8_8,flags)
        cudaHostMalloc_r8_0_source = cudaMemcpy(cptr,c_loc(source),8_8,&
&cudaMemcpyHostToHost)
        call c_f_pointer(cptr,ptr)
      else
        cudaHostMalloc_r8_0_source = cudaHostMalloc_(cptr,8_8,flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaHostMalloc_r8_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      real(c_double),target,dimension(:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_double),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r8_1_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_r8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r8_1_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_r8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r8_1_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r8_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r8_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r8_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:) :: ptr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_1_c_int
      !
      cudaHostMalloc_r8_1_c_int = cudaHostMalloc_(cptr,length1*8_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_r8_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:) :: ptr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_1_c_size_t
      !
      cudaHostMalloc_r8_1_c_size_t = cudaHostMalloc_(cptr,length1*8_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_r8_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      real(c_double),target,dimension(:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_double),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r8_2_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_r8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r8_2_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_r8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r8_2_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r8_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r8_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r8_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_2_c_int
      !
      cudaHostMalloc_r8_2_c_int = cudaHostMalloc_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_r8_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_2_c_size_t
      !
      cudaHostMalloc_r8_2_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_r8_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      real(c_double),target,dimension(:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_double),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r8_3_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_r8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaHostMalloc_r8_3_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_r8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r8_3_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r8_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r8_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r8_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_3_c_int
      !
      cudaHostMalloc_r8_3_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_r8_3_c_size_t(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_3_c_size_t
      !
      cudaHostMalloc_r8_3_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_r8_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      real(c_double),target,dimension(:,:,:,:),intent(in),optional :: &
&dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_double),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r8_4_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_r8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r8_4_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_r8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r8_4_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r8_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r8_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r8_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_4_c_int
      !
      cudaHostMalloc_r8_4_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_r8_4_c_size_t(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_4_c_size_t
      !
      cudaHostMalloc_r8_4_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_r8_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      real(c_double),target,dimension(:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_double),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r8_5_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_r8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r8_5_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_r8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r8_5_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r8_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r8_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r8_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_5_c_int
      !
      cudaHostMalloc_r8_5_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_r8_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_5_c_size_t
      !
      cudaHostMalloc_r8_5_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_r8_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      real(c_double),target,dimension(:,:,:,:,:,:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_double),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r8_6_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_r8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r8_6_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_r8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r8_6_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r8_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r8_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r8_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_6_c_int
      !
      cudaHostMalloc_r8_6_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_r8_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_6_c_size_t
      !
      cudaHostMalloc_r8_6_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_r8_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      real(c_double),target,dimension(:,:,:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      real(c_double),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_r8_7_source = cudaHostMalloc_(cptr,&
&size(dsource)*8_8,flags)
        cudaHostMalloc_r8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_r8_7_source = cudaHostMalloc_(cptr,&
&size(source)*8_8,flags)
        cudaHostMalloc_r8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_r8_7_source = cudaHostMalloc_(cptr,&
&size(mold)*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_r8_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_r8_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_r8_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_7_c_int
      !
      cudaHostMalloc_r8_7_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaHostMalloc_r8_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_r8_7_c_size_t
      !
      cudaHostMalloc_r8_7_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaHostMalloc_c4_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,intent(inout) :: ptr
      complex(c_float_complex),target,intent(in),optional :: dsource,&
&source
      integer(kind=4),intent(in) :: flags
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."

      if ( present(dsource) ) then
        cudaHostMalloc_c4_0_source = cudaHostMalloc_(cptr,2*4_8,flags)
        cudaHostMalloc_c4_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaHostMalloc_c4_0_source = cudaHostMalloc_(cptr,2*4_8,flags)
        cudaHostMalloc_c4_0_source = cudaMemcpy(cptr,c_loc(source),&
&2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,ptr)
      else
        cudaHostMalloc_c4_0_source = cudaHostMalloc_(cptr,2*4_8,flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaHostMalloc_c4_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:),intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      complex(c_float_complex),target,dimension(:),intent(in),optional &
&:: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_float_complex),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c4_1_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaHostMalloc_c4_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c4_1_source = cudaHostMalloc_(cptr,&
&size(source)*2*4_8,flags)
        cudaHostMalloc_c4_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c4_1_source = cudaHostMalloc_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c4_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c4_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c4_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:) :: ptr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_1_c_int
      !
      cudaHostMalloc_c4_1_c_int = cudaHostMalloc_(cptr,length1*2*4_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_c4_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:) :: ptr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_1_c_size_t
      !
      cudaHostMalloc_c4_1_c_size_t = cudaHostMalloc_(cptr,&
&length1*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_c4_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      complex(c_float_complex),target,dimension(:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_float_complex),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c4_2_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaHostMalloc_c4_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c4_2_source = cudaHostMalloc_(cptr,&
&size(source)*2*4_8,flags)
        cudaHostMalloc_c4_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c4_2_source = cudaHostMalloc_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c4_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c4_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c4_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_2_c_int
      !
      cudaHostMalloc_c4_2_c_int = cudaHostMalloc_(cptr,&
&length1*length2*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_c4_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_2_c_size_t
      !
      cudaHostMalloc_c4_2_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_c4_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      complex(c_float_complex),target,dimension(:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_float_complex),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c4_3_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaHostMalloc_c4_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaHostMalloc_c4_3_source = cudaHostMalloc_(cptr,&
&size(source)*2*4_8,flags)
        cudaHostMalloc_c4_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c4_3_source = cudaHostMalloc_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c4_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c4_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c4_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_3_c_int
      !
      cudaHostMalloc_c4_3_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_c4_3_c_size_t(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_3_c_size_t
      !
      cudaHostMalloc_c4_3_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_c4_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      complex(c_float_complex),target,dimension(:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_float_complex),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c4_4_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaHostMalloc_c4_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c4_4_source = cudaHostMalloc_(cptr,&
&size(source)*2*4_8,flags)
        cudaHostMalloc_c4_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c4_4_source = cudaHostMalloc_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c4_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c4_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c4_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_4_c_int
      !
      cudaHostMalloc_c4_4_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_c4_4_c_size_t(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_4_c_size_t
      !
      cudaHostMalloc_c4_4_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_c4_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      complex(c_float_complex),target,dimension(:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_float_complex),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c4_5_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaHostMalloc_c4_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c4_5_source = cudaHostMalloc_(cptr,&
&size(source)*2*4_8,flags)
        cudaHostMalloc_c4_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c4_5_source = cudaHostMalloc_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c4_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c4_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c4_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_5_c_int
      !
      cudaHostMalloc_c4_5_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_c4_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_5_c_size_t
      !
      cudaHostMalloc_c4_5_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_c4_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      complex(c_float_complex),target,dimension(:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c4_6_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaHostMalloc_c4_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c4_6_source = cudaHostMalloc_(cptr,&
&size(source)*2*4_8,flags)
        cudaHostMalloc_c4_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c4_6_source = cudaHostMalloc_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c4_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c4_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c4_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_6_c_int
      !
      cudaHostMalloc_c4_6_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_c4_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_6_c_size_t
      !
      cudaHostMalloc_c4_6_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_c4_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      complex(c_float_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c4_7_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*4_8,flags)
        cudaHostMalloc_c4_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*4_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c4_7_source = cudaHostMalloc_(cptr,&
&size(source)*2*4_8,flags)
        cudaHostMalloc_c4_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*4_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c4_7_source = cudaHostMalloc_(cptr,&
&size(mold)*2*4_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c4_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c4_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*4_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c4_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_7_c_int
      !
      cudaHostMalloc_c4_7_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaHostMalloc_c4_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c4_7_c_size_t
      !
      cudaHostMalloc_c4_7_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

! scalars
                                                              
    function cudaHostMalloc_c8_0_source(ptr,dsource,source,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,intent(inout) :: ptr
      complex(c_double_complex),target,intent(in),optional :: dsource,&
&source
      integer(kind=4),intent(in) :: flags
      !
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_0_source
      !
      nOptArgs = 0
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc (scalar &
&version): Only one optional argument ('dsource','source') must be &
&specified."

      if ( present(dsource) ) then
        cudaHostMalloc_c8_0_source = cudaHostMalloc_(cptr,2*8_8,flags)
        cudaHostMalloc_c8_0_source = cudaMemcpy(cptr,c_loc(dsource),&
&2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,ptr)
      else if ( present(source) ) then
        cudaHostMalloc_c8_0_source = cudaHostMalloc_(cptr,2*8_8,flags)
        cudaHostMalloc_c8_0_source = cudaMemcpy(cptr,c_loc(source),&
&2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,ptr)
      else
        cudaHostMalloc_c8_0_source = cudaHostMalloc_(cptr,2*8_8,flags)
        call c_f_pointer(cptr,ptr)
      end if
    end function

! arrays
                                                              
    function cudaHostMalloc_c8_1_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:),intent(inout) :: &
&ptr
      integer(4),intent(in),optional :: lbounds(1),dims(1)
      integer(8),intent(in),optional :: lbounds8(1),dims8(1)
      complex(c_double_complex),target,dimension(:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_double_complex),pointer,dimension(:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_1_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c8_1_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaHostMalloc_c8_1_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c8_1_source = cudaHostMalloc_(cptr,&
&size(source)*2*8_8,flags)
        cudaHostMalloc_c8_1_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c8_1_source = cudaHostMalloc_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c8_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c8_1_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c8_1_c_int(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:) :: ptr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_1_c_int
      !
      cudaHostMalloc_c8_1_c_int = cudaHostMalloc_(cptr,length1*2*8_8,&
&flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_c8_1_c_size_t(ptr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:) :: ptr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_1_c_size_t
      !
      cudaHostMalloc_c8_1_c_size_t = cudaHostMalloc_(cptr,&
&length1*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1])
    end function

                                                              
    function cudaHostMalloc_c8_2_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(2),dims(2)
      integer(8),intent(in),optional :: lbounds8(2),dims8(2)
      complex(c_double_complex),target,dimension(:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_double_complex),pointer,dimension(:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_2_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c8_2_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaHostMalloc_c8_2_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c8_2_source = cudaHostMalloc_(cptr,&
&size(source)*2*8_8,flags)
        cudaHostMalloc_c8_2_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c8_2_source = cudaHostMalloc_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c8_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c8_2_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c8_2_c_int(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_2_c_int
      !
      cudaHostMalloc_c8_2_c_int = cudaHostMalloc_(cptr,&
&length1*length2*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_c8_2_c_size_t(ptr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_2_c_size_t
      !
      cudaHostMalloc_c8_2_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2])
    end function

                                                              
    function cudaHostMalloc_c8_3_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:),intent(inout) &
&:: ptr
      integer(4),intent(in),optional :: lbounds(3),dims(3)
      integer(8),intent(in),optional :: lbounds8(3),dims8(3)
      complex(c_double_complex),target,dimension(:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_double_complex),pointer,dimension(:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_3_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c8_3_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaHostMalloc_c8_3_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):) &
&=> tmp
      else if ( present(source) ) then
        cudaHostMalloc_c8_3_source = cudaHostMalloc_(cptr,&
&size(source)*2*8_8,flags)
        cudaHostMalloc_c8_3_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):) => &
&tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c8_3_source = cudaHostMalloc_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c8_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c8_3_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c8_3_c_int(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_3_c_int
      !
      cudaHostMalloc_c8_3_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_c8_3_c_size_t(ptr,length1,length2,length3,&
&flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_3_c_size_t
      !
      cudaHostMalloc_c8_3_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3])
    end function

                                                              
    function cudaHostMalloc_c8_4_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(4),dims(4)
      integer(8),intent(in),optional :: lbounds8(4),dims8(4)
      complex(c_double_complex),target,dimension(:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_double_complex),pointer,dimension(:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_4_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c8_4_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaHostMalloc_c8_4_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c8_4_source = cudaHostMalloc_(cptr,&
&size(source)*2*8_8,flags)
        cudaHostMalloc_c8_4_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c8_4_source = cudaHostMalloc_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c8_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):) => &
&tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c8_4_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c8_4_c_int(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_4_c_int
      !
      cudaHostMalloc_c8_4_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_c8_4_c_size_t(ptr,length1,length2,length3,&
&length4,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_4_c_size_t
      !
      cudaHostMalloc_c8_4_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4])
    end function

                                                              
    function cudaHostMalloc_c8_5_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(5),dims(5)
      integer(8),intent(in),optional :: lbounds8(5),dims8(5)
      complex(c_double_complex),target,dimension(:,:,:,:,:),intent(in),&
&optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_double_complex),pointer,dimension(:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_5_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c8_5_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaHostMalloc_c8_5_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c8_5_source = cudaHostMalloc_(cptr,&
&size(source)*2*8_8,flags)
        cudaHostMalloc_c8_5_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c8_5_source = cudaHostMalloc_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c8_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c8_5_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c8_5_c_int(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_5_c_int
      !
      cudaHostMalloc_c8_5_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_c8_5_c_size_t(ptr,length1,length2,length3,&
&length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_5_c_size_t
      !
      cudaHostMalloc_c8_5_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5])
    end function

                                                              
    function cudaHostMalloc_c8_6_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(6),dims(6)
      integer(8),intent(in),optional :: lbounds8(6),dims8(6)
      complex(c_double_complex),target,dimension(:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_6_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c8_6_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaHostMalloc_c8_6_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c8_6_source = cudaHostMalloc_(cptr,&
&size(source)*2*8_8,flags)
        cudaHostMalloc_c8_6_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c8_6_source = cudaHostMalloc_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):) => tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c8_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c8_6_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c8_6_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_6_c_int
      !
      cudaHostMalloc_c8_6_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_c8_6_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_6_c_size_t
      !
      cudaHostMalloc_c8_6_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6])
    end function

                                                              
    function cudaHostMalloc_c8_7_source(ptr,dims,dims8,lbounds,&
&lbounds8,dsource,source,mold,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      use cudafor_cudamemcpy, ONLY: cudaMemcpy
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: ptr
      integer(4),intent(in),optional :: lbounds(7),dims(7)
      integer(8),intent(in),optional :: lbounds8(7),dims8(7)
      complex(c_double_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in),optional :: dsource,source,mold
      integer(kind=4),intent(in) :: flags
      !
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:) :: tmp
      type(c_ptr) :: cptr
      integer :: nOptArgs 
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_7_source
      !
      nOptArgs = 0
      if ( present(dims) ) nOptArgs = nOptArgs + 1
      if ( present(dims8) ) nOptArgs = nOptArgs + 1
      if ( present(dsource) ) nOptArgs = nOptArgs + 1
      if ( present(source) ) nOptArgs = nOptArgs + 1
      if ( present(mold) ) nOptArgs = nOptArgs + 1
      if ( nOptArgs == 0 ) ERROR STOP "ERROR: cudaHostMalloc: At least &
&one optional argument ('dims','dims8','dsource','source','mold') must &
&be specified."
      if ( nOptArgs > 1 ) ERROR STOP "ERROR: cudaHostMalloc: Only one &
&optional argument ('dims','dims8','dsource','source','mold') must be &
&specified."
      
      if ( present(lbounds8) .and. .not. present(dims8) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds8' may only be &
&specified in combination with 'dims8'."
      else if ( present(lbounds) .and. .not. present(dims) ) then
        ERROR STOP "ERROR: cudaMalloc: 'lbounds' may only be specified &
&in combination with 'dims'."
      endif

      if ( present(dsource) ) then
        cudaHostMalloc_c8_7_source = cudaHostMalloc_(cptr,&
&size(dsource)*2*8_8,flags)
        cudaHostMalloc_c8_7_source = cudaMemcpy(cptr,c_loc(dsource),&
&size(dsource)*2*8_8,cudaMemcpyDeviceToHost)
        call c_f_pointer(cptr,tmp,shape=shape(dsource))
        ptr(LBOUND(dsource,1):,LBOUND(dsource,2):,LBOUND(dsource,3):,&
&LBOUND(dsource,4):,LBOUND(dsource,5):,LBOUND(dsource,6):,&
&LBOUND(dsource,7):) => tmp
      else if ( present(source) ) then
        cudaHostMalloc_c8_7_source = cudaHostMalloc_(cptr,&
&size(source)*2*8_8,flags)
        cudaHostMalloc_c8_7_source = cudaMemcpy(cptr,c_loc(source),&
&size(source)*2*8_8,cudaMemcpyHostToHost)
        call c_f_pointer(cptr,tmp,shape=shape(source))
        ptr(LBOUND(source,1):,LBOUND(source,2):,LBOUND(source,3):,&
&LBOUND(source,4):,LBOUND(source,5):,LBOUND(source,6):,LBOUND(source,&
&7):) => tmp
      else if ( present(mold) ) then
        cudaHostMalloc_c8_7_source = cudaHostMalloc_(cptr,&
&size(mold)*2*8_8,flags)
        call c_f_pointer(cptr,ptr,shape=shape(mold))
        ptr(LBOUND(mold,1):,LBOUND(mold,2):,LBOUND(mold,3):,&
&LBOUND(mold,4):,LBOUND(mold,5):,LBOUND(mold,6):,LBOUND(mold,7):) => &
&tmp
      else if ( present(dims8) ) then
        cudaHostMalloc_c8_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims8)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims8)
        if ( present(lbounds8) ) then 
          ptr(lbounds8(1):,lbounds8(2):,lbounds8(3):,lbounds8(4):,&
&lbounds8(5):,lbounds8(6):,lbounds8(7):) => tmp
        else
          ptr => tmp
        end if
      else if ( present(dims) ) then
        cudaHostMalloc_c8_7_source = cudaHostMalloc_(cptr,&
&PRODUCT(dims)*2*8_8,flags)
        call c_f_pointer(cptr,tmp,shape=dims)
        if ( present(lbounds) ) then 
          ptr(lbounds(1):,lbounds(2):,lbounds(3):,lbounds(4):,&
&lbounds(5):,lbounds(6):,lbounds(7):) => tmp
        else
          ptr => tmp
        end if
      end if
    end function
    
                                                              
    function cudaHostMalloc_c8_7_c_int(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_7_c_int
      !
      cudaHostMalloc_c8_7_c_int = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function

                                                              
    function cudaHostMalloc_c8_7_c_size_t(ptr,length1,length2,length3,&
&length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostMalloc_c8_7_c_size_t
      !
      cudaHostMalloc_c8_7_c_size_t = cudaHostMalloc_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8,flags)
      call c_f_pointer(cptr,ptr,shape=[length1,length2,length3,length4,&
&length5,length6,length7])
    end function


                                        
    function cudaFree_l_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_0
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_0 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_l_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_1
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_1 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_l_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_2
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_2 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_l_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_3
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_3 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_l_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_4
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_4 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_l_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_5
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_5 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_l_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_6
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_6 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_l_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_l_7
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_l_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_l_7 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaFree_i4_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_0
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_0 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i4_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_1
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_1 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i4_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_2
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_2 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i4_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_3
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_3 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i4_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_4
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_4 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i4_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_5
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_5 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i4_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_6
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_6 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i4_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i4_7
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i4_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i4_7 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaFree_i8_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_0
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_0 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i8_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_1
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_1 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i8_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_2
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_2 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i8_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_3
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_3 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i8_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_4
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_4 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i8_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_5
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_5 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i8_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_6
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_6 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_i8_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_i8_7
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_i8_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_i8_7 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaFree_r4_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_0
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_0 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r4_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_1
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_1 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r4_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_2
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_2 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r4_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_3
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_3 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r4_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_4
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_4 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r4_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_5
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_5 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r4_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_6
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_6 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r4_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r4_7
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r4_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r4_7 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaFree_r8_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_0
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_0 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r8_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_1
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_1 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r8_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_2
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_2 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r8_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_3
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_3 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r8_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_4
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_4 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r8_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_5
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_5 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r8_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_6
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_6 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_r8_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_r8_7
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_r8_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_r8_7 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaFree_c4_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_0
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_0 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c4_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_1
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_1 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c4_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_2
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_2 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c4_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_3
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_3 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c4_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_4
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_4 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c4_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_5
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_5 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c4_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_6
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_6 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c4_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c4_7
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c4_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c4_7 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaFree_c8_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_0
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_0 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c8_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_1
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_1 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c8_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_2
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_2 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c8_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_3
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_3 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c8_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_4
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_4 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c8_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_5
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_5 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c8_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_6
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_6 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
 
                                        
    function cudaFree_c8_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaFree_c8_7
      !
      logical :: opt_only_if_allocated
      !
      cudaFree_c8_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaFree_c8_7 = cudaFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
    
                                        
    function cudaHostFree_l_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_0
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_0 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_l_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_1
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_1 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_l_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_2
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_2 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_l_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_3
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_3 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_l_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_4
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_4 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_l_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_5
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_5 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_l_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_6
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_6 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_l_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_l_7
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_l_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_l_7 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_0
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_0 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_1
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_1 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_2
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_2 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_3
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_3 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_4
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_4 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_5
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_5 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_6
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_6 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i4_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i4_7
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i4_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i4_7 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_0
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_0 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_1
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_1 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_2
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_2 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_3
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_3 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_4
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_4 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_5
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_5 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_6
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_6 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_i8_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_i8_7
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_i8_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_i8_7 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_0
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_0 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_1
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_1 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_2
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_2 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_3
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_3 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_4
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_4 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_5
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_5 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_6
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_6 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r4_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r4_7
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r4_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r4_7 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_0
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_0 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_1
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_1 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_2
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_2 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_3
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_3 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_4
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_4 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_5
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_5 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_6
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_6 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_r8_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_r8_7
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_r8_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_r8_7 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_0
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_0 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_1
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_1 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_2
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_2 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_3
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_3 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_4
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_4 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_5
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_5 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_6
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_6 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c4_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c4_7
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c4_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c4_7 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_0(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_0
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_0 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_0 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_1(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_1
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_1 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_1 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_2(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_2
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_2 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_2 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_3(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_3
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_3 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_3 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_4(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_4
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_4 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_4 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_5(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_5
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_5 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_5 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_6(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_6
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_6 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_6 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function
                                        
    function cudaHostFree_c8_7(ptr,only_if_allocated)
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:) :: ptr
      logical,intent(in),optional :: only_if_allocated
      integer(kind(cudaSuccess)) :: cudaHostFree_c8_7
      !
      logical :: opt_only_if_allocated
      !
      cudaHostFree_c8_7 = cudaSuccess
      opt_only_if_allocated = .FALSE.
      if ( present(only_if_allocated) ) opt_only_if_allocated = &
&only_if_allocated
      if ( .not. opt_only_if_allocated .or. associated(ptr) ) then
        cudaHostFree_c8_7 = cudaHostFree_(c_loc(ptr))
        nullify(ptr)
      endif
    end function

end module