


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ==============================================================================
! hipfort: FORTRAN Interfaces for GPU kernels
! ==============================================================================
! Copyright (c) 2020-2022 Advanced Micro Devices, Inc. All rights reserved.
! [MITx11 License]
! 
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
! IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
! AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
! OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
! THE SOFTWARE.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

module cudafor_cudahostregister

  !> 
  !>    @brief Register host memory so it can be accessed from the current device.
  !>  
  !>    @param[out] hostPtr Pointer to host memory to be registered.
  !>    @param[in] sizeBytes size of the host memory
  !>    @param[in] flags.  See below.
  !>  
  !>    Flags:
  !>    - #hipHostRegisterDefault   Memory is Mapped and Portable
  !>    - #hipHostRegisterPortable  Memory is considered registered by all contexts.  HIP only supports
  !>   one context so this is always assumed true.
  !>    - #hipHostRegisterMapped    Map the allocation into the address space for the current device.
  !>   The device pointer can be obtained with #hipHostGetDevicePointer.
  !>  
  !>  
  !>    After registering the memory, use #hipHostGetDevicePointer to obtain the mapped device pointer.
  !>    On many systems, the mapped device pointer will have a different value than the mapped host
  !>   pointer.  Applications must use the device pointer in device code, and the host pointer in device
  !>   code.
  !>  
  !>    On some systems, registered memory is pinned.  On some systems, registered memory may not be
  !>   actually be pinned but uses OS or hardware facilities to all GPU access to the host memory.
  !>  
  !>    Developers are strongly encouraged to register memory blocks which are aligned to the host
  !>   cache-line size. (typically 64-bytes but can be obtains from the CPUID instruction).
  !>  
  !>    If registering non-aligned pointers, the application must take care when register pointers from
  !>   the same cache line on different devices.  HIP's coarse-grained synchronization model does not
  !>   guarantee correct results if different devices write to different parts of the same cache block -
  !>   typically one of the writes will "win" and overwrite data from the other registered memory
  !>   region.
  !>  
  !>    @return #hipSuccess, #hipErrorOutOfMemory
  !>  
  !>    @see hipHostUnregister, hipHostGetFlags, hipHostGetDevicePointer
  !>  
  interface cudaHostRegister
    function cudaHostRegister_(hostPtr,sizeBytes,flags) bind(c, &
&name="cudaHostRegister")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaHostRegister_
      type(c_ptr),value :: hostPtr
      integer(c_size_t),value :: sizeBytes
      integer(kind=4),value :: flags
    end function

    module procedure cudaHostRegister_l_0_nosize,&
      cudaHostRegister_l_1_nosize,&
      cudaHostRegister_l_1_c_int,&
      cudaHostRegister_l_1_c_size_t,&
      cudaHostRegister_l_2_nosize,&
      cudaHostRegister_l_2_c_int,&
      cudaHostRegister_l_2_c_size_t,&
      cudaHostRegister_l_3_nosize,&
      cudaHostRegister_l_3_c_int,&
      cudaHostRegister_l_3_c_size_t,&
      cudaHostRegister_l_4_nosize,&
      cudaHostRegister_l_4_c_int,&
      cudaHostRegister_l_4_c_size_t,&
      cudaHostRegister_l_5_nosize,&
      cudaHostRegister_l_5_c_int,&
      cudaHostRegister_l_5_c_size_t,&
      cudaHostRegister_l_6_nosize,&
      cudaHostRegister_l_6_c_int,&
      cudaHostRegister_l_6_c_size_t,&
      cudaHostRegister_l_7_nosize,&
      cudaHostRegister_l_7_c_int,&
      cudaHostRegister_l_7_c_size_t,&
      cudaHostRegister_i4_0_nosize,&
      cudaHostRegister_i4_1_nosize,&
      cudaHostRegister_i4_1_c_int,&
      cudaHostRegister_i4_1_c_size_t,&
      cudaHostRegister_i4_2_nosize,&
      cudaHostRegister_i4_2_c_int,&
      cudaHostRegister_i4_2_c_size_t,&
      cudaHostRegister_i4_3_nosize,&
      cudaHostRegister_i4_3_c_int,&
      cudaHostRegister_i4_3_c_size_t,&
      cudaHostRegister_i4_4_nosize,&
      cudaHostRegister_i4_4_c_int,&
      cudaHostRegister_i4_4_c_size_t,&
      cudaHostRegister_i4_5_nosize,&
      cudaHostRegister_i4_5_c_int,&
      cudaHostRegister_i4_5_c_size_t,&
      cudaHostRegister_i4_6_nosize,&
      cudaHostRegister_i4_6_c_int,&
      cudaHostRegister_i4_6_c_size_t,&
      cudaHostRegister_i4_7_nosize,&
      cudaHostRegister_i4_7_c_int,&
      cudaHostRegister_i4_7_c_size_t,&
      cudaHostRegister_i8_0_nosize,&
      cudaHostRegister_i8_1_nosize,&
      cudaHostRegister_i8_1_c_int,&
      cudaHostRegister_i8_1_c_size_t,&
      cudaHostRegister_i8_2_nosize,&
      cudaHostRegister_i8_2_c_int,&
      cudaHostRegister_i8_2_c_size_t,&
      cudaHostRegister_i8_3_nosize,&
      cudaHostRegister_i8_3_c_int,&
      cudaHostRegister_i8_3_c_size_t,&
      cudaHostRegister_i8_4_nosize,&
      cudaHostRegister_i8_4_c_int,&
      cudaHostRegister_i8_4_c_size_t,&
      cudaHostRegister_i8_5_nosize,&
      cudaHostRegister_i8_5_c_int,&
      cudaHostRegister_i8_5_c_size_t,&
      cudaHostRegister_i8_6_nosize,&
      cudaHostRegister_i8_6_c_int,&
      cudaHostRegister_i8_6_c_size_t,&
      cudaHostRegister_i8_7_nosize,&
      cudaHostRegister_i8_7_c_int,&
      cudaHostRegister_i8_7_c_size_t,&
      cudaHostRegister_r4_0_nosize,&
      cudaHostRegister_r4_1_nosize,&
      cudaHostRegister_r4_1_c_int,&
      cudaHostRegister_r4_1_c_size_t,&
      cudaHostRegister_r4_2_nosize,&
      cudaHostRegister_r4_2_c_int,&
      cudaHostRegister_r4_2_c_size_t,&
      cudaHostRegister_r4_3_nosize,&
      cudaHostRegister_r4_3_c_int,&
      cudaHostRegister_r4_3_c_size_t,&
      cudaHostRegister_r4_4_nosize,&
      cudaHostRegister_r4_4_c_int,&
      cudaHostRegister_r4_4_c_size_t,&
      cudaHostRegister_r4_5_nosize,&
      cudaHostRegister_r4_5_c_int,&
      cudaHostRegister_r4_5_c_size_t,&
      cudaHostRegister_r4_6_nosize,&
      cudaHostRegister_r4_6_c_int,&
      cudaHostRegister_r4_6_c_size_t,&
      cudaHostRegister_r4_7_nosize,&
      cudaHostRegister_r4_7_c_int,&
      cudaHostRegister_r4_7_c_size_t,&
      cudaHostRegister_r8_0_nosize,&
      cudaHostRegister_r8_1_nosize,&
      cudaHostRegister_r8_1_c_int,&
      cudaHostRegister_r8_1_c_size_t,&
      cudaHostRegister_r8_2_nosize,&
      cudaHostRegister_r8_2_c_int,&
      cudaHostRegister_r8_2_c_size_t,&
      cudaHostRegister_r8_3_nosize,&
      cudaHostRegister_r8_3_c_int,&
      cudaHostRegister_r8_3_c_size_t,&
      cudaHostRegister_r8_4_nosize,&
      cudaHostRegister_r8_4_c_int,&
      cudaHostRegister_r8_4_c_size_t,&
      cudaHostRegister_r8_5_nosize,&
      cudaHostRegister_r8_5_c_int,&
      cudaHostRegister_r8_5_c_size_t,&
      cudaHostRegister_r8_6_nosize,&
      cudaHostRegister_r8_6_c_int,&
      cudaHostRegister_r8_6_c_size_t,&
      cudaHostRegister_r8_7_nosize,&
      cudaHostRegister_r8_7_c_int,&
      cudaHostRegister_r8_7_c_size_t,&
      cudaHostRegister_c4_0_nosize,&
      cudaHostRegister_c4_1_nosize,&
      cudaHostRegister_c4_1_c_int,&
      cudaHostRegister_c4_1_c_size_t,&
      cudaHostRegister_c4_2_nosize,&
      cudaHostRegister_c4_2_c_int,&
      cudaHostRegister_c4_2_c_size_t,&
      cudaHostRegister_c4_3_nosize,&
      cudaHostRegister_c4_3_c_int,&
      cudaHostRegister_c4_3_c_size_t,&
      cudaHostRegister_c4_4_nosize,&
      cudaHostRegister_c4_4_c_int,&
      cudaHostRegister_c4_4_c_size_t,&
      cudaHostRegister_c4_5_nosize,&
      cudaHostRegister_c4_5_c_int,&
      cudaHostRegister_c4_5_c_size_t,&
      cudaHostRegister_c4_6_nosize,&
      cudaHostRegister_c4_6_c_int,&
      cudaHostRegister_c4_6_c_size_t,&
      cudaHostRegister_c4_7_nosize,&
      cudaHostRegister_c4_7_c_int,&
      cudaHostRegister_c4_7_c_size_t,&
      cudaHostRegister_c8_0_nosize,&
      cudaHostRegister_c8_1_nosize,&
      cudaHostRegister_c8_1_c_int,&
      cudaHostRegister_c8_1_c_size_t,&
      cudaHostRegister_c8_2_nosize,&
      cudaHostRegister_c8_2_c_int,&
      cudaHostRegister_c8_2_c_size_t,&
      cudaHostRegister_c8_3_nosize,&
      cudaHostRegister_c8_3_c_int,&
      cudaHostRegister_c8_3_c_size_t,&
      cudaHostRegister_c8_4_nosize,&
      cudaHostRegister_c8_4_c_int,&
      cudaHostRegister_c8_4_c_size_t,&
      cudaHostRegister_c8_5_nosize,&
      cudaHostRegister_c8_5_c_int,&
      cudaHostRegister_c8_5_c_size_t,&
      cudaHostRegister_c8_6_nosize,&
      cudaHostRegister_c8_6_c_int,&
      cudaHostRegister_c8_6_c_size_t,&
      cudaHostRegister_c8_7_nosize,&
      cudaHostRegister_c8_7_c_int,&
      cudaHostRegister_c8_7_c_size_t 
  end interface
  !> 
  !>    @brief Un-register host pointer
  !>  
  !>    @param[in] hostPtr Host pointer previously registered with #hipHostRegister
  !>    @return Error code
  !>  
  !>    @see hipHostRegister
  !>  
  interface cudaHostUnregister
    function cudaHostUnregister_(hostPtr) bind(c, &
&name="cudaHostUnregister")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaHostUnregister_
      type(c_ptr),value :: hostPtr
    end function

    module procedure cudaHostUnregister_l_0,&
      cudaHostUnregister_l_1,&
      cudaHostUnregister_l_2,&
      cudaHostUnregister_l_3,&
      cudaHostUnregister_l_4,&
      cudaHostUnregister_l_5,&
      cudaHostUnregister_l_6,&
      cudaHostUnregister_l_7,&
      cudaHostUnregister_i4_0,&
      cudaHostUnregister_i4_1,&
      cudaHostUnregister_i4_2,&
      cudaHostUnregister_i4_3,&
      cudaHostUnregister_i4_4,&
      cudaHostUnregister_i4_5,&
      cudaHostUnregister_i4_6,&
      cudaHostUnregister_i4_7,&
      cudaHostUnregister_i8_0,&
      cudaHostUnregister_i8_1,&
      cudaHostUnregister_i8_2,&
      cudaHostUnregister_i8_3,&
      cudaHostUnregister_i8_4,&
      cudaHostUnregister_i8_5,&
      cudaHostUnregister_i8_6,&
      cudaHostUnregister_i8_7,&
      cudaHostUnregister_r4_0,&
      cudaHostUnregister_r4_1,&
      cudaHostUnregister_r4_2,&
      cudaHostUnregister_r4_3,&
      cudaHostUnregister_r4_4,&
      cudaHostUnregister_r4_5,&
      cudaHostUnregister_r4_6,&
      cudaHostUnregister_r4_7,&
      cudaHostUnregister_r8_0,&
      cudaHostUnregister_r8_1,&
      cudaHostUnregister_r8_2,&
      cudaHostUnregister_r8_3,&
      cudaHostUnregister_r8_4,&
      cudaHostUnregister_r8_5,&
      cudaHostUnregister_r8_6,&
      cudaHostUnregister_r8_7,&
      cudaHostUnregister_c4_0,&
      cudaHostUnregister_c4_1,&
      cudaHostUnregister_c4_2,&
      cudaHostUnregister_c4_3,&
      cudaHostUnregister_c4_4,&
      cudaHostUnregister_c4_5,&
      cudaHostUnregister_c4_6,&
      cudaHostUnregister_c4_7,&
      cudaHostUnregister_c8_0,&
      cudaHostUnregister_c8_1,&
      cudaHostUnregister_c8_2,&
      cudaHostUnregister_c8_3,&
      cudaHostUnregister_c8_4,&
      cudaHostUnregister_c8_5,&
      cudaHostUnregister_c8_6,&
      cudaHostUnregister_c8_7 
  end interface
 
  !> 
  !>    @brief Get Device pointer from Host Pointer allocated through hipHostMalloc
  !>  
  !>    @param[out] dstPtr Device Pointer mapped to passed host pointer
  !>    @param[in]  hstPtr Host Pointer allocated through hipHostMalloc
  !>    @param[in]  flags Flags to be passed for extension
  !>  
  !>    @return #hipSuccess, #hipErrorInvalidValue, #hipErrorOutOfMemory
  !>  
  !>    @see hipSetDeviceFlags, hipHostMalloc
  !>  
  interface cudaHostGetDevicePointer
    function cudaHostGetDevicePointer_(devPtr,hstPtr,flags) bind(c, &
&name="cudaHostGetDevicePointer")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_
      type(c_ptr) :: devPtr
      type(c_ptr),value :: hstPtr
      integer(kind=4),value :: flags
    end function

    module procedure cudaHostGetDevicePointer_l_0,&
      cudaHostGetDevicePointer_l_1,&
      cudaHostGetDevicePointer_l_2,&
      cudaHostGetDevicePointer_l_3,&
      cudaHostGetDevicePointer_l_4,&
      cudaHostGetDevicePointer_l_5,&
      cudaHostGetDevicePointer_l_6,&
      cudaHostGetDevicePointer_l_7,&
      cudaHostGetDevicePointer_i4_0,&
      cudaHostGetDevicePointer_i4_1,&
      cudaHostGetDevicePointer_i4_2,&
      cudaHostGetDevicePointer_i4_3,&
      cudaHostGetDevicePointer_i4_4,&
      cudaHostGetDevicePointer_i4_5,&
      cudaHostGetDevicePointer_i4_6,&
      cudaHostGetDevicePointer_i4_7,&
      cudaHostGetDevicePointer_i8_0,&
      cudaHostGetDevicePointer_i8_1,&
      cudaHostGetDevicePointer_i8_2,&
      cudaHostGetDevicePointer_i8_3,&
      cudaHostGetDevicePointer_i8_4,&
      cudaHostGetDevicePointer_i8_5,&
      cudaHostGetDevicePointer_i8_6,&
      cudaHostGetDevicePointer_i8_7,&
      cudaHostGetDevicePointer_r4_0,&
      cudaHostGetDevicePointer_r4_1,&
      cudaHostGetDevicePointer_r4_2,&
      cudaHostGetDevicePointer_r4_3,&
      cudaHostGetDevicePointer_r4_4,&
      cudaHostGetDevicePointer_r4_5,&
      cudaHostGetDevicePointer_r4_6,&
      cudaHostGetDevicePointer_r4_7,&
      cudaHostGetDevicePointer_r8_0,&
      cudaHostGetDevicePointer_r8_1,&
      cudaHostGetDevicePointer_r8_2,&
      cudaHostGetDevicePointer_r8_3,&
      cudaHostGetDevicePointer_r8_4,&
      cudaHostGetDevicePointer_r8_5,&
      cudaHostGetDevicePointer_r8_6,&
      cudaHostGetDevicePointer_r8_7,&
      cudaHostGetDevicePointer_c4_0,&
      cudaHostGetDevicePointer_c4_1,&
      cudaHostGetDevicePointer_c4_2,&
      cudaHostGetDevicePointer_c4_3,&
      cudaHostGetDevicePointer_c4_4,&
      cudaHostGetDevicePointer_c4_5,&
      cudaHostGetDevicePointer_c4_6,&
      cudaHostGetDevicePointer_c4_7,&
      cudaHostGetDevicePointer_c8_0,&
      cudaHostGetDevicePointer_c8_1,&
      cudaHostGetDevicePointer_c8_2,&
      cudaHostGetDevicePointer_c8_3,&
      cudaHostGetDevicePointer_c8_4,&
      cudaHostGetDevicePointer_c8_5,&
      cudaHostGetDevicePointer_c8_6,&
      cudaHostGetDevicePointer_c8_7 
  end interface
  !> 
  !>    @brief Return flags associated with host pointer
  !>  
  !>    @param[out] flagsPtr Memory location to store flags
  !>    @param[in]  hostPtr Host Pointer allocated through hipHostMalloc
  !>    @return #hipSuccess, #hipErrorInvalidValue
  !>  
  !>    @see hipHostMalloc
  !>  
  interface cudaHostGetFlags
    function cudaHostGetFlags_(flagsPtr,hostPtr) bind(c, &
&name="cudaHostGetFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_
      integer(kind=4)   :: flagsPtr
      type(c_ptr),value :: hostPtr
    end function
    
    module procedure cudaHostGetFlags_l_0,&
      cudaHostGetFlags_l_1,&
      cudaHostGetFlags_l_2,&
      cudaHostGetFlags_l_3,&
      cudaHostGetFlags_l_4,&
      cudaHostGetFlags_l_5,&
      cudaHostGetFlags_l_6,&
      cudaHostGetFlags_l_7,&
      cudaHostGetFlags_i4_0,&
      cudaHostGetFlags_i4_1,&
      cudaHostGetFlags_i4_2,&
      cudaHostGetFlags_i4_3,&
      cudaHostGetFlags_i4_4,&
      cudaHostGetFlags_i4_5,&
      cudaHostGetFlags_i4_6,&
      cudaHostGetFlags_i4_7,&
      cudaHostGetFlags_i8_0,&
      cudaHostGetFlags_i8_1,&
      cudaHostGetFlags_i8_2,&
      cudaHostGetFlags_i8_3,&
      cudaHostGetFlags_i8_4,&
      cudaHostGetFlags_i8_5,&
      cudaHostGetFlags_i8_6,&
      cudaHostGetFlags_i8_7,&
      cudaHostGetFlags_r4_0,&
      cudaHostGetFlags_r4_1,&
      cudaHostGetFlags_r4_2,&
      cudaHostGetFlags_r4_3,&
      cudaHostGetFlags_r4_4,&
      cudaHostGetFlags_r4_5,&
      cudaHostGetFlags_r4_6,&
      cudaHostGetFlags_r4_7,&
      cudaHostGetFlags_r8_0,&
      cudaHostGetFlags_r8_1,&
      cudaHostGetFlags_r8_2,&
      cudaHostGetFlags_r8_3,&
      cudaHostGetFlags_r8_4,&
      cudaHostGetFlags_r8_5,&
      cudaHostGetFlags_r8_6,&
      cudaHostGetFlags_r8_7,&
      cudaHostGetFlags_c4_0,&
      cudaHostGetFlags_c4_1,&
      cudaHostGetFlags_c4_2,&
      cudaHostGetFlags_c4_3,&
      cudaHostGetFlags_c4_4,&
      cudaHostGetFlags_c4_5,&
      cudaHostGetFlags_c4_6,&
      cudaHostGetFlags_c4_7,&
      cudaHostGetFlags_c8_0,&
      cudaHostGetFlags_c8_1,&
      cudaHostGetFlags_c8_2,&
      cudaHostGetFlags_c8_3,&
      cudaHostGetFlags_c8_4,&
      cudaHostGetFlags_c8_5,&
      cudaHostGetFlags_c8_6,&
      cudaHostGetFlags_c8_7 
  end interface

contains 
     
                                        
    function cudaHostRegister_l_0_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_0_nosize
      !
      cudaHostRegister_l_0_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8,flags)
    end function

                                        
    function cudaHostRegister_l_1_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_1_nosize
      !
      cudaHostRegister_l_1_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_l_1_c_int(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_1_c_int
      !
      cudaHostRegister_l_1_c_int = cudaHostRegister_(cptr,length1*1_8,&
&flags)
    end function
                                                              
    function cudaHostRegister_l_1_c_size_t(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_1_c_size_t
      !
      cudaHostRegister_l_1_c_size_t = cudaHostRegister_(cptr,&
&length1*1_8,flags)
    end function
                                        
    function cudaHostRegister_l_2_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_2_nosize
      !
      cudaHostRegister_l_2_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_l_2_c_int(hostPtr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_2_c_int
      !
      cudaHostRegister_l_2_c_int = cudaHostRegister_(cptr,&
&length1*length2*1_8,flags)
    end function
                                                              
    function cudaHostRegister_l_2_c_size_t(hostPtr,length1,length2,&
&flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_2_c_size_t
      !
      cudaHostRegister_l_2_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*1_8,flags)
    end function
                                        
    function cudaHostRegister_l_3_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_3_nosize
      !
      cudaHostRegister_l_3_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_l_3_c_int(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_3_c_int
      !
      cudaHostRegister_l_3_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*1_8,flags)
    end function
                                                              
    function cudaHostRegister_l_3_c_size_t(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_3_c_size_t
      !
      cudaHostRegister_l_3_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*1_8,flags)
    end function
                                        
    function cudaHostRegister_l_4_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_4_nosize
      !
      cudaHostRegister_l_4_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_l_4_c_int(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_4_c_int
      !
      cudaHostRegister_l_4_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*1_8,flags)
    end function
                                                              
    function cudaHostRegister_l_4_c_size_t(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_4_c_size_t
      !
      cudaHostRegister_l_4_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*1_8,flags)
    end function
                                        
    function cudaHostRegister_l_5_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_5_nosize
      !
      cudaHostRegister_l_5_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_l_5_c_int(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_5_c_int
      !
      cudaHostRegister_l_5_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*1_8,flags)
    end function
                                                              
    function cudaHostRegister_l_5_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_5_c_size_t
      !
      cudaHostRegister_l_5_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*1_8,flags)
    end function
                                        
    function cudaHostRegister_l_6_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_6_nosize
      !
      cudaHostRegister_l_6_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_l_6_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_6_c_int
      !
      cudaHostRegister_l_6_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8,flags)
    end function
                                                              
    function cudaHostRegister_l_6_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_6_c_size_t
      !
      cudaHostRegister_l_6_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*1_8,flags)
    end function
                                        
    function cudaHostRegister_l_7_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_7_nosize
      !
      cudaHostRegister_l_7_nosize = cudaHostRegister_(c_loc(hostPtr),&
&1_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_l_7_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_7_c_int
      !
      cudaHostRegister_l_7_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8,flags)
    end function
                                                              
    function cudaHostRegister_l_7_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_l_7_c_size_t
      !
      cudaHostRegister_l_7_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*1_8,flags)
    end function
                                        
    function cudaHostRegister_i4_0_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_0_nosize
      !
      cudaHostRegister_i4_0_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8,flags)
    end function

                                        
    function cudaHostRegister_i4_1_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_1_nosize
      !
      cudaHostRegister_i4_1_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i4_1_c_int(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_1_c_int
      !
      cudaHostRegister_i4_1_c_int = cudaHostRegister_(cptr,length1*4_8,&
&flags)
    end function
                                                              
    function cudaHostRegister_i4_1_c_size_t(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_1_c_size_t
      !
      cudaHostRegister_i4_1_c_size_t = cudaHostRegister_(cptr,&
&length1*4_8,flags)
    end function
                                        
    function cudaHostRegister_i4_2_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_2_nosize
      !
      cudaHostRegister_i4_2_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i4_2_c_int(hostPtr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_2_c_int
      !
      cudaHostRegister_i4_2_c_int = cudaHostRegister_(cptr,&
&length1*length2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_i4_2_c_size_t(hostPtr,length1,length2,&
&flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_2_c_size_t
      !
      cudaHostRegister_i4_2_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*4_8,flags)
    end function
                                        
    function cudaHostRegister_i4_3_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_3_nosize
      !
      cudaHostRegister_i4_3_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i4_3_c_int(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_3_c_int
      !
      cudaHostRegister_i4_3_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*4_8,flags)
    end function
                                                              
    function cudaHostRegister_i4_3_c_size_t(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_3_c_size_t
      !
      cudaHostRegister_i4_3_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*4_8,flags)
    end function
                                        
    function cudaHostRegister_i4_4_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_4_nosize
      !
      cudaHostRegister_i4_4_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i4_4_c_int(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_4_c_int
      !
      cudaHostRegister_i4_4_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*4_8,flags)
    end function
                                                              
    function cudaHostRegister_i4_4_c_size_t(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_4_c_size_t
      !
      cudaHostRegister_i4_4_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*4_8,flags)
    end function
                                        
    function cudaHostRegister_i4_5_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_5_nosize
      !
      cudaHostRegister_i4_5_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i4_5_c_int(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_5_c_int
      !
      cudaHostRegister_i4_5_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
    end function
                                                              
    function cudaHostRegister_i4_5_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_5_c_size_t
      !
      cudaHostRegister_i4_5_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
    end function
                                        
    function cudaHostRegister_i4_6_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_6_nosize
      !
      cudaHostRegister_i4_6_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i4_6_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_6_c_int
      !
      cudaHostRegister_i4_6_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
    end function
                                                              
    function cudaHostRegister_i4_6_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_6_c_size_t
      !
      cudaHostRegister_i4_6_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
    end function
                                        
    function cudaHostRegister_i4_7_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_7_nosize
      !
      cudaHostRegister_i4_7_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i4_7_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_7_c_int
      !
      cudaHostRegister_i4_7_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
    end function
                                                              
    function cudaHostRegister_i4_7_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i4_7_c_size_t
      !
      cudaHostRegister_i4_7_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
    end function
                                        
    function cudaHostRegister_i8_0_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_0_nosize
      !
      cudaHostRegister_i8_0_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8,flags)
    end function

                                        
    function cudaHostRegister_i8_1_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_1_nosize
      !
      cudaHostRegister_i8_1_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i8_1_c_int(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_1_c_int
      !
      cudaHostRegister_i8_1_c_int = cudaHostRegister_(cptr,length1*8_8,&
&flags)
    end function
                                                              
    function cudaHostRegister_i8_1_c_size_t(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_1_c_size_t
      !
      cudaHostRegister_i8_1_c_size_t = cudaHostRegister_(cptr,&
&length1*8_8,flags)
    end function
                                        
    function cudaHostRegister_i8_2_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_2_nosize
      !
      cudaHostRegister_i8_2_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i8_2_c_int(hostPtr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_2_c_int
      !
      cudaHostRegister_i8_2_c_int = cudaHostRegister_(cptr,&
&length1*length2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_i8_2_c_size_t(hostPtr,length1,length2,&
&flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_2_c_size_t
      !
      cudaHostRegister_i8_2_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*8_8,flags)
    end function
                                        
    function cudaHostRegister_i8_3_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_3_nosize
      !
      cudaHostRegister_i8_3_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i8_3_c_int(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_3_c_int
      !
      cudaHostRegister_i8_3_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*8_8,flags)
    end function
                                                              
    function cudaHostRegister_i8_3_c_size_t(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_3_c_size_t
      !
      cudaHostRegister_i8_3_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*8_8,flags)
    end function
                                        
    function cudaHostRegister_i8_4_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_4_nosize
      !
      cudaHostRegister_i8_4_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i8_4_c_int(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_4_c_int
      !
      cudaHostRegister_i8_4_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*8_8,flags)
    end function
                                                              
    function cudaHostRegister_i8_4_c_size_t(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_4_c_size_t
      !
      cudaHostRegister_i8_4_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*8_8,flags)
    end function
                                        
    function cudaHostRegister_i8_5_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_5_nosize
      !
      cudaHostRegister_i8_5_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i8_5_c_int(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_5_c_int
      !
      cudaHostRegister_i8_5_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
    end function
                                                              
    function cudaHostRegister_i8_5_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_5_c_size_t
      !
      cudaHostRegister_i8_5_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
    end function
                                        
    function cudaHostRegister_i8_6_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_6_nosize
      !
      cudaHostRegister_i8_6_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i8_6_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_6_c_int
      !
      cudaHostRegister_i8_6_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
    end function
                                                              
    function cudaHostRegister_i8_6_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_6_c_size_t
      !
      cudaHostRegister_i8_6_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
    end function
                                        
    function cudaHostRegister_i8_7_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_7_nosize
      !
      cudaHostRegister_i8_7_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_i8_7_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_7_c_int
      !
      cudaHostRegister_i8_7_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
    end function
                                                              
    function cudaHostRegister_i8_7_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_i8_7_c_size_t
      !
      cudaHostRegister_i8_7_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
    end function
                                        
    function cudaHostRegister_r4_0_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_0_nosize
      !
      cudaHostRegister_r4_0_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8,flags)
    end function

                                        
    function cudaHostRegister_r4_1_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_1_nosize
      !
      cudaHostRegister_r4_1_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r4_1_c_int(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_1_c_int
      !
      cudaHostRegister_r4_1_c_int = cudaHostRegister_(cptr,length1*4_8,&
&flags)
    end function
                                                              
    function cudaHostRegister_r4_1_c_size_t(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_1_c_size_t
      !
      cudaHostRegister_r4_1_c_size_t = cudaHostRegister_(cptr,&
&length1*4_8,flags)
    end function
                                        
    function cudaHostRegister_r4_2_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_2_nosize
      !
      cudaHostRegister_r4_2_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r4_2_c_int(hostPtr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_2_c_int
      !
      cudaHostRegister_r4_2_c_int = cudaHostRegister_(cptr,&
&length1*length2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_r4_2_c_size_t(hostPtr,length1,length2,&
&flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_2_c_size_t
      !
      cudaHostRegister_r4_2_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*4_8,flags)
    end function
                                        
    function cudaHostRegister_r4_3_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_3_nosize
      !
      cudaHostRegister_r4_3_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r4_3_c_int(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_3_c_int
      !
      cudaHostRegister_r4_3_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*4_8,flags)
    end function
                                                              
    function cudaHostRegister_r4_3_c_size_t(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_3_c_size_t
      !
      cudaHostRegister_r4_3_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*4_8,flags)
    end function
                                        
    function cudaHostRegister_r4_4_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_4_nosize
      !
      cudaHostRegister_r4_4_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r4_4_c_int(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_4_c_int
      !
      cudaHostRegister_r4_4_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*4_8,flags)
    end function
                                                              
    function cudaHostRegister_r4_4_c_size_t(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_4_c_size_t
      !
      cudaHostRegister_r4_4_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*4_8,flags)
    end function
                                        
    function cudaHostRegister_r4_5_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_5_nosize
      !
      cudaHostRegister_r4_5_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r4_5_c_int(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_5_c_int
      !
      cudaHostRegister_r4_5_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
    end function
                                                              
    function cudaHostRegister_r4_5_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_5_c_size_t
      !
      cudaHostRegister_r4_5_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*4_8,flags)
    end function
                                        
    function cudaHostRegister_r4_6_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_6_nosize
      !
      cudaHostRegister_r4_6_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r4_6_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_6_c_int
      !
      cudaHostRegister_r4_6_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
    end function
                                                              
    function cudaHostRegister_r4_6_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_6_c_size_t
      !
      cudaHostRegister_r4_6_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*4_8,flags)
    end function
                                        
    function cudaHostRegister_r4_7_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_7_nosize
      !
      cudaHostRegister_r4_7_nosize = cudaHostRegister_(c_loc(hostPtr),&
&4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r4_7_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_7_c_int
      !
      cudaHostRegister_r4_7_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
    end function
                                                              
    function cudaHostRegister_r4_7_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r4_7_c_size_t
      !
      cudaHostRegister_r4_7_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*4_8,flags)
    end function
                                        
    function cudaHostRegister_r8_0_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_0_nosize
      !
      cudaHostRegister_r8_0_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8,flags)
    end function

                                        
    function cudaHostRegister_r8_1_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_1_nosize
      !
      cudaHostRegister_r8_1_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r8_1_c_int(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_1_c_int
      !
      cudaHostRegister_r8_1_c_int = cudaHostRegister_(cptr,length1*8_8,&
&flags)
    end function
                                                              
    function cudaHostRegister_r8_1_c_size_t(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_1_c_size_t
      !
      cudaHostRegister_r8_1_c_size_t = cudaHostRegister_(cptr,&
&length1*8_8,flags)
    end function
                                        
    function cudaHostRegister_r8_2_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_2_nosize
      !
      cudaHostRegister_r8_2_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r8_2_c_int(hostPtr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_2_c_int
      !
      cudaHostRegister_r8_2_c_int = cudaHostRegister_(cptr,&
&length1*length2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_r8_2_c_size_t(hostPtr,length1,length2,&
&flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_2_c_size_t
      !
      cudaHostRegister_r8_2_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*8_8,flags)
    end function
                                        
    function cudaHostRegister_r8_3_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_3_nosize
      !
      cudaHostRegister_r8_3_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r8_3_c_int(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_3_c_int
      !
      cudaHostRegister_r8_3_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*8_8,flags)
    end function
                                                              
    function cudaHostRegister_r8_3_c_size_t(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_3_c_size_t
      !
      cudaHostRegister_r8_3_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*8_8,flags)
    end function
                                        
    function cudaHostRegister_r8_4_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_4_nosize
      !
      cudaHostRegister_r8_4_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r8_4_c_int(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_4_c_int
      !
      cudaHostRegister_r8_4_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*8_8,flags)
    end function
                                                              
    function cudaHostRegister_r8_4_c_size_t(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_4_c_size_t
      !
      cudaHostRegister_r8_4_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*8_8,flags)
    end function
                                        
    function cudaHostRegister_r8_5_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_5_nosize
      !
      cudaHostRegister_r8_5_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r8_5_c_int(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:), intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_5_c_int
      !
      cudaHostRegister_r8_5_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
    end function
                                                              
    function cudaHostRegister_r8_5_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:), intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_5_c_size_t
      !
      cudaHostRegister_r8_5_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*8_8,flags)
    end function
                                        
    function cudaHostRegister_r8_6_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_6_nosize
      !
      cudaHostRegister_r8_6_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r8_6_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_6_c_int
      !
      cudaHostRegister_r8_6_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
    end function
                                                              
    function cudaHostRegister_r8_6_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_6_c_size_t
      !
      cudaHostRegister_r8_6_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*8_8,flags)
    end function
                                        
    function cudaHostRegister_r8_7_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_7_nosize
      !
      cudaHostRegister_r8_7_nosize = cudaHostRegister_(c_loc(hostPtr),&
&8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_r8_7_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_7_c_int
      !
      cudaHostRegister_r8_7_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
    end function
                                                              
    function cudaHostRegister_r8_7_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_r8_7_c_size_t
      !
      cudaHostRegister_r8_7_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*8_8,flags)
    end function
                                        
    function cudaHostRegister_c4_0_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_0_nosize
      !
      cudaHostRegister_c4_0_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8,flags)
    end function

                                        
    function cudaHostRegister_c4_1_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:),intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_1_nosize
      !
      cudaHostRegister_c4_1_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c4_1_c_int(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_1_c_int
      !
      cudaHostRegister_c4_1_c_int = cudaHostRegister_(cptr,&
&length1*2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_c4_1_c_size_t(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_1_c_size_t
      !
      cudaHostRegister_c4_1_c_size_t = cudaHostRegister_(cptr,&
&length1*2*4_8,flags)
    end function
                                        
    function cudaHostRegister_c4_2_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_2_nosize
      !
      cudaHostRegister_c4_2_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c4_2_c_int(hostPtr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_2_c_int
      !
      cudaHostRegister_c4_2_c_int = cudaHostRegister_(cptr,&
&length1*length2*2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_c4_2_c_size_t(hostPtr,length1,length2,&
&flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_2_c_size_t
      !
      cudaHostRegister_c4_2_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*2*4_8,flags)
    end function
                                        
    function cudaHostRegister_c4_3_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_3_nosize
      !
      cudaHostRegister_c4_3_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c4_3_c_int(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_3_c_int
      !
      cudaHostRegister_c4_3_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_c4_3_c_size_t(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_3_c_size_t
      !
      cudaHostRegister_c4_3_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*2*4_8,flags)
    end function
                                        
    function cudaHostRegister_c4_4_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_4_nosize
      !
      cudaHostRegister_c4_4_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c4_4_c_int(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:), intent(in) &
&:: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_4_c_int
      !
      cudaHostRegister_c4_4_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_c4_4_c_size_t(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:), intent(in) &
&:: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_4_c_size_t
      !
      cudaHostRegister_c4_4_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*2*4_8,flags)
    end function
                                        
    function cudaHostRegister_c4_5_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_5_nosize
      !
      cudaHostRegister_c4_5_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c4_5_c_int(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_5_c_int
      !
      cudaHostRegister_c4_5_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_c4_5_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_5_c_size_t
      !
      cudaHostRegister_c4_5_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*2*4_8,flags)
    end function
                                        
    function cudaHostRegister_c4_6_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_6_nosize
      !
      cudaHostRegister_c4_6_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c4_6_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_6_c_int
      !
      cudaHostRegister_c4_6_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_c4_6_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_6_c_size_t
      !
      cudaHostRegister_c4_6_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*2*4_8,flags)
    end function
                                        
    function cudaHostRegister_c4_7_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_7_nosize
      !
      cudaHostRegister_c4_7_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*4_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c4_7_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_7_c_int
      !
      cudaHostRegister_c4_7_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8,flags)
    end function
                                                              
    function cudaHostRegister_c4_7_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c4_7_c_size_t
      !
      cudaHostRegister_c4_7_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*4_8,flags)
    end function
                                        
    function cudaHostRegister_c8_0_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_0_nosize
      !
      cudaHostRegister_c8_0_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8,flags)
    end function

                                        
    function cudaHostRegister_c8_1_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_1_nosize
      !
      cudaHostRegister_c8_1_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c8_1_c_int(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_1_c_int
      !
      cudaHostRegister_c8_1_c_int = cudaHostRegister_(cptr,&
&length1*2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_c8_1_c_size_t(hostPtr,length1,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_1_c_size_t
      !
      cudaHostRegister_c8_1_c_size_t = cudaHostRegister_(cptr,&
&length1*2*8_8,flags)
    end function
                                        
    function cudaHostRegister_c8_2_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_2_nosize
      !
      cudaHostRegister_c8_2_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c8_2_c_int(hostPtr,length1,length2,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:), intent(in) :: &
&hostPtr
      integer(c_int),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_2_c_int
      !
      cudaHostRegister_c8_2_c_int = cudaHostRegister_(cptr,&
&length1*length2*2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_c8_2_c_size_t(hostPtr,length1,length2,&
&flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:), intent(in) :: &
&hostPtr
      integer(c_size_t),intent(in) :: length1,length2 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_2_c_size_t
      !
      cudaHostRegister_c8_2_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*2*8_8,flags)
    end function
                                        
    function cudaHostRegister_c8_3_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:),intent(in) :: &
&hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_3_nosize
      !
      cudaHostRegister_c8_3_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c8_3_c_int(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:), intent(in) &
&:: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_3_c_int
      !
      cudaHostRegister_c8_3_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_c8_3_c_size_t(hostPtr,length1,length2,&
&length3,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:), intent(in) &
&:: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_3_c_size_t
      !
      cudaHostRegister_c8_3_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*2*8_8,flags)
    end function
                                        
    function cudaHostRegister_c8_4_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_4_nosize
      !
      cudaHostRegister_c8_4_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c8_4_c_int(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:), intent(in) &
&:: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_4_c_int
      !
      cudaHostRegister_c8_4_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_c8_4_c_size_t(hostPtr,length1,length2,&
&length3,length4,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:), intent(in) &
&:: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_4_c_size_t
      !
      cudaHostRegister_c8_4_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*2*8_8,flags)
    end function
                                        
    function cudaHostRegister_c8_5_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_5_nosize
      !
      cudaHostRegister_c8_5_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c8_5_c_int(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_5_c_int
      !
      cudaHostRegister_c8_5_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_c8_5_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_5_c_size_t
      !
      cudaHostRegister_c8_5_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*2*8_8,flags)
    end function
                                        
    function cudaHostRegister_c8_6_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_6_nosize
      !
      cudaHostRegister_c8_6_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c8_6_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_6_c_int
      !
      cudaHostRegister_c8_6_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_c8_6_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_6_c_size_t
      !
      cudaHostRegister_c8_6_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*2*8_8,flags)
    end function
                                        
    function cudaHostRegister_c8_7_nosize(hostPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind=4),intent(in) :: flags
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_7_nosize
      !
      cudaHostRegister_c8_7_nosize = cudaHostRegister_(c_loc(hostPtr),&
&2*8_8*size(hostPtr),flags)
    end function

                                                              
    function cudaHostRegister_c8_7_c_int(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_int),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_7_c_int
      !
      cudaHostRegister_c8_7_c_int = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8,flags)
    end function
                                                              
    function cudaHostRegister_c8_7_c_size_t(hostPtr,length1,length2,&
&length3,length4,length5,length6,length7,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:), &
&intent(in) :: hostPtr
      integer(c_size_t),intent(in) :: length1,length2,length3,length4,&
&length5,length6,length7 
      integer(kind=4),intent(in) :: flags 
      !
      type(c_ptr) :: cptr
      integer(kind(cudaSuccess)) :: cudaHostRegister_c8_7_c_size_t
      !
      cudaHostRegister_c8_7_c_size_t = cudaHostRegister_(cptr,&
&length1*length2*length3*length4*length5*length6*length7*2*8_8,flags)
    end function
  
                                        
    function cudaHostGetDevicePointer_l_0(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,intent(inout) :: devPtr
      logical(c_bool),target,intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_0
      !
      cudaHostGetDevicePointer_l_0 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_l_1(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:),intent(inout) :: devPtr
      logical(c_bool),target,dimension(:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_1
      !
      cudaHostGetDevicePointer_l_1 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_l_2(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:),intent(inout) :: devPtr
      logical(c_bool),target,dimension(:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_2
      !
      cudaHostGetDevicePointer_l_2 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_l_3(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:),intent(inout) :: devPtr
      logical(c_bool),target,dimension(:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_3
      !
      cudaHostGetDevicePointer_l_3 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_l_4(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:),intent(inout) :: devPtr
      logical(c_bool),target,dimension(:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_4
      !
      cudaHostGetDevicePointer_l_4 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_l_5(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:),intent(inout) :: &
&devPtr
      logical(c_bool),target,dimension(:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_5
      !
      cudaHostGetDevicePointer_l_5 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_l_6(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&devPtr
      logical(c_bool),target,dimension(:,:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_6
      !
      cudaHostGetDevicePointer_l_6 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_l_7(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: devPtr
      logical(c_bool),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_l_7
      !
      cudaHostGetDevicePointer_l_7 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_0(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,intent(inout) :: devPtr
      integer(c_int),target,intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_0
      !
      cudaHostGetDevicePointer_i4_0 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_1(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:),intent(inout) :: devPtr
      integer(c_int),target,dimension(:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_1
      !
      cudaHostGetDevicePointer_i4_1 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_2(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:),intent(inout) :: devPtr
      integer(c_int),target,dimension(:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_2
      !
      cudaHostGetDevicePointer_i4_2 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_3(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:),intent(inout) :: devPtr
      integer(c_int),target,dimension(:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_3
      !
      cudaHostGetDevicePointer_i4_3 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_4(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:),intent(inout) :: devPtr
      integer(c_int),target,dimension(:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_4
      !
      cudaHostGetDevicePointer_i4_4 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_5(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:),intent(inout) :: &
&devPtr
      integer(c_int),target,dimension(:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_5
      !
      cudaHostGetDevicePointer_i4_5 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_6(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&devPtr
      integer(c_int),target,dimension(:,:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_6
      !
      cudaHostGetDevicePointer_i4_6 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i4_7(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&devPtr
      integer(c_int),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i4_7
      !
      cudaHostGetDevicePointer_i4_7 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_0(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,intent(inout) :: devPtr
      integer(c_long),target,intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_0
      !
      cudaHostGetDevicePointer_i8_0 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_1(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:),intent(inout) :: devPtr
      integer(c_long),target,dimension(:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_1
      !
      cudaHostGetDevicePointer_i8_1 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_2(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:),intent(inout) :: devPtr
      integer(c_long),target,dimension(:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_2
      !
      cudaHostGetDevicePointer_i8_2 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_3(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:),intent(inout) :: devPtr
      integer(c_long),target,dimension(:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_3
      !
      cudaHostGetDevicePointer_i8_3 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_4(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:),intent(inout) :: devPtr
      integer(c_long),target,dimension(:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_4
      !
      cudaHostGetDevicePointer_i8_4 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_5(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:),intent(inout) :: &
&devPtr
      integer(c_long),target,dimension(:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_5
      !
      cudaHostGetDevicePointer_i8_5 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_6(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&devPtr
      integer(c_long),target,dimension(:,:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_6
      !
      cudaHostGetDevicePointer_i8_6 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_i8_7(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),pointer,dimension(:,:,:,:,:,:,:),intent(inout) &
&:: devPtr
      integer(c_long),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_i8_7
      !
      cudaHostGetDevicePointer_i8_7 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_0(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,intent(inout) :: devPtr
      real(c_float),target,intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_0
      !
      cudaHostGetDevicePointer_r4_0 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_1(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:),intent(inout) :: devPtr
      real(c_float),target,dimension(:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_1
      !
      cudaHostGetDevicePointer_r4_1 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_2(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:),intent(inout) :: devPtr
      real(c_float),target,dimension(:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_2
      !
      cudaHostGetDevicePointer_r4_2 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_3(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:),intent(inout) :: devPtr
      real(c_float),target,dimension(:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_3
      !
      cudaHostGetDevicePointer_r4_3 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_4(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:),intent(inout) :: devPtr
      real(c_float),target,dimension(:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_4
      !
      cudaHostGetDevicePointer_r4_4 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_5(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:),intent(inout) :: devPtr
      real(c_float),target,dimension(:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_5
      !
      cudaHostGetDevicePointer_r4_5 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_6(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&devPtr
      real(c_float),target,dimension(:,:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_6
      !
      cudaHostGetDevicePointer_r4_6 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r4_7(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&devPtr
      real(c_float),target,dimension(:,:,:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r4_7
      !
      cudaHostGetDevicePointer_r4_7 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_0(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,intent(inout) :: devPtr
      real(c_double),target,intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_0
      !
      cudaHostGetDevicePointer_r8_0 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_1(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:),intent(inout) :: devPtr
      real(c_double),target,dimension(:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_1
      !
      cudaHostGetDevicePointer_r8_1 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_2(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:),intent(inout) :: devPtr
      real(c_double),target,dimension(:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_2
      !
      cudaHostGetDevicePointer_r8_2 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_3(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:),intent(inout) :: devPtr
      real(c_double),target,dimension(:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_3
      !
      cudaHostGetDevicePointer_r8_3 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_4(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:),intent(inout) :: devPtr
      real(c_double),target,dimension(:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_4
      !
      cudaHostGetDevicePointer_r8_4 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_5(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:),intent(inout) :: &
&devPtr
      real(c_double),target,dimension(:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_5
      !
      cudaHostGetDevicePointer_r8_5 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_6(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:),intent(inout) :: &
&devPtr
      real(c_double),target,dimension(:,:,:,:,:,:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_6
      !
      cudaHostGetDevicePointer_r8_6 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_r8_7(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),pointer,dimension(:,:,:,:,:,:,:),intent(inout) :: &
&devPtr
      real(c_double),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_r8_7
      !
      cudaHostGetDevicePointer_r8_7 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_0(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,intent(inout) :: devPtr
      complex(c_float_complex),target,intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_0
      !
      cudaHostGetDevicePointer_c4_0 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_1(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:),intent(inout) :: &
&devPtr
      complex(c_float_complex),target,dimension(:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_1
      !
      cudaHostGetDevicePointer_c4_1 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_2(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:),intent(inout) :: &
&devPtr
      complex(c_float_complex),target,dimension(:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_2
      !
      cudaHostGetDevicePointer_c4_2 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_3(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:),intent(inout) &
&:: devPtr
      complex(c_float_complex),target,dimension(:,:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_3
      !
      cudaHostGetDevicePointer_c4_3 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_4(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_float_complex),target,dimension(:,:,:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_4
      !
      cudaHostGetDevicePointer_c4_4 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_5(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_float_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_5
      !
      cudaHostGetDevicePointer_c4_5 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_6(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_float_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_6
      !
      cudaHostGetDevicePointer_c4_6 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c4_7(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_float_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c4_7
      !
      cudaHostGetDevicePointer_c4_7 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_0(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,intent(inout) :: devPtr
      complex(c_double_complex),target,intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_0
      !
      cudaHostGetDevicePointer_c8_0 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_1(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:),intent(inout) :: &
&devPtr
      complex(c_double_complex),target,dimension(:),intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_1
      !
      cudaHostGetDevicePointer_c8_1 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_2(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:),intent(inout) &
&:: devPtr
      complex(c_double_complex),target,dimension(:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_2
      !
      cudaHostGetDevicePointer_c8_2 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_3(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:),intent(inout) &
&:: devPtr
      complex(c_double_complex),target,dimension(:,:,:),intent(in) :: &
&hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_3
      !
      cudaHostGetDevicePointer_c8_3 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_4(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_double_complex),target,dimension(:,:,:,:),intent(in) &
&:: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_4
      !
      cudaHostGetDevicePointer_c8_4 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_5(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_double_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_5
      !
      cudaHostGetDevicePointer_c8_5 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_6(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_double_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_6
      !
      cudaHostGetDevicePointer_c8_6 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function
                                        
    function cudaHostGetDevicePointer_c8_7(devPtr,hstPtr,flags)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),pointer,dimension(:,:,:,:,:,:,:),&
&intent(inout) :: devPtr
      complex(c_double_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hstPtr
      integer(kind=4),intent(in) :: flags 
      integer(kind(cudaSuccess)) :: cudaHostGetDevicePointer_c8_7
      !
      cudaHostGetDevicePointer_c8_7 = &
&cudaHostGetDevicePointer_(c_loc(devPtr),c_loc(hstPtr),flags)
    end function

                                        
    function cudaHostGetFlags_l_0(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_0
      !
      cudaHostGetFlags_l_0 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_l_1(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_1
      !
      cudaHostGetFlags_l_1 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_l_2(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_2
      !
      cudaHostGetFlags_l_2 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_l_3(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_3
      !
      cudaHostGetFlags_l_3 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_l_4(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_4
      !
      cudaHostGetFlags_l_4 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_l_5(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_5
      !
      cudaHostGetFlags_l_5 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_l_6(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,dimension(:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_6
      !
      cudaHostGetFlags_l_6 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_l_7(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      logical(c_bool),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_l_7
      !
      cudaHostGetFlags_l_7 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_0(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_0
      !
      cudaHostGetFlags_i4_0 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_1(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_1
      !
      cudaHostGetFlags_i4_1 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_2(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_2
      !
      cudaHostGetFlags_i4_2 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_3(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_3
      !
      cudaHostGetFlags_i4_3 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_4(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_4
      !
      cudaHostGetFlags_i4_4 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_5(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_5
      !
      cudaHostGetFlags_i4_5 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_6(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_6
      !
      cudaHostGetFlags_i4_6 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i4_7(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_int),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i4_7
      !
      cudaHostGetFlags_i4_7 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_0(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_0
      !
      cudaHostGetFlags_i8_0 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_1(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_1
      !
      cudaHostGetFlags_i8_1 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_2(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_2
      !
      cudaHostGetFlags_i8_2 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_3(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_3
      !
      cudaHostGetFlags_i8_3 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_4(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_4
      !
      cudaHostGetFlags_i8_4 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_5(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_5
      !
      cudaHostGetFlags_i8_5 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_6(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,dimension(:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_6
      !
      cudaHostGetFlags_i8_6 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_i8_7(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      integer(c_long),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_i8_7
      !
      cudaHostGetFlags_i8_7 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_0(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_0
      !
      cudaHostGetFlags_r4_0 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_1(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_1
      !
      cudaHostGetFlags_r4_1 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_2(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_2
      !
      cudaHostGetFlags_r4_2 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_3(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_3
      !
      cudaHostGetFlags_r4_3 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_4(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_4
      !
      cudaHostGetFlags_r4_4 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_5(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_5
      !
      cudaHostGetFlags_r4_5 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_6(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_6
      !
      cudaHostGetFlags_r4_6 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r4_7(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_float),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r4_7
      !
      cudaHostGetFlags_r4_7 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_0(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_0
      !
      cudaHostGetFlags_r8_0 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_1(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_1
      !
      cudaHostGetFlags_r8_1 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_2(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_2
      !
      cudaHostGetFlags_r8_2 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_3(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_3
      !
      cudaHostGetFlags_r8_3 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_4(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_4
      !
      cudaHostGetFlags_r8_4 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_5(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_5
      !
      cudaHostGetFlags_r8_5 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_6(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_6
      !
      cudaHostGetFlags_r8_6 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_r8_7(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      real(c_double),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_r8_7
      !
      cudaHostGetFlags_r8_7 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_0(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_0
      !
      cudaHostGetFlags_c4_0 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_1(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_1
      !
      cudaHostGetFlags_c4_1 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_2(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,dimension(:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_2
      !
      cudaHostGetFlags_c4_2 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_3(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,dimension(:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_3
      !
      cudaHostGetFlags_c4_3 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_4(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,dimension(:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_4
      !
      cudaHostGetFlags_c4_4 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_5(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_5
      !
      cudaHostGetFlags_c4_5 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_6(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_6
      !
      cudaHostGetFlags_c4_6 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c4_7(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_float_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c4_7
      !
      cudaHostGetFlags_c4_7 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_0(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_0
      !
      cudaHostGetFlags_c8_0 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_1(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,dimension(:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_1
      !
      cudaHostGetFlags_c8_1 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_2(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,dimension(:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_2
      !
      cudaHostGetFlags_c8_2 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_3(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,dimension(:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_3
      !
      cudaHostGetFlags_c8_3 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_4(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,dimension(:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_4
      !
      cudaHostGetFlags_c8_4 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_5(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_5
      !
      cudaHostGetFlags_c8_5 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_6(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_6
      !
      cudaHostGetFlags_c8_6 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function
                                        
    function cudaHostGetFlags_c8_7(flagsPtr,hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(kind=4),intent(out) :: flagsPtr
      complex(c_double_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostGetFlags_c8_7
      !
      cudaHostGetFlags_c8_7 = cudaHostGetFlags_(flagsPtr,c_loc(hostPtr))
    end function

                                        
    function cudaHostUnregister_l_0(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_0
      !
      cudaHostUnregister_l_0 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_l_1(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_1
      !
      cudaHostUnregister_l_1 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_l_2(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_2
      !
      cudaHostUnregister_l_2 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_l_3(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_3
      !
      cudaHostUnregister_l_3 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_l_4(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_4
      !
      cudaHostUnregister_l_4 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_l_5(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_5
      !
      cudaHostUnregister_l_5 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_l_6(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_6
      !
      cudaHostUnregister_l_6 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_l_7(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      logical(c_bool),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_l_7
      !
      cudaHostUnregister_l_7 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_0(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_0
      !
      cudaHostUnregister_i4_0 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_1(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_1
      !
      cudaHostUnregister_i4_1 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_2(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_2
      !
      cudaHostUnregister_i4_2 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_3(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_3
      !
      cudaHostUnregister_i4_3 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_4(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_4
      !
      cudaHostUnregister_i4_4 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_5(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_5
      !
      cudaHostUnregister_i4_5 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_6(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_6
      !
      cudaHostUnregister_i4_6 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i4_7(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_int),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i4_7
      !
      cudaHostUnregister_i4_7 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_0(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_0
      !
      cudaHostUnregister_i8_0 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_1(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_1
      !
      cudaHostUnregister_i8_1 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_2(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_2
      !
      cudaHostUnregister_i8_2 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_3(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_3
      !
      cudaHostUnregister_i8_3 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_4(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_4
      !
      cudaHostUnregister_i8_4 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_5(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_5
      !
      cudaHostUnregister_i8_5 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_6(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_6
      !
      cudaHostUnregister_i8_6 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_i8_7(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      integer(c_long),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_i8_7
      !
      cudaHostUnregister_i8_7 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_0(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_0
      !
      cudaHostUnregister_r4_0 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_1(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_1
      !
      cudaHostUnregister_r4_1 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_2(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_2
      !
      cudaHostUnregister_r4_2 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_3(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_3
      !
      cudaHostUnregister_r4_3 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_4(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_4
      !
      cudaHostUnregister_r4_4 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_5(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_5
      !
      cudaHostUnregister_r4_5 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_6(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_6
      !
      cudaHostUnregister_r4_6 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r4_7(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_float),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r4_7
      !
      cudaHostUnregister_r4_7 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_0(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_0
      !
      cudaHostUnregister_r8_0 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_1(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_1
      !
      cudaHostUnregister_r8_1 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_2(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_2
      !
      cudaHostUnregister_r8_2 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_3(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_3
      !
      cudaHostUnregister_r8_3 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_4(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_4
      !
      cudaHostUnregister_r8_4 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_5(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_5
      !
      cudaHostUnregister_r8_5 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_6(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:,:,:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_6
      !
      cudaHostUnregister_r8_6 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_r8_7(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      real(c_double),target,dimension(:,:,:,:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_r8_7
      !
      cudaHostUnregister_r8_7 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_0(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_0
      !
      cudaHostUnregister_c4_0 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_1(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:),intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_1
      !
      cudaHostUnregister_c4_1 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_2(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_2
      !
      cudaHostUnregister_c4_2 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_3(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_3
      !
      cudaHostUnregister_c4_3 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_4(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_4
      !
      cudaHostUnregister_c4_4 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_5(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_5
      !
      cudaHostUnregister_c4_5 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_6(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_6
      !
      cudaHostUnregister_c4_6 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c4_7(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_float_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c4_7
      !
      cudaHostUnregister_c4_7 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_0(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_0
      !
      cudaHostUnregister_c8_0 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_1(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_1
      !
      cudaHostUnregister_c8_1 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_2(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_2
      !
      cudaHostUnregister_c8_2 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_3(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:),intent(in) :: &
&hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_3
      !
      cudaHostUnregister_c8_3 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_4(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_4
      !
      cudaHostUnregister_c8_4 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_5(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:,:),intent(in) &
&:: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_5
      !
      cudaHostUnregister_c8_5 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_6(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_6
      !
      cudaHostUnregister_c8_6 = cudaHostUnregister_(c_loc(hostPtr))
    end function
                                        
    function cudaHostUnregister_c8_7(hostPtr)
      use iso_c_binding
      use cudafor_enums
      implicit none
      complex(c_double_complex),target,dimension(:,:,:,:,:,:,:),&
&intent(in) :: hostPtr
      integer(kind(cudaSuccess)) :: cudaHostUnregister_c8_7
      !
      cudaHostUnregister_c8_7 = cudaHostUnregister_(c_loc(hostPtr))
    end function
end module