


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ==============================================================================
! hipfort: FORTRAN Interfaces for GPU kernels
! ==============================================================================
! Copyright (c) 2020-2022 Advanced Micro Devices, Inc. All rights reserved.
! [MITx11 License]
! 
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
! IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
! AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
! OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
! THE SOFTWARE.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
          
module cudafor_cufft
  use cudafor_cufft_enums
  use cudafor_cufft_params
  implicit none

 
  !>  @brief Create a new one-dimensional FFT plan.
  !> 
  !>   @details Allocate and initialize a new one-dimensional FFT plan.
  !> 
  !>   @param[out] plan Pointer to the FFT plan handle.
  !>   @param[in] nx FFT length.
  !>   @param[in] type FFT type.
  !>   @param[in] batch Number of batched transforms to compute.
  interface cufftPlan1d
    function cufftPlan1d_(plan,nx,myType,batch) bind(c, &
&name="cufftPlan1d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftPlan1d_
      type(c_ptr) :: plan
      integer(c_int),value :: nx
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
    end function

  end interface
  !>  @brief Create a new two-dimensional FFT plan.
  !> 
  !>   @details Allocate and initialize a new two-dimensional FFT plan.
  !>   Two-dimensional data should be stored in C ordering (row-major
  !>   format), so that indexes in y-direction (j index) vary the
  !>   fastest.
  !> 
  !>   @param[out] plan Pointer to the FFT plan handle.
  !>   @param[in] nx Number of elements in the x-direction (slow index).
  !>   @param[in] ny Number of elements in the y-direction (fast index).
  !>   @param[in] type FFT type.
  interface cufftPlan2d
    function cufftPlan2d_(plan,nx,ny,myType) bind(c, name="cufftPlan2d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftPlan2d_
      type(c_ptr) :: plan
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(kind(CUFFT_R2C)),value :: myType
    end function

  end interface
  !>  @brief Create a new three-dimensional FFT plan.
  !> 
  !>   @details Allocate and initialize a new three-dimensional FFT plan.
  !>   Three-dimensional data should be stored in C ordering (row-major
  !>   format), so that indexes in z-direction (k index) vary the
  !>   fastest.
  !> 
  !>   @param[out] plan Pointer to the FFT plan handle.
  !>   @param[in] nx Number of elements in the x-direction (slowest index).
  !>   @param[in] ny Number of elements in the y-direction.
  !>   @param[in] nz Number of elements in the z-direction (fastest index).
  !>   @param[in] type FFT type.
  interface cufftPlan3d
    function cufftPlan3d_(plan,nx,ny,nz,myType) bind(c, &
&name="cufftPlan3d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftPlan3d_
      type(c_ptr) :: plan
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(c_int),value :: nz
      integer(kind(CUFFT_R2C)),value :: myType
    end function

  end interface
  !>  @brief Create a new batched rank-dimensional FFT plan.
  !> 
  !>  @details Allocate and initialize a new batched rank-dimensional
  !>   FFT.  The batch parameter tells hipFFT how many transforms to
  !>   perform.  Used in complicated usage case like flexible input and
  !>   output layout.
  !> 
  !>   @param[out] plan Pointer to the FFT plan handle.
  !>   @param[in] rank Dimension of FFT transform (1, 2, or 3).
  !>   @param[in] n Number of elements in the x/y/z directions.
  !>   @param[in] inembed
  !>   @param[in] istride
  !>   @param[in] idist Distance between input batches.
  !>   @param[in] onembed
  !>   @param[in] ostride
  !>   @param[in] odist Distance between output batches.
  !>   @param[in] type FFT type.
  !>   @param[in] batch Number of batched transforms to perform.
  interface cufftPlanMany
    function cufftPlanMany_(plan,rank,n,inembed,istride,idist,onembed,&
&ostride,odist,myType,batch) bind(c, name="cufftPlanMany")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftPlanMany_
      type(c_ptr) :: plan
      integer(c_int),value :: rank
      type(c_ptr),value :: n
      type(c_ptr),value :: inembed
      integer(c_int),value :: istride
      integer(c_int),value :: idist
      type(c_ptr),value :: onembed
      integer(c_int),value :: ostride
      integer(c_int),value :: odist
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
    end function

    module procedure &
      cufftPlanMany_rank_0,&
      cufftPlanMany_rank_1
  end interface
  !>  @brief Allocate a new plan.
  interface cufftCreate
    function cufftCreate_(plan) bind(c, name="cufftCreate")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftCreate_
      type(c_ptr) :: plan
    end function

  end interface
  !>  @brief Initialize a new one-dimensional FFT plan.
  !> 
  !>   @details Assumes that the plan has been created already, and
  !>   modifies the plan associated with the plan handle.
  !> 
  !>   @param[in] plan Handle of the FFT plan.
  !>   @param[in] nx FFT length.
  !>   @param[in] type FFT type.
  !>   @param[in] batch Number of batched transforms to compute.
  interface cufftMakePlan1d
    function cufftMakePlan1d_(plan,nx,myType,batch,workSize) bind(c, &
&name="cufftMakePlan1d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan1d_
      type(c_ptr),value :: plan
      integer(c_int),value :: nx
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftMakePlan1d_rank_0,&
      cufftMakePlan1d_rank_1
  end interface
  !>  @brief Initialize a new two-dimensional FFT plan.
  !> 
  !>   @details Assumes that the plan has been created already, and
  !>   modifies the plan associated with the plan handle.
  !>   Two-dimensional data should be stored in C ordering (row-major
  !>   format), so that indexes in y-direction (j index) vary the
  !>   fastest.
  !> 
  !>   @param[in] plan Handle of the FFT plan.
  !>   @param[in] nx Number of elements in the x-direction (slow index).
  !>   @param[in] ny Number of elements in the y-direction (fast index).
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftMakePlan2d
    function cufftMakePlan2d_(plan,nx,ny,myType,workSize) bind(c, &
&name="cufftMakePlan2d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan2d_
      type(c_ptr),value :: plan
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(kind(CUFFT_R2C)),value :: myType
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftMakePlan2d_rank_0,&
      cufftMakePlan2d_rank_1
  end interface
  !>  @brief Initialize a new two-dimensional FFT plan.
  !> 
  !>   @details Assumes that the plan has been created already, and
  !>   modifies the plan associated with the plan handle.
  !>   Three-dimensional data should be stored in C ordering (row-major
  !>   format), so that indexes in z-direction (k index) vary the
  !>   fastest.
  !> 
  !>   @param[in] plan Handle of the FFT plan.
  !>   @param[in] nx Number of elements in the x-direction (slowest index).
  !>   @param[in] ny Number of elements in the y-direction.
  !>   @param[in] nz Number of elements in the z-direction (fastest index).
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftMakePlan3d
    function cufftMakePlan3d_(plan,nx,ny,nz,myType,workSize) bind(c, &
&name="cufftMakePlan3d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan3d_
      type(c_ptr),value :: plan
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(c_int),value :: nz
      integer(kind(CUFFT_R2C)),value :: myType
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftMakePlan3d_rank_0,&
      cufftMakePlan3d_rank_1
  end interface
  !>  @brief Initialize a new batched rank-dimensional FFT plan.
  !> 
  !>   @details Assumes that the plan has been created already, and
  !>   modifies the plan associated with the plan handle.  The
  !>   batch parameter tells hipFFT how many transforms to perform.  Used
  !>   in complicated usage case like flexible input and output layout.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  !>   @param[in] rank Dimension of FFT transform (1, 2, or 3).
  !>   @param[in] n Number of elements in the x/y/z directions.
  !>   @param[in] inembed
  !>   @param[in] istride
  !>   @param[in] idist Distance between input batches.
  !>   @param[in] onembed
  !>   @param[in] ostride
  !>   @param[in] odist Distance between output batches.
  !>   @param[in] type FFT type.
  !>   @param[in] batch Number of batched transforms to perform.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftMakePlanMany
    function cufftMakePlanMany_(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize) bind(c, &
&name="cufftMakePlanMany")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlanMany_
      type(c_ptr),value :: plan
      integer(c_int),value :: rank
      type(c_ptr),value :: n
      type(c_ptr),value :: inembed
      integer(c_int),value :: istride
      integer(c_int),value :: idist
      type(c_ptr),value :: onembed
      integer(c_int),value :: ostride
      integer(c_int),value :: odist
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftMakePlanMany_rank_0,&
      cufftMakePlanMany_rank_1
  end interface
  
  interface cufftMakePlanMany64
    function cufftMakePlanMany64_(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize) bind(c, &
&name="cufftMakePlanMany64")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlanMany64_
      type(c_ptr),value :: plan
      integer(c_int),value :: rank
      type(c_ptr),value :: n
      type(c_ptr),value :: inembed
      integer(c_long_long),value :: istride
      integer(c_long_long),value :: idist
      type(c_ptr),value :: onembed
      integer(c_long_long),value :: ostride
      integer(c_long_long),value :: odist
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_long_long),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftMakePlanMany64_rank_0,&
      cufftMakePlanMany64_rank_1
  end interface
  !>  @brief Return an estimate of the work area size required for a 1D plan.
  !> 
  !>   @param[in] nx Number of elements in the x-direction.
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftEstimate1d
    function cufftEstimate1d_(nx,myType,batch,workSize) bind(c, &
&name="cufftEstimate1d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate1d_
      integer(c_int),value :: nx
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftEstimate1d_rank_0,&
      cufftEstimate1d_rank_1
  end interface
  !>  @brief Return an estimate of the work area size required for a 2D plan.
  !> 
  !>   @param[in] nx Number of elements in the x-direction.
  !>   @param[in] ny Number of elements in the y-direction.
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftEstimate2d
    function cufftEstimate2d_(nx,ny,myType,workSize) bind(c, &
&name="cufftEstimate2d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate2d_
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(kind(CUFFT_R2C)),value :: myType
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftEstimate2d_rank_0,&
      cufftEstimate2d_rank_1
  end interface
  !>  @brief Return an estimate of the work area size required for a 3D plan.
  !> 
  !>   @param[in] nx Number of elements in the x-direction.
  !>   @param[in] ny Number of elements in the y-direction.
  !>   @param[in] nz Number of elements in the z-direction.
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftEstimate3d
    function cufftEstimate3d_(nx,ny,nz,myType,workSize) bind(c, &
&name="cufftEstimate3d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate3d_
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(c_int),value :: nz
      integer(kind(CUFFT_R2C)),value :: myType
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftEstimate3d_rank_0,&
      cufftEstimate3d_rank_1
  end interface
  !>  @brief Return an estimate of the work area size required for a rank-dimensional plan.
  !> 
  !>   @param[in] rank Dimension of FFT transform (1, 2, or 3).
  !>   @param[in] n Number of elements in the x/y/z directions.
  !>   @param[in] inembed
  !>   @param[in] istride
  !>   @param[in] idist Distance between input batches.
  !>   @param[in] onembed
  !>   @param[in] ostride
  !>   @param[in] odist Distance between output batches.
  !>   @param[in] type FFT type.
  !>   @param[in] batch Number of batched transforms to perform.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftEstimateMany
    function cufftEstimateMany_(rank,n,inembed,istride,idist,onembed,&
&ostride,odist,myType,batch,workSize) bind(c, &
&name="cufftEstimateMany")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimateMany_
      integer(c_int),value :: rank
      type(c_ptr),value :: n
      type(c_ptr),value :: inembed
      integer(c_int),value :: istride
      integer(c_int),value :: idist
      type(c_ptr),value :: onembed
      integer(c_int),value :: ostride
      integer(c_int),value :: odist
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftEstimateMany_rank_0,&
      cufftEstimateMany_rank_1
  end interface
  !>  @brief Return size of the work area size required for a 1D plan.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  !>   @param[in] nx Number of elements in the x-direction.
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftGetSize1d
    function cufftGetSize1d_(plan,nx,myType,batch,workSize) bind(c, &
&name="cufftGetSize1d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize1d_
      type(c_ptr),value :: plan
      integer(c_int),value :: nx
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftGetSize1d_rank_0,&
      cufftGetSize1d_rank_1
  end interface
  !>  @brief Return size of the work area size required for a 2D plan.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  !>   @param[in] nx Number of elements in the x-direction.
  !>   @param[in] ny Number of elements in the y-direction.
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftGetSize2d
    function cufftGetSize2d_(plan,nx,ny,myType,workSize) bind(c, &
&name="cufftGetSize2d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize2d_
      type(c_ptr),value :: plan
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(kind(CUFFT_R2C)),value :: myType
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftGetSize2d_rank_0,&
      cufftGetSize2d_rank_1
  end interface
  !>  @brief Return size of the work area size required for a 3D plan.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  !>   @param[in] nx Number of elements in the x-direction.
  !>   @param[in] ny Number of elements in the y-direction.
  !>   @param[in] nz Number of elements in the z-direction.
  !>   @param[in] type FFT type.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftGetSize3d
    function cufftGetSize3d_(plan,nx,ny,nz,myType,workSize) bind(c, &
&name="cufftGetSize3d")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize3d_
      type(c_ptr),value :: plan
      integer(c_int),value :: nx
      integer(c_int),value :: ny
      integer(c_int),value :: nz
      integer(kind(CUFFT_R2C)),value :: myType
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftGetSize3d_rank_0,&
      cufftGetSize3d_rank_1
  end interface
  !>  @brief Return size of the work area size required for a rank-dimensional plan.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  !>   @param[in] rank Dimension of FFT transform (1, 2, or 3).
  !>   @param[in] n Number of elements in the x/y/z directions.
  !>   @param[in] inembed
  !>   @param[in] istride
  !>   @param[in] idist Distance between input batches.
  !>   @param[in] onembed
  !>   @param[in] ostride
  !>   @param[in] odist Distance between output batches.
  !>   @param[in] type FFT type.
  !>   @param[in] batch Number of batched transforms to perform.
  !>   @param[out] workSize Pointer to work area size (returned value).
  interface cufftGetSizeMany
    function cufftGetSizeMany_(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize) bind(c, &
&name="cufftGetSizeMany")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSizeMany_
      type(c_ptr),value :: plan
      integer(c_int),value :: rank
      type(c_ptr),value :: n
      type(c_ptr),value :: inembed
      integer(c_int),value :: istride
      integer(c_int),value :: idist
      type(c_ptr),value :: onembed
      integer(c_int),value :: ostride
      integer(c_int),value :: odist
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_int),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftGetSizeMany_rank_0,&
      cufftGetSizeMany_rank_1
  end interface
  
  interface cufftGetSizeMany64
    function cufftGetSizeMany64_(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize) bind(c, &
&name="cufftGetSizeMany64")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSizeMany64_
      type(c_ptr),value :: plan
      integer(c_int),value :: rank
      type(c_ptr),value :: n
      type(c_ptr),value :: inembed
      integer(c_long_long),value :: istride
      integer(c_long_long),value :: idist
      type(c_ptr),value :: onembed
      integer(c_long_long),value :: ostride
      integer(c_long_long),value :: odist
      integer(kind(CUFFT_R2C)),value :: myType
      integer(c_long_long),value :: batch
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftGetSizeMany64_rank_0,&
      cufftGetSizeMany64_rank_1
  end interface
  !>  @brief Return size of the work area size required for a rank-dimensional plan.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  interface cufftGetSize
    function cufftGetSize_(plan,workSize) bind(c, name="cufftGetSize")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize_
      type(c_ptr),value :: plan
      type(c_ptr),value :: workSize
    end function

    module procedure &
      cufftGetSize_rank_0,&
      cufftGetSize_rank_1
  end interface
  !>  @brief Set the plan's auto-allocation flag.  The plan will allocate its own workarea.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  !>   @param[in] autoAllocate 0 to disable auto-allocation, non-zero to enable.
  interface cufftSetAutoAllocation
    function cufftSetAutoAllocation_(plan,autoAllocate) bind(c, &
&name="cufftSetAutoAllocation")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftSetAutoAllocation_
      type(c_ptr),value :: plan
      integer(c_int),value :: autoAllocate
    end function

  end interface
  !>  @brief Set the plan's work area.
  !> 
  !>   @param[in] plan Pointer to the FFT plan.
  !>   @param[in] workArea Pointer to the work area (on device).
  interface cufftSetWorkArea
    function cufftSetWorkArea_(plan,workArea) bind(c, &
&name="cufftSetWorkArea")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftSetWorkArea_
      type(c_ptr),value :: plan
      type(c_ptr),value :: workArea
    end function

  end interface
  !>  @brief Execute a (float) complex-to-complex FFT.
  !> 
  !>   @details If the input and output buffers are equal, an in-place
  !>   transform is performed.
  !> 
  !>   @param plan The FFT plan.
  !>   @param idata Input data (on device).
  !>   @param odata Output data (on device).
  !>   @param direction Either `HIPFFT_FORWARD` or `HIPFFT_BACKWARD`.
  interface cufftExecC2C
    function cufftExecC2C_(plan,idata,odata,direction) bind(c, &
&name="cufftExecC2C")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2C_
      type(c_ptr),value :: plan
      type(c_ptr),value :: idata
      type(c_ptr),value :: odata
      integer(c_int),value :: direction
    end function

    module procedure &
      cufftExecC2C_rank_0,&
      cufftExecC2C_rank_1,&
      cufftExecC2C_rank_2,&
      cufftExecC2C_rank_3
  end interface
  !>  @brief Execute a (float) real-to-complex FFT.
  !> 
  !>   @details If the input and output buffers are equal, an in-place
  !>   transform is performed.
  !> 
  !>   @param plan The FFT plan.
  !>   @param idata Input data (on device).
  !>   @param odata Output data (on device).
  interface cufftExecR2C
    function cufftExecR2C_(plan,idata,odata) bind(c, &
&name="cufftExecR2C")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecR2C_
      type(c_ptr),value :: plan
      type(c_ptr),value :: idata
      type(c_ptr),value :: odata
    end function

    module procedure &
      cufftExecR2C_rank_0,&
      cufftExecR2C_rank_1,&
      cufftExecR2C_rank_2,&
      cufftExecR2C_rank_3
  end interface
  !>  @brief Execute a (float) complex-to-real FFT.
  !> 
  !>   @details If the input and output buffers are equal, an in-place
  !>   transform is performed.
  !> 
  !>   @param plan The FFT plan.
  !>   @param idata Input data (on device).
  !>   @param odata Output data (on device).
  interface cufftExecC2R
    function cufftExecC2R_(plan,idata,odata) bind(c, &
&name="cufftExecC2R")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2R_
      type(c_ptr),value :: plan
      type(c_ptr),value :: idata
      type(c_ptr),value :: odata
    end function

    module procedure &
      cufftExecC2R_rank_0,&
      cufftExecC2R_rank_1,&
      cufftExecC2R_rank_2,&
      cufftExecC2R_rank_3
  end interface
  !>  @brief Execute a (double) complex-to-complex FFT.
  !> 
  !>   @details If the input and output buffers are equal, an in-place
  !>   transform is performed.
  !> 
  !>   @param plan The FFT plan.
  !>   @param idata Input data (on device).
  !>   @param odata Output data (on device).
  !>   @param direction Either `HIPFFT_FORWARD` or `HIPFFT_BACKWARD`.
  interface cufftExecZ2Z
    function cufftExecZ2Z_(plan,idata,odata,direction) bind(c, &
&name="cufftExecZ2Z")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2Z_
      type(c_ptr),value :: plan
      type(c_ptr),value :: idata
      type(c_ptr),value :: odata
      integer(c_int),value :: direction
    end function

    module procedure &
      cufftExecZ2Z_rank_0,&
      cufftExecZ2Z_rank_1,&
      cufftExecZ2Z_rank_2,&
      cufftExecZ2Z_rank_3
  end interface
  !>  @brief Execute a (double) real-to-complex FFT.
  !> 
  !>   @details If the input and output buffers are equal, an in-place
  !>   transform is performed.
  !> 
  !>   @param plan The FFT plan.
  !>   @param idata Input data (on device).
  !>   @param odata Output data (on device).
  interface cufftExecD2Z
    function cufftExecD2Z_(plan,idata,odata) bind(c, &
&name="cufftExecD2Z")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecD2Z_
      type(c_ptr),value :: plan
      type(c_ptr),value :: idata
      type(c_ptr),value :: odata
    end function

    module procedure &
      cufftExecD2Z_rank_0,&
      cufftExecD2Z_rank_1,&
      cufftExecD2Z_rank_2,&
      cufftExecD2Z_rank_3
  end interface
  !>  @brief Execute a (double) complex-to-real FFT.
  !> 
  !>   @details If the input and output buffers are equal, an in-place
  !>   transform is performed.
  !> 
  !>   @param plan The FFT plan.
  !>   @param idata Input data (on device).
  !>   @param odata Output data (on device).
  interface cufftExecZ2D
    function cufftExecZ2D_(plan,idata,odata) bind(c, &
&name="cufftExecZ2D")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2D_
      type(c_ptr),value :: plan
      type(c_ptr),value :: idata
      type(c_ptr),value :: odata
    end function

    module procedure &
      cufftExecZ2D_rank_0,&
      cufftExecZ2D_rank_1,&
      cufftExecZ2D_rank_2,&
      cufftExecZ2D_rank_3
  end interface
  !>  @brief Set HIP stream to execute plan on.
  !> 
  !>  @details Associates a HIP stream with a hipFFT plan.  All kernels
  !>  launched by this plan are associated with the provided stream.
  !> 
  !>  @param plan The FFT plan.
  !>  @param stream The HIP stream.
  interface cufftSetStream
    function cufftSetStream_(plan,stream) bind(c, name="cufftSetStream")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftSetStream_
      type(c_ptr),value :: plan
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Destroy and deallocate an existing plan.
  interface cufftDestroy
    function cufftDestroy_(plan) bind(c, name="cufftDestroy")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftDestroy_
      type(c_ptr),value :: plan
    end function

  end interface
  !>  @brief Get rocFFT/cuFFT version.
  !> 
  !>   @param[out] version cuFFT/rocFFT version (returned value).
  interface cufftGetVersion
    function cufftGetVersion_(version) bind(c, name="cufftGetVersion")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetVersion_
      type(c_ptr),value :: version
    end function

  end interface
  !>  @brief Get library property.
  !> 
  !>   @param[in] type Property type.
  !>   @param[out] value Returned value.
  interface cufftGetProperty
    function cufftGetProperty_(myType,myValue) bind(c, &
&name="cufftGetProperty")
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetProperty_
      integer(kind(CUFFT_MAJOR_VERSION)),value :: myType
      type(c_ptr),value :: myValue
    end function

  end interface

  contains
    function cufftPlanMany_rank_0(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftPlanMany_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_int),target :: n
      integer(c_int),target :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      !
      cufftPlanMany_rank_0 = cufftPlanMany_(plan,rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch)
    end function

    function cufftPlanMany_rank_1(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftPlanMany_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_int),target,dimension(:) :: n
      integer(c_int),target,dimension(:) :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target,dimension(:) :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      !
      cufftPlanMany_rank_1 = cufftPlanMany_(plan,rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch)
    end function

    function cufftMakePlan1d_rank_0(plan,nx,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan1d_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftMakePlan1d_rank_0 = cufftMakePlan1d_(plan,nx,myType,batch,&
&c_loc(workSize))
    end function

    function cufftMakePlan1d_rank_1(plan,nx,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan1d_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftMakePlan1d_rank_1 = cufftMakePlan1d_(plan,nx,myType,batch,&
&c_loc(workSize))
    end function

    function cufftMakePlan2d_rank_0(plan,nx,ny,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan2d_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target :: workSize
      !
      cufftMakePlan2d_rank_0 = cufftMakePlan2d_(plan,nx,ny,myType,&
&c_loc(workSize))
    end function

    function cufftMakePlan2d_rank_1(plan,nx,ny,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan2d_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftMakePlan2d_rank_1 = cufftMakePlan2d_(plan,nx,ny,myType,&
&c_loc(workSize))
    end function

    function cufftMakePlan3d_rank_0(plan,nx,ny,nz,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan3d_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(c_int) :: nz
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target :: workSize
      !
      cufftMakePlan3d_rank_0 = cufftMakePlan3d_(plan,nx,ny,nz,myType,&
&c_loc(workSize))
    end function

    function cufftMakePlan3d_rank_1(plan,nx,ny,nz,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlan3d_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(c_int) :: nz
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftMakePlan3d_rank_1 = cufftMakePlan3d_(plan,nx,ny,nz,myType,&
&c_loc(workSize))
    end function

    function cufftMakePlanMany_rank_0(plan,rank,n,inembed,istride,&
&idist,onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlanMany_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_int),target :: n
      integer(c_int),target :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftMakePlanMany_rank_0 = cufftMakePlanMany_(plan,rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch,c_loc(workSize))
    end function

    function cufftMakePlanMany_rank_1(plan,rank,n,inembed,istride,&
&idist,onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlanMany_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_int),target,dimension(:) :: n
      integer(c_int),target,dimension(:) :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target,dimension(:) :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftMakePlanMany_rank_1 = cufftMakePlanMany_(plan,rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch,c_loc(workSize))
    end function

    function cufftMakePlanMany64_rank_0(plan,rank,n,inembed,istride,&
&idist,onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlanMany64_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_long_long),target :: n
      integer(c_long_long),target :: inembed
      integer(c_long_long) :: istride
      integer(c_long_long) :: idist
      integer(c_long_long),target :: onembed
      integer(c_long_long) :: ostride
      integer(c_long_long) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_long_long) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftMakePlanMany64_rank_0 = cufftMakePlanMany64_(plan,rank,&
&c_loc(n),c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,&
&myType,batch,c_loc(workSize))
    end function

    function cufftMakePlanMany64_rank_1(plan,rank,n,inembed,istride,&
&idist,onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftMakePlanMany64_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_long_long),target,dimension(:) :: n
      integer(c_long_long),target,dimension(:) :: inembed
      integer(c_long_long) :: istride
      integer(c_long_long) :: idist
      integer(c_long_long),target,dimension(:) :: onembed
      integer(c_long_long) :: ostride
      integer(c_long_long) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_long_long) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftMakePlanMany64_rank_1 = cufftMakePlanMany64_(plan,rank,&
&c_loc(n),c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,&
&myType,batch,c_loc(workSize))
    end function

    function cufftEstimate1d_rank_0(nx,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate1d_rank_0
      integer(c_int) :: nx
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftEstimate1d_rank_0 = cufftEstimate1d_(nx,myType,batch,&
&c_loc(workSize))
    end function

    function cufftEstimate1d_rank_1(nx,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate1d_rank_1
      integer(c_int) :: nx
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftEstimate1d_rank_1 = cufftEstimate1d_(nx,myType,batch,&
&c_loc(workSize))
    end function

    function cufftEstimate2d_rank_0(nx,ny,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate2d_rank_0
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target :: workSize
      !
      cufftEstimate2d_rank_0 = cufftEstimate2d_(nx,ny,myType,&
&c_loc(workSize))
    end function

    function cufftEstimate2d_rank_1(nx,ny,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate2d_rank_1
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftEstimate2d_rank_1 = cufftEstimate2d_(nx,ny,myType,&
&c_loc(workSize))
    end function

    function cufftEstimate3d_rank_0(nx,ny,nz,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate3d_rank_0
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(c_int) :: nz
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target :: workSize
      !
      cufftEstimate3d_rank_0 = cufftEstimate3d_(nx,ny,nz,myType,&
&c_loc(workSize))
    end function

    function cufftEstimate3d_rank_1(nx,ny,nz,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimate3d_rank_1
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(c_int) :: nz
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftEstimate3d_rank_1 = cufftEstimate3d_(nx,ny,nz,myType,&
&c_loc(workSize))
    end function

    function cufftEstimateMany_rank_0(rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimateMany_rank_0
      integer(c_int) :: rank
      integer(c_int),target :: n
      integer(c_int),target :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftEstimateMany_rank_0 = cufftEstimateMany_(rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch,c_loc(workSize))
    end function

    function cufftEstimateMany_rank_1(rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftEstimateMany_rank_1
      integer(c_int) :: rank
      integer(c_int),target,dimension(:) :: n
      integer(c_int),target,dimension(:) :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target,dimension(:) :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftEstimateMany_rank_1 = cufftEstimateMany_(rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch,c_loc(workSize))
    end function

    function cufftGetSize1d_rank_0(plan,nx,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize1d_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftGetSize1d_rank_0 = cufftGetSize1d_(plan,nx,myType,batch,&
&c_loc(workSize))
    end function

    function cufftGetSize1d_rank_1(plan,nx,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize1d_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftGetSize1d_rank_1 = cufftGetSize1d_(plan,nx,myType,batch,&
&c_loc(workSize))
    end function

    function cufftGetSize2d_rank_0(plan,nx,ny,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize2d_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target :: workSize
      !
      cufftGetSize2d_rank_0 = cufftGetSize2d_(plan,nx,ny,myType,&
&c_loc(workSize))
    end function

    function cufftGetSize2d_rank_1(plan,nx,ny,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize2d_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftGetSize2d_rank_1 = cufftGetSize2d_(plan,nx,ny,myType,&
&c_loc(workSize))
    end function

    function cufftGetSize3d_rank_0(plan,nx,ny,nz,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize3d_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(c_int) :: nz
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target :: workSize
      !
      cufftGetSize3d_rank_0 = cufftGetSize3d_(plan,nx,ny,nz,myType,&
&c_loc(workSize))
    end function

    function cufftGetSize3d_rank_1(plan,nx,ny,nz,myType,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize3d_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: nx
      integer(c_int) :: ny
      integer(c_int) :: nz
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftGetSize3d_rank_1 = cufftGetSize3d_(plan,nx,ny,nz,myType,&
&c_loc(workSize))
    end function

    function cufftGetSizeMany_rank_0(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSizeMany_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_int),target :: n
      integer(c_int),target :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftGetSizeMany_rank_0 = cufftGetSizeMany_(plan,rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch,c_loc(workSize))
    end function

    function cufftGetSizeMany_rank_1(plan,rank,n,inembed,istride,idist,&
&onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSizeMany_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_int),target,dimension(:) :: n
      integer(c_int),target,dimension(:) :: inembed
      integer(c_int) :: istride
      integer(c_int) :: idist
      integer(c_int),target,dimension(:) :: onembed
      integer(c_int) :: ostride
      integer(c_int) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_int) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftGetSizeMany_rank_1 = cufftGetSizeMany_(plan,rank,c_loc(n),&
&c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,myType,&
&batch,c_loc(workSize))
    end function

    function cufftGetSizeMany64_rank_0(plan,rank,n,inembed,istride,&
&idist,onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSizeMany64_rank_0
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_long_long),target :: n
      integer(c_long_long),target :: inembed
      integer(c_long_long) :: istride
      integer(c_long_long) :: idist
      integer(c_long_long),target :: onembed
      integer(c_long_long) :: ostride
      integer(c_long_long) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_long_long) :: batch
      integer(c_size_t),target :: workSize
      !
      cufftGetSizeMany64_rank_0 = cufftGetSizeMany64_(plan,rank,&
&c_loc(n),c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,&
&myType,batch,c_loc(workSize))
    end function

    function cufftGetSizeMany64_rank_1(plan,rank,n,inembed,istride,&
&idist,onembed,ostride,odist,myType,batch,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSizeMany64_rank_1
      type(c_ptr) :: plan
      integer(c_int) :: rank
      integer(c_long_long),target,dimension(:) :: n
      integer(c_long_long),target,dimension(:) :: inembed
      integer(c_long_long) :: istride
      integer(c_long_long) :: idist
      integer(c_long_long),target,dimension(:) :: onembed
      integer(c_long_long) :: ostride
      integer(c_long_long) :: odist
      integer(kind(CUFFT_R2C)) :: myType
      integer(c_long_long) :: batch
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftGetSizeMany64_rank_1 = cufftGetSizeMany64_(plan,rank,&
&c_loc(n),c_loc(inembed),istride,idist,c_loc(onembed),ostride,odist,&
&myType,batch,c_loc(workSize))
    end function

    function cufftGetSize_rank_0(plan,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize_rank_0
      type(c_ptr) :: plan
      integer(c_size_t),target :: workSize
      !
      cufftGetSize_rank_0 = cufftGetSize_(plan,c_loc(workSize))
    end function

    function cufftGetSize_rank_1(plan,workSize)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftGetSize_rank_1
      type(c_ptr) :: plan
      integer(c_size_t),target,dimension(:) :: workSize
      !
      cufftGetSize_rank_1 = cufftGetSize_(plan,c_loc(workSize))
    end function

    function cufftExecC2C_rank_0(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2C_rank_0
      type(c_ptr) :: plan
      complex(c_float_complex),target :: idata
      complex(c_float_complex),target :: odata
      integer(c_int) :: direction
      !
      cufftExecC2C_rank_0 = cufftExecC2C_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecC2C_rank_1(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2C_rank_1
      type(c_ptr) :: plan
      complex(c_float_complex),target,dimension(:) :: idata
      complex(c_float_complex),target,dimension(:) :: odata
      integer(c_int) :: direction
      !
      cufftExecC2C_rank_1 = cufftExecC2C_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecR2C_rank_0(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecR2C_rank_0
      type(c_ptr) :: plan
      real(c_float),target :: idata
      complex(c_float_complex),target :: odata
      !
      cufftExecR2C_rank_0 = cufftExecR2C_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecR2C_rank_1(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecR2C_rank_1
      type(c_ptr) :: plan
      real(c_float),target,dimension(:) :: idata
      complex(c_float_complex),target,dimension(:) :: odata
      !
      cufftExecR2C_rank_1 = cufftExecR2C_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecC2R_rank_0(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2R_rank_0
      type(c_ptr) :: plan
      complex(c_float_complex),target :: idata
      real(c_float),target :: odata
      !
      cufftExecC2R_rank_0 = cufftExecC2R_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecC2R_rank_1(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2R_rank_1
      type(c_ptr) :: plan
      complex(c_float_complex),target,dimension(:) :: idata
      real(c_float),target,dimension(:) :: odata
      !
      cufftExecC2R_rank_1 = cufftExecC2R_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecZ2Z_rank_0(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2Z_rank_0
      type(c_ptr) :: plan
      complex(c_double_complex),target :: idata
      complex(c_double_complex),target :: odata
      integer(c_int) :: direction
      !
      cufftExecZ2Z_rank_0 = cufftExecZ2Z_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecZ2Z_rank_1(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2Z_rank_1
      type(c_ptr) :: plan
      complex(c_double_complex),target,dimension(:) :: idata
      complex(c_double_complex),target,dimension(:) :: odata
      integer(c_int) :: direction
      !
      cufftExecZ2Z_rank_1 = cufftExecZ2Z_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecD2Z_rank_0(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecD2Z_rank_0
      type(c_ptr) :: plan
      real(c_double),target :: idata
      complex(c_double_complex),target :: odata
      !
      cufftExecD2Z_rank_0 = cufftExecD2Z_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecD2Z_rank_1(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecD2Z_rank_1
      type(c_ptr) :: plan
      real(c_double),target,dimension(:) :: idata
      complex(c_double_complex),target,dimension(:) :: odata
      !
      cufftExecD2Z_rank_1 = cufftExecD2Z_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecZ2D_rank_0(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2D_rank_0
      type(c_ptr) :: plan
      complex(c_double_complex),target :: idata
      real(c_double),target :: odata
      !
      cufftExecZ2D_rank_0 = cufftExecZ2D_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecZ2D_rank_1(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2D_rank_1
      type(c_ptr) :: plan
      complex(c_double_complex),target,dimension(:) :: idata
      real(c_double),target,dimension(:) :: odata
      !
      cufftExecZ2D_rank_1 = cufftExecZ2D_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    ! 2D    
    function cufftExecC2C_rank_2(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2C_rank_2
      type(c_ptr),value :: plan
      complex(c_float_complex),target,dimension(:,:) :: idata
      complex(c_float_complex),target,dimension(:,:) :: odata
      integer(c_int),value :: direction
      !
      cufftExecC2C_rank_2 = cufftExecC2C_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecR2C_rank_2(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecR2C_rank_2
      type(c_ptr),value :: plan
      real(c_float),target,dimension(:,:) :: idata
      complex(c_float_complex),target,dimension(:,:) :: odata
      !
      cufftExecR2C_rank_2 = cufftExecR2C_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecC2R_rank_2(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2R_rank_2
      type(c_ptr),value :: plan
      complex(c_float_complex),target,dimension(:,:) :: idata
      real(c_float),target,dimension(:,:) :: odata
      !
      cufftExecC2R_rank_2 = cufftExecC2R_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecZ2Z_rank_2(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2Z_rank_2
      type(c_ptr),value :: plan
      complex(c_double_complex),target,dimension(:,:) :: idata
      complex(c_double_complex),target,dimension(:,:) :: odata
      integer(c_int),value :: direction
      !
      cufftExecZ2Z_rank_2 = cufftExecZ2Z_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecD2Z_rank_2(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecD2Z_rank_2
      type(c_ptr),value :: plan
      real(c_double),target,dimension(:,:) :: idata
      complex(c_double_complex),target,dimension(:,:) :: odata
      !
      cufftExecD2Z_rank_2 = cufftExecD2Z_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecZ2D_rank_2(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2D_rank_2
      type(c_ptr),value :: plan
      complex(c_double_complex),target,dimension(:,:) :: idata
      real(c_double),target,dimension(:,:) :: odata
      !
      cufftExecZ2D_rank_2 = cufftExecZ2D_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    ! 3D
    function cufftExecC2C_rank_3(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2C_rank_3
      type(c_ptr),value :: plan
      complex(c_float_complex),target,dimension(:,:,:) :: idata
      complex(c_float_complex),target,dimension(:,:,:) :: odata
      integer(c_int),value :: direction
      !
      cufftExecC2C_rank_3 = cufftExecC2C_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecR2C_rank_3(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecR2C_rank_3
      type(c_ptr),value :: plan
      real(c_float),target,dimension(:,:,:) :: idata
      complex(c_float_complex),target,dimension(:,:,:) :: odata
      !
      cufftExecR2C_rank_3 = cufftExecR2C_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecC2R_rank_3(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecC2R_rank_3
      type(c_ptr),value :: plan
      complex(c_float_complex),target,dimension(:,:,:) :: idata
      real(c_float),target,dimension(:,:,:) :: odata
      !
      cufftExecC2R_rank_3 = cufftExecC2R_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecZ2Z_rank_3(plan,idata,odata,direction)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2Z_rank_3
      type(c_ptr),value :: plan
      complex(c_double_complex),target,dimension(:,:,:) :: idata
      complex(c_double_complex),target,dimension(:,:,:) :: odata
      integer(c_int),value :: direction
      !
      cufftExecZ2Z_rank_3 = cufftExecZ2Z_(plan,c_loc(idata),&
&c_loc(odata),direction)
    end function

    function cufftExecD2Z_rank_3(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecD2Z_rank_3
      type(c_ptr),value :: plan
      real(c_double),target,dimension(:,:,:) :: idata
      complex(c_double_complex),target,dimension(:,:,:) :: odata
      !
      cufftExecD2Z_rank_3 = cufftExecD2Z_(plan,c_loc(idata),&
&c_loc(odata))
    end function

    function cufftExecZ2D_rank_3(plan,idata,odata)
      use iso_c_binding
      use cudafor_cufft_enums
      implicit none
      integer(kind(CUFFT_SUCCESS)) :: cufftExecZ2D_rank_3
      type(c_ptr),value :: plan
      complex(c_double_complex),target,dimension(:,:,:) :: idata
      real(c_double),target,dimension(:,:,:) :: odata
      !
      cufftExecZ2D_rank_3 = cufftExecZ2D_(plan,c_loc(idata),&
&c_loc(odata))
    end function
end module cudafor_cufft