


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ==============================================================================
! hipfort: FORTRAN Interfaces for GPU kernels
! ==============================================================================
! Copyright (c) 2020-2022 Advanced Micro Devices, Inc. All rights reserved.
! [MITx11 License]
! 
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
! IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
! AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
! OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
! THE SOFTWARE.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
          
module cudafor_cublas
  use cudafor_cublas_enums
  use cudafor_cublas_auxiliary
  implicit none

 
  
  interface cublasCreate
    function cublasCreate_(handle) bind(c, name="cublasCreate")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCreate_
      type(c_ptr) :: handle
    end function

  end interface
  
  interface cublasDestroy
    function cublasDestroy_(handle) bind(c, name="cublasDestroy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDestroy_
      type(c_ptr),value :: handle
    end function

  end interface
  
  interface cublasSetStream
    function cublasSetStream_(handle,streamId) bind(c, &
&name="cublasSetStream")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSetStream_
      type(c_ptr),value :: handle
      type(c_ptr),value :: streamId
    end function

  end interface
  
  interface cublasGetStream
    function cublasGetStream_(handle,streamId) bind(c, &
&name="cublasGetStream")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasGetStream_
      type(c_ptr),value :: handle
      type(c_ptr) :: streamId
    end function

  end interface
  
  interface cublasSetPointerMode
    function cublasSetPointerMode_(handle,mode) bind(c, &
&name="cublasSetPointerMode")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSetPointerMode_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_POINTER_MODE_HOST)),value :: mode
    end function

  end interface
  
  interface cublasGetPointerMode
    function cublasGetPointerMode_(handle,mode) bind(c, &
&name="cublasGetPointerMode")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasGetPointerMode_
      type(c_ptr),value :: handle
      type(c_ptr),value :: mode
    end function

  end interface
  
  interface cublasSetAtomicsMode
    function cublasSetAtomicsMode_(handle,atomics_mode) bind(c, &
&name="cublasSetAtomicsMode")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSetAtomicsMode_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_ATOMICS_NOT_ALLOWED)),value :: atomics_mode
    end function

  end interface
  
  interface cublasGetAtomicsMode
    function cublasGetAtomicsMode_(handle,atomics_mode) bind(c, &
&name="cublasGetAtomicsMode")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasGetAtomicsMode_
      type(c_ptr),value :: handle
      type(c_ptr),value :: atomics_mode
    end function

  end interface
  
  interface cublasIsamax
    function cublasIsamax_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIsamax")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamax_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIsamax_rank_0,&
      cublasIsamax_rank_1
  end interface
  
  interface cublasIdamax
    function cublasIdamax_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIdamax")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamax_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIdamax_rank_0,&
      cublasIdamax_rank_1
  end interface
  
  interface cublasIcamax
    function cublasIcamax_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIcamax")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamax_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIcamax_rank_0,&
      cublasIcamax_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     amax finds the first index of the element of maximum magnitude of a vector x.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     result
  !>               device pointer or host pointer to store the amax index.
  !>               return is 0.0 if n, incx<=0.
  interface cublasIzamax
    function cublasIzamax_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIzamax")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamax_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIzamax_rank_0,&
      cublasIzamax_rank_1
  end interface
  
  interface cublasIsamaxBatched
    function cublasIsamaxBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIsamaxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamaxBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIsamaxBatched_full_rank,&
      cublasIsamaxBatched_rank_0,&
      cublasIsamaxBatched_rank_1
  end interface
  
  interface cublasIdamaxBatched
    function cublasIdamaxBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIdamaxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamaxBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIdamaxBatched_full_rank,&
      cublasIdamaxBatched_rank_0,&
      cublasIdamaxBatched_rank_1
  end interface
  
  interface cublasIcamaxBatched
    function cublasIcamaxBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIcamaxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamaxBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIcamaxBatched_full_rank,&
      cublasIcamaxBatched_rank_0,&
      cublasIcamaxBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>      amaxBatched finds the first index of the element of maximum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each vector x_i
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch, must be > 0.
  !>     @param[out]
  !>     result
  !>               device or host array of pointers of batchCount size for results.
  !>               return is 0 if n, incx<=0.
  interface cublasIzamaxBatched
    function cublasIzamaxBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIzamaxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamaxBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIzamaxBatched_full_rank,&
      cublasIzamaxBatched_rank_0,&
      cublasIzamaxBatched_rank_1
  end interface
  
  interface cublasIsamaxStridedBatched
    function cublasIsamaxStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIsamaxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsamaxStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIsamaxStridedBatched_rank_0,&
      cublasIsamaxStridedBatched_rank_1
  end interface
  
  interface cublasIdamaxStridedBatched
    function cublasIdamaxStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIdamaxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdamaxStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIdamaxStridedBatched_rank_0,&
      cublasIdamaxStridedBatched_rank_1
  end interface
  
  interface cublasIcamaxStridedBatched
    function cublasIcamaxStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIcamaxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcamaxStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIcamaxStridedBatched_rank_0,&
      cublasIcamaxStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>      amaxStridedBatched finds the first index of the element of maximum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each vector x_i
  !>     @param[in]
  !>     x         device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               specifies the pointer increment between one x_i and the next x_(i + 1).
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>     @param[out]
  !>     result
  !>               device or host pointer for storing contiguous batchCount results.
  !>               return is 0 if n <= 0, incx<=0.
  !>
  interface cublasIzamaxStridedBatched
    function cublasIzamaxStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIzamaxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzamaxStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIzamaxStridedBatched_rank_0,&
      cublasIzamaxStridedBatched_rank_1
  end interface
  
  interface cublasIsamin
    function cublasIsamin_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIsamin")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamin_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIsamin_rank_0,&
      cublasIsamin_rank_1
  end interface
  
  interface cublasIdamin
    function cublasIdamin_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIdamin")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamin_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIdamin_rank_0,&
      cublasIdamin_rank_1
  end interface
  
  interface cublasIcamin
    function cublasIcamin_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIcamin")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamin_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIcamin_rank_0,&
      cublasIcamin_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     amin finds the first index of the element of minimum magnitude of a vector x.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     result
  !>               device pointer or host pointer to store the amin index.
  !>               return is 0.0 if n, incx<=0.
  interface cublasIzamin
    function cublasIzamin_(handle,n,x,incx,myResult) bind(c, &
&name="cublasIzamin")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamin_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIzamin_rank_0,&
      cublasIzamin_rank_1
  end interface
  
  interface cublasIsaminBatched
    function cublasIsaminBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIsaminBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsaminBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIsaminBatched_full_rank,&
      cublasIsaminBatched_rank_0,&
      cublasIsaminBatched_rank_1
  end interface
  
  interface cublasIdaminBatched
    function cublasIdaminBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIdaminBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdaminBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIdaminBatched_full_rank,&
      cublasIdaminBatched_rank_0,&
      cublasIdaminBatched_rank_1
  end interface
  
  interface cublasIcaminBatched
    function cublasIcaminBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIcaminBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcaminBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIcaminBatched_full_rank,&
      cublasIcaminBatched_rank_0,&
      cublasIcaminBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     aminBatched finds the first index of the element of minimum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each vector x_i
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch, must be > 0.
  !>     @param[out]
  !>     result
  !>               device or host pointers to array of batchCount size for results.
  !>               return is 0 if n, incx<=0.
  interface cublasIzaminBatched
    function cublasIzaminBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasIzaminBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzaminBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIzaminBatched_full_rank,&
      cublasIzaminBatched_rank_0,&
      cublasIzaminBatched_rank_1
  end interface
  
  interface cublasIsaminStridedBatched
    function cublasIsaminStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIsaminStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsaminStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIsaminStridedBatched_rank_0,&
      cublasIsaminStridedBatched_rank_1
  end interface
  
  interface cublasIdaminStridedBatched
    function cublasIdaminStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIdaminStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdaminStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIdaminStridedBatched_rank_0,&
      cublasIdaminStridedBatched_rank_1
  end interface
  
  interface cublasIcaminStridedBatched
    function cublasIcaminStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIcaminStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcaminStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIcaminStridedBatched_rank_0,&
      cublasIcaminStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>      aminStridedBatched finds the first index of the element of minimum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each vector x_i
  !>     @param[in]
  !>     x         device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               specifies the pointer increment between one x_i and the next x_(i + 1)
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>     @param[out]
  !>     result
  !>               device or host pointer to array for storing contiguous batchCount results.
  !>               return is 0 if n <= 0, incx<=0.
  !>
  interface cublasIzaminStridedBatched
    function cublasIzaminStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasIzaminStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzaminStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasIzaminStridedBatched_rank_0,&
      cublasIzaminStridedBatched_rank_1
  end interface
  
  interface cublasSasum
    function cublasSasum_(handle,n,x,incx,myResult) bind(c, &
&name="cublasSasum")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSasum_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSasum_rank_0,&
      cublasSasum_rank_1
  end interface
  
  interface cublasDasum
    function cublasDasum_(handle,n,x,incx,myResult) bind(c, &
&name="cublasDasum")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDasum_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDasum_rank_0,&
      cublasDasum_rank_1
  end interface
  
  interface cublasScasum
    function cublasScasum_(handle,n,x,incx,myResult) bind(c, &
&name="cublasScasum")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScasum_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasScasum_rank_0,&
      cublasScasum_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     asum computes the sum of the magnitudes of elements of a real vector x,
  !>          or the sum of magnitudes of the real and imaginary parts of elements if x is a complex vector.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x and y.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x. incx must be > 0.
  !>     @param[inout]
  !>     result
  !>               device pointer or host pointer to store the asum product.
  !>               return is 0.0 if n <= 0.
  !>
  interface cublasDzasum
    function cublasDzasum_(handle,n,x,incx,myResult) bind(c, &
&name="cublasDzasum")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDzasum_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDzasum_rank_0,&
      cublasDzasum_rank_1
  end interface
  
  interface cublasSasumBatched
    function cublasSasumBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasSasumBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSasumBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSasumBatched_full_rank,&
      cublasSasumBatched_rank_0,&
      cublasSasumBatched_rank_1
  end interface
  
  interface cublasDasumBatched
    function cublasDasumBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasDasumBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDasumBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDasumBatched_full_rank,&
      cublasDasumBatched_rank_0,&
      cublasDasumBatched_rank_1
  end interface
  
  interface cublasScasumBatched
    function cublasScasumBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasScasumBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScasumBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasScasumBatched_full_rank,&
      cublasScasumBatched_rank_0,&
      cublasScasumBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     asumBatched computes the sum of the magnitudes of the elements in a batch of real vectors x_i,
  !>         or the sum of magnitudes of the real and imaginary parts of elements if x_i is a complex
  !>         vector, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each vector x_i
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch.
  !>     @param[out]
  !>     result
  !>               device array or host array of batchCount size for results.
  !>               return is 0.0 if n, incx<=0.
  interface cublasDzasumBatched
    function cublasDzasumBatched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasDzasumBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDzasumBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDzasumBatched_full_rank,&
      cublasDzasumBatched_rank_0,&
      cublasDzasumBatched_rank_1
  end interface
  
  interface cublasSasumStridedBatched
    function cublasSasumStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasSasumStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSasumStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSasumStridedBatched_rank_0,&
      cublasSasumStridedBatched_rank_1
  end interface
  
  interface cublasDasumStridedBatched
    function cublasDasumStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasDasumStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDasumStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDasumStridedBatched_rank_0,&
      cublasDasumStridedBatched_rank_1
  end interface
  
  interface cublasScasumStridedBatched
    function cublasScasumStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasScasumStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScasumStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasScasumStridedBatched_rank_0,&
      cublasScasumStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     asumStridedBatched computes the sum of the magnitudes of elements of a real vectors x_i,
  !>         or the sum of magnitudes of the real and imaginary parts of elements if x_i is a complex
  !>         vector, for i = 1, ..., batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each vector x_i
  !>     @param[in]
  !>     x         device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>               There are no restrictions placed on stride_x, however the user should
  !>               take care to ensure that stride_x is of appropriate size, for a typical
  !>               case this means stride_x >= n * incx.
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>     @param[out]
  !>     result
  !>               device pointer or host pointer to array for storing contiguous batchCount results.
  !>               return is 0.0 if n, incx<=0.
  interface cublasDzasumStridedBatched
    function cublasDzasumStridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasDzasumStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDzasumStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDzasumStridedBatched_rank_0,&
      cublasDzasumStridedBatched_rank_1
  end interface
  
  interface cublasSaxpy
    function cublasSaxpy_(handle,n,alpha,x,incx,y,incy) bind(c, &
&name="cublasSaxpy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSaxpy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasSaxpy_rank_0,&
      cublasSaxpy_rank_1
  end interface
  
  interface cublasDaxpy
    function cublasDaxpy_(handle,n,alpha,x,incx,y,incy) bind(c, &
&name="cublasDaxpy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDaxpy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDaxpy_rank_0,&
      cublasDaxpy_rank_1
  end interface
  
  interface cublasCaxpy
    function cublasCaxpy_(handle,n,alpha,x,incx,y,incy) bind(c, &
&name="cublasCaxpy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCaxpy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasCaxpy_rank_0,&
      cublasCaxpy_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     axpy   computes ant alpha multiplied by vector x, plus vector y
  !> 
  !>         y := alpha * x + y
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x and y.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to specify the scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[out]
  !>     y         device pointer storing vector y.
  !>     @param[inout]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZaxpy
    function cublasZaxpy_(handle,n,alpha,x,incx,y,incy) bind(c, &
&name="cublasZaxpy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZaxpy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZaxpy_rank_0,&
      cublasZaxpy_rank_1
  end interface
  
  interface cublasSaxpyBatched
    function cublasSaxpyBatched_(handle,n,alpha,x,incx,y,incy,&
&batchCount) bind(c, name="cublasSaxpyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSaxpyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSaxpyBatched_full_rank,&
      cublasSaxpyBatched_rank_0,&
      cublasSaxpyBatched_rank_1
  end interface
  
  interface cublasDaxpyBatched
    function cublasDaxpyBatched_(handle,n,alpha,x,incx,y,incy,&
&batchCount) bind(c, name="cublasDaxpyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDaxpyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDaxpyBatched_full_rank,&
      cublasDaxpyBatched_rank_0,&
      cublasDaxpyBatched_rank_1
  end interface
  
  interface cublasCaxpyBatched
    function cublasCaxpyBatched_(handle,n,alpha,x,incx,y,incy,&
&batchCount) bind(c, name="cublasCaxpyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCaxpyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCaxpyBatched_full_rank,&
      cublasCaxpyBatched_rank_0,&
      cublasCaxpyBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     axpyBatched   compute y := alpha * x + y over a set of batched vectors.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x and y.
  !>     @param[in]
  !>     alpha     specifies the scalar alpha.
  !>     @param[in]
  !>     x         pointer storing vector x on the GPU.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[out]
  !>     y         pointer storing vector y on the GPU.
  !>     @param[inout]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  interface cublasZaxpyBatched
    function cublasZaxpyBatched_(handle,n,alpha,x,incx,y,incy,&
&batchCount) bind(c, name="cublasZaxpyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZaxpyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZaxpyBatched_full_rank,&
      cublasZaxpyBatched_rank_0,&
      cublasZaxpyBatched_rank_1
  end interface
  
  interface cublasSaxpyStridedBatched
    function cublasSaxpyStridedBatched_(handle,n,alpha,x,incx,stridex,&
&y,incy,stridey,batchCount) bind(c, name="cublasSaxpyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSaxpyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSaxpyStridedBatched_rank_0,&
      cublasSaxpyStridedBatched_rank_1
  end interface
  
  interface cublasDaxpyStridedBatched
    function cublasDaxpyStridedBatched_(handle,n,alpha,x,incx,stridex,&
&y,incy,stridey,batchCount) bind(c, name="cublasDaxpyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDaxpyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDaxpyStridedBatched_rank_0,&
      cublasDaxpyStridedBatched_rank_1
  end interface
  
  interface cublasCaxpyStridedBatched
    function cublasCaxpyStridedBatched_(handle,n,alpha,x,incx,stridex,&
&y,incy,stridey,batchCount) bind(c, name="cublasCaxpyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCaxpyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCaxpyStridedBatched_rank_0,&
      cublasCaxpyStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     axpyStridedBatched   compute y := alpha * x + y over a set of strided batched vectors.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>     @param[in]
  !>     alpha     specifies the scalar alpha.
  !>     @param[in]
  !>     x         pointer storing vector x on the GPU.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               specifies the increment between vectors of x.
  !>     @param[out]
  !>     y         pointer storing vector y on the GPU.
  !>     @param[inout]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     stridey   [hipblasStride]
  !>               specifies the increment between vectors of y.
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>
  interface cublasZaxpyStridedBatched
    function cublasZaxpyStridedBatched_(handle,n,alpha,x,incx,stridex,&
&y,incy,stridey,batchCount) bind(c, name="cublasZaxpyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZaxpyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZaxpyStridedBatched_rank_0,&
      cublasZaxpyStridedBatched_rank_1
  end interface
  
  interface cublasScopy
    function cublasScopy_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasScopy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScopy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasScopy_rank_0,&
      cublasScopy_rank_1
  end interface
  
  interface cublasDcopy
    function cublasDcopy_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasDcopy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDcopy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDcopy_rank_0,&
      cublasDcopy_rank_1
  end interface
  
  interface cublasCcopy
    function cublasCcopy_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasCcopy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCcopy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasCcopy_rank_0,&
      cublasCcopy_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     copy  copies each element x[i] into y[i], for  i = 1 , ... , n
  !> 
  !>         y := x,
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x to be copied to y.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[out]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZcopy
    function cublasZcopy_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasZcopy")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZcopy_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZcopy_rank_0,&
      cublasZcopy_rank_1
  end interface
  
  interface cublasScopyBatched
    function cublasScopyBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasScopyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScopyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasScopyBatched_full_rank,&
      cublasScopyBatched_rank_0,&
      cublasScopyBatched_rank_1
  end interface
  
  interface cublasDcopyBatched
    function cublasDcopyBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasDcopyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDcopyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDcopyBatched_full_rank,&
      cublasDcopyBatched_rank_0,&
      cublasDcopyBatched_rank_1
  end interface
  
  interface cublasCcopyBatched
    function cublasCcopyBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasCcopyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCcopyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCcopyBatched_full_rank,&
      cublasCcopyBatched_rank_0,&
      cublasCcopyBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     copyBatched copies each element x_i[j] into y_i[j], for  j = 1 , ... , n; i = 1 , ... , batchCount
  !> 
  !>         y_i := x_i,
  !> 
  !>     where (x_i, y_i) is the i-th instance of the batch.
  !>     x_i and y_i are vectors.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in each x_i to be copied to y_i.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i.
  !>     @param[out]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZcopyBatched
    function cublasZcopyBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasZcopyBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZcopyBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZcopyBatched_full_rank,&
      cublasZcopyBatched_rank_0,&
      cublasZcopyBatched_rank_1
  end interface
  
  interface cublasScopyStridedBatched
    function cublasScopyStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasScopyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScopyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasScopyStridedBatched_rank_0,&
      cublasScopyStridedBatched_rank_1
  end interface
  
  interface cublasDcopyStridedBatched
    function cublasDcopyStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasDcopyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDcopyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDcopyStridedBatched_rank_0,&
      cublasDcopyStridedBatched_rank_1
  end interface
  
  interface cublasCcopyStridedBatched
    function cublasCcopyStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasCcopyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCcopyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCcopyStridedBatched_rank_0,&
      cublasCcopyStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     copyStridedBatched copies each element x_i[j] into y_i[j], for  j = 1 , ... , n; i = 1 , ... , batchCount
  !> 
  !>         y_i := x_i,
  !> 
  !>     where (x_i, y_i) is the i-th instance of the batch.
  !>     x_i and y_i are vectors.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in each x_i to be copied to y_i.
  !>     @param[in]
  !>     x         device pointer to the first vector (x_1) in the batch.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increments for the elements of vectors x_i.
  !>     @param[in]
  !>     stridex     [hipblasStride]
  !>                 stride from the start of one vector (x_i) and the next one (x_i+1).
  !>                 There are no restrictions placed on stride_x, however the user should
  !>                 take care to ensure that stride_x is of appropriate size, for a typical
  !>                 case this means stride_x >= n * incx.
  !>     @param[out]
  !>     y         device pointer to the first vector (y_1) in the batch.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of vectors y_i.
  !>     @param[in]
  !>     stridey     [hipblasStride]
  !>                 stride from the start of one vector (y_i) and the next one (y_i+1).
  !>                 There are no restrictions placed on stride_y, however the user should
  !>                 take care to ensure that stride_y is of appropriate size, for a typical
  !>                 case this means stride_y >= n * incy. stridey should be non zero.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZcopyStridedBatched
    function cublasZcopyStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasZcopyStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZcopyStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZcopyStridedBatched_rank_0,&
      cublasZcopyStridedBatched_rank_1
  end interface
  
  interface cublasSdot
    function cublasSdot_(handle,n,x,incx,y,incy,myResult) bind(c, &
&name="cublasSdot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSdot_rank_0,&
      cublasSdot_rank_1
  end interface
  
  interface cublasDdot
    function cublasDdot_(handle,n,x,incx,y,incy,myResult) bind(c, &
&name="cublasDdot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDdot_rank_0,&
      cublasDdot_rank_1
  end interface
  
  interface cublasCdotc
    function cublasCdotc_(handle,n,x,incx,y,incy,myResult) bind(c, &
&name="cublasCdotc")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotc_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasCdotc_rank_0,&
      cublasCdotc_rank_1
  end interface
  
  interface cublasCdotu
    function cublasCdotu_(handle,n,x,incx,y,incy,myResult) bind(c, &
&name="cublasCdotu")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotu_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasCdotu_rank_0,&
      cublasCdotu_rank_1
  end interface
  
  interface cublasZdotc
    function cublasZdotc_(handle,n,x,incx,y,incy,myResult) bind(c, &
&name="cublasZdotc")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotc_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasZdotc_rank_0,&
      cublasZdotc_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     dot(u)  performs the dot product of vectors x and y
  !> 
  !>         result = x * y;
  !> 
  !>     dotc  performs the dot product of the conjugate of complex vector x and complex vector y
  !> 
  !>         result = conjugate (x) * y;
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x and y.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     result
  !>               device pointer or host pointer to store the dot product.
  !>               return is 0.0 if n <= 0.
  !>
  interface cublasZdotu
    function cublasZdotu_(handle,n,x,incx,y,incy,myResult) bind(c, &
&name="cublasZdotu")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotu_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasZdotu_rank_0,&
      cublasZdotu_rank_1
  end interface
  
  interface cublasSdotBatched
    function cublasSdotBatched_(handle,n,x,incx,y,incy,batchCount,&
&myResult) bind(c, name="cublasSdotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSdotBatched_full_rank,&
      cublasSdotBatched_rank_0,&
      cublasSdotBatched_rank_1
  end interface
  
  interface cublasDdotBatched
    function cublasDdotBatched_(handle,n,x,incx,y,incy,batchCount,&
&myResult) bind(c, name="cublasDdotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDdotBatched_full_rank,&
      cublasDdotBatched_rank_0,&
      cublasDdotBatched_rank_1
  end interface
  
  interface cublasCdotcBatched
    function cublasCdotcBatched_(handle,n,x,incx,y,incy,batchCount,&
&myResult) bind(c, name="cublasCdotcBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotcBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasCdotcBatched_full_rank,&
      cublasCdotcBatched_rank_0,&
      cublasCdotcBatched_rank_1
  end interface
  
  interface cublasCdotuBatched
    function cublasCdotuBatched_(handle,n,x,incx,y,incy,batchCount,&
&myResult) bind(c, name="cublasCdotuBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotuBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasCdotuBatched_full_rank,&
      cublasCdotuBatched_rank_0,&
      cublasCdotuBatched_rank_1
  end interface
  
  interface cublasZdotcBatched
    function cublasZdotcBatched_(handle,n,x,incx,y,incy,batchCount,&
&myResult) bind(c, name="cublasZdotcBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotcBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasZdotcBatched_full_rank,&
      cublasZdotcBatched_rank_0,&
      cublasZdotcBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     dotBatched(u) performs a batch of dot products of vectors x and y
  !> 
  !>         result_i = x_i * y_i;
  !> 
  !>     dotcBatched  performs a batch of dot products of the conjugate of complex vector x and complex vector y
  !> 
  !>         result_i = conjugate (x_i) * y_i;
  !> 
  !>     where (x_i, y_i) is the i-th instance of the batch.
  !>     x_i and y_i are vectors, for i = 1, ..., batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in each x_i and y_i.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>     @param[inout]
  !>     result
  !>               device array or host array of batchCount size to store the dot products of each batch.
  !>               return 0.0 for each element if n <= 0.
  !>
  interface cublasZdotuBatched
    function cublasZdotuBatched_(handle,n,x,incx,y,incy,batchCount,&
&myResult) bind(c, name="cublasZdotuBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotuBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasZdotuBatched_full_rank,&
      cublasZdotuBatched_rank_0,&
      cublasZdotuBatched_rank_1
  end interface
  
  interface cublasSdotStridedBatched
    function cublasSdotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount,myResult) bind(c, name="cublasSdotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSdotStridedBatched_rank_0,&
      cublasSdotStridedBatched_rank_1
  end interface
  
  interface cublasDdotStridedBatched
    function cublasDdotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount,myResult) bind(c, name="cublasDdotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDdotStridedBatched_rank_0,&
      cublasDdotStridedBatched_rank_1
  end interface
  
  interface cublasCdotcStridedBatched
    function cublasCdotcStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount,myResult) bind(c, &
&name="cublasCdotcStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotcStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasCdotcStridedBatched_rank_0,&
      cublasCdotcStridedBatched_rank_1
  end interface
  
  interface cublasCdotuStridedBatched
    function cublasCdotuStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount,myResult) bind(c, &
&name="cublasCdotuStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotuStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasCdotuStridedBatched_rank_0,&
      cublasCdotuStridedBatched_rank_1
  end interface
  
  interface cublasZdotcStridedBatched
    function cublasZdotcStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount,myResult) bind(c, &
&name="cublasZdotcStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotcStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasZdotcStridedBatched_rank_0,&
      cublasZdotcStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     dotStridedBatched(u)  performs a batch of dot products of vectors x and y
  !> 
  !>         result_i = x_i * y_i;
  !> 
  !>     dotcStridedBatched  performs a batch of dot products of the conjugate of complex vector x and complex vector y
  !> 
  !>         result_i = conjugate (x_i) * y_i;
  !> 
  !>     where (x_i, y_i) is the i-th instance of the batch.
  !>     x_i and y_i are vectors, for i = 1, ..., batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in each x_i and y_i.
  !>     @param[in]
  !>     x         device pointer to the first vector (x_1) in the batch.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex     [hipblasStride]
  !>                 stride from the start of one vector (x_i) and the next one (x_i+1)
  !>     @param[in]
  !>     y         device pointer to the first vector (y_1) in the batch.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     stridey     [hipblasStride]
  !>                 stride from the start of one vector (y_i) and the next one (y_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>     @param[inout]
  !>     result
  !>               device array or host array of batchCount size to store the dot products of each batch.
  !>               return 0.0 for each element if n <= 0.
  !>
  interface cublasZdotuStridedBatched
    function cublasZdotuStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount,myResult) bind(c, &
&name="cublasZdotuStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotuStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasZdotuStridedBatched_rank_0,&
      cublasZdotuStridedBatched_rank_1
  end interface
  
  interface cublasSnrm2
    function cublasSnrm2_(handle,n,x,incx,myResult) bind(c, &
&name="cublasSnrm2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSnrm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSnrm2_rank_0,&
      cublasSnrm2_rank_1
  end interface
  
  interface cublasDnrm2
    function cublasDnrm2_(handle,n,x,incx,myResult) bind(c, &
&name="cublasDnrm2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDnrm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDnrm2_rank_0,&
      cublasDnrm2_rank_1
  end interface
  
  interface cublasScnrm2
    function cublasScnrm2_(handle,n,x,incx,myResult) bind(c, &
&name="cublasScnrm2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScnrm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasScnrm2_rank_0,&
      cublasScnrm2_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     nrm2 computes the euclidean norm of a real or complex vector
  !> 
  !>               result := sqrt( x'*x ) for real vectors
  !>               result := sqrt( x**H*x ) for complex vectors
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     result
  !>               device pointer or host pointer to store the nrm2 product.
  !>               return is 0.0 if n, incx<=0.
  interface cublasDznrm2
    function cublasDznrm2_(handle,n,x,incx,myResult) bind(c, &
&name="cublasDznrm2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDznrm2_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDznrm2_rank_0,&
      cublasDznrm2_rank_1
  end interface
  
  interface cublasSnrm2Batched
    function cublasSnrm2Batched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasSnrm2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSnrm2Batched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSnrm2Batched_full_rank,&
      cublasSnrm2Batched_rank_0,&
      cublasSnrm2Batched_rank_1
  end interface
  
  interface cublasDnrm2Batched
    function cublasDnrm2Batched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasDnrm2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDnrm2Batched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDnrm2Batched_full_rank,&
      cublasDnrm2Batched_rank_0,&
      cublasDnrm2Batched_rank_1
  end interface
  
  interface cublasScnrm2Batched
    function cublasScnrm2Batched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasScnrm2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScnrm2Batched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasScnrm2Batched_full_rank,&
      cublasScnrm2Batched_rank_0,&
      cublasScnrm2Batched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     nrm2Batched computes the euclidean norm over a batch of real or complex vectors
  !> 
  !>               result := sqrt( x_i'*x_i ) for real vectors x, for i = 1, ..., batchCount
  !>               result := sqrt( x_i**H*x_i ) for complex vectors x, for i = 1, ..., batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each x_i.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>     @param[out]
  !>     result
  !>               device pointer or host pointer to array of batchCount size for nrm2 results.
  !>               return is 0.0 for each element if n <= 0, incx<=0.
  !>
  interface cublasDznrm2Batched
    function cublasDznrm2Batched_(handle,n,x,incx,batchCount,myResult) &
&bind(c, name="cublasDznrm2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDznrm2Batched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDznrm2Batched_full_rank,&
      cublasDznrm2Batched_rank_0,&
      cublasDznrm2Batched_rank_1
  end interface
  
  interface cublasSnrm2StridedBatched
    function cublasSnrm2StridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasSnrm2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSnrm2StridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasSnrm2StridedBatched_rank_0,&
      cublasSnrm2StridedBatched_rank_1
  end interface
  
  interface cublasDnrm2StridedBatched
    function cublasDnrm2StridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasDnrm2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDnrm2StridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDnrm2StridedBatched_rank_0,&
      cublasDnrm2StridedBatched_rank_1
  end interface
  
  interface cublasScnrm2StridedBatched
    function cublasScnrm2StridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasScnrm2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScnrm2StridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasScnrm2StridedBatched_rank_0,&
      cublasScnrm2StridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     nrm2StridedBatched computes the euclidean norm over a batch of real or complex vectors
  !> 
  !>               := sqrt( x_i'*x_i ) for real vectors x, for i = 1, ..., batchCount
  !>               := sqrt( x_i**H*x_i ) for complex vectors, for i = 1, ..., batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               number of elements in each x_i.
  !>     @param[in]
  !>     x         device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i. incx must be > 0.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>               There are no restrictions placed on stride_x, however the user should
  !>               take care to ensure that stride_x is of appropriate size, for a typical
  !>               case this means stride_x >= n * incx.
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>     @param[out]
  !>     result
  !>               device pointer or host pointer to array for storing contiguous batch_count results.
  !>               return is 0.0 for each element if n <= 0, incx<=0.
  !>
  interface cublasDznrm2StridedBatched
    function cublasDznrm2StridedBatched_(handle,n,x,incx,stridex,&
&batchCount,myResult) bind(c, name="cublasDznrm2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDznrm2StridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
      type(c_ptr),value :: myResult
    end function

    module procedure &
      cublasDznrm2StridedBatched_rank_0,&
      cublasDznrm2StridedBatched_rank_1
  end interface
  
  interface cublasSrot
    function cublasSrot_(handle,n,x,incx,y,incy,c,s) bind(c, &
&name="cublasSrot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

    module procedure &
      cublasSrot_rank_0,&
      cublasSrot_rank_1
  end interface
  
  interface cublasDrot
    function cublasDrot_(handle,n,x,incx,y,incy,c,s) bind(c, &
&name="cublasDrot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

    module procedure &
      cublasDrot_rank_0,&
      cublasDrot_rank_1
  end interface
  
  interface cublasCrot
    function cublasCrot_(handle,n,x,incx,y,incy,c,s) bind(c, &
&name="cublasCrot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

    module procedure &
      cublasCrot_rank_0,&
      cublasCrot_rank_1
  end interface
  
  interface cublasCsrot
    function cublasCsrot_(handle,n,x,incx,y,incy,c,s) bind(c, &
&name="cublasCsrot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsrot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

    module procedure &
      cublasCsrot_rank_0,&
      cublasCsrot_rank_1
  end interface
  
  interface cublasZrot
    function cublasZrot_(handle,n,x,incx,y,incy,c,s) bind(c, &
&name="cublasZrot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

    module procedure &
      cublasZrot_rank_0,&
      cublasZrot_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rot applies the Givens rotation matrix defined by c=cos(alpha) and s=sin(alpha) to vectors x and y.
  !>         Scalars c and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[in]
  !>     n       [int]
  !>             number of elements in the x and y vectors.
  !>     @param[inout]
  !>     x       device pointer storing vector x.
  !>     @param[in]
  !>     incx    [int]
  !>             specifies the increment between elements of x.
  !>     @param[inout]
  !>     y       device pointer storing vector y.
  !>     @param[in]
  !>     incy    [int]
  !>             specifies the increment between elements of y.
  !>     @param[in]
  !>     c       device pointer or host pointer storing scalar cosine component of the rotation matrix.
  !>     @param[in]
  !>     s       device pointer or host pointer storing scalar sine component of the rotation matrix.
  !>
  interface cublasZdrot
    function cublasZdrot_(handle,n,x,incx,y,incy,c,s) bind(c, &
&name="cublasZdrot")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdrot_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

    module procedure &
      cublasZdrot_rank_0,&
      cublasZdrot_rank_1
  end interface
  
  interface cublasSrotBatched
    function cublasSrotBatched_(handle,n,x,incx,y,incy,c,s,batchCount) &
&bind(c, name="cublasSrotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSrotBatched_full_rank,&
      cublasSrotBatched_rank_0,&
      cublasSrotBatched_rank_1
  end interface
  
  interface cublasDrotBatched
    function cublasDrotBatched_(handle,n,x,incx,y,incy,c,s,batchCount) &
&bind(c, name="cublasDrotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDrotBatched_full_rank,&
      cublasDrotBatched_rank_0,&
      cublasDrotBatched_rank_1
  end interface
  
  interface cublasCrotBatched
    function cublasCrotBatched_(handle,n,x,incx,y,incy,c,s,batchCount) &
&bind(c, name="cublasCrotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCrotBatched_full_rank,&
      cublasCrotBatched_rank_0,&
      cublasCrotBatched_rank_1
  end interface
  
  interface cublasCsrotBatched
    function cublasCsrotBatched_(handle,n,x,incx,y,incy,c,s,&
&batchCount) bind(c, name="cublasCsrotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsrotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsrotBatched_full_rank,&
      cublasCsrotBatched_rank_0,&
      cublasCsrotBatched_rank_1
  end interface
  
  interface cublasZrotBatched
    function cublasZrotBatched_(handle,n,x,incx,y,incy,c,s,batchCount) &
&bind(c, name="cublasZrotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZrotBatched_full_rank,&
      cublasZrotBatched_rank_0,&
      cublasZrotBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotBatched applies the Givens rotation matrix defined by c=cos(alpha) and s=sin(alpha) to batched vectors x_i and y_i, for i = 1, ..., batchCount.
  !>         Scalars c and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[in]
  !>     n       [int]
  !>             number of elements in each x_i and y_i vectors.
  !>     @param[inout]
  !>     x       device array of deivce pointers storing each vector x_i.
  !>     @param[in]
  !>     incx    [int]
  !>             specifies the increment between elements of each x_i.
  !>     @param[inout]
  !>     y       device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy    [int]
  !>             specifies the increment between elements of each y_i.
  !>     @param[in]
  !>     c       device pointer or host pointer to scalar cosine component of the rotation matrix.
  !>     @param[in]
  !>     s       device pointer or host pointer to scalar sine component of the rotation matrix.
  !>     @param[in]
  !>     batchCount [int]
  !>                 the number of x and y arrays, i.e. the number of batches.
  !>
  interface cublasZdrotBatched
    function cublasZdrotBatched_(handle,n,x,incx,y,incy,c,s,&
&batchCount) bind(c, name="cublasZdrotBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdrotBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZdrotBatched_full_rank,&
      cublasZdrotBatched_rank_0,&
      cublasZdrotBatched_rank_1
  end interface
  
  interface cublasSrotStridedBatched
    function cublasSrotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,c,s,batchCount) bind(c, name="cublasSrotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSrotStridedBatched_rank_0,&
      cublasSrotStridedBatched_rank_1
  end interface
  
  interface cublasDrotStridedBatched
    function cublasDrotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,c,s,batchCount) bind(c, name="cublasDrotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDrotStridedBatched_rank_0,&
      cublasDrotStridedBatched_rank_1
  end interface
  
  interface cublasCrotStridedBatched
    function cublasCrotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,c,s,batchCount) bind(c, name="cublasCrotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCrotStridedBatched_rank_0,&
      cublasCrotStridedBatched_rank_1
  end interface
  
  interface cublasCsrotStridedBatched
    function cublasCsrotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,c,s,batchCount) bind(c, name="cublasCsrotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsrotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsrotStridedBatched_rank_0,&
      cublasCsrotStridedBatched_rank_1
  end interface
  
  interface cublasZrotStridedBatched
    function cublasZrotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,c,s,batchCount) bind(c, name="cublasZrotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZrotStridedBatched_rank_0,&
      cublasZrotStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotStridedBatched applies the Givens rotation matrix defined by c=cos(alpha) and s=sin(alpha) to strided batched vectors x_i and y_i, for i = 1, ..., batchCount.
  !>         Scalars c and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[in]
  !>     n       [int]
  !>             number of elements in each x_i and y_i vectors.
  !>     @param[inout]
  !>     x       device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx    [int]
  !>             specifies the increment between elements of each x_i.
  !>     @param[in]
  !>     stride_x [hipblasStride]
  !>              specifies the increment from the beginning of x_i to the beginning of x_(i+1)
  !>     @param[inout]
  !>     y       device pointer to the first vector y_1.
  !>     @param[in]
  !>     incy    [int]
  !>             specifies the increment between elements of each y_i.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>              specifies the increment from the beginning of y_i to the beginning of y_(i+1)
  !>     @param[in]
  !>     c       device pointer or host pointer to scalar cosine component of the rotation matrix.
  !>     @param[in]
  !>     s       device pointer or host pointer to scalar sine component of the rotation matrix.
  !>     @param[in]
  !>     batchCount [int]
  !>             the number of x and y arrays, i.e. the number of batches.
  !>
  interface cublasZdrotStridedBatched
    function cublasZdrotStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,c,s,batchCount) bind(c, name="cublasZdrotStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdrotStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZdrotStridedBatched_rank_0,&
      cublasZdrotStridedBatched_rank_1
  end interface
  
  interface cublasSrotg
    function cublasSrotg_(handle,a,b,c,s) bind(c, name="cublasSrotg")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotg_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      type(c_ptr),value :: b
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

  end interface
  
  interface cublasDrotg
    function cublasDrotg_(handle,a,b,c,s) bind(c, name="cublasDrotg")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotg_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      type(c_ptr),value :: b
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

  end interface
  
  interface cublasCrotg
    function cublasCrotg_(handle,a,b,c,s) bind(c, name="cublasCrotg")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrotg_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      type(c_ptr),value :: b
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotg creates the Givens rotation matrix for the vector (a b).
  !>          Scalars c and s and arrays a and b may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>          If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
  !>          If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[inout]
  !>     a       device pointer or host pointer to input vector element, overwritten with r.
  !>     @param[inout]
  !>     b       device pointer or host pointer to input vector element, overwritten with z.
  !>     @param[inout]
  !>     c       device pointer or host pointer to cosine element of Givens rotation.
  !>     @param[inout]
  !>     s       device pointer or host pointer sine element of Givens rotation.
  !>
  interface cublasZrotg
    function cublasZrotg_(handle,a,b,c,s) bind(c, name="cublasZrotg")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrotg_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      type(c_ptr),value :: b
      type(c_ptr),value :: c
      type(c_ptr),value :: s
    end function

  end interface
  
  interface cublasSrotgBatched
    function cublasSrotgBatched_(handle,a,b,c,s,batchCount) bind(c, &
&name="cublasSrotgBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotgBatched_
      type(c_ptr),value :: handle
      type(c_ptr) :: a
      type(c_ptr) :: b
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasDrotgBatched
    function cublasDrotgBatched_(handle,a,b,c,s,batchCount) bind(c, &
&name="cublasDrotgBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotgBatched_
      type(c_ptr),value :: handle
      type(c_ptr) :: a
      type(c_ptr) :: b
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasCrotgBatched
    function cublasCrotgBatched_(handle,a,b,c,s,batchCount) bind(c, &
&name="cublasCrotgBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrotgBatched_
      type(c_ptr),value :: handle
      type(c_ptr) :: a
      type(c_ptr) :: b
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int),value :: batchCount
    end function

  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotgBatched creates the Givens rotation matrix for the batched vectors (a_i b_i), for i = 1, ..., batchCount.
  !>          a, b, c, and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>          If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
  !>          If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[inout]
  !>     a       device array of device pointers storing each single input vector element a_i, overwritten with r_i.
  !>     @param[inout]
  !>     b       device array of device pointers storing each single input vector element b_i, overwritten with z_i.
  !>     @param[inout]
  !>     c       device array of device pointers storing each cosine element of Givens rotation for the batch.
  !>     @param[inout]
  !>     s       device array of device pointers storing each sine element of Givens rotation for the batch.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of batches (length of arrays a, b, c, and s).
  !>
  interface cublasZrotgBatched
    function cublasZrotgBatched_(handle,a,b,c,s,batchCount) bind(c, &
&name="cublasZrotgBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrotgBatched_
      type(c_ptr),value :: handle
      type(c_ptr) :: a
      type(c_ptr) :: b
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasSrotgStridedBatched
    function cublasSrotgStridedBatched_(handle,a,stride_a,b,stride_b,c,&
&stride_c,s,stride_s,batchCount) bind(c, &
&name="cublasSrotgStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotgStridedBatched_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      integer(c_int64_t),value :: stride_a
      type(c_ptr),value :: b
      integer(c_int64_t),value :: stride_b
      type(c_ptr),value :: c
      integer(c_int64_t),value :: stride_c
      type(c_ptr),value :: s
      integer(c_int64_t),value :: stride_s
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasDrotgStridedBatched
    function cublasDrotgStridedBatched_(handle,a,stride_a,b,stride_b,c,&
&stride_c,s,stride_s,batchCount) bind(c, &
&name="cublasDrotgStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotgStridedBatched_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      integer(c_int64_t),value :: stride_a
      type(c_ptr),value :: b
      integer(c_int64_t),value :: stride_b
      type(c_ptr),value :: c
      integer(c_int64_t),value :: stride_c
      type(c_ptr),value :: s
      integer(c_int64_t),value :: stride_s
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasCrotgStridedBatched
    function cublasCrotgStridedBatched_(handle,a,stride_a,b,stride_b,c,&
&stride_c,s,stride_s,batchCount) bind(c, &
&name="cublasCrotgStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrotgStridedBatched_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      integer(c_int64_t),value :: stride_a
      type(c_ptr),value :: b
      integer(c_int64_t),value :: stride_b
      type(c_ptr),value :: c
      integer(c_int64_t),value :: stride_c
      type(c_ptr),value :: s
      integer(c_int64_t),value :: stride_s
      integer(c_int),value :: batchCount
    end function

  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotgStridedBatched creates the Givens rotation matrix for the strided batched vectors (a_i b_i), for i = 1, ..., batchCount.
  !>          a, b, c, and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>          If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
  !>          If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function returns immediately and synchronization is required to read the results.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[inout]
  !>     a       device strided_batched pointer or host strided_batched pointer to first single input vector element a_1, overwritten with r.
  !>     @param[in]
  !>     stride_a [hipblasStride]
  !>              distance between elements of a in batch (distance between a_i and a_(i + 1))
  !>     @param[inout]
  !>     b       device strided_batched pointer or host strided_batched pointer to first single input vector element b_1, overwritten with z.
  !>     @param[in]
  !>     stride_b [hipblasStride]
  !>              distance between elements of b in batch (distance between b_i and b_(i + 1))
  !>     @param[inout]
  !>     c       device strided_batched pointer or host strided_batched pointer to first cosine element of Givens rotations c_1.
  !>     @param[in]
  !>     stride_c [hipblasStride]
  !>              distance between elements of c in batch (distance between c_i and c_(i + 1))
  !>     @param[inout]
  !>     s       device strided_batched pointer or host strided_batched pointer to sine element of Givens rotations s_1.
  !>     @param[in]
  !>     stride_s [hipblasStride]
  !>              distance between elements of s in batch (distance between s_i and s_(i + 1))
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of batches (length of arrays a, b, c, and s).
  !>
  interface cublasZrotgStridedBatched
    function cublasZrotgStridedBatched_(handle,a,stride_a,b,stride_b,c,&
&stride_c,s,stride_s,batchCount) bind(c, &
&name="cublasZrotgStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrotgStridedBatched_
      type(c_ptr),value :: handle
      type(c_ptr),value :: a
      integer(c_int64_t),value :: stride_a
      type(c_ptr),value :: b
      integer(c_int64_t),value :: stride_b
      type(c_ptr),value :: c
      integer(c_int64_t),value :: stride_c
      type(c_ptr),value :: s
      integer(c_int64_t),value :: stride_s
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasSrotm
    function cublasSrotm_(handle,n,x,incx,y,incy,param) bind(c, &
&name="cublasSrotm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotm_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: param
    end function

    module procedure &
      cublasSrotm_rank_0,&
      cublasSrotm_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotm applies the modified Givens rotation matrix defined by param to vectors x and y.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[in]
  !>     n       [int]
  !>             number of elements in the x and y vectors.
  !>     @param[inout]
  !>     x       device pointer storing vector x.
  !>     @param[in]
  !>     incx    [int]
  !>             specifies the increment between elements of x.
  !>     @param[inout]
  !>     y       device pointer storing vector y.
  !>     @param[in]
  !>     incy    [int]
  !>             specifies the increment between elements of y.
  !>     @param[in]
  !>     param   device vector or host vector of 5 elements defining the rotation.
  !>             param[0] = flag
  !>             param[1] = H11
  !>             param[2] = H21
  !>             param[3] = H12
  !>             param[4] = H22
  !>             The flag parameter defines the form of H:
  !>             flag = -1 => H = ( H11 H12 H21 H22 )
  !>             flag =  0 => H = ( 1.0 H12 H21 1.0 )
  !>             flag =  1 => H = ( H11 1.0 -1.0 H22 )
  !>             flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
  !>             param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>
  interface cublasDrotm
    function cublasDrotm_(handle,n,x,incx,y,incy,param) bind(c, &
&name="cublasDrotm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotm_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: param
    end function

    module procedure &
      cublasDrotm_rank_0,&
      cublasDrotm_rank_1
  end interface
  
  interface cublasSrotmBatched
    function cublasSrotmBatched_(handle,n,x,incx,y,incy,param,&
&batchCount) bind(c, name="cublasSrotmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotmBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: param
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSrotmBatched_full_rank,&
      cublasSrotmBatched_rank_0,&
      cublasSrotmBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotmBatched applies the modified Givens rotation matrix defined by param_i to batched vectors x_i and y_i, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[in]
  !>     n       [int]
  !>             number of elements in the x and y vectors.
  !>     @param[inout]
  !>     x       device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx    [int]
  !>             specifies the increment between elements of each x_i.
  !>     @param[inout]
  !>     y       device array of device pointers storing each vector y_1.
  !>     @param[in]
  !>     incy    [int]
  !>             specifies the increment between elements of each y_i.
  !>     @param[in]
  !>     param   device array of device vectors of 5 elements defining the rotation.
  !>             param[0] = flag
  !>             param[1] = H11
  !>             param[2] = H21
  !>             param[3] = H12
  !>             param[4] = H22
  !>             The flag parameter defines the form of H:
  !>             flag = -1 => H = ( H11 H12 H21 H22 )
  !>             flag =  0 => H = ( 1.0 H12 H21 1.0 )
  !>             flag =  1 => H = ( H11 1.0 -1.0 H22 )
  !>             flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
  !>             param may ONLY be stored on the device for the batched version of this function.
  !>     @param[in]
  !>     batchCount [int]
  !>                 the number of x and y arrays, i.e. the number of batches.
  !>
  interface cublasDrotmBatched
    function cublasDrotmBatched_(handle,n,x,incx,y,incy,param,&
&batchCount) bind(c, name="cublasDrotmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotmBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: param
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDrotmBatched_full_rank,&
      cublasDrotmBatched_rank_0,&
      cublasDrotmBatched_rank_1
  end interface
  
  interface cublasSrotmStridedBatched
    function cublasSrotmStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,param,strideParam,batchCount) bind(c, &
&name="cublasSrotmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotmStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: param
      integer(c_int64_t),value :: strideParam
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSrotmStridedBatched_rank_0,&
      cublasSrotmStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotmStridedBatched applies the modified Givens rotation matrix defined by param_i to strided batched vectors x_i and y_i, for i = 1, ..., batchCount
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[in]
  !>     n       [int]
  !>             number of elements in the x and y vectors.
  !>     @param[inout]
  !>     x       device pointer pointing to first strided batched vector x_1.
  !>     @param[in]
  !>     incx    [int]
  !>             specifies the increment between elements of each x_i.
  !>     @param[in]
  !>     stride_x [hipblasStride]
  !>              specifies the increment between the beginning of x_i and x_(i + 1)
  !>     @param[inout]
  !>     y       device pointer pointing to first strided batched vector y_1.
  !>     @param[in]
  !>     incy    [int]
  !>             specifies the increment between elements of each y_i.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>              specifies the increment between the beginning of y_i and y_(i + 1)
  !>     @param[in]
  !>     param   device pointer pointing to first array of 5 elements defining the rotation (param_1).
  !>             param[0] = flag
  !>             param[1] = H11
  !>             param[2] = H21
  !>             param[3] = H12
  !>             param[4] = H22
  !>             The flag parameter defines the form of H:
  !>             flag = -1 => H = ( H11 H12 H21 H22 )
  !>             flag =  0 => H = ( 1.0 H12 H21 1.0 )
  !>             flag =  1 => H = ( H11 1.0 -1.0 H22 )
  !>             flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
  !>             param may ONLY be stored on the device for the strided_batched version of this function.
  !>     @param[in]
  !>     strideParam [hipblasStride]
  !>                  specifies the increment between the beginning of param_i and param_(i + 1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 the number of x and y arrays, i.e. the number of batches.
  !>
  interface cublasDrotmStridedBatched
    function cublasDrotmStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,param,strideParam,batchCount) bind(c, &
&name="cublasDrotmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotmStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: param
      integer(c_int64_t),value :: strideParam
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDrotmStridedBatched_rank_0,&
      cublasDrotmStridedBatched_rank_1
  end interface
  
  interface cublasSrotmg
    function cublasSrotmg_(handle,d1,d2,x1,y1,param) bind(c, &
&name="cublasSrotmg")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotmg_
      type(c_ptr),value :: handle
      type(c_ptr),value :: d1
      type(c_ptr),value :: d2
      type(c_ptr),value :: x1
      type(c_ptr),value :: y1
      type(c_ptr),value :: param
    end function

  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotmg creates the modified Givens rotation matrix for the vector (d1 * x1, d2 * y1).
  !>           Parameters may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>           If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
  !>           If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[inout]
  !>     d1      device pointer or host pointer to input scalar that is overwritten.
  !>     @param[inout]
  !>     d2      device pointer or host pointer to input scalar that is overwritten.
  !>     @param[inout]
  !>     x1      device pointer or host pointer to input scalar that is overwritten.
  !>     @param[in]
  !>     y1      device pointer or host pointer to input scalar.
  !>     @param[out]
  !>     param   device vector or host vector of 5 elements defining the rotation.
  !>             param[0] = flag
  !>             param[1] = H11
  !>             param[2] = H21
  !>             param[3] = H12
  !>             param[4] = H22
  !>             The flag parameter defines the form of H:
  !>             flag = -1 => H = ( H11 H12 H21 H22 )
  !>             flag =  0 => H = ( 1.0 H12 H21 1.0 )
  !>             flag =  1 => H = ( H11 1.0 -1.0 H22 )
  !>             flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
  !>             param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>
  interface cublasDrotmg
    function cublasDrotmg_(handle,d1,d2,x1,y1,param) bind(c, &
&name="cublasDrotmg")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotmg_
      type(c_ptr),value :: handle
      type(c_ptr),value :: d1
      type(c_ptr),value :: d2
      type(c_ptr),value :: x1
      type(c_ptr),value :: y1
      type(c_ptr),value :: param
    end function

  end interface
  
  interface cublasSrotmgBatched
    function cublasSrotmgBatched_(handle,d1,d2,x1,y1,param,batchCount) &
&bind(c, name="cublasSrotmgBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotmgBatched_
      type(c_ptr),value :: handle
      type(c_ptr) :: d1
      type(c_ptr) :: d2
      type(c_ptr) :: x1
      type(c_ptr) :: y1
      type(c_ptr) :: param
      integer(c_int),value :: batchCount
    end function

  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotmgBatched creates the modified Givens rotation matrix for the batched vectors (d1_i * x1_i, d2_i * y1_i), for i = 1, ..., batchCount.
  !>           Parameters may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>           If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
  !>           If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[inout]
  !>     d1      device batched array or host batched array of input scalars that is overwritten.
  !>     @param[inout]
  !>     d2      device batched array or host batched array of input scalars that is overwritten.
  !>     @param[inout]
  !>     x1      device batched array or host batched array of input scalars that is overwritten.
  !>     @param[in]
  !>     y1      device batched array or host batched array of input scalars.
  !>     @param[out]
  !>     param   device batched array or host batched array of vectors of 5 elements defining the rotation.
  !>             param[0] = flag
  !>             param[1] = H11
  !>             param[2] = H21
  !>             param[3] = H12
  !>             param[4] = H22
  !>             The flag parameter defines the form of H:
  !>             flag = -1 => H = ( H11 H12 H21 H22 )
  !>             flag =  0 => H = ( 1.0 H12 H21 1.0 )
  !>             flag =  1 => H = ( H11 1.0 -1.0 H22 )
  !>             flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
  !>             param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>     @param[in]
  !>     batchCount [int]
  !>                 the number of instances in the batch.
  !>
  interface cublasDrotmgBatched
    function cublasDrotmgBatched_(handle,d1,d2,x1,y1,param,batchCount) &
&bind(c, name="cublasDrotmgBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotmgBatched_
      type(c_ptr),value :: handle
      type(c_ptr) :: d1
      type(c_ptr) :: d2
      type(c_ptr) :: x1
      type(c_ptr) :: y1
      type(c_ptr) :: param
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasSrotmgStridedBatched
    function cublasSrotmgStridedBatched_(handle,d1,stride_d1,d2,&
&stride_d2,x1,stride_x1,y1,stride_y1,param,strideParam,batchCount) &
&bind(c, name="cublasSrotmgStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSrotmgStridedBatched_
      type(c_ptr),value :: handle
      type(c_ptr),value :: d1
      integer(c_int64_t),value :: stride_d1
      type(c_ptr),value :: d2
      integer(c_int64_t),value :: stride_d2
      type(c_ptr),value :: x1
      integer(c_int64_t),value :: stride_x1
      type(c_ptr),value :: y1
      integer(c_int64_t),value :: stride_y1
      type(c_ptr),value :: param
      integer(c_int64_t),value :: strideParam
      integer(c_int),value :: batchCount
    end function

  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     rotmgStridedBatched creates the modified Givens rotation matrix for the strided batched vectors (d1_i * x1_i, d2_i * y1_i), for i = 1, ..., batchCount.
  !>           Parameters may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>           If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
  !>           If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.
  !> 
  !>     @param[in]
  !>     handle  [hipblasHandle_t]
  !>             handle to the hipblas library context queue.
  !>     @param[inout]
  !>     d1      device strided_batched array or host strided_batched array of input scalars that is overwritten.
  !>     @param[in]
  !>     stride_d1 [hipblasStride]
  !>               specifies the increment between the beginning of d1_i and d1_(i+1)
  !>     @param[inout]
  !>     d2      device strided_batched array or host strided_batched array of input scalars that is overwritten.
  !>     @param[in]
  !>     stride_d2 [hipblasStride]
  !>               specifies the increment between the beginning of d2_i and d2_(i+1)
  !>     @param[inout]
  !>     x1      device strided_batched array or host strided_batched array of input scalars that is overwritten.
  !>     @param[in]
  !>     stride_x1 [hipblasStride]
  !>               specifies the increment between the beginning of x1_i and x1_(i+1)
  !>     @param[in]
  !>     y1      device strided_batched array or host strided_batched array of input scalars.
  !>     @param[in]
  !>     stride_y1 [hipblasStride]
  !>               specifies the increment between the beginning of y1_i and y1_(i+1)
  !>     @param[out]
  !>     param   device stridedBatched array or host stridedBatched array of vectors of 5 elements defining the rotation.
  !>             param[0] = flag
  !>             param[1] = H11
  !>             param[2] = H21
  !>             param[3] = H12
  !>             param[4] = H22
  !>             The flag parameter defines the form of H:
  !>             flag = -1 => H = ( H11 H12 H21 H22 )
  !>             flag =  0 => H = ( 1.0 H12 H21 1.0 )
  !>             flag =  1 => H = ( H11 1.0 -1.0 H22 )
  !>             flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
  !>             param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
  !>     @param[in]
  !>     strideParam [hipblasStride]
  !>                  specifies the increment between the beginning of param_i and param_(i + 1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 the number of instances in the batch.
  !>
  interface cublasDrotmgStridedBatched
    function cublasDrotmgStridedBatched_(handle,d1,stride_d1,d2,&
&stride_d2,x1,stride_x1,y1,stride_y1,param,strideParam,batchCount) &
&bind(c, name="cublasDrotmgStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDrotmgStridedBatched_
      type(c_ptr),value :: handle
      type(c_ptr),value :: d1
      integer(c_int64_t),value :: stride_d1
      type(c_ptr),value :: d2
      integer(c_int64_t),value :: stride_d2
      type(c_ptr),value :: x1
      integer(c_int64_t),value :: stride_x1
      type(c_ptr),value :: y1
      integer(c_int64_t),value :: stride_y1
      type(c_ptr),value :: param
      integer(c_int64_t),value :: strideParam
      integer(c_int),value :: batchCount
    end function

  end interface
  
  interface cublasSscal
    function cublasSscal_(handle,n,alpha,x,incx) bind(c, &
&name="cublasSscal")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSscal_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasSscal_rank_0,&
      cublasSscal_rank_1
  end interface
  
  interface cublasDscal
    function cublasDscal_(handle,n,alpha,x,incx) bind(c, &
&name="cublasDscal")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDscal_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasDscal_rank_0,&
      cublasDscal_rank_1
  end interface
  
  interface cublasCscal
    function cublasCscal_(handle,n,alpha,x,incx) bind(c, &
&name="cublasCscal")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCscal_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCscal_rank_0,&
      cublasCscal_rank_1
  end interface
  
  interface cublasCsscal
    function cublasCsscal_(handle,n,alpha,x,incx) bind(c, &
&name="cublasCsscal")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsscal_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCsscal_rank_0,&
      cublasCsscal_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     scal  scales each element of vector x with scalar alpha.
  !> 
  !>         x := alpha * x
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x.
  !>     @param[in]
  !>     alpha     device pointer or host pointer for the scalar alpha.
  !>     @param[inout]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !> 
  !>
  interface cublasZscal
    function cublasZscal_(handle,n,alpha,x,incx) bind(c, &
&name="cublasZscal")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZscal_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZscal_rank_0,&
      cublasZscal_rank_1
  end interface
  
  interface cublasZdscal
    function cublasZdscal_(handle,n,alpha,x,incx) bind(c, &
&name="cublasZdscal")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdscal_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZdscal_rank_0,&
      cublasZdscal_rank_1
  end interface
  
  interface cublasSscalBatched
    function cublasSscalBatched_(handle,n,alpha,x,incx,batchCount) &
&bind(c, name="cublasSscalBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSscalBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSscalBatched_full_rank,&
      cublasSscalBatched_rank_0,&
      cublasSscalBatched_rank_1
  end interface
  
  interface cublasDscalBatched
    function cublasDscalBatched_(handle,n,alpha,x,incx,batchCount) &
&bind(c, name="cublasDscalBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDscalBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDscalBatched_full_rank,&
      cublasDscalBatched_rank_0,&
      cublasDscalBatched_rank_1
  end interface
  
  interface cublasCscalBatched
    function cublasCscalBatched_(handle,n,alpha,x,incx,batchCount) &
&bind(c, name="cublasCscalBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCscalBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCscalBatched_full_rank,&
      cublasCscalBatched_rank_0,&
      cublasCscalBatched_rank_1
  end interface
  
  interface cublasZscalBatched
    function cublasZscalBatched_(handle,n,alpha,x,incx,batchCount) &
&bind(c, name="cublasZscalBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZscalBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZscalBatched_full_rank,&
      cublasZscalBatched_rank_0,&
      cublasZscalBatched_rank_1
  end interface
  
  interface cublasCsscalBatched
    function cublasCsscalBatched_(handle,n,alpha,x,incx,batchCount) &
&bind(c, name="cublasCsscalBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsscalBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsscalBatched_full_rank,&
      cublasCsscalBatched_rank_0,&
      cublasCsscalBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !>      \details
  !>     scalBatched  scales each element of vector x_i with scalar alpha, for i = 1, ... , batchCount.
  !> 
  !>          x_i := alpha * x_i
  !> 
  !>      where (x_i) is the i-th instance of the batch.
  !>     @param[in]
  !>     handle      [hipblasHandle_t]
  !>                 handle to the hipblas library context queue.
  !>     @param[in]
  !>     n           [int]
  !>                 the number of elements in each x_i.
  !>     @param[in]
  !>     alpha       host pointer or device pointer for the scalar alpha.
  !>     @param[inout]
  !>     x           device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx        [int]
  !>                 specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 specifies the number of batches in x.
  interface cublasZdscalBatched
    function cublasZdscalBatched_(handle,n,alpha,x,incx,batchCount) &
&bind(c, name="cublasZdscalBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdscalBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZdscalBatched_full_rank,&
      cublasZdscalBatched_rank_0,&
      cublasZdscalBatched_rank_1
  end interface
  
  interface cublasSscalStridedBatched
    function cublasSscalStridedBatched_(handle,n,alpha,x,incx,stridex,&
&batchCount) bind(c, name="cublasSscalStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSscalStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSscalStridedBatched_rank_0,&
      cublasSscalStridedBatched_rank_1
  end interface
  
  interface cublasDscalStridedBatched
    function cublasDscalStridedBatched_(handle,n,alpha,x,incx,stridex,&
&batchCount) bind(c, name="cublasDscalStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDscalStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDscalStridedBatched_rank_0,&
      cublasDscalStridedBatched_rank_1
  end interface
  
  interface cublasCscalStridedBatched
    function cublasCscalStridedBatched_(handle,n,alpha,x,incx,stridex,&
&batchCount) bind(c, name="cublasCscalStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCscalStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCscalStridedBatched_rank_0,&
      cublasCscalStridedBatched_rank_1
  end interface
  
  interface cublasZscalStridedBatched
    function cublasZscalStridedBatched_(handle,n,alpha,x,incx,stridex,&
&batchCount) bind(c, name="cublasZscalStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZscalStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZscalStridedBatched_rank_0,&
      cublasZscalStridedBatched_rank_1
  end interface
  
  interface cublasCsscalStridedBatched
    function cublasCsscalStridedBatched_(handle,n,alpha,x,incx,stridex,&
&batchCount) bind(c, name="cublasCsscalStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCsscalStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsscalStridedBatched_rank_0,&
      cublasCsscalStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !>      \details
  !>     scalStridedBatched  scales each element of vector x_i with scalar alpha, for i = 1, ... , batchCount.
  !> 
  !>          x_i := alpha * x_i ,
  !> 
  !>      where (x_i) is the i-th instance of the batch.
  !>      @param[in]
  !>     handle      [hipblasHandle_t]
  !>                 handle to the hipblas library context queue.
  !>     @param[in]
  !>     n           [int]
  !>                 the number of elements in each x_i.
  !>     @param[in]
  !>     alpha       host pointer or device pointer for the scalar alpha.
  !>     @param[inout]
  !>     x           device pointer to the first vector (x_1) in the batch.
  !>     @param[in]
  !>     incx        [int]
  !>                 specifies the increment for the elements of x.
  !>     @param[in]
  !>     stridex     [hipblasStride]
  !>                 stride from the start of one vector (x_i) and the next one (x_i+1).
  !>                 There are no restrictions placed on stride_x, however the user should
  !>                 take care to ensure that stride_x is of appropriate size, for a typical
  !>                 case this means stride_x >= n * incx.
  !>     @param[in]
  !>     batchCount [int]
  !>                 specifies the number of batches in x.
  interface cublasZdscalStridedBatched
    function cublasZdscalStridedBatched_(handle,n,alpha,x,incx,stridex,&
&batchCount) bind(c, name="cublasZdscalStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZdscalStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZdscalStridedBatched_rank_0,&
      cublasZdscalStridedBatched_rank_1
  end interface
  
  interface cublasSswap
    function cublasSswap_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasSswap")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSswap_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasSswap_rank_0,&
      cublasSswap_rank_1
  end interface
  
  interface cublasDswap
    function cublasDswap_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasDswap")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDswap_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDswap_rank_0,&
      cublasDswap_rank_1
  end interface
  
  interface cublasCswap
    function cublasCswap_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasCswap")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCswap_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasCswap_rank_0,&
      cublasCswap_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     swap  interchanges vectors x and y.
  !> 
  !>         y := x; x := y
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in x and y.
  !>     @param[inout]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[inout]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZswap
    function cublasZswap_(handle,n,x,incx,y,incy) bind(c, &
&name="cublasZswap")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZswap_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZswap_rank_0,&
      cublasZswap_rank_1
  end interface
  
  interface cublasSswapBatched
    function cublasSswapBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasSswapBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSswapBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSswapBatched_full_rank,&
      cublasSswapBatched_rank_0,&
      cublasSswapBatched_rank_1
  end interface
  
  interface cublasDswapBatched
    function cublasDswapBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasDswapBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDswapBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDswapBatched_full_rank,&
      cublasDswapBatched_rank_0,&
      cublasDswapBatched_rank_1
  end interface
  
  interface cublasCswapBatched
    function cublasCswapBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasCswapBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCswapBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCswapBatched_full_rank,&
      cublasCswapBatched_rank_0,&
      cublasCswapBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     swapBatched interchanges vectors x_i and y_i, for i = 1 , ... , batchCount
  !> 
  !>         y_i := x_i; x_i := y_i
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in each x_i and y_i.
  !>     @param[inout]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[inout]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZswapBatched
    function cublasZswapBatched_(handle,n,x,incx,y,incy,batchCount) &
&bind(c, name="cublasZswapBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZswapBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZswapBatched_full_rank,&
      cublasZswapBatched_rank_0,&
      cublasZswapBatched_rank_1
  end interface
  
  interface cublasSswapStridedBatched
    function cublasSswapStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasSswapStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSswapStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSswapStridedBatched_rank_0,&
      cublasSswapStridedBatched_rank_1
  end interface
  
  interface cublasDswapStridedBatched
    function cublasDswapStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasDswapStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDswapStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDswapStridedBatched_rank_0,&
      cublasDswapStridedBatched_rank_1
  end interface
  
  interface cublasCswapStridedBatched
    function cublasCswapStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasCswapStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCswapStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCswapStridedBatched_rank_0,&
      cublasCswapStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 1 API
  !> 
  !>     \details
  !>     swapStridedBatched interchanges vectors x_i and y_i, for i = 1 , ... , batchCount
  !> 
  !>         y_i := x_i; x_i := y_i
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     n         [int]
  !>               the number of elements in each x_i and y_i.
  !>     @param[inout]
  !>     x         device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>               There are no restrictions placed on stride_x, however the user should
  !>               take care to ensure that stride_x is of appropriate size, for a typical
  !>               case this means stride_x >= n * incx.
  !>     @param[inout]
  !>     y         device pointer to the first vector y_1.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     stridey   [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (y_i+1).
  !>               There are no restrictions placed on stride_x, however the user should
  !>               take care to ensure that stride_y is of appropriate size, for a typical
  !>               case this means stride_y >= n * incy. stridey should be non zero.
  !>      @param[in]
  !>      batchCount [int]
  !>                  number of instances in the batch.
  !>
  interface cublasZswapStridedBatched
    function cublasZswapStridedBatched_(handle,n,x,incx,stridex,y,incy,&
&stridey,batchCount) bind(c, name="cublasZswapStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZswapStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZswapStridedBatched_rank_0,&
      cublasZswapStridedBatched_rank_1
  end interface
  
  interface cublasSgbmv
    function cublasSgbmv_(handle,trans,m,n,kl,ku,alpha,A,lda,x,incx,&
&beta,y,incy) bind(c, name="cublasSgbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasSgbmv_full_rank,&
      cublasSgbmv_rank_0,&
      cublasSgbmv_rank_1
  end interface
  
  interface cublasDgbmv
    function cublasDgbmv_(handle,trans,m,n,kl,ku,alpha,A,lda,x,incx,&
&beta,y,incy) bind(c, name="cublasDgbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDgbmv_full_rank,&
      cublasDgbmv_rank_0,&
      cublasDgbmv_rank_1
  end interface
  
  interface cublasCgbmv
    function cublasCgbmv_(handle,trans,m,n,kl,ku,alpha,A,lda,x,incx,&
&beta,y,incy) bind(c, name="cublasCgbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasCgbmv_full_rank,&
      cublasCgbmv_rank_0,&
      cublasCgbmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gbmv performs one of the matrix-vector operations
  !> 
  !>         y := alpha*A*x    + beta*y,   or
  !>         y := alpha*A**T*x + beta*y,   or
  !>         y := alpha*A**H*x + beta*y,
  !> 
  !>     where alpha and beta are scalars, x and y are vectors and A is an
  !>     m by n banded matrix with kl sub-diagonals and ku super-diagonals.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     trans     [hipblasOperation_t]
  !>               indicates whether matrix A is tranposed (conjugated) or not
  !>     @param[in]
  !>     m         [int]
  !>               number of rows of matrix A
  !>     @param[in]
  !>     n         [int]
  !>               number of columns of matrix A
  !>     @param[in]
  !>     kl        [int]
  !>               number of sub-diagonals of A
  !>     @param[in]
  !>     ku        [int]
  !>               number of super-diagonals of A
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>         A     device pointer storing banded matrix A.
  !>               Leading (kl + ku + 1) by n part of the matrix contains the coefficients
  !>               of the banded matrix. The leading diagonal resides in row (ku + 1) with
  !>               the first super-diagonal above on the RHS of row ku. The first sub-diagonal
  !>               resides below on the LHS of row ku + 2. This propogates up and down across
  !>               sub/super-diagonals.
  !>                 Ex: (m = n = 7; ku = 2, kl = 2)
  !>                 1 2 3 0 0 0 0             0 0 3 3 3 3 3
  !>                 4 1 2 3 0 0 0             0 2 2 2 2 2 2
  !>                 5 4 1 2 3 0 0    ---->    1 1 1 1 1 1 1
  !>                 0 5 4 1 2 3 0             4 4 4 4 4 4 0
  !>                 0 0 5 4 1 2 0             5 5 5 5 5 0 0
  !>                 0 0 0 5 4 1 2             0 0 0 0 0 0 0
  !>                 0 0 0 0 5 4 1             0 0 0 0 0 0 0
  !>               Note that the empty elements which don't correspond to data will not
  !>               be referenced.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A. Must be >= (kl + ku + 1)
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZgbmv
    function cublasZgbmv_(handle,trans,m,n,kl,ku,alpha,A,lda,x,incx,&
&beta,y,incy) bind(c, name="cublasZgbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZgbmv_full_rank,&
      cublasZgbmv_rank_0,&
      cublasZgbmv_rank_1
  end interface
  
  interface cublasSgbmvBatched
    function cublasSgbmvBatched_(handle,trans,m,n,kl,ku,alpha,A,lda,x,&
&incx,beta,y,incy,batchCount) bind(c, name="cublasSgbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgbmvBatched_full_rank,&
      cublasSgbmvBatched_rank_0,&
      cublasSgbmvBatched_rank_1
  end interface
  
  interface cublasDgbmvBatched
    function cublasDgbmvBatched_(handle,trans,m,n,kl,ku,alpha,A,lda,x,&
&incx,beta,y,incy,batchCount) bind(c, name="cublasDgbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgbmvBatched_full_rank,&
      cublasDgbmvBatched_rank_0,&
      cublasDgbmvBatched_rank_1
  end interface
  
  interface cublasCgbmvBatched
    function cublasCgbmvBatched_(handle,trans,m,n,kl,ku,alpha,A,lda,x,&
&incx,beta,y,incy,batchCount) bind(c, name="cublasCgbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgbmvBatched_full_rank,&
      cublasCgbmvBatched_rank_0,&
      cublasCgbmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gbmvBatched performs one of the matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i    + beta*y_i,   or
  !>         y_i := alpha*A_i**T*x_i + beta*y_i,   or
  !>         y_i := alpha*A_i**H*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     m by n banded matrix with kl sub-diagonals and ku super-diagonals,
  !>     for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     trans     [hipblasOperation_t]
  !>               indicates whether matrix A is tranposed (conjugated) or not
  !>     @param[in]
  !>     m         [int]
  !>               number of rows of each matrix A_i
  !>     @param[in]
  !>     n         [int]
  !>               number of columns of each matrix A_i
  !>     @param[in]
  !>     kl        [int]
  !>               number of sub-diagonals of each A_i
  !>     @param[in]
  !>     ku        [int]
  !>               number of super-diagonals of each A_i
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>         A     device array of device pointers storing each banded matrix A_i.
  !>               Leading (kl + ku + 1) by n part of the matrix contains the coefficients
  !>               of the banded matrix. The leading diagonal resides in row (ku + 1) with
  !>               the first super-diagonal above on the RHS of row ku. The first sub-diagonal
  !>               resides below on the LHS of row ku + 2. This propogates up and down across
  !>               sub/super-diagonals.
  !>                 Ex: (m = n = 7; ku = 2, kl = 2)
  !>                 1 2 3 0 0 0 0             0 0 3 3 3 3 3
  !>                 4 1 2 3 0 0 0             0 2 2 2 2 2 2
  !>                 5 4 1 2 3 0 0    ---->    1 1 1 1 1 1 1
  !>                 0 5 4 1 2 3 0             4 4 4 4 4 4 0
  !>                 0 0 5 4 1 2 0             5 5 5 5 5 0 0
  !>                 0 0 0 5 4 1 2             0 0 0 0 0 0 0
  !>                 0 0 0 0 5 4 1             0 0 0 0 0 0 0
  !>               Note that the empty elements which don't correspond to data will not
  !>               be referenced.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. Must be >= (kl + ku + 1)
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 specifies the number of instances in the batch.
  !>
  interface cublasZgbmvBatched
    function cublasZgbmvBatched_(handle,trans,m,n,kl,ku,alpha,A,lda,x,&
&incx,beta,y,incy,batchCount) bind(c, name="cublasZgbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgbmvBatched_full_rank,&
      cublasZgbmvBatched_rank_0,&
      cublasZgbmvBatched_rank_1
  end interface
  
  interface cublasSgbmvStridedBatched
    function cublasSgbmvStridedBatched_(handle,trans,m,n,kl,ku,alpha,A,&
&lda,strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasSgbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgbmvStridedBatched_full_rank,&
      cublasSgbmvStridedBatched_rank_0,&
      cublasSgbmvStridedBatched_rank_1
  end interface
  
  interface cublasDgbmvStridedBatched
    function cublasDgbmvStridedBatched_(handle,trans,m,n,kl,ku,alpha,A,&
&lda,strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasDgbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgbmvStridedBatched_full_rank,&
      cublasDgbmvStridedBatched_rank_0,&
      cublasDgbmvStridedBatched_rank_1
  end interface
  
  interface cublasCgbmvStridedBatched
    function cublasCgbmvStridedBatched_(handle,trans,m,n,kl,ku,alpha,A,&
&lda,strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasCgbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgbmvStridedBatched_full_rank,&
      cublasCgbmvStridedBatched_rank_0,&
      cublasCgbmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gbmvStridedBatched performs one of the matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i    + beta*y_i,   or
  !>         y_i := alpha*A_i**T*x_i + beta*y_i,   or
  !>         y_i := alpha*A_i**H*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     m by n banded matrix with kl sub-diagonals and ku super-diagonals,
  !>     for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     trans     [hipblasOperation_t]
  !>               indicates whether matrix A is tranposed (conjugated) or not
  !>     @param[in]
  !>     m         [int]
  !>               number of rows of matrix A
  !>     @param[in]
  !>     n         [int]
  !>               number of columns of matrix A
  !>     @param[in]
  !>     kl        [int]
  !>               number of sub-diagonals of A
  !>     @param[in]
  !>     ku        [int]
  !>               number of super-diagonals of A
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>         A     device pointer to first banded matrix (A_1).
  !>               Leading (kl + ku + 1) by n part of the matrix contains the coefficients
  !>               of the banded matrix. The leading diagonal resides in row (ku + 1) with
  !>               the first super-diagonal above on the RHS of row ku. The first sub-diagonal
  !>               resides below on the LHS of row ku + 2. This propogates up and down across
  !>               sub/super-diagonals.
  !>                 Ex: (m = n = 7; ku = 2, kl = 2)
  !>                 1 2 3 0 0 0 0             0 0 3 3 3 3 3
  !>                 4 1 2 3 0 0 0             0 2 2 2 2 2 2
  !>                 5 4 1 2 3 0 0    ---->    1 1 1 1 1 1 1
  !>                 0 5 4 1 2 3 0             4 4 4 4 4 4 0
  !>                 0 0 5 4 1 2 0             5 5 5 5 5 0 0
  !>                 0 0 0 5 4 1 2             0 0 0 0 0 0 0
  !>                 0 0 0 0 5 4 1             0 0 0 0 0 0 0
  !>               Note that the empty elements which don't correspond to data will not
  !>               be referenced.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A. Must be >= (kl + ku + 1)
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     x         device pointer to first vector (x_1).
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1)
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device pointer to first vector (y_1).
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (x_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 specifies the number of instances in the batch.
  !>
  interface cublasZgbmvStridedBatched
    function cublasZgbmvStridedBatched_(handle,trans,m,n,kl,ku,alpha,A,&
&lda,strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasZgbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: kl
      integer(c_int),value :: ku
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgbmvStridedBatched_full_rank,&
      cublasZgbmvStridedBatched_rank_0,&
      cublasZgbmvStridedBatched_rank_1
  end interface
  
  interface cublasSgemv
    function cublasSgemv_(handle,trans,m,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasSgemv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgemv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasSgemv_full_rank,&
      cublasSgemv_rank_0,&
      cublasSgemv_rank_1
  end interface
  
  interface cublasDgemv
    function cublasDgemv_(handle,trans,m,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasDgemv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgemv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDgemv_full_rank,&
      cublasDgemv_rank_0,&
      cublasDgemv_rank_1
  end interface
  
  interface cublasCgemv
    function cublasCgemv_(handle,trans,m,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasCgemv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgemv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasCgemv_full_rank,&
      cublasCgemv_rank_0,&
      cublasCgemv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gemv performs one of the matrix-vector operations
  !> 
  !>         y := alpha*A*x    + beta*y,   or
  !>         y := alpha*A**T*x + beta*y,   or
  !>         y := alpha*A**H*x + beta*y,
  !> 
  !>     where alpha and beta are scalars, x and y are vectors and A is an
  !>     m by n matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     trans     [hipblasOperation_t]
  !>               indicates whether matrix A is tranposed (conjugated) or not
  !>     @param[in]
  !>     m         [int]
  !>               number of rows of matrix A
  !>     @param[in]
  !>     n         [int]
  !>               number of columns of matrix A
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZgemv
    function cublasZgemv_(handle,trans,m,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasZgemv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgemv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZgemv_full_rank,&
      cublasZgemv_rank_0,&
      cublasZgemv_rank_1
  end interface
  
  interface cublasSgemvBatched
    function cublasSgemvBatched_(handle,trans,m,n,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasSgemvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgemvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgemvBatched_full_rank,&
      cublasSgemvBatched_rank_0,&
      cublasSgemvBatched_rank_1
  end interface
  
  interface cublasDgemvBatched
    function cublasDgemvBatched_(handle,trans,m,n,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasDgemvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgemvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgemvBatched_full_rank,&
      cublasDgemvBatched_rank_0,&
      cublasDgemvBatched_rank_1
  end interface
  
  interface cublasCgemvBatched
    function cublasCgemvBatched_(handle,trans,m,n,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasCgemvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgemvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgemvBatched_full_rank,&
      cublasCgemvBatched_rank_0,&
      cublasCgemvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gemvBatched performs a batch of matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i    + beta*y_i,   or
  !>         y_i := alpha*A_i**T*x_i + beta*y_i,   or
  !>         y_i := alpha*A_i**H*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     m by n matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle      [hipblasHandle_t]
  !>                 handle to the hipblas library context queue.
  !>     @param[in]
  !>     trans       [hipblasOperation_t]
  !>                 indicates whether matrices A_i are tranposed (conjugated) or not
  !>     @param[in]
  !>     m           [int]
  !>                 number of rows of each matrix A_i
  !>     @param[in]
  !>     n           [int]
  !>                 number of columns of each matrix A_i
  !>     @param[in]
  !>     alpha       device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A           device array of device pointers storing each matrix A_i.
  !>     @param[in]
  !>     lda         [int]
  !>                 specifies the leading dimension of each matrix A_i.
  !>     @param[in]
  !>     x           device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx        [int]
  !>                 specifies the increment for the elements of each vector x_i.
  !>     @param[in]
  !>     beta        device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y           device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy        [int]
  !>                 specifies the increment for the elements of each vector y_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZgemvBatched
    function cublasZgemvBatched_(handle,trans,m,n,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasZgemvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgemvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgemvBatched_full_rank,&
      cublasZgemvBatched_rank_0,&
      cublasZgemvBatched_rank_1
  end interface
  
  interface cublasSgemvStridedBatched
    function cublasSgemvStridedBatched_(handle,trans,m,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasSgemvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgemvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgemvStridedBatched_full_rank,&
      cublasSgemvStridedBatched_rank_0,&
      cublasSgemvStridedBatched_rank_1
  end interface
  
  interface cublasDgemvStridedBatched
    function cublasDgemvStridedBatched_(handle,trans,m,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasDgemvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgemvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgemvStridedBatched_full_rank,&
      cublasDgemvStridedBatched_rank_0,&
      cublasDgemvStridedBatched_rank_1
  end interface
  
  interface cublasCgemvStridedBatched
    function cublasCgemvStridedBatched_(handle,trans,m,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasCgemvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgemvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgemvStridedBatched_full_rank,&
      cublasCgemvStridedBatched_rank_0,&
      cublasCgemvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gemvStridedBatched performs a batch of matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i    + beta*y_i,   or
  !>         y_i := alpha*A_i**T*x_i + beta*y_i,   or
  !>         y_i := alpha*A_i**H*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     m by n matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle      [hipblasHandle_t]
  !>                 handle to the hipblas library context queue.
  !>     @param[in]
  !>     transA      [hipblasOperation_t]
  !>                 indicates whether matrices A_i are tranposed (conjugated) or not
  !>     @param[in]
  !>     m           [int]
  !>                 number of rows of matrices A_i
  !>     @param[in]
  !>     n           [int]
  !>                 number of columns of matrices A_i
  !>     @param[in]
  !>     alpha       device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A           device pointer to the first matrix (A_1) in the batch.
  !>     @param[in]
  !>     lda         [int]
  !>                 specifies the leading dimension of matrices A_i.
  !>     @param[in]
  !>     strideA     [hipblasStride]
  !>                 stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     x           device pointer to the first vector (x_1) in the batch.
  !>     @param[in]
  !>     incx        [int]
  !>                 specifies the increment for the elements of vectors x_i.
  !>     @param[in]
  !>     stridex     [hipblasStride]
  !>                 stride from the start of one vector (x_i) and the next one (x_i+1).
  !>                 There are no restrictions placed on stridex, however the user should
  !>                 take care to ensure that stridex is of appropriate size. When trans equals HIPBLAS_OP_N
  !>                 this typically means stridex >= n * incx, otherwise stridex >= m * incx.
  !>     @param[in]
  !>     beta        device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y           device pointer to the first vector (y_1) in the batch.
  !>     @param[in]
  !>     incy        [int]
  !>                 specifies the increment for the elements of vectors y_i.
  !>     @param[in]
  !>     stridey     [hipblasStride]
  !>                 stride from the start of one vector (y_i) and the next one (y_i+1).
  !>                 There are no restrictions placed on stridey, however the user should
  !>                 take care to ensure that stridey is of appropriate size. When trans equals HIPBLAS_OP_N
  !>                 this typically means stridey >= m * incy, otherwise stridey >= n * incy. stridey should be non zero.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZgemvStridedBatched
    function cublasZgemvStridedBatched_(handle,trans,m,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasZgemvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgemvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgemvStridedBatched_full_rank,&
      cublasZgemvStridedBatched_rank_0,&
      cublasZgemvStridedBatched_rank_1
  end interface
  
  interface cublasSger
    function cublasSger_(handle,m,n,alpha,x,incx,y,incy,A,lda) bind(c, &
&name="cublasSger")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSger_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasSger_full_rank,&
      cublasSger_rank_0,&
      cublasSger_rank_1
  end interface
  
  interface cublasDger
    function cublasDger_(handle,m,n,alpha,x,incx,y,incy,A,lda) bind(c, &
&name="cublasDger")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDger_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasDger_full_rank,&
      cublasDger_rank_0,&
      cublasDger_rank_1
  end interface
  
  interface cublasCgeru
    function cublasCgeru_(handle,m,n,alpha,x,incx,y,incy,A,lda) bind(c,&
& name="cublasCgeru")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeru_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasCgeru_full_rank,&
      cublasCgeru_rank_0,&
      cublasCgeru_rank_1
  end interface
  
  interface cublasCgerc
    function cublasCgerc_(handle,m,n,alpha,x,incx,y,incy,A,lda) bind(c,&
& name="cublasCgerc")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgerc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasCgerc_full_rank,&
      cublasCgerc_rank_0,&
      cublasCgerc_rank_1
  end interface
  
  interface cublasZgeru
    function cublasZgeru_(handle,m,n,alpha,x,incx,y,incy,A,lda) bind(c,&
& name="cublasZgeru")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeru_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasZgeru_full_rank,&
      cublasZgeru_rank_0,&
      cublasZgeru_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     ger,geru,gerc performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*y**T , OR
  !>         A := A + alpha*x*y**H for gerc
  !> 
  !>     where alpha is a scalar, x and y are vectors, and A is an
  !>     m by n matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     m         [int]
  !>               the number of rows of the matrix A.
  !>     @param[in]
  !>     n         [int]
  !>               the number of columns of the matrix A.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>
  interface cublasZgerc
    function cublasZgerc_(handle,m,n,alpha,x,incx,y,incy,A,lda) bind(c,&
& name="cublasZgerc")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgerc_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasZgerc_full_rank,&
      cublasZgerc_rank_0,&
      cublasZgerc_rank_1
  end interface
  
  interface cublasSgerBatched
    function cublasSgerBatched_(handle,m,n,alpha,x,incx,y,incy,A,lda,&
&batchCount) bind(c, name="cublasSgerBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgerBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgerBatched_full_rank,&
      cublasSgerBatched_rank_0,&
      cublasSgerBatched_rank_1
  end interface
  
  interface cublasDgerBatched
    function cublasDgerBatched_(handle,m,n,alpha,x,incx,y,incy,A,lda,&
&batchCount) bind(c, name="cublasDgerBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgerBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgerBatched_full_rank,&
      cublasDgerBatched_rank_0,&
      cublasDgerBatched_rank_1
  end interface
  
  interface cublasCgeruBatched
    function cublasCgeruBatched_(handle,m,n,alpha,x,incx,y,incy,A,lda,&
&batchCount) bind(c, name="cublasCgeruBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeruBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgeruBatched_full_rank,&
      cublasCgeruBatched_rank_0,&
      cublasCgeruBatched_rank_1
  end interface
  
  interface cublasCgercBatched
    function cublasCgercBatched_(handle,m,n,alpha,x,incx,y,incy,A,lda,&
&batchCount) bind(c, name="cublasCgercBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgercBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgercBatched_full_rank,&
      cublasCgercBatched_rank_0,&
      cublasCgercBatched_rank_1
  end interface
  
  interface cublasZgeruBatched
    function cublasZgeruBatched_(handle,m,n,alpha,x,incx,y,incy,A,lda,&
&batchCount) bind(c, name="cublasZgeruBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeruBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgeruBatched_full_rank,&
      cublasZgeruBatched_rank_0,&
      cublasZgeruBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gerBatched,geruBatched,gercBatched performs a batch of the matrix-vector operations
  !> 
  !>         A := A + alpha*x*y**T , OR
  !>         A := A + alpha*x*y**H for gerc
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha is a scalar, x_i and y_i are vectors and A_i is an
  !>     m by n matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     m         [int]
  !>               the number of rows of each matrix A_i.
  !>     @param[in]
  !>     n         [int]
  !>               the number of columns of eaceh matrix A_i.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i.
  !>     @param[in]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i.
  !>     @param[inout]
  !>     A         device array of device pointers storing each matrix A_i.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZgercBatched
    function cublasZgercBatched_(handle,m,n,alpha,x,incx,y,incy,A,lda,&
&batchCount) bind(c, name="cublasZgercBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgercBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgercBatched_full_rank,&
      cublasZgercBatched_rank_0,&
      cublasZgercBatched_rank_1
  end interface
  
  interface cublasSgerStridedBatched
    function cublasSgerStridedBatched_(handle,m,n,alpha,x,incx,stridex,&
&y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasSgerStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgerStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgerStridedBatched_full_rank,&
      cublasSgerStridedBatched_rank_0,&
      cublasSgerStridedBatched_rank_1
  end interface
  
  interface cublasDgerStridedBatched
    function cublasDgerStridedBatched_(handle,m,n,alpha,x,incx,stridex,&
&y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasDgerStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgerStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgerStridedBatched_full_rank,&
      cublasDgerStridedBatched_rank_0,&
      cublasDgerStridedBatched_rank_1
  end interface
  
  interface cublasCgeruStridedBatched
    function cublasCgeruStridedBatched_(handle,m,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasCgeruStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeruStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgeruStridedBatched_full_rank,&
      cublasCgeruStridedBatched_rank_0,&
      cublasCgeruStridedBatched_rank_1
  end interface
  
  interface cublasCgercStridedBatched
    function cublasCgercStridedBatched_(handle,m,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasCgercStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgercStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgercStridedBatched_full_rank,&
      cublasCgercStridedBatched_rank_0,&
      cublasCgercStridedBatched_rank_1
  end interface
  
  interface cublasZgeruStridedBatched
    function cublasZgeruStridedBatched_(handle,m,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasZgeruStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeruStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgeruStridedBatched_full_rank,&
      cublasZgeruStridedBatched_rank_0,&
      cublasZgeruStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     gerStridedBatched,geruStridedBatched,gercStridedBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*y_i**T, OR
  !>         A_i := A_i + alpha*x_i*y_i**H  for gerc
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha is a scalar, x_i and y_i are vectors and A_i is an
  !>     m by n matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     m         [int]
  !>               the number of rows of each matrix A_i.
  !>     @param[in]
  !>     n         [int]
  !>               the number of columns of each matrix A_i.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer to the first vector (x_1) in the batch.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increments for the elements of each vector x_i.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>               There are no restrictions placed on stridex, however the user should
  !>               take care to ensure that stridex is of appropriate size, for a typical
  !>               case this means stridex >= m * incx.
  !>     @param[inout]
  !>     y         device pointer to the first vector (y_1) in the batch.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i.
  !>     @param[in]
  !>     stridey   [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (y_i+1).
  !>               There are no restrictions placed on stridey, however the user should
  !>               take care to ensure that stridey is of appropriate size, for a typical
  !>               case this means stridey >= n * incy.
  !>     @param[inout]
  !>     A         device pointer to the first matrix (A_1) in the batch.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     strideA     [hipblasStride]
  !>                 stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZgercStridedBatched
    function cublasZgercStridedBatched_(handle,m,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasZgercStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgercStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgercStridedBatched_full_rank,&
      cublasZgercStridedBatched_rank_0,&
      cublasZgercStridedBatched_rank_1
  end interface
  
  interface cublasChbmv
    function cublasChbmv_(handle,uplo,n,k,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasChbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasChbmv_full_rank,&
      cublasChbmv_rank_0,&
      cublasChbmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hbmv performs the matrix-vector operations
  !> 
  !>         y := alpha*A*x + beta*y
  !> 
  !>     where alpha and beta are scalars, x and y are n element vectors and A is an
  !>     n by n Hermitian band matrix, with k super-diagonals.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is being supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is being supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the order of the matrix A.
  !>     @param[in]
  !>     k         [int]
  !>               the number of super-diagonals of the matrix A. Must be >= 0.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A         device pointer storing matrix A. Of dimension (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The leading (k + 1) by n part of A must contain the upper
  !>                 triangular band part of the Hermitian matrix, with the leading
  !>                 diagonal in row (k + 1), the first super-diagonal on the RHS
  !>                 of row k, etc.
  !>                 The top left k by x triangle of A will not be referenced.
  !>                     Ex (upper, lda = n = 4, k = 1):
  !>                     A                             Represented matrix
  !>                     (0,0) (5,9) (6,8) (7,7)       (1, 0) (5, 9) (0, 0) (0, 0)
  !>                     (1,0) (2,0) (3,0) (4,0)       (5,-9) (2, 0) (6, 8) (0, 0)
  !>                     (0,0) (0,0) (0,0) (0,0)       (0, 0) (6,-8) (3, 0) (7, 7)
  !>                     (0,0) (0,0) (0,0) (0,0)       (0, 0) (0, 0) (7,-7) (4, 0)
  !> 
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The leading (k + 1) by n part of A must contain the lower
  !>                 triangular band part of the Hermitian matrix, with the leading
  !>                 diagonal in row (1), the first sub-diagonal on the LHS of
  !>                 row 2, etc.
  !>                 The bottom right k by k triangle of A will not be referenced.
  !>                     Ex (lower, lda = 2, n = 4, k = 1):
  !>                     A                               Represented matrix
  !>                     (1,0) (2,0) (3,0) (4,0)         (1, 0) (5,-9) (0, 0) (0, 0)
  !>                     (5,9) (6,8) (7,7) (0,0)         (5, 9) (2, 0) (6,-8) (0, 0)
  !>                                                     (0, 0) (6, 8) (3, 0) (7,-7)
  !>                                                     (0, 0) (0, 0) (7, 7) (4, 0)
  !> 
  !>               As a Hermitian matrix, the imaginary part of the main diagonal
  !>               of A will not be referenced and is assumed to be == 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A. must be >= k + 1
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZhbmv
    function cublasZhbmv_(handle,uplo,n,k,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasZhbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZhbmv_full_rank,&
      cublasZhbmv_rank_0,&
      cublasZhbmv_rank_1
  end interface
  
  interface cublasChbmvBatched
    function cublasChbmvBatched_(handle,uplo,n,k,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasChbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChbmvBatched_full_rank,&
      cublasChbmvBatched_rank_0,&
      cublasChbmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hbmvBatched performs one of the matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i
  !> 
  !>     where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
  !>     n by n Hermitian band matrix with k super-diagonals, for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is being supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is being supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the order of each matrix A_i.
  !>     @param[in]
  !>     k         [int]
  !>               the number of super-diagonals of each matrix A_i. Must be >= 0.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix_i A of dimension (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The leading (k + 1) by n part of each A_i must contain the upper
  !>                 triangular band part of the Hermitian matrix, with the leading
  !>                 diagonal in row (k + 1), the first super-diagonal on the RHS
  !>                 of row k, etc.
  !>                 The top left k by x triangle of each A_i will not be referenced.
  !>                     Ex (upper, lda = n = 4, k = 1):
  !>                     A                             Represented matrix
  !>                     (0,0) (5,9) (6,8) (7,7)       (1, 0) (5, 9) (0, 0) (0, 0)
  !>                     (1,0) (2,0) (3,0) (4,0)       (5,-9) (2, 0) (6, 8) (0, 0)
  !>                     (0,0) (0,0) (0,0) (0,0)       (0, 0) (6,-8) (3, 0) (7, 7)
  !>                     (0,0) (0,0) (0,0) (0,0)       (0, 0) (0, 0) (7,-7) (4, 0)
  !> 
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The leading (k + 1) by n part of each A_i must contain the lower
  !>                 triangular band part of the Hermitian matrix, with the leading
  !>                 diagonal in row (1), the first sub-diagonal on the LHS of
  !>                 row 2, etc.
  !>                 The bottom right k by k triangle of each A_i will not be referenced.
  !>                     Ex (lower, lda = 2, n = 4, k = 1):
  !>                     A                               Represented matrix
  !>                     (1,0) (2,0) (3,0) (4,0)         (1, 0) (5,-9) (0, 0) (0, 0)
  !>                     (5,9) (6,8) (7,7) (0,0)         (5, 9) (2, 0) (6,-8) (0, 0)
  !>                                                     (0, 0) (6, 8) (3, 0) (7,-7)
  !>                                                     (0, 0) (0, 0) (7, 7) (4, 0)
  !> 
  !>               As a Hermitian matrix, the imaginary part of the main diagonal
  !>               of each A_i will not be referenced and is assumed to be == 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. must be >= max(1, n)
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZhbmvBatched
    function cublasZhbmvBatched_(handle,uplo,n,k,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasZhbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhbmvBatched_full_rank,&
      cublasZhbmvBatched_rank_0,&
      cublasZhbmvBatched_rank_1
  end interface
  
  interface cublasChbmvStridedBatched
    function cublasChbmvStridedBatched_(handle,uplo,n,k,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasChbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChbmvStridedBatched_full_rank,&
      cublasChbmvStridedBatched_rank_0,&
      cublasChbmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hbmvStridedBatched performs one of the matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i
  !> 
  !>     where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
  !>     n by n Hermitian band matrix with k super-diagonals, for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is being supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is being supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the order of each matrix A_i.
  !>     @param[in]
  !>     k         [int]
  !>               the number of super-diagonals of each matrix A_i. Must be >= 0.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A         device array pointing to the first matrix A_1. Each A_i is of dimension (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The leading (k + 1) by n part of each A_i must contain the upper
  !>                 triangular band part of the Hermitian matrix, with the leading
  !>                 diagonal in row (k + 1), the first super-diagonal on the RHS
  !>                 of row k, etc.
  !>                 The top left k by x triangle of each A_i will not be referenced.
  !>                     Ex (upper, lda = n = 4, k = 1):
  !>                     A                             Represented matrix
  !>                     (0,0) (5,9) (6,8) (7,7)       (1, 0) (5, 9) (0, 0) (0, 0)
  !>                     (1,0) (2,0) (3,0) (4,0)       (5,-9) (2, 0) (6, 8) (0, 0)
  !>                     (0,0) (0,0) (0,0) (0,0)       (0, 0) (6,-8) (3, 0) (7, 7)
  !>                     (0,0) (0,0) (0,0) (0,0)       (0, 0) (0, 0) (7,-7) (4, 0)
  !> 
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The leading (k + 1) by n part of each A_i must contain the lower
  !>                 triangular band part of the Hermitian matrix, with the leading
  !>                 diagonal in row (1), the first sub-diagonal on the LHS of
  !>                 row 2, etc.
  !>                 The bottom right k by k triangle of each A_i will not be referenced.
  !>                     Ex (lower, lda = 2, n = 4, k = 1):
  !>                     A                               Represented matrix
  !>                     (1,0) (2,0) (3,0) (4,0)         (1, 0) (5,-9) (0, 0) (0, 0)
  !>                     (5,9) (6,8) (7,7) (0,0)         (5, 9) (2, 0) (6,-8) (0, 0)
  !>                                                     (0, 0) (6, 8) (3, 0) (7,-7)
  !>                                                     (0, 0) (0, 0) (7, 7) (4, 0)
  !> 
  !>               As a Hermitian matrix, the imaginary part of the main diagonal
  !>               of each A_i will not be referenced and is assumed to be == 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. must be >= max(1, n)
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     x         device array pointing to the first vector y_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1)
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device array pointing to the first vector y_1.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (y_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZhbmvStridedBatched
    function cublasZhbmvStridedBatched_(handle,uplo,n,k,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasZhbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhbmvStridedBatched_full_rank,&
      cublasZhbmvStridedBatched_rank_0,&
      cublasZhbmvStridedBatched_rank_1
  end interface
  
  interface cublasChemv
    function cublasChemv_(handle,uplo,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasChemv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChemv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasChemv_full_rank,&
      cublasChemv_rank_0,&
      cublasChemv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hemv performs one of the matrix-vector operations
  !> 
  !>         y := alpha*A*x + beta*y
  !> 
  !>     where alpha and beta are scalars, x and y are n element vectors and A is an
  !>     n by n Hermitian matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the order of the matrix A.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A         device pointer storing matrix A. Of dimension (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular part of A must contain
  !>                 the upper triangular part of a Hermitian matrix. The lower
  !>                 triangular part of A will not be referenced.
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular part of A must contain
  !>                 the lower triangular part of a Hermitian matrix. The upper
  !>                 triangular part of A will not be referenced.
  !>               As a Hermitian matrix, the imaginary part of the main diagonal
  !>               of A will not be referenced and is assumed to be == 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A. must be >= max(1, n)
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZhemv
    function cublasZhemv_(handle,uplo,n,alpha,A,da,x,incx,beta,y,incy) &
&bind(c, name="cublasZhemv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhemv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: da
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZhemv_full_rank,&
      cublasZhemv_rank_0,&
      cublasZhemv_rank_1
  end interface
  
  interface cublasChemvBatched
    function cublasChemvBatched_(handle,uplo,n,alpha,A,lda,x,incx,beta,&
&y,incy,batchCount) bind(c, name="cublasChemvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChemvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChemvBatched_full_rank,&
      cublasChemvBatched_rank_0,&
      cublasChemvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hemvBatched performs one of the matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i
  !> 
  !>     where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
  !>     n by n Hermitian matrix, for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the order of each matrix A_i.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i of dimension (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular part of each A_i must contain
  !>                 the upper triangular part of a Hermitian matrix. The lower
  !>                 triangular part of each A_i will not be referenced.
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular part of each A_i must contain
  !>                 the lower triangular part of a Hermitian matrix. The upper
  !>                 triangular part of each A_i will not be referenced.
  !>               As a Hermitian matrix, the imaginary part of the main diagonal
  !>               of each A_i will not be referenced and is assumed to be == 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. must be >= max(1, n)
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZhemvBatched
    function cublasZhemvBatched_(handle,uplo,n,alpha,A,lda,x,incx,beta,&
&y,incy,batchCount) bind(c, name="cublasZhemvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhemvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhemvBatched_full_rank,&
      cublasZhemvBatched_rank_0,&
      cublasZhemvBatched_rank_1
  end interface
  
  interface cublasChemvStridedBatched
    function cublasChemvStridedBatched_(handle,uplo,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasChemvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChemvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChemvStridedBatched_full_rank,&
      cublasChemvStridedBatched_rank_0,&
      cublasChemvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hemvStridedBatched performs one of the matrix-vector operations
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i
  !> 
  !>     where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
  !>     n by n Hermitian matrix, for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the order of each matrix A_i.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i of dimension (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular part of each A_i must contain
  !>                 the upper triangular part of a Hermitian matrix. The lower
  !>                 triangular part of each A_i will not be referenced.
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular part of each A_i must contain
  !>                 the lower triangular part of a Hermitian matrix. The upper
  !>                 triangular part of each A_i will not be referenced.
  !>               As a Hermitian matrix, the imaginary part of the main diagonal
  !>               of each A_i will not be referenced and is assumed to be == 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. must be >= max(1, n)
  !>     @param[in]
  !>     strideA    [hipblasStride]
  !>                 stride from the start of one (A_i) to the next (A_i+1)
  !> 
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (y_i+1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZhemvStridedBatched
    function cublasZhemvStridedBatched_(handle,uplo,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasZhemvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhemvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhemvStridedBatched_full_rank,&
      cublasZhemvStridedBatched_rank_0,&
      cublasZhemvStridedBatched_rank_1
  end interface
  
  interface cublasCher
    function cublasCher_(handle,uplo,n,alpha,x,incx,A,lda) bind(c, &
&name="cublasCher")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCher_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasCher_full_rank,&
      cublasCher_rank_0,&
      cublasCher_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     her performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*x**H
  !> 
  !>     where alpha is a real scalar, x is a vector, and A is an
  !>     n by n Hermitian matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in A.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in A.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[inout]
  !>     A         device pointer storing the specified triangular portion of
  !>               the Hermitian matrix A. Of size (lda * n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of the Hermitian matrix A is supplied. The lower
  !>                 triangluar portion will not be touched.
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of the Hermitian matrix A is supplied. The upper
  !>                 triangular portion will not be touched.
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A. Must be at least max(1, n).
  interface cublasZher
    function cublasZher_(handle,uplo,n,alpha,x,incx,A,lda) bind(c, &
&name="cublasZher")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZher_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasZher_full_rank,&
      cublasZher_rank_0,&
      cublasZher_rank_1
  end interface
  
  interface cublasCherBatched
    function cublasCherBatched_(handle,uplo,n,alpha,x,incx,A,lda,&
&batchCount) bind(c, name="cublasCherBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCherBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCherBatched_full_rank,&
      cublasCherBatched_rank_0,&
      cublasCherBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     herBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*x_i**H
  !> 
  !>     where alpha is a real scalar, x_i is a vector, and A_i is an
  !>     n by n symmetric matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in A.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in A.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[inout]
  !>     A         device array of device pointers storing the specified triangular portion of
  !>               each Hermitian matrix A_i of at least size ((n * (n + 1)) 2). Array is of at least size batchCount.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular portion
  !>                 of each A_i will not be touched.
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular portion
  !>                 of each A_i will not be touched.
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. Must be at least max(1, n).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZherBatched
    function cublasZherBatched_(handle,uplo,n,alpha,x,incx,A,lda,&
&batchCount) bind(c, name="cublasZherBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZherBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZherBatched_full_rank,&
      cublasZherBatched_rank_0,&
      cublasZherBatched_rank_1
  end interface
  
  interface cublasCherStridedBatched
    function cublasCherStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,A,lda,strideA,batchCount) bind(c, &
&name="cublasCherStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCherStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCherStridedBatched_full_rank,&
      cublasCherStridedBatched_rank_0,&
      cublasCherStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     herStridedBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*x_i**H
  !> 
  !>     where alpha is a real scalar, x_i is a vector, and A_i is an
  !>     n by n Hermitian matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in A.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in A.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer pointing to the first vector (x_1).
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>     @param[inout]
  !>     A         device array of device pointers storing the specified triangular portion of
  !>               each Hermitian matrix A_i. Points to the first matrix (A_1).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular
  !>                 portion of each A_i will not be touched.
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular
  !>                 portion of each A_i will not be touched.
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     strideA    [hipblasStride]
  !>                 stride from the start of one (A_i) and the next (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZherStridedBatched
    function cublasZherStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,A,lda,strideA,batchCount) bind(c, &
&name="cublasZherStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZherStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZherStridedBatched_full_rank,&
      cublasZherStridedBatched_rank_0,&
      cublasZherStridedBatched_rank_1
  end interface
  
  interface cublasCher2
    function cublasCher2_(handle,uplo,n,alpha,x,incx,y,incy,A,lda) &
&bind(c, name="cublasCher2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCher2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasCher2_full_rank,&
      cublasCher2_rank_0,&
      cublasCher2_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     her2 performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*y**H + conj(alpha)*y*x**H
  !> 
  !>     where alpha is a complex scalar, x and y are vectors, and A is an
  !>     n by n Hermitian matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     A         device pointer storing the specified triangular portion of
  !>               the Hermitian matrix A. Of size (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of the Hermitian matrix A is supplied. The lower triangular
  !>                 portion of A will not be touched.
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of the Hermitian matrix A is supplied. The upper triangular
  !>                 portion of A will not be touched.
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A. Must be at least max(lda, 1).
  interface cublasZher2
    function cublasZher2_(handle,uplo,n,alpha,x,incx,y,incy,A,lda) &
&bind(c, name="cublasZher2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZher2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasZher2_full_rank,&
      cublasZher2_rank_0,&
      cublasZher2_rank_1
  end interface
  
  interface cublasCher2Batched
    function cublasCher2Batched_(handle,uplo,n,alpha,x,incx,y,incy,A,&
&lda,batchCount) bind(c, name="cublasCher2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCher2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCher2Batched_full_rank,&
      cublasCher2Batched_rank_0,&
      cublasCher2Batched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     her2Batched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H
  !> 
  !>     where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
  !>     n by n Hermitian matrix for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[inout]
  !>     A         device array of device pointers storing the specified triangular portion of
  !>               each Hermitian matrix A_i of size (lda, n).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular
  !>                 portion of each A_i will not be touched.
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular
  !>                 portion of each A_i will not be touched.
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. Must be at least max(lda, 1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZher2Batched
    function cublasZher2Batched_(handle,uplo,n,alpha,x,incx,y,incy,A,&
&lda,batchCount) bind(c, name="cublasZher2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZher2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZher2Batched_full_rank,&
      cublasZher2Batched_rank_0,&
      cublasZher2Batched_rank_1
  end interface
  
  interface cublasCher2StridedBatched
    function cublasCher2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasCher2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCher2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCher2StridedBatched_full_rank,&
      cublasCher2StridedBatched_rank_0,&
      cublasCher2StridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     her2StridedBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H
  !> 
  !>     where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
  !>     n by n Hermitian matrix for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer pointing to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               specifies the stride between the beginning of one vector (x_i) and the next (x_i+1).
  !>     @param[in]
  !>     y         device pointer pointing to the first vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>               specifies the stride between the beginning of one vector (y_i) and the next (y_i+1).
  !>     @param[inout]
  !>     A         device pointer pointing to the first matrix (A_1). Stores the specified triangular portion of
  !>               each Hermitian matrix A_i.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular
  !>                 portion of each A_i will not be touched.
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular
  !>                 portion of each A_i will not be touched.
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. Must be at least max(lda, 1).
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               specifies the stride between the beginning of one matrix (A_i) and the next (A_i+1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZher2StridedBatched
    function cublasZher2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasZher2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZher2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZher2StridedBatched_full_rank,&
      cublasZher2StridedBatched_rank_0,&
      cublasZher2StridedBatched_rank_1
  end interface
  
  interface cublasChpmv
    function cublasChpmv_(handle,uplo,n,alpha,AP,x,incx,beta,y,incy) &
&bind(c, name="cublasChpmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChpmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasChpmv_rank_0,&
      cublasChpmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hpmv performs the matrix-vector operation
  !> 
  !>         y := alpha*A*x + beta*y
  !> 
  !>     where alpha and beta are scalars, x and y are n element vectors and A is an
  !>     n by n Hermitian matrix, supplied in packed form (see description below).
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the order of the matrix A, must be >= 0.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     AP        device pointer storing the packed version of the specified triangular portion of
  !>               the Hermitian matrix A. Of at least size ((n * (n + 1)) 2).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of the Hermitian matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (3, 2)
  !>                         (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,1), (4,0), (3,2), (5,-1), (6,0)]
  !>                         (3,-2) (5, 1) (6, 0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of the Hermitian matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (3, 2)
  !>                         (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,-1), (3,-2), (4,0), (5,1), (6,0)]
  !>                         (3,-2) (5, 1) (6, 0)
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>
  interface cublasZhpmv
    function cublasZhpmv_(handle,uplo,n,alpha,AP,x,incx,beta,y,incy) &
&bind(c, name="cublasZhpmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhpmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZhpmv_rank_0,&
      cublasZhpmv_rank_1
  end interface
  
  interface cublasChpmvBatched
    function cublasChpmvBatched_(handle,uplo,n,alpha,AP,x,incx,beta,y,&
&incy,batchCount) bind(c, name="cublasChpmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChpmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChpmvBatched_full_rank,&
      cublasChpmvBatched_rank_0,&
      cublasChpmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hpmvBatched performs the matrix-vector operation
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i
  !> 
  !>     where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
  !>     n by n Hermitian matrix, supplied in packed form (see description below),
  !>     for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: the upper triangular part of each Hermitian matrix A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: the lower triangular part of each Hermitian matrix A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the order of each matrix A_i.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     AP      device pointer of device pointers storing the packed version of the specified triangular
  !>             portion of each Hermitian matrix A_i. Each A_i is of at least size ((n * (n + 1)) 2).
  !>             if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>             The upper triangular portion of each Hermitian matrix A_i is supplied.
  !>             The matrix is compacted so that each AP_i contains the triangular portion column-by-column
  !>             so that:
  !>             AP(0) = A(0,0)
  !>             AP(1) = A(0,1)
  !>             AP(2) = A(1,1), etc.
  !>                 Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                     (1, 0) (2, 1) (3, 2)
  !>                     (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,1), (4,0), (3,2), (5,-1), (6,0)]
  !>                     (3,-2) (5, 1) (6, 0)
  !>         if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>             The lower triangular portion of each Hermitian matrix A_i is supplied.
  !>             The matrix is compacted so that each AP_i contains the triangular portion column-by-column
  !>             so that:
  !>             AP(0) = A(0,0)
  !>             AP(1) = A(1,0)
  !>             AP(2) = A(2,1), etc.
  !>                 Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                     (1, 0) (2, 1) (3, 2)
  !>                     (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,-1), (3,-2), (4,0), (5,1), (6,0)]
  !>                     (3,-2) (5, 1) (6, 0)
  !>         Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>         to be 0.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZhpmvBatched
    function cublasZhpmvBatched_(handle,uplo,n,alpha,AP,x,incx,beta,y,&
&incy,batchCount) bind(c, name="cublasZhpmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhpmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhpmvBatched_full_rank,&
      cublasZhpmvBatched_rank_0,&
      cublasZhpmvBatched_rank_1
  end interface
  
  interface cublasChpmvStridedBatched
    function cublasChpmvStridedBatched_(handle,uplo,n,alpha,AP,&
&strideAP,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasChpmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChpmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChpmvStridedBatched_rank_0,&
      cublasChpmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hpmvStridedBatched performs the matrix-vector operation
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i
  !> 
  !>     where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
  !>     n by n Hermitian matrix, supplied in packed form (see description below),
  !>     for each batch in i = [1, batchCount].
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: the upper triangular part of each Hermitian matrix A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: the lower triangular part of each Hermitian matrix A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the order of each matrix A_i.
  !>     @param[in]
  !>     alpha     device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     AP        device pointer pointing to the beginning of the first matrix (AP_1). Stores the packed
  !>               version of the specified triangular portion of each Hermitian matrix AP_i of size ((n * (n + 1)) 2).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that each AP_i contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (3, 2)
  !>                         (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,1), (4,0), (3,2), (5,-1), (6,0)]
  !>                         (3,-2) (5, 1) (6, 0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that each AP_i contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (3, 2)
  !>                         (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,-1), (3,-2), (4,0), (5,1), (6,0)]
  !>                         (3,-2) (5, 1) (6, 0)
  !>         Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>         to be 0.
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (AP_i) and the next one (AP_i+1).
  !>     @param[in]
  !>     x         device array pointing to the beginning of the first vector (x_1).
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta.
  !>     @param[inout]
  !>     y         device array pointing to the beginning of the first vector (y_1).
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (y_i+1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZhpmvStridedBatched
    function cublasZhpmvStridedBatched_(handle,uplo,n,alpha,AP,&
&strideAP,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasZhpmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhpmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhpmvStridedBatched_rank_0,&
      cublasZhpmvStridedBatched_rank_1
  end interface
  
  interface cublasChpr
    function cublasChpr_(handle,uplo,n,alpha,x,incx,AP) bind(c, &
&name="cublasChpr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChpr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasChpr_rank_0,&
      cublasChpr_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hpr performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*x**H
  !> 
  !>     where alpha is a real scalar, x is a vector, and A is an
  !>     n by n Hermitian matrix, supplied in packed form.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[inout]
  !>     AP        device pointer storing the packed version of the specified triangular portion of
  !>               the Hermitian matrix A. Of at least size ((n * (n + 1)) 2).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of the Hermitian matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of the Hermitian matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  interface cublasZhpr
    function cublasZhpr_(handle,uplo,n,alpha,x,incx,AP) bind(c, &
&name="cublasZhpr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhpr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasZhpr_rank_0,&
      cublasZhpr_rank_1
  end interface
  
  interface cublasChprBatched
    function cublasChprBatched_(handle,uplo,n,alpha,x,incx,AP,&
&batchCount) bind(c, name="cublasChprBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChprBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChprBatched_full_rank,&
      cublasChprBatched_rank_0,&
      cublasChprBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hprBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*x_i**H
  !> 
  !>     where alpha is a real scalar, x_i is a vector, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[inout]
  !>     AP        device array of device pointers storing the packed version of the specified triangular portion of
  !>               each Hermitian matrix A_i of at least size ((n * (n + 1)) 2). Array is of at least size batchCount.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZhprBatched
    function cublasZhprBatched_(handle,uplo,n,alpha,x,incx,AP,&
&batchCount) bind(c, name="cublasZhprBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhprBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhprBatched_full_rank,&
      cublasZhprBatched_rank_0,&
      cublasZhprBatched_rank_1
  end interface
  
  interface cublasChprStridedBatched
    function cublasChprStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,AP,strideAP,batchCount) bind(c, &
&name="cublasChprStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChprStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChprStridedBatched_rank_0,&
      cublasChprStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hprStridedBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*x_i**H
  !> 
  !>     where alpha is a real scalar, x_i is a vector, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer pointing to the first vector (x_1).
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>     @param[inout]
  !>     AP        device array of device pointers storing the packed version of the specified triangular portion of
  !>               each Hermitian matrix A_i. Points to the first matrix (A_1).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     strideAP   [hipblasStride]
  !>                 stride from the start of one (A_i) and the next (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZhprStridedBatched
    function cublasZhprStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,AP,strideAP,batchCount) bind(c, &
&name="cublasZhprStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhprStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhprStridedBatched_rank_0,&
      cublasZhprStridedBatched_rank_1
  end interface
  
  interface cublasChpr2
    function cublasChpr2_(handle,uplo,n,alpha,x,incx,y,incy,AP) bind(c,&
& name="cublasChpr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChpr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasChpr2_rank_0,&
      cublasChpr2_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hpr2 performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*y**H + conj(alpha)*y*x**H
  !> 
  !>     where alpha is a complex scalar, x and y are vectors, and A is an
  !>     n by n Hermitian matrix, supplied in packed form.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     AP        device pointer storing the packed version of the specified triangular portion of
  !>               the Hermitian matrix A. Of at least size ((n * (n + 1)) 2).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of the Hermitian matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of the Hermitian matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  interface cublasZhpr2
    function cublasZhpr2_(handle,uplo,n,alpha,x,incx,y,incy,AP) bind(c,&
& name="cublasZhpr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhpr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasZhpr2_rank_0,&
      cublasZhpr2_rank_1
  end interface
  
  interface cublasChpr2Batched
    function cublasChpr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,AP,&
&batchCount) bind(c, name="cublasChpr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChpr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChpr2Batched_full_rank,&
      cublasChpr2Batched_rank_0,&
      cublasChpr2Batched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hpr2Batched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H
  !> 
  !>     where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[inout]
  !>     AP        device array of device pointers storing the packed version of the specified triangular portion of
  !>               each Hermitian matrix A_i of at least size ((n * (n + 1)) 2). Array is of at least size batchCount.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZhpr2Batched
    function cublasZhpr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,AP,&
&batchCount) bind(c, name="cublasZhpr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhpr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhpr2Batched_full_rank,&
      cublasZhpr2Batched_rank_0,&
      cublasZhpr2Batched_rank_1
  end interface
  
  interface cublasChpr2StridedBatched
    function cublasChpr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,AP,strideAP,batchCount) bind(c, &
&name="cublasChpr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChpr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChpr2StridedBatched_rank_0,&
      cublasChpr2StridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     hpr2StridedBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H
  !> 
  !>     where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer pointing to the first vector (x_1).
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>     @param[in]
  !>     y         device pointer pointing to the first vector (y_1).
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (y_i+1).
  !>     @param[inout]
  !>     AP        device array of device pointers storing the packed version of the specified triangular portion of
  !>               each Hermitian matrix A_i. Points to the first matrix (A_1).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each Hermitian matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
  !>                         (1, 0) (2, 1) (4,9)
  !>                         (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
  !>                         (4,-9) (5,-3) (6,0)
  !>             Note that the imaginary part of the diagonal elements are not accessed and are assumed
  !>             to be 0.
  !>     @param[in]
  !>     strideAP    [hipblasStride]
  !>                 stride from the start of one (A_i) and the next (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZhpr2StridedBatched
    function cublasZhpr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,AP,strideAP,batchCount) bind(c, &
&name="cublasZhpr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhpr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhpr2StridedBatched_rank_0,&
      cublasZhpr2StridedBatched_rank_1
  end interface
  
  interface cublasSsbmv
    function cublasSsbmv_(handle,uplo,n,k,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasSsbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasSsbmv_full_rank,&
      cublasSsbmv_rank_0,&
      cublasSsbmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     sbmv performs the matrix-vector operation:
  !> 
  !>         y := alpha*A*x + beta*y,
  !> 
  !>     where alpha and beta are scalars, x and y are n element vectors and
  !>     A should contain an upper or lower triangular n by n symmetric banded matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>     @param[in]
  !>     k         [int]
  !>               specifies the number of sub- and super-diagonals
  !>     @param[in]
  !>     alpha
  !>               specifies the scalar alpha
  !>     @param[in]
  !>     A         pointer storing matrix A on the GPU
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of matrix A
  !>     @param[in]
  !>     x         pointer storing vector x on the GPU
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x
  !>     @param[in]
  !>     beta      specifies the scalar beta
  !>     @param[out]
  !>     y         pointer storing vector y on the GPU
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y
  !>
  interface cublasDsbmv
    function cublasDsbmv_(handle,uplo,n,k,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasDsbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDsbmv_full_rank,&
      cublasDsbmv_rank_0,&
      cublasDsbmv_rank_1
  end interface
  
  interface cublasSsbmvBatched
    function cublasSsbmvBatched_(handle,uplo,n,k,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasSsbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsbmvBatched_full_rank,&
      cublasSsbmvBatched_rank_0,&
      cublasSsbmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     sbmvBatched performs the matrix-vector operation:
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     n by n symmetric banded matrix, for i = 1, ..., batchCount.
  !>     A should contain an upper or lower triangular n by n symmetric banded matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               number of rows and columns of each matrix A_i
  !>     @param[in]
  !>     k         [int]
  !>               specifies the number of sub- and super-diagonals
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each matrix A_i
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta
  !>     @param[out]
  !>     y         device array of device pointers storing each vector y_i
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasDsbmvBatched
    function cublasDsbmvBatched_(handle,uplo,n,k,alpha,A,lda,x,incx,&
&beta,y,incy,batchCount) bind(c, name="cublasDsbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsbmvBatched_full_rank,&
      cublasDsbmvBatched_rank_0,&
      cublasDsbmvBatched_rank_1
  end interface
  
  interface cublasSsbmvStridedBatched
    function cublasSsbmvStridedBatched_(handle,uplo,n,k,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasSsbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsbmvStridedBatched_full_rank,&
      cublasSsbmvStridedBatched_rank_0,&
      cublasSsbmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     sbmvStridedBatched performs the matrix-vector operation:
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     n by n symmetric banded matrix, for i = 1, ..., batchCount.
  !>     A should contain an upper or lower triangular n by n symmetric banded matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               number of rows and columns of each matrix A_i
  !>     @param[in]
  !>     k         [int]
  !>               specifies the number of sub- and super-diagonals
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha
  !>     @param[in]
  !>     A         Device pointer to the first matrix A_1 on the GPU
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each matrix A_i
  !>     @param[in]
  !>     strideA     [hipblasStride]
  !>                 stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     x         Device pointer to the first vector x_1 on the GPU
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i
  !>     @param[in]
  !>     stridex     [hipblasStride]
  !>                 stride from the start of one vector (x_i) and the next one (x_i+1).
  !>                 There are no restrictions placed on stridex, however the user should
  !>                 take care to ensure that stridex is of appropriate size.
  !>                 This typically means stridex >= n * incx. stridex should be non zero.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta
  !>     @param[out]
  !>     y         Device pointer to the first vector y_1 on the GPU
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i
  !>     @param[in]
  !>     stridey     [hipblasStride]
  !>                 stride from the start of one vector (y_i) and the next one (y_i+1).
  !>                 There are no restrictions placed on stridey, however the user should
  !>                 take care to ensure that stridey is of appropriate size.
  !>                 This typically means stridey >= n * incy. stridey should be non zero.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasDsbmvStridedBatched
    function cublasDsbmvStridedBatched_(handle,uplo,n,k,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasDsbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsbmvStridedBatched_full_rank,&
      cublasDsbmvStridedBatched_rank_0,&
      cublasDsbmvStridedBatched_rank_1
  end interface
  
  interface cublasSspmv
    function cublasSspmv_(handle,uplo,n,alpha,AP,x,incx,beta,y,incy) &
&bind(c, name="cublasSspmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSspmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasSspmv_rank_0,&
      cublasSspmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     spmv performs the matrix-vector operation:
  !> 
  !>         y := alpha*A*x + beta*y,
  !> 
  !>     where alpha and beta are scalars, x and y are n element vectors and
  !>     A should contain an upper or lower triangular n by n packed symmetric matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>     @param[in]
  !>     alpha
  !>               specifies the scalar alpha
  !>     @param[in]
  !>     A         pointer storing matrix A on the GPU
  !>     @param[in]
  !>     x         pointer storing vector x on the GPU
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x
  !>     @param[in]
  !>     beta      specifies the scalar beta
  !>     @param[out]
  !>     y         pointer storing vector y on the GPU
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y
  !>
  interface cublasDspmv
    function cublasDspmv_(handle,uplo,n,alpha,AP,x,incx,beta,y,incy) &
&bind(c, name="cublasDspmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDspmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDspmv_rank_0,&
      cublasDspmv_rank_1
  end interface
  
  interface cublasSspmvBatched
    function cublasSspmvBatched_(handle,uplo,n,alpha,AP,x,incx,beta,y,&
&incy,batchCount) bind(c, name="cublasSspmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSspmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSspmvBatched_full_rank,&
      cublasSspmvBatched_rank_0,&
      cublasSspmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     spmvBatched performs the matrix-vector operation:
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     n by n symmetric matrix, for i = 1, ..., batchCount.
  !>     A should contain an upper or lower triangular n by n packed symmetric matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               number of rows and columns of each matrix A_i
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta
  !>     @param[out]
  !>     y         device array of device pointers storing each vector y_i
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasDspmvBatched
    function cublasDspmvBatched_(handle,uplo,n,alpha,AP,x,incx,beta,y,&
&incy,batchCount) bind(c, name="cublasDspmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDspmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDspmvBatched_full_rank,&
      cublasDspmvBatched_rank_0,&
      cublasDspmvBatched_rank_1
  end interface
  
  interface cublasSspmvStridedBatched
    function cublasSspmvStridedBatched_(handle,uplo,n,alpha,AP,&
&strideAP,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasSspmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSspmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSspmvStridedBatched_rank_0,&
      cublasSspmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     spmvStridedBatched performs the matrix-vector operation:
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     n by n symmetric matrix, for i = 1, ..., batchCount.
  !>     A should contain an upper or lower triangular n by n packed symmetric matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               number of rows and columns of each matrix A_i
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha
  !>     @param[in]
  !>     A         Device pointer to the first matrix A_1 on the GPU
  !>     @param[in]
  !>     strideA     [hipblasStride]
  !>                 stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     x         Device pointer to the first vector x_1 on the GPU
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i
  !>     @param[in]
  !>     stridex     [hipblasStride]
  !>                 stride from the start of one vector (x_i) and the next one (x_i+1).
  !>                 There are no restrictions placed on stridex, however the user should
  !>                 take care to ensure that stridex is of appropriate size.
  !>                 This typically means stridex >= n * incx. stridex should be non zero.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta
  !>     @param[out]
  !>     y         Device pointer to the first vector y_1 on the GPU
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i
  !>     @param[in]
  !>     stridey     [hipblasStride]
  !>                 stride from the start of one vector (y_i) and the next one (y_i+1).
  !>                 There are no restrictions placed on stridey, however the user should
  !>                 take care to ensure that stridey is of appropriate size.
  !>                 This typically means stridey >= n * incy. stridey should be non zero.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasDspmvStridedBatched
    function cublasDspmvStridedBatched_(handle,uplo,n,alpha,AP,&
&strideAP,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasDspmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDspmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDspmvStridedBatched_rank_0,&
      cublasDspmvStridedBatched_rank_1
  end interface
  
  interface cublasSspr
    function cublasSspr_(handle,uplo,n,alpha,x,incx,AP) bind(c, &
&name="cublasSspr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSspr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasSspr_rank_0,&
      cublasSspr_rank_1
  end interface
  
  interface cublasDspr
    function cublasDspr_(handle,uplo,n,alpha,x,incx,AP) bind(c, &
&name="cublasDspr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDspr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasDspr_rank_0,&
      cublasDspr_rank_1
  end interface
  
  interface cublasCspr
    function cublasCspr_(handle,uplo,n,alpha,x,incx,AP) bind(c, &
&name="cublasCspr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCspr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasCspr_rank_0,&
      cublasCspr_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     spr performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*x**T
  !> 
  !>     where alpha is a scalar, x is a vector, and A is an
  !>     n by n symmetric matrix, supplied in packed form.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[inout]
  !>     AP        device pointer storing the packed version of the specified triangular portion of
  !>               the symmetric matrix A. Of at least size ((n * (n + 1)) 2).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of the symmetric matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
  !>                         1 2 4 7
  !>                         2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         4 5 6 9
  !>                         7 8 9 0
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of the symmetric matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
  !>                         1 2 3 4
  !>                         2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         3 6 8 9
  !>                         4 7 9 0
  interface cublasZspr
    function cublasZspr_(handle,uplo,n,alpha,x,incx,AP) bind(c, &
&name="cublasZspr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZspr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasZspr_rank_0,&
      cublasZspr_rank_1
  end interface
  
  interface cublasSsprBatched
    function cublasSsprBatched_(handle,uplo,n,alpha,x,incx,AP,&
&batchCount) bind(c, name="cublasSsprBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsprBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsprBatched_full_rank,&
      cublasSsprBatched_rank_0,&
      cublasSsprBatched_rank_1
  end interface
  
  interface cublasDsprBatched
    function cublasDsprBatched_(handle,uplo,n,alpha,x,incx,AP,&
&batchCount) bind(c, name="cublasDsprBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsprBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsprBatched_full_rank,&
      cublasDsprBatched_rank_0,&
      cublasDsprBatched_rank_1
  end interface
  
  interface cublasCsprBatched
    function cublasCsprBatched_(handle,uplo,n,alpha,x,incx,AP,&
&batchCount) bind(c, name="cublasCsprBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsprBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsprBatched_full_rank,&
      cublasCsprBatched_rank_0,&
      cublasCsprBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     sprBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*x_i**T
  !> 
  !>     where alpha is a scalar, x_i is a vector, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[inout]
  !>     AP        device array of device pointers storing the packed version of the specified triangular portion of
  !>               each symmetric matrix A_i of at least size ((n * (n + 1)) 2). Array is of at least size batchCount.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
  !>                         1 2 4 7
  !>                         2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         4 5 6 9
  !>                         7 8 9 0
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
  !>                         1 2 3 4
  !>                         2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         3 6 8 9
  !>                         4 7 9 0
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZsprBatched
    function cublasZsprBatched_(handle,uplo,n,alpha,x,incx,AP,&
&batchCount) bind(c, name="cublasZsprBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsprBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsprBatched_full_rank,&
      cublasZsprBatched_rank_0,&
      cublasZsprBatched_rank_1
  end interface
  
  interface cublasSsprStridedBatched
    function cublasSsprStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,AP,strideAP,batchCount) bind(c, &
&name="cublasSsprStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsprStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsprStridedBatched_rank_0,&
      cublasSsprStridedBatched_rank_1
  end interface
  
  interface cublasDsprStridedBatched
    function cublasDsprStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,AP,strideAP,batchCount) bind(c, &
&name="cublasDsprStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsprStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsprStridedBatched_rank_0,&
      cublasDsprStridedBatched_rank_1
  end interface
  
  interface cublasCsprStridedBatched
    function cublasCsprStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,AP,strideAP,batchCount) bind(c, &
&name="cublasCsprStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsprStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsprStridedBatched_rank_0,&
      cublasCsprStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     sprStridedBatched performs the matrix-vector operations
  !> 
  !>         A_i := A_i + alpha*x_i*x_i**T
  !> 
  !>     where alpha is a scalar, x_i is a vector, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer pointing to the first vector (x_1).
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>     @param[inout]
  !>     AP        device pointer storing the packed version of the specified triangular portion of
  !>               each symmetric matrix A_i. Points to the first A_1.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
  !>                         1 2 4 7
  !>                         2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         4 5 6 9
  !>                         7 8 9 0
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(2) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
  !>                         1 2 3 4
  !>                         2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         3 6 8 9
  !>                         4 7 9 0
  !>     @param[in]
  !>     strideA    [hipblasStride]
  !>                 stride from the start of one (A_i) and the next (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasZsprStridedBatched
    function cublasZsprStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,AP,strideAP,batchCount) bind(c, &
&name="cublasZsprStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsprStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsprStridedBatched_rank_0,&
      cublasZsprStridedBatched_rank_1
  end interface
  
  interface cublasSspr2
    function cublasSspr2_(handle,uplo,n,alpha,x,incx,y,incy,AP) bind(c,&
& name="cublasSspr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSspr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasSspr2_rank_0,&
      cublasSspr2_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     spr2 performs the matrix-vector operation
  !> 
  !>         A := A + alpha*x*y**T + alpha*y*x**T
  !> 
  !>     where alpha is a scalar, x and y are vectors, and A is an
  !>     n by n symmetric matrix, supplied in packed form.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     AP        device pointer storing the packed version of the specified triangular portion of
  !>               the symmetric matrix A. Of at least size ((n * (n + 1)) 2).
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of the symmetric matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
  !>                         1 2 4 7
  !>                         2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         4 5 6 9
  !>                         7 8 9 0
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of the symmetric matrix A is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(n) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
  !>                         1 2 3 4
  !>                         2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         3 6 8 9
  !>                         4 7 9 0
  interface cublasDspr2
    function cublasDspr2_(handle,uplo,n,alpha,x,incx,y,incy,AP) bind(c,&
& name="cublasDspr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDspr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: AP
    end function

    module procedure &
      cublasDspr2_rank_0,&
      cublasDspr2_rank_1
  end interface
  
  interface cublasSspr2Batched
    function cublasSspr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,AP,&
&batchCount) bind(c, name="cublasSspr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSspr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSspr2Batched_full_rank,&
      cublasSspr2Batched_rank_0,&
      cublasSspr2Batched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     spr2Batched performs the matrix-vector operation
  !> 
  !>         A_i := A_i + alpha*x_i*y_i**T + alpha*y_i*x_i**T
  !> 
  !>     where alpha is a scalar, x_i and y_i are vectors, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[inout]
  !>     AP        device array of device pointers storing the packed version of the specified triangular portion of
  !>               each symmetric matrix A_i of at least size ((n * (n + 1)) 2). Array is of at least size batchCount.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
  !>                         1 2 4 7
  !>                         2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         4 5 6 9
  !>                         7 8 9 0
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(n) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
  !>                         1 2 3 4
  !>                         2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         3 6 8 9
  !>                         4 7 9 0
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasDspr2Batched
    function cublasDspr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,AP,&
&batchCount) bind(c, name="cublasDspr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDspr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: AP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDspr2Batched_full_rank,&
      cublasDspr2Batched_rank_0,&
      cublasDspr2Batched_rank_1
  end interface
  
  interface cublasSspr2StridedBatched
    function cublasSspr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,AP,strideAP,batchCount) bind(c, &
&name="cublasSspr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSspr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSspr2StridedBatched_rank_0,&
      cublasSspr2StridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     spr2StridedBatched performs the matrix-vector operation
  !> 
  !>         A_i := A_i + alpha*x_i*y_i**T + alpha*y_i*x_i**T
  !> 
  !>     where alpha is a scalar, x_i amd y_i are vectors, and A_i is an
  !>     n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
  !>               HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A_i, must be at least 0.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer pointing to the first vector (x_1).
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector (x_i) and the next one (x_i+1).
  !>     @param[in]
  !>     y         device pointer pointing to the first vector (y_1).
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     stridey  [hipblasStride]
  !>               stride from the start of one vector (y_i) and the next one (y_i+1).
  !>     @param[inout]
  !>     AP        device pointer storing the packed version of the specified triangular portion of
  !>               each symmetric matrix A_i. Points to the first A_1.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The upper triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(0,1)
  !>                 AP(2) = A(1,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
  !>                         1 2 4 7
  !>                         2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         4 5 6 9
  !>                         7 8 9 0
  !>             if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The lower triangular portion of each symmetric matrix A_i is supplied.
  !>                 The matrix is compacted so that AP contains the triangular portion column-by-column
  !>                 so that:
  !>                 AP(0) = A(0,0)
  !>                 AP(1) = A(1,0)
  !>                 AP(n) = A(2,1), etc.
  !>                     Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
  !>                         1 2 3 4
  !>                         2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
  !>                         3 6 8 9
  !>                         4 7 9 0
  !>     @param[in]
  !>     strideA    [hipblasStride]
  !>                 stride from the start of one (A_i) and the next (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  interface cublasDspr2StridedBatched
    function cublasDspr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,AP,strideAP,batchCount) bind(c, &
&name="cublasDspr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDspr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDspr2StridedBatched_rank_0,&
      cublasDspr2StridedBatched_rank_1
  end interface
  
  interface cublasSsymv
    function cublasSsymv_(handle,uplo,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasSsymv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsymv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasSsymv_full_rank,&
      cublasSsymv_rank_0,&
      cublasSsymv_rank_1
  end interface
  
  interface cublasDsymv
    function cublasDsymv_(handle,uplo,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasDsymv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsymv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasDsymv_full_rank,&
      cublasDsymv_rank_0,&
      cublasDsymv_rank_1
  end interface
  
  interface cublasCsymv
    function cublasCsymv_(handle,uplo,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasCsymv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsymv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasCsymv_full_rank,&
      cublasCsymv_rank_0,&
      cublasCsymv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     symv performs the matrix-vector operation:
  !> 
  !>         y := alpha*A*x + beta*y,
  !> 
  !>     where alpha and beta are scalars, x and y are n element vectors and
  !>     A should contain an upper or lower triangular n by n symmetric matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>     @param[in]
  !>     alpha
  !>               specifies the scalar alpha
  !>     @param[in]
  !>     A         pointer storing matrix A on the GPU
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A
  !>     @param[in]
  !>     x         pointer storing vector x on the GPU
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x
  !>     @param[in]
  !>     beta      specifies the scalar beta
  !>     @param[out]
  !>     y         pointer storing vector y on the GPU
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y
  !>
  interface cublasZsymv
    function cublasZsymv_(handle,uplo,n,alpha,A,lda,x,incx,beta,y,&
&incy) bind(c, name="cublasZsymv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsymv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
    end function

    module procedure &
      cublasZsymv_full_rank,&
      cublasZsymv_rank_0,&
      cublasZsymv_rank_1
  end interface
  
  interface cublasSsymvBatched
    function cublasSsymvBatched_(handle,uplo,n,alpha,A,lda,x,incx,beta,&
&y,incy,batchCount) bind(c, name="cublasSsymvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsymvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_float) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsymvBatched_full_rank,&
      cublasSsymvBatched_rank_0,&
      cublasSsymvBatched_rank_1
  end interface
  
  interface cublasDsymvBatched
    function cublasDsymvBatched_(handle,uplo,n,alpha,A,lda,x,incx,beta,&
&y,incy,batchCount) bind(c, name="cublasDsymvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsymvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      real(c_double) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsymvBatched_full_rank,&
      cublasDsymvBatched_rank_0,&
      cublasDsymvBatched_rank_1
  end interface
  
  interface cublasCsymvBatched
    function cublasCsymvBatched_(handle,uplo,n,alpha,A,lda,x,incx,beta,&
&y,incy,batchCount) bind(c, name="cublasCsymvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsymvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_float_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsymvBatched_full_rank,&
      cublasCsymvBatched_rank_0,&
      cublasCsymvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     symvBatched performs the matrix-vector operation:
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     n by n symmetric matrix, for i = 1, ..., batchCount.
  !>     A a should contain an upper or lower triangular symmetric matrix
  !>     and the opposing triangular part of A is not referenced
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               number of rows and columns of each matrix A_i
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each matrix A_i
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta
  !>     @param[out]
  !>     y         device array of device pointers storing each vector y_i
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZsymvBatched
    function cublasZsymvBatched_(handle,uplo,n,alpha,A,lda,x,incx,beta,&
&y,incy,batchCount) bind(c, name="cublasZsymvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsymvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      complex(c_double_complex) :: beta
      type(c_ptr) :: y
      integer(c_int),value :: incy
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsymvBatched_full_rank,&
      cublasZsymvBatched_rank_0,&
      cublasZsymvBatched_rank_1
  end interface
  
  interface cublasSsymvStridedBatched
    function cublasSsymvStridedBatched_(handle,uplo,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasSsymvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsymvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_float) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsymvStridedBatched_full_rank,&
      cublasSsymvStridedBatched_rank_0,&
      cublasSsymvStridedBatched_rank_1
  end interface
  
  interface cublasDsymvStridedBatched
    function cublasDsymvStridedBatched_(handle,uplo,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasDsymvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsymvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      real(c_double) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsymvStridedBatched_full_rank,&
      cublasDsymvStridedBatched_rank_0,&
      cublasDsymvStridedBatched_rank_1
  end interface
  
  interface cublasCsymvStridedBatched
    function cublasCsymvStridedBatched_(handle,uplo,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasCsymvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsymvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_float_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsymvStridedBatched_full_rank,&
      cublasCsymvStridedBatched_rank_0,&
      cublasCsymvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     symvStridedBatched performs the matrix-vector operation:
  !> 
  !>         y_i := alpha*A_i*x_i + beta*y_i,
  !> 
  !>     where (A_i, x_i, y_i) is the i-th instance of the batch.
  !>     alpha and beta are scalars, x_i and y_i are vectors and A_i is an
  !>     n by n symmetric matrix, for i = 1, ..., batchCount.
  !>     A a should contain an upper or lower triangular symmetric matrix
  !>     and the opposing triangular part of A is not referenced
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               number of rows and columns of each matrix A_i
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha
  !>     @param[in]
  !>     A         Device pointer to the first matrix A_1 on the GPU
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each matrix A_i
  !>     @param[in]
  !>     strideA     [hipblasStride]
  !>                 stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     x         Device pointer to the first vector x_1 on the GPU
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each vector x_i
  !>     @param[in]
  !>     stridex     [hipblasStride]
  !>                 stride from the start of one vector (x_i) and the next one (x_i+1).
  !>                 There are no restrictions placed on stridex, however the user should
  !>                 take care to ensure that stridex is of appropriate size.
  !>                 This typically means stridex >= n * incx. stridex should be non zero.
  !>     @param[in]
  !>     beta      device pointer or host pointer to scalar beta
  !>     @param[out]
  !>     y         Device pointer to the first vector y_1 on the GPU
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each vector y_i
  !>     @param[in]
  !>     stridey     [hipblasStride]
  !>                 stride from the start of one vector (y_i) and the next one (y_i+1).
  !>                 There are no restrictions placed on stridey, however the user should
  !>                 take care to ensure that stridey is of appropriate size.
  !>                 This typically means stridey >= n * incy. stridey should be non zero.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZsymvStridedBatched
    function cublasZsymvStridedBatched_(handle,uplo,n,alpha,A,lda,&
&strideA,x,incx,stridex,beta,y,incy,stridey,batchCount) bind(c, &
&name="cublasZsymvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsymvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      complex(c_double_complex) :: beta
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsymvStridedBatched_full_rank,&
      cublasZsymvStridedBatched_rank_0,&
      cublasZsymvStridedBatched_rank_1
  end interface
  
  interface cublasSsyr
    function cublasSsyr_(handle,uplo,n,alpha,x,incx,A,lda) bind(c, &
&name="cublasSsyr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasSsyr_full_rank,&
      cublasSsyr_rank_0,&
      cublasSsyr_rank_1
  end interface
  
  interface cublasDsyr
    function cublasDsyr_(handle,uplo,n,alpha,x,incx,A,lda) bind(c, &
&name="cublasDsyr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasDsyr_full_rank,&
      cublasDsyr_rank_0,&
      cublasDsyr_rank_1
  end interface
  
  interface cublasCsyr
    function cublasCsyr_(handle,uplo,n,alpha,x,incx,A,lda) bind(c, &
&name="cublasCsyr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasCsyr_full_rank,&
      cublasCsyr_rank_0,&
      cublasCsyr_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     syr performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*x**T
  !> 
  !>     where alpha is a scalar, x is a vector, and A is an
  !>     n by n symmetric matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[inout]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>
  interface cublasZsyr
    function cublasZsyr_(handle,uplo,n,alpha,x,incx,A,lda) bind(c, &
&name="cublasZsyr")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyr_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasZsyr_full_rank,&
      cublasZsyr_rank_0,&
      cublasZsyr_rank_1
  end interface
  
  interface cublasSsyrBatched
    function cublasSsyrBatched_(handle,uplo,n,alpha,x,incx,A,lda,&
&batchCount) bind(c, name="cublasSsyrBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyrBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyrBatched_full_rank,&
      cublasSsyrBatched_rank_0,&
      cublasSsyrBatched_rank_1
  end interface
  
  interface cublasDsyrBatched
    function cublasDsyrBatched_(handle,uplo,n,alpha,x,incx,A,lda,&
&batchCount) bind(c, name="cublasDsyrBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyrBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyrBatched_full_rank,&
      cublasDsyrBatched_rank_0,&
      cublasDsyrBatched_rank_1
  end interface
  
  interface cublasCsyrBatched
    function cublasCsyrBatched_(handle,uplo,n,alpha,x,incx,A,lda,&
&batchCount) bind(c, name="cublasCsyrBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyrBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyrBatched_full_rank,&
      cublasCsyrBatched_rank_0,&
      cublasCsyrBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     syrBatched performs a batch of matrix-vector operations
  !> 
  !>         A[i] := A[i] + alpha*x[i]*x[i]**T
  !> 
  !>     where alpha is a scalar, x is an array of vectors, and A is an array of
  !>     n by n symmetric matrices, for i = 1 , ... , batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[inout]
  !>     A         device array of device pointers storing each matrix A_i.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZsyrBatched
    function cublasZsyrBatched_(handle,uplo,n,alpha,x,incx,A,lda,&
&batchCount) bind(c, name="cublasZsyrBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyrBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyrBatched_full_rank,&
      cublasZsyrBatched_rank_0,&
      cublasZsyrBatched_rank_1
  end interface
  
  interface cublasSsyrStridedBatched
    function cublasSsyrStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,A,lda,stridey,batchCount) bind(c, &
&name="cublasSsyrStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyrStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyrStridedBatched_full_rank,&
      cublasSsyrStridedBatched_rank_0,&
      cublasSsyrStridedBatched_rank_1
  end interface
  
  interface cublasDsyrStridedBatched
    function cublasDsyrStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,A,lda,stridey,batchCount) bind(c, &
&name="cublasDsyrStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyrStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyrStridedBatched_full_rank,&
      cublasDsyrStridedBatched_rank_0,&
      cublasDsyrStridedBatched_rank_1
  end interface
  
  interface cublasCsyrStridedBatched
    function cublasCsyrStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,A,lda,stridey,batchCount) bind(c, &
&name="cublasCsyrStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyrStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyrStridedBatched_full_rank,&
      cublasCsyrStridedBatched_rank_0,&
      cublasCsyrStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     syrStridedBatched performs the matrix-vector operations
  !> 
  !>         A[i] := A[i] + alpha*x[i]*x[i]**T
  !> 
  !>     where alpha is a scalar, vectors, and A is an array of
  !>     n by n symmetric matrices, for i = 1 , ... , batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               specifies the pointer increment between vectors (x_i) and (x_i+1).
  !>     @param[inout]
  !>     A         device pointer to the first matrix A_1.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     strideA   [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>
  interface cublasZsyrStridedBatched
    function cublasZsyrStridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,A,lda,stridey,batchCount) bind(c, &
&name="cublasZsyrStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyrStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyrStridedBatched_full_rank,&
      cublasZsyrStridedBatched_rank_0,&
      cublasZsyrStridedBatched_rank_1
  end interface
  
  interface cublasSsyr2
    function cublasSsyr2_(handle,uplo,n,alpha,x,incx,y,incy,A,lda) &
&bind(c, name="cublasSsyr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasSsyr2_full_rank,&
      cublasSsyr2_rank_0,&
      cublasSsyr2_rank_1
  end interface
  
  interface cublasDsyr2
    function cublasDsyr2_(handle,uplo,n,alpha,x,incx,y,incy,A,lda) &
&bind(c, name="cublasDsyr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasDsyr2_full_rank,&
      cublasDsyr2_rank_0,&
      cublasDsyr2_rank_1
  end interface
  
  interface cublasCsyr2
    function cublasCsyr2_(handle,uplo,n,alpha,x,incx,y,incy,A,lda) &
&bind(c, name="cublasCsyr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasCsyr2_full_rank,&
      cublasCsyr2_rank_0,&
      cublasCsyr2_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     syr2 performs the matrix-vector operations
  !> 
  !>         A := A + alpha*x*y**T + alpha*y*x**T
  !> 
  !>     where alpha is a scalar, x and y are vectors, and A is an
  !>     n by n symmetric matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>     @param[in]
  !>     y         device pointer storing vector y.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of y.
  !>     @param[inout]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>
  interface cublasZsyr2
    function cublasZsyr2_(handle,uplo,n,alpha,x,incx,y,incy,A,lda) &
&bind(c, name="cublasZsyr2")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyr2_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      type(c_ptr),value :: A
      integer(c_int),value :: lda
    end function

    module procedure &
      cublasZsyr2_full_rank,&
      cublasZsyr2_rank_0,&
      cublasZsyr2_rank_1
  end interface
  
  interface cublasSsyr2Batched
    function cublasSsyr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,A,&
&lda,batchCount) bind(c, name="cublasSsyr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyr2Batched_full_rank,&
      cublasSsyr2Batched_rank_0,&
      cublasSsyr2Batched_rank_1
  end interface
  
  interface cublasDsyr2Batched
    function cublasDsyr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,A,&
&lda,batchCount) bind(c, name="cublasDsyr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyr2Batched_full_rank,&
      cublasDsyr2Batched_rank_0,&
      cublasDsyr2Batched_rank_1
  end interface
  
  interface cublasCsyr2Batched
    function cublasCsyr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,A,&
&lda,batchCount) bind(c, name="cublasCsyr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyr2Batched_full_rank,&
      cublasCsyr2Batched_rank_0,&
      cublasCsyr2Batched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     syr2Batched performs a batch of matrix-vector operations
  !> 
  !>         A[i] := A[i] + alpha*x[i]*y[i]**T + alpha*y[i]*x[i]**T
  !> 
  !>     where alpha is a scalar, x[i] and y[i] are vectors, and A[i] is a
  !>     n by n symmetric matrix, for i = 1 , ... , batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of matrix A.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     y         device array of device pointers storing each vector y_i.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[inout]
  !>     A         device array of device pointers storing each matrix A_i.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZsyr2Batched
    function cublasZsyr2Batched_(handle,uplo,n,alpha,x,incx,y,incy,A,&
&lda,batchCount) bind(c, name="cublasZsyr2Batched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyr2Batched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: y
      integer(c_int),value :: incy
      type(c_ptr) :: A
      integer(c_int),value :: lda
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyr2Batched_full_rank,&
      cublasZsyr2Batched_rank_0,&
      cublasZsyr2Batched_rank_1
  end interface
  
  interface cublasSsyr2StridedBatched
    function cublasSsyr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasSsyr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyr2StridedBatched_full_rank,&
      cublasSsyr2StridedBatched_rank_0,&
      cublasSsyr2StridedBatched_rank_1
  end interface
  
  interface cublasDsyr2StridedBatched
    function cublasDsyr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasDsyr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyr2StridedBatched_full_rank,&
      cublasDsyr2StridedBatched_rank_0,&
      cublasDsyr2StridedBatched_rank_1
  end interface
  
  interface cublasCsyr2StridedBatched
    function cublasCsyr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasCsyr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyr2StridedBatched_full_rank,&
      cublasCsyr2StridedBatched_rank_0,&
      cublasCsyr2StridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     syr2StridedBatched the matrix-vector operations
  !> 
  !>         A[i] := A[i] + alpha*x[i]*y[i]**T + alpha*y[i]*x[i]**T
  !> 
  !>     where alpha is a scalar, x[i] and y[i] are vectors, and A[i] is a
  !>     n by n symmetric matrices, for i = 1 , ... , batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     n         [int]
  !>               the number of rows and columns of each matrix A.
  !>     @param[in]
  !>     alpha
  !>               device pointer or host pointer to scalar alpha.
  !>     @param[in]
  !>     x         device pointer to the first vector x_1.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex   [hipblasStride]
  !>               specifies the pointer increment between vectors (x_i) and (x_i+1).
  !>     @param[in]
  !>     y         device pointer to the first vector y_1.
  !>     @param[in]
  !>     incy      [int]
  !>               specifies the increment for the elements of each y_i.
  !>     @param[in]
  !>     stridey   [hipblasStride]
  !>               specifies the pointer increment between vectors (y_i) and (y_i+1).
  !>     @param[inout]
  !>     A         device pointer to the first matrix A_1.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     strideA   [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>               number of instances in the batch
  !>
  interface cublasZsyr2StridedBatched
    function cublasZsyr2StridedBatched_(handle,uplo,n,alpha,x,incx,&
&stridex,y,incy,stridey,A,lda,strideA,batchCount) bind(c, &
&name="cublasZsyr2StridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyr2StridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyr2StridedBatched_full_rank,&
      cublasZsyr2StridedBatched_rank_0,&
      cublasZsyr2StridedBatched_rank_1
  end interface
  
  interface cublasStbmv
    function cublasStbmv_(handle,uplo,transA,diag,m,k,A,lda,x,incx) &
&bind(c, name="cublasStbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasStbmv_full_rank,&
      cublasStbmv_rank_0,&
      cublasStbmv_rank_1
  end interface
  
  interface cublasDtbmv
    function cublasDtbmv_(handle,uplo,transA,diag,m,k,A,lda,x,incx) &
&bind(c, name="cublasDtbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasDtbmv_full_rank,&
      cublasDtbmv_rank_0,&
      cublasDtbmv_rank_1
  end interface
  
  interface cublasCtbmv
    function cublasCtbmv_(handle,uplo,transA,diag,m,k,A,lda,x,incx) &
&bind(c, name="cublasCtbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCtbmv_full_rank,&
      cublasCtbmv_rank_0,&
      cublasCtbmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tbmv performs one of the matrix-vector operations
  !> 
  !>         x := A*x      or
  !>         x := A**T*x   or
  !>         x := A**H*x,
  !> 
  !>     x is a vectors and A is a banded m by m matrix (see description below).
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: A is an upper banded triangular matrix.
  !>               HIPBLAS_FILL_MODE_LOWER: A is a  lower banded triangular matrix.
  !>     @param[in]
  !>     trans     [hipblasOperation_t]
  !>               indicates whether matrix A is tranposed (conjugated) or not.
  !>     @param[in]
  !>     diag      [hipblasDiagType_t]
  !>               HIPBLAS_DIAG_UNIT: The main diagonal of A is assumed to consist of only
  !>                                      1's and is not referenced.
  !>               HIPBLAS_DIAG_NON_UNIT: No assumptions are made of A's main diagonal.
  !>     @param[in]
  !>     m         [int]
  !>               the number of rows and columns of the matrix represented by A.
  !>     @param[in]
  !>     k         [int]
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER, k specifies the number of super-diagonals
  !>               of the matrix A.
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER, k specifies the number of sub-diagonals
  !>               of the matrix A.
  !>               k must satisfy k > 0 && k < lda.
  !>     @param[in]
  !>     A         device pointer storing banded triangular matrix A.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The matrix represented is an upper banded triangular matrix
  !>                 with the main diagonal and k super-diagonals, everything
  !>                 else can be assumed to be 0.
  !>                 The matrix is compacted so that the main diagonal resides on the k'th
  !>                 row, the first super diagonal resides on the RHS of the k-1'th row, etc,
  !>                 with the k'th diagonal on the RHS of the 0'th row.
  !>                    Ex: (HIPBLAS_FILL_MODE_UPPER; m = 5; k = 2)
  !>                       1 6 9 0 0              0 0 9 8 7
  !>                       0 2 7 8 0              0 6 7 8 9
  !>                       0 0 3 8 7     ---->    1 2 3 4 5
  !>                       0 0 0 4 9              0 0 0 0 0
  !>                       0 0 0 0 5              0 0 0 0 0
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The matrix represnted is a lower banded triangular matrix
  !>                 with the main diagonal and k sub-diagonals, everything else can be
  !>                 assumed to be 0.
  !>                 The matrix is compacted so that the main diagonal resides on the 0'th row,
  !>                 working up to the k'th diagonal residing on the LHS of the k'th row.
  !>                    Ex: (HIPBLAS_FILL_MODE_LOWER; m = 5; k = 2)
  !>                       1 0 0 0 0              1 2 3 4 5
  !>                       6 2 0 0 0              6 7 8 9 0
  !>                       9 7 3 0 0     ---->    9 8 7 0 0
  !>                       0 8 8 4 0              0 0 0 0 0
  !>                       0 0 7 9 5              0 0 0 0 0
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A. lda must satisfy lda > k.
  !>     @param[inout]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>
  interface cublasZtbmv
    function cublasZtbmv_(handle,uplo,transA,diag,m,k,A,lda,x,incx) &
&bind(c, name="cublasZtbmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtbmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZtbmv_full_rank,&
      cublasZtbmv_rank_0,&
      cublasZtbmv_rank_1
  end interface
  
  interface cublasStbmvBatched
    function cublasStbmvBatched_(handle,uplo,transA,diag,m,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasStbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStbmvBatched_full_rank,&
      cublasStbmvBatched_rank_0,&
      cublasStbmvBatched_rank_1
  end interface
  
  interface cublasDtbmvBatched
    function cublasDtbmvBatched_(handle,uplo,transA,diag,m,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasDtbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtbmvBatched_full_rank,&
      cublasDtbmvBatched_rank_0,&
      cublasDtbmvBatched_rank_1
  end interface
  
  interface cublasCtbmvBatched
    function cublasCtbmvBatched_(handle,uplo,transA,diag,m,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasCtbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtbmvBatched_full_rank,&
      cublasCtbmvBatched_rank_0,&
      cublasCtbmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tbmvBatched performs one of the matrix-vector operations
  !> 
  !>         x_i := A_i*x_i      or
  !>         x_i := A_i**T*x_i   or
  !>         x_i := A_i**H*x_i,
  !> 
  !>     where (A_i, x_i) is the i-th instance of the batch.
  !>     x_i is a vector and A_i is an m by m matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: each A_i is an upper banded triangular matrix.
  !>               HIPBLAS_FILL_MODE_LOWER: each A_i is a  lower banded triangular matrix.
  !>     @param[in]
  !>     trans     [hipblasOperation_t]
  !>               indicates whether each matrix A_i is tranposed (conjugated) or not.
  !>     @param[in]
  !>     diag      [hipblasDiagType_t]
  !>               HIPBLAS_DIAG_UNIT: The main diagonal of each A_i is assumed to consist of only
  !>                                      1's and is not referenced.
  !>               HIPBLAS_DIAG_NON_UNIT: No assumptions are made of each A_i's main diagonal.
  !>     @param[in]
  !>     m         [int]
  !>               the number of rows and columns of the matrix represented by each A_i.
  !>     @param[in]
  !>     k         [int]
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER, k specifies the number of super-diagonals
  !>               of each matrix A_i.
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER, k specifies the number of sub-diagonals
  !>               of each matrix A_i.
  !>               k must satisfy k > 0 && k < lda.
  !>     @param[in]
  !>     A         device array of device pointers storing each banded triangular matrix A_i.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The matrix represented is an upper banded triangular matrix
  !>                 with the main diagonal and k super-diagonals, everything
  !>                 else can be assumed to be 0.
  !>                 The matrix is compacted so that the main diagonal resides on the k'th
  !>                 row, the first super diagonal resides on the RHS of the k-1'th row, etc,
  !>                 with the k'th diagonal on the RHS of the 0'th row.
  !>                    Ex: (HIPBLAS_FILL_MODE_UPPER; m = 5; k = 2)
  !>                       1 6 9 0 0              0 0 9 8 7
  !>                       0 2 7 8 0              0 6 7 8 9
  !>                       0 0 3 8 7     ---->    1 2 3 4 5
  !>                       0 0 0 4 9              0 0 0 0 0
  !>                       0 0 0 0 5              0 0 0 0 0
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The matrix represnted is a lower banded triangular matrix
  !>                 with the main diagonal and k sub-diagonals, everything else can be
  !>                 assumed to be 0.
  !>                 The matrix is compacted so that the main diagonal resides on the 0'th row,
  !>                 working up to the k'th diagonal residing on the LHS of the k'th row.
  !>                    Ex: (HIPBLAS_FILL_MODE_LOWER; m = 5; k = 2)
  !>                       1 0 0 0 0              1 2 3 4 5
  !>                       6 2 0 0 0              6 7 8 9 0
  !>                       9 7 3 0 0     ---->    9 8 7 0 0
  !>                       0 8 8 4 0              0 0 0 0 0
  !>                       0 0 7 9 5              0 0 0 0 0
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. lda must satisfy lda > k.
  !>     @param[inout]
  !>     x         device array of device pointer storing each vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZtbmvBatched
    function cublasZtbmvBatched_(handle,uplo,transA,diag,m,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasZtbmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtbmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtbmvBatched_full_rank,&
      cublasZtbmvBatched_rank_0,&
      cublasZtbmvBatched_rank_1
  end interface
  
  interface cublasStbmvStridedBatched
    function cublasStbmvStridedBatched_(handle,uplo,transA,diag,m,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasStbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStbmvStridedBatched_full_rank,&
      cublasStbmvStridedBatched_rank_0,&
      cublasStbmvStridedBatched_rank_1
  end interface
  
  interface cublasDtbmvStridedBatched
    function cublasDtbmvStridedBatched_(handle,uplo,transA,diag,m,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasDtbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtbmvStridedBatched_full_rank,&
      cublasDtbmvStridedBatched_rank_0,&
      cublasDtbmvStridedBatched_rank_1
  end interface
  
  interface cublasCtbmvStridedBatched
    function cublasCtbmvStridedBatched_(handle,uplo,transA,diag,m,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasCtbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtbmvStridedBatched_full_rank,&
      cublasCtbmvStridedBatched_rank_0,&
      cublasCtbmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tbmvStridedBatched performs one of the matrix-vector operations
  !> 
  !>         x_i := A_i*x_i      or
  !>         x_i := A_i**T*x_i   or
  !>         x_i := A_i**H*x_i,
  !> 
  !>     where (A_i, x_i) is the i-th instance of the batch.
  !>     x_i is a vector and A_i is an m by m matrix, for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               HIPBLAS_FILL_MODE_UPPER: each A_i is an upper banded triangular matrix.
  !>               HIPBLAS_FILL_MODE_LOWER: each A_i is a  lower banded triangular matrix.
  !>     @param[in]
  !>     trans     [hipblasOperation_t]
  !>               indicates whether each matrix A_i is tranposed (conjugated) or not.
  !>     @param[in]
  !>     diag      [hipblasDiagType_t]
  !>               HIPBLAS_DIAG_UNIT: The main diagonal of each A_i is assumed to consist of only
  !>                                      1's and is not referenced.
  !>               HIPBLAS_DIAG_NON_UNIT: No assumptions are made of each A_i's main diagonal.
  !>     @param[in]
  !>     m         [int]
  !>               the number of rows and columns of the matrix represented by each A_i.
  !>     @param[in]
  !>     k         [int]
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER, k specifies the number of super-diagonals
  !>               of each matrix A_i.
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER, k specifies the number of sub-diagonals
  !>               of each matrix A_i.
  !>               k must satisfy k > 0 && k < lda.
  !>     @param[in]
  !>     A         device array to the first matrix A_i of the batch. Stores each banded triangular matrix A_i.
  !>               if uplo == HIPBLAS_FILL_MODE_UPPER:
  !>                 The matrix represented is an upper banded triangular matrix
  !>                 with the main diagonal and k super-diagonals, everything
  !>                 else can be assumed to be 0.
  !>                 The matrix is compacted so that the main diagonal resides on the k'th
  !>                 row, the first super diagonal resides on the RHS of the k-1'th row, etc,
  !>                 with the k'th diagonal on the RHS of the 0'th row.
  !>                    Ex: (HIPBLAS_FILL_MODE_UPPER; m = 5; k = 2)
  !>                       1 6 9 0 0              0 0 9 8 7
  !>                       0 2 7 8 0              0 6 7 8 9
  !>                       0 0 3 8 7     ---->    1 2 3 4 5
  !>                       0 0 0 4 9              0 0 0 0 0
  !>                       0 0 0 0 5              0 0 0 0 0
  !>               if uplo == HIPBLAS_FILL_MODE_LOWER:
  !>                 The matrix represnted is a lower banded triangular matrix
  !>                 with the main diagonal and k sub-diagonals, everything else can be
  !>                 assumed to be 0.
  !>                 The matrix is compacted so that the main diagonal resides on the 0'th row,
  !>                 working up to the k'th diagonal residing on the LHS of the k'th row.
  !>                    Ex: (HIPBLAS_FILL_MODE_LOWER; m = 5; k = 2)
  !>                       1 0 0 0 0              1 2 3 4 5
  !>                       6 2 0 0 0              6 7 8 9 0
  !>                       9 7 3 0 0     ---->    9 8 7 0 0
  !>                       0 8 8 4 0              0 0 0 0 0
  !>                       0 0 7 9 5              0 0 0 0 0
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i. lda must satisfy lda > k.
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one A_i matrix to the next A_(i + 1).
  !>     @param[inout]
  !>     x         device array to the first vector x_i of the batch.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one x_i matrix to the next x_(i + 1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZtbmvStridedBatched
    function cublasZtbmvStridedBatched_(handle,uplo,transA,diag,m,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasZtbmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtbmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtbmvStridedBatched_full_rank,&
      cublasZtbmvStridedBatched_rank_0,&
      cublasZtbmvStridedBatched_rank_1
  end interface
  
  interface cublasStbsv
    function cublasStbsv_(handle,uplo,transA,diag,n,k,A,lda,x,incx) &
&bind(c, name="cublasStbsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStbsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasStbsv_full_rank,&
      cublasStbsv_rank_0,&
      cublasStbsv_rank_1
  end interface
  
  interface cublasDtbsv
    function cublasDtbsv_(handle,uplo,transA,diag,n,k,A,lda,x,incx) &
&bind(c, name="cublasDtbsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtbsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasDtbsv_full_rank,&
      cublasDtbsv_rank_0,&
      cublasDtbsv_rank_1
  end interface
  
  interface cublasCtbsv
    function cublasCtbsv_(handle,uplo,transA,diag,n,k,A,lda,x,incx) &
&bind(c, name="cublasCtbsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtbsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCtbsv_full_rank,&
      cublasCtbsv_rank_0,&
      cublasCtbsv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tbsv solves
  !> 
  !>          A*x = b or A**T*x = b or A**H*x = b,
  !> 
  !>     where x and b are vectors and A is a banded triangular matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !>                HIPBLAS_OP_N: Solves A*x = b
  !>                HIPBLAS_OP_T: Solves A**T*x = b
  !>                HIPBLAS_OP_C: Solves A**H*x = b
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular (i.e. the diagonal elements
  !>                                        of A are not used in computations).
  !>             HIPBLAS_DIAG_NON_UNIT: A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               n specifies the number of rows of b. n >= 0.
  !>     @param[in]
  !>     k         [int]
  !>               if(uplo == HIPBLAS_FILL_MODE_UPPER)
  !>                 k specifies the number of super-diagonals of A.
  !>               if(uplo == HIPBLAS_FILL_MODE_LOWER)
  !>                 k specifies the number of sub-diagonals of A.
  !>               k >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer storing the matrix A in banded format.
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>               lda >= (k + 1).
  !> 
  !>     @param[inout]
  !>     x         device pointer storing input vector b. Overwritten by the output vector x.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>
  interface cublasZtbsv
    function cublasZtbsv_(handle,uplo,transA,diag,n,k,A,lda,x,incx) &
&bind(c, name="cublasZtbsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtbsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZtbsv_full_rank,&
      cublasZtbsv_rank_0,&
      cublasZtbsv_rank_1
  end interface
  
  interface cublasStbsvBatched
    function cublasStbsvBatched_(handle,uplo,transA,diag,n,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasStbsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStbsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStbsvBatched_full_rank,&
      cublasStbsvBatched_rank_0,&
      cublasStbsvBatched_rank_1
  end interface
  
  interface cublasDtbsvBatched
    function cublasDtbsvBatched_(handle,uplo,transA,diag,n,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasDtbsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtbsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtbsvBatched_full_rank,&
      cublasDtbsvBatched_rank_0,&
      cublasDtbsvBatched_rank_1
  end interface
  
  interface cublasCtbsvBatched
    function cublasCtbsvBatched_(handle,uplo,transA,diag,n,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasCtbsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtbsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtbsvBatched_full_rank,&
      cublasCtbsvBatched_rank_0,&
      cublasCtbsvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tbsvBatched solves
  !> 
  !>          A_i*x_i = b_i or A_i**T*x_i = b_i or A_i**H*x_i = b_i,
  !> 
  !>     where x_i and b_i are vectors and A_i is a banded triangular matrix,
  !>     for i = [1, batchCount].
  !> 
  !>     The input vectors b_i are overwritten by the output vectors x_i.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !>                HIPBLAS_OP_N: Solves A_i*x_i = b_i
  !>                HIPBLAS_OP_T: Solves A_i**T*x_i = b_i
  !>                HIPBLAS_OP_C: Solves A_i**H*x_i = b_i
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     each A_i is assumed to be unit triangular (i.e. the diagonal elements
  !>                                        of each A_i are not used in computations).
  !>             HIPBLAS_DIAG_NON_UNIT: each A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               n specifies the number of rows of each b_i. n >= 0.
  !>     @param[in]
  !>     k         [int]
  !>               if(uplo == HIPBLAS_FILL_MODE_UPPER)
  !>                 k specifies the number of super-diagonals of each A_i.
  !>               if(uplo == HIPBLAS_FILL_MODE_LOWER)
  !>                 k specifies the number of sub-diagonals of each A_i.
  !>               k >= 0.
  !> 
  !>     @param[in]
  !>     A         device vector of device pointers storing each matrix A_i in banded format.
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>               lda >= (k + 1).
  !> 
  !>     @param[inout]
  !>     x         device vector of device pointers storing each input vector b_i. Overwritten by each output
  !>               vector x_i.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZtbsvBatched
    function cublasZtbsvBatched_(handle,uplo,transA,diag,n,k,A,lda,x,&
&incx,batchCount) bind(c, name="cublasZtbsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtbsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtbsvBatched_full_rank,&
      cublasZtbsvBatched_rank_0,&
      cublasZtbsvBatched_rank_1
  end interface
  
  interface cublasStbsvStridedBatched
    function cublasStbsvStridedBatched_(handle,uplo,transA,diag,n,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasStbsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStbsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStbsvStridedBatched_full_rank,&
      cublasStbsvStridedBatched_rank_0,&
      cublasStbsvStridedBatched_rank_1
  end interface
  
  interface cublasDtbsvStridedBatched
    function cublasDtbsvStridedBatched_(handle,uplo,transA,diag,n,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasDtbsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtbsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtbsvStridedBatched_full_rank,&
      cublasDtbsvStridedBatched_rank_0,&
      cublasDtbsvStridedBatched_rank_1
  end interface
  
  interface cublasCtbsvStridedBatched
    function cublasCtbsvStridedBatched_(handle,uplo,transA,diag,n,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasCtbsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtbsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtbsvStridedBatched_full_rank,&
      cublasCtbsvStridedBatched_rank_0,&
      cublasCtbsvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tbsvStridedBatched solves
  !> 
  !>          A_i*x_i = b_i or A_i**T*x_i = b_i or A_i**H*x_i = b_i,
  !> 
  !>     where x_i and b_i are vectors and A_i is a banded triangular matrix,
  !>     for i = [1, batchCount].
  !> 
  !>     The input vectors b_i are overwritten by the output vectors x_i.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !>                HIPBLAS_OP_N: Solves A_i*x_i = b_i
  !>                HIPBLAS_OP_T: Solves A_i**T*x_i = b_i
  !>                HIPBLAS_OP_C: Solves A_i**H*x_i = b_i
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     each A_i is assumed to be unit triangular (i.e. the diagonal elements
  !>                                        of each A_i are not used in computations).
  !>             HIPBLAS_DIAG_NON_UNIT: each A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               n specifies the number of rows of each b_i. n >= 0.
  !>     @param[in]
  !>     k         [int]
  !>               if(uplo == HIPBLAS_FILL_MODE_UPPER)
  !>                 k specifies the number of super-diagonals of each A_i.
  !>               if(uplo == HIPBLAS_FILL_MODE_LOWER)
  !>                 k specifies the number of sub-diagonals of each A_i.
  !>               k >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer pointing to the first banded matrix A_1.
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>               lda >= (k + 1).
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               specifies the distance between the start of one matrix (A_i) and the next (A_i+1).
  !> 
  !>     @param[inout]
  !>     x         device pointer pointing to the first input vector b_1. Overwritten by output vectors x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               specifies the distance between the start of one vector (x_i) and the next (x_i+1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZtbsvStridedBatched
    function cublasZtbsvStridedBatched_(handle,uplo,transA,diag,n,k,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasZtbsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtbsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtbsvStridedBatched_full_rank,&
      cublasZtbsvStridedBatched_rank_0,&
      cublasZtbsvStridedBatched_rank_1
  end interface
  
  interface cublasStpmv
    function cublasStpmv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasStpmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStpmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasStpmv_rank_0,&
      cublasStpmv_rank_1
  end interface
  
  interface cublasDtpmv
    function cublasDtpmv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasDtpmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtpmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasDtpmv_rank_0,&
      cublasDtpmv_rank_1
  end interface
  
  interface cublasCtpmv
    function cublasCtpmv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasCtpmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtpmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCtpmv_rank_0,&
      cublasCtpmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tpmv performs one of the matrix-vector operations
  !> 
  !>          x = A*x or x = A**T*x,
  !> 
  !>     where x is an n element vector and A is an n by n unit, or non-unit, upper or lower triangular matrix, supplied in the pack form.
  !> 
  !>     The vector x is overwritten.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of A. m >= 0.
  !> 
  !>     @param[in]
  !>     A       device pointer storing matrix A,
  !>             of dimension at leat ( m * ( m + 1 ) 2 ).
  !>           Before entry with uplo = HIPBLAS_FILL_MODE_UPPER, the array A
  !>           must contain the upper triangular matrix packed sequentially,
  !>           column by column, so that A[0] contains a_{0,0}, A[1] and A[2] contain
  !>           a_{0,1} and a_{1, 1} respectively, and so on.
  !>           Before entry with uplo = HIPBLAS_FILL_MODE_LOWER, the array A
  !>           must contain the lower triangular matrix packed sequentially,
  !>           column by column, so that A[0] contains a_{0,0}, A[1] and A[2] contain
  !>           a_{1,0} and a_{2,0} respectively, and so on.
  !>           Note that when DIAG = HIPBLAS_DIAG_UNIT, the diagonal elements of A are
  !>           not referenced, but are assumed to be unity.
  !> 
  !>     @param[in]
  !>     x       device pointer storing vector x.
  !> 
  !>     @param[in]
  !>     incx    [int]
  !>             specifies the increment for the elements of x. incx must not be zero.
  !>
  interface cublasZtpmv
    function cublasZtpmv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasZtpmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtpmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZtpmv_rank_0,&
      cublasZtpmv_rank_1
  end interface
  
  interface cublasStpmvBatched
    function cublasStpmvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasStpmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStpmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStpmvBatched_full_rank,&
      cublasStpmvBatched_rank_0,&
      cublasStpmvBatched_rank_1
  end interface
  
  interface cublasDtpmvBatched
    function cublasDtpmvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasDtpmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtpmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtpmvBatched_full_rank,&
      cublasDtpmvBatched_rank_0,&
      cublasDtpmvBatched_rank_1
  end interface
  
  interface cublasCtpmvBatched
    function cublasCtpmvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasCtpmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtpmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtpmvBatched_full_rank,&
      cublasCtpmvBatched_rank_0,&
      cublasCtpmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tpmvBatched performs one of the matrix-vector operations
  !> 
  !>          x_i = A_i*x_i or x_i = A**T*x_i, 0 \le i < batchCount
  !> 
  !>     where x_i is an n element vector and A_i is an n by n (unit, or non-unit, upper or lower triangular matrix)
  !> 
  !>     The vectors x_i are overwritten.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of matrices A_i. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer storing pointer of matrices A_i,
  !>               of dimension ( lda, m )
  !> 
  !>     @param[in]
  !>     x         device pointer storing vectors x_i.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of vectors x_i.
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>               The number of batched matrices/vectors.
  !> 
  !>
  interface cublasZtpmvBatched
    function cublasZtpmvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasZtpmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtpmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtpmvBatched_full_rank,&
      cublasZtpmvBatched_rank_0,&
      cublasZtpmvBatched_rank_1
  end interface
  
  interface cublasStpmvStridedBatched
    function cublasStpmvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stride,batchCount) bind(c, &
&name="cublasStpmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStpmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stride
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStpmvStridedBatched_rank_0,&
      cublasStpmvStridedBatched_rank_1
  end interface
  
  interface cublasDtpmvStridedBatched
    function cublasDtpmvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stride,batchCount) bind(c, &
&name="cublasDtpmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtpmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stride
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtpmvStridedBatched_rank_0,&
      cublasDtpmvStridedBatched_rank_1
  end interface
  
  interface cublasCtpmvStridedBatched
    function cublasCtpmvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stride,batchCount) bind(c, &
&name="cublasCtpmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtpmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stride
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtpmvStridedBatched_rank_0,&
      cublasCtpmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tpmvStridedBatched performs one of the matrix-vector operations
  !> 
  !>          x_i = A_i*x_i or x_i = A**T*x_i, 0 \le i < batchCount
  !> 
  !>     where x_i is an n element vector and A_i is an n by n (unit, or non-unit, upper or lower triangular matrix)
  !>     with strides specifying how to retrieve $x_i$ (resp. $A_i$) from $x_{i-1}$ (resp. $A_i$).
  !> 
  !>     The vectors x_i are overwritten.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of matrices A_i. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer of the matrix A_0,
  !>               of dimension ( lda, m )
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one A_i matrix to the next A_{i + 1}
  !> 
  !>     @param[in]
  !>     x         device pointer storing the vector x_0.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of one vector x.
  !> 
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one x_i vector to the next x_{i + 1}
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>               The number of batched matrices/vectors.
  !> 
  !>
  interface cublasZtpmvStridedBatched
    function cublasZtpmvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stride,batchCount) bind(c, &
&name="cublasZtpmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtpmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stride
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtpmvStridedBatched_rank_0,&
      cublasZtpmvStridedBatched_rank_1
  end interface
  
  interface cublasStpsv
    function cublasStpsv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasStpsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStpsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasStpsv_rank_0,&
      cublasStpsv_rank_1
  end interface
  
  interface cublasDtpsv
    function cublasDtpsv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasDtpsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtpsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasDtpsv_rank_0,&
      cublasDtpsv_rank_1
  end interface
  
  interface cublasCtpsv
    function cublasCtpsv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasCtpsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtpsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCtpsv_rank_0,&
      cublasCtpsv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tpsv solves
  !> 
  !>          A*x = b or A**T*x = b, or A**H*x = b,
  !> 
  !>     where x and b are vectors and A is a triangular matrix stored in the packed format.
  !> 
  !>     The input vector b is overwritten by the output vector x.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_N: Solves A*x = b
  !>             HIPBLAS_OP_T: Solves A**T*x = b
  !>             HIPBLAS_OP_C: Solves A**H*x = b
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular (i.e. the diagonal elements
  !>                                        of A are not used in computations).
  !>             HIPBLAS_DIAG_NON_UNIT: A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               n specifies the number of rows of b. n >= 0.
  !> 
  !>     @param[in]
  !>     AP        device pointer storing the packed version of matrix A,
  !>               of dimension >= (n * (n + 1) 2)
  !> 
  !>     @param[inout]
  !>     x         device pointer storing vector b on input, overwritten by x on output.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>
  interface cublasZtpsv
    function cublasZtpsv_(handle,uplo,transA,diag,m,AP,x,incx) bind(c, &
&name="cublasZtpsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtpsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZtpsv_rank_0,&
      cublasZtpsv_rank_1
  end interface
  
  interface cublasStpsvBatched
    function cublasStpsvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasStpsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStpsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStpsvBatched_full_rank,&
      cublasStpsvBatched_rank_0,&
      cublasStpsvBatched_rank_1
  end interface
  
  interface cublasDtpsvBatched
    function cublasDtpsvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasDtpsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtpsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtpsvBatched_full_rank,&
      cublasDtpsvBatched_rank_0,&
      cublasDtpsvBatched_rank_1
  end interface
  
  interface cublasCtpsvBatched
    function cublasCtpsvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasCtpsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtpsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtpsvBatched_full_rank,&
      cublasCtpsvBatched_rank_0,&
      cublasCtpsvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tpsvBatched solves
  !> 
  !>          A_i*x_i = b_i or A_i**T*x_i = b_i, or A_i**H*x_i = b_i,
  !> 
  !>     where x_i and b_i are vectors and A_i is a triangular matrix stored in the packed format,
  !>     for i in [1, batchCount].
  !> 
  !>     The input vectors b_i are overwritten by the output vectors x_i.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  each A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  each A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_N: Solves A*x = b
  !>             HIPBLAS_OP_T: Solves A**T*x = b
  !>             HIPBLAS_OP_C: Solves A**H*x = b
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     each A_i is assumed to be unit triangular (i.e. the diagonal elements
  !>                                        of each A_i are not used in computations).
  !>             HIPBLAS_DIAG_NON_UNIT: each A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               n specifies the number of rows of each b_i. n >= 0.
  !> 
  !>     @param[in]
  !>     AP        device array of device pointers storing the packed versions of each matrix A_i,
  !>               of dimension >= (n * (n + 1) 2)
  !> 
  !>     @param[inout]
  !>     x         device array of device pointers storing each input vector b_i, overwritten by x_i on output.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 specifies the number of instances in the batch.
  !>
  interface cublasZtpsvBatched
    function cublasZtpsvBatched_(handle,uplo,transA,diag,m,AP,x,incx,&
&batchCount) bind(c, name="cublasZtpsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtpsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: AP
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtpsvBatched_full_rank,&
      cublasZtpsvBatched_rank_0,&
      cublasZtpsvBatched_rank_1
  end interface
  
  interface cublasStpsvStridedBatched
    function cublasStpsvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stridex,batchCount) bind(c, &
&name="cublasStpsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStpsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStpsvStridedBatched_rank_0,&
      cublasStpsvStridedBatched_rank_1
  end interface
  
  interface cublasDtpsvStridedBatched
    function cublasDtpsvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stridex,batchCount) bind(c, &
&name="cublasDtpsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtpsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtpsvStridedBatched_rank_0,&
      cublasDtpsvStridedBatched_rank_1
  end interface
  
  interface cublasCtpsvStridedBatched
    function cublasCtpsvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stridex,batchCount) bind(c, &
&name="cublasCtpsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtpsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtpsvStridedBatched_rank_0,&
      cublasCtpsvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     tpsvStridedBatched solves
  !> 
  !>          A_i*x_i = b_i or A_i**T*x_i = b_i, or A_i**H*x_i = b_i,
  !> 
  !>     where x_i and b_i are vectors and A_i is a triangular matrix stored in the packed format,
  !>     for i in [1, batchCount].
  !> 
  !>     The input vectors b_i are overwritten by the output vectors x_i.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  each A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  each A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_N: Solves A*x = b
  !>             HIPBLAS_OP_T: Solves A**T*x = b
  !>             HIPBLAS_OP_C: Solves A**H*x = b
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     each A_i is assumed to be unit triangular (i.e. the diagonal elements
  !>                                        of each A_i are not used in computations).
  !>             HIPBLAS_DIAG_NON_UNIT: each A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               n specifies the number of rows of each b_i. n >= 0.
  !> 
  !>     @param[in]
  !>     AP        device pointer pointing to the first packed matrix A_1,
  !>               of dimension >= (n * (n + 1) 2)
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the beginning of one packed matrix (AP_i) and the next (AP_i+1).
  !> 
  !>     @param[inout]
  !>     x         device pointer pointing to the first input vector b_1. Overwritten by each x_i on output.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the beginning of one vector (x_i) and the next (x_i+1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 specifies the number of instances in the batch.
  !>
  interface cublasZtpsvStridedBatched
    function cublasZtpsvStridedBatched_(handle,uplo,transA,diag,m,AP,&
&strideAP,x,incx,stridex,batchCount) bind(c, &
&name="cublasZtpsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtpsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: AP
      integer(c_int64_t),value :: strideAP
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtpsvStridedBatched_rank_0,&
      cublasZtpsvStridedBatched_rank_1
  end interface
  
  interface cublasStrmv
    function cublasStrmv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasStrmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasStrmv_full_rank,&
      cublasStrmv_rank_0,&
      cublasStrmv_rank_1
  end interface
  
  interface cublasDtrmv
    function cublasDtrmv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasDtrmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasDtrmv_full_rank,&
      cublasDtrmv_rank_0,&
      cublasDtrmv_rank_1
  end interface
  
  interface cublasCtrmv
    function cublasCtrmv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasCtrmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCtrmv_full_rank,&
      cublasCtrmv_rank_0,&
      cublasCtrmv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     trmv performs one of the matrix-vector operations
  !> 
  !>          x = A*x or x = A**T*x,
  !> 
  !>     where x is an n element vector and A is an n by n unit, or non-unit, upper or lower triangular matrix.
  !> 
  !>     The vector x is overwritten.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of A. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer storing matrix A,
  !>               of dimension ( lda, m )
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>               lda = max( 1, m ).
  !> 
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>
  interface cublasZtrmv
    function cublasZtrmv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasZtrmv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrmv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZtrmv_full_rank,&
      cublasZtrmv_rank_0,&
      cublasZtrmv_rank_1
  end interface
  
  interface cublasStrmvBatched
    function cublasStrmvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasStrmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrmvBatched_full_rank,&
      cublasStrmvBatched_rank_0,&
      cublasStrmvBatched_rank_1
  end interface
  
  interface cublasDtrmvBatched
    function cublasDtrmvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasDtrmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrmvBatched_full_rank,&
      cublasDtrmvBatched_rank_0,&
      cublasDtrmvBatched_rank_1
  end interface
  
  interface cublasCtrmvBatched
    function cublasCtrmvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasCtrmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrmvBatched_full_rank,&
      cublasCtrmvBatched_rank_0,&
      cublasCtrmvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     trmvBatched performs one of the matrix-vector operations
  !> 
  !>          x_i = A_i*x_i or x_i = A**T*x_i, 0 \le i < batchCount
  !> 
  !>     where x_i is an n element vector and A_i is an n by n (unit, or non-unit, upper or lower triangular matrix)
  !> 
  !>     The vectors x_i are overwritten.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of matrices A_i. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer storing pointer of matrices A_i,
  !>               of dimension ( lda, m )
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A_i.
  !>               lda >= max( 1, m ).
  !> 
  !>     @param[in]
  !>     x         device pointer storing vectors x_i.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of vectors x_i.
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>               The number of batched matrices/vectors.
  !> 
  !>
  interface cublasZtrmvBatched
    function cublasZtrmvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasZtrmvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrmvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrmvBatched_full_rank,&
      cublasZtrmvBatched_rank_0,&
      cublasZtrmvBatched_rank_1
  end interface
  
  interface cublasStrmvStridedBatched
    function cublasStrmvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasStrmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrmvStridedBatched_full_rank,&
      cublasStrmvStridedBatched_rank_0,&
      cublasStrmvStridedBatched_rank_1
  end interface
  
  interface cublasDtrmvStridedBatched
    function cublasDtrmvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasDtrmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrmvStridedBatched_full_rank,&
      cublasDtrmvStridedBatched_rank_0,&
      cublasDtrmvStridedBatched_rank_1
  end interface
  
  interface cublasCtrmvStridedBatched
    function cublasCtrmvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasCtrmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrmvStridedBatched_full_rank,&
      cublasCtrmvStridedBatched_rank_0,&
      cublasCtrmvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     trmvStridedBatched performs one of the matrix-vector operations
  !> 
  !>          x_i = A_i*x_i or x_i = A**T*x_i, 0 \le i < batchCount
  !> 
  !>     where x_i is an n element vector and A_i is an n by n (unit, or non-unit, upper or lower triangular matrix)
  !>     with strides specifying how to retrieve $x_i$ (resp. $A_i$) from $x_{i-1}$ (resp. $A_i$).
  !> 
  !>     The vectors x_i are overwritten.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of matrices A_i. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer of the matrix A_0,
  !>               of dimension ( lda, m )
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A_i.
  !>               lda >= max( 1, m ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one A_i matrix to the next A_{i + 1}
  !> 
  !>     @param[in]
  !>     x         device pointer storing the vector x_0.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of one vector x.
  !> 
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one x_i vector to the next x_{i + 1}
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>               The number of batched matrices/vectors.
  !> 
  !>
  interface cublasZtrmvStridedBatched
    function cublasZtrmvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasZtrmvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrmvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrmvStridedBatched_full_rank,&
      cublasZtrmvStridedBatched_rank_0,&
      cublasZtrmvStridedBatched_rank_1
  end interface
  
  interface cublasStrsv
    function cublasStrsv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasStrsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasStrsv_full_rank,&
      cublasStrsv_rank_0,&
      cublasStrsv_rank_1
  end interface
  
  interface cublasDtrsv
    function cublasDtrsv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasDtrsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasDtrsv_full_rank,&
      cublasDtrsv_rank_0,&
      cublasDtrsv_rank_1
  end interface
  
  interface cublasCtrsv
    function cublasCtrsv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasCtrsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasCtrsv_full_rank,&
      cublasCtrsv_rank_0,&
      cublasCtrsv_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     trsv solves
  !> 
  !>          A*x = b or A**T*x = b,
  !> 
  !>     where x and b are vectors and A is a triangular matrix.
  !> 
  !>     The vector x is overwritten on b.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of b. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer storing matrix A,
  !>               of dimension ( lda, m )
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>               lda = max( 1, m ).
  !> 
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !>
  interface cublasZtrsv
    function cublasZtrsv_(handle,uplo,transA,diag,m,A,lda,x,incx) &
&bind(c, name="cublasZtrsv")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrsv_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
    end function

    module procedure &
      cublasZtrsv_full_rank,&
      cublasZtrsv_rank_0,&
      cublasZtrsv_rank_1
  end interface
  
  interface cublasStrsvBatched
    function cublasStrsvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasStrsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrsvBatched_full_rank,&
      cublasStrsvBatched_rank_0,&
      cublasStrsvBatched_rank_1
  end interface
  
  interface cublasDtrsvBatched
    function cublasDtrsvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasDtrsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrsvBatched_full_rank,&
      cublasDtrsvBatched_rank_0,&
      cublasDtrsvBatched_rank_1
  end interface
  
  interface cublasCtrsvBatched
    function cublasCtrsvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasCtrsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrsvBatched_full_rank,&
      cublasCtrsvBatched_rank_0,&
      cublasCtrsvBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     trsvBatched solves
  !> 
  !>          A_i*x_i = b_i or A_i**T*x_i = b_i,
  !> 
  !>     where (A_i, x_i, b_i) is the i-th instance of the batch.
  !>     x_i and b_i are vectors and A_i is an
  !>     m by m triangular matrix.
  !> 
  !>     The vector x is overwritten on b.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of b. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i.
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>               lda = max(1, m)
  !> 
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i.
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of x.
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZtrsvBatched
    function cublasZtrsvBatched_(handle,uplo,transA,diag,m,A,lda,x,&
&incx,batchCount) bind(c, name="cublasZtrsvBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrsvBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrsvBatched_full_rank,&
      cublasZtrsvBatched_rank_0,&
      cublasZtrsvBatched_rank_1
  end interface
  
  interface cublasStrsvStridedBatched
    function cublasStrsvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasStrsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrsvStridedBatched_full_rank,&
      cublasStrsvStridedBatched_rank_0,&
      cublasStrsvStridedBatched_rank_1
  end interface
  
  interface cublasDtrsvStridedBatched
    function cublasDtrsvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasDtrsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrsvStridedBatched_full_rank,&
      cublasDtrsvStridedBatched_rank_0,&
      cublasDtrsvStridedBatched_rank_1
  end interface
  
  interface cublasCtrsvStridedBatched
    function cublasCtrsvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasCtrsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrsvStridedBatched_full_rank,&
      cublasCtrsvStridedBatched_rank_0,&
      cublasCtrsvStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 2 API
  !> 
  !>     \details
  !>     trsvStridedBatched solves
  !> 
  !>          A_i*x_i = b_i or A_i**T*x_i = b_i,
  !> 
  !>     where (A_i, x_i, b_i) is the i-th instance of the batch.
  !>     x_i and b_i are vectors and A_i is an m by m triangular matrix, for i = 1, ..., batchCount.
  !> 
  !>     The vector x is overwritten on b.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA     [hipblasOperation_t]
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               m specifies the number of rows of each b_i. m >= 0.
  !> 
  !>     @param[in]
  !>     A         device pointer to the first matrix (A_1) in the batch, of dimension ( lda, m )
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one A_i matrix to the next A_(i + 1)
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>               lda = max( 1, m ).
  !> 
  !>     @param[in, out]
  !>     x         device pointer to the first vector (x_1) in the batch.
  !> 
  !>     @param[in]
  !>     stridex [hipblasStride]
  !>              stride from the start of one x_i vector to the next x_(i + 1)
  !> 
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment for the elements of each x_i.
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZtrsvStridedBatched
    function cublasZtrsvStridedBatched_(handle,uplo,transA,diag,m,A,&
&lda,strideA,x,incx,stridex,batchCount) bind(c, &
&name="cublasZtrsvStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrsvStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrsvStridedBatched_full_rank,&
      cublasZtrsvStridedBatched_rank_0,&
      cublasZtrsvStridedBatched_rank_1
  end interface
  
  interface cublasSgemm
    function cublasSgemm_(handle,transa,transb,m,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasSgemm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgemm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasSgemm_full_rank,&
      cublasSgemm_rank_0,&
      cublasSgemm_rank_1
  end interface
  
  interface cublasDgemm
    function cublasDgemm_(handle,transa,transb,m,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasDgemm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgemm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasDgemm_full_rank,&
      cublasDgemm_rank_0,&
      cublasDgemm_rank_1
  end interface
  
  interface cublasCgemm
    function cublasCgemm_(handle,transa,transb,m,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasCgemm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgemm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCgemm_full_rank,&
      cublasCgemm_rank_0,&
      cublasCgemm_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     gemm performs one of the matrix-matrix operations
  !> 
  !>         C = alpha*op( A )*op( B ) + beta*C,
  !> 
  !>     where op( X ) is one of
  !> 
  !>         op( X ) = X      or
  !>         op( X ) = X**T   or
  !>         op( X ) = X**H,
  !> 
  !>     alpha and beta are scalars, and A, B and C are matrices, with
  !>     op( A ) an m by k matrix, op( B ) a k by n matrix and C an m by n matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     transA    [hipblasOperation_t]
  !>               specifies the form of op( A )
  !>     @param[in]
  !>     transB    [hipblasOperation_t]
  !>               specifies the form of op( B )
  !>     @param[in]
  !>     m         [int]
  !>               number or rows of matrices op( A ) and C
  !>     @param[in]
  !>     n         [int]
  !>               number of columns of matrices op( B ) and C
  !>     @param[in]
  !>     k         [int]
  !>               number of columns of matrix op( A ) and number of rows of matrix op( B )
  !>     @param[in]
  !>     alpha     device pointer or host pointer specifying the scalar alpha.
  !>     @param[in]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>     @param[in]
  !>     B         device pointer storing matrix B.
  !>     @param[in]
  !>     ldb       [int]
  !>               specifies the leading dimension of B.
  !>     @param[in]
  !>     beta      device pointer or host pointer specifying the scalar beta.
  !>     @param[in, out]
  !>     C         device pointer storing matrix C on the GPU.
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of C.
  !>
  interface cublasZgemm
    function cublasZgemm_(handle,transa,transb,m,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasZgemm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgemm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZgemm_full_rank,&
      cublasZgemm_rank_0,&
      cublasZgemm_rank_1
  end interface
  
  interface cublasSgemmBatched
    function cublasSgemmBatched_(handle,transa,transb,m,n,k,alpha,A,&
&lda,B,ldb,beta,C,ldc,batchCount) bind(c, name="cublasSgemmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgemmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgemmBatched_full_rank,&
      cublasSgemmBatched_rank_0,&
      cublasSgemmBatched_rank_1
  end interface
  
  interface cublasDgemmBatched
    function cublasDgemmBatched_(handle,transa,transb,m,n,k,alpha,A,&
&lda,B,ldb,beta,C,ldc,batchCount) bind(c, name="cublasDgemmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgemmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgemmBatched_full_rank,&
      cublasDgemmBatched_rank_0,&
      cublasDgemmBatched_rank_1
  end interface
  
  interface cublasCgemmBatched
    function cublasCgemmBatched_(handle,transa,transb,m,n,k,alpha,A,&
&lda,B,ldb,beta,C,ldc,batchCount) bind(c, name="cublasCgemmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgemmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgemmBatched_full_rank,&
      cublasCgemmBatched_rank_0,&
      cublasCgemmBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !>      \details
  !>     gemmBatched performs one of the batched matrix-matrix operations
  !>          C_i = alpha*op( A_i )*op( B_i ) + beta*C_i, for i = 1, ..., batchCount.
  !>      where op( X ) is one of
  !>          op( X ) = X      or
  !>         op( X ) = X**T   or
  !>         op( X ) = X**H,
  !>      alpha and beta are scalars, and A, B and C are strided batched matrices, with
  !>     op( A ) an m by k by batchCount strided_batched matrix,
  !>     op( B ) an k by n by batchCount strided_batched matrix and
  !>     C an m by n by batchCount strided_batched matrix.
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     transA    [hipblasOperation_t]
  !>               specifies the form of op( A )
  !>     @param[in]
  !>     transB    [hipblasOperation_t]
  !>               specifies the form of op( B )
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimention m.
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimention n.
  !>     @param[in]
  !>     k         [int]
  !>               matrix dimention k.
  !>     @param[in]
  !>     alpha     device pointer or host pointer specifying the scalar alpha.
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     B         device array of device pointers storing each matrix B_i.
  !>     @param[in]
  !>     ldb       [int]
  !>               specifies the leading dimension of each B_i.
  !>     @param[in]
  !>     beta      device pointer or host pointer specifying the scalar beta.
  !>     @param[in, out]
  !>     C         device array of device pointers storing each matrix C_i.
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of each C_i.
  !>     @param[in]
  !>     batchCount
  !>               [int]
  !>               number of gemm operations in the batch
  interface cublasZgemmBatched
    function cublasZgemmBatched_(handle,transa,transb,m,n,k,alpha,A,&
&lda,B,ldb,beta,C,ldc,batchCount) bind(c, name="cublasZgemmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgemmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgemmBatched_full_rank,&
      cublasZgemmBatched_rank_0,&
      cublasZgemmBatched_rank_1
  end interface
  
  interface cublasSgemmStridedBatched
    function cublasSgemmStridedBatched_(handle,transa,transb,m,n,k,&
&alpha,A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) &
&bind(c, name="cublasSgemmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgemmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_long_long),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_long_long),value :: strideB
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_long_long),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgemmStridedBatched_full_rank,&
      cublasSgemmStridedBatched_rank_0,&
      cublasSgemmStridedBatched_rank_1
  end interface
  
  interface cublasDgemmStridedBatched
    function cublasDgemmStridedBatched_(handle,transa,transb,m,n,k,&
&alpha,A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) &
&bind(c, name="cublasDgemmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgemmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_long_long),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_long_long),value :: strideB
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_long_long),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgemmStridedBatched_full_rank,&
      cublasDgemmStridedBatched_rank_0,&
      cublasDgemmStridedBatched_rank_1
  end interface
  
  interface cublasCgemmStridedBatched
    function cublasCgemmStridedBatched_(handle,transa,transb,m,n,k,&
&alpha,A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) &
&bind(c, name="cublasCgemmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgemmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_long_long),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_long_long),value :: strideB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_long_long),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgemmStridedBatched_full_rank,&
      cublasCgemmStridedBatched_rank_0,&
      cublasCgemmStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     gemmStridedBatched performs one of the strided batched matrix-matrix operations
  !> 
  !>         C_i = alpha*op( A_i )*op( B_i ) + beta*C_i, for i = 1, ..., batchCount.
  !> 
  !>     where op( X ) is one of
  !> 
  !>         op( X ) = X      or
  !>         op( X ) = X**T   or
  !>         op( X ) = X**H,
  !> 
  !>     alpha and beta are scalars, and A, B and C are strided batched matrices, with
  !>     op( A ) an m by k by batchCount strided_batched matrix,
  !>     op( B ) an k by n by batchCount strided_batched matrix and
  !>     C an m by n by batchCount strided_batched matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     transA    [hipblasOperation_t]
  !>               specifies the form of op( A )
  !>     @param[in]
  !>     transB    [hipblasOperation_t]
  !>               specifies the form of op( B )
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimention m.
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimention n.
  !>     @param[in]
  !>     k         [int]
  !>               matrix dimention k.
  !>     @param[in]
  !>     alpha     device pointer or host pointer specifying the scalar alpha.
  !>     @param[in]
  !>     A         device pointer pointing to the first matrix A_1.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one A_i matrix to the next A_(i + 1).
  !>     @param[in]
  !>     B         device pointer pointing to the first matrix B_1.
  !>     @param[in]
  !>     ldb       [int]
  !>               specifies the leading dimension of each B_i.
  !>     @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one B_i matrix to the next B_(i + 1).
  !>     @param[in]
  !>     beta      device pointer or host pointer specifying the scalar beta.
  !>     @param[in, out]
  !>     C         device pointer pointing to the first matrix C_1.
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of each C_i.
  !>     @param[in]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one C_i matrix to the next C_(i + 1).
  !>     @param[in]
  !>     batchCount
  !>               [int]
  !>               number of gemm operatons in the batch
  !>
  interface cublasZgemmStridedBatched
    function cublasZgemmStridedBatched_(handle,transa,transb,m,n,k,&
&alpha,A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) &
&bind(c, name="cublasZgemmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgemmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_long_long),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_long_long),value :: strideB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_long_long),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgemmStridedBatched_full_rank,&
      cublasZgemmStridedBatched_rank_0,&
      cublasZgemmStridedBatched_rank_1
  end interface
  
  interface cublasCherk
    function cublasCherk_(handle,uplo,transA,n,k,alpha,A,lda,beta,C,&
&ldc) bind(c, name="cublasCherk")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCherk_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCherk_full_rank,&
      cublasCherk_rank_0,&
      cublasCherk_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     herk performs one of the matrix-matrix operations for a Hermitian rank-k update
  !> 
  !>     C := alpha*op( A )*op( A )^H + beta*C
  !> 
  !>     where  alpha and beta are scalars, op(A) is an n by k matrix, and
  !>     C is a n x n Hermitian matrix stored as either upper or lower.
  !> 
  !>         op( A ) = A,  and A is n by k if transA == HIPBLAS_OP_N
  !>         op( A ) = A^H and A is k by n if transA == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_C:  op(A) = A^H
  !>             HIPBLAS_ON_N:  op(A) = A
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             Martrix dimension is ( lda, k ) when if transA = HIPBLAS_OP_N, otherwise (lda, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if transA = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>
  interface cublasZherk
    function cublasZherk_(handle,uplo,transA,n,k,alpha,A,lda,beta,C,&
&ldc) bind(c, name="cublasZherk")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZherk_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZherk_full_rank,&
      cublasZherk_rank_0,&
      cublasZherk_rank_1
  end interface
  
  interface cublasCherkBatched
    function cublasCherkBatched_(handle,uplo,transA,n,k,alpha,A,lda,&
&beta,C,ldc,batchCount) bind(c, name="cublasCherkBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCherkBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCherkBatched_full_rank,&
      cublasCherkBatched_rank_0,&
      cublasCherkBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     herkBatched performs a batch of the matrix-matrix operations for a Hermitian rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( A_i )^H + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A) is an n by k matrix, and
  !>     C_i is a n x n Hermitian matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, and A_i is n by k if transA == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^H and A_i is k by n if transA == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_C: op(A) = A^H
  !>             HIPBLAS_OP_N: op(A) = A
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix_i A of dimension (lda, k)
  !>             when transA is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if transA = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZherkBatched
    function cublasZherkBatched_(handle,uplo,transA,n,k,alpha,A,lda,&
&beta,C,ldc,batchCount) bind(c, name="cublasZherkBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZherkBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZherkBatched_full_rank,&
      cublasZherkBatched_rank_0,&
      cublasZherkBatched_rank_1
  end interface
  
  interface cublasCherkStridedBatched
    function cublasCherkStridedBatched_(handle,uplo,transA,n,k,alpha,A,&
&lda,strideA,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasCherkStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCherkStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCherkStridedBatched_full_rank,&
      cublasCherkStridedBatched_rank_0,&
      cublasCherkStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     herkStridedBatched performs a batch of the matrix-matrix operations for a Hermitian rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( A_i )^H + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A) is an n by k matrix, and
  !>     C_i is a n x n Hermitian matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, and A_i is n by k if transA == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^H and A_i is k by n if transA == HIPBLAS_OP_C
  !> 
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_C: op(A) = A^H
  !>             HIPBLAS_OP_N: op(A) = A
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to the first matrix A_1 on the GPU of dimension (lda, k)
  !>             when transA is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if transA = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       Device pointer to the first matrix C_1 on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZherkStridedBatched
    function cublasZherkStridedBatched_(handle,uplo,transA,n,k,alpha,A,&
&lda,strideA,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZherkStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZherkStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZherkStridedBatched_full_rank,&
      cublasZherkStridedBatched_rank_0,&
      cublasZherkStridedBatched_rank_1
  end interface
  
  interface cublasCherkx
    function cublasCherkx_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasCherkx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCherkx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCherkx_full_rank,&
      cublasCherkx_rank_0,&
      cublasCherkx_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     herkx performs one of the matrix-matrix operations for a Hermitian rank-k update
  !> 
  !>     C := alpha*op( A )*op( B )^H + beta*C
  !> 
  !>     where  alpha and beta are scalars, op(A) and op(B) are n by k matrices, and
  !>     C is a n x n Hermitian matrix stored as either upper or lower.
  !>     This routine should only be used when the caller can guarantee that the result of op( A )*op( B )^T will be Hermitian.
  !> 
  !> 
  !>         op( A ) = A, op( B ) = B, and A and B are n by k if trans == HIPBLAS_OP_N
  !>         op( A ) = A^H, op( B ) = B^H,  and A and B are k by n if trans == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_C:  op( A ) = A^H, op( B ) = B^H
  !>             HIPBLAS_OP_N:  op( A ) = A, op( B ) = B
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             Martrix dimension is ( lda, k ) when if trans = HIPBLAS_OP_N, otherwise (lda, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !>     @param[in]
  !>     B       pointer storing matrix B on the GPU.
  !>             Martrix dimension is ( ldb, k ) when if trans = HIPBLAS_OP_N, otherwise (ldb, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>
  interface cublasZherkx
    function cublasZherkx_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasZherkx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZherkx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZherkx_full_rank,&
      cublasZherkx_rank_0,&
      cublasZherkx_rank_1
  end interface
  
  interface cublasCherkxBatched
    function cublasCherkxBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasCherkxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCherkxBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCherkxBatched_full_rank,&
      cublasCherkxBatched_rank_0,&
      cublasCherkxBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     herkxBatched performs a batch of the matrix-matrix operations for a Hermitian rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( B_i )^H + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrices, and
  !>     C_i is a n x n Hermitian matrix stored as either upper or lower.
  !>     This routine should only be used when the caller can guarantee that the result of op( A )*op( B )^T will be Hermitian.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^H, op( B_i ) = B_i^H,  and A_i and B_i are k by n if trans == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_C: op(A) = A^H
  !>             HIPBLAS_OP_N: op(A) = A
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix_i A of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     B       device array of device pointers storing each matrix_i B of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZherkxBatched
    function cublasZherkxBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasZherkxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZherkxBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZherkxBatched_full_rank,&
      cublasZherkxBatched_rank_0,&
      cublasZherkxBatched_rank_1
  end interface
  
  interface cublasCherkxStridedBatched
    function cublasCherkxStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasCherkxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCherkxStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCherkxStridedBatched_full_rank,&
      cublasCherkxStridedBatched_rank_0,&
      cublasCherkxStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     herkxStridedBatched performs a batch of the matrix-matrix operations for a Hermitian rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( B_i )^H + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrices, and
  !>     C_i is a n x n Hermitian matrix stored as either upper or lower.
  !>     This routine should only be used when the caller can guarantee that the result of op( A )*op( B )^T will be Hermitian.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^H, op( B_i ) = B_i^H,  and A_i and B_i are k by n if trans == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_C: op( A_i ) = A_i^H, op( B_i ) = B_i^H
  !>             HIPBLAS_OP_N: op( A_i ) = A_i, op( B_i ) = B_i
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to the first matrix A_1 on the GPU of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     B       Device pointer to the first matrix B_1 on the GPU of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       Device pointer to the first matrix C_1 on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZherkxStridedBatched
    function cublasZherkxStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZherkxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZherkxStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZherkxStridedBatched_full_rank,&
      cublasZherkxStridedBatched_rank_0,&
      cublasZherkxStridedBatched_rank_1
  end interface
  
  interface cublasCher2k
    function cublasCher2k_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasCher2k")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCher2k_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCher2k_full_rank,&
      cublasCher2k_rank_0,&
      cublasCher2k_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     her2k performs one of the matrix-matrix operations for a Hermitian rank-2k update
  !> 
  !>     C := alpha*op( A )*op( B )^H + conj(alpha)*op( B )*op( A )^H + beta*C
  !> 
  !>     where  alpha and beta are scalars, op(A) and op(B) are n by k matrices, and
  !>     C is a n x n Hermitian matrix stored as either upper or lower.
  !> 
  !>         op( A ) = A, op( B ) = B, and A and B are n by k if trans == HIPBLAS_OP_N
  !>         op( A ) = A^H, op( B ) = B^H,  and A and B are k by n if trans == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_C:  op( A ) = A^H, op( B ) = B^H
  !>             HIPBLAS_OP_N:  op( A ) = A, op( B ) = B
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             Martrix dimension is ( lda, k ) when if trans = HIPBLAS_OP_N, otherwise (lda, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !>     @param[in]
  !>     B       pointer storing matrix B on the GPU.
  !>             Martrix dimension is ( ldb, k ) when if trans = HIPBLAS_OP_N, otherwise (ldb, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>
  interface cublasZher2k
    function cublasZher2k_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasZher2k")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZher2k_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZher2k_full_rank,&
      cublasZher2k_rank_0,&
      cublasZher2k_rank_1
  end interface
  
  interface cublasCher2kBatched
    function cublasCher2kBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasCher2kBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCher2kBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCher2kBatched_full_rank,&
      cublasCher2kBatched_rank_0,&
      cublasCher2kBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     her2kBatched performs a batch of the matrix-matrix operations for a Hermitian rank-2k update
  !> 
  !>     C_i := alpha*op( A_i )*op( B_i )^H + conj(alpha)*op( B_i )*op( A_i )^H + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrices, and
  !>     C_i is a n x n Hermitian matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^H, op( B_i ) = B_i^H,  and A_i and B_i are k by n if trans == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_C: op(A) = A^H
  !>             HIPBLAS_OP_N: op(A) = A
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix_i A of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !>     @param[in]
  !>     B       device array of device pointers storing each matrix_i B of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZher2kBatched
    function cublasZher2kBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasZher2kBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZher2kBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZher2kBatched_full_rank,&
      cublasZher2kBatched_rank_0,&
      cublasZher2kBatched_rank_1
  end interface
  
  interface cublasCher2kStridedBatched
    function cublasCher2kStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasCher2kStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCher2kStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCher2kStridedBatched_full_rank,&
      cublasCher2kStridedBatched_rank_0,&
      cublasCher2kStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     her2kStridedBatched performs a batch of the matrix-matrix operations for a Hermitian rank-2k update
  !> 
  !>     C_i := alpha*op( A_i )*op( B_i )^H + conj(alpha)*op( B_i )*op( A_i )^H + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrices, and
  !>     C_i is a n x n Hermitian matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^H, op( B_i ) = B_i^H,  and A_i and B_i are k by n if trans == HIPBLAS_OP_C
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_C: op( A_i ) = A_i^H, op( B_i ) = B_i^H
  !>             HIPBLAS_OP_N: op( A_i ) = A_i, op( B_i ) = B_i
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to the first matrix A_1 on the GPU of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     B       Device pointer to the first matrix B_1 on the GPU of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       Device pointer to the first matrix C_1 on the GPU.
  !>             The imaginary component of the diagonal elements are not used but are set to zero unless quick return.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZher2kStridedBatched
    function cublasZher2kStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZher2kStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZher2kStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZher2kStridedBatched_full_rank,&
      cublasZher2kStridedBatched_rank_0,&
      cublasZher2kStridedBatched_rank_1
  end interface
  
  interface cublasSsymm
    function cublasSsymm_(handle,side,uplo,m,n,alpha,A,lda,B,ldb,beta,&
&C,ldc) bind(c, name="cublasSsymm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsymm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasSsymm_full_rank,&
      cublasSsymm_rank_0,&
      cublasSsymm_rank_1
  end interface
  
  interface cublasDsymm
    function cublasDsymm_(handle,side,uplo,m,n,alpha,A,lda,B,ldb,beta,&
&C,ldc) bind(c, name="cublasDsymm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsymm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasDsymm_full_rank,&
      cublasDsymm_rank_0,&
      cublasDsymm_rank_1
  end interface
  
  interface cublasCsymm
    function cublasCsymm_(handle,side,uplo,m,n,alpha,A,lda,B,ldb,beta,&
&C,ldc) bind(c, name="cublasCsymm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsymm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCsymm_full_rank,&
      cublasCsymm_rank_0,&
      cublasCsymm_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     symm performs one of the matrix-matrix operations:
  !> 
  !>     C := alpha*A*B + beta*C if side == HIPBLAS_SIDE_LEFT,
  !>     C := alpha*B*A + beta*C if side == HIPBLAS_SIDE_RIGHT,
  !> 
  !>     where alpha and beta are scalars, B and C are m by n matrices, and
  !>     A is a symmetric matrix stored as either upper or lower.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side  [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:      C := alpha*A*B + beta*C
  !>             HIPBLAS_SIDE_RIGHT:     C := alpha*B*A + beta*C
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B and C. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B and C. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A and B are not referenced.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             A is m by m if side == HIPBLAS_SIDE_LEFT
  !>             A is n by n if side == HIPBLAS_SIDE_RIGHT
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             otherwise lda >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     B       pointer storing matrix B on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B. ldb >= max( 1, m )
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, m )
  !>
  interface cublasZsymm
    function cublasZsymm_(handle,side,uplo,m,n,alpha,A,lda,B,ldb,beta,&
&C,ldc) bind(c, name="cublasZsymm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsymm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZsymm_full_rank,&
      cublasZsymm_rank_0,&
      cublasZsymm_rank_1
  end interface
  
  interface cublasSsymmBatched
    function cublasSsymmBatched_(handle,side,uplo,m,n,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasSsymmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsymmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsymmBatched_full_rank,&
      cublasSsymmBatched_rank_0,&
      cublasSsymmBatched_rank_1
  end interface
  
  interface cublasDsymmBatched
    function cublasDsymmBatched_(handle,side,uplo,m,n,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasDsymmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsymmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsymmBatched_full_rank,&
      cublasDsymmBatched_rank_0,&
      cublasDsymmBatched_rank_1
  end interface
  
  interface cublasCsymmBatched
    function cublasCsymmBatched_(handle,side,uplo,m,n,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasCsymmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsymmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsymmBatched_full_rank,&
      cublasCsymmBatched_rank_0,&
      cublasCsymmBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     symmBatched performs a batch of the matrix-matrix operations:
  !> 
  !>     C_i := alpha*A_i*B_i + beta*C_i if side == HIPBLAS_SIDE_LEFT,
  !>     C_i := alpha*B_i*A_i + beta*C_i if side == HIPBLAS_SIDE_RIGHT,
  !> 
  !>     where alpha and beta are scalars, B_i and C_i are m by n matrices, and
  !>     A_i is a symmetric matrix stored as either upper or lower.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side  [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:      C_i := alpha*A_i*B_i + beta*C_i
  !>             HIPBLAS_SIDE_RIGHT:     C_i := alpha*B_i*A_i + beta*C_i
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B_i and C_i. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B_i and C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A_i and B_i are not referenced.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix A_i on the GPU.
  !>             A_i is m by m if side == HIPBLAS_SIDE_LEFT
  !>             A_i is n by n if side == HIPBLAS_SIDE_RIGHT
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             otherwise lda >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     B       device array of device pointers storing each matrix B_i on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i. ldb >= max( 1, m )
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C_i need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C_i. ldc >= max( 1, m )
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZsymmBatched
    function cublasZsymmBatched_(handle,side,uplo,m,n,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasZsymmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsymmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsymmBatched_full_rank,&
      cublasZsymmBatched_rank_0,&
      cublasZsymmBatched_rank_1
  end interface
  
  interface cublasSsymmStridedBatched
    function cublasSsymmStridedBatched_(handle,side,uplo,m,n,alpha,A,&
&lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasSsymmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsymmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsymmStridedBatched_full_rank,&
      cublasSsymmStridedBatched_rank_0,&
      cublasSsymmStridedBatched_rank_1
  end interface
  
  interface cublasDsymmStridedBatched
    function cublasDsymmStridedBatched_(handle,side,uplo,m,n,alpha,A,&
&lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasDsymmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsymmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsymmStridedBatched_full_rank,&
      cublasDsymmStridedBatched_rank_0,&
      cublasDsymmStridedBatched_rank_1
  end interface
  
  interface cublasCsymmStridedBatched
    function cublasCsymmStridedBatched_(handle,side,uplo,m,n,alpha,A,&
&lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasCsymmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsymmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsymmStridedBatched_full_rank,&
      cublasCsymmStridedBatched_rank_0,&
      cublasCsymmStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     symmStridedBatched performs a batch of the matrix-matrix operations:
  !> 
  !>     C_i := alpha*A_i*B_i + beta*C_i if side == HIPBLAS_SIDE_LEFT,
  !>     C_i := alpha*B_i*A_i + beta*C_i if side == HIPBLAS_SIDE_RIGHT,
  !> 
  !>     where alpha and beta are scalars, B_i and C_i are m by n matrices, and
  !>     A_i is a symmetric matrix stored as either upper or lower.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side  [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:      C_i := alpha*A_i*B_i + beta*C_i
  !>             HIPBLAS_SIDE_RIGHT:     C_i := alpha*B_i*A_i + beta*C_i
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B_i and C_i. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B_i and C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A_i and B_i are not referenced.
  !> 
  !>     @param[in]
  !>     A       device pointer to first matrix A_1
  !>             A_i is m by m if side == HIPBLAS_SIDE_LEFT
  !>             A_i is n by n if side == HIPBLAS_SIDE_RIGHT
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             otherwise lda >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     B       device pointer to first matrix B_1 of dimension (ldb, n) on the GPU.
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i. ldb >= max( 1, m )
  !> 
  !>     @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C        device pointer to first matrix C_1 of dimension (ldc, n) on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, m ).
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZsymmStridedBatched
    function cublasZsymmStridedBatched_(handle,side,uplo,m,n,alpha,A,&
&lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZsymmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsymmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsymmStridedBatched_full_rank,&
      cublasZsymmStridedBatched_rank_0,&
      cublasZsymmStridedBatched_rank_1
  end interface
  
  interface cublasSsyrk
    function cublasSsyrk_(handle,uplo,transA,n,k,alpha,A,lda,beta,C,&
&ldc) bind(c, name="cublasSsyrk")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyrk_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasSsyrk_full_rank,&
      cublasSsyrk_rank_0,&
      cublasSsyrk_rank_1
  end interface
  
  interface cublasDsyrk
    function cublasDsyrk_(handle,uplo,transA,n,k,alpha,A,lda,beta,C,&
&ldc) bind(c, name="cublasDsyrk")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyrk_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasDsyrk_full_rank,&
      cublasDsyrk_rank_0,&
      cublasDsyrk_rank_1
  end interface
  
  interface cublasCsyrk
    function cublasCsyrk_(handle,uplo,transA,n,k,alpha,A,lda,beta,C,&
&ldc) bind(c, name="cublasCsyrk")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyrk_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCsyrk_full_rank,&
      cublasCsyrk_rank_0,&
      cublasCsyrk_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syrk performs one of the matrix-matrix operations for a symmetric rank-k update
  !> 
  !>     C := alpha*op( A )*op( A )^T + beta*C
  !> 
  !>     where  alpha and beta are scalars, op(A) is an n by k matrix, and
  !>     C is a symmetric n x n matrix stored as either upper or lower.
  !> 
  !>         op( A ) = A, and A is n by k if transA == HIPBLAS_OP_N
  !>         op( A ) = A^T and A is k by n if transA == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_T: op(A) = A^T
  !>             HIPBLAS_OP_N: op(A) = A
  !>             HIPBLAS_OP_C: op(A) = A^T
  !> 
  !>             HIPBLAS_OP_C is not supported for complex types, see cherk
  !>             and zherk.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             Martrix dimension is ( lda, k ) when if transA = HIPBLAS_OP_N, otherwise (lda, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if transA = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>
  interface cublasZsyrk
    function cublasZsyrk_(handle,uplo,transA,n,k,alpha,A,lda,beta,C,&
&ldc) bind(c, name="cublasZsyrk")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyrk_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZsyrk_full_rank,&
      cublasZsyrk_rank_0,&
      cublasZsyrk_rank_1
  end interface
  
  interface cublasSsyrkBatched
    function cublasSsyrkBatched_(handle,uplo,transA,n,k,alpha,A,lda,&
&beta,C,ldc,batchCount) bind(c, name="cublasSsyrkBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyrkBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyrkBatched_full_rank,&
      cublasSsyrkBatched_rank_0,&
      cublasSsyrkBatched_rank_1
  end interface
  
  interface cublasDsyrkBatched
    function cublasDsyrkBatched_(handle,uplo,transA,n,k,alpha,A,lda,&
&beta,C,ldc,batchCount) bind(c, name="cublasDsyrkBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyrkBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyrkBatched_full_rank,&
      cublasDsyrkBatched_rank_0,&
      cublasDsyrkBatched_rank_1
  end interface
  
  interface cublasCsyrkBatched
    function cublasCsyrkBatched_(handle,uplo,transA,n,k,alpha,A,lda,&
&beta,C,ldc,batchCount) bind(c, name="cublasCsyrkBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyrkBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      complex(c_float_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyrkBatched_full_rank,&
      cublasCsyrkBatched_rank_0,&
      cublasCsyrkBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syrkBatched performs a batch of the matrix-matrix operations for a symmetric rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( A_i )^T + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) is an n by k matrix, and
  !>     C_i is a symmetric n x n matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, and A_i is n by k if transA == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^T and A_i is k by n if transA == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_T: op(A) = A^T
  !>             HIPBLAS_OP_N: op(A) = A
  !>             HIPBLAS_OP_C: op(A) = A^T
  !> 
  !>             HIPBLAS_OP_C is not supported for complex types, see cherk
  !>             and zherk.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix_i A of dimension (lda, k)
  !>             when transA is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if transA = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZsyrkBatched
    function cublasZsyrkBatched_(handle,uplo,transA,n,k,alpha,A,lda,&
&beta,C,ldc,batchCount) bind(c, name="cublasZsyrkBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyrkBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      complex(c_double_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyrkBatched_full_rank,&
      cublasZsyrkBatched_rank_0,&
      cublasZsyrkBatched_rank_1
  end interface
  
  interface cublasSsyrkStridedBatched
    function cublasSsyrkStridedBatched_(handle,uplo,transA,n,k,alpha,A,&
&lda,strideA,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasSsyrkStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyrkStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyrkStridedBatched_full_rank,&
      cublasSsyrkStridedBatched_rank_0,&
      cublasSsyrkStridedBatched_rank_1
  end interface
  
  interface cublasDsyrkStridedBatched
    function cublasDsyrkStridedBatched_(handle,uplo,transA,n,k,alpha,A,&
&lda,strideA,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasDsyrkStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyrkStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyrkStridedBatched_full_rank,&
      cublasDsyrkStridedBatched_rank_0,&
      cublasDsyrkStridedBatched_rank_1
  end interface
  
  interface cublasCsyrkStridedBatched
    function cublasCsyrkStridedBatched_(handle,uplo,transA,n,k,alpha,A,&
&lda,strideA,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasCsyrkStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyrkStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyrkStridedBatched_full_rank,&
      cublasCsyrkStridedBatched_rank_0,&
      cublasCsyrkStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syrkStridedBatched performs a batch of the matrix-matrix operations for a symmetric rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( A_i )^T + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) is an n by k matrix, and
  !>     C_i is a symmetric n x n matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, and A_i is n by k if transA == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^T and A_i is k by n if transA == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_T: op(A) = A^T
  !>             HIPBLAS_OP_N: op(A) = A
  !>             HIPBLAS_OP_C: op(A) = A^T
  !> 
  !>             HIPBLAS_OP_C is not supported for complex types, see cherk
  !>             and zherk.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to the first matrix A_1 on the GPU of dimension (lda, k)
  !>             when transA is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if transA = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       Device pointer to the first matrix C_1 on the GPU. on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZsyrkStridedBatched
    function cublasZsyrkStridedBatched_(handle,uplo,transA,n,k,alpha,A,&
&lda,strideA,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZsyrkStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyrkStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyrkStridedBatched_full_rank,&
      cublasZsyrkStridedBatched_rank_0,&
      cublasZsyrkStridedBatched_rank_1
  end interface
  
  interface cublasSsyr2k
    function cublasSsyr2k_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasSsyr2k")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyr2k_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasSsyr2k_full_rank,&
      cublasSsyr2k_rank_0,&
      cublasSsyr2k_rank_1
  end interface
  
  interface cublasDsyr2k
    function cublasDsyr2k_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasDsyr2k")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyr2k_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasDsyr2k_full_rank,&
      cublasDsyr2k_rank_0,&
      cublasDsyr2k_rank_1
  end interface
  
  interface cublasCsyr2k
    function cublasCsyr2k_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasCsyr2k")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyr2k_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCsyr2k_full_rank,&
      cublasCsyr2k_rank_0,&
      cublasCsyr2k_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syr2k performs one of the matrix-matrix operations for a symmetric rank-2k update
  !> 
  !>     C := alpha*(op( A )*op( B )^T + op( B )*op( A )^T) + beta*C
  !> 
  !>     where  alpha and beta are scalars, op(A) and op(B) are n by k matrix, and
  !>     C is a symmetric n x n matrix stored as either upper or lower.
  !> 
  !>         op( A ) = A, op( B ) = B, and A and B are n by k if trans == HIPBLAS_OP_N
  !>         op( A ) = A^T, op( B ) = B^T,  and A and B are k by n if trans == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_T:      op( A ) = A^T, op( B ) = B^T
  !>             HIPBLAS_OP_N:           op( A ) = A, op( B ) = B
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A) and op(B). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             Martrix dimension is ( lda, k ) when if trans = HIPBLAS_OP_N, otherwise (lda, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !>     @param[in]
  !>     B       pointer storing matrix B on the GPU.
  !>             Martrix dimension is ( ldb, k ) when if trans = HIPBLAS_OP_N, otherwise (ldb, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>
  interface cublasZsyr2k
    function cublasZsyr2k_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasZsyr2k")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyr2k_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZsyr2k_full_rank,&
      cublasZsyr2k_rank_0,&
      cublasZsyr2k_rank_1
  end interface
  
  interface cublasSsyr2kBatched
    function cublasSsyr2kBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasSsyr2kBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyr2kBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyr2kBatched_full_rank,&
      cublasSsyr2kBatched_rank_0,&
      cublasSsyr2kBatched_rank_1
  end interface
  
  interface cublasDsyr2kBatched
    function cublasDsyr2kBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasDsyr2kBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyr2kBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyr2kBatched_full_rank,&
      cublasDsyr2kBatched_rank_0,&
      cublasDsyr2kBatched_rank_1
  end interface
  
  interface cublasCsyr2kBatched
    function cublasCsyr2kBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasCsyr2kBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyr2kBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyr2kBatched_full_rank,&
      cublasCsyr2kBatched_rank_0,&
      cublasCsyr2kBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syr2kBatched performs a batch of the matrix-matrix operations for a symmetric rank-2k update
  !> 
  !>     C_i := alpha*(op( A_i )*op( B_i )^T + op( B_i )*op( A_i )^T) + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrix, and
  !>     C_i is a symmetric n x n matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^T, op( B_i ) = B_i^T,  and A_i and B_i are k by n if trans == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_T:      op( A_i ) = A_i^T, op( B_i ) = B_i^T
  !>             HIPBLAS_OP_N:           op( A_i ) = A_i, op( B_i ) = B_i
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix_i A of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !>     @param[in]
  !>     B       device array of device pointers storing each matrix_i B of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZsyr2kBatched
    function cublasZsyr2kBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasZsyr2kBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyr2kBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyr2kBatched_full_rank,&
      cublasZsyr2kBatched_rank_0,&
      cublasZsyr2kBatched_rank_1
  end interface
  
  interface cublasSsyr2kStridedBatched
    function cublasSsyr2kStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasSsyr2kStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSsyr2kStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyr2kStridedBatched_full_rank,&
      cublasSsyr2kStridedBatched_rank_0,&
      cublasSsyr2kStridedBatched_rank_1
  end interface
  
  interface cublasDsyr2kStridedBatched
    function cublasDsyr2kStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasDsyr2kStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDsyr2kStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyr2kStridedBatched_full_rank,&
      cublasDsyr2kStridedBatched_rank_0,&
      cublasDsyr2kStridedBatched_rank_1
  end interface
  
  interface cublasCsyr2kStridedBatched
    function cublasCsyr2kStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasCsyr2kStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCsyr2kStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyr2kStridedBatched_full_rank,&
      cublasCsyr2kStridedBatched_rank_0,&
      cublasCsyr2kStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syr2kStridedBatched performs a batch of the matrix-matrix operations for a symmetric rank-2k update
  !> 
  !>     C_i := alpha*(op( A_i )*op( B_i )^T + op( B_i )*op( A_i )^T) + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrix, and
  !>     C_i is a symmetric n x n matrix stored as either upper or lower.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^T, op( B_i ) = B_i^T,  and A_i and B_i are k by n if trans == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_T:      op( A_i ) = A_i^T, op( B_i ) = B_i^T
  !>             HIPBLAS_OP_N:           op( A_i ) = A_i, op( B_i ) = B_i
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to the first matrix A_1 on the GPU of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     B       Device pointer to the first matrix B_1 on the GPU of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     stride_B  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       Device pointer to the first matrix C_1 on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZsyr2kStridedBatched
    function cublasZsyr2kStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZsyr2kStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZsyr2kStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyr2kStridedBatched_full_rank,&
      cublasZsyr2kStridedBatched_rank_0,&
      cublasZsyr2kStridedBatched_rank_1
  end interface
  
  interface cublasSsyrkx
    function cublasSsyrkx_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasSsyrkx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyrkx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasSsyrkx_full_rank,&
      cublasSsyrkx_rank_0,&
      cublasSsyrkx_rank_1
  end interface
  
  interface cublasDsyrkx
    function cublasDsyrkx_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasDsyrkx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyrkx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasDsyrkx_full_rank,&
      cublasDsyrkx_rank_0,&
      cublasDsyrkx_rank_1
  end interface
  
  interface cublasCsyrkx
    function cublasCsyrkx_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasCsyrkx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyrkx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCsyrkx_full_rank,&
      cublasCsyrkx_rank_0,&
      cublasCsyrkx_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syrkx performs one of the matrix-matrix operations for a symmetric rank-k update
  !> 
  !>     C := alpha*op( A )*op( B )^T + beta*C
  !> 
  !>     where  alpha and beta are scalars, op(A) and op(B) are n by k matrix, and
  !>     C is a symmetric n x n matrix stored as either upper or lower.
  !>     This routine should only be used when the caller can guarantee that the result of op( A )*op( B )^T will be symmetric.
  !> 
  !>         op( A ) = A, op( B ) = B, and A and B are n by k if trans == HIPBLAS_OP_N
  !>         op( A ) = A^T, op( B ) = B^T,  and A and B are k by n if trans == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_T:      op( A ) = A^T, op( B ) = B^T
  !>             HIPBLAS_OP_N:           op( A ) = A, op( B ) = B
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A) and op(B). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             Martrix dimension is ( lda, k ) when if trans = HIPBLAS_OP_N, otherwise (lda, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     B       pointer storing matrix B on the GPU.
  !>             Martrix dimension is ( ldb, k ) when if trans = HIPBLAS_OP_N, otherwise (ldb, n)
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !>
  interface cublasZsyrkx
    function cublasZsyrkx_(handle,uplo,transA,n,k,alpha,A,lda,B,ldb,&
&beta,C,ldc) bind(c, name="cublasZsyrkx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyrkx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZsyrkx_full_rank,&
      cublasZsyrkx_rank_0,&
      cublasZsyrkx_rank_1
  end interface
  
  interface cublasSsyrkxBatched
    function cublasSsyrkxBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasSsyrkxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSsyrkxBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_float) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyrkxBatched_full_rank,&
      cublasSsyrkxBatched_rank_0,&
      cublasSsyrkxBatched_rank_1
  end interface
  
  interface cublasDsyrkxBatched
    function cublasDsyrkxBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasDsyrkxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDsyrkxBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      real(c_double) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyrkxBatched_full_rank,&
      cublasDsyrkxBatched_rank_0,&
      cublasDsyrkxBatched_rank_1
  end interface
  
  interface cublasCsyrkxBatched
    function cublasCsyrkxBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasCsyrkxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsyrkxBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyrkxBatched_full_rank,&
      cublasCsyrkxBatched_rank_0,&
      cublasCsyrkxBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syrkxBatched performs a batch of the matrix-matrix operations for a symmetric rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( B_i )^T + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrix, and
  !>     C_i is a symmetric n x n matrix stored as either upper or lower.
  !>     This routine should only be used when the caller can guarantee that the result of op( A_i )*op( B_i )^T will be symmetric.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^T, op( B_i ) = B_i^T,  and A_i and B_i are k by n if trans == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_T:      op( A_i ) = A_i^T, op( B_i ) = B_i^T
  !>             HIPBLAS_OP_N:           op( A_i ) = A_i, op( B_i ) = B_i
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix_i A of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     B       device array of device pointers storing each matrix_i B of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>             number of instances in the batch.
  !>
  interface cublasZsyrkxBatched
    function cublasZsyrkxBatched_(handle,uplo,transA,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasZsyrkxBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZsyrkxBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyrkxBatched_full_rank,&
      cublasZsyrkxBatched_rank_0,&
      cublasZsyrkxBatched_rank_1
  end interface
  
  interface cublasSsyrkxStridedBatched
    function cublasSsyrkxStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,stridec,batchCount) bind(c, &
&name="cublasSsyrkxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSsyrkxStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_float) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: stridec
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSsyrkxStridedBatched_full_rank,&
      cublasSsyrkxStridedBatched_rank_0,&
      cublasSsyrkxStridedBatched_rank_1
  end interface
  
  interface cublasDsyrkxStridedBatched
    function cublasDsyrkxStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,stridec,batchCount) bind(c, &
&name="cublasDsyrkxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDsyrkxStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      real(c_double) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: stridec
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDsyrkxStridedBatched_full_rank,&
      cublasDsyrkxStridedBatched_rank_0,&
      cublasDsyrkxStridedBatched_rank_1
  end interface
  
  interface cublasCsyrkxStridedBatched
    function cublasCsyrkxStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,stridec,batchCount) bind(c, &
&name="cublasCsyrkxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCsyrkxStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: stridec
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCsyrkxStridedBatched_full_rank,&
      cublasCsyrkxStridedBatched_rank_0,&
      cublasCsyrkxStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     syrkxStridedBatched performs a batch of the matrix-matrix operations for a symmetric rank-k update
  !> 
  !>     C_i := alpha*op( A_i )*op( B_i )^T + beta*C_i
  !> 
  !>     where  alpha and beta are scalars, op(A_i) and op(B_i) are n by k matrix, and
  !>     C_i is a symmetric n x n matrix stored as either upper or lower.
  !>     This routine should only be used when the caller can guarantee that the result of op( A_i )*op( B_i )^T will be symmetric.
  !> 
  !>         op( A_i ) = A_i, op( B_i ) = B_i, and A_i and B_i are n by k if trans == HIPBLAS_OP_N
  !>         op( A_i ) = A_i^T, op( B_i ) = B_i^T,  and A_i and B_i are k by n if trans == HIPBLAS_OP_T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  C_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  C_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     trans  [hipblasOperation_t]
  !>             HIPBLAS_OP_T:      op( A_i ) = A_i^T, op( B_i ) = B_i^T
  !>             HIPBLAS_OP_N:           op( A_i ) = A_i, op( B_i ) = B_i
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of rows and columns of C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     k       [int]
  !>             k specifies the number of columns of op(A). k >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and A need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to the first matrix A_1 on the GPU of dimension (lda, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (lda, n)
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if trans = HIPBLAS_OP_N,  lda >= max( 1, n ),
  !>             otherwise lda >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     B       Device pointer to the first matrix B_1 on the GPU of dimension (ldb, k)
  !>             when trans is HIPBLAS_OP_N, otherwise of dimension (ldb, n)
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i.
  !>             if trans = HIPBLAS_OP_N,  ldb >= max( 1, n ),
  !>             otherwise ldb >= max( 1, k ).
  !> 
  !>     @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       Device pointer to the first matrix C_1 on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZsyrkxStridedBatched
    function cublasZsyrkxStridedBatched_(handle,uplo,transA,n,k,alpha,&
&A,lda,strideA,B,ldb,strideB,beta,C,ldc,stridec,batchCount) bind(c, &
&name="cublasZsyrkxStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZsyrkxStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: stridec
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZsyrkxStridedBatched_full_rank,&
      cublasZsyrkxStridedBatched_rank_0,&
      cublasZsyrkxStridedBatched_rank_1
  end interface
  
  interface cublasSgeam
    function cublasSgeam_(handle,transa,transb,m,n,alpha,A,lda,beta,B,&
&ldb,C,ldc) bind(c, name="cublasSgeam")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgeam_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_float) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasSgeam_full_rank,&
      cublasSgeam_rank_0,&
      cublasSgeam_rank_1
  end interface
  
  interface cublasDgeam
    function cublasDgeam_(handle,transa,transb,m,n,alpha,A,lda,beta,B,&
&ldb,C,ldc) bind(c, name="cublasDgeam")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgeam_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      real(c_double) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasDgeam_full_rank,&
      cublasDgeam_rank_0,&
      cublasDgeam_rank_1
  end interface
  
  interface cublasCgeam
    function cublasCgeam_(handle,transa,transb,m,n,alpha,A,lda,beta,B,&
&ldb,C,ldc) bind(c, name="cublasCgeam")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeam_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      complex(c_float_complex) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCgeam_full_rank,&
      cublasCgeam_rank_0,&
      cublasCgeam_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     geam performs one of the matrix-matrix operations
  !> 
  !>         C = alpha*op( A ) + beta*op( B ),
  !> 
  !>     where op( X ) is one of
  !> 
  !>         op( X ) = X      or
  !>         op( X ) = X**T   or
  !>         op( X ) = X**H,
  !> 
  !>     alpha and beta are scalars, and A, B and C are matrices, with
  !>     op( A ) an m by n matrix, op( B ) an m by n matrix, and C an m by n matrix.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     transA    [hipblasOperation_t]
  !>               specifies the form of op( A )
  !>     @param[in]
  !>     transB    [hipblasOperation_t]
  !>               specifies the form of op( B )
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimension m.
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimension n.
  !>     @param[in]
  !>     alpha     device pointer or host pointer specifying the scalar alpha.
  !>     @param[in]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>     @param[in]
  !>     beta      device pointer or host pointer specifying the scalar beta.
  !>     @param[in]
  !>     B         device pointer storing matrix B.
  !>     @param[in]
  !>     ldb       [int]
  !>               specifies the leading dimension of B.
  !>     @param[in, out]
  !>     C         device pointer storing matrix C.
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of C.
  !>
  interface cublasZgeam
    function cublasZgeam_(handle,transa,transb,m,n,alpha,A,lda,beta,B,&
&ldb,C,ldc) bind(c, name="cublasZgeam")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeam_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      complex(c_double_complex) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZgeam_full_rank,&
      cublasZgeam_rank_0,&
      cublasZgeam_rank_1
  end interface
  
  interface cublasSgeamBatched
    function cublasSgeamBatched_(handle,transa,transb,m,n,alpha,A,lda,&
&beta,B,ldb,C,ldc,batchCount) bind(c, name="cublasSgeamBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgeamBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      real(c_float) :: beta
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgeamBatched_full_rank,&
      cublasSgeamBatched_rank_0,&
      cublasSgeamBatched_rank_1
  end interface
  
  interface cublasDgeamBatched
    function cublasDgeamBatched_(handle,transa,transb,m,n,alpha,A,lda,&
&beta,B,ldb,C,ldc,batchCount) bind(c, name="cublasDgeamBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgeamBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      real(c_double) :: beta
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgeamBatched_full_rank,&
      cublasDgeamBatched_rank_0,&
      cublasDgeamBatched_rank_1
  end interface
  
  interface cublasCgeamBatched
    function cublasCgeamBatched_(handle,transa,transb,m,n,alpha,A,lda,&
&beta,B,ldb,C,ldc,batchCount) bind(c, name="cublasCgeamBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeamBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      complex(c_float_complex) :: beta
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgeamBatched_full_rank,&
      cublasCgeamBatched_rank_0,&
      cublasCgeamBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     geamBatched performs one of the batched matrix-matrix operations
  !> 
  !>         C_i = alpha*op( A_i ) + beta*op( B_i )  for i = 0, 1, ... batchCount - 1
  !> 
  !>     where alpha and beta are scalars, and op(A_i), op(B_i) and C_i are m by n matrices
  !>     and op( X ) is one of
  !> 
  !>         op( X ) = X      or
  !>         op( X ) = X**T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     transA    [hipblasOperation_t]
  !>               specifies the form of op( A )
  !>     @param[in]
  !>     transB    [hipblasOperation_t]
  !>               specifies the form of op( B )
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimension m.
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimension n.
  !>     @param[in]
  !>     alpha     device pointer or host pointer specifying the scalar alpha.
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i on the GPU.
  !>               Each A_i is of dimension ( lda, k ), where k is m
  !>               when  transA == HIPBLAS_OP_N and
  !>               is  n  when  transA == HIPBLAS_OP_T.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>     @param[in]
  !>     beta      device pointer or host pointer specifying the scalar beta.
  !>     @param[in]
  !>     B         device array of device pointers storing each matrix B_i on the GPU.
  !>               Each B_i is of dimension ( ldb, k ), where k is m
  !>               when  transB == HIPBLAS_OP_N and
  !>               is  n  when  transB == HIPBLAS_OP_T.
  !>     @param[in]
  !>     ldb       [int]
  !>               specifies the leading dimension of B.
  !>     @param[in, out]
  !>     C         device array of device pointers storing each matrix C_i on the GPU.
  !>               Each C_i is of dimension ( ldc, n ).
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of C.
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances i in the batch.
  !>
  interface cublasZgeamBatched
    function cublasZgeamBatched_(handle,transa,transb,m,n,alpha,A,lda,&
&beta,B,ldb,C,ldc,batchCount) bind(c, name="cublasZgeamBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeamBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      complex(c_double_complex) :: beta
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgeamBatched_full_rank,&
      cublasZgeamBatched_rank_0,&
      cublasZgeamBatched_rank_1
  end interface
  
  interface cublasSgeamStridedBatched
    function cublasSgeamStridedBatched_(handle,transa,transb,m,n,alpha,&
&A,lda,strideA,beta,B,ldb,strideB,C,ldc,strideC,batchCount) bind(c, &
&name="cublasSgeamStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgeamStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      real(c_float) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgeamStridedBatched_full_rank,&
      cublasSgeamStridedBatched_rank_0,&
      cublasSgeamStridedBatched_rank_1
  end interface
  
  interface cublasDgeamStridedBatched
    function cublasDgeamStridedBatched_(handle,transa,transb,m,n,alpha,&
&A,lda,strideA,beta,B,ldb,strideB,C,ldc,strideC,batchCount) bind(c, &
&name="cublasDgeamStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgeamStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      real(c_double) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgeamStridedBatched_full_rank,&
      cublasDgeamStridedBatched_rank_0,&
      cublasDgeamStridedBatched_rank_1
  end interface
  
  interface cublasCgeamStridedBatched
    function cublasCgeamStridedBatched_(handle,transa,transb,m,n,alpha,&
&A,lda,strideA,beta,B,ldb,strideB,C,ldc,strideC,batchCount) bind(c, &
&name="cublasCgeamStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeamStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      complex(c_float_complex) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgeamStridedBatched_full_rank,&
      cublasCgeamStridedBatched_rank_0,&
      cublasCgeamStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     geamStridedBatched performs one of the batched matrix-matrix operations
  !> 
  !>         C_i = alpha*op( A_i ) + beta*op( B_i )  for i = 0, 1, ... batchCount - 1
  !> 
  !>     where alpha and beta are scalars, and op(A_i), op(B_i) and C_i are m by n matrices
  !>     and op( X ) is one of
  !> 
  !>         op( X ) = X      or
  !>         op( X ) = X**T
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     transA    [hipblasOperation_t]
  !>               specifies the form of op( A )
  !> 
  !>     @param[in]
  !>     transB    [hipblasOperation_t]
  !>               specifies the form of op( B )
  !> 
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimension m.
  !> 
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimension n.
  !> 
  !>     @param[in]
  !>     alpha     device pointer or host pointer specifying the scalar alpha.
  !> 
  !>     @param[in]
  !>     A         device pointer to the first matrix A_0 on the GPU.
  !>               Each A_i is of dimension ( lda, k ), where k is m
  !>               when  transA == HIPBLAS_OP_N and
  !>               is  n  when  transA == HIPBLAS_OP_T.
  !> 
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     beta      device pointer or host pointer specifying the scalar beta.
  !> 
  !>     @param[in]
  !>     B         pointer to the first matrix B_0 on the GPU.
  !>               Each B_i is of dimension ( ldb, k ), where k is m
  !>               when  transB == HIPBLAS_OP_N and
  !>               is  n  when  transB == HIPBLAS_OP_T.
  !> 
  !>     @param[in]
  !>     ldb       [int]
  !>               specifies the leading dimension of B.
  !> 
  !>     @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !> 
  !>     @param[in, out]
  !>     C         pointer to the first matrix C_0 on the GPU.
  !>               Each C_i is of dimension ( ldc, n ).
  !> 
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of C.
  !> 
  !>     @param[in]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances i in the batch.
  !>
  interface cublasZgeamStridedBatched
    function cublasZgeamStridedBatched_(handle,transa,transb,m,n,alpha,&
&A,lda,strideA,beta,B,ldb,strideB,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZgeamStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeamStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: transa
      integer(kind(CUBLAS_OP_N)),value :: transb
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      complex(c_double_complex) :: beta
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgeamStridedBatched_full_rank,&
      cublasZgeamStridedBatched_rank_0,&
      cublasZgeamStridedBatched_rank_1
  end interface
  
  interface cublasChemm
    function cublasChemm_(handle,side,uplo,n,k,alpha,A,lda,B,ldb,beta,&
&C,ldc) bind(c, name="cublasChemm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChemm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasChemm_full_rank,&
      cublasChemm_rank_0,&
      cublasChemm_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     hemm performs one of the matrix-matrix operations:
  !> 
  !>     C := alpha*A*B + beta*C if side == HIPBLAS_SIDE_LEFT,
  !>     C := alpha*B*A + beta*C if side == HIPBLAS_SIDE_RIGHT,
  !> 
  !>     where alpha and beta are scalars, B and C are m by n matrices, and
  !>     A is a Hermitian matrix stored as either upper or lower.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side  [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:      C := alpha*A*B + beta*C
  !>             HIPBLAS_SIDE_RIGHT:     C := alpha*B*A + beta*C
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B and C. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B and C. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A and B are not referenced.
  !> 
  !>     @param[in]
  !>     A       pointer storing matrix A on the GPU.
  !>             A is m by m if side == HIPBLAS_SIDE_LEFT
  !>             A is n by n if side == HIPBLAS_SIDE_RIGHT
  !>             Only the upper/lower triangular part is accessed.
  !>             The imaginary component of the diagonal elements is not used.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             otherwise lda >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     B       pointer storing matrix B on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B. ldb >= max( 1, m )
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       pointer storing matrix C on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, m )
  !>
  interface cublasZhemm
    function cublasZhemm_(handle,side,uplo,n,k,alpha,A,lda,B,ldb,beta,&
&C,ldc) bind(c, name="cublasZhemm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhemm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZhemm_full_rank,&
      cublasZhemm_rank_0,&
      cublasZhemm_rank_1
  end interface
  
  interface cublasChemmBatched
    function cublasChemmBatched_(handle,side,uplo,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasChemmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChemmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_float_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChemmBatched_full_rank,&
      cublasChemmBatched_rank_0,&
      cublasChemmBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     hemmBatched performs a batch of the matrix-matrix operations:
  !> 
  !>     C_i := alpha*A_i*B_i + beta*C_i if side == HIPBLAS_SIDE_LEFT,
  !>     C_i := alpha*B_i*A_i + beta*C_i if side == HIPBLAS_SIDE_RIGHT,
  !> 
  !>     where alpha and beta are scalars, B_i and C_i are m by n matrices, and
  !>     A_i is a Hermitian matrix stored as either upper or lower.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side  [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:      C_i := alpha*A_i*B_i + beta*C_i
  !>             HIPBLAS_SIDE_RIGHT:     C_i := alpha*B_i*A_i + beta*C_i
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B_i and C_i. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B_i and C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A_i and B_i are not referenced.
  !> 
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix A_i on the GPU.
  !>             A_i is m by m if side == HIPBLAS_SIDE_LEFT
  !>             A_i is n by n if side == HIPBLAS_SIDE_RIGHT
  !>             Only the upper/lower triangular part is accessed.
  !>             The imaginary component of the diagonal elements is not used.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             otherwise lda >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     B       device array of device pointers storing each matrix B_i on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i. ldb >= max( 1, m )
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C_i need not be set before entry.
  !> 
  !>     @param[in]
  !>     C       device array of device pointers storing each matrix C_i on the GPU.
  !>             Matrix dimension is m by n
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C_i. ldc >= max( 1, m )
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZhemmBatched
    function cublasZhemmBatched_(handle,side,uplo,n,k,alpha,A,lda,B,&
&ldb,beta,C,ldc,batchCount) bind(c, name="cublasZhemmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhemmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      complex(c_double_complex) :: beta
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhemmBatched_full_rank,&
      cublasZhemmBatched_rank_0,&
      cublasZhemmBatched_rank_1
  end interface
  
  interface cublasChemmStridedBatched
    function cublasChemmStridedBatched_(handle,side,uplo,n,k,alpha,A,&
&lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasChemmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasChemmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_float_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasChemmStridedBatched_full_rank,&
      cublasChemmStridedBatched_rank_0,&
      cublasChemmStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     hemmStridedBatched performs a batch of the matrix-matrix operations:
  !> 
  !>     C_i := alpha*A_i*B_i + beta*C_i if side == HIPBLAS_SIDE_LEFT,
  !>     C_i := alpha*B_i*A_i + beta*C_i if side == HIPBLAS_SIDE_RIGHT,
  !> 
  !>     where alpha and beta are scalars, B_i and C_i are m by n matrices, and
  !>     A_i is a Hermitian matrix stored as either upper or lower.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side  [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:      C_i := alpha*A_i*B_i + beta*C_i
  !>             HIPBLAS_SIDE_RIGHT:     C_i := alpha*B_i*A_i + beta*C_i
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A_i is an upper triangular matrix
  !>             HIPBLAS_FILL_MODE_LOWER:  A_i is a  lower triangular matrix
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B_i and C_i. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B_i and C_i. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A_i and B_i are not referenced.
  !> 
  !>     @param[in]
  !>     A       device pointer to first matrix A_1
  !>             A_i is m by m if side == HIPBLAS_SIDE_LEFT
  !>             A_i is n by n if side == HIPBLAS_SIDE_RIGHT
  !>             Only the upper/lower triangular part is accessed.
  !>             The imaginary component of the diagonal elements is not used.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A_i.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             otherwise lda >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[in]
  !>     B       device pointer to first matrix B_1 of dimension (ldb, n) on the GPU
  !> 
  !>     @param[in]
  !>     ldb     [int]
  !>             ldb specifies the first dimension of B_i.
  !>             if side = HIPBLAS_OP_N,  ldb >= max( 1, m ),
  !>             otherwise ldb >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !> 
  !>     @param[in]
  !>     beta
  !>             beta specifies the scalar beta. When beta is
  !>             zero then C need not be set before entry.
  !> 
  !>     @param[in]
  !>     C        device pointer to first matrix C_1 of dimension (ldc, n) on the GPU.
  !> 
  !>     @param[in]
  !>     ldc    [int]
  !>            ldc specifies the first dimension of C. ldc >= max( 1, m )
  !> 
  !>     @param[inout]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch
  !>
  interface cublasZhemmStridedBatched
    function cublasZhemmStridedBatched_(handle,side,uplo,n,k,alpha,A,&
&lda,strideA,B,ldb,strideB,beta,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZhemmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZhemmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(c_int),value :: n
      integer(c_int),value :: k
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      complex(c_double_complex) :: beta
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZhemmStridedBatched_full_rank,&
      cublasZhemmStridedBatched_rank_0,&
      cublasZhemmStridedBatched_rank_1
  end interface
  
  interface cublasStrmm
    function cublasStrmm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasStrmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasStrmm_full_rank,&
      cublasStrmm_rank_0,&
      cublasStrmm_rank_1
  end interface
  
  interface cublasDtrmm
    function cublasDtrmm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasDtrmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasDtrmm_full_rank,&
      cublasDtrmm_rank_0,&
      cublasDtrmm_rank_1
  end interface
  
  interface cublasCtrmm
    function cublasCtrmm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasCtrmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasCtrmm_full_rank,&
      cublasCtrmm_rank_0,&
      cublasCtrmm_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     trmm performs one of the matrix-matrix operations
  !> 
  !>     B := alpha*op( A )*B,   or   B := alpha*B*op( A )
  !> 
  !>     where  alpha  is a scalar,  B  is an m by n matrix,  A  is a unit, or
  !>     non-unit,  upper or lower triangular matrix  and  op( A )  is one  of
  !> 
  !>         op( A ) = A   or   op( A ) = A^T   or   op( A ) = A^H.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side    [hipblasSideMode_t]
  !>             Specifies whether op(A) multiplies B from the left or right as follows:
  !>             HIPBLAS_SIDE_LEFT:       B := alpha*op( A )*B.
  !>             HIPBLAS_SIDE_RIGHT:      B := alpha*B*op( A ).
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             Specifies whether the matrix A is an upper or lower triangular matrix as follows:
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             Specifies the form of op(A) to be used in the matrix multiplication as follows:
  !>             HIPBLAS_OP_N: op(A) = A.
  !>             HIPBLAS_OP_T: op(A) = A^T.
  !>             HIPBLAS_OP_C:  op(A) = A^H.
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             Specifies whether or not A is unit triangular as follows:
  !>             HIPBLAS_DIAG_UNIT:      A is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A is not referenced and B need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to matrix A on the GPU.
  !>             A has dimension ( lda, k ), where k is m
  !>             when  side == HIPBLAS_SIDE_LEFT  and
  !>             is  n  when  side == HIPBLAS_SIDE_RIGHT.
  !> 
  !>         When uplo == HIPBLAS_FILL_MODE_UPPER the  leading  k by k
  !>         upper triangular part of the array  A must contain the upper
  !>         triangular matrix  and the strictly lower triangular part of
  !>         A is not referenced.
  !> 
  !>         When uplo == HIPBLAS_FILL_MODE_LOWER the  leading  k by k
  !>         lower triangular part of the array  A must contain the lower
  !>         triangular matrix  and the strictly upper triangular part of
  !>         A is not referenced.
  !> 
  !>         Note that when  diag == HIPBLAS_DIAG_UNIT  the diagonal elements of
  !>         A  are not referenced either,  but are assumed to be  unity.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if side == HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             if side == HIPBLAS_SIDE_RIGHT, lda >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     B       Device pointer to the first matrix B_0 on the GPU.
  !>             On entry,  the leading  m by n part of the array  B must
  !>            contain the matrix  B,  and  on exit  is overwritten  by the
  !>            transformed matrix.
  !> 
  !>     @param[in]
  !>     ldb    [int]
  !>            ldb specifies the first dimension of B. ldb >= max( 1, m ).
  !>
  interface cublasZtrmm
    function cublasZtrmm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasZtrmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasZtrmm_full_rank,&
      cublasZtrmm_rank_0,&
      cublasZtrmm_rank_1
  end interface
  
  interface cublasStrmmBatched
    function cublasStrmmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasStrmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrmmBatched_full_rank,&
      cublasStrmmBatched_rank_0,&
      cublasStrmmBatched_rank_1
  end interface
  
  interface cublasDtrmmBatched
    function cublasDtrmmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasDtrmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrmmBatched_full_rank,&
      cublasDtrmmBatched_rank_0,&
      cublasDtrmmBatched_rank_1
  end interface
  
  interface cublasCtrmmBatched
    function cublasCtrmmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasCtrmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrmmBatched_full_rank,&
      cublasCtrmmBatched_rank_0,&
      cublasCtrmmBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     trmmBatched performs one of the batched matrix-matrix operations
  !> 
  !>     B_i := alpha*op( A_i )*B_i,   or   B_i := alpha*B_i*op( A_i )  for i = 0, 1, ... batchCount -1
  !> 
  !>     where  alpha  is a scalar,  B_i  is an m by n matrix,  A_i  is a unit, or
  !>     non-unit,  upper or lower triangular matrix  and  op( A_i )  is one  of
  !> 
  !>         op( A_i ) = A_i   or   op( A_i ) = A_i^T   or   op( A_i ) = A_i^H.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side    [hipblasSideMode_t]
  !>             Specifies whether op(A_i) multiplies B_i from the left or right as follows:
  !>             HIPBLAS_SIDE_LEFT:       B_i := alpha*op( A_i )*B_i.
  !>             HIPBLAS_SIDE_RIGHT:      B_i := alpha*B_i*op( A_i ).
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             Specifies whether the matrix A is an upper or lower triangular matrix as follows:
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             Specifies the form of op(A_i) to be used in the matrix multiplication as follows:
  !>             HIPBLAS_OP_N:    op(A_i) = A_i.
  !>             HIPBLAS_OP_T:      op(A_i) = A_i^T.
  !>             HIPBLAS_OP_C:  op(A_i) = A_i^H.
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             Specifies whether or not A_i is unit triangular as follows:
  !>             HIPBLAS_DIAG_UNIT:      A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B_i. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B_i. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A_i is not referenced and B_i need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device array of device pointers storing each matrix A_i on the GPU.
  !>             Each A_i is of dimension ( lda, k ), where k is m
  !>             when  side == HIPBLAS_SIDE_LEFT  and
  !>             is  n  when  side == HIPBLAS_SIDE_RIGHT.
  !> 
  !>         When uplo == HIPBLAS_FILL_MODE_UPPER the  leading  k by k
  !>         upper triangular part of the array  A must contain the upper
  !>         triangular matrix  and the strictly lower triangular part of
  !>         A is not referenced.
  !> 
  !>         When uplo == HIPBLAS_FILL_MODE_LOWER the  leading  k by k
  !>         lower triangular part of the array  A must contain the lower
  !>         triangular matrix  and the strictly upper triangular part of
  !>         A is not referenced.
  !> 
  !>         Note that when  diag == HIPBLAS_DIAG_UNIT  the diagonal elements of
  !>         A_i  are not referenced either,  but are assumed to be  unity.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if side == HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             if side == HIPBLAS_SIDE_RIGHT, lda >= max( 1, n ).
  !> 
  !>     @param[inout]
  !>     B       device array of device pointers storing each matrix B_i on the GPU.
  !>             On entry,  the leading  m by n part of the array  B_i must
  !>            contain the matrix  B_i,  and  on exit  is overwritten  by the
  !>            transformed matrix.
  !> 
  !>     @param[in]
  !>     ldb    [int]
  !>            ldb specifies the first dimension of B_i. ldb >= max( 1, m ).
  !> 
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances i in the batch.
  interface cublasZtrmmBatched
    function cublasZtrmmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasZtrmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrmmBatched_full_rank,&
      cublasZtrmmBatched_rank_0,&
      cublasZtrmmBatched_rank_1
  end interface
  
  interface cublasStrmmStridedBatched
    function cublasStrmmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasStrmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrmmStridedBatched_full_rank,&
      cublasStrmmStridedBatched_rank_0,&
      cublasStrmmStridedBatched_rank_1
  end interface
  
  interface cublasDtrmmStridedBatched
    function cublasDtrmmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasDtrmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrmmStridedBatched_full_rank,&
      cublasDtrmmStridedBatched_rank_0,&
      cublasDtrmmStridedBatched_rank_1
  end interface
  
  interface cublasCtrmmStridedBatched
    function cublasCtrmmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasCtrmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrmmStridedBatched_full_rank,&
      cublasCtrmmStridedBatched_rank_0,&
      cublasCtrmmStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     trmmStridedBatched performs one of the strided_batched matrix-matrix operations
  !> 
  !>     B_i := alpha*op( A_i )*B_i,   or   B_i := alpha*B_i*op( A_i )  for i = 0, 1, ... batchCount -1
  !> 
  !>     where  alpha  is a scalar,  B_i  is an m by n matrix,  A_i  is a unit, or
  !>     non-unit,  upper or lower triangular matrix  and  op( A_i )  is one  of
  !> 
  !>         op( A_i ) = A_i   or   op( A_i ) = A_i^T   or   op( A_i ) = A_i^H.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side    [hipblasSideMode_t]
  !>             Specifies whether op(A_i) multiplies B_i from the left or right as follows:
  !>             HIPBLAS_SIDE_LEFT:       B_i := alpha*op( A_i )*B_i.
  !>             HIPBLAS_SIDE_RIGHT:      B_i := alpha*B_i*op( A_i ).
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             Specifies whether the matrix A is an upper or lower triangular matrix as follows:
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             Specifies the form of op(A_i) to be used in the matrix multiplication as follows:
  !>             HIPBLAS_OP_N:    op(A_i) = A_i.
  !>             HIPBLAS_OP_T:      op(A_i) = A_i^T.
  !>             HIPBLAS_OP_C:  op(A_i) = A_i^H.
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             Specifies whether or not A_i is unit triangular as follows:
  !>             HIPBLAS_DIAG_UNIT:      A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A_i is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B_i. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B_i. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             alpha specifies the scalar alpha. When alpha is
  !>             zero then A_i is not referenced and B_i need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       Device pointer to the first matrix A_0 on the GPU.
  !>             Each A_i is of dimension ( lda, k ), where k is m
  !>             when  side == HIPBLAS_SIDE_LEFT  and
  !>             is  n  when  side == HIPBLAS_SIDE_RIGHT.
  !> 
  !>         When uplo == HIPBLAS_FILL_MODE_UPPER the  leading  k by k
  !>         upper triangular part of the array  A must contain the upper
  !>         triangular matrix  and the strictly lower triangular part of
  !>         A is not referenced.
  !> 
  !>         When uplo == HIPBLAS_FILL_MODE_LOWER the  leading  k by k
  !>         lower triangular part of the array  A must contain the lower
  !>         triangular matrix  and the strictly upper triangular part of
  !>         A is not referenced.
  !> 
  !>         Note that when  diag == HIPBLAS_DIAG_UNIT  the diagonal elements of
  !>         A_i  are not referenced either,  but are assumed to be  unity.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if side == HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             if side == HIPBLAS_SIDE_RIGHT, lda >= max( 1, n ).
  !> 
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !> 
  !>     @param[inout]
  !>     B       Device pointer to the first matrix B_0 on the GPU.
  !>             On entry,  the leading  m by n part of the array  B_i must
  !>            contain the matrix  B_i,  and  on exit  is overwritten  by the
  !>            transformed matrix.
  !> 
  !>     @param[in]
  !>     ldb    [int]
  !>            ldb specifies the first dimension of B_i. ldb >= max( 1, m ).
  !> 
  !>            @param[in]
  !>     strideB  [hipblasStride]
  !>               stride from the start of one matrix (B_i) and the next one (B_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances i in the batch.
  interface cublasZtrmmStridedBatched
    function cublasZtrmmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasZtrmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrmmStridedBatched_full_rank,&
      cublasZtrmmStridedBatched_rank_0,&
      cublasZtrmmStridedBatched_rank_1
  end interface
  
  interface cublasStrsm
    function cublasStrsm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasStrsm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrsm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasStrsm_full_rank,&
      cublasStrsm_rank_0,&
      cublasStrsm_rank_1
  end interface
  
  interface cublasDtrsm
    function cublasDtrsm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasDtrsm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrsm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasDtrsm_full_rank,&
      cublasDtrsm_rank_0,&
      cublasDtrsm_rank_1
  end interface
  
  interface cublasCtrsm
    function cublasCtrsm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasCtrsm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrsm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasCtrsm_full_rank,&
      cublasCtrsm_rank_0,&
      cublasCtrsm_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !> 
  !>     trsm solves
  !> 
  !>         op(A)*X = alpha*B or  X*op(A) = alpha*B,
  !> 
  !>     where alpha is a scalar, X and B are m by n matrices,
  !>     A is triangular matrix and op(A) is one of
  !> 
  !>         op( A ) = A   or   op( A ) = A^T   or   op( A ) = A^H.
  !> 
  !>     The matrix X is overwritten on B.
  !> 
  !>     Note about memory allocation:
  !>     When trsm is launched with a k evenly divisible by the internal block size of 128,
  !>     and is no larger than 10 of these blocks, the API takes advantage of utilizing pre-allocated
  !>     memory found in the handle to increase overall performance. This memory can be managed by using
  !>     the environment variable WORKBUF_TRSM_B_CHNK. When this variable is not set the device memory
  !>     used for temporary storage will default to 1 MB and may result in chunking, which in turn may
  !>     reduce performance. Under these circumstances it is recommended that WORKBUF_TRSM_B_CHNK be set
  !>     to the desired chunk of right hand sides to be used at a time.
  !> 
  !>     (where k is m when HIPBLAS_SIDE_LEFT and is n when HIPBLAS_SIDE_RIGHT)
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !> 
  !>     @param[in]
  !>     side    [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:       op(A)*X = alpha*B.
  !>             HIPBLAS_SIDE_RIGHT:      X*op(A) = alpha*B.
  !> 
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  A is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  A is a  lower triangular matrix.
  !> 
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_N: op(A) = A.
  !>             HIPBLAS_OP_T: op(A) = A^T.
  !>             HIPBLAS_OP_C: op(A) = A^H.
  !> 
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     A is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  A is not assumed to be unit triangular.
  !> 
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of B. m >= 0.
  !> 
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of B. n >= 0.
  !> 
  !>     @param[in]
  !>     alpha
  !>             device pointer or host pointer specifying the scalar alpha. When alpha is
  !>             &zero then A is not referenced and B need not be set before
  !>             entry.
  !> 
  !>     @param[in]
  !>     A       device pointer storing matrix A.
  !>             of dimension ( lda, k ), where k is m
  !>             when  HIPBLAS_SIDE_LEFT  and
  !>             is  n  when  HIPBLAS_SIDE_RIGHT
  !>             only the upper/lower triangular part is accessed.
  !> 
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of A.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             if side = HIPBLAS_SIDE_RIGHT, lda >= max( 1, n ).
  !> 
  !>     @param[in,out]
  !>     B       device pointer storing matrix B.
  !> 
  !>     @param[in]
  !>     ldb    [int]
  !>            ldb specifies the first dimension of B. ldb >= max( 1, m ).
  !>
  interface cublasZtrsm
    function cublasZtrsm_(handle,side,uplo,transA,diag,m,n,alpha,A,lda,&
&B,ldb) bind(c, name="cublasZtrsm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrsm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
    end function

    module procedure &
      cublasZtrsm_full_rank,&
      cublasZtrsm_rank_0,&
      cublasZtrsm_rank_1
  end interface
  
  interface cublasStrsmBatched
    function cublasStrsmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasStrsmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrsmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrsmBatched_full_rank,&
      cublasStrsmBatched_rank_0,&
      cublasStrsmBatched_rank_1
  end interface
  
  interface cublasDtrsmBatched
    function cublasDtrsmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasDtrsmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrsmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrsmBatched_full_rank,&
      cublasDtrsmBatched_rank_0,&
      cublasDtrsmBatched_rank_1
  end interface
  
  interface cublasCtrsmBatched
    function cublasCtrsmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasCtrsmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrsmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrsmBatched_full_rank,&
      cublasCtrsmBatched_rank_0,&
      cublasCtrsmBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !>     \details
  !>     trsmBatched performs the following batched operation:
  !> 
  !>         op(A_i)*X_i = alpha*B_i or  X_i*op(A_i) = alpha*B_i, for i = 1, ..., batchCount.
  !> 
  !>     where alpha is a scalar, X and B are batched m by n matrices,
  !>     A is triangular batched matrix and op(A) is one of
  !> 
  !>         op( A ) = A   or   op( A ) = A^T   or   op( A ) = A^H.
  !> 
  !>     Each matrix X_i is overwritten on B_i for i = 1, ..., batchCount.
  !> 
  !>     Note about memory allocation:
  !>     When trsm is launched with a k evenly divisible by the internal block size of 128,
  !>     and is no larger than 10 of these blocks, the API takes advantage of utilizing pre-allocated
  !>     memory found in the handle to increase overall performance. This memory can be managed by using
  !>     the environment variable WORKBUF_TRSM_B_CHNK. When this variable is not set the device memory
  !>     used for temporary storage will default to 1 MB and may result in chunking, which in turn may
  !>     reduce performance. Under these circumstances it is recommended that WORKBUF_TRSM_B_CHNK be set
  !>     to the desired chunk of right hand sides to be used at a time.
  !>     (where k is m when HIPBLAS_SIDE_LEFT and is n when HIPBLAS_SIDE_RIGHT)
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     side    [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:       op(A)*X = alpha*B.
  !>             HIPBLAS_SIDE_RIGHT:      X*op(A) = alpha*B.
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  each A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  each A_i is a  lower triangular matrix.
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_N: op(A) = A.
  !>             HIPBLAS_OP_T: op(A) = A^T.
  !>             HIPBLAS_OP_C: op(A) = A^H.
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     each A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  each A_i is not assumed to be unit triangular.
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of each B_i. m >= 0.
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of each B_i. n >= 0.
  !>     @param[in]
  !>     alpha
  !>             device pointer or host pointer specifying the scalar alpha. When alpha is
  !>             &zero then A is not referenced and B need not be set before
  !>             entry.
  !>     @param[in]
  !>     A       device array of device pointers storing each matrix A_i on the GPU.
  !>             Matricies are of dimension ( lda, k ), where k is m
  !>             when  HIPBLAS_SIDE_LEFT  and is  n  when  HIPBLAS_SIDE_RIGHT
  !>             only the upper/lower triangular part is accessed.
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of each A_i.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             if side = HIPBLAS_SIDE_RIGHT, lda >= max( 1, n ).
  !>     @param[in,out]
  !>     B       device array of device pointers storing each matrix B_i on the GPU.
  !>     @param[in]
  !>     ldb    [int]
  !>            ldb specifies the first dimension of each B_i. ldb >= max( 1, m ).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of trsm operatons in the batch.
  interface cublasZtrsmBatched
    function cublasZtrsmBatched_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batchCount) bind(c, name="cublasZtrsmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrsmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrsmBatched_full_rank,&
      cublasZtrsmBatched_rank_0,&
      cublasZtrsmBatched_rank_1
  end interface
  
  interface cublasStrsmStridedBatched
    function cublasStrsmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasStrsmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrsmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_float) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrsmStridedBatched_full_rank,&
      cublasStrsmStridedBatched_rank_0,&
      cublasStrsmStridedBatched_rank_1
  end interface
  
  interface cublasDtrsmStridedBatched
    function cublasDtrsmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasDtrsmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrsmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      real(c_double) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrsmStridedBatched_full_rank,&
      cublasDtrsmStridedBatched_rank_0,&
      cublasDtrsmStridedBatched_rank_1
  end interface
  
  interface cublasCtrsmStridedBatched
    function cublasCtrsmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasCtrsmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrsmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_float_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrsmStridedBatched_full_rank,&
      cublasCtrsmStridedBatched_rank_0,&
      cublasCtrsmStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !>     \details
  !>     trsmSridedBatched performs the following strided batched operation:
  !> 
  !>         op(A_i)*X_i = alpha*B_i or  X_i*op(A_i) = alpha*B_i, for i = 1, ..., batchCount.
  !> 
  !>     where alpha is a scalar, X and B are strided batched m by n matrices,
  !>     A is triangular strided batched matrix and op(A) is one of
  !> 
  !>         op( A ) = A   or   op( A ) = A^T   or   op( A ) = A^H.
  !> 
  !>     Each matrix X_i is overwritten on B_i for i = 1, ..., batchCount.
  !> 
  !>     Note about memory allocation:
  !>     When trsm is launched with a k evenly divisible by the internal block size of 128,
  !>     and is no larger than 10 of these blocks, the API takes advantage of utilizing pre-allocated
  !>     memory found in the handle to increase overall performance. This memory can be managed by using
  !>     the environment variable WORKBUF_TRSM_B_CHNK. When this variable is not set the device memory
  !>     used for temporary storage will default to 1 MB and may result in chunking, which in turn may
  !>     reduce performance. Under these circumstances it is recommended that WORKBUF_TRSM_B_CHNK be set
  !>     to the desired chunk of right hand sides to be used at a time.
  !>     (where k is m when HIPBLAS_SIDE_LEFT and is n when HIPBLAS_SIDE_RIGHT)
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     side    [hipblasSideMode_t]
  !>             HIPBLAS_SIDE_LEFT:       op(A)*X = alpha*B.
  !>             HIPBLAS_SIDE_RIGHT:      X*op(A) = alpha*B.
  !>     @param[in]
  !>     uplo    [hipblasFillMode_t]
  !>             HIPBLAS_FILL_MODE_UPPER:  each A_i is an upper triangular matrix.
  !>             HIPBLAS_FILL_MODE_LOWER:  each A_i is a  lower triangular matrix.
  !>     @param[in]
  !>     transA  [hipblasOperation_t]
  !>             HIPBLAS_OP_N: op(A) = A.
  !>             HIPBLAS_OP_T: op(A) = A^T.
  !>             HIPBLAS_OP_C: op(A) = A^H.
  !>     @param[in]
  !>     diag    [hipblasDiagType_t]
  !>             HIPBLAS_DIAG_UNIT:     each A_i is assumed to be unit triangular.
  !>             HIPBLAS_DIAG_NON_UNIT:  each A_i is not assumed to be unit triangular.
  !>     @param[in]
  !>     m       [int]
  !>             m specifies the number of rows of each B_i. m >= 0.
  !>     @param[in]
  !>     n       [int]
  !>             n specifies the number of columns of each B_i. n >= 0.
  !>     @param[in]
  !>     alpha
  !>             device pointer or host pointer specifying the scalar alpha. When alpha is
  !>             &zero then A is not referenced and B need not be set before
  !>             entry.
  !>     @param[in]
  !>     A       device pointer pointing to the first matrix A_1.
  !>             of dimension ( lda, k ), where k is m
  !>             when  HIPBLAS_SIDE_LEFT  and
  !>             is  n  when  HIPBLAS_SIDE_RIGHT
  !>             only the upper/lower triangular part is accessed.
  !>     @param[in]
  !>     lda     [int]
  !>             lda specifies the first dimension of each A_i.
  !>             if side = HIPBLAS_SIDE_LEFT,  lda >= max( 1, m ),
  !>             if side = HIPBLAS_SIDE_RIGHT, lda >= max( 1, n ).
  !>     @param[in]
  !>     strideA [hipblasStride]
  !>              stride from the start of one A_i matrix to the next A_(i + 1).
  !>     @param[in,out]
  !>     B       device pointer pointing to the first matrix B_1.
  !>     @param[in]
  !>     ldb    [int]
  !>            ldb specifies the first dimension of each B_i. ldb >= max( 1, m ).
  !>     @param[in]
  !>     strideB [hipblasStride]
  !>              stride from the start of one B_i matrix to the next B_(i + 1).
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of trsm operatons in the batch.
  interface cublasZtrsmStridedBatched
    function cublasZtrsmStridedBatched_(handle,side,uplo,transA,diag,m,&
&n,alpha,A,lda,strideA,B,ldb,strideB,batchCount) bind(c, &
&name="cublasZtrsmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrsmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      complex(c_double_complex) :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrsmStridedBatched_full_rank,&
      cublasZtrsmStridedBatched_rank_0,&
      cublasZtrsmStridedBatched_rank_1
  end interface
  
  interface cublasStrtri
    function cublasStrtri_(handle,uplo,diag,n,A,lda,invA,ldinvA) &
&bind(c, name="cublasStrtri")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrtri_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
    end function

    module procedure &
      cublasStrtri_full_rank,&
      cublasStrtri_rank_0,&
      cublasStrtri_rank_1
  end interface
  
  interface cublasDtrtri
    function cublasDtrtri_(handle,uplo,diag,n,A,lda,invA,ldinvA) &
&bind(c, name="cublasDtrtri")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrtri_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
    end function

    module procedure &
      cublasDtrtri_full_rank,&
      cublasDtrtri_rank_0,&
      cublasDtrtri_rank_1
  end interface
  
  interface cublasCtrtri
    function cublasCtrtri_(handle,uplo,diag,n,A,lda,invA,ldinvA) &
&bind(c, name="cublasCtrtri")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrtri_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
    end function

    module procedure &
      cublasCtrtri_full_rank,&
      cublasCtrtri_rank_0,&
      cublasCtrtri_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     trtri  compute the inverse of a matrix A, namely, invA
  !> 
  !>         and write the result into invA;
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>               if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
  !>               if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
  !>     @param[in]
  !>     diag      [hipblasDiagType_t]
  !>               = 'HIPBLAS_DIAG_NON_UNIT', A is non-unit triangular;
  !>               = 'HIPBLAS_DIAG_UNIT', A is unit triangular;
  !>     @param[in]
  !>     n         [int]
  !>               size of matrix A and invA
  !>     @param[in]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>     @param[out]
  !>     invA      device pointer storing matrix invA.
  !>     @param[in]
  !>     ldinvA    [int]
  !>               specifies the leading dimension of invA.
  !>
  interface cublasZtrtri
    function cublasZtrtri_(handle,uplo,diag,n,A,lda,invA,ldinvA) &
&bind(c, name="cublasZtrtri")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrtri_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
    end function

    module procedure &
      cublasZtrtri_full_rank,&
      cublasZtrtri_rank_0,&
      cublasZtrtri_rank_1
  end interface
  
  interface cublasStrtriBatched
    function cublasStrtriBatched_(handle,uplo,diag,n,A,lda,invA,ldinvA,&
&batchCount) bind(c, name="cublasStrtriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasStrtriBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: invA
      integer(c_int),value :: ldinvA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrtriBatched_full_rank,&
      cublasStrtriBatched_rank_0,&
      cublasStrtriBatched_rank_1
  end interface
  
  interface cublasDtrtriBatched
    function cublasDtrtriBatched_(handle,uplo,diag,n,A,lda,invA,ldinvA,&
&batchCount) bind(c, name="cublasDtrtriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDtrtriBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: invA
      integer(c_int),value :: ldinvA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrtriBatched_full_rank,&
      cublasDtrtriBatched_rank_0,&
      cublasDtrtriBatched_rank_1
  end interface
  
  interface cublasCtrtriBatched
    function cublasCtrtriBatched_(handle,uplo,diag,n,A,lda,invA,ldinvA,&
&batchCount) bind(c, name="cublasCtrtriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCtrtriBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: invA
      integer(c_int),value :: ldinvA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrtriBatched_full_rank,&
      cublasCtrtriBatched_rank_0,&
      cublasCtrtriBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     trtriBatched  compute the inverse of A_i and write into invA_i where
  !>                    A_i and invA_i are the i-th matrices in the batch,
  !>                    for i = 1, ..., batchCount.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>     @param[in]
  !>     diag      [hipblasDiagType_t]
  !>               = 'HIPBLAS_DIAG_NON_UNIT', A is non-unit triangular;
  !>               = 'HIPBLAS_DIAG_UNIT', A is unit triangular;
  !>     @param[in]
  !>     n         [int]
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A_i.
  !>     @param[out]
  !>     invA      device array of device pointers storing the inverse of each matrix A_i.
  !>               Partial inplace operation is supported, see below.
  !>               If UPLO = 'U', the leading N-by-N upper triangular part of the invA will store
  !>               the inverse of the upper triangular matrix, and the strictly lower
  !>               triangular part of invA is cleared.
  !>               If UPLO = 'L', the leading N-by-N lower triangular part of the invA will store
  !>               the inverse of the lower triangular matrix, and the strictly upper
  !>               triangular part of invA is cleared.
  !>     @param[in]
  !>     ldinvA    [int]
  !>               specifies the leading dimension of each invA_i.
  !>     @param[in]
  !>     batchCount [int]
  !>               numbers of matrices in the batch
  interface cublasZtrtriBatched
    function cublasZtrtriBatched_(handle,uplo,diag,n,A,lda,invA,ldinvA,&
&batchCount) bind(c, name="cublasZtrtriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZtrtriBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: invA
      integer(c_int),value :: ldinvA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrtriBatched_full_rank,&
      cublasZtrtriBatched_rank_0,&
      cublasZtrtriBatched_rank_1
  end interface
  
  interface cublasStrtriStridedBatched
    function cublasStrtriStridedBatched_(handle,uplo,diag,n,A,lda,&
&strideA,invA,ldinvA,stride_invA,batchCount) bind(c, &
&name="cublasStrtriStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasStrtriStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
      integer(c_int64_t),value :: stride_invA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasStrtriStridedBatched_full_rank,&
      cublasStrtriStridedBatched_rank_0,&
      cublasStrtriStridedBatched_rank_1
  end interface
  
  interface cublasDtrtriStridedBatched
    function cublasDtrtriStridedBatched_(handle,uplo,diag,n,A,lda,&
&strideA,invA,ldinvA,stride_invA,batchCount) bind(c, &
&name="cublasDtrtriStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDtrtriStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
      integer(c_int64_t),value :: stride_invA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDtrtriStridedBatched_full_rank,&
      cublasDtrtriStridedBatched_rank_0,&
      cublasDtrtriStridedBatched_rank_1
  end interface
  
  interface cublasCtrtriStridedBatched
    function cublasCtrtriStridedBatched_(handle,uplo,diag,n,A,lda,&
&strideA,invA,ldinvA,stride_invA,batchCount) bind(c, &
&name="cublasCtrtriStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCtrtriStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
      integer(c_int64_t),value :: stride_invA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCtrtriStridedBatched_full_rank,&
      cublasCtrtriStridedBatched_rank_0,&
      cublasCtrtriStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     trtriStridedBatched compute the inverse of A_i and write into invA_i where
  !>                    A_i and invA_i are the i-th matrices in the batch,
  !>                    for i = 1, ..., batchCount
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     uplo      [hipblasFillMode_t]
  !>               specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
  !>     @param[in]
  !>     diag      [hipblasDiagType_t]
  !>               = 'HIPBLAS_DIAG_NON_UNIT', A is non-unit triangular;
  !>               = 'HIPBLAS_DIAG_UNIT', A is unit triangular;
  !>     @param[in]
  !>     n         [int]
  !>     @param[in]
  !>     A         device pointer pointing to address of first matrix A_1.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of each A.
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>              "batch stride a": stride from the start of one A_i matrix to the next A_(i + 1).
  !>     @param[out]
  !>     invA      device pointer storing the inverses of each matrix A_i.
  !>               Partial inplace operation is supported, see below.
  !>               If UPLO = 'U', the leading N-by-N upper triangular part of the invA will store
  !>               the inverse of the upper triangular matrix, and the strictly lower
  !>               triangular part of invA is cleared.
  !>               If UPLO = 'L', the leading N-by-N lower triangular part of the invA will store
  !>               the inverse of the lower triangular matrix, and the strictly upper
  !>               triangular part of invA is cleared.
  !>     @param[in]
  !>     ldinvA    [int]
  !>               specifies the leading dimension of each invA_i.
  !>     @param[in]
  !>     stride_invA  [hipblasStride]
  !>                  "batch stride invA": stride from the start of one invA_i matrix to the next invA_(i + 1).
  !>     @param[in]
  !>     batchCount  [int]
  !>                  numbers of matrices in the batch
  interface cublasZtrtriStridedBatched
    function cublasZtrtriStridedBatched_(handle,uplo,diag,n,A,lda,&
&strideA,invA,ldinvA,stride_invA,batchCount) bind(c, &
&name="cublasZtrtriStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZtrtriStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: invA
      integer(c_int),value :: ldinvA
      integer(c_int64_t),value :: stride_invA
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZtrtriStridedBatched_full_rank,&
      cublasZtrtriStridedBatched_rank_0,&
      cublasZtrtriStridedBatched_rank_1
  end interface
  
  interface cublasSdgmm
    function cublasSdgmm_(handle,side,m,n,A,lda,x,incx,C,ldc) bind(c, &
&name="cublasSdgmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdgmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasSdgmm_full_rank,&
      cublasSdgmm_rank_0,&
      cublasSdgmm_rank_1
  end interface
  
  interface cublasDdgmm
    function cublasDdgmm_(handle,side,m,n,A,lda,x,incx,C,ldc) bind(c, &
&name="cublasDdgmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdgmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasDdgmm_full_rank,&
      cublasDdgmm_rank_0,&
      cublasDdgmm_rank_1
  end interface
  
  interface cublasCdgmm
    function cublasCdgmm_(handle,side,m,n,A,lda,x,incx,C,ldc) bind(c, &
&name="cublasCdgmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdgmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasCdgmm_full_rank,&
      cublasCdgmm_rank_0,&
      cublasCdgmm_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     dgmm performs one of the matrix-matrix operations
  !> 
  !>         C = A * diag(x) if side == HIPBLAS_SIDE_RIGHT
  !>         C = diag(x) * A if side == HIPBLAS_SIDE_LEFT
  !> 
  !>     where C and A are m by n dimensional matrices. diag( x ) is a diagonal matrix
  !>     and x is vector of dimension n if side == HIPBLAS_SIDE_RIGHT and dimension m
  !>     if side == HIPBLAS_SIDE_LEFT.
  !> 
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     side      [hipblasSideMode_t]
  !>               specifies the side of diag(x)
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimension m.
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimension n.
  !>     @param[in]
  !>     A         device pointer storing matrix A.
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>     @param[in]
  !>     x         device pointer storing vector x.
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment between values of x
  !>     @param[in, out]
  !>     C         device pointer storing matrix C.
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of C.
  !>
  interface cublasZdgmm
    function cublasZdgmm_(handle,side,m,n,A,lda,x,incx,C,ldc) bind(c, &
&name="cublasZdgmm")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdgmm_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
    end function

    module procedure &
      cublasZdgmm_full_rank,&
      cublasZdgmm_rank_0,&
      cublasZdgmm_rank_1
  end interface
  
  interface cublasSdgmmBatched
    function cublasSdgmmBatched_(handle,side,m,n,A,lda,x,incx,C,ldc,&
&batchCount) bind(c, name="cublasSdgmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdgmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSdgmmBatched_full_rank,&
      cublasSdgmmBatched_rank_0,&
      cublasSdgmmBatched_rank_1
  end interface
  
  interface cublasDdgmmBatched
    function cublasDdgmmBatched_(handle,side,m,n,A,lda,x,incx,C,ldc,&
&batchCount) bind(c, name="cublasDdgmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdgmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDdgmmBatched_full_rank,&
      cublasDdgmmBatched_rank_0,&
      cublasDdgmmBatched_rank_1
  end interface
  
  interface cublasCdgmmBatched
    function cublasCdgmmBatched_(handle,side,m,n,A,lda,x,incx,C,ldc,&
&batchCount) bind(c, name="cublasCdgmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdgmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCdgmmBatched_full_rank,&
      cublasCdgmmBatched_rank_0,&
      cublasCdgmmBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     dgmmBatched performs one of the batched matrix-matrix operations
  !> 
  !>         C_i = A_i * diag(x_i) for i = 0, 1, ... batchCount-1 if side == HIPBLAS_SIDE_RIGHT
  !>         C_i = diag(x_i) * A_i for i = 0, 1, ... batchCount-1 if side == HIPBLAS_SIDE_LEFT
  !> 
  !>     where C_i and A_i are m by n dimensional matrices. diag(x_i) is a diagonal matrix
  !>     and x_i is vector of dimension n if side == HIPBLAS_SIDE_RIGHT and dimension m
  !>     if side == HIPBLAS_SIDE_LEFT.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     side      [hipblasSideMode_t]
  !>               specifies the side of diag(x)
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimension m.
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimension n.
  !>     @param[in]
  !>     A         device array of device pointers storing each matrix A_i on the GPU.
  !>               Each A_i is of dimension ( lda, n )
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A_i.
  !>     @param[in]
  !>     x         device array of device pointers storing each vector x_i on the GPU.
  !>               Each x_i is of dimension n if side == HIPBLAS_SIDE_RIGHT and dimension
  !>               m if side == HIPBLAS_SIDE_LEFT
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment between values of x_i
  !>     @param[in, out]
  !>     C         device array of device pointers storing each matrix C_i on the GPU.
  !>               Each C_i is of dimension ( ldc, n ).
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of C_i.
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances in the batch.
  !>
  interface cublasZdgmmBatched
    function cublasZdgmmBatched_(handle,side,m,n,A,lda,x,incx,C,ldc,&
&batchCount) bind(c, name="cublasZdgmmBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdgmmBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: x
      integer(c_int),value :: incx
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZdgmmBatched_full_rank,&
      cublasZdgmmBatched_rank_0,&
      cublasZdgmmBatched_rank_1
  end interface
  
  interface cublasSdgmmStridedBatched
    function cublasSdgmmStridedBatched_(handle,side,m,n,A,lda,strideA,&
&x,incx,stridex,C,ldc,strideC,batchCount) bind(c, &
&name="cublasSdgmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdgmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSdgmmStridedBatched_full_rank,&
      cublasSdgmmStridedBatched_rank_0,&
      cublasSdgmmStridedBatched_rank_1
  end interface
  
  interface cublasDdgmmStridedBatched
    function cublasDdgmmStridedBatched_(handle,side,m,n,A,lda,strideA,&
&x,incx,stridex,C,ldc,strideC,batchCount) bind(c, &
&name="cublasDdgmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdgmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDdgmmStridedBatched_full_rank,&
      cublasDdgmmStridedBatched_rank_0,&
      cublasDdgmmStridedBatched_rank_1
  end interface
  
  interface cublasCdgmmStridedBatched
    function cublasCdgmmStridedBatched_(handle,side,m,n,A,lda,stride_A,&
&x,incx,stride_x,C,ldc,stride_C,batchCount) bind(c, &
&name="cublasCdgmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdgmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: stride_A
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stride_x
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: stride_C
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCdgmmStridedBatched_full_rank,&
      cublasCdgmmStridedBatched_rank_0,&
      cublasCdgmmStridedBatched_rank_1
  end interface
  !>  \brief BLAS Level 3 API
  !> 
  !>     \details
  !>     dgmmStridedBatched performs one of the batched matrix-matrix operations
  !> 
  !>         C_i = A_i * diag(x_i)   if side == HIPBLAS_SIDE_RIGHT   for i = 0, 1, ... batchCount-1
  !>         C_i = diag(x_i) * A_i   if side == HIPBLAS_SIDE_LEFT    for i = 0, 1, ... batchCount-1
  !> 
  !>     where C_i and A_i are m by n dimensional matrices. diag(x_i) is a diagonal matrix
  !>     and x_i is vector of dimension n if side == HIPBLAS_SIDE_RIGHT and dimension m
  !>     if side == HIPBLAS_SIDE_LEFT.
  !> 
  !>     @param[in]
  !>     handle    [hipblasHandle_t]
  !>               handle to the hipblas library context queue.
  !>     @param[in]
  !>     side      [hipblasSideMode_t]
  !>               specifies the side of diag(x)
  !>     @param[in]
  !>     m         [int]
  !>               matrix dimension m.
  !>     @param[in]
  !>     n         [int]
  !>               matrix dimension n.
  !>     @param[in]
  !>     A         device pointer to the first matrix A_0 on the GPU.
  !>               Each A_i is of dimension ( lda, n )
  !>     @param[in]
  !>     lda       [int]
  !>               specifies the leading dimension of A.
  !>     @param[in]
  !>     strideA  [hipblasStride]
  !>               stride from the start of one matrix (A_i) and the next one (A_i+1)
  !>     @param[in]
  !>     x         pointer to the first vector x_0 on the GPU.
  !>               Each x_i is of dimension n if side == HIPBLAS_SIDE_RIGHT and dimension
  !>               m if side == HIPBLAS_SIDE_LEFT
  !>     @param[in]
  !>     incx      [int]
  !>               specifies the increment between values of x
  !>     @param[in]
  !>     stridex  [hipblasStride]
  !>               stride from the start of one vector(x_i) and the next one (x_i+1)
  !>     @param[in, out]
  !>     C         device pointer to the first matrix C_0 on the GPU.
  !>               Each C_i is of dimension ( ldc, n ).
  !>     @param[in]
  !>     ldc       [int]
  !>               specifies the leading dimension of C.
  !>     @param[in]
  !>     strideC  [hipblasStride]
  !>               stride from the start of one matrix (C_i) and the next one (C_i+1)
  !>     @param[in]
  !>     batchCount [int]
  !>                 number of instances i in the batch.
  !>
  interface cublasZdgmmStridedBatched
    function cublasZdgmmStridedBatched_(handle,side,m,n,A,lda,strideA,&
&x,incx,stridex,C,ldc,strideC,batchCount) bind(c, &
&name="cublasZdgmmStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdgmmStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: x
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: C
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: strideC
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZdgmmStridedBatched_full_rank,&
      cublasZdgmmStridedBatched_rank_0,&
      cublasZdgmmStridedBatched_rank_1
  end interface
  
  interface cublasSgetrf
    function cublasSgetrf_(handle,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasSgetrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgetrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasSgetrf_full_rank,&
      cublasSgetrf_rank_0,&
      cublasSgetrf_rank_1
  end interface
  
  interface cublasDgetrf
    function cublasDgetrf_(handle,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasDgetrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgetrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasDgetrf_full_rank,&
      cublasDgetrf_rank_0,&
      cublasDgetrf_rank_1
  end interface
  
  interface cublasCgetrf
    function cublasCgetrf_(handle,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasCgetrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgetrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasCgetrf_full_rank,&
      cublasCgetrf_rank_0,&
      cublasCgetrf_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     getrf computes the LU factorization of a general n-by-n matrix A
  !>     using partial pivoting with row interchanges. The LU factorization can
  !>     be done without pivoting if ipiv is passed as a nullptr.
  !> 
  !>     In the case that ipiv is not null, the factorization has the form:
  !> 
  !>     \f[
  !>         A = PLU
  !>     \f]
  !> 
  !>     where P is a permutation matrix, L is lower triangular with unit
  !>     diagonal elements, and U is upper triangular.
  !> 
  !>     In the case that ipiv is null, the factorization is done without pivoting:
  !> 
  !>     \f[
  !>         A = LU
  !>     \f]
  !> 
  !>     @param[in]
  !>     handle    hipblasHandle_t.
  !>     @param[in]
  !>     n         int. n >= 0.\n
  !>               The number of columns and rows of the matrix A.
  !>     @param[inout]
  !>     A         pointer to type. Array on the GPU of dimension lda*n.\n
  !>               On entry, the n-by-n matrix A to be factored.
  !>               On exit, the factors L and U from the factorization.
  !>               The unit diagonal elements of L are not stored.
  !>     @param[in]
  !>     lda       int. lda >= n.\n
  !>               Specifies the leading dimension of A.
  !>     @param[out]
  !>     ipiv      pointer to int. Array on the GPU of dimension n.\n
  !>               The vector of pivot indices. Elements of ipiv are 1-based indices.
  !>               For 1 <= i <= n, the row i of the
  !>               matrix was interchanged with row ipiv[i].
  !>               Matrix P of the factorization can be derived from ipiv.
  !>               The factorization here can be done without pivoting if ipiv is passed
  !>               in as a nullptr.
  !>     @param[out]
  !>     info      pointer to a int on the GPU.\n
  !>               If info = 0, successful exit.
  !>               If info = j > 0, U is singular. U[j,j] is the first zero pivot.
  interface cublasZgetrf
    function cublasZgetrf_(handle,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasZgetrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgetrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasZgetrf_full_rank,&
      cublasZgetrf_rank_0,&
      cublasZgetrf_rank_1
  end interface
  
  interface cublasSgetrfBatched
    function cublasSgetrfBatched_(handle,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasSgetrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgetrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgetrfBatched_full_rank,&
      cublasSgetrfBatched_rank_0,&
      cublasSgetrfBatched_rank_1
  end interface
  
  interface cublasDgetrfBatched
    function cublasDgetrfBatched_(handle,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasDgetrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgetrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgetrfBatched_full_rank,&
      cublasDgetrfBatched_rank_0,&
      cublasDgetrfBatched_rank_1
  end interface
  
  interface cublasCgetrfBatched
    function cublasCgetrfBatched_(handle,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasCgetrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgetrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgetrfBatched_full_rank,&
      cublasCgetrfBatched_rank_0,&
      cublasCgetrfBatched_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     getrfBatched computes the LU factorization of a batch of general
  !>     n-by-n matrices using partial pivoting with row interchanges. The LU factorization can
  !>     be done without pivoting if ipiv is passed as a nullptr.
  !> 
  !>     In the case that ipiv is not null, the factorization of matrix \f$A_i\f$ in the batch has the form:
  !> 
  !>     \f[
  !>         A_i = P_iL_iU_i
  !>     \f]
  !> 
  !>     where \f$P_i\f$ is a permutation matrix, \f$L_i\f$ is lower triangular with unit
  !>     diagonal elements, and \f$U_i\f$ is upper triangular.
  !> 
  !>     In the case that ipiv is null, the factorization is done without pivoting:
  !> 
  !>     \f[
  !>         A_i = L_iU_i
  !>     \f]
  !> 
  !>     @param[in]
  !>     handle    hipblasHandle_t.
  !>     @param[in]
  !>     n         int. n >= 0.\n
  !>               The number of columns and rows of all matrices A_i in the batch.
  !>     @param[inout]
  !>     A         array of pointers to type. Each pointer points to an array on the GPU of dimension lda*n.\n
  !>               On entry, the n-by-n matrices A_i to be factored.
  !>               On exit, the factors L_i and U_i from the factorizations.
  !>               The unit diagonal elements of L_i are not stored.
  !>     @param[in]
  !>     lda       int. lda >= n.\n
  !>               Specifies the leading dimension of matrices A_i.
  !>     @param[out]
  !>     ipiv      pointer to int. Array on the GPU.\n
  !>               Contains the vectors of pivot indices ipiv_i (corresponding to A_i).
  !>               Dimension of ipiv_i is n.
  !>               Elements of ipiv_i are 1-based indices.
  !>               For each instance A_i in the batch and for 1 <= j <= n, the row j of the
  !>               matrix A_i was interchanged with row ipiv_i[j].
  !>               Matrix P_i of the factorization can be derived from ipiv_i.
  !>               The factorization here can be done without pivoting if ipiv is passed
  !>               in as a nullptr.
  !>     @param[out]
  !>     info      pointer to int. Array of batchCount integers on the GPU.\n
  !>               If info[i] = 0, successful exit for factorization of A_i.
  !>               If info[i] = j > 0, U_i is singular. U_i[j,j] is the first zero pivot.
  !>     @param[in]
  !>     batchCount int. batchCount >= 0.\n
  !>                 Number of matrices in the batch.
  interface cublasZgetrfBatched
    function cublasZgetrfBatched_(handle,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasZgetrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgetrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgetrfBatched_full_rank,&
      cublasZgetrfBatched_rank_0,&
      cublasZgetrfBatched_rank_1
  end interface
  
  interface cublasSgetrfStridedBatched
    function cublasSgetrfStridedBatched_(handle,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasSgetrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSgetrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgetrfStridedBatched_full_rank,&
      cublasSgetrfStridedBatched_rank_0,&
      cublasSgetrfStridedBatched_rank_1
  end interface
  
  interface cublasDgetrfStridedBatched
    function cublasDgetrfStridedBatched_(handle,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasDgetrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDgetrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgetrfStridedBatched_full_rank,&
      cublasDgetrfStridedBatched_rank_0,&
      cublasDgetrfStridedBatched_rank_1
  end interface
  
  interface cublasCgetrfStridedBatched
    function cublasCgetrfStridedBatched_(handle,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasCgetrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCgetrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgetrfStridedBatched_full_rank,&
      cublasCgetrfStridedBatched_rank_0,&
      cublasCgetrfStridedBatched_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     getrfStridedBatched computes the LU factorization of a batch of
  !>     general n-by-n matrices using partial pivoting with row interchanges. The LU factorization can
  !>     be done without pivoting if ipiv is passed as a nullptr.
  !> 
  !>     In the case that ipiv is not null, the factorization of matrix \f$A_i\f$ in the batch has the form:
  !> 
  !>     \f[
  !>         A_i = P_iL_iU_i
  !>     \f]
  !> 
  !>     where \f$P_i\f$ is a permutation matrix, \f$L_i\f$ is lower triangular with unit
  !>     diagonal elements, and \f$U_i\f$ is upper triangular.
  !> 
  !>     In the case that ipiv is null, the factorization is done without pivoting:
  !> 
  !>     \f[
  !>         A_i = L_iU_i
  !>     \f]
  !> 
  !>     @param[in]
  !>     handle    hipblasHandle_t.
  !>     @param[in]
  !>     n         int. n >= 0.\n
  !>               The number of columns and rows of all matrices A_i in the batch.
  !>     @param[inout]
  !>     A         pointer to type. Array on the GPU (the size depends on the value of strideA).\n
  !>               On entry, the n-by-n matrices A_i to be factored.
  !>               On exit, the factors L_i and U_i from the factorization.
  !>               The unit diagonal elements of L_i are not stored.
  !>     @param[in]
  !>     lda       int. lda >= n.\n
  !>               Specifies the leading dimension of matrices A_i.
  !>     @param[in]
  !>     strideA   hipblasStride.\n
  !>               Stride from the start of one matrix A_i to the next one A_(i+1).
  !>               There is no restriction for the value of strideA. Normal use case is strideA >= lda*n
  !>     @param[out]
  !>     ipiv      pointer to int. Array on the GPU (the size depends on the value of strideP).\n
  !>               Contains the vectors of pivots indices ipiv_i (corresponding to A_i).
  !>               Dimension of ipiv_i is n.
  !>               Elements of ipiv_i are 1-based indices.
  !>               For each instance A_i in the batch and for 1 <= j <= n, the row j of the
  !>               matrix A_i was interchanged with row ipiv_i[j].
  !>               Matrix P_i of the factorization can be derived from ipiv_i.
  !>               The factorization here can be done without pivoting if ipiv is passed
  !>               in as a nullptr.
  !>     @param[in]
  !>     strideP   hipblasStride.\n
  !>               Stride from the start of one vector ipiv_i to the next one ipiv_(i+1).
  !>               There is no restriction for the value of strideP. Normal use case is strideP >= n.
  !>     @param[out]
  !>     info      pointer to int. Array of batchCount integers on the GPU.\n
  !>               If info[i] = 0, successful exit for factorization of A_i.
  !>               If info[i] = j > 0, U_i is singular. U_i[j,j] is the first zero pivot.
  !>     @param[in]
  !>     batchCount int. batchCount >= 0.\n
  !>                 Number of matrices in the batch.
  interface cublasZgetrfStridedBatched
    function cublasZgetrfStridedBatched_(handle,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasZgetrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZgetrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgetrfStridedBatched_full_rank,&
      cublasZgetrfStridedBatched_rank_0,&
      cublasZgetrfStridedBatched_rank_1
  end interface
  
  interface cublasSgetrs
    function cublasSgetrs_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo) bind(c, name="cublasSgetrs")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgetrs_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasSgetrs_full_rank,&
      cublasSgetrs_rank_0,&
      cublasSgetrs_rank_1
  end interface
  
  interface cublasDgetrs
    function cublasDgetrs_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo) bind(c, name="cublasDgetrs")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgetrs_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasDgetrs_full_rank,&
      cublasDgetrs_rank_0,&
      cublasDgetrs_rank_1
  end interface
  
  interface cublasCgetrs
    function cublasCgetrs_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo) bind(c, name="cublasCgetrs")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgetrs_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasCgetrs_full_rank,&
      cublasCgetrs_rank_0,&
      cublasCgetrs_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     getrs solves a system of n linear equations on n variables in its factorized form.
  !> 
  !>     It solves one of the following systems, depending on the value of trans:
  !> 
  !>     \f[
  !>         \begin{array}{cl}
  !>         A X = B & \: \text{not transposed,}\\
  !>         A^T X = B & \: \text{transposed, or}\\
  !>         A^H X = B & \: \text{conjugate transposed.}
  !>         \end{array}
  !>     \f]
  !> 
  !>     Matrix A is defined by its triangular factors as returned by \ref hipblasSgetrf "getrf".
  !> 
  !>     @param[in]
  !>     handle      hipblasHandle_t.
  !>     @param[in]
  !>     trans       hipblasOperation_t.\n
  !>                 Specifies the form of the system of equations.
  !>     @param[in]
  !>     n           int. n >= 0.\n
  !>                 The order of the system, i.e. the number of columns and rows of A.
  !>     @param[in]
  !>     nrhs        int. nrhs >= 0.\n
  !>                 The number of right hand sides, i.e., the number of columns
  !>                 of the matrix B.
  !>     @param[in]
  !>     A           pointer to type. Array on the GPU of dimension lda*n.\n
  !>                 The factors L and U of the factorization A = P*L*U returned by \ref hipblasSgetrf "getrf".
  !>     @param[in]
  !>     lda         int. lda >= n.\n
  !>                 The leading dimension of A.
  !>     @param[in]
  !>     ipiv        pointer to int. Array on the GPU of dimension n.\n
  !>                 The pivot indices returned by \ref hipblasSgetrf "getrf".
  !>     @param[in,out]
  !>     B           pointer to type. Array on the GPU of dimension ldb*nrhs.\n
  !>                 On entry, the right hand side matrix B.
  !>                 On exit, the solution matrix X.
  !>     @param[in]
  !>     ldb         int. ldb >= n.\n
  !>                 The leading dimension of B.
  !>     @param[out]
  !>     info      pointer to a int on the host.\n
  !>               If info = 0, successful exit.
  !>               If info = j < 0, the j-th argument is invalid.
  interface cublasZgetrs
    function cublasZgetrs_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo) bind(c, name="cublasZgetrs")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgetrs_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasZgetrs_full_rank,&
      cublasZgetrs_rank_0,&
      cublasZgetrs_rank_1
  end interface
  
  interface cublasSgetrsBatched
    function cublasSgetrsBatched_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo,batchCount) bind(c, name="cublasSgetrsBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgetrsBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgetrsBatched_full_rank,&
      cublasSgetrsBatched_rank_0,&
      cublasSgetrsBatched_rank_1
  end interface
  
  interface cublasDgetrsBatched
    function cublasDgetrsBatched_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo,batchCount) bind(c, name="cublasDgetrsBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgetrsBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgetrsBatched_full_rank,&
      cublasDgetrsBatched_rank_0,&
      cublasDgetrsBatched_rank_1
  end interface
  
  interface cublasCgetrsBatched
    function cublasCgetrsBatched_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo,batchCount) bind(c, name="cublasCgetrsBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgetrsBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgetrsBatched_full_rank,&
      cublasCgetrsBatched_rank_0,&
      cublasCgetrsBatched_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details getrsBatched solves a batch of systems of n linear equations on n
  !>     variables in its factorized forms.
  !> 
  !>     For each instance i in the batch, it solves one of the following systems, depending on the value of trans:
  !> 
  !>     \f[
  !>         \begin{array}{cl}
  !>         A_i X_i = B_i & \: \text{not transposed,}\\
  !>         A_i^T X_i = B_i & \: \text{transposed, or}\\
  !>         A_i^H X_i = B_i & \: \text{conjugate transposed.}
  !>         \end{array}
  !>     \f]
  !> 
  !>     Matrix \f$A_i\f$ is defined by its triangular factors as returned by \ref hipblasSgetrfBatched "getrfBatched".
  !> 
  !>     @param[in]
  !>     handle      hipblasHandle_t.
  !>     @param[in]
  !>     trans       hipblasOperation_t.\n
  !>                 Specifies the form of the system of equations of each instance in the batch.
  !>     @param[in]
  !>     n           int. n >= 0.\n
  !>                 The order of the system, i.e. the number of columns and rows of all A_i matrices.
  !>     @param[in]
  !>     nrhs        int. nrhs >= 0.\n
  !>                 The number of right hand sides, i.e., the number of columns
  !>                 of all the matrices B_i.
  !>     @param[in]
  !>     A           Array of pointers to type. Each pointer points to an array on the GPU of dimension lda*n.\n
  !>                 The factors L_i and U_i of the factorization A_i = P_i*L_i*U_i returned by \ref hipblasSgetrfBatched "getrfBatched".
  !>     @param[in]
  !>     lda         int. lda >= n.\n
  !>                 The leading dimension of matrices A_i.
  !>     @param[in]
  !>     ipiv        pointer to int. Array on the GPU.\n
  !>                 Contains the vectors ipiv_i of pivot indices returned by \ref hipblasSgetrfBatched "getrfBatched".
  !>     @param[in,out]
  !>     B           Array of pointers to type. Each pointer points to an array on the GPU of dimension ldb*nrhs.\n
  !>                 On entry, the right hand side matrices B_i.
  !>                 On exit, the solution matrix X_i of each system in the batch.
  !>     @param[in]
  !>     ldb         int. ldb >= n.\n
  !>                 The leading dimension of matrices B_i.
  !>     @param[out]
  !>     info      pointer to a int on the host.\n
  !>               If info = 0, successful exit.
  !>               If info = j < 0, the j-th argument is invalid.
  !>     @param[in]
  !>     batchCount int. batchCount >= 0.\n
  !>                 Number of instances (systems) in the batch.
  !>
  interface cublasZgetrsBatched
    function cublasZgetrsBatched_(handle,trans,n,nrhs,A,lda,ipiv,B,ldb,&
&myInfo,batchCount) bind(c, name="cublasZgetrsBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgetrsBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgetrsBatched_full_rank,&
      cublasZgetrsBatched_rank_0,&
      cublasZgetrsBatched_rank_1
  end interface
  
  interface cublasSgetrsStridedBatched
    function cublasSgetrsStridedBatched_(handle,trans,n,nrhs,A,lda,&
&strideA,ipiv,strideP,B,ldb,strideB,myInfo,batchCount) bind(c, &
&name="cublasSgetrsStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSgetrsStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgetrsStridedBatched_full_rank,&
      cublasSgetrsStridedBatched_rank_0,&
      cublasSgetrsStridedBatched_rank_1
  end interface
  
  interface cublasDgetrsStridedBatched
    function cublasDgetrsStridedBatched_(handle,trans,n,nrhs,A,lda,&
&strideA,ipiv,strideP,B,ldb,strideB,myInfo,batchCount) bind(c, &
&name="cublasDgetrsStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDgetrsStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgetrsStridedBatched_full_rank,&
      cublasDgetrsStridedBatched_rank_0,&
      cublasDgetrsStridedBatched_rank_1
  end interface
  
  interface cublasCgetrsStridedBatched
    function cublasCgetrsStridedBatched_(handle,trans,n,nrhs,A,lda,&
&strideA,ipiv,strideP,B,ldb,strideB,myInfo,batchCount) bind(c, &
&name="cublasCgetrsStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCgetrsStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgetrsStridedBatched_full_rank,&
      cublasCgetrsStridedBatched_rank_0,&
      cublasCgetrsStridedBatched_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     getrsStridedBatched solves a batch of systems of n linear equations
  !>     on n variables in its factorized forms.
  !> 
  !>     For each instance i in the batch, it solves one of the following systems, depending on the value of trans:
  !> 
  !>     \f[
  !>         \begin{array}{cl}
  !>         A_i X_i = B_i & \: \text{not transposed,}\\
  !>         A_i^T X_i = B_i & \: \text{transposed, or}\\
  !>         A_i^H X_i = B_i & \: \text{conjugate transposed.}
  !>         \end{array}
  !>     \f]
  !> 
  !>     Matrix \f$A_i\f$ is defined by its triangular factors as returned by \ref hipblasSgetrfStridedBatched "getrfStridedBatched".
  !> 
  !>     @param[in]
  !>     handle      hipblasHandle_t.
  !>     @param[in]
  !>     trans       hipblasOperation_t.\n
  !>                 Specifies the form of the system of equations of each instance in the batch.
  !>     @param[in]
  !>     n           int. n >= 0.\n
  !>                 The order of the system, i.e. the number of columns and rows of all A_i matrices.
  !>     @param[in]
  !>     nrhs        int. nrhs >= 0.\n
  !>                 The number of right hand sides, i.e., the number of columns
  !>                 of all the matrices B_i.
  !>     @param[in]
  !>     A           pointer to type. Array on the GPU (the size depends on the value of strideA).\n
  !>                 The factors L_i and U_i of the factorization A_i = P_i*L_i*U_i returned by \ref hipblasSgetrfStridedBatched "getrfStridedBatched".
  !>     @param[in]
  !>     lda         int. lda >= n.\n
  !>                 The leading dimension of matrices A_i.
  !>     @param[in]
  !>     strideA     hipblasStride.\n
  !>                 Stride from the start of one matrix A_i to the next one A_(i+1).
  !>                 There is no restriction for the value of strideA. Normal use case is strideA >= lda*n.
  !>     @param[in]
  !>     ipiv        pointer to int. Array on the GPU (the size depends on the value of strideP).\n
  !>                 Contains the vectors ipiv_i of pivot indices returned by \ref hipblasSgetrfStridedBatched "getrfStridedBatched".
  !>     @param[in]
  !>     strideP     hipblasStride.\n
  !>                 Stride from the start of one vector ipiv_i to the next one ipiv_(i+1).
  !>                 There is no restriction for the value of strideP. Normal use case is strideP >= n.
  !>     @param[in,out]
  !>     B           pointer to type. Array on the GPU (size depends on the value of strideB).\n
  !>                 On entry, the right hand side matrices B_i.
  !>                 On exit, the solution matrix X_i of each system in the batch.
  !>     @param[in]
  !>     ldb         int. ldb >= n.\n
  !>                 The leading dimension of matrices B_i.
  !>     @param[in]
  !>     strideB     hipblasStride.\n
  !>                 Stride from the start of one matrix B_i to the next one B_(i+1).
  !>                 There is no restriction for the value of strideB. Normal use case is strideB >= ldb*nrhs.
  !>     @param[out]
  !>     info      pointer to a int on the host.\n
  !>               If info = 0, successful exit.
  !>               If info = j < 0, the j-th argument is invalid.
  !>     @param[in]
  !>     batchCount int. batchCount >= 0.\n
  !>                 Number of instances (systems) in the batch.
  !>
  interface cublasZgetrsStridedBatched
    function cublasZgetrsStridedBatched_(handle,trans,n,nrhs,A,lda,&
&strideA,ipiv,strideP,B,ldb,strideB,myInfo,batchCount) bind(c, &
&name="cublasZgetrsStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZgetrsStridedBatched_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans
      integer(c_int),value :: n
      integer(c_int),value :: nrhs
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: strideB
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgetrsStridedBatched_full_rank,&
      cublasZgetrsStridedBatched_rank_0,&
      cublasZgetrsStridedBatched_rank_1
  end interface
  
  interface cublasSgetriBatched
    function cublasSgetriBatched_(handle,n,A,lda,ipiv,C,ldc,myInfo,&
&batchCount) bind(c, name="cublasSgetriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgetriBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgetriBatched_full_rank,&
      cublasSgetriBatched_rank_0,&
      cublasSgetriBatched_rank_1
  end interface
  
  interface cublasDgetriBatched
    function cublasDgetriBatched_(handle,n,A,lda,ipiv,C,ldc,myInfo,&
&batchCount) bind(c, name="cublasDgetriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgetriBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgetriBatched_full_rank,&
      cublasDgetriBatched_rank_0,&
      cublasDgetriBatched_rank_1
  end interface
  
  interface cublasCgetriBatched
    function cublasCgetriBatched_(handle,n,A,lda,ipiv,C,ldc,myInfo,&
&batchCount) bind(c, name="cublasCgetriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgetriBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgetriBatched_full_rank,&
      cublasCgetriBatched_rank_0,&
      cublasCgetriBatched_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     getriBatched computes the inverse \f$C_i = A_i^{-1}\f$ of a batch of general n-by-n matrices \f$A_i\f$.
  !> 
  !>     The inverse is computed by solving the linear system
  !> 
  !>     \f[
  !>         A_i C_i = I
  !>     \f]
  !> 
  !>     where I is the identity matrix, and \f$A_i\f$ is factorized as \f$A_i = P_i  L_i  U_i\f$ as given by \ref hipblasSgetrfBatched "getrfBatched".
  !> 
  !>     @param[in]
  !>     handle    hipblasHandle_t.
  !>     @param[in]
  !>     n         int. n >= 0.\n
  !>               The number of rows and columns of all matrices A_i in the batch.
  !>     @param[in]
  !>     A         array of pointers to type. Each pointer points to an array on the GPU of dimension lda*n.\n
  !>               The factors L_i and U_i of the factorization A_i = P_i*L_i*U_i returned by \ref hipblasSgetrfBatched "getrfBatched".
  !>     @param[in]
  !>     lda       int. lda >= n.\n
  !>               Specifies the leading dimension of matrices A_i.
  !>     @param[in]
  !>     ipiv      pointer to int. Array on the GPU (the size depends on the value of strideP).\n
  !>               The pivot indices returned by \ref hipblasSgetrfBatched "getrfBatched".
  !>               ipiv can be passed in as a nullptr, this will assume that getrfBatched was called without partial pivoting.
  !>     @param[out]
  !>     C         array of pointers to type. Each pointer points to an array on the GPU of dimension ldc*n.\n
  !>               If info[i] = 0, the inverse of matrices A_i. Otherwise, undefined.
  !>     @param[in]
  !>     ldc       int. ldc >= n.\n
  !>               Specifies the leading dimension of C_i.
  !>     @param[out]
  !>     info      pointer to int. Array of batchCount integers on the GPU.\n
  !>               If info[i] = 0, successful exit for inversion of A_i.
  !>               If info[i] = j > 0, U_i is singular. U_i[j,j] is the first zero pivot.
  !>     @param[in]
  !>     batchCount int. batchCount >= 0.\n
  !>                 Number of matrices in the batch.
  !>
  interface cublasZgetriBatched
    function cublasZgetriBatched_(handle,n,A,lda,ipiv,C,ldc,myInfo,&
&batchCount) bind(c, name="cublasZgetriBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgetriBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr) :: C
      integer(c_int),value :: ldc
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgetriBatched_full_rank,&
      cublasZgetriBatched_rank_0,&
      cublasZgetriBatched_rank_1
  end interface
  
  interface cublasSgeqrf
    function cublasSgeqrf_(handle,m,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasSgeqrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgeqrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasSgeqrf_full_rank,&
      cublasSgeqrf_rank_0,&
      cublasSgeqrf_rank_1
  end interface
  
  interface cublasDgeqrf
    function cublasDgeqrf_(handle,m,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasDgeqrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgeqrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasDgeqrf_full_rank,&
      cublasDgeqrf_rank_0,&
      cublasDgeqrf_rank_1
  end interface
  
  interface cublasCgeqrf
    function cublasCgeqrf_(handle,m,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasCgeqrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeqrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasCgeqrf_full_rank,&
      cublasCgeqrf_rank_0,&
      cublasCgeqrf_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     geqrf computes a QR factorization of a general m-by-n matrix A.
  !> 
  !>     The factorization has the form
  !> 
  !>     \f[
  !>         A = Q\left[\begin{array}{c}
  !>         R\\
  !>         0
  !>         \end{array}\right]
  !>     \f]
  !> 
  !>     where R is upper triangular (upper trapezoidal if m < n), and Q is
  !>     a m-by-m orthogonal/unitary matrix represented as the product of Householder matrices
  !> 
  !>     \f[
  !>         Q = H_1H_2\cdots H_k, \quad \text{with} \: k = \text{min}(m,n)
  !>     \f]
  !> 
  !>     Each Householder matrix \f$H_i\f$ is given by
  !> 
  !>     \f[
  !>         H_i = I - \text{ipiv}[i] \cdot v_i v_i'
  !>     \f]
  !> 
  !>     where the first i-1 elements of the Householder vector \f$v_i\f$ are zero, and \f$v_i[i] = 1\f$.
  !> 
  !>     @param[in]
  !>     handle    hipblasHandle_t.
  !>     @param[in]
  !>     m         int. m >= 0.\n
  !>               The number of rows of the matrix A.
  !>     @param[in]
  !>     n         int. n >= 0.\n
  !>               The number of columns of the matrix A.
  !>     @param[inout]
  !>     A         pointer to type. Array on the GPU of dimension lda*n.\n
  !>               On entry, the m-by-n matrix to be factored.
  !>               On exit, the elements on and above the diagonal contain the
  !>               factor R; the elements below the diagonal are the last m - i elements
  !>               of Householder vector v_i.
  !>     @param[in]
  !>     lda       int. lda >= m.\n
  !>               Specifies the leading dimension of A.
  !>     @param[out]
  !>     ipiv      pointer to type. Array on the GPU of dimension min(m,n).\n
  !>               The Householder scalars.
  !>     @param[out]
  !>     info      pointer to a int on the host.\n
  !>               If info = 0, successful exit.
  !>               If info = j < 0, the j-th argument is invalid.
  !>
  interface cublasZgeqrf
    function cublasZgeqrf_(handle,m,n,A,lda,ipiv,myInfo) bind(c, &
&name="cublasZgeqrf")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeqrf_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: ipiv
      type(c_ptr),value :: myInfo
    end function

    module procedure &
      cublasZgeqrf_full_rank,&
      cublasZgeqrf_rank_0,&
      cublasZgeqrf_rank_1
  end interface
  
  interface cublasSgeqrfBatched
    function cublasSgeqrfBatched_(handle,m,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasSgeqrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSgeqrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgeqrfBatched_full_rank,&
      cublasSgeqrfBatched_rank_0,&
      cublasSgeqrfBatched_rank_1
  end interface
  
  interface cublasDgeqrfBatched
    function cublasDgeqrfBatched_(handle,m,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasDgeqrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDgeqrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgeqrfBatched_full_rank,&
      cublasDgeqrfBatched_rank_0,&
      cublasDgeqrfBatched_rank_1
  end interface
  
  interface cublasCgeqrfBatched
    function cublasCgeqrfBatched_(handle,m,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasCgeqrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCgeqrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgeqrfBatched_full_rank,&
      cublasCgeqrfBatched_rank_0,&
      cublasCgeqrfBatched_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     geqrfBatched computes the QR factorization of a batch of general
  !>     m-by-n matrices.
  !> 
  !>     The factorization of matrix \f$A_i\f$ in the batch has the form
  !> 
  !>     \f[
  !>         A_i = Q_i\left[\begin{array}{c}
  !>         R_i\\
  !>         0
  !>         \end{array}\right]
  !>     \f]
  !> 
  !>     where \f$R_i\f$ is upper triangular (upper trapezoidal if m < n), and \f$Q_i\f$ is
  !>     a m-by-m orthogonal/unitary matrix represented as the product of Householder matrices
  !> 
  !>     \f[
  !>         Q_i = H_{i_1}H_{i_2}\cdots H_{i_k}, \quad \text{with} \: k = \text{min}(m,n)
  !>     \f]
  !> 
  !>     Each Householder matrix \f$H_{i_j}\f$ is given by
  !> 
  !>     \f[
  !>         H_{i_j} = I - \text{ipiv}_i[j] \cdot v_{i_j} v_{i_j}'
  !>     \f]
  !> 
  !>     where the first j-1 elements of Householder vector \f$v_{i_j}\f$ are zero, and \f$v_{i_j}[j] = 1\f$.
  !> 
  !>     @param[in]
  !>     handle    hipblasHandle_t.
  !>     @param[in]
  !>     m         int. m >= 0.\n
  !>               The number of rows of all the matrices A_i in the batch.
  !>     @param[in]
  !>     n         int. n >= 0.\n
  !>               The number of columns of all the matrices A_i in the batch.
  !>     @param[inout]
  !>     A         Array of pointers to type. Each pointer points to an array on the GPU of dimension lda*n.\n
  !>               On entry, the m-by-n matrices A_i to be factored.
  !>               On exit, the elements on and above the diagonal contain the
  !>               factor R_i. The elements below the diagonal are the last m - j elements
  !>               of Householder vector v_(i_j).
  !>     @param[in]
  !>     lda       int. lda >= m.\n
  !>               Specifies the leading dimension of matrices A_i.
  !>     @param[out]
  !>     ipiv      array of pointers to type. Each pointer points to an array on the GPU
  !>               of dimension min(m, n).\n
  !>               Contains the vectors ipiv_i of corresponding Householder scalars.
  !>     @param[out]
  !>     info      pointer to a int on the host.\n
  !>               If info = 0, successful exit.
  !>               If info = k < 0, the k-th argument is invalid.
  !>     @param[in]
  !>     batchCount  int. batchCount >= 0.\n
  !>                  Number of matrices in the batch.
  interface cublasZgeqrfBatched
    function cublasZgeqrfBatched_(handle,m,n,A,lda,ipiv,myInfo,&
&batchCount) bind(c, name="cublasZgeqrfBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZgeqrfBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr) :: A
      integer(c_int),value :: lda
      type(c_ptr) :: ipiv
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgeqrfBatched_full_rank,&
      cublasZgeqrfBatched_rank_0,&
      cublasZgeqrfBatched_rank_1
  end interface
  
  interface cublasSgeqrfStridedBatched
    function cublasSgeqrfStridedBatched_(handle,m,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasSgeqrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSgeqrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasSgeqrfStridedBatched_full_rank,&
      cublasSgeqrfStridedBatched_rank_0,&
      cublasSgeqrfStridedBatched_rank_1
  end interface
  
  interface cublasDgeqrfStridedBatched
    function cublasDgeqrfStridedBatched_(handle,m,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasDgeqrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDgeqrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasDgeqrfStridedBatched_full_rank,&
      cublasDgeqrfStridedBatched_rank_0,&
      cublasDgeqrfStridedBatched_rank_1
  end interface
  
  interface cublasCgeqrfStridedBatched
    function cublasCgeqrfStridedBatched_(handle,m,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasCgeqrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCgeqrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasCgeqrfStridedBatched_full_rank,&
      cublasCgeqrfStridedBatched_rank_0,&
      cublasCgeqrfStridedBatched_rank_1
  end interface
  !>  \brief SOLVER API
  !> 
  !>     \details
  !>     geqrfStridedBatched computes the QR factorization of a batch of
  !>     general m-by-n matrices.
  !> 
  !>     The factorization of matrix \f$A_i\f$ in the batch has the form
  !> 
  !>     \f[
  !>         A_i = Q_i\left[\begin{array}{c}
  !>         R_i\\
  !>         0
  !>         \end{array}\right]
  !>     \f]
  !> 
  !>     where \f$R_i\f$ is upper triangular (upper trapezoidal if m < n), and \f$Q_i\f$ is
  !>     a m-by-m orthogonal/unitary matrix represented as the product of Householder matrices
  !> 
  !>     \f[
  !>         Q_i = H_{i_1}H_{i_2}\cdots H_{i_k}, \quad \text{with} \: k = \text{min}(m,n)
  !>     \f]
  !> 
  !>     Each Householder matrix \f$H_{i_j}\f$ is given by
  !> 
  !>     \f[
  !>         H_{i_j} = I - \text{ipiv}_j[j] \cdot v_{i_j} v_{i_j}'
  !>     \f]
  !> 
  !>     where the first j-1 elements of Householder vector \f$v_{i_j}\f$ are zero, and \f$v_{i_j}[j] = 1\f$.
  !> 
  !>     @param[in]
  !>     handle    hipblasHandle_t.
  !>     @param[in]
  !>     m         int. m >= 0.\n
  !>               The number of rows of all the matrices A_i in the batch.
  !>     @param[in]
  !>     n         int. n >= 0.\n
  !>               The number of columns of all the matrices A_i in the batch.
  !>     @param[inout]
  !>     A         pointer to type. Array on the GPU (the size depends on the value of strideA).\n
  !>               On entry, the m-by-n matrices A_i to be factored.
  !>               On exit, the elements on and above the diagonal contain the
  !>               factor R_i. The elements below the diagonal are the last m - j elements
  !>               of Householder vector v_(i_j).
  !>     @param[in]
  !>     lda       int. lda >= m.\n
  !>               Specifies the leading dimension of matrices A_i.
  !>     @param[in]
  !>     strideA   hipblasStride.\n
  !>               Stride from the start of one matrix A_i to the next one A_(i+1).
  !>               There is no restriction for the value of strideA. Normal use case is strideA >= lda*n.
  !>     @param[out]
  !>     ipiv      pointer to type. Array on the GPU (the size depends on the value of strideP).\n
  !>               Contains the vectors ipiv_i of corresponding Householder scalars.
  !>     @param[in]
  !>     strideP   hipblasStride.\n
  !>               Stride from the start of one vector ipiv_i to the next one ipiv_(i+1).
  !>               There is no restriction for the value
  !>               of strideP. Normal use is strideP >= min(m,n).
  !>     @param[out]
  !>     info      pointer to a int on the host.\n
  !>               If info = 0, successful exit.
  !>               If info = k < 0, the k-th argument is invalid.
  !>     @param[in]
  !>     batchCount  int. batchCount >= 0.\n
  !>                  Number of matrices in the batch.
  interface cublasZgeqrfStridedBatched
    function cublasZgeqrfStridedBatched_(handle,m,n,A,lda,strideA,ipiv,&
&strideP,myInfo,batchCount) bind(c, name="cublasZgeqrfStridedBatched")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZgeqrfStridedBatched_
      type(c_ptr),value :: handle
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: strideA
      type(c_ptr),value :: ipiv
      integer(c_int64_t),value :: strideP
      type(c_ptr),value :: myInfo
      integer(c_int),value :: batchCount
    end function

    module procedure &
      cublasZgeqrfStridedBatched_full_rank,&
      cublasZgeqrfStridedBatched_rank_0,&
      cublasZgeqrfStridedBatched_rank_1
  end interface
  
  interface cublasGemmEx
    function cublasGemmEx_(handle,trans_a,trans_b,m,n,k,alpha,a,a_type,&
&lda,b,b_type,ldb,beta,c,c_type,ldc,compute_type,algo) bind(c, &
&name="cublasGemmEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasGemmEx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans_a
      integer(kind(CUBLAS_OP_N)),value :: trans_b
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: alpha
      type(c_ptr),value :: a
      integer(kind(CUDA_R_16F)),value :: a_type
      integer(c_int),value :: lda
      type(c_ptr),value :: b
      integer(kind(CUDA_R_16F)),value :: b_type
      integer(c_int),value :: ldb
      type(c_ptr),value :: beta
      type(c_ptr),value :: c
      integer(kind(CUDA_R_16F)),value :: c_type
      integer(c_int),value :: ldc
      integer(kind(CUDA_R_16F)),value :: compute_type
      integer(kind(CUBLAS_GEMM_DEFAULT)),value :: algo
    end function

  end interface
  
  interface cublasGemmBatchedEx
    function cublasGemmBatchedEx_(handle,trans_a,trans_b,m,n,k,alpha,a,&
&a_type,lda,b,b_type,ldb,beta,c,c_type,ldc,batch_count,compute_type,&
&algo) bind(c, name="cublasGemmBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasGemmBatchedEx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans_a
      integer(kind(CUBLAS_OP_N)),value :: trans_b
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: alpha
      type(c_ptr) :: a
      integer(kind(CUDA_R_16F)),value :: a_type
      integer(c_int),value :: lda
      type(c_ptr) :: b
      integer(kind(CUDA_R_16F)),value :: b_type
      integer(c_int),value :: ldb
      type(c_ptr),value :: beta
      type(c_ptr) :: c
      integer(kind(CUDA_R_16F)),value :: c_type
      integer(c_int),value :: ldc
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: compute_type
      integer(kind(CUBLAS_GEMM_DEFAULT)),value :: algo
    end function

  end interface
  
  interface cublasGemmStridedBatchedEx
    function cublasGemmStridedBatchedEx_(handle,trans_a,trans_b,m,n,k,&
&alpha,a,a_type,lda,stride_A,b,b_type,ldb,stride_B,beta,c,c_type,ldc,&
&stride_C,batch_count,compute_type,algo) bind(c, &
&name="cublasGemmStridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasGemmStridedBatchedEx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_OP_N)),value :: trans_a
      integer(kind(CUBLAS_OP_N)),value :: trans_b
      integer(c_int),value :: m
      integer(c_int),value :: n
      integer(c_int),value :: k
      type(c_ptr),value :: alpha
      type(c_ptr),value :: a
      integer(kind(CUDA_R_16F)),value :: a_type
      integer(c_int),value :: lda
      integer(c_int64_t),value :: stride_A
      type(c_ptr),value :: b
      integer(kind(CUDA_R_16F)),value :: b_type
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: stride_B
      type(c_ptr),value :: beta
      type(c_ptr),value :: c
      integer(kind(CUDA_R_16F)),value :: c_type
      integer(c_int),value :: ldc
      integer(c_int64_t),value :: stride_C
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: compute_type
      integer(kind(CUBLAS_GEMM_DEFAULT)),value :: algo
    end function

  end interface
  
  interface cublasTrsmEx
    function cublasTrsmEx_(handle,side,uplo,transA,diag,m,n,alpha,A,&
&lda,B,ldb,invA,invA_size,compute_type) bind(c, name="cublasTrsmEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasTrsmEx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      type(c_ptr),value :: invA
      integer(c_int),value :: invA_size
      integer(kind(CUDA_R_16F)),value :: compute_type
    end function

  end interface
  
  interface cublasTrsmBatchedEx
    function cublasTrsmBatchedEx_(handle,side,uplo,transA,diag,m,n,&
&alpha,A,lda,B,ldb,batch_count,invA,invA_size,compute_type) bind(c, &
&name="cublasTrsmBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasTrsmBatchedEx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int),value :: batch_count
      type(c_ptr),value :: invA
      integer(c_int),value :: invA_size
      integer(kind(CUDA_R_16F)),value :: compute_type
    end function

  end interface
  
  interface cublasTrsmStridedBatchedEx
    function cublasTrsmStridedBatchedEx_(handle,side,uplo,transA,diag,&
&m,n,alpha,A,lda,stride_A,B,ldb,stride_B,batch_count,invA,invA_size,&
&stride_invA,compute_type) bind(c, name="cublasTrsmStridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasTrsmStridedBatchedEx_
      type(c_ptr),value :: handle
      integer(kind(CUBLAS_SIDE_LEFT)),value :: side
      integer(kind(CUBLAS_FILL_MODE_UPPER)),value :: uplo
      integer(kind(CUBLAS_OP_N)),value :: transA
      integer(kind(CUBLAS_DIAG_NON_UNIT)),value :: diag
      integer(c_int),value :: m
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      type(c_ptr),value :: A
      integer(c_int),value :: lda
      integer(c_int64_t),value :: stride_A
      type(c_ptr),value :: B
      integer(c_int),value :: ldb
      integer(c_int64_t),value :: stride_B
      integer(c_int),value :: batch_count
      type(c_ptr),value :: invA
      integer(c_int),value :: invA_size
      integer(c_int64_t),value :: stride_invA
      integer(kind(CUDA_R_16F)),value :: compute_type
    end function

  end interface
  
  interface cublasAxpyEx
    function cublasAxpyEx_(handle,n,alpha,alphaType,x,xType,incx,y,&
&yType,incy,executionType) bind(c, name="cublasAxpyEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasAxpyEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      integer(kind(CUDA_R_16F)),value :: alphaType
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasAxpyBatchedEx
    function cublasAxpyBatchedEx_(handle,n,alpha,alphaType,x,xType,&
&incx,y,yType,incy,batch_count,executionType) bind(c, &
&name="cublasAxpyBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasAxpyBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      integer(kind(CUDA_R_16F)),value :: alphaType
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasAxpyStridedBatchedEx
    function cublasAxpyStridedBatchedEx_(handle,n,alpha,alphaType,x,&
&xType,incx,stridex,y,yType,incy,stridey,batch_count,executionType) &
&bind(c, name="cublasAxpyStridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasAxpyStridedBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      integer(kind(CUDA_R_16F)),value :: alphaType
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasDotEx
    function cublasDotEx_(handle,n,x,xType,incx,y,yType,incy,myResult,&
&resultType,executionType) bind(c, name="cublasDotEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDotEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasDotcEx
    function cublasDotcEx_(handle,n,x,xType,incx,y,yType,incy,myResult,&
&resultType,executionType) bind(c, name="cublasDotcEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDotcEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasDotBatchedEx
    function cublasDotBatchedEx_(handle,n,x,xType,incx,y,yType,incy,&
&batch_count,myResult,resultType,executionType) bind(c, &
&name="cublasDotBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDotBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(c_int),value :: batch_count
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasDotcBatchedEx
    function cublasDotcBatchedEx_(handle,n,x,xType,incx,y,yType,incy,&
&batch_count,myResult,resultType,executionType) bind(c, &
&name="cublasDotcBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDotcBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(c_int),value :: batch_count
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasDotStridedBatchedEx
    function cublasDotStridedBatchedEx_(handle,n,x,xType,incx,stridex,&
&y,yType,incy,stridey,batch_count,myResult,resultType,executionType) &
&bind(c, name="cublasDotStridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDotStridedBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batch_count
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasDotcStridedBatchedEx
    function cublasDotcStridedBatchedEx_(handle,n,x,xType,incx,stridex,&
&y,yType,incy,stridey,batch_count,myResult,resultType,executionType) &
&bind(c, name="cublasDotcStridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDotcStridedBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      integer(c_int),value :: batch_count
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasNrm2Ex
    function cublasNrm2Ex_(handle,n,x,xType,incx,myResult,resultType,&
&executionType) bind(c, name="cublasNrm2Ex")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasNrm2Ex_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasNrm2BatchedEx
    function cublasNrm2BatchedEx_(handle,n,x,xType,incx,batch_count,&
&myResult,resultType,executionType) bind(c, &
&name="cublasNrm2BatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasNrm2BatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int),value :: batch_count
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasNrm2StridedBatchedEx
    function cublasNrm2StridedBatchedEx_(handle,n,x,xType,incx,stridex,&
&batch_count,myResult,resultType,executionType) bind(c, &
&name="cublasNrm2StridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasNrm2StridedBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batch_count
      type(c_ptr),value :: myResult
      integer(kind(CUDA_R_16F)),value :: resultType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasRotEx
    function cublasRotEx_(handle,n,x,xType,incx,y,yType,incy,c,s,&
&csType,executionType) bind(c, name="cublasRotEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasRotEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(kind(CUDA_R_16F)),value :: csType
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasRotBatchedEx
    function cublasRotBatchedEx_(handle,n,x,xType,incx,y,yType,incy,c,&
&s,csType,batch_count,executionType) bind(c, &
&name="cublasRotBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasRotBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(kind(CUDA_R_16F)),value :: csType
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasRotStridedBatchedEx
    function cublasRotStridedBatchedEx_(handle,n,x,xType,incx,stridex,&
&y,yType,incy,stridey,c,s,csType,batch_count,executionType) bind(c, &
&name="cublasRotStridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasRotStridedBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      type(c_ptr),value :: y
      integer(kind(CUDA_R_16F)),value :: yType
      integer(c_int),value :: incy
      integer(c_int64_t),value :: stridey
      type(c_ptr),value :: c
      type(c_ptr),value :: s
      integer(kind(CUDA_R_16F)),value :: csType
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasScalEx
    function cublasScalEx_(handle,n,alpha,alphaType,x,xType,incx,&
&executionType) bind(c, name="cublasScalEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScalEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      integer(kind(CUDA_R_16F)),value :: alphaType
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasScalBatchedEx
    function cublasScalBatchedEx_(handle,n,alpha,alphaType,x,xType,&
&incx,batch_count,executionType) bind(c, name="cublasScalBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScalBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      integer(kind(CUDA_R_16F)),value :: alphaType
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface
  
  interface cublasScalStridedBatchedEx
    function cublasScalStridedBatchedEx_(handle,n,alpha,alphaType,x,&
&xType,incx,stridex,batch_count,executionType) bind(c, &
&name="cublasScalStridedBatchedEx")
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScalStridedBatchedEx_
      type(c_ptr),value :: handle
      integer(c_int),value :: n
      type(c_ptr),value :: alpha
      integer(kind(CUDA_R_16F)),value :: alphaType
      type(c_ptr),value :: x
      integer(kind(CUDA_R_16F)),value :: xType
      integer(c_int),value :: incx
      integer(c_int64_t),value :: stridex
      integer(c_int),value :: batch_count
      integer(kind(CUDA_R_16F)),value :: executionType
    end function

  end interface

  contains
    function cublasIsamax_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamax_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIsamax_rank_0 = cublasIsamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIsamax_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamax_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIsamax_rank_1 = cublasIsamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIdamax_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamax_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIdamax_rank_0 = cublasIdamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIdamax_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamax_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIdamax_rank_1 = cublasIdamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIcamax_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamax_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIcamax_rank_0 = cublasIcamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIcamax_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamax_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIcamax_rank_1 = cublasIcamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIzamax_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamax_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIzamax_rank_0 = cublasIzamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIzamax_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamax_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIzamax_rank_1 = cublasIzamax_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIsamaxBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsamaxBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsamaxBatched_full_rank = cublasIsamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIsamaxBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamaxBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsamaxBatched_rank_0 = cublasIsamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIsamaxBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamaxBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsamaxBatched_rank_1 = cublasIsamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIdamaxBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdamaxBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdamaxBatched_full_rank = cublasIdamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIdamaxBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamaxBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdamaxBatched_rank_0 = cublasIdamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIdamaxBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamaxBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdamaxBatched_rank_1 = cublasIdamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIcamaxBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcamaxBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcamaxBatched_full_rank = cublasIcamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIcamaxBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamaxBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcamaxBatched_rank_0 = cublasIcamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIcamaxBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamaxBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcamaxBatched_rank_1 = cublasIcamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIzamaxBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzamaxBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzamaxBatched_full_rank = cublasIzamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIzamaxBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamaxBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzamaxBatched_rank_0 = cublasIzamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIzamaxBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamaxBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzamaxBatched_rank_1 = cublasIzamaxBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIsamaxStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsamaxStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsamaxStridedBatched_rank_0 = &
&cublasIsamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIsamaxStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsamaxStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsamaxStridedBatched_rank_1 = &
&cublasIsamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIdamaxStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdamaxStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdamaxStridedBatched_rank_0 = &
&cublasIdamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIdamaxStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdamaxStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdamaxStridedBatched_rank_1 = &
&cublasIdamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIcamaxStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcamaxStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcamaxStridedBatched_rank_0 = &
&cublasIcamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIcamaxStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcamaxStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcamaxStridedBatched_rank_1 = &
&cublasIcamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIzamaxStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzamaxStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzamaxStridedBatched_rank_0 = &
&cublasIzamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIzamaxStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzamaxStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzamaxStridedBatched_rank_1 = &
&cublasIzamaxStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIsamin_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamin_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIsamin_rank_0 = cublasIsamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIsamin_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsamin_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIsamin_rank_1 = cublasIsamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIdamin_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamin_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIdamin_rank_0 = cublasIdamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIdamin_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdamin_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIdamin_rank_1 = cublasIdamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIcamin_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamin_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIcamin_rank_0 = cublasIcamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIcamin_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcamin_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIcamin_rank_1 = cublasIcamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIzamin_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamin_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIzamin_rank_0 = cublasIzamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIzamin_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzamin_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasIzamin_rank_1 = cublasIzamin_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasIsaminBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsaminBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsaminBatched_full_rank = cublasIsaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIsaminBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsaminBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsaminBatched_rank_0 = cublasIsaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIsaminBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIsaminBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsaminBatched_rank_1 = cublasIsaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIdaminBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdaminBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdaminBatched_full_rank = cublasIdaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIdaminBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdaminBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdaminBatched_rank_0 = cublasIdaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIdaminBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIdaminBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdaminBatched_rank_1 = cublasIdaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIcaminBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcaminBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcaminBatched_full_rank = cublasIcaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIcaminBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcaminBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcaminBatched_rank_0 = cublasIcaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIcaminBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIcaminBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcaminBatched_rank_1 = cublasIcaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIzaminBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzaminBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzaminBatched_full_rank = cublasIzaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIzaminBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzaminBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzaminBatched_rank_0 = cublasIzaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIzaminBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasIzaminBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzaminBatched_rank_1 = cublasIzaminBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasIsaminStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsaminStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsaminStridedBatched_rank_0 = &
&cublasIsaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIsaminStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIsaminStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIsaminStridedBatched_rank_1 = &
&cublasIsaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIdaminStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdaminStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdaminStridedBatched_rank_0 = &
&cublasIdaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIdaminStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIdaminStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIdaminStridedBatched_rank_1 = &
&cublasIdaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIcaminStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcaminStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcaminStridedBatched_rank_0 = &
&cublasIcaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIcaminStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIcaminStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIcaminStridedBatched_rank_1 = &
&cublasIcaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIzaminStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzaminStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzaminStridedBatched_rank_0 = &
&cublasIzaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasIzaminStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasIzaminStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasIzaminStridedBatched_rank_1 = &
&cublasIzaminStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasSasum_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSasum_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasSasum_rank_0 = cublasSasum_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasSasum_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSasum_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasSasum_rank_1 = cublasSasum_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasDasum_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDasum_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDasum_rank_0 = cublasDasum_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasDasum_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDasum_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDasum_rank_1 = cublasDasum_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasScasum_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScasum_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasScasum_rank_0 = cublasScasum_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasScasum_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScasum_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasScasum_rank_1 = cublasScasum_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasDzasum_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDzasum_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDzasum_rank_0 = cublasDzasum_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasDzasum_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDzasum_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDzasum_rank_1 = cublasDzasum_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasSasumBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSasumBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSasumBatched_full_rank = cublasSasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasSasumBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSasumBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSasumBatched_rank_0 = cublasSasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasSasumBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSasumBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSasumBatched_rank_1 = cublasSasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDasumBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDasumBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDasumBatched_full_rank = cublasDasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDasumBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDasumBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDasumBatched_rank_0 = cublasDasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDasumBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDasumBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDasumBatched_rank_1 = cublasDasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasScasumBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScasumBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScasumBatched_full_rank = cublasScasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasScasumBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScasumBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScasumBatched_rank_0 = cublasScasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasScasumBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScasumBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScasumBatched_rank_1 = cublasScasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDzasumBatched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDzasumBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDzasumBatched_full_rank = cublasDzasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDzasumBatched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDzasumBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDzasumBatched_rank_0 = cublasDzasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDzasumBatched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDzasumBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDzasumBatched_rank_1 = cublasDzasumBatched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasSasumStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSasumStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSasumStridedBatched_rank_0 = &
&cublasSasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasSasumStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSasumStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSasumStridedBatched_rank_1 = &
&cublasSasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDasumStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDasumStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDasumStridedBatched_rank_0 = &
&cublasDasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDasumStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDasumStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDasumStridedBatched_rank_1 = &
&cublasDasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasScasumStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScasumStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScasumStridedBatched_rank_0 = &
&cublasScasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasScasumStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScasumStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScasumStridedBatched_rank_1 = &
&cublasScasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDzasumStridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDzasumStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDzasumStridedBatched_rank_0 = &
&cublasDzasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDzasumStridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDzasumStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDzasumStridedBatched_rank_1 = &
&cublasDzasumStridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasSaxpy_rank_0(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSaxpy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      !
      cublasSaxpy_rank_0 = cublasSaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasSaxpy_rank_1(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSaxpy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasSaxpy_rank_1 = cublasSaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasDaxpy_rank_0(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDaxpy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      !
      cublasDaxpy_rank_0 = cublasDaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasDaxpy_rank_1(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDaxpy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasDaxpy_rank_1 = cublasDaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasCaxpy_rank_0(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCaxpy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      !
      cublasCaxpy_rank_0 = cublasCaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasCaxpy_rank_1(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCaxpy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasCaxpy_rank_1 = cublasCaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasZaxpy_rank_0(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZaxpy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      !
      cublasZaxpy_rank_0 = cublasZaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasZaxpy_rank_1(handle,n,alpha,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZaxpy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasZaxpy_rank_1 = cublasZaxpy_(handle,n,alpha,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasSaxpyBatched_full_rank(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSaxpyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasSaxpyBatched_full_rank = cublasSaxpyBatched_(handle,n,&
&alpha,c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasSaxpyBatched_rank_0(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSaxpyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasSaxpyBatched_rank_0 = cublasSaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasSaxpyBatched_rank_1(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSaxpyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasSaxpyBatched_rank_1 = cublasSaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasDaxpyBatched_full_rank(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDaxpyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasDaxpyBatched_full_rank = cublasDaxpyBatched_(handle,n,&
&alpha,c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasDaxpyBatched_rank_0(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDaxpyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasDaxpyBatched_rank_0 = cublasDaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasDaxpyBatched_rank_1(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDaxpyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasDaxpyBatched_rank_1 = cublasDaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasCaxpyBatched_full_rank(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCaxpyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasCaxpyBatched_full_rank = cublasCaxpyBatched_(handle,n,&
&alpha,c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasCaxpyBatched_rank_0(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCaxpyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasCaxpyBatched_rank_0 = cublasCaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasCaxpyBatched_rank_1(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCaxpyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasCaxpyBatched_rank_1 = cublasCaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasZaxpyBatched_full_rank(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZaxpyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasZaxpyBatched_full_rank = cublasZaxpyBatched_(handle,n,&
&alpha,c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasZaxpyBatched_rank_0(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZaxpyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasZaxpyBatched_rank_0 = cublasZaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasZaxpyBatched_rank_1(handle,n,alpha,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZaxpyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasZaxpyBatched_rank_1 = cublasZaxpyBatched_(handle,n,alpha,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasSaxpyStridedBatched_rank_0(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSaxpyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_float),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasSaxpyStridedBatched_rank_0 = &
&cublasSaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasSaxpyStridedBatched_rank_1(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSaxpyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float) :: alpha
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasSaxpyStridedBatched_rank_1 = &
&cublasSaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasDaxpyStridedBatched_rank_0(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDaxpyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_double),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasDaxpyStridedBatched_rank_0 = &
&cublasDaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasDaxpyStridedBatched_rank_1(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDaxpyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double) :: alpha
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasDaxpyStridedBatched_rank_1 = &
&cublasDaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasCaxpyStridedBatched_rank_0(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCaxpyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasCaxpyStridedBatched_rank_0 = &
&cublasCaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasCaxpyStridedBatched_rank_1(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCaxpyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex) :: alpha
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasCaxpyStridedBatched_rank_1 = &
&cublasCaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasZaxpyStridedBatched_rank_0(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZaxpyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasZaxpyStridedBatched_rank_0 = &
&cublasZaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasZaxpyStridedBatched_rank_1(handle,n,alpha,x,incx,&
&stridex,y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZaxpyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex) :: alpha
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasZaxpyStridedBatched_rank_1 = &
&cublasZaxpyStridedBatched_(handle,n,alpha,c_loc(x),incx,stridex,&
&c_loc(y),incy,stridey,batchCount)
    end function

    function cublasScopy_rank_0(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScopy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      !
      cublasScopy_rank_0 = cublasScopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasScopy_rank_1(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScopy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasScopy_rank_1 = cublasScopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasDcopy_rank_0(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDcopy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      !
      cublasDcopy_rank_0 = cublasDcopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasDcopy_rank_1(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDcopy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasDcopy_rank_1 = cublasDcopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasCcopy_rank_0(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCcopy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      !
      cublasCcopy_rank_0 = cublasCcopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasCcopy_rank_1(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCcopy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasCcopy_rank_1 = cublasCcopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasZcopy_rank_0(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZcopy_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      !
      cublasZcopy_rank_0 = cublasZcopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasZcopy_rank_1(handle,n,x,incx,y,incy)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZcopy_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      !
      cublasZcopy_rank_1 = cublasZcopy_(handle,n,c_loc(x),incx,&
&c_loc(y),incy)
    end function

    function cublasScopyBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScopyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasScopyBatched_full_rank = cublasScopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasScopyBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScopyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasScopyBatched_rank_0 = cublasScopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasScopyBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScopyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasScopyBatched_rank_1 = cublasScopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasDcopyBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDcopyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasDcopyBatched_full_rank = cublasDcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasDcopyBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDcopyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasDcopyBatched_rank_0 = cublasDcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasDcopyBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDcopyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasDcopyBatched_rank_1 = cublasDcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasCcopyBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCcopyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasCcopyBatched_full_rank = cublasCcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasCcopyBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCcopyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasCcopyBatched_rank_0 = cublasCcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasCcopyBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCcopyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasCcopyBatched_rank_1 = cublasCcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasZcopyBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZcopyBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasZcopyBatched_full_rank = cublasZcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasZcopyBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZcopyBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasZcopyBatched_rank_0 = cublasZcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasZcopyBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZcopyBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      !
      cublasZcopyBatched_rank_1 = cublasZcopyBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount)
    end function

    function cublasScopyStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScopyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_float),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasScopyStridedBatched_rank_0 = &
&cublasScopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasScopyStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScopyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasScopyStridedBatched_rank_1 = &
&cublasScopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasDcopyStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDcopyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_double),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasDcopyStridedBatched_rank_0 = &
&cublasDcopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasDcopyStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDcopyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasDcopyStridedBatched_rank_1 = &
&cublasDcopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasCcopyStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCcopyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasCcopyStridedBatched_rank_0 = &
&cublasCcopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasCcopyStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCcopyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasCcopyStridedBatched_rank_1 = &
&cublasCcopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasZcopyStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZcopyStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasZcopyStridedBatched_rank_0 = &
&cublasZcopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasZcopyStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZcopyStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      !
      cublasZcopyStridedBatched_rank_1 = &
&cublasZcopyStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount)
    end function

    function cublasSdot_rank_0(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasSdot_rank_0 = cublasSdot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,myResult)
    end function

    function cublasSdot_rank_1(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasSdot_rank_1 = cublasSdot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,myResult)
    end function

    function cublasDdot_rank_0(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasDdot_rank_0 = cublasDdot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,myResult)
    end function

    function cublasDdot_rank_1(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasDdot_rank_1 = cublasDdot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,myResult)
    end function

    function cublasCdotc_rank_0(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotc_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasCdotc_rank_0 = cublasCdotc_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasCdotc_rank_1(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotc_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasCdotc_rank_1 = cublasCdotc_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasCdotu_rank_0(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotu_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasCdotu_rank_0 = cublasCdotu_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasCdotu_rank_1(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotu_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasCdotu_rank_1 = cublasCdotu_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasZdotc_rank_0(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotc_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasZdotc_rank_0 = cublasZdotc_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasZdotc_rank_1(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotc_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasZdotc_rank_1 = cublasZdotc_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasZdotu_rank_0(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotu_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasZdotu_rank_0 = cublasZdotu_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasZdotu_rank_1(handle,n,x,incx,y,incy,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotu_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: myResult
      !
      cublasZdotu_rank_1 = cublasZdotu_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,myResult)
    end function

    function cublasSdotBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSdotBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSdotBatched_full_rank = cublasSdotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasSdotBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdotBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSdotBatched_rank_0 = cublasSdotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasSdotBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSdotBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSdotBatched_rank_1 = cublasSdotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasDdotBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDdotBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDdotBatched_full_rank = cublasDdotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasDdotBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdotBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDdotBatched_rank_0 = cublasDdotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasDdotBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDdotBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDdotBatched_rank_1 = cublasDdotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasCdotcBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCdotcBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotcBatched_full_rank = cublasCdotcBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasCdotcBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotcBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotcBatched_rank_0 = cublasCdotcBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasCdotcBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotcBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotcBatched_rank_1 = cublasCdotcBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasCdotuBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCdotuBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotuBatched_full_rank = cublasCdotuBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasCdotuBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotuBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotuBatched_rank_0 = cublasCdotuBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasCdotuBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCdotuBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotuBatched_rank_1 = cublasCdotuBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasZdotcBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZdotcBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotcBatched_full_rank = cublasZdotcBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasZdotcBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotcBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotcBatched_rank_0 = cublasZdotcBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasZdotcBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotcBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotcBatched_rank_1 = cublasZdotcBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasZdotuBatched_full_rank(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZdotuBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotuBatched_full_rank = cublasZdotuBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasZdotuBatched_rank_0(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotuBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotuBatched_rank_0 = cublasZdotuBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasZdotuBatched_rank_1(handle,n,x,incx,y,incy,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdotuBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotuBatched_rank_1 = cublasZdotuBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,batchCount,myResult)
    end function

    function cublasSdotStridedBatched_rank_0(handle,n,x,incx,stridex,y,&
&incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSdotStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_float),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSdotStridedBatched_rank_0 = &
&cublasSdotStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasSdotStridedBatched_rank_1(handle,n,x,incx,stridex,y,&
&incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSdotStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSdotStridedBatched_rank_1 = &
&cublasSdotStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasDdotStridedBatched_rank_0(handle,n,x,incx,stridex,y,&
&incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDdotStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_double),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDdotStridedBatched_rank_0 = &
&cublasDdotStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasDdotStridedBatched_rank_1(handle,n,x,incx,stridex,y,&
&incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDdotStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDdotStridedBatched_rank_1 = &
&cublasDdotStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasCdotcStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCdotcStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotcStridedBatched_rank_0 = &
&cublasCdotcStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasCdotcStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCdotcStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotcStridedBatched_rank_1 = &
&cublasCdotcStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasCdotuStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCdotuStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotuStridedBatched_rank_0 = &
&cublasCdotuStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasCdotuStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCdotuStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasCdotuStridedBatched_rank_1 = &
&cublasCdotuStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasZdotcStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZdotcStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotcStridedBatched_rank_0 = &
&cublasZdotcStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasZdotcStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZdotcStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotcStridedBatched_rank_1 = &
&cublasZdotcStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasZdotuStridedBatched_rank_0(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZdotuStridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotuStridedBatched_rank_0 = &
&cublasZdotuStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasZdotuStridedBatched_rank_1(handle,n,x,incx,stridex,&
&y,incy,stridey,batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZdotuStridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      integer(c_int64_t) :: stridey
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasZdotuStridedBatched_rank_1 = &
&cublasZdotuStridedBatched_(handle,n,c_loc(x),incx,stridex,c_loc(y),&
&incy,stridey,batchCount,myResult)
    end function

    function cublasSnrm2_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSnrm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasSnrm2_rank_0 = cublasSnrm2_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasSnrm2_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSnrm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasSnrm2_rank_1 = cublasSnrm2_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasDnrm2_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDnrm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDnrm2_rank_0 = cublasDnrm2_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasDnrm2_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDnrm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDnrm2_rank_1 = cublasDnrm2_(handle,n,c_loc(x),incx,myResult)
    end function

    function cublasScnrm2_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScnrm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasScnrm2_rank_0 = cublasScnrm2_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasScnrm2_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScnrm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasScnrm2_rank_1 = cublasScnrm2_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasDznrm2_rank_0(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDznrm2_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDznrm2_rank_0 = cublasDznrm2_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasDznrm2_rank_1(handle,n,x,incx,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDznrm2_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      type(c_ptr) :: myResult
      !
      cublasDznrm2_rank_1 = cublasDznrm2_(handle,n,c_loc(x),incx,&
&myResult)
    end function

    function cublasSnrm2Batched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSnrm2Batched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSnrm2Batched_full_rank = cublasSnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasSnrm2Batched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSnrm2Batched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSnrm2Batched_rank_0 = cublasSnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasSnrm2Batched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSnrm2Batched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSnrm2Batched_rank_1 = cublasSnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDnrm2Batched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDnrm2Batched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDnrm2Batched_full_rank = cublasDnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDnrm2Batched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDnrm2Batched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDnrm2Batched_rank_0 = cublasDnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDnrm2Batched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDnrm2Batched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDnrm2Batched_rank_1 = cublasDnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasScnrm2Batched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScnrm2Batched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScnrm2Batched_full_rank = cublasScnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasScnrm2Batched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScnrm2Batched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScnrm2Batched_rank_0 = cublasScnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasScnrm2Batched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasScnrm2Batched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScnrm2Batched_rank_1 = cublasScnrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDznrm2Batched_full_rank(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDznrm2Batched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDznrm2Batched_full_rank = cublasDznrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDznrm2Batched_rank_0(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDznrm2Batched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDznrm2Batched_rank_0 = cublasDznrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasDznrm2Batched_rank_1(handle,n,x,incx,batchCount,&
&myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDznrm2Batched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDznrm2Batched_rank_1 = cublasDznrm2Batched_(handle,n,&
&c_loc(x),incx,batchCount,myResult)
    end function

    function cublasSnrm2StridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSnrm2StridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSnrm2StridedBatched_rank_0 = &
&cublasSnrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasSnrm2StridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSnrm2StridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasSnrm2StridedBatched_rank_1 = &
&cublasSnrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDnrm2StridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDnrm2StridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDnrm2StridedBatched_rank_0 = &
&cublasDnrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDnrm2StridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDnrm2StridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDnrm2StridedBatched_rank_1 = &
&cublasDnrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasScnrm2StridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScnrm2StridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScnrm2StridedBatched_rank_0 = &
&cublasScnrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasScnrm2StridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasScnrm2StridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasScnrm2StridedBatched_rank_1 = &
&cublasScnrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDznrm2StridedBatched_rank_0(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDznrm2StridedBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDznrm2StridedBatched_rank_0 = &
&cublasDznrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasDznrm2StridedBatched_rank_1(handle,n,x,incx,stridex,&
&batchCount,myResult)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDznrm2StridedBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      integer(c_int64_t) :: stridex
      integer(c_int) :: batchCount
      type(c_ptr) :: myResult
      !
      cublasDznrm2StridedBatched_rank_1 = &
&cublasDznrm2StridedBatched_(handle,n,c_loc(x),incx,stridex,batchCount,&
&myResult)
    end function

    function cublasSrot_rank_0(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasSrot_rank_0 = cublasSrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasSrot_rank_1(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasSrot_rank_1 = cublasSrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasDrot_rank_0(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasDrot_rank_0 = cublasDrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasDrot_rank_1(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasDrot_rank_1 = cublasDrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasCrot_rank_0(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasCrot_rank_0 = cublasCrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasCrot_rank_1(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasCrot_rank_1 = cublasCrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasCsrot_rank_0(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsrot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasCsrot_rank_0 = cublasCsrot_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,c,s)
    end function

    function cublasCsrot_rank_1(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsrot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasCsrot_rank_1 = cublasCsrot_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,c,s)
    end function

    function cublasZrot_rank_0(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasZrot_rank_0 = cublasZrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasZrot_rank_1(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasZrot_rank_1 = cublasZrot_(handle,n,c_loc(x),incx,c_loc(y),&
&incy,c,s)
    end function

    function cublasZdrot_rank_0(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdrot_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasZdrot_rank_0 = cublasZdrot_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,c,s)
    end function

    function cublasZdrot_rank_1(handle,n,x,incx,y,incy,c,s)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZdrot_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      !
      cublasZdrot_rank_1 = cublasZdrot_(handle,n,c_loc(x),incx,&
&c_loc(y),incy,c,s)
    end function

    function cublasSrotBatched_full_rank(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasSrotBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:,:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasSrotBatched_full_rank = cublasSrotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasSrotBatched_rank_0(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target :: x
      integer(c_int) :: incx
      real(c_float),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasSrotBatched_rank_0 = cublasSrotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasSrotBatched_rank_1(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasSrotBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_float),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_float),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasSrotBatched_rank_1 = cublasSrotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasDrotBatched_full_rank(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasDrotBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:,:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:,:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasDrotBatched_full_rank = cublasDrotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasDrotBatched_rank_0(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target :: x
      integer(c_int) :: incx
      real(c_double),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasDrotBatched_rank_0 = cublasDrotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasDrotBatched_rank_1(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasDrotBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      real(c_double),target,dimension(:) :: x
      integer(c_int) :: incx
      real(c_double),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasDrotBatched_rank_1 = cublasDrotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasCrotBatched_full_rank(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCrotBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasCrotBatched_full_rank = cublasCrotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasCrotBatched_rank_0(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrotBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasCrotBatched_rank_0 = cublasCrotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasCrotBatched_rank_1(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCrotBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasCrotBatched_rank_1 = cublasCrotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasCsrotBatched_full_rank(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasCsrotBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasCsrotBatched_full_rank = cublasCsrotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasCsrotBatched_rank_0(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsrotBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target :: x
      integer(c_int) :: incx
      complex(c_float_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasCsrotBatched_rank_0 = cublasCsrotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasCsrotBatched_rank_1(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasCsrotBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_float_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_float_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasCsrotBatched_rank_1 = cublasCsrotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasZrotBatched_full_rank(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: &
&cublasZrotBatched_full_rank
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:,:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:,:) :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasZrotBatched_full_rank = cublasZrotBatched_(handle,n,&
&c_loc(x),incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasZrotBatched_rank_0(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrotBatched_rank_0
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target :: x
      integer(c_int) :: incx
      complex(c_double_complex),target :: y
      integer(c_int) :: incy
      type(c_ptr) :: c
      type(c_ptr) :: s
      integer(c_int) :: batchCount
      !
      cublasZrotBatched_rank_0 = cublasZrotBatched_(handle,n,c_loc(x),&
&incx,c_loc(y),incy,c,s,batchCount)
    end function

    function cublasZrotBatched_rank_1(handle,n,x,incx,y,incy,c,s,&
&batchCount)
      use iso_c_binding
      use cudafor_cublas_enums
      implicit none
      integer(kind(CUBLAS_STATUS_SUCCESS)) :: cublasZrotBatched_rank_1
      type(c_ptr) :: handle
      integer(c_int) :: n
      complex(c_double_complex),target,dimension(:) :: x
      integer(c_int) :: incx
      complex(c_double_complex),target,dimension(:) :: y
      integer(c_int) :: incy
      type(c_p