// Copyright (c) 2017 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#ifndef HIPRAND_KERNEL_HCC_H_
#define HIPRAND_KERNEL_HCC_H_

/// \addtogroup curanddevice
/// @{

#ifndef QUALIFIERS
#define QUALIFIERS __forceinline__ __device__
#endif // QUALIFIERS

#include <cuda_runtime.h>
#include "curand.h"
#include <type_traits>

#include "curand_common.h"
#include "curand_philox4x32_x.h"
#include "curand_mrg32k3a.h"
#include "curand_xorwow.h"
#include "curand_sobol32.h"
#include "curand_sobol64.h"
#include "curand_mtgp32.h"

#include "curand_uniform.h"
#include "curand_normal.h"
#include "curand_lognormal.h"
#include "curand_poisson.h"
#include "curand_discrete.h"

/// \cond
#ifndef FQUALIFIERS
#define FQUALIFIERS QUALIFIERS
#endif
/// \endcond

/// \cond

#define DEFINE_HIPRAND_STATE(curand_name, rand_name) \
    struct curand_name : public rand_name \
    { \
        typedef rand_name base; \
    }; \
    typedef struct curand_name curand_name ## _t;

DEFINE_HIPRAND_STATE(curandState, curand_state_xorwow)
DEFINE_HIPRAND_STATE(curandStateXORWOW, curand_state_xorwow)
DEFINE_HIPRAND_STATE(curandStatePhilox4_32_10, curand_state_philox4x32_10)
DEFINE_HIPRAND_STATE(curandStateMRG32k3a, curand_state_mrg32k3a)
DEFINE_HIPRAND_STATE(curandStateMtgp32, curand_state_mtgp32)
DEFINE_HIPRAND_STATE(curandStateSobol32, curand_state_sobol32)
DEFINE_HIPRAND_STATE(curandStateSobol64, curand_state_sobol64)
#undef DEFINE_HIPRAND_STATE

#define CURAND_PHILOX4x32_DEFAULT_SEED 0ULL
#define CURAND_XORWOW_DEFAULT_SEED 0ULL
#define CURAND_MRG32K3A_DEFAULT_SEED 12345ULL

typedef unsigned int curandDirectionVectors32_t[32];
//typedef unsigned int curandDirectionVectors64_t[64];

typedef mtgp32_params mtgp32_kernel_params_t;
typedef mtgp32_fast_params mtgp32_fast_param_t;
/// \endcond

/// \cond
namespace detail {

template<typename T, typename... R>
struct is_any_of : std::false_type { };

template<typename T, typename F>
struct is_any_of<T, F> : std::is_same<T, F> { };

template<typename T, typename F, typename... R>
struct is_any_of<T, F, R...>
    : std::integral_constant<
        bool,
        std::is_same<T, F>::value || is_any_of<T, R...>::value
      >
{ };

} // end namespace detail
/// \endcond

/// \cond

template<typename StateType>
QUALIFIERS
void check_state_type()
{
    static_assert(
        detail::is_any_of<
            StateType,
            curandState_t,
            curandStateXORWOW_t,
            curandStatePhilox4_32_10_t,
            curandStateMRG32k3a_t,
            curandStateMtgp32_t,
            curandStateSobol32_t, 
            curandStateSobol64_t 
        >::value,
        "StateType is not a cuRAND generator state"
    );
}
/// \endcond

/**
 * \brief Copy MTGP32 state to another state using block of threads
 *
 * Copies a MTGP32 state \p src to \p dest using a block of threads
 * efficiently. Example usage would be:
 *
 * \code
 * __global__
 * void generate_kernel(curandStateMtgp32_t * states, unsigned int * output, const size_t size)
 * {
 *      const unsigned int state_id = cuBlockIdx_x;
 *      unsigned int index = cuBlockIdx_x * cuBlockDim_x + cuThreadIdx_x;
 *      unsigned int stride = cuGridDim_x * cuBlockDim_x;
 *
 *      __shared__ GeneratorState state;
 *      curand_mtgp32_block_copy(&states[state_id], &state);
 *
 *      while(index < size)
 *      {
 *          output[index] = rand(&state);
 *          index += stride;
 *      }
 *
 *      curand_mtgp32_block_copy(&state, &states[state_id]);
 * }
 * \endcode
 *
 * \param src - Pointer to a state to copy from
 * \param dest - Pointer to a state to copy to
 */

QUALIFIERS
void curand_mtgp32_block_copy(curandStateMtgp32_t * src,
                               curandStateMtgp32_t * dest)
{
    rand_mtgp32_block_copy(src, dest);
}


/**
 * \brief Changes parameters of a MTGP32 state.
 *
 * \param state - Pointer to a MTGP32 state
 * \param params - Pointer to new parameters
 */
/*
QUALIFIERS
void curand_mtgp32_set_params(curandStateMtgp32_t * state,
                               mtgp32_kernel_params_t * params)
{
    rand_mtgp32_set_params(state, params);
}
*/

/// \brief Initializes a PRNG state.
///
/// \tparam StateType - Pseudorandom number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, or \p curandStateMRG32k3a_t
///
/// \param seed - Pseudorandom number generator's seed
/// \param subsequence - Number of subsequence to skipahead
/// \param offset - Absolute subsequence offset, i.e. how many states from
/// current subsequence should be skipped
/// \param state - Pointer to a state to initialize
///
/// See also: curandMakeMTGP32KernelState()
template<class StateType>
QUALIFIERS
void curand_init(const unsigned long long seed,
                  const unsigned long long subsequence,
                  const unsigned long long offset,
                  StateType * state)
{
    check_state_type<StateType>();
    static_assert(
        !std::is_same<
            StateType,
            curandStateMtgp32_t
        >::value,
        "curandStateMtgp32_t does not have curand_init function, "
        "check curandMakeMTGP32KernelState() host function"
    );
    static_assert(
        !detail::is_any_of<
            StateType,
            curandStateSobol32_t
        >::value,
        "Quasirandom generators use different curand_init() function"
    );
    rand_init(seed, subsequence, offset, state);
}

/// \brief Initializes a Sobol32 state.
///
/// \param direction_vectors - Pointer to array of 32 <tt>unsigned int</tt>s that
/// represent the direction numbers
/// \param offset - Absolute subsequence offset, i.e. how many states should be skipped
/// \param state - Pointer to a state to initialize
QUALIFIERS
void curand_init(curandDirectionVectors32_t direction_vectors,
                  unsigned int offset,
                  curandStateSobol32_t * state)
{
    rand_init(direction_vectors, offset, state);
}

QUALIFIERS
void curand_init(curandDirectionVectors64_t direction_vectors,
                  unsigned int offset,
                  curandStateSobol64_t * state)
{
    rand_init(direction_vectors, offset, state);
}

/// \brief Updates RNG state skipping \p n states ahead.
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// or \p curandStateSobol32_t
///
/// \param n - Number of states to skipahead
/// \param state - Pointer to a state to modify
template<class StateType>
QUALIFIERS
void cu_skipahead(unsigned long long n, StateType * state)
{
    check_state_type<StateType>();
    static_assert(
        !std::is_same<
            StateType,
            curandStateMtgp32_t
        >::value,
        "curandStateMtgp32_t does not have skipahead function"
    );
    typedef typename StateType::base base_type;
    skipahead(n, static_cast<base_type*>(state));
}

/// \brief Updates PRNG state skipping \p n sequences ahead.
///
///      PRNG     | Sequence size [Number of elements]
/// ------------- | -------------
/// XORWOW        | 2^67
/// Philox        | 4 * 2^64
/// MRG32k3a      | 2^67
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t,
/// or \p curandStateMRG32k3a_t
///
/// \param n - Number of subsequences to skipahead
/// \param state - Pointer to a state to update
template<class StateType>
QUALIFIERS
void skipahead_sequence(unsigned long long n, StateType * state)
{
    check_state_type<StateType>();
    static_assert(
        !detail::is_any_of<
            StateType,
            curandStateMtgp32_t,
            curandStateSobol32_t
        >::value,
        "StateType does not have skipahead_sequence function"
    );
    typedef typename StateType::base base_type;
    skipahead_subsequence(n, static_cast<base_type*>(state));
}

/// \brief Updates PRNG state skipping \p n subsequences ahead.
///
///      PRNG     | Subsequence size [Number of elements]
/// ------------- | -------------
/// XORWOW        | 2^67
/// Philox        | 4 * 2^64
/// MRG32k3a      | 2^127
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t,
/// or \p curandStateMRG32k3a_t
///
/// \param n - Number of subsequences to skipahead
/// \param state - Pointer to a state to update
template<class StateType>
QUALIFIERS
void skipahead_subsequence(unsigned long long n, StateType * state)
{
    check_state_type<StateType>();
    static_assert(
        !detail::is_any_of<
            StateType,
            curandStateMtgp32_t,
            curandStateSobol32_t
        >::value,
        "StateType does not have skipahead_subsequence function"
    );
    typedef typename StateType::base base_type;
    skipahead_subsequence(n, static_cast<base_type*>(state));
}

/// \brief Generates uniformly distributed random <tt>unsigned int</tt>
/// from [0; 2^32 - 1] range.
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \return Uniformly distributed random 32-bit <tt>unsigned int</tt>
template<class StateType>
QUALIFIERS
unsigned int curand(StateType * state)
{
    check_state_type<StateType>();
    return rand(state);
}

/// \brief Generates four uniformly distributed random <tt>unsigned int</tt>s
/// from [0; 2^32 - 1] range.
///
/// \param state - Pointer to a Philox state to use
/// \return Four uniformly distributed random 32-bit <tt>unsigned int</tt>s as \p uint4
QUALIFIERS
uint4 curand4(curandStatePhilox4_32_10_t * state)
{
    return rand4(state);
}

/// \brief Generates uniformly distributed random <tt>float</tt> value
/// from (0; 1] range.
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \return Uniformly distributed random <tt>float</tt> value
template<class StateType>
QUALIFIERS
float curand_uniform(StateType * state)
{
    check_state_type<StateType>();
    return rand_uniform(state);
}

/// \brief Generates four uniformly distributed random <tt>float</tt> value
/// from (0; 1] range.
///
/// \param state - Pointer to a Philox state to use
/// \return Four uniformly distributed random <tt>float</tt> values as \p float4
QUALIFIERS
float4 curand_uniform4(curandStatePhilox4_32_10_t * state)
{
    return rand_uniform4(state);
}

/// \brief Generates uniformly distributed random <tt>double</tt> value from (0; 1] range
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \return Uniformly distributed random <tt>double</tt> value
///
/// Note: When \p state is of type: \p curandStateMRG32k3a_t, \p curandStateMtgp32_t,
/// or \p curandStateSobol32_t, then the returned \p double value is generated
/// using only 32 random bits (one <tt>unsigned int</tt> value).
/// In case of type \p curandStateSobol32_t, this is done to guarantee the quasirandom
/// properties of the Sobol32 sequence.
template<class StateType>
QUALIFIERS
double curand_uniform_double(StateType * state)
{
    check_state_type<StateType>();
    return rand_uniform_double(state);
}

/// \brief Generates two uniformly distributed random <tt>double</tt> values
/// from (0; 1] range.
///
/// \param state - Pointer to a Philox state to use
/// \return Two uniformly distributed random <tt>double</tt> values as \p double2
QUALIFIERS
double2 curand_uniform2_double(curandStatePhilox4_32_10_t * state)
{
    return rand_uniform_double2(state);
}

/// \brief Generates four uniformly distributed random <tt>double</tt> values
/// from (0; 1] range.
///
/// \param state - Pointer to a Philox state to use
/// \return Four uniformly distributed random <tt>double</tt> values as \p double4
QUALIFIERS
double4 curand_uniform4_double(curandStatePhilox4_32_10_t * state)
{
    return rand_uniform_double4(state);
}

/// \brief Generates normally distributed random <tt>float</tt> value
///
/// Mean value of normal distribution is equal to 0.0, and standard deviation
/// equals 1.0.
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \return Normally distributed random <tt>float</tt> value
template<class StateType>
QUALIFIERS
float curand_normal(StateType * state)
{
    check_state_type<StateType>();
    return rand_normal(state);
}

/// \brief Generates two normally distributed random <tt>float</tt> values
///
/// Mean value of normal distribution is equal to 0.0, and standard deviation
/// equals 1.0.
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t,
/// or \p curandStateMRG32k3a_t
///
/// \param state - Pointer to a RNG state to use
/// \return Two normally distributed random <tt>float</tt> values as \p float2
template<class StateType>
QUALIFIERS
float2 curand_normal2(StateType * state)
{
    check_state_type<StateType>();
    static_assert(
        detail::is_any_of<
            StateType,
            curandState_t,
            curandStateXORWOW_t,
            curandStatePhilox4_32_10_t,
            curandStateMRG32k3a_t
        >::value,
        "Used StateType is not supported"
    );
    return rand_normal2(state);
}

/// \brief Generates four normally distributed random <tt>float</tt> values
///
/// Mean value of normal distribution is equal to 0.0, and standard deviation
/// equals 1.0.
///
/// \param state - Pointer to a Philox state to use
/// \return Four normally distributed random <tt>float</tt> values as \p float4
QUALIFIERS
float4 curand_normal4(curandStatePhilox4_32_10_t * state)
{
    return rand_normal4(state);
}

/// \brief Generates normally distributed random <tt>double</tt> value
///
/// Mean value of normal distribution is equal to 0.0, and standard deviation
/// equals 1.0.
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \return Normally distributed random <tt>double</tt> value
template<class StateType>
QUALIFIERS
double curand_normal_double(StateType * state)
{
    check_state_type<StateType>();
    return rand_normal_double(state);
}

/// \brief Generates two normally distributed random <tt>double</tt> values
///
/// Mean value of normal distribution is equal to 0.0, and standard deviation
/// equals 1.0.
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t,
/// or \p curandStateMRG32k3a_t
///
/// \param state - Pointer to a RNG state to use
/// \return Two normally distributed random <tt>double</tt> values as \p double2
template<class StateType>
QUALIFIERS
double2 curand_normal2_double(StateType * state)
{
    check_state_type<StateType>();
    static_assert(
        detail::is_any_of<
            StateType,
            curandState_t,
            curandStateXORWOW_t,
            curandStatePhilox4_32_10_t,
            curandStateMRG32k3a_t
        >::value,
        "Used StateType is not supported"
    );
    return rand_normal_double2(state);
}

/// \brief Generates four normally distributed random <tt>double</tt> values
///
/// Mean value of normal distribution is equal to 0.0, and standard deviation
/// equals 1.0.
///
/// \param state - Pointer to a Philox state to use
/// \return Four normally distributed random <tt>double</tt> values as \p double4
QUALIFIERS
double4 curand_normal4_double(curandStatePhilox4_32_10_t * state)
{
    return rand_normal_double4(state);
}

/// \brief Generates log-normally distributed random <tt>float</tt> value
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \param mean - Mean value of log-normal distribution
/// \param stddev - Standard deviation value of log-normal distribution
/// \return Log-normally distributed random <tt>float</tt> value

template<class StateType>
QUALIFIERS
float curand_log_normal(StateType * state,
                         float mean, float stddev)
{
    check_state_type<StateType>();
    return rand_log_normal(state, mean, stddev);
}

/// \brief Generates two log-normally distributed random <tt>float</tt> values
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t,
/// or \p curandStateMRG32k3a_t
///
/// \param state - Pointer to a RNG state to use
/// \param mean - Mean value of log-normal distribution
/// \param stddev - Standard deviation value of log-normal distribution
/// \return Two log-normally distributed random <tt>float</tt> values as \p float2
template<class StateType>
QUALIFIERS
float2 curand_log_normal2(StateType * state,
                           float mean, float stddev)
{
    check_state_type<StateType>();
    static_assert(
        detail::is_any_of<
            StateType,
            curandState_t,
            curandStateXORWOW_t,
            curandStatePhilox4_32_10_t,
            curandStateMRG32k3a_t
        >::value,
        "Used StateType is not supported"
    );
    return rand_log_normal2(state, mean, stddev);
}

/// \brief Generates four log-normally distributed random <tt>float</tt> values
///
/// \param state - Pointer to a Philox state to use
/// \param mean - Mean value of log-normal distribution
/// \param stddev - Standard deviation value of log-normal distribution
/// \return Four log-normally distributed random <tt>float</tt> values as \p float4
QUALIFIERS
float4 curand_log_normal4(curandStatePhilox4_32_10_t * state,
                           float mean, float stddev)
{
    return rand_log_normal4(state, mean, stddev);
}

/// \brief Generates log-normally distributed random <tt>double</tt> value
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \param mean - Mean value of log-normal distribution
/// \param stddev - Standard deviation value of log-normal distribution
/// \return Log-normally distributed random <tt>double</tt> value
template<class StateType>
QUALIFIERS
double curand_log_normal_double(StateType * state,
                                 double mean, double stddev)
{
    check_state_type<StateType>();
    return rand_log_normal_double(state, mean, stddev);
}

/// \brief Generates two log-normally distributed random <tt>double</tt> values
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \param mean - Mean value of log-normal distribution
/// \param stddev - Standard deviation value of log-normal distribution
/// \return Two log-normally distributed random <tt>double</tt> values as \p double2
template<class StateType>
QUALIFIERS
double2 curand_log_normal2_double(StateType * state,
                                   double mean, double stddev)
{
    check_state_type<StateType>();
    static_assert(
        detail::is_any_of<
            StateType,
            curandState_t,
            curandStateXORWOW_t,
            curandStatePhilox4_32_10_t,
            curandStateMRG32k3a_t
        >::value,
        "Used StateType is not supported"
    );
    return rand_log_normal_double2(state, mean, stddev);
}

/// \brief Generates four log-normally distributed random <tt>double</tt> values
///
/// \param state - Pointer to a Philox state to use
/// \param mean - Mean value of log-normal distribution
/// \param stddev - Standard deviation value of log-normal distribution
/// \return Four log-normally distributed random <tt>double</tt> values as \p double4
QUALIFIERS
double4 curand_log_normal4_double(curandStatePhilox4_32_10_t * state,
                                   double mean, double stddev)
{
    return rand_log_normal_double4(state, mean, stddev);
}

/// \brief Generates Poisson-distributed random <tt>unsigned int</tt> value
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \param lambda - Lambda (mean) parameter of Poisson distribution
/// \return Poisson-distributed random <tt>unsigned int</tt> value
template<class StateType>
QUALIFIERS
uint curand_poisson(StateType * state, double lambda)
{
    check_state_type<StateType>();
    return rand_poisson(state, lambda);
}

/// \brief Generates four Poisson-distributed random <tt>unsigned int</tt> values
///
/// \param state - Pointer to a Philox state to use
/// \param lambda - Lambda (mean) parameter of Poisson distribution
/// \return Four Poisson-distributed random <tt>unsigned int</tt> values as \p uint4
QUALIFIERS
uint4 curand_poisson4(curandStatePhilox4_32_10_t * state, double lambda)
{
    return rand_poisson4(state, lambda);
}

/// \brief Generates random <tt>unsigned int</tt> value according to
/// given discrete distribution
///
/// \tparam StateType - Random number generator state type.
/// \p StateType type must be one of following types:
/// \p curandStateXORWOW_t, \p curandStatePhilox4_32_10_t, \p curandStateMRG32k3a_t,
/// \p curandStateMtgp32_t, or \p curandStateSobol32_t
///
/// \param state - Pointer to a RNG state to use
/// \param discrete_distribution - Discrete distribution
/// \return Random <tt>unsigned int</tt> value
///
/// See also: curandCreatePoissonDistribution()


template<class StateType>
QUALIFIERS
uint curand_discrete(StateType * state, curandDiscreteDistribution_t discrete_distribution)
{
    check_state_type<StateType>();
    return rand_discrete(state, discrete_distribution);
}

/// \brief Generates four random <tt>unsigned int</tt> values according to
/// given discrete distribution
///
/// \param state - Pointer to a Philox state to use
/// \param discrete_distribution - Discrete distribution
/// \return Four random <tt>unsigned int</tt> values as \p uint4
///
/// See also: curandCreatePoissonDistribution()
/*
QUALIFIERS
uint4 curand_discrete4(curandStatePhilox4_32_10_t * state,
                        curandDiscreteDistribution_t discrete_distribution)
{
    return curand_discrete4(state, discrete_distribution);
}
*/

/// @} // end of group curanddevice

#endif // HIPRAND_KERNEL_HCC_H_
