// Copyright (c) 2017 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#ifndef HIPRAND_MTGP32_HOST_H_
#define HIPRAND_MTGP32_HOST_H_

/// \addtogroup curanddevice
/// @{

#include "curand_kernel.h"

/// \cond
typedef mtgp32_params mtgp32_kernel_params_t;
typedef mtgp32_fast_params mtgp32_fast_param_t;
/// \endcond

/**
 * \brief Loads parameters for MTGP32
 *
 * Loads parameters for use by kernel functions on the host-side and copies the
 * results to the specified location in device memory.
 *
 * \param params - Pointer to an array of type mtgp32_params_fast_t allocated in host memory
 * \param p - Pointer to a mtgp32_kernel_params_t structure allocated in device memory
 *
 * \return
 * - HIPRAND_STATUS_ALLOCATION_FAILED if parameters could not be loaded
 * - HIPRAND_STATUS_SUCCESS if parameters are loaded
 */
inline __host__
curandStatus_t curandMakeMTGP32Constants(const mtgp32_params_fast_t params[],
                                           mtgp32_kernel_params_t * p)
{
    curandStatus status = CURAND_STATUS_SUCCESS;
    status = curand_make_constant(params, p);
    return status;
}

/**
 * \brief Initializes MTGP32 states
 *
 * Initializes MTGP32 states on the host-side by allocating a state array in host
 * memory, initializes that array, and copies the result to device memory.
 *
 * \param s - Pointer to an array of states in device memory
 * \param params - Pointer to an array of type mtgp32_params_fast_t in host memory
 * \param k - Pointer to a mtgp32_kernel_params_t structure allocated in device memory
 * \param n - Number of states to initialize
 * \param seed - Seed value
 *
 * \return
 * - HIPRAND_STATUS_ALLOCATION_FAILED if states could not be initialized
 * - HIPRAND_STATUS_SUCCESS if states are initialized
 */
inline __host__
curandStatus_t curandMakeMTGP32KernelState(curandStateMtgp32_t *s,
                                             mtgp32_params_fast_t params[],
                                             mtgp32_kernel_params_t *k,
                                             int n,
                                             unsigned long long seed)
{
    (void) k;
    curandStatus status = CURAND_STATUS_SUCCESS;
    status = curand_make_state_mtgp32(s, params, n, seed);
    return status;
}


/// @} // end of group curanddevice

#endif // HIPRAND_MTGP32_HOST_H_
