


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ==============================================================================
! hipfort: FORTRAN Interfaces for GPU kernels
! ==============================================================================
! Copyright (c) 2020-2022 Advanced Micro Devices, Inc. All rights reserved.
! [MITx11 License]
! 
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
! IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
! AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
! OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
! THE SOFTWARE.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
          
include'cudafor_enums.f'
include'cudafor_types.f'
include'cudafor_cudahostregister.f'
include'cudafor_auxiliary.f'
include'cudafor_cudamemcpy.f'
include'cudafor_cudamalloc.f'
include'cudafor_cublas_enums.f'
include'cudafor_cufft_enums.f'
include'cudafor_cusparse_enums.f'
include'cudafor_cusolver_enums.f'
include'cudafor_curand_enums.f'
include'cudafor_cufft_params.f'
include'cudafor_cublas_auxiliary.f'
include'cudafor_cufft.f'
include'cudafor_cusparse.f'
include'cudafor_cusolver.f'
include'cudafor_curand.f'
include'cudafor_cublas.f'
include'cudafor_check.f'
module cudafor
  use cudafor_enums
  use cudafor_types
  use cudafor_cudamalloc
  use cudafor_cudamemcpy
  use cudafor_auxiliary
  implicit none

 
  
  interface cuInit
    function cuInit_(flags) bind(c, name="cuInit")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuInit_
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Returns the approximate HIP driver version.
  !> 
  !>  @param [out] driverVersion
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidValue
  !> 
  !>  @warning The HIP feature set does not correspond to an exact CUDA SDK driver revision.
  !>  This function always set *driverVersion to 4 as an approximation though HIP supports
  !>  some features which were introduced in later CUDA SDK revisions.
  !>  HIP apps code should not rely on the driver revision number here and should
  !>  use arch feature flags to test device capabilities or conditional compilation.
  !> 
  !>  @see hipRuntimeGetVersion
  interface cudaDriverGetVersion
    function cudaDriverGetVersion_(driverVersion) bind(c, &
&name="cudaDriverGetVersion")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDriverGetVersion_
      type(c_ptr),value :: driverVersion
    end function

  end interface
  !>  @brief Returns the approximate HIP Runtime version.
  !> 
  !>  @param [out] runtimeVersion
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidValue
  !> 
  !>  @warning The version definition of HIP runtime is different from CUDA.
  !>  On AMD platform, the function returns HIP runtime version,
  !>  while on NVIDIA platform, it returns CUDA runtime version.
  !>  And there is no mapping/correlation between HIP version and CUDA version.
  !> 
  !>  @see hipDriverGetVersion
  interface cudaRuntimeGetVersion
    function cudaRuntimeGetVersion_(runtimeVersion) bind(c, &
&name="cudaRuntimeGetVersion")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaRuntimeGetVersion_
      type(c_ptr),value :: runtimeVersion
    end function

  end interface
  !>  @brief Returns a handle to a compute device
  !>  @param [out] device
  !>  @param [in] ordinal
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidDevice
  interface cuDeviceGet
    function cuDeviceGet_(device,ordinal) bind(c, &
&name="cuDeviceGet")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDeviceGet_
      integer(c_int) :: device
      integer(c_int),value :: ordinal
    end function

  end interface
  !>  @brief Returns the compute capability of the device
  !>  @param [out] major
  !>  @param [out] minor
  !>  @param [in] device
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidDevice
  interface cuDeviceComputeCapability
    function cuDeviceComputeCapability_(major,minor,device) bind(c, &
&name="cuDeviceComputeCapability")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDeviceComputeCapability_
      type(c_ptr),value :: major
      type(c_ptr),value :: minor
      integer(c_int),value :: device
    end function

  end interface
  !>  @brief Returns an identifer string for the device.
  !>  @param [out] name
  !>  @param [in] len
  !>  @param [in] device
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidDevice
  interface cuDeviceGetName
    function cuDeviceGetName_(name,len,device) bind(c, &
&name="cuDeviceGetName")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDeviceGetName_
      type(c_ptr),value :: name
      integer(c_int),value :: len
      integer(c_int),value :: device
    end function

  end interface
  !>  @brief Returns a value for attr of link between two devices
  !>  @param [out] value
  !>  @param [in] attr
  !>  @param [in] srcDevice
  !>  @param [in] dstDevice
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidDevice
  interface cudaDeviceGetP2PAttribute
    function cudaDeviceGetP2PAttribute_(myValue,attr,srcDevice,&
&dstDevice) bind(c, name="cudaDeviceGetP2PAttribute")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceGetP2PAttribute_
      type(c_ptr),value :: myValue
      integer(kind(cudaDevP2PAttrPerformanceRank)),value :: attr
      integer(c_int),value :: srcDevice
      integer(c_int),value :: dstDevice
    end function

  end interface
  !>  @brief Returns a PCI Bus Id string for the device, overloaded to take int device ID.
  !>  @param [out] pciBusId
  !>  @param [in] len
  !>  @param [in] device
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidDevice
  interface cudaDeviceGetPCIBusId
    function cudaDeviceGetPCIBusId_(pciBusId,len,device) bind(c, &
&name="cudaDeviceGetPCIBusId")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceGetPCIBusId_
      type(c_ptr),value :: pciBusId
      integer(c_int),value :: len
      integer(c_int),value :: device
    end function

  end interface
  !>  @brief Returns a handle to a compute device.
  !>  @param [out] device handle
  !>  @param [in] PCI Bus ID
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidDevice, #hipErrorInvalidValue
  interface cudaDeviceGetByPCIBusId
    function cudaDeviceGetByPCIBusId_(device,pciBusId) bind(c, &
&name="cudaDeviceGetByPCIBusId")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceGetByPCIBusId_
      integer(c_int) :: device
      type(c_ptr),value :: pciBusId
    end function

  end interface
  !>  @brief Returns the total amount of memory on the device.
  !>  @param [out] bytes
  !>  @param [in] device
  !> 
  !>  @returns #hipSuccess, #hipErrorInavlidDevice
  interface cuDeviceTotalMem
    function cuDeviceTotalMem_(bytes,device) bind(c, &
&name="cuDeviceTotalMem")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDeviceTotalMem_
      integer(c_size_t) :: bytes
      integer(c_int),value :: device
    end function

  end interface
  !>  @brief Waits on all active streams on current device
  !> 
  !>  When this command is invoked, the host thread gets blocked until all the commands associated
  !>  with streams associated with the device. HIP does not support multiple blocking modes (yet!).
  !> 
  !>  @returns #hipSuccess
  !> 
  !>  @see hipSetDevice, hipDeviceReset
  interface cudaDeviceSynchronize
    function cudaDeviceSynchronize_() bind(c, &
&name="cudaDeviceSynchronize")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceSynchronize_
    end function

  end interface
  !>  @brief The state of current device is discarded and updated to a fresh state.
  !> 
  !>  Calling this function deletes all streams created, memory allocated, kernels running, events
  !>  created. Make sure that no other thread is using the device or streams, memory, kernels, events
  !>  associated with the current device.
  !> 
  !>  @returns #hipSuccess
  !> 
  !>  @see hipDeviceSynchronize
  interface cudaDeviceReset
    function cudaDeviceReset_() bind(c, name="cudaDeviceReset")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceReset_
    end function

  end interface
  !>  @brief Set default device to be used for subsequent hip API calls from this thread.
  !> 
  !>  @param[in] deviceId Valid device in range 0...hipGetDeviceCount().
  !> 
  !>  Sets @p device as the default device for the calling host thread.  Valid device id's are 0...
  !>  (hipGetDeviceCount()-1).
  !> 
  !>  Many HIP APIs implicitly use the "default device" :
  !> 
  !>  - Any device memory subsequently allocated from this host thread (using hipMalloc) will be
  !>  allocated on device.
  !>  - Any streams or events created from this host thread will be associated with device.
  !>  - Any kernels launched from this host thread (using hipLaunchKernel) will be executed on device
  !>  (unless a specific stream is specified, in which case the device associated with that stream will
  !>  be used).
  !> 
  !>  This function may be called from any host thread.  Multiple host threads may use the same device.
  !>  This function does no synchronization with the previous or new device, and has very little
  !>  runtime overhead. Applications can use hipSetDevice to quickly switch the default device before
  !>  making a HIP runtime call which uses the default device.
  !> 
  !>  The default device is stored in thread-local-storage for each thread.
  !>  Thread-pool implementations may inherit the default device of the previous thread.  A good
  !>  practice is to always call hipSetDevice at the start of HIP coding sequency to establish a known
  !>  standard device.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDevice, #hipErrorDeviceAlreadyInUse
  !> 
  !>  @see hipGetDevice, hipGetDeviceCount
  interface cudaSetDevice
    function cudaSetDevice_(deviceId) bind(c, name="cudaSetDevice")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaSetDevice_
      integer(c_int),value :: deviceId
    end function

  end interface
  !>  @brief Return the default device id for the calling host thread.
  !> 
  !>  @param [out] device *device is written with the default device
  !> 
  !>  HIP maintains an default device for each thread using thread-local-storage.
  !>  This device is used implicitly for HIP runtime APIs called by this thread.
  !>  hipGetDevice returns in * @p device the default device for the calling host thread.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>  @see hipSetDevice, hipGetDevicesizeBytes
  interface cudaGetDevice
    function cudaGetDevice_(deviceId) bind(c, name="cudaGetDevice")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetDevice_
      integer(c_int) :: deviceId
    end function

  end interface
  !>  @brief Return number of compute-capable devices.
  !> 
  !>  @param [output] count Returns number of compute-capable devices.
  !> 
  !>  @returns #hipSuccess, #hipErrorNoDevice
  !> 
  !> 
  !>  Returns in @p *count the number of devices that have ability to run compute commands.  If there
  !>  are no such devices, then @ref hipGetDeviceCount will return #hipErrorNoDevice. If 1 or more
  !>  devices can be found, then hipGetDeviceCount returns #hipSuccess.
  interface cudaGetDeviceCount
    function cudaGetDeviceCount_(count) bind(c, &
&name="cudaGetDeviceCount")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetDeviceCount_
      integer(c_int) :: count
    end function

  end interface
  !>  @brief Query for a specific device attribute.
  !> 
  !>  @param [out] pi pointer to value to return
  !>  @param [in] attr attribute to query
  !>  @param [in] deviceId which device to query for information
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  interface cuDeviceGetAttribute
    function cuDeviceGetAttribute_(pi,attr,deviceId) bind(c, &
&name="cuDeviceGetAttribute")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDeviceGetAttribute_
      type(c_ptr),value :: pi
      integer(kind(cudaDevAttrCudaCompatibleBegin)),value :: &
&attr
      integer(c_int),value :: deviceId
    end function

  end interface
  !>  @brief Set L1/Shared cache partition.
  !> 
  !>  @param [in] cacheConfig
  !> 
  !>  @returns #hipSuccess, #hipErrorNotInitialized
  !>  Note: AMD devices and some Nvidia GPUS do not support reconfigurable cache.  This hint is ignored
  !>  on those architectures.
  !>
  interface cudaDeviceSetCacheConfig
    function cudaDeviceSetCacheConfig_(cacheConfig) bind(c, &
&name="cudaDeviceSetCacheConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceSetCacheConfig_
      integer(kind(cudaFuncCachePreferNone)),value :: cacheConfig
    end function

  end interface
  !>  @brief Set Cache configuration for a specific function
  !> 
  !>  @param [in] cacheConfig
  !> 
  !>  @returns #hipSuccess, #hipErrorNotInitialized
  !>  Note: AMD devices and some Nvidia GPUS do not support reconfigurable cache.  This hint is ignored
  !>  on those architectures.
  !>
  interface cudaDeviceGetCacheConfig
    function cudaDeviceGetCacheConfig_(cacheConfig) bind(c, &
&name="cudaDeviceGetCacheConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceGetCacheConfig_
      type(c_ptr),value :: cacheConfig
    end function

  end interface
  !>  @brief Get Resource limits of current device
  !> 
  !>  @param [out] pValue
  !>  @param [in]  limit
  !> 
  !>  @returns #hipSuccess, #hipErrorUnsupportedLimit, #hipErrorInvalidValue
  !>  Note: Currently, only hipLimitMallocHeapSize is available
  !>
  interface cudaDeviceGetLimit
    function cudaDeviceGetLimit_(pValue,limit) bind(c, &
&name="cudaDeviceGetLimit")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceGetLimit_
      integer(c_size_t) :: pValue
      integer(kind(cudaLimitPrintfFifoSize)),value :: limit
    end function

  end interface
  !>  @brief Returns bank width of shared memory for current device
  !> 
  !>  @param [out] pConfig
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  !> 
  !>  Note: AMD devices and some Nvidia GPUS do not support shared cache banking, and the hint is
  !>  ignored on those architectures.
  !>
  interface cudaDeviceGetSharedMemConfig
    function cudaDeviceGetSharedMemConfig_(pConfig) bind(c, &
&name="cudaDeviceGetSharedMemConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceGetSharedMemConfig_
      type(c_ptr),value :: pConfig
    end function

  end interface
  !>  @brief Gets the flags set for current device
  !> 
  !>  @param [out] flags
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  interface cudaGetDeviceFlags
    function cudaGetDeviceFlags_(flags) bind(c, &
&name="cudaGetDeviceFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetDeviceFlags_
      type(c_ptr),value :: flags
    end function

  end interface
  !>  @brief The bank width of shared memory on current device is set
  !> 
  !>  @param [in] config
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  !> 
  !>  Note: AMD devices and some Nvidia GPUS do not support shared cache banking, and the hint is
  !>  ignored on those architectures.
  !>
  interface cudaDeviceSetSharedMemConfig
    function cudaDeviceSetSharedMemConfig_(config) bind(c, &
&name="cudaDeviceSetSharedMemConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceSetSharedMemConfig_
      integer(kind(cudaSharedMemBankSizeDefault)),value :: config
    end function

  end interface
  !>  @brief The current device behavior is changed according the flags passed.
  !> 
  !>  @param [in] flags
  !> 
  !>  The schedule flags impact how HIP waits for the completion of a command running on a device.
  !>  hipDeviceScheduleSpin         : HIP runtime will actively spin in the thread which submitted the
  !>  work until the command completes.  This offers the lowest latency, but will consume a CPU core
  !>  and may increase power. hipDeviceScheduleYield        : The HIP runtime will yield the CPU to
  !>  system so that other tasks can use it.  This may increase latency to detect the completion but
  !>  will consume less power and is friendlier to other tasks in the system.
  !>  hipDeviceScheduleBlockingSync : On ROCm platform, this is a synonym for hipDeviceScheduleYield.
  !>  hipDeviceScheduleAuto         : Use a hueristic to select between Spin and Yield modes.  If the
  !>  number of HIP contexts is greater than the number of logical processors in the system, use Spin
  !>  scheduling.  Else use Yield scheduling.
  !> 
  !> 
  !>  hipDeviceMapHost              : Allow mapping host memory.  On ROCM, this is always allowed and
  !>  the flag is ignored. hipDeviceLmemResizeToMax      : @warning ROCm silently ignores this flag.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDevice, #hipErrorSetOnActiveProcess
  !> 
  !>
  interface cudaSetDeviceFlags
    function cudaSetDeviceFlags_(flags) bind(c, &
&name="cudaSetDeviceFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaSetDeviceFlags_
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Device which matches hipDeviceProp_t is returned
  !> 
  !>  @param [out] device ID
  !>  @param [in]  device properties pointer
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  interface cudaChooseDevice
    function cudaChooseDevice_(device,prop) bind(c, &
&name="cudaChooseDevice")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaChooseDevice_
      integer(c_int) :: device
      type(c_ptr) :: prop
    end function

  end interface
  !>  @brief Returns the link type and hop count between two devices
  !> 
  !>  @param [in] device1 Ordinal for device1
  !>  @param [in] device2 Ordinal for device2
  !>  @param [out] linktype Returns the link type (See hsa_amd_link_info_type_t) between the two devices
  !>  @param [out] hopcount Returns the hop count between the two devices
  !> 
  !>  Queries and returns the HSA link type and the hop count between the two specified devices.
  !> 
  !>  @returns #hipSuccess, #hipInvalidDevice, #hipErrorRuntimeOther
  interface cudaExtGetLinkTypeAndHopCount
    function cudaExtGetLinkTypeAndHopCount_(device1,device2,linktype,&
&hopcount) bind(c, name="cudaExtGetLinkTypeAndHopCount")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaExtGetLinkTypeAndHopCount_
      integer(c_int),value :: device1
      integer(c_int),value :: device2
      type(c_ptr),value :: linktype
      type(c_ptr),value :: hopcount
    end function

  end interface
  !>  @brief Gets an interprocess memory handle for an existing device memory
  !>           allocation
  !> 
  !>  Takes a pointer to the base of an existing device memory allocation created
  !>  with hipMalloc and exports it for use in another process. This is a
  !>  lightweight operation and may be called multiple times on an allocation
  !>  without adverse effects.
  !> 
  !>  If a region of memory is freed with hipFree and a subsequent call
  !>  to hipMalloc returns memory with the same device address,
  !>  hipIpcGetMemHandle will return a unique handle for the
  !>  new memory.
  !> 
  !>  @param handle - Pointer to user allocated hipIpcMemHandle to return
  !>                     the handle in.
  !>  @param devPtr - Base pointer to previously allocated device memory
  !> 
  !>  @returns
  !>  hipSuccess,
  !>  hipErrorInvalidHandle,
  !>  hipErrorOutOfMemory,
  !>  hipErrorMapFailed,
  !>
  interface cudaIpcGetMemHandle
    function cudaIpcGetMemHandle_(handle,devPtr) bind(c, &
&name="cudaIpcGetMemHandle")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaIpcGetMemHandle_
      type(c_ptr) :: handle
      type(c_ptr),value :: devPtr
    end function

  end interface
  !>  @brief Opens an interprocess memory handle exported from another process
  !>           and returns a device pointer usable in the local process.
  !> 
  !>  Maps memory exported from another process with hipIpcGetMemHandle into
  !>  the current device address space. For contexts on different devices
  !>  hipIpcOpenMemHandle can attempt to enable peer access between the
  !>  devices as if the user called hipDeviceEnablePeerAccess. This behavior is
  !>  controlled by the hipIpcMemLazyEnablePeerAccess flag.
  !>  hipDeviceCanAccessPeer can determine if a mapping is possible.
  !> 
  !>  Contexts that may open hipIpcMemHandles are restricted in the following way.
  !>  hipIpcMemHandles from each device in a given process may only be opened
  !>  by one context per device per other process.
  !> 
  !>  Memory returned from hipIpcOpenMemHandle must be freed with
  !>  hipIpcCloseMemHandle.
  !> 
  !>  Calling hipFree on an exported memory region before calling
  !>  hipIpcCloseMemHandle in the importing context will result in undefined
  !>  behavior.
  !> 
  !>  @param devPtr - Returned device pointer
  !>  @param handle - hipIpcMemHandle to open
  !>  @param flags  - Flags for this operation. Must be specified as hipIpcMemLazyEnablePeerAccess
  !> 
  !>  @returns
  !>  hipSuccess,
  !>  hipErrorMapFailed,
  !>  hipErrorInvalidHandle,
  !>  hipErrorTooManyPeers
  !> 
  !>  @note No guarantees are made about the address returned in @p *devPtr.
  !>  In particular, multiple processes may not receive the same address for the same @p handle.
  !>
  interface cudaIpcOpenMemHandle
    function cudaIpcOpenMemHandle_(devPtr,handle,flags) bind(c, &
&name="cudaIpcOpenMemHandle")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaIpcOpenMemHandle_
      type(c_ptr) :: devPtr
      type(c_ptr),value :: handle
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Close memory mapped with hipIpcOpenMemHandle
  !> 
  !>  Unmaps memory returnd by hipIpcOpenMemHandle. The original allocation
  !>  in the exporting process as well as imported mappings in other processes
  !>  will be unaffected.
  !> 
  !>  Any resources used to enable peer access will be freed if this is the
  !>  last mapping using them.
  !> 
  !>  @param devPtr - Device pointer returned by hipIpcOpenMemHandle
  !> 
  !>  @returns
  !>  hipSuccess,
  !>  hipErrorMapFailed,
  !>  hipErrorInvalidHandle,
  !>
  interface cudaIpcCloseMemHandle
    function cudaIpcCloseMemHandle_(devPtr) bind(c, &
&name="cudaIpcCloseMemHandle")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaIpcCloseMemHandle_
      type(c_ptr),value :: devPtr
    end function

  end interface
  !>  @brief Gets an opaque interprocess handle for an event.
  !> 
  !>  This opaque handle may be copied into other processes and opened with cudaIpcOpenEventHandle.
  !>  Then cudaEventRecord, cudaEventSynchronize, cudaStreamWaitEvent and cudaEventQuery may be used in
  !>  either process. Operations on the imported event after the exported event has been freed with hipEventDestroy
  !>  will result in undefined behavior.
  !> 
  !>  @param[out]  handle Pointer to cudaIpcEventHandle to return the opaque event handle
  !>  @param[in]   event  Event allocated with cudaEventInterprocess and cudaEventDisableTiming flags
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidConfiguration, #hipErrorInvalidValue
  !>
  interface cudaIpcGetEventHandle
    function cudaIpcGetEventHandle_(handle,event) bind(c, &
&name="cudaIpcGetEventHandle")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaIpcGetEventHandle_
      type(c_ptr) :: handle
      type(c_ptr),value :: event
    end function

  end interface
  !>  @brief Opens an interprocess event handles.
  !> 
  !>  Opens an interprocess event handle exported from another process with cudaIpcGetEventHandle. The returned
  !>  hipEvent_t behaves like a locally created event with the hipEventDisableTiming flag specified. This event
  !>  need be freed with hipEventDestroy. Operations on the imported event after the exported event has been freed
  !>  with hipEventDestroy will result in undefined behavior. If the function is called within the same process where
  !>  handle is returned by hipIpcGetEventHandle, it will return hipErrorInvalidContext.
  !> 
  !>  @param[out]  event  Pointer to hipEvent_t to return the event
  !>  @param[in]   handle The opaque interprocess handle to open
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidContext
  !>
  interface cudaIpcOpenEventHandle
    function cudaIpcOpenEventHandle_(event,handle) bind(c, &
&name="cudaIpcOpenEventHandle")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaIpcOpenEventHandle_
      type(c_ptr) :: event
      type(c_ptr),value :: handle
    end function

  end interface
  !>  @brief Set attribute for a specific function
  !> 
  !>  @param [in] func;
  !>  @param [in] attr;
  !>  @param [in] value;
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDeviceFunction, #hipErrorInvalidValue
  !> 
  !>  Note: AMD devices and some Nvidia GPUS do not support shared cache banking, and the hint is
  !>  ignored on those architectures.
  !>
  interface cudaFuncSetAttribute
    function cudaFuncSetAttribute_(func,attr,myValue) bind(c, &
&name="cudaFuncSetAttribute")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFuncSetAttribute_
      type(c_ptr),value :: func
      integer(kind(cudaFuncAttributeMaxDynamicSharedMemorySize)),value &
&:: attr
      integer(c_int),value :: myValue
    end function

  end interface
  !>  @brief Set Cache configuration for a specific function
  !> 
  !>  @param [in] config;
  !> 
  !>  @returns #hipSuccess, #hipErrorNotInitialized
  !>  Note: AMD devices and some Nvidia GPUS do not support reconfigurable cache.  This hint is ignored
  !>  on those architectures.
  !>
  interface cudaFuncSetCacheConfig
    function cudaFuncSetCacheConfig_(func,config) bind(c, &
&name="cudaFuncSetCacheConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFuncSetCacheConfig_
      type(c_ptr),value :: func
      integer(kind(cudaFuncCachePreferNone)),value :: config
    end function

  end interface
  !>  @brief Set shared memory configuation for a specific function
  !> 
  !>  @param [in] func
  !>  @param [in] config
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDeviceFunction, #hipErrorInvalidValue
  !> 
  !>  Note: AMD devices and some Nvidia GPUS do not support shared cache banking, and the hint is
  !>  ignored on those architectures.
  !>
  interface cudaFuncSetSharedMemConfig
    function cudaFuncSetSharedMemConfig_(func,config) bind(c, &
&name="cudaFuncSetSharedMemConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFuncSetSharedMemConfig_
      type(c_ptr),value :: func
      integer(kind(cudaSharedMemBankSizeDefault)),value :: config
    end function

  end interface
  !>  @brief Return last error returned by any HIP runtime API call and resets the stored error code to
  !>  #hipSuccess
  !> 
  !>  @returns return code from last HIP called from the active host thread
  !> 
  !>  Returns the last error that has been returned by any of the runtime calls in the same host
  !>  thread, and then resets the saved error to #hipSuccess.
  !> 
  !>  @see hipGetErrorString, hipGetLastError, hipPeakAtLastError, hipError_t
  interface cudaGetLastError
    function cudaGetLastError_() bind(c, name="cudaGetLastError")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetLastError_
    end function

  end interface
  !>  @brief Return last error returned by any HIP runtime API call.
  !> 
  !>  @return #hipSuccess
  !> 
  !>  Returns the last error that has been returned by any of the runtime calls in the same host
  !>  thread. Unlike hipGetLastError, this function does not reset the saved error code.
  !> 
  !>  @see hipGetErrorString, hipGetLastError, hipPeakAtLastError, hipError_t
  interface cudaPeekAtLastError
    function cudaPeekAtLastError_() bind(c, name="cudaPeekAtLastError")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaPeekAtLastError_
    end function

  end interface
  !>  @brief Create an asynchronous stream.
  !> 
  !>  @param[in, out] stream Valid pointer to hipStream_t.  This function writes the memory with the
  !>  newly created stream.
  !>  @return #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  Create a new asynchronous stream.  @p stream returns an opaque handle that can be used to
  !>  reference the newly created stream in subsequent hipStream* commands.  The stream is allocated on
  !>  the heap and will remain allocated even if the handle goes out-of-scope.  To release the memory
  !>  used by the stream, applicaiton must call hipStreamDestroy.
  !> 
  !>  @return #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @see hipStreamCreateWithFlags, hipStreamCreateWithPriority, hipStreamSynchronize, hipStreamWaitEvent, hipStreamDestroy
  interface cudaStreamCreate
    function cudaStreamCreate_(stream) bind(c, name="cudaStreamCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamCreate_
      type(c_ptr) :: stream
    end function

  end interface
  !>  @brief Create an asynchronous stream.
  !> 
  !>  @param[in, out] stream Pointer to new stream
  !>  @param[in ] flags to control stream creation.
  !>  @return #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  Create a new asynchronous stream.  @p stream returns an opaque handle that can be used to
  !>  reference the newly created stream in subsequent hipStream* commands.  The stream is allocated on
  !>  the heap and will remain allocated even if the handle goes out-of-scope.  To release the memory
  !>  used by the stream, applicaiton must call hipStreamDestroy. Flags controls behavior of the
  !>  stream.  See #hipStreamDefault, #hipStreamNonBlocking.
  !> 
  !> 
  !>  @see hipStreamCreate, hipStreamCreateWithPriority, hipStreamSynchronize, hipStreamWaitEvent, hipStreamDestroy
  interface cudaStreamCreateWithFlags
    function cudaStreamCreateWithFlags_(stream,flags) bind(c, &
&name="cudaStreamCreateWithFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamCreateWithFlags_
      type(c_ptr) :: stream
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Create an asynchronous stream with the specified priority.
  !> 
  !>  @param[in, out] stream Pointer to new stream
  !>  @param[in ] flags to control stream creation.
  !>  @param[in ] priority of the stream. Lower numbers represent higher priorities.
  !>  @return #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  Create a new asynchronous stream with the specified priority.  @p stream returns an opaque handle
  !>  that can be used to reference the newly created stream in subsequent hipStream* commands.  The
  !>  stream is allocated on the heap and will remain allocated even if the handle goes out-of-scope.
  !>  To release the memory used by the stream, applicaiton must call hipStreamDestroy. Flags controls
  !>  behavior of the stream.  See #hipStreamDefault, #hipStreamNonBlocking.
  !> 
  !> 
  !>  @see hipStreamCreate, hipStreamSynchronize, hipStreamWaitEvent, hipStreamDestroy
  interface cudaStreamCreateWithPriority
    function cudaStreamCreateWithPriority_(stream,flags,priority) &
&bind(c, name="cudaStreamCreateWithPriority")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamCreateWithPriority_
      type(c_ptr) :: stream
      integer(c_int),value :: flags
      integer(c_int),value :: priority
    end function

  end interface
  !>  @brief Returns numerical values that correspond to the least and greatest stream priority.
  !> 
  !>  @param[in, out] leastPriority pointer in which value corresponding to least priority is returned.
  !>  @param[in, out] greatestPriority pointer in which value corresponding to greatest priority is returned.
  !> 
  !>  Returns in *leastPriority and *greatestPriority the numerical values that correspond to the least
  !>  and greatest stream priority respectively. Stream priorities follow a convention where lower numbers
  !>  imply greater priorities. The range of meaningful stream priorities is given by
  !>  [*greatestPriority, *leastPriority]. If the user attempts to create a stream with a priority value
  !>  that is outside the the meaningful range as specified by this API, the priority is automatically
  !>  clamped to within the valid range.
  interface cudaDeviceGetStreamPriorityRange
    function cudaDeviceGetStreamPriorityRange_(leastPriority,&
&greatestPriority) bind(c, name="cudaDeviceGetStreamPriorityRange")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceGetStreamPriorityRange_
      type(c_ptr),value :: leastPriority
      type(c_ptr),value :: greatestPriority
    end function

  end interface
  !>  @brief Destroys the specified stream.
  !> 
  !>  @param[in, out] stream Valid pointer to hipStream_t.  This function writes the memory with the
  !>  newly created stream.
  !>  @return #hipSuccess #hipErrorInvalidHandle
  !> 
  !>  Destroys the specified stream.
  !> 
  !>  If commands are still executing on the specified stream, some may complete execution before the
  !>  queue is deleted.
  !> 
  !>  The queue may be destroyed while some commands are still inflight, or may wait for all commands
  !>  queued to the stream before destroying it.
  !> 
  !>  @see hipStreamCreate, hipStreamCreateWithFlags, hipStreamCreateWithPriority, hipStreamQuery, hipStreamWaitEvent,
  !>  hipStreamSynchronize
  interface cudaStreamDestroy
    function cudaStreamDestroy_(stream) bind(c, &
&name="cudaStreamDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamDestroy_
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Return #hipSuccess if all of the operations in the specified @p stream have completed, or
  !>  #hipErrorNotReady if not.
  !> 
  !>  @param[in] stream stream to query
  !> 
  !>  @return #hipSuccess, #hipErrorNotReady, #hipErrorInvalidHandle
  !> 
  !>  This is thread-safe and returns a snapshot of the current state of the queue.  However, if other
  !>  host threads are sending work to the stream, the status may change immediately after the function
  !>  is called.  It is typically used for debug.
  !> 
  !>  @see hipStreamCreate, hipStreamCreateWithFlags, hipStreamCreateWithPriority, hipStreamWaitEvent, hipStreamSynchronize,
  !>  hipStreamDestroy
  interface cudaStreamQuery
    function cudaStreamQuery_(stream) bind(c, name="cudaStreamQuery")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamQuery_
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Wait for all commands in stream to complete.
  !> 
  !>  @param[in] stream stream identifier.
  !> 
  !>  @return #hipSuccess, #hipErrorInvalidHandle
  !> 
  !>  This command is host-synchronous : the host will block until the specified stream is empty.
  !> 
  !>  This command follows standard null-stream semantics.  Specifically, specifying the null stream
  !>  will cause the command to wait for other streams on the same device to complete all pending
  !>  operations.
  !> 
  !>  This command honors the hipDeviceLaunchBlocking flag, which controls whether the wait is active
  !>  or blocking.
  !> 
  !>  @see hipStreamCreate, hipStreamCreateWithFlags, hipStreamCreateWithPriority, hipStreamWaitEvent, hipStreamDestroy
  !>
  interface cudaStreamSynchronize
    function cudaStreamSynchronize_(stream) bind(c, &
&name="cudaStreamSynchronize")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamSynchronize_
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Make the specified compute stream wait for an event
  !> 
  !>  @param[in] stream stream to make wait.
  !>  @param[in] event event to wait on
  !>  @param[in] flags control operation [must be 0]
  !> 
  !>  @return #hipSuccess, #hipErrorInvalidHandle
  !> 
  !>  This function inserts a wait operation into the specified stream.
  !>  All future work submitted to @p stream will wait until @p event reports completion before
  !>  beginning execution.
  !> 
  !>  This function only waits for commands in the current stream to complete.  Notably,, this function
  !>  does not impliciy wait for commands in the default stream to complete, even if the specified
  !>  stream is created with hipStreamNonBlocking = 0.
  !> 
  !>  @see hipStreamCreate, hipStreamCreateWithFlags, hipStreamCreateWithPriority, hipStreamSynchronize, hipStreamDestroy
  interface cudaStreamWaitEvent
    function cudaStreamWaitEvent_(stream,event,flags) bind(c, &
&name="cudaStreamWaitEvent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamWaitEvent_
      type(c_ptr),value :: stream
      type(c_ptr),value :: event
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Return flags associated with this stream.
  !> 
  !>  @param[in] stream stream to be queried
  !>  @param[in,out] flags Pointer to an unsigned integer in which the stream's flags are returned
  !>  @return #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidHandle
  !> 
  !>  @returns #hipSuccess #hipErrorInvalidValue #hipErrorInvalidHandle
  !> 
  !>  Return flags associated with this stream in *@p flags.
  !> 
  !>  @see hipStreamCreateWithFlags
  interface cudaStreamGetFlags
    function cudaStreamGetFlags_(stream,flags) bind(c, &
&name="cudaStreamGetFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamGetFlags_
      type(c_ptr),value :: stream
      type(c_ptr),value :: flags
    end function

  end interface
  !>  @brief Query the priority of a stream.
  !> 
  !>  @param[in] stream stream to be queried
  !>  @param[in,out] priority Pointer to an unsigned integer in which the stream's priority is returned
  !>  @return #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidHandle
  !> 
  !>  @returns #hipSuccess #hipErrorInvalidValue #hipErrorInvalidHandle
  !> 
  !>  Query the priority of a stream. The priority is returned in in priority.
  !> 
  !>  @see hipStreamCreateWithFlags
  interface cudaStreamGetPriority
    function cudaStreamGetPriority_(stream,priority) bind(c, &
&name="cudaStreamGetPriority")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamGetPriority_
      type(c_ptr),value :: stream
      type(c_ptr),value :: priority
    end function

  end interface
  !>  @brief Create an asynchronous stream with the specified CU mask.
  !> 
  !>  @param[in, out] stream Pointer to new stream
  !>  @param[in ] cuMaskSize Size of CU mask bit array passed in.
  !>  @param[in ] cuMask Bit-vector representing the CU mask. Each active bit represents using one CU.
  !>  The first 32 bits represent the first 32 CUs, and so on. If its size is greater than physical
  !>  CU number (i.e., multiProcessorCount member of hipDeviceProp_t), the extra elements are ignored.
  !>  It is user's responsibility to make sure the input is meaningful.
  !>  @return #hipSuccess, #hipErrorInvalidHandle, #hipErrorInvalidValue
  !> 
  !>  Create a new asynchronous stream with the specified CU mask.  @p stream returns an opaque handle
  !>  that can be used to reference the newly created stream in subsequent hipStream* commands.  The
  !>  stream is allocated on the heap and will remain allocated even if the handle goes out-of-scope.
  !>  To release the memory used by the stream, application must call hipStreamDestroy.
  !> 
  !> 
  !>  @see hipStreamCreate, hipStreamSynchronize, hipStreamWaitEvent, hipStreamDestroy
  interface cudaExtStreamCreateWithCUMask
    function cudaExtStreamCreateWithCUMask_(stream,cuMaskSize,cuMask) &
&bind(c, name="cudaExtStreamCreateWithCUMask")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaExtStreamCreateWithCUMask_
      type(c_ptr) :: stream
      integer(c_int),value :: cuMaskSize
      type(c_ptr),value :: cuMask
    end function

  end interface
  !>  @brief Get CU mask associated with an asynchronous stream
  !> 
  !>  @param[in] stream stream to be queried
  !>  @param[in] cuMaskSize number of the block of memories (uint32_t *) allocated by user
  !>  @param[out] cuMask Pointer to a pre-allocated block of memories (uint32_t *) in which
  !>  the stream's CU mask is returned. The CU mask is returned in a chunck of 32 bits where
  !>  each active bit represents one active CU
  !>  @return #hipSuccess, #hipErrorInvalidHandle, #hipErrorInvalidValue
  !> 
  !>  @see hipStreamCreate, hipStreamSynchronize, hipStreamWaitEvent, hipStreamDestroy
  interface cudaExtStreamGetCUMask
    function cudaExtStreamGetCUMask_(stream,cuMaskSize,cuMask) bind(c, &
&name="cudaExtStreamGetCUMask")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaExtStreamGetCUMask_
      type(c_ptr),value :: stream
      integer(c_int),value :: cuMaskSize
      type(c_ptr),value :: cuMask
    end function

  end interface
  !>  @brief Adds a callback to be called on the host after all currently enqueued
  !>  items in the stream have completed.  For each
  !>  hipStreamAddCallback call, a callback will be executed exactly once.
  !>  The callback will block later work in the stream until it is finished.
  !>  @param[in] stream   - Stream to add callback to
  !>  @param[in] callback - The function to call once preceding stream operations are complete
  !>  @param[in] userData - User specified data to be passed to the callback function
  !>  @param[in] flags    - Reserved for future use, must be 0
  !>  @return #hipSuccess, #hipErrorInvalidHandle, #hipErrorNotSupported
  !> 
  !>  @see hipStreamCreate, hipStreamCreateWithFlags, hipStreamQuery, hipStreamSynchronize,
  !>  hipStreamWaitEvent, hipStreamDestroy, hipStreamCreateWithPriority
  !>
  interface cudaStreamAddCallback
    function cudaStreamAddCallback_(stream,callback,userData,flags) &
&bind(c, name="cudaStreamAddCallback")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamAddCallback_
      type(c_ptr),value :: stream
      type(c_funptr),value :: callback
      type(c_ptr),value :: userData
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Enqueues a write command to the stream.[BETA]
  !> 
  !>  @param [in] stream - Stream identifier
  !>  @param [in] ptr    - Pointer to a GPU accessible memory object
  !>  @param [in] value  - Value to be written
  !>  @param [in] flags  - reserved, ignored for now, will be used in future releases
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  Enqueues a write command to the stream, write operation is performed after all earlier commands
  !>  on this stream have completed the execution.
  !> 
  !>  @beta This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !> 
  !>  @see hipExtMallocWithFlags, hipFree, hipStreamWriteValue32, hipStreamWaitValue32,
  !>  hipStreamWaitValue64
  interface cuStreamWriteValue32
    function cuStreamWriteValue32_(stream,ptr,myValue,flags) bind(c, &
&name="cuStreamWriteValue32")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuStreamWriteValue32_
      type(c_ptr),value :: stream
      type(c_ptr),value :: ptr
      integer(c_int),value :: myValue
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Enqueues a write command to the stream.[BETA]
  !> 
  !>  @param [in] stream - Stream identifier
  !>  @param [in] ptr    - Pointer to a GPU accessible memory object
  !>  @param [in] value  - Value to be written
  !>  @param [in] flags  - reserved, ignored for now, will be used in future releases
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  Enqueues a write command to the stream, write operation is performed after all earlier commands
  !>  on this stream have completed the execution.
  !> 
  !>  @beta This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !> 
  !>  @see hipExtMallocWithFlags, hipFree, hipStreamWriteValue32, hipStreamWaitValue32,
  !>  hipStreamWaitValue64
  interface cuStreamWriteValue64
    function cuStreamWriteValue64_(stream,ptr,myValue,flags) bind(c, &
&name="cuStreamWriteValue64")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuStreamWriteValue64_
      type(c_ptr),value :: stream
      type(c_ptr),value :: ptr
      integer(c_long),value :: myValue
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Create an event with the specified flags
  !> 
  !>  @param[in,out] event Returns the newly created event.
  !>  @param[in] flags     Flags to control event behavior.  Valid values are #hipEventDefault,
  !>  #hipEventBlockingSync, #hipEventDisableTiming, #hipEventInterprocess
  !>  #hipEventDefault : Default flag.  The event will use active synchronization and will support
  !>  timing.  Blocking synchronization provides lowest possible latency at the expense of dedicating a
  !>  CPU to poll on the event.
  !>  #hipEventBlockingSync : The event will use blocking synchronization : if hipEventSynchronize is
  !>  called on this event, the thread will block until the event completes.  This can increase latency
  !>  for the synchroniation but can result in lower power and more resources for other CPU threads.
  !>  #hipEventDisableTiming : Disable recording of timing information. Events created with this flag
  !>  would not record profiling data and provide best performance if used for synchronization.
  !>  @warning On AMD platform, hipEventInterprocess support is under development.  Use of this flag
  !>  will return an error.
  !> 
  !>  @returns #hipSuccess, #hipErrorNotInitialized, #hipErrorInvalidValue,
  !>  #hipErrorLaunchFailure, #hipErrorOutOfMemory
  !> 
  !>  @see hipEventCreate, hipEventSynchronize, hipEventDestroy, hipEventElapsedTime
  interface cudaEventCreateWithFlags
    function cudaEventCreateWithFlags_(event,flags) bind(c, &
&name="cudaEventCreateWithFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaEventCreateWithFlags_
      type(c_ptr) :: event
      integer(c_int),value :: flags
    end function

  end interface
  !>   Create an event
  !> 
  !>  @param[in,out] event Returns the newly created event.
  !> 
  !>  @returns #hipSuccess, #hipErrorNotInitialized, #hipErrorInvalidValue,
  !>  #hipErrorLaunchFailure, #hipErrorOutOfMemory
  !> 
  !>  @see hipEventCreateWithFlags, hipEventRecord, hipEventQuery, hipEventSynchronize,
  !>  hipEventDestroy, hipEventElapsedTime
  interface cudaEventCreate
    function cudaEventCreate_(event) bind(c, name="cudaEventCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaEventCreate_
      type(c_ptr) :: event
    end function

  end interface
  !>  @brief Record an event in the specified stream.
  !> 
  !>  @param[in] event event to record.
  !>  @param[in] stream stream in which to record event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized,
  !>  #hipErrorInvalidHandle, #hipErrorLaunchFailure
  !> 
  !>  hipEventQuery() or hipEventSynchronize() must be used to determine when the event
  !>  transitions from "recording" (after hipEventRecord() is called) to "recorded"
  !>  (when timestamps are set, if requested).
  !> 
  !>  Events which are recorded in a non-NULL stream will transition to
  !>  from recording to "recorded" state when they reach the head of
  !>  the specified stream, after all previous
  !>  commands in that stream have completed executing.
  !> 
  !>  If hipEventRecord() has been previously called on this event, then this call will overwrite any
  !>  existing state in event.
  !> 
  !>  If this function is called on an event that is currently being recorded, results are undefined
  !>  - either outstanding recording may save state into the event, and the order is not guaranteed.
  !> 
  !>  @see hipEventCreate, hipEventCreateWithFlags, hipEventQuery, hipEventSynchronize,
  !>  hipEventDestroy, hipEventElapsedTime
  !>
  interface cudaEventRecord
    function cudaEventRecord_(event,stream) bind(c, &
&name="cudaEventRecord")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaEventRecord_
      type(c_ptr),value :: event
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Destroy the specified event.
  !> 
  !>   @param[in] event Event to destroy.
  !>   @returns #hipSuccess, #hipErrorNotInitialized, #hipErrorInvalidValue,
  !>  #hipErrorLaunchFailure
  !> 
  !>   Releases memory associated with the event.  If the event is recording but has not completed
  !>  recording when hipEventDestroy() is called, the function will return immediately and the
  !>  completion_future resources will be released later, when the hipDevice is synchronized.
  !> 
  !>  @see hipEventCreate, hipEventCreateWithFlags, hipEventQuery, hipEventSynchronize, hipEventRecord,
  !>  hipEventElapsedTime
  !> 
  !>  @returns #hipSuccess
  interface cudaEventDestroy
    function cudaEventDestroy_(event) bind(c, name="cudaEventDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaEventDestroy_
      type(c_ptr),value :: event
    end function

  end interface
  !>   @brief Wait for an event to complete.
  !> 
  !>   This function will block until the event is ready, waiting for all previous work in the stream
  !>  specified when event was recorded with hipEventRecord().
  !> 
  !>   If hipEventRecord() has not been called on @p event, this function returns immediately.
  !> 
  !>   TODO-hip- This function needs to support hipEventBlockingSync parameter.
  !> 
  !>   @param[in] event Event on which to wait.
  !>   @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized,
  !>  #hipErrorInvalidHandle, #hipErrorLaunchFailure
  !> 
  !>   @see hipEventCreate, hipEventCreateWithFlags, hipEventQuery, hipEventDestroy, hipEventRecord,
  !>  hipEventElapsedTime
  interface cudaEventSynchronize
    function cudaEventSynchronize_(event) bind(c, &
&name="cudaEventSynchronize")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaEventSynchronize_
      type(c_ptr),value :: event
    end function

  end interface
  !>  @brief Return the elapsed time between two events.
  !> 
  !>  @param[out] ms : Return time between start and stop in ms.
  !>  @param[in]   start : Start event.
  !>  @param[in]   stop  : Stop event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorNotReady, #hipErrorInvalidHandle,
  !>  #hipErrorNotInitialized, #hipErrorLaunchFailure
  !> 
  !>  Computes the elapsed time between two events. Time is computed in ms, with
  !>  a resolution of approximately 1 us.
  !> 
  !>  Events which are recorded in a NULL stream will block until all commands
  !>  on all other streams complete execution, and then record the timestamp.
  !> 
  !>  Events which are recorded in a non-NULL stream will record their timestamp
  !>  when they reach the head of the specified stream, after all previous
  !>  commands in that stream have completed executing.  Thus the time that
  !>  the event recorded may be significantly after the host calls hipEventRecord().
  !> 
  !>  If hipEventRecord() has not been called on either event, then #hipErrorInvalidHandle is
  !>  returned. If hipEventRecord() has been called on both events, but the timestamp has not yet been
  !>  recorded on one or both events (that is, hipEventQuery() would return #hipErrorNotReady on at
  !>  least one of the events), then #hipErrorNotReady is returned.
  !> 
  !>  Note, for HIP Events used in kernel dispatch using hipExtLaunchKernelGGL/hipExtLaunchKernel,
  !>  events passed in hipExtLaunchKernelGGL/hipExtLaunchKernel are not explicitly recorded and should
  !>  only be used to get elapsed time for that specific launch. In case events are used across
  !>  multiple dispatches, for example, start and stop events from different hipExtLaunchKernelGGL/
  !>  hipExtLaunchKernel calls, they will be treated as invalid unrecorded events, HIP will throw
  !>  error "hipErrorInvalidHandle" from hipEventElapsedTime.
  !> 
  !>  @see hipEventCreate, hipEventCreateWithFlags, hipEventQuery, hipEventDestroy, hipEventRecord,
  !>  hipEventSynchronize
  interface cudaEventElapsedTime
    function cudaEventElapsedTime_(ms,start,myStop) bind(c, &
&name="cudaEventElapsedTime")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaEventElapsedTime_
      type(c_ptr),value :: ms
      type(c_ptr),value :: start
      type(c_ptr),value :: myStop
    end function

  end interface
  !>  @brief Query event status
  !> 
  !>  @param[in] event Event to query.
  !>  @returns #hipSuccess, #hipErrorNotReady, #hipErrorInvalidHandle, #hipErrorInvalidValue,
  !>  #hipErrorNotInitialized, #hipErrorLaunchFailure
  !> 
  !>  Query the status of the specified event.  This function will return #hipErrorNotReady if all
  !>  commands in the appropriate stream (specified to hipEventRecord()) have completed.  If that work
  !>  has not completed, or if hipEventRecord() was not called on the event, then #hipSuccess is
  !>  returned.
  !> 
  !>  @see hipEventCreate, hipEventCreateWithFlags, hipEventRecord, hipEventDestroy,
  !>  hipEventSynchronize, hipEventElapsedTime
  interface cudaEventQuery
    function cudaEventQuery_(event) bind(c, name="cudaEventQuery")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaEventQuery_
      type(c_ptr),value :: event
    end function

  end interface
  !>   @brief Return attributes for the specified pointer
  !> 
  !>   @param [out]  attributes  attributes for the specified pointer
  !>   @param [in]   ptr         pointer to get attributes for
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @see hipPointerGetAttribute
  interface cudaPointerGetAttributes
    function cudaPointerGetAttributes_(attributes,ptr) bind(c, &
&name="cudaPointerGetAttributes")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaPointerGetAttributes_
      type(c_ptr) :: attributes
      type(c_ptr),value :: ptr
    end function

  end interface
  !>   @brief Returns information about the specified pointer.[BETA]
  !> 
  !>   @param [in, out] data     returned pointer attribute value
  !>   @param [in]      atribute attribute to query for
  !>   @param [in]      ptr      pointer to get attributes for
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @beta This API is marked as beta, meaning, while this is feature complete,
  !>   it is still open to changes and may have outstanding issues.
  !> 
  !>   @see hipPointerGetAttributes
  interface cudaPointerGetAttribute
    function cudaPointerGetAttribute_(myData,attribute,ptr) bind(c, &
&name="cudaPointerGetAttribute")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaPointerGetAttribute_
      type(c_ptr),value :: myData
      integer(kind(CU_POINTER_ATTRIBUTE_CONTEXT)),value :: attribute
      type(c_ptr),value :: ptr
    end function

  end interface
  !>   @brief Returns information about the specified pointer.[BETA]
  !> 
  !>   @param [in]  numAttributes   number of attributes to query for
  !>   @param [in]  attributes      attributes to query for
  !>   @param [in, out] data        a two-dimensional containing pointers to memory locations
  !>                                where the result of each attribute query will be written to
  !>   @param [in]  ptr             pointer to get attributes for
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @beta This API is marked as beta, meaning, while this is feature complete,
  !>   it is still open to changes and may have outstanding issues.
  !> 
  !>   @see hipPointerGetAttribute
  interface cudaDrvPointerGetAttributes
    function cudaDrvPointerGetAttributes_(numAttributes,attributes,&
&myData,ptr) bind(c, name="cudaDrvPointerGetAttributes")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDrvPointerGetAttributes_
      integer(c_int),value :: numAttributes
      type(c_ptr),value :: attributes
      type(c_ptr) :: myData
      type(c_ptr),value :: ptr
    end function

  end interface
  !>   @brief Signals a set of external semaphore objects.
  !> 
  !>   @param[in] extSem_out  External semaphores to be waited on
  !>   @param[in] paramsArray Array of semaphore parameters
  !>   @param[in] numExtSems Number of semaphores to wait on
  !>   @param[in] stream Stream to enqueue the wait operations in
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @see
  interface cudaSignalExternalSemaphoresAsync
    function cudaSignalExternalSemaphoresAsync_(extSemArray,&
&paramsArray,numExtSems,stream) bind(c, &
&name="cudaSignalExternalSemaphoresAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaSignalExternalSemaphoresAsync_
      type(c_ptr) :: extSemArray
      type(c_ptr),value :: paramsArray
      integer(c_int),value :: numExtSems
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Waits on a set of external semaphore objects
  !> 
  !>   @param[in] extSem_out  External semaphores to be waited on
  !>   @param[in] paramsArray Array of semaphore parameters
  !>   @param[in] numExtSems Number of semaphores to wait on
  !>   @param[in] stream Stream to enqueue the wait operations in
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @see
  interface cudaWaitExternalSemaphoresAsync
    function cudaWaitExternalSemaphoresAsync_(extSemArray,paramsArray,&
&numExtSems,stream) bind(c, name="cudaWaitExternalSemaphoresAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaWaitExternalSemaphoresAsync_
      type(c_ptr) :: extSemArray
      type(c_ptr),value :: paramsArray
      integer(c_int),value :: numExtSems
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Destroys an external semaphore object and releases any references to the underlying resource. Any outstanding signals or waits must have completed before the semaphore is destroyed.
  !> 
  !>   @param[in] extSem handle to an external memory object
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @see
  interface cudaDestroyExternalSemaphore
    function cudaDestroyExternalSemaphore_(extSem) bind(c, &
&name="cudaDestroyExternalSemaphore")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDestroyExternalSemaphore_
      type(c_ptr),value :: extSem
    end function

  end interface
  !>   @brief Maps a buffer onto an imported memory object.
  !> 
  !>   @param[out] devPtr Returned device pointer to buffer
  !>   @param[in]  extMem  Handle to external memory object
  !>   @param[in]  bufferDesc  Buffer descriptor
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @see
  interface cudaExternalMemoryGetMappedBuffer
    function cudaExternalMemoryGetMappedBuffer_(devPtr,extMem,&
&bufferDesc) bind(c, name="cudaExternalMemoryGetMappedBuffer")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaExternalMemoryGetMappedBuffer_
      type(c_ptr) :: devPtr
      type(c_ptr),value :: extMem
      type(c_ptr),value :: bufferDesc
    end function

  end interface
  !>   @brief Destroys an external memory object.
  !> 
  !>   @param[in] extMem  External memory object to be destroyed
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !> 
  !>   @see
  interface cudaDestroyExternalMemory
    function cudaDestroyExternalMemory_(extMem) bind(c, &
&name="cudaDestroyExternalMemory")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDestroyExternalMemory_
      type(c_ptr),value :: extMem
    end function

  end interface
  !>   @brief Allocate memory on the default accelerator
  !> 
  !>   @param[out] ptr Pointer to the allocated memory
  !>   @param[in]  size Requested memory size
  !>   @param[in]  flags Type of memory allocation
  !> 
  !>   If size is 0, no memory is allocated, *ptr returns nullptr, and hipSuccess is returned.
  !> 
  !>   @return #hipSuccess, #hipErrorOutOfMemory, #hipErrorInvalidValue (bad context, null *ptr)
  !> 
  !>   @see hipMallocPitch, hipFree, hipMallocArray, hipFreeArray, hipMalloc3D, hipMalloc3DArray,
  !>  hipHostFree, hipHostMalloc
  interface cudaExtMallocWithFlags
    function cudaExtMallocWithFlags_(ptr,sizeBytes,flags) bind(c, &
&name="cudaExtMallocWithFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaExtMallocWithFlags_
      type(c_ptr) :: ptr
      integer(c_size_t),value :: sizeBytes
      integer(c_int),value :: flags
    end function

  end interface
  
  interface cudaMallocHost
    function cudaMallocHost_(ptr,mySize) bind(c, name="cudaMallocHost")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMallocHost_
      type(c_ptr) :: ptr
      integer(c_size_t),value :: mySize
    end function

  end interface
  
  interface cudaMemAllocHost
    function cudaMemAllocHost_(ptr,mySize) bind(c, &
&name="cudaMemAllocHost")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemAllocHost_
      type(c_ptr) :: ptr
      integer(c_size_t),value :: mySize
    end function

  end interface
  !>  @brief Prefetches memory to the specified destination device using HIP.
  !> 
  !>  @param [in] dev_ptr  pointer to be prefetched
  !>  @param [in] count    size in bytes for prefetching
  !>  @param [in] device   destination device to prefetch to
  !>  @param [in] stream   stream to enqueue prefetch operation
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  interface cudaMemPrefetchAsync
    function cudaMemPrefetchAsync_(dev_ptr,count,device,stream) bind(c,&
& name="cudaMemPrefetchAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemPrefetchAsync_
      type(c_ptr),value :: dev_ptr
      integer(c_size_t),value :: count
      integer(c_int),value :: device
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Advise about the usage of a given memory range to HIP.
  !> 
  !>  @param [in] dev_ptr  pointer to memory to set the advice for
  !>  @param [in] count    size in bytes of the memory range
  !>  @param [in] advice   advice to be applied for the specified memory range
  !>  @param [in] device   device to apply the advice for
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  interface cudaMemAdvise
    function cudaMemAdvise_(dev_ptr,count,advice,device) bind(c, &
&name="cudaMemAdvise")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemAdvise_
      type(c_ptr),value :: dev_ptr
      integer(c_size_t),value :: count
      integer(kind(cudaMemAdviseSetReadMostly)),value :: advice
      integer(c_int),value :: device
    end function

  end interface
  !>  @brief Query an attribute of a given memory range in HIP.
  !> 
  !>  @param [in,out] data   a pointer to a memory location where the result of each
  !>                         attribute query will be written to
  !>  @param [in] data_size  the size of data
  !>  @param [in] attribute  the attribute to query
  !>  @param [in] dev_ptr    start of the range to query
  !>  @param [in] count      size of the range to query
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  interface cudaMemRangeGetAttribute
    function cudaMemRangeGetAttribute_(myData,data_size,attribute,&
&dev_ptr,count) bind(c, name="cudaMemRangeGetAttribute")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemRangeGetAttribute_
      type(c_ptr),value :: myData
      integer(c_size_t),value :: data_size
      integer(kind(cudaMemRangeAttributeReadMostly)),value :: attribute
      type(c_ptr),value :: dev_ptr
      integer(c_size_t),value :: count
    end function

  end interface
  !>  @brief Query attributes of a given memory range in HIP.
  !> 
  !>  @param [in,out] data     a two-dimensional array containing pointers to memory locations
  !>                           where the result of each attribute query will be written to
  !>  @param [in] data_sizes   an array, containing the sizes of each result
  !>  @param [in] attributes   the attribute to query
  !>  @param [in] num_attributes  an array of attributes to query (numAttributes and the number
  !>                           of attributes in this array should match)
  !>  @param [in] dev_ptr      start of the range to query
  !>  @param [in] count        size of the range to query
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  interface cudaMemRangeGetAttributes
    function cudaMemRangeGetAttributes_(myData,data_sizes,attributes,&
&num_attributes,dev_ptr,count) bind(c, &
&name="cudaMemRangeGetAttributes")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemRangeGetAttributes_
      type(c_ptr) :: myData
      type(c_ptr),value :: data_sizes
      type(c_ptr),value :: attributes
      integer(c_size_t),value :: num_attributes
      type(c_ptr),value :: dev_ptr
      integer(c_size_t),value :: count
    end function

  end interface
  !>  @brief Attach memory to a stream asynchronously in HIP.
  !> 
  !>  @param [in] stream     - stream in which to enqueue the attach operation
  !>  @param [in] dev_ptr    - pointer to memory (must be a pointer to managed memory or
  !>                           to a valid host-accessible region of system-allocated memory)
  !>  @param [in] length     - length of memory (defaults to zero)
  !>  @param [in] flags      - must be one of hipMemAttachGlobal, hipMemAttachHost or
  !>                           hipMemAttachSingle (defaults to hipMemAttachSingle)
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  interface cudaStreamAttachMemAsync
    function cudaStreamAttachMemAsync_(stream,dev_ptr,length,flags) &
&bind(c, name="cudaStreamAttachMemAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamAttachMemAsync_
      type(c_ptr),value :: stream
      type(c_ptr),value :: dev_ptr
      integer(c_size_t),value :: length
      integer(c_int),value :: flags
    end function

  end interface
  
  interface cudaHostAlloc
    function cudaHostAlloc_(ptr,mySize,flags) bind(c, &
&name="cudaHostAlloc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaHostAlloc_
      type(c_ptr) :: ptr
      integer(c_size_t),value :: mySize
      integer(c_int),value :: flags
    end function

  end interface
  !>   Allocates at least width (in bytes) * height bytes of linear memory
  !>   Padding may occur to ensure alighnment requirements are met for the given row
  !>   The change in width size due to padding will be returned in *pitch.
  !>   Currently the alignment is set to 128 bytes
  !> 
  !>   @param[out] ptr Pointer to the allocated device memory
  !>   @param[out] pitch Pitch for allocation (in bytes)
  !>   @param[in]  width Requested pitched allocation width (in bytes)
  !>   @param[in]  height Requested pitched allocation height
  !> 
  !>   If size is 0, no memory is allocated, *ptr returns nullptr, and hipSuccess is returned.
  !> 
  !>   @return Error code
  !> 
  !>   @see hipMalloc, hipFree, hipMallocArray, hipFreeArray, hipHostFree, hipMalloc3D,
  !>  hipMalloc3DArray, hipHostMalloc
  interface cudaMallocPitch
    function cudaMallocPitch_(ptr,pitch,width,height) bind(c, &
&name="cudaMallocPitch")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMallocPitch_
      type(c_ptr) :: ptr
      integer(c_size_t) :: pitch
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
    end function

  end interface
  !>   Allocates at least width (in bytes) * height bytes of linear memory
  !>   Padding may occur to ensure alighnment requirements are met for the given row
  !>   The change in width size due to padding will be returned in *pitch.
  !>   Currently the alignment is set to 128 bytes
  !> 
  !>   @param[out] dptr Pointer to the allocated device memory
  !>   @param[out] pitch Pitch for allocation (in bytes)
  !>   @param[in]  width Requested pitched allocation width (in bytes)
  !>   @param[in]  height Requested pitched allocation height
  !> 
  !>   If size is 0, no memory is allocated, *ptr returns nullptr, and hipSuccess is returned.
  !>   The intended usage of pitch is as a separate parameter of the allocation, used to compute addresses within the 2D array.
  !>   Given the row and column of an array element of type T, the address is computed as:
  !>   T* pElement = (T*)((char*)BaseAddress + Row * Pitch) + Column;
  !> 
  !>   @return Error code
  !> 
  !>   @see hipMalloc, hipFree, hipMallocArray, hipFreeArray, hipHostFree, hipMalloc3D,
  !>  hipMalloc3DArray, hipHostMalloc
  interface cuMemAllocPitch
    function cuMemAllocPitch_(dptr,pitch,widthInBytes,height,&
&elementSizeBytes) bind(c, name="cuMemAllocPitch")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemAllocPitch_
      type(c_ptr) :: dptr
      integer(c_size_t) :: pitch
      integer(c_size_t),value :: widthInBytes
      integer(c_size_t),value :: height
      integer(c_int),value :: elementSizeBytes
    end function

  end interface
  
  interface cudaFreeHost
    function cudaFreeHost_(ptr) bind(c, name="cudaFreeHost")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFreeHost_
      type(c_ptr),value :: ptr
    end function

  end interface
  
  interface cudaMemcpyWithStream
    function cudaMemcpyWithStream_(dst,src,sizeBytes,myKind,stream) &
&bind(c, name="cudaMemcpyWithStream")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyWithStream_
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Copy data from Host to Device
  !> 
  !>   @param[out]  dst Data being copy to
  !>   @param[in]   src Data being copy from
  !>   @param[in]   sizeBytes Data size in bytes
  !> 
  !>   @return #hipSuccess, #hipErrorDeInitialized, #hipErrorNotInitialized, #hipErrorInvalidContext,
  !>  #hipErrorInvalidValue
  !> 
  !>   @see hipArrayCreate, hipArrayDestroy, hipArrayGetDescriptor, hipMemAlloc, hipMemAllocHost,
  !>  hipMemAllocPitch, hipMemcpy2D, hipMemcpy2DAsync, hipMemcpy2DUnaligned, hipMemcpyAtoA,
  !>  hipMemcpyAtoD, hipMemcpyAtoH, hipMemcpyAtoHAsync, hipMemcpyDtoA, hipMemcpyDtoD,
  !>  hipMemcpyDtoDAsync, hipMemcpyDtoH, hipMemcpyDtoHAsync, hipMemcpyHtoA, hipMemcpyHtoAAsync,
  !>  hipMemcpyHtoDAsync, hipMemFree, hipMemFreeHost, hipMemGetAddressRange, hipMemGetInfo,
  !>  hipMemHostAlloc, hipMemHostGetDevicePointer
  interface cuMemcpyHtoD
    function cuMemcpyHtoD_(dst,src,sizeBytes) bind(c, &
&name="cuMemcpyHtoD")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyHtoD_
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
    end function

  end interface
  !>   @brief Copy data from Device to Host
  !> 
  !>   @param[out]  dst Data being copy to
  !>   @param[in]   src Data being copy from
  !>   @param[in]   sizeBytes Data size in bytes
  !> 
  !>   @return #hipSuccess, #hipErrorDeInitialized, #hipErrorNotInitialized, #hipErrorInvalidContext,
  !>  #hipErrorInvalidValue
  !> 
  !>   @see hipArrayCreate, hipArrayDestroy, hipArrayGetDescriptor, hipMemAlloc, hipMemAllocHost,
  !>  hipMemAllocPitch, hipMemcpy2D, hipMemcpy2DAsync, hipMemcpy2DUnaligned, hipMemcpyAtoA,
  !>  hipMemcpyAtoD, hipMemcpyAtoH, hipMemcpyAtoHAsync, hipMemcpyDtoA, hipMemcpyDtoD,
  !>  hipMemcpyDtoDAsync, hipMemcpyDtoH, hipMemcpyDtoHAsync, hipMemcpyHtoA, hipMemcpyHtoAAsync,
  !>  hipMemcpyHtoDAsync, hipMemFree, hipMemFreeHost, hipMemGetAddressRange, hipMemGetInfo,
  !>  hipMemHostAlloc, hipMemHostGetDevicePointer
  interface cuMemcpyDtoH
    function cuMemcpyDtoH_(dst,src,sizeBytes) bind(c, &
&name="cuMemcpyDtoH")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyDtoH_
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
    end function

  end interface
  !>   @brief Copy data from Device to Device
  !> 
  !>   @param[out]  dst Data being copy to
  !>   @param[in]   src Data being copy from
  !>   @param[in]   sizeBytes Data size in bytes
  !> 
  !>   @return #hipSuccess, #hipErrorDeInitialized, #hipErrorNotInitialized, #hipErrorInvalidContext,
  !>  #hipErrorInvalidValue
  !> 
  !>   @see hipArrayCreate, hipArrayDestroy, hipArrayGetDescriptor, hipMemAlloc, hipMemAllocHost,
  !>  hipMemAllocPitch, hipMemcpy2D, hipMemcpy2DAsync, hipMemcpy2DUnaligned, hipMemcpyAtoA,
  !>  hipMemcpyAtoD, hipMemcpyAtoH, hipMemcpyAtoHAsync, hipMemcpyDtoA, hipMemcpyDtoD,
  !>  hipMemcpyDtoDAsync, hipMemcpyDtoH, hipMemcpyDtoHAsync, hipMemcpyHtoA, hipMemcpyHtoAAsync,
  !>  hipMemcpyHtoDAsync, hipMemFree, hipMemFreeHost, hipMemGetAddressRange, hipMemGetInfo,
  !>  hipMemHostAlloc, hipMemHostGetDevicePointer
  interface cuMemcpyDtoD
    function cuMemcpyDtoD_(dst,src,sizeBytes) bind(c, &
&name="cuMemcpyDtoD")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyDtoD_
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
    end function

  end interface
  !>   @brief Copy data from Host to Device asynchronously
  !> 
  !>   @param[out]  dst Data being copy to
  !>   @param[in]   src Data being copy from
  !>   @param[in]   sizeBytes Data size in bytes
  !> 
  !>   @return #hipSuccess, #hipErrorDeInitialized, #hipErrorNotInitialized, #hipErrorInvalidContext,
  !>  #hipErrorInvalidValue
  !> 
  !>   @see hipArrayCreate, hipArrayDestroy, hipArrayGetDescriptor, hipMemAlloc, hipMemAllocHost,
  !>  hipMemAllocPitch, hipMemcpy2D, hipMemcpy2DAsync, hipMemcpy2DUnaligned, hipMemcpyAtoA,
  !>  hipMemcpyAtoD, hipMemcpyAtoH, hipMemcpyAtoHAsync, hipMemcpyDtoA, hipMemcpyDtoD,
  !>  hipMemcpyDtoDAsync, hipMemcpyDtoH, hipMemcpyDtoHAsync, hipMemcpyHtoA, hipMemcpyHtoAAsync,
  !>  hipMemcpyHtoDAsync, hipMemFree, hipMemFreeHost, hipMemGetAddressRange, hipMemGetInfo,
  !>  hipMemHostAlloc, hipMemHostGetDevicePointer
  interface cuMemcpyHtoDAsync
    function cuMemcpyHtoDAsync_(dst,src,sizeBytes,stream) bind(c, &
&name="cuMemcpyHtoDAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyHtoDAsync_
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Copy data from Device to Host asynchronously
  !> 
  !>   @param[out]  dst Data being copy to
  !>   @param[in]   src Data being copy from
  !>   @param[in]   sizeBytes Data size in bytes
  !> 
  !>   @return #hipSuccess, #hipErrorDeInitialized, #hipErrorNotInitialized, #hipErrorInvalidContext,
  !>  #hipErrorInvalidValue
  !> 
  !>   @see hipArrayCreate, hipArrayDestroy, hipArrayGetDescriptor, hipMemAlloc, hipMemAllocHost,
  !>  hipMemAllocPitch, hipMemcpy2D, hipMemcpy2DAsync, hipMemcpy2DUnaligned, hipMemcpyAtoA,
  !>  hipMemcpyAtoD, hipMemcpyAtoH, hipMemcpyAtoHAsync, hipMemcpyDtoA, hipMemcpyDtoD,
  !>  hipMemcpyDtoDAsync, hipMemcpyDtoH, hipMemcpyDtoHAsync, hipMemcpyHtoA, hipMemcpyHtoAAsync,
  !>  hipMemcpyHtoDAsync, hipMemFree, hipMemFreeHost, hipMemGetAddressRange, hipMemGetInfo,
  !>  hipMemHostAlloc, hipMemHostGetDevicePointer
  interface cuMemcpyDtoHAsync
    function cuMemcpyDtoHAsync_(dst,src,sizeBytes,stream) bind(c, &
&name="cuMemcpyDtoHAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyDtoHAsync_
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Copy data from Device to Device asynchronously
  !> 
  !>   @param[out]  dst Data being copy to
  !>   @param[in]   src Data being copy from
  !>   @param[in]   sizeBytes Data size in bytes
  !> 
  !>   @return #hipSuccess, #hipErrorDeInitialized, #hipErrorNotInitialized, #hipErrorInvalidContext,
  !>  #hipErrorInvalidValue
  !> 
  !>   @see hipArrayCreate, hipArrayDestroy, hipArrayGetDescriptor, hipMemAlloc, hipMemAllocHost,
  !>  hipMemAllocPitch, hipMemcpy2D, hipMemcpy2DAsync, hipMemcpy2DUnaligned, hipMemcpyAtoA,
  !>  hipMemcpyAtoD, hipMemcpyAtoH, hipMemcpyAtoHAsync, hipMemcpyDtoA, hipMemcpyDtoD,
  !>  hipMemcpyDtoDAsync, hipMemcpyDtoH, hipMemcpyDtoHAsync, hipMemcpyHtoA, hipMemcpyHtoAAsync,
  !>  hipMemcpyHtoDAsync, hipMemFree, hipMemFreeHost, hipMemGetAddressRange, hipMemGetInfo,
  !>  hipMemHostAlloc, hipMemHostGetDevicePointer
  interface cuMemcpyDtoDAsync
    function cuMemcpyDtoDAsync_(dst,src,sizeBytes,stream) bind(c, &
&name="cuMemcpyDtoDAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyDtoDAsync_
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Returns a global pointer from a module.
  !>   Returns in *dptr and *bytes the pointer and size of the global of name name located in module hmod.
  !>   If no variable of that name exists, it returns hipErrorNotFound. Both parameters dptr and bytes are optional.
  !>   If one of them is NULL, it is ignored and hipSuccess is returned.
  !> 
  !>   @param[out]  dptr  Returned global device pointer
  !>   @param[out]  bytes Returned global size in bytes
  !>   @param[in]   hmod  Module to retrieve global from
  !>   @param[in]   name  Name of global to retrieve
  !> 
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorNotFound, #hipErrorInvalidContext
  !>
  interface cuModuleGetGlobal
    function cuModuleGetGlobal_(dptr,bytes,hmod,name) bind(c, &
&name="cuModuleGetGlobal")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuModuleGetGlobal_
      type(c_ptr) :: dptr
      integer(c_size_t) :: bytes
      type(c_ptr),value :: hmod
      type(c_ptr),value :: name
    end function

  end interface
  
  interface cudaGetSymbolAddress
    function cudaGetSymbolAddress_(devPtr,symbol) bind(c, &
&name="cudaGetSymbolAddress")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetSymbolAddress_
      type(c_ptr) :: devPtr
      type(c_ptr),value :: symbol
    end function

  end interface
  
  interface cudaGetSymbolSize
    function cudaGetSymbolSize_(mySize,symbol) bind(c, &
&name="cudaGetSymbolSize")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetSymbolSize_
      integer(c_size_t) :: mySize
      type(c_ptr),value :: symbol
    end function

  end interface
  
  interface cudaMemcpyToSymbol
    function cudaMemcpyToSymbol_(symbol,src,sizeBytes,offset,myKind) &
&bind(c, name="cudaMemcpyToSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyToSymbol_
      type(c_ptr),value :: symbol
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  
  interface cudaMemcpyToSymbolAsync
    function cudaMemcpyToSymbolAsync_(symbol,src,sizeBytes,offset,&
&myKind,stream) bind(c, name="cudaMemcpyToSymbolAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyToSymbolAsync_
      type(c_ptr),value :: symbol
      type(c_ptr),value :: src
      integer(c_size_t),value :: sizeBytes
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
      type(c_ptr),value :: stream
    end function

  end interface
  
  interface cudaMemcpyFromSymbol
    function cudaMemcpyFromSymbol_(dst,symbol,sizeBytes,offset,myKind) &
&bind(c, name="cudaMemcpyFromSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyFromSymbol_
      type(c_ptr),value :: dst
      type(c_ptr),value :: symbol
      integer(c_size_t),value :: sizeBytes
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  
  interface cudaMemcpyFromSymbolAsync
    function cudaMemcpyFromSymbolAsync_(dst,symbol,sizeBytes,offset,&
&myKind,stream) bind(c, name="cudaMemcpyFromSymbolAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyFromSymbolAsync_
      type(c_ptr),value :: dst
      type(c_ptr),value :: symbol
      integer(c_size_t),value :: sizeBytes
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Fills the first sizeBytes bytes of the memory area pointed to by dest with the ant
  !>  byte value value.
  !> 
  !>   @param[out] dst Data being filled
  !>   @param[in]  ant value to be set
  !>   @param[in]  sizeBytes Data size in bytes
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  interface cudaMemset
    function cudaMemset_(dst,myValue,sizeBytes) bind(c, &
&name="cudaMemset")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemset_
      type(c_ptr),value :: dst
      integer(c_int),value :: myValue
      integer(c_size_t),value :: sizeBytes
    end function

  end interface
  !>   @brief Fills the first sizeBytes bytes of the memory area pointed to by dest with the ant
  !>  byte value value.
  !> 
  !>   @param[out] dst Data ptr to be filled
  !>   @param[in]  ant value to be set
  !>   @param[in]  number of values to be set
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  interface cuMemsetD8
    function cuMemsetD8_(dest,myValue,count) bind(c, name="cuMemsetD8")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemsetD8_
      type(c_ptr),value :: dest
      character(c_char),value :: myValue
      integer(c_size_t),value :: count
    end function

  end interface
  !>   @brief Fills the first sizeBytes bytes of the memory area pointed to by dest with the ant
  !>  byte value value.
  !> 
  !>  hipMemsetD8Async() is asynchronous with respect to the host, so the call may return before the
  !>  memset is complete. The operation can optionally be associated to a stream by passing a non-zero
  !>  stream argument. If stream is non-zero, the operation may overlap with operations in other
  !>  streams.
  !> 
  !>   @param[out] dst Data ptr to be filled
  !>   @param[in]  ant value to be set
  !>   @param[in]  number of values to be set
  !>   @param[in]  stream - Stream identifier
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  interface cuMemsetD8Async
    function cuMemsetD8Async_(dest,myValue,count,stream) bind(c, &
&name="cuMemsetD8Async")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemsetD8Async_
      type(c_ptr),value :: dest
      character(c_char),value :: myValue
      integer(c_size_t),value :: count
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Fills the first sizeBytes bytes of the memory area pointed to by dest with the ant
  !>  short value value.
  !> 
  !>   @param[out] dst Data ptr to be filled
  !>   @param[in]  ant value to be set
  !>   @param[in]  number of values to be set
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  interface cuMemsetD16
    function cuMemsetD16_(dest,myValue,count) bind(c, &
&name="cuMemsetD16")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemsetD16_
      type(c_ptr),value :: dest
      integer(c_short),value :: myValue
      integer(c_size_t),value :: count
    end function

  end interface
  !>   @brief Fills the first sizeBytes bytes of the memory area pointed to by dest with the ant
  !>  short value value.
  !> 
  !>  hipMemsetD16Async() is asynchronous with respect to the host, so the call may return before the
  !>  memset is complete. The operation can optionally be associated to a stream by passing a non-zero
  !>  stream argument. If stream is non-zero, the operation may overlap with operations in other
  !>  streams.
  !> 
  !>   @param[out] dst Data ptr to be filled
  !>   @param[in]  ant value to be set
  !>   @param[in]  number of values to be set
  !>   @param[in]  stream - Stream identifier
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  interface cuMemsetD16Async
    function cuMemsetD16Async_(dest,myValue,count,stream) bind(c, &
&name="cuMemsetD16Async")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemsetD16Async_
      type(c_ptr),value :: dest
      integer(c_short),value :: myValue
      integer(c_size_t),value :: count
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Fills the memory area pointed to by dest with the ant integer
  !>  value for specified number of times.
  !> 
  !>   @param[out] dst Data being filled
  !>   @param[in]  ant value to be set
  !>   @param[in]  number of values to be set
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  interface cuMemsetD32
    function cuMemsetD32_(dest,myValue,count) bind(c, &
&name="cuMemsetD32")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemsetD32_
      type(c_ptr),value :: dest
      integer(c_int),value :: myValue
      integer(c_size_t),value :: count
    end function

  end interface
  !>   @brief Fills the first sizeBytes bytes of the memory area pointed to by dev with the ant
  !>  byte value value.
  !> 
  !>   hipMemsetAsync() is asynchronous with respect to the host, so the call may return before the
  !>  memset is complete. The operation can optionally be associated to a stream by passing a non-zero
  !>  stream argument. If stream is non-zero, the operation may overlap with operations in other
  !>  streams.
  !> 
  !>   @param[out] dst Pointer to device memory
  !>   @param[in]  value - Value to set for each byte of specified memory
  !>   @param[in]  sizeBytes - Size in bytes to set
  !>   @param[in]  stream - Stream identifier
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryFree
  interface cudaMemsetAsync
    function cudaMemsetAsync_(dst,myValue,sizeBytes,stream) bind(c, &
&name="cudaMemsetAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemsetAsync_
      type(c_ptr),value :: dst
      integer(c_int),value :: myValue
      integer(c_size_t),value :: sizeBytes
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Fills the memory area pointed to by dev with the ant integer
  !>  value for specified number of times.
  !> 
  !>   hipMemsetD32Async() is asynchronous with respect to the host, so the call may return before the
  !>  memset is complete. The operation can optionally be associated to a stream by passing a non-zero
  !>  stream argument. If stream is non-zero, the operation may overlap with operations in other
  !>  streams.
  !> 
  !>   @param[out] dst Pointer to device memory
  !>   @param[in]  value - Value to set for each byte of specified memory
  !>   @param[in]  count - number of values to be set
  !>   @param[in]  stream - Stream identifier
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryFree
  interface cuMemsetD32Async
    function cuMemsetD32Async_(dst,myValue,count,stream) bind(c, &
&name="cuMemsetD32Async")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemsetD32Async_
      type(c_ptr),value :: dst
      integer(c_int),value :: myValue
      integer(c_size_t),value :: count
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Fills the memory area pointed to by dst with the ant value.
  !> 
  !>   @param[out] dst Pointer to device memory
  !>   @param[in]  pitch - data size in bytes
  !>   @param[in]  value - ant value to be set
  !>   @param[in]  width
  !>   @param[in]  height
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryFree
  interface cudaMemset2D
    function cudaMemset2D_(dst,pitch,myValue,width,height) bind(c, &
&name="cudaMemset2D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemset2D_
      type(c_ptr),value :: dst
      integer(c_size_t),value :: pitch
      integer(c_int),value :: myValue
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
    end function

  end interface
  !>   @brief Fills asynchronously the memory area pointed to by dst with the ant value.
  !> 
  !>   @param[in]  dst Pointer to device memory
  !>   @param[in]  pitch - data size in bytes
  !>   @param[in]  value - ant value to be set
  !>   @param[in]  width
  !>   @param[in]  height
  !>   @param[in]  stream
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryFree
  interface cudaMemset2DAsync
    function cudaMemset2DAsync_(dst,pitch,myValue,width,height,stream) &
&bind(c, name="cudaMemset2DAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemset2DAsync_
      type(c_ptr),value :: dst
      integer(c_size_t),value :: pitch
      integer(c_int),value :: myValue
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Fills synchronously the memory area pointed to by pitchedDevPtr with the ant value.
  !> 
  !>   @param[in] pitchedDevPtr
  !>   @param[in]  value - ant value to be set
  !>   @param[in]  extent
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryFree
  interface cudaMemset3D
    function cudaMemset3D_(pitchedDevPtr,myValue,extent) bind(c, &
&name="cudaMemset3D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemset3D_
      type(c_ptr),value :: pitchedDevPtr
      integer(c_int),value :: myValue
      type(c_ptr),value :: extent
    end function

  end interface
  !>   @brief Fills asynchronously the memory area pointed to by pitchedDevPtr with the ant value.
  !> 
  !>   @param[in] pitchedDevPtr
  !>   @param[in]  value - ant value to be set
  !>   @param[in]  extent
  !>   @param[in]  stream
  !>   @return #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryFree
  interface cudaMemset3DAsync
    function cudaMemset3DAsync_(pitchedDevPtr,myValue,extent,stream) &
&bind(c, name="cudaMemset3DAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemset3DAsync_
      type(c_ptr),value :: pitchedDevPtr
      integer(c_int),value :: myValue
      type(c_ptr),value :: extent
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Query memory info.
  !>  Return snapshot of free memory, and total allocatable memory on the device.
  !> 
  !>  Returns in *free a snapshot of the current free memory.
  !>  @returns #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue
  !>  @warning On HCC, the free memory only accounts for memory allocated by this process and may be
  !> optimistic.
  interface cudaMemGetInfo
    function cudaMemGetInfo_(free,total) bind(c, name="cudaMemGetInfo")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemGetInfo_
      integer(c_size_t) :: free
      integer(c_size_t) :: total
    end function

  end interface
  
  interface cudaMemPtrGetInfo
    function cudaMemPtrGetInfo_(ptr,mySize) bind(c, &
&name="cudaMemPtrGetInfo")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemPtrGetInfo_
      type(c_ptr),value :: ptr
      integer(c_size_t) :: mySize
    end function

  end interface
  !>   @brief Allocate an array on the device.
  !> 
  !>   @param[out]  array  Pointer to allocated array in device memory
  !>   @param[in]   desc   Requested channel format
  !>   @param[in]   width  Requested array allocation width
  !>   @param[in]   height Requested array allocation height
  !>   @param[in]   flags  Requested properties of allocated array
  !>   @return      #hipSuccess, #hipErrorOutOfMemory
  !> 
  !>   @see hipMalloc, hipMallocPitch, hipFree, hipFreeArray, hipHostMalloc, hipHostFree
  interface cudaMallocArray
    function cudaMallocArray_(array,desc,width,height,flags) bind(c, &
&name="cudaMallocArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMallocArray_
      type(c_ptr) :: array
      type(c_ptr) :: desc
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
      integer(c_int),value :: flags
    end function

  end interface
  
  interface cuArrayCreate
    function cuArrayCreate_(pHandle,pAllocateArray) bind(c, &
&name="cuArrayCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuArrayCreate_
      type(c_ptr) :: pHandle
      type(c_ptr) :: pAllocateArray
    end function

  end interface
  
  interface cuArrayDestroy
    function cuArrayDestroy_(array) bind(c, name="cuArrayDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuArrayDestroy_
      type(c_ptr) :: array
    end function

  end interface
  
  interface cuArray3DCreate
    function cuArray3DCreate_(array,pAllocateArray) bind(c, &
&name="cuArray3DCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuArray3DCreate_
      type(c_ptr) :: array
      type(c_ptr) :: pAllocateArray
    end function

  end interface
  
  interface cudaMalloc3D
    function cudaMalloc3D_(pitchedDevPtr,extent) bind(c, &
&name="cudaMalloc3D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMalloc3D_
      type(c_ptr) :: pitchedDevPtr
      type(c_ptr),value :: extent
    end function

  end interface
  !>   @brief Frees an array on the device.
  !> 
  !>   @param[in]  array  Pointer to array to free
  !>   @return     #hipSuccess, #hipErrorInvalidValue, #hipErrorNotInitialized
  !> 
  !>   @see hipMalloc, hipMallocPitch, hipFree, hipMallocArray, hipHostMalloc, hipHostFree
  interface cudaFreeArray
    function cudaFreeArray_(array) bind(c, name="cudaFreeArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFreeArray_
      type(c_ptr) :: array
    end function

  end interface
  !>  @brief Frees a mipmapped array on the device
  !> 
  !>  @param[in] mipmappedArray - Pointer to mipmapped array to free
  !> 
  !>  @return #hipSuccess, #hipErrorInvalidValue
  interface cudaFreeMipmappedArray
    function cudaFreeMipmappedArray_(mipmappedArray) bind(c, &
&name="cudaFreeMipmappedArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFreeMipmappedArray_
      type(c_ptr),value :: mipmappedArray
    end function

  end interface
  !>   @brief Allocate an array on the device.
  !> 
  !>   @param[out]  array  Pointer to allocated array in device memory
  !>   @param[in]   desc   Requested channel format
  !>   @param[in]   extent Requested array allocation width, height and depth
  !>   @param[in]   flags  Requested properties of allocated array
  !>   @return      #hipSuccess, #hipErrorOutOfMemory
  !> 
  !>   @see hipMalloc, hipMallocPitch, hipFree, hipFreeArray, hipHostMalloc, hipHostFree
  interface cudaMalloc3DArray
    function cudaMalloc3DArray_(array,desc,extent,flags) bind(c, &
&name="cudaMalloc3DArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMalloc3DArray_
      type(c_ptr) :: array
      type(c_ptr) :: desc
      type(c_ptr),value :: extent
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Allocate a mipmapped array on the device
  !> 
  !>  @param[out] mipmappedArray  - Pointer to allocated mipmapped array in device memory
  !>  @param[in]  desc            - Requested channel format
  !>  @param[in]  extent          - Requested allocation size (width field in elements)
  !>  @param[in]  numLevels       - Number of mipmap levels to allocate
  !>  @param[in]  flags           - Flags for extensions
  !> 
  !>  @return #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryAllocation
  interface cudaMallocMipmappedArray
    function cudaMallocMipmappedArray_(mipmappedArray,desc,extent,&
&numLevels,flags) bind(c, name="cudaMallocMipmappedArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMallocMipmappedArray_
      type(c_ptr) :: mipmappedArray
      type(c_ptr) :: desc
      type(c_ptr),value :: extent
      integer(c_int),value :: numLevels
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Gets a mipmap level of a HIP mipmapped array
  !> 
  !>  @param[out] levelArray     - Returned mipmap level HIP array
  !>  @param[in]  mipmappedArray - HIP mipmapped array
  !>  @param[in]  level          - Mipmap level
  !> 
  !>  @return #hipSuccess, #hipErrorInvalidValue
  interface cudaGetMipmappedArrayLevel
    function cudaGetMipmappedArrayLevel_(levelArray,mipmappedArray,&
&level) bind(c, name="cudaGetMipmappedArrayLevel")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetMipmappedArrayLevel_
      type(c_ptr) :: levelArray
      type(c_ptr),value :: mipmappedArray
      integer(c_int),value :: level
    end function

  end interface
  !>   @brief Copies memory for 2D arrays.
  !>   @param[in]   pCopy Parameters for the memory copy
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>   #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2D, hipMemcpyToArray, hipMemcpy2DToArray, hipMemcpyFromArray,
  !>  hipMemcpyToSymbol, hipMemcpyAsync
  interface cuMemcpy2D
    function cuMemcpy2D_(pCopy) bind(c, name="cuMemcpy2D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpy2D_
      type(c_ptr) :: pCopy
    end function

  end interface
  !>   @brief Copies memory for 2D arrays.
  !>   @param[in]   pCopy Parameters for the memory copy
  !>   @param[in]   stream Stream to use
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2D, hipMemcpyToArray, hipMemcpy2DToArray, hipMemcpyFromArray,
  !>  hipMemcpyToSymbol, hipMemcpyAsync
  interface cuMemcpy2DAsync
    function cuMemcpy2DAsync_(pCopy,stream) bind(c, &
&name="cuMemcpy2DAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpy2DAsync_
      type(c_ptr) :: pCopy
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Copies data between host and device.
  !> 
  !>   @param[in]   dst     Destination memory address
  !>   @param[in]   wOffset Destination starting X offset
  !>   @param[in]   hOffset Destination starting Y offset
  !>   @param[in]   src     Source memory address
  !>   @param[in]   spitch  Pitch of source memory
  !>   @param[in]   width   Width of matrix transfer (columns in bytes)
  !>   @param[in]   height  Height of matrix transfer (rows)
  !>   @param[in]   kind    Type of transfer
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpyToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cudaMemcpy2DToArray
    function cudaMemcpy2DToArray_(dst,wOffset,hOffset,src,spitch,width,&
&height,myKind) bind(c, name="cudaMemcpy2DToArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpy2DToArray_
      type(c_ptr) :: dst
      integer(c_size_t),value :: wOffset
      integer(c_size_t),value :: hOffset
      type(c_ptr),value :: src
      integer(c_size_t),value :: spitch
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>   @brief Copies data between host and device.
  !> 
  !>   @param[in]   dst     Destination memory address
  !>   @param[in]   wOffset Destination starting X offset
  !>   @param[in]   hOffset Destination starting Y offset
  !>   @param[in]   src     Source memory address
  !>   @param[in]   spitch  Pitch of source memory
  !>   @param[in]   width   Width of matrix transfer (columns in bytes)
  !>   @param[in]   height  Height of matrix transfer (rows)
  !>   @param[in]   kind    Type of transfer
  !>   @param[in]   stream    Accelerator view which the copy is being enqueued
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpyToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cudaMemcpy2DToArrayAsync
    function cudaMemcpy2DToArrayAsync_(dst,wOffset,hOffset,src,spitch,&
&width,height,myKind,stream) bind(c, name="cudaMemcpy2DToArrayAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpy2DToArrayAsync_
      type(c_ptr) :: dst
      integer(c_size_t),value :: wOffset
      integer(c_size_t),value :: hOffset
      type(c_ptr),value :: src
      integer(c_size_t),value :: spitch
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
      type(c_ptr),value :: stream
    end function

  end interface
  
  interface cudaMemcpyToArray
    function cudaMemcpyToArray_(dst,wOffset,hOffset,src,count,myKind) &
&bind(c, name="cudaMemcpyToArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyToArray_
      type(c_ptr) :: dst
      integer(c_size_t),value :: wOffset
      integer(c_size_t),value :: hOffset
      type(c_ptr),value :: src
      integer(c_size_t),value :: count
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  
  interface cudaMemcpyFromArray
    function cudaMemcpyFromArray_(dst,srcArray,wOffset,hOffset,count,&
&myKind) bind(c, name="cudaMemcpyFromArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyFromArray_
      type(c_ptr),value :: dst
      type(c_ptr),value :: srcArray
      integer(c_size_t),value :: wOffset
      integer(c_size_t),value :: hOffset
      integer(c_size_t),value :: count
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>   @brief Copies data between host and device.
  !> 
  !>   @param[in]   dst       Destination memory address
  !>   @param[in]   dpitch    Pitch of destination memory
  !>   @param[in]   src       Source memory address
  !>   @param[in]   wOffset   Source starting X offset
  !>   @param[in]   hOffset   Source starting Y offset
  !>   @param[in]   width     Width of matrix transfer (columns in bytes)
  !>   @param[in]   height    Height of matrix transfer (rows)
  !>   @param[in]   kind      Type of transfer
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cudaMemcpy2DFromArray
    function cudaMemcpy2DFromArray_(dst,dpitch,src,wOffset,hOffset,&
&width,height,myKind) bind(c, name="cudaMemcpy2DFromArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpy2DFromArray_
      type(c_ptr),value :: dst
      integer(c_size_t),value :: dpitch
      type(c_ptr),value :: src
      integer(c_size_t),value :: wOffset
      integer(c_size_t),value :: hOffset
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>   @brief Copies data between host and device asynchronously.
  !> 
  !>   @param[in]   dst       Destination memory address
  !>   @param[in]   dpitch    Pitch of destination memory
  !>   @param[in]   src       Source memory address
  !>   @param[in]   wOffset   Source starting X offset
  !>   @param[in]   hOffset   Source starting Y offset
  !>   @param[in]   width     Width of matrix transfer (columns in bytes)
  !>   @param[in]   height    Height of matrix transfer (rows)
  !>   @param[in]   kind      Type of transfer
  !>   @param[in]   stream    Accelerator view which the copy is being enqueued
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cudaMemcpy2DFromArrayAsync
    function cudaMemcpy2DFromArrayAsync_(dst,dpitch,src,wOffset,&
&hOffset,width,height,myKind,stream) bind(c, &
&name="cudaMemcpy2DFromArrayAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpy2DFromArrayAsync_
      type(c_ptr),value :: dst
      integer(c_size_t),value :: dpitch
      type(c_ptr),value :: src
      integer(c_size_t),value :: wOffset
      integer(c_size_t),value :: hOffset
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Copies data between host and device.
  !> 
  !>   @param[in]   dst       Destination memory address
  !>   @param[in]   srcArray  Source array
  !>   @param[in]   srcoffset Offset in bytes of source array
  !>   @param[in]   count     Size of memory copy in bytes
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cuMemcpyAtoH
    function cuMemcpyAtoH_(dst,srcArray,srcOffset,count) bind(c, &
&name="cuMemcpyAtoH")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyAtoH_
      type(c_ptr),value :: dst
      type(c_ptr) :: srcArray
      integer(c_size_t),value :: srcOffset
      integer(c_size_t),value :: count
    end function

  end interface
  !>   @brief Copies data between host and device.
  !> 
  !>   @param[in]   dstArray   Destination memory address
  !>   @param[in]   dstOffset  Offset in bytes of destination array
  !>   @param[in]   srcHost    Source host pointer
  !>   @param[in]   count      Size of memory copy in bytes
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cuMemcpyHtoA
    function cuMemcpyHtoA_(dstArray,dstOffset,srcHost,count) bind(c, &
&name="cuMemcpyHtoA")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpyHtoA_
      type(c_ptr) :: dstArray
      integer(c_size_t),value :: dstOffset
      type(c_ptr),value :: srcHost
      integer(c_size_t),value :: count
    end function

  end interface
  !>   @brief Copies data between host and device.
  !> 
  !>   @param[in]   p   3D memory copy parameters
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cudaMemcpy3D
    function cudaMemcpy3D_(p) bind(c, name="cudaMemcpy3D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpy3D_
      type(c_ptr) :: p
    end function

  end interface
  !>   @brief Copies data between host and device asynchronously.
  !> 
  !>   @param[in]   p        3D memory copy parameters
  !>   @param[in]   stream   Stream to use
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>  #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cudaMemcpy3DAsync
    function cudaMemcpy3DAsync_(p,stream) bind(c, &
&name="cudaMemcpy3DAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpy3DAsync_
      type(c_ptr) :: p
      type(c_ptr),value :: stream
    end function

  end interface
  !>   @brief Copies data between host and device.
  !> 
  !>   @param[in]   pCopy   3D memory copy parameters
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>   #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cuMemcpy3D
    function cuMemcpy3D_(pCopy) bind(c, name="cuMemcpy3D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpy3D_
      type(c_ptr) :: pCopy
    end function

  end interface
  !>   @brief Copies data between host and device asynchronously.
  !> 
  !>   @param[in]   pCopy    3D memory copy parameters
  !>   @param[in]   stream   Stream to use
  !>   @return      #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidPitchValue,
  !>   #hipErrorInvalidDevicePointer, #hipErrorInvalidMemcpyDirection
  !> 
  !>   @see hipMemcpy, hipMemcpy2DToArray, hipMemcpy2D, hipMemcpyFromArray, hipMemcpyToSymbol,
  !>  hipMemcpyAsync
  interface cuMemcpy3DAsync
    function cuMemcpy3DAsync_(pCopy,stream) bind(c, &
&name="cuMemcpy3DAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpy3DAsync_
      type(c_ptr) :: pCopy
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Determine if a device can access a peer's memory.
  !> 
  !>  @param [out] canAccessPeer Returns the peer access capability (0 or 1)
  !>  @param [in] device - device from where memory may be accessed.
  !>  @param [in] peerDevice - device where memory is physically located
  !> 
  !>  Returns "1" in @p canAccessPeer if the specified @p device is capable
  !>  of directly accessing memory physically located on peerDevice , or "0" if not.
  !> 
  !>  Returns "0" in @p canAccessPeer if deviceId == peerDeviceId, and both are valid devices : a
  !>  device is not a peer of itself.
  !> 
  !>  @returns #hipSuccess,
  !>  @returns #hipErrorInvalidDevice if deviceId or peerDeviceId are not valid devices
  interface cudaDeviceCanAccessPeer
    function cudaDeviceCanAccessPeer_(canAccessPeer,deviceId,&
&peerDeviceId) bind(c, name="cudaDeviceCanAccessPeer")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceCanAccessPeer_
      type(c_ptr),value :: canAccessPeer
      integer(c_int),value :: deviceId
      integer(c_int),value :: peerDeviceId
    end function

  end interface
  !>  @brief Enable direct access from current device's virtual address space to memory allocations
  !>  physically located on a peer device.
  !> 
  !>  Memory which already allocated on peer device will be mapped into the address space of the
  !>  current device.  In addition, all future memory allocations on peerDeviceId will be mapped into
  !>  the address space of the current device when the memory is allocated. The peer memory remains
  !>  accessible from the current device until a call to hipDeviceDisablePeerAccess or hipDeviceReset.
  !> 
  !> 
  !>  @param [in] peerDeviceId
  !>  @param [in] flags
  !> 
  !>  Returns #hipSuccess, #hipErrorInvalidDevice, #hipErrorInvalidValue,
  !>  @returns #hipErrorPeerAccessAlreadyEnabled if peer access is already enabled for this device.
  interface cudaDeviceEnablePeerAccess
    function cudaDeviceEnablePeerAccess_(peerDeviceId,flags) bind(c, &
&name="cudaDeviceEnablePeerAccess")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceEnablePeerAccess_
      integer(c_int),value :: peerDeviceId
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Disable direct access from current device's virtual address space to memory allocations
  !>  physically located on a peer device.
  !> 
  !>  Returns hipErrorPeerAccessNotEnabled if direct access to memory on peerDevice has not yet been
  !>  enabled from the current device.
  !> 
  !>  @param [in] peerDeviceId
  !> 
  !>  @returns #hipSuccess, #hipErrorPeerAccessNotEnabled
  interface cudaDeviceDisablePeerAccess
    function cudaDeviceDisablePeerAccess_(peerDeviceId) bind(c, &
&name="cudaDeviceDisablePeerAccess")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDeviceDisablePeerAccess_
      integer(c_int),value :: peerDeviceId
    end function

  end interface
  !>  @brief Get information on memory allocations.
  !> 
  !>  @param [out] pbase - BAse pointer address
  !>  @param [out] psize - Size of allocation
  !>  @param [in]  dptr- Device Pointer
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidDevicePointer
  !> 
  !>  @see hipCtxCreate, hipCtxDestroy, hipCtxGetFlags, hipCtxPopCurrent, hipCtxGetCurrent,
  !>  hipCtxSetCurrent, hipCtxPushCurrent, hipCtxSetCacheConfig, hipCtxSynchronize, hipCtxGetDevice
  interface cuMemGetAddressRange
    function cuMemGetAddressRange_(pbase,psize,dptr) bind(c, &
&name="cuMemGetAddressRange")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemGetAddressRange_
      type(c_ptr) :: pbase
      integer(c_size_t) :: psize
      type(c_ptr),value :: dptr
    end function

  end interface
  !>  @brief Copies memory from one device to memory on another device.
  !> 
  !>  @param [out] dst - Destination device pointer.
  !>  @param [in] dstDeviceId - Destination device
  !>  @param [in] src - Source device pointer
  !>  @param [in] srcDeviceId - Source device
  !>  @param [in] sizeBytes - Size of memory copy in bytes
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidDevice
  interface cudaMemcpyPeer
    function cudaMemcpyPeer_(dst,dstDeviceId,src,srcDeviceId,&
&sizeBytes) bind(c, name="cudaMemcpyPeer")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyPeer_
      type(c_ptr),value :: dst
      integer(c_int),value :: dstDeviceId
      type(c_ptr),value :: src
      integer(c_int),value :: srcDeviceId
      integer(c_size_t),value :: sizeBytes
    end function

  end interface
  !>  @brief Copies memory from one device to memory on another device.
  !> 
  !>  @param [out] dst - Destination device pointer.
  !>  @param [in] dstDevice - Destination device
  !>  @param [in] src - Source device pointer
  !>  @param [in] srcDevice - Source device
  !>  @param [in] sizeBytes - Size of memory copy in bytes
  !>  @param [in] stream - Stream identifier
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidDevice
  interface cudaMemcpyPeerAsync
    function cudaMemcpyPeerAsync_(dst,dstDeviceId,src,srcDevice,&
&sizeBytes,stream) bind(c, name="cudaMemcpyPeerAsync")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaMemcpyPeerAsync_
      type(c_ptr),value :: dst
      integer(c_int),value :: dstDeviceId
      type(c_ptr),value :: src
      integer(c_int),value :: srcDevice
      integer(c_size_t),value :: sizeBytes
      type(c_ptr),value :: stream
    end function

  end interface
  
  interface cuCtxCreate
    function cuCtxCreate_(ctx,flags,device) bind(c, name="cuCtxCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxCreate_
      type(c_ptr) :: ctx
      integer(c_int),value :: flags
      integer(c_int),value :: device
    end function

  end interface
  
  interface cuCtxDestroy
    function cuCtxDestroy_(ctx) bind(c, name="cuCtxDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxDestroy_
      type(c_ptr),value :: ctx
    end function

  end interface
  
  interface cuCtxPopCurrent
    function cuCtxPopCurrent_(ctx) bind(c, name="cuCtxPopCurrent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxPopCurrent_
      type(c_ptr) :: ctx
    end function

  end interface
  
  interface cuCtxPushCurrent
    function cuCtxPushCurrent_(ctx) bind(c, name="cuCtxPushCurrent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxPushCurrent_
      type(c_ptr),value :: ctx
    end function

  end interface
  
  interface cuCtxSetCurrent
    function cuCtxSetCurrent_(ctx) bind(c, name="cuCtxSetCurrent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxSetCurrent_
      type(c_ptr),value :: ctx
    end function

  end interface
  
  interface cuCtxGetCurrent
    function cuCtxGetCurrent_(ctx) bind(c, name="cuCtxGetCurrent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxGetCurrent_
      type(c_ptr) :: ctx
    end function

  end interface
  
  interface cuCtxGetDevice
    function cuCtxGetDevice_(device) bind(c, name="cuCtxGetDevice")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxGetDevice_
      integer(c_int) :: device
    end function

  end interface
  
  interface cuCtxGetApiVersion
    function cuCtxGetApiVersion_(ctx,apiVersion) bind(c, &
&name="cuCtxGetApiVersion")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxGetApiVersion_
      type(c_ptr),value :: ctx
      type(c_ptr),value :: apiVersion
    end function

  end interface
  
  interface cuCtxGetCacheConfig
    function cuCtxGetCacheConfig_(cacheConfig) bind(c, &
&name="cuCtxGetCacheConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxGetCacheConfig_
      type(c_ptr),value :: cacheConfig
    end function

  end interface
  
  interface cuCtxSetCacheConfig
    function cuCtxSetCacheConfig_(cacheConfig) bind(c, &
&name="cuCtxSetCacheConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxSetCacheConfig_
      integer(kind(cudaFuncCachePreferNone)),value :: cacheConfig
    end function

  end interface
  
  interface cuCtxSetSharedMemConfig
    function cuCtxSetSharedMemConfig_(config) bind(c, &
&name="cuCtxSetSharedMemConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxSetSharedMemConfig_
      integer(kind(cudaSharedMemBankSizeDefault)),value :: config
    end function

  end interface
  
  interface cuCtxGetSharedMemConfig
    function cuCtxGetSharedMemConfig_(pConfig) bind(c, &
&name="cuCtxGetSharedMemConfig")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxGetSharedMemConfig_
      type(c_ptr),value :: pConfig
    end function

  end interface
  
  interface cuCtxSynchronize
    function cuCtxSynchronize_() bind(c, name="cuCtxSynchronize")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxSynchronize_
    end function

  end interface
  
  interface cuCtxGetFlags
    function cuCtxGetFlags_(flags) bind(c, name="cuCtxGetFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxGetFlags_
      type(c_ptr),value :: flags
    end function

  end interface
  
  interface cuCtxEnablePeerAccess
    function cuCtxEnablePeerAccess_(peerCtx,flags) bind(c, &
&name="cuCtxEnablePeerAccess")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxEnablePeerAccess_
      type(c_ptr),value :: peerCtx
      integer(c_int),value :: flags
    end function

  end interface
  
  interface cuCtxDisablePeerAccess
    function cuCtxDisablePeerAccess_(peerCtx) bind(c, &
&name="cuCtxDisablePeerAccess")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuCtxDisablePeerAccess_
      type(c_ptr),value :: peerCtx
    end function

  end interface
  !>  @brief Get the state of the primary context.
  !> 
  !>  @param [in] Device to get primary context flags for
  !>  @param [out] Pointer to store flags
  !>  @param [out] Pointer to store context state; 0 = inactive, 1 = active
  !> 
  !>  @returns #hipSuccess
  !> 
  !>  @see hipCtxCreate, hipCtxDestroy, hipCtxGetFlags, hipCtxPopCurrent, hipCtxGetCurrent,
  !>  hipCtxSetCurrent, hipCtxPushCurrent, hipCtxSetCacheConfig, hipCtxSynchronize, hipCtxGetDevice
  interface cuDevicePrimaryCtxGetState
    function cuDevicePrimaryCtxGetState_(dev,flags,active) bind(c, &
&name="cuDevicePrimaryCtxGetState")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDevicePrimaryCtxGetState_
      integer(c_int),value :: dev
      type(c_ptr),value :: flags
      type(c_ptr),value :: active
    end function

  end interface
  !>  @brief Release the primary context on the GPU.
  !> 
  !>  @param [in] Device which primary context is released
  !> 
  !>  @returns #hipSuccess
  !> 
  !>  @see hipCtxCreate, hipCtxDestroy, hipCtxGetFlags, hipCtxPopCurrent, hipCtxGetCurrent,
  !>  hipCtxSetCurrent, hipCtxPushCurrent, hipCtxSetCacheConfig, hipCtxSynchronize, hipCtxGetDevice
  !>  @warning This function return #hipSuccess though doesn't release the primaryCtx by design on
  !>  HIP/HCC path.
  interface cuDevicePrimaryCtxRelease
    function cuDevicePrimaryCtxRelease_(dev) bind(c, &
&name="cuDevicePrimaryCtxRelease")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDevicePrimaryCtxRelease_
      integer(c_int),value :: dev
    end function

  end interface
  !>  @brief Retain the primary context on the GPU.
  !> 
  !>  @param [out] Returned context handle of the new context
  !>  @param [in] Device which primary context is released
  !> 
  !>  @returns #hipSuccess
  !> 
  !>  @see hipCtxCreate, hipCtxDestroy, hipCtxGetFlags, hipCtxPopCurrent, hipCtxGetCurrent,
  !>  hipCtxSetCurrent, hipCtxPushCurrent, hipCtxSetCacheConfig, hipCtxSynchronize, hipCtxGetDevice
  interface cuDevicePrimaryCtxRetain
    function cuDevicePrimaryCtxRetain_(pctx,dev) bind(c, &
&name="cuDevicePrimaryCtxRetain")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDevicePrimaryCtxRetain_
      type(c_ptr) :: pctx
      integer(c_int),value :: dev
    end function

  end interface
  !>  @brief Resets the primary context on the GPU.
  !> 
  !>  @param [in] Device which primary context is reset
  !> 
  !>  @returns #hipSuccess
  !> 
  !>  @see hipCtxCreate, hipCtxDestroy, hipCtxGetFlags, hipCtxPopCurrent, hipCtxGetCurrent,
  !>  hipCtxSetCurrent, hipCtxPushCurrent, hipCtxSetCacheConfig, hipCtxSynchronize, hipCtxGetDevice
  interface cuDevicePrimaryCtxReset
    function cuDevicePrimaryCtxReset_(dev) bind(c, &
&name="cuDevicePrimaryCtxReset")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDevicePrimaryCtxReset_
      integer(c_int),value :: dev
    end function

  end interface
  !>  @brief Set flags for the primary context.
  !> 
  !>  @param [in] Device for which the primary context flags are set
  !>  @param [in] New flags for the device
  !> 
  !>  @returns #hipSuccess, #hipErrorContextAlreadyInUse
  !> 
  !>  @see hipCtxCreate, hipCtxDestroy, hipCtxGetFlags, hipCtxPopCurrent, hipCtxGetCurrent,
  !>  hipCtxSetCurrent, hipCtxPushCurrent, hipCtxSetCacheConfig, hipCtxSynchronize, hipCtxGetDevice
  interface cuDevicePrimaryCtxSetFlags
    function cuDevicePrimaryCtxSetFlags_(dev,flags) bind(c, &
&name="cuDevicePrimaryCtxSetFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuDevicePrimaryCtxSetFlags_
      integer(c_int),value :: dev
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Loads code object from file into a hipModule_t
  !> 
  !>  @param [in] fname
  !>  @param [out] module
  !> 
  !>  @returns hipSuccess, hipErrorInvalidValue, hipErrorInvalidContext, hipErrorFileNotFound,
  !>  hipErrorOutOfMemory, hipErrorSharedObjectInitFailed, hipErrorNotInitialized
  !> 
  !>
  interface cuModuleLoad
    function cuModuleLoad_(myModule,fname) bind(c, name="cuModuleLoad")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuModuleLoad_
      type(c_ptr) :: myModule
      type(c_ptr),value :: fname
    end function

  end interface
  !>  @brief Frees the module
  !> 
  !>  @param [in] module
  !> 
  !>  @returns hipSuccess, hipInvalidValue
  !>  module is freed and the code objects associated with it are destroyed
  !>
  interface cuModuleUnload
    function cuModuleUnload_(myModule) bind(c, name="cuModuleUnload")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuModuleUnload_
      type(c_ptr),value :: myModule
    end function

  end interface
  !>  @brief Function with kname will be extracted if present in module
  !> 
  !>  @param [in] module
  !>  @param [in] kname
  !>  @param [out] function
  !> 
  !>  @returns hipSuccess, hipErrorInvalidValue, hipErrorInvalidContext, hipErrorNotInitialized,
  !>  hipErrorNotFound,
  interface cuModuleGetFunction
    function cuModuleGetFunction_(myFunction,myModule,kname) bind(c, &
&name="cuModuleGetFunction")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuModuleGetFunction_
      type(c_ptr) :: myFunction
      type(c_ptr),value :: myModule
      type(c_ptr),value :: kname
    end function

  end interface
  !>  @brief Find out attributes for a given function.
  !> 
  !>  @param [out] attr
  !>  @param [in] func
  !> 
  !>  @returns hipSuccess, hipErrorInvalidValue, hipErrorInvalidDeviceFunction
  interface cudaFuncGetAttributes
    function cudaFuncGetAttributes_(attr,func) bind(c, &
&name="cudaFuncGetAttributes")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFuncGetAttributes_
      type(c_ptr) :: attr
      type(c_ptr),value :: func
    end function

  end interface
  !>  @brief Find out a specific attribute for a given function.
  !> 
  !>  @param [out] value
  !>  @param [in]  attrib
  !>  @param [in]  hfunc
  !> 
  !>  @returns hipSuccess, hipErrorInvalidValue, hipErrorInvalidDeviceFunction
  interface cudaFuncGetAttribute
    function cudaFuncGetAttribute_(myValue,attrib,hfunc) bind(c, &
&name="cudaFuncGetAttribute")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaFuncGetAttribute_
      type(c_ptr),value :: myValue
      integer(kind(CU_FUNC_ATTRIBUTE_MAX_THREADS_PER_BLOCK)),value :: &
&attrib
      type(c_ptr),value :: hfunc
    end function

  end interface
  !>  @brief returns the handle of the texture reference with the name from the module.
  !> 
  !>  @param [in] hmod
  !>  @param [in] name
  !>  @param [out] texRef
  !> 
  !>  @returns hipSuccess, hipErrorNotInitialized, hipErrorNotFound, hipErrorInvalidValue
  interface cuModuleGetTexRef
    function cuModuleGetTexRef_(texRef,hmod,name) bind(c, &
&name="cuModuleGetTexRef")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuModuleGetTexRef_
      type(c_ptr) :: texRef
      type(c_ptr),value :: hmod
      type(c_ptr),value :: name
    end function

  end interface
  !>  @brief builds module from code object which resides in host memory. Image is pointer to that
  !>  location.
  !> 
  !>  @param [in] image
  !>  @param [out] module
  !> 
  !>  @returns hipSuccess, hipErrorNotInitialized, hipErrorOutOfMemory, hipErrorNotInitialized
  interface cuModuleLoadData
    function cuModuleLoadData_(myModule,image) bind(c, &
&name="cuModuleLoadData")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuModuleLoadData_
      type(c_ptr) :: myModule
      type(c_ptr),value :: image
    end function

  end interface
  !>  @brief builds module from code object which resides in host memory. Image is pointer to that
  !>  location. Options are not used. hipModuleLoadData is called.
  !> 
  !>  @param [in] image
  !>  @param [out] module
  !>  @param [in] number of options
  !>  @param [in] options for JIT
  !>  @param [in] option values for JIT
  !> 
  !>  @returns hipSuccess, hipErrorNotInitialized, hipErrorOutOfMemory, hipErrorNotInitialized
  interface cuModuleLoadDataEx
    function cuModuleLoadDataEx_(myModule,image,numOptions,options,&
&optionValues) bind(c, name="cuModuleLoadDataEx")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuModuleLoadDataEx_
      type(c_ptr) :: myModule
      type(c_ptr),value :: image
      integer(c_int),value :: numOptions
      type(c_ptr),value :: options
      type(c_ptr) :: optionValues
    end function

  end interface
  !>  @brief launches kernel f with launch parameters and shared memory on stream with arguments passed
  !>  to kernelparams or extra
  !> 
  !>  @param [in] f         Kernel to launch.
  !>  @param [in] gridDimX  X grid dimension specified as multiple of blockDimX.
  !>  @param [in] gridDimY  Y grid dimension specified as multiple of blockDimY.
  !>  @param [in] gridDimZ  Z grid dimension specified as multiple of blockDimZ.
  !>  @param [in] blockDimX X block dimensions specified in work-items
  !>  @param [in] blockDimY Y grid dimension specified in work-items
  !>  @param [in] blockDimZ Z grid dimension specified in work-items
  !>  @param [in] sharedMemBytes Amount of dynamic shared memory to allocate for this kernel. The
  !>  HIP-Clang compiler provides support for extern shared declarations.
  !>  @param [in] stream    Stream where the kernel should be dispatched.  May be 0, in which case th
  !>  default stream is used with associated synchronization rules.
  !>  @param [in] kernelParams
  !>  @param [in] extra     Pointer to kernel arguments.   These are passed directly to the kernel and
  !>  must be in the memory layout and alignment expected by the kernel.
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue
  !> 
  !>  @warning kernellParams argument is not yet implemented in HIP. Please use extra instead. Please
  !>  refer to hip_porting_driver_api.md for sample usage.
  interface cuLaunchKernel
    function cuLaunchKernel_(f,gridDimX,gridDimY,gridDimZ,blockDimX,&
&blockDimY,blockDimZ,sharedMemBytes,stream,kernelParams,extra) bind(c, &
&name="cuLaunchKernel")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuLaunchKernel_
      type(c_ptr),value :: f
      integer(c_int),value :: gridDimX
      integer(c_int),value :: gridDimY
      integer(c_int),value :: gridDimZ
      integer(c_int),value :: blockDimX
      integer(c_int),value :: blockDimY
      integer(c_int),value :: blockDimZ
      integer(c_int),value :: sharedMemBytes
      type(c_ptr),value :: stream
      type(c_ptr) :: kernelParams
      type(c_ptr) :: extra
    end function

  end interface
  !>  @brief launches kernel f with launch parameters and shared memory on stream with arguments passed
  !>  to kernelparams or extra, where thread blocks can cooperate and synchronize as they execute
  !> 
  !>  @param [in] f         Kernel to launch.
  !>  @param [in] gridDim   Grid dimensions specified as multiple of blockDim.
  !>  @param [in] blockDim  Block dimensions specified in work-items
  !>  @param [in] kernelParams A list of kernel arguments
  !>  @param [in] sharedMemBytes Amount of dynamic shared memory to allocate for this kernel. The
  !>  HIP-Clang compiler provides support for extern shared declarations.
  !>  @param [in] stream    Stream where the kernel should be dispatched.  May be 0, in which case th
  !>  default stream is used with associated synchronization rules.
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue, hipErrorCooperativeLaunchTooLarge
  interface cudaLaunchCooperativeKernel
    function cudaLaunchCooperativeKernel_(f,gridDim,blockDimX,&
&kernelParams,sharedMemBytes,stream) bind(c, &
&name="cudaLaunchCooperativeKernel")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaLaunchCooperativeKernel_
      type(c_ptr),value :: f
      type(dim3),value :: gridDim
      type(dim3),value :: blockDimX
      type(c_ptr) :: kernelParams
      integer(c_int),value :: sharedMemBytes
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Launches kernels on multiple devices where thread blocks can cooperate and
  !>  synchronize as they execute.
  !> 
  !>  @param [in] launchParamsList         List of launch parameters, one per device.
  !>  @param [in] numDevices               Size of the launchParamsList array.
  !>  @param [in] flags                    Flags to control launch behavior.
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue, hipErrorCooperativeLaunchTooLarge
  interface cudaLaunchCooperativeKernelMultiDevice
    function cudaLaunchCooperativeKernelMultiDevice_(launchParamsList,&
&numDevices,flags) bind(c, &
&name="cudaLaunchCooperativeKernelMultiDevice")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaLaunchCooperativeKernelMultiDevice_
      type(c_ptr),value :: launchParamsList
      integer(c_int),value :: numDevices
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Launches kernels on multiple devices and guarantees all specified kernels are dispatched
  !>  on respective streams before enqueuing any other work on the specified streams from any other threads
  !> 
  !> 
  !>  @param [in] hipLaunchParams          List of launch parameters, one per device.
  !>  @param [in] numDevices               Size of the launchParamsList array.
  !>  @param [in] flags                    Flags to control launch behavior.
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue
  interface cudaExtLaunchMultiKernelMultiDevice
    function cudaExtLaunchMultiKernelMultiDevice_(launchParamsList,&
&numDevices,flags) bind(c, name="cudaExtLaunchMultiKernelMultiDevice")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaExtLaunchMultiKernelMultiDevice_
      type(c_ptr),value :: launchParamsList
      integer(c_int),value :: numDevices
      integer(c_int),value :: flags
    end function

  end interface
  
  interface cuOccupancyMaxPotentialBlockSize
    function cuOccupancyMaxPotentialBlockSize_(gridSize,blockSize,f,&
&dynSharedMemPerBlk,blockSizeLimit) bind(c, &
&name="cuOccupancyMaxPotentialBlockSize")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuOccupancyMaxPotentialBlockSize_
      type(c_ptr),value :: gridSize
      type(c_ptr),value :: blockSize
      type(c_ptr),value :: f
      integer(c_size_t),value :: dynSharedMemPerBlk
      integer(c_int),value :: blockSizeLimit
    end function

  end interface
  
  interface cuOccupancyMaxPotentialBlockSizeWithFlags
    function cuOccupancyMaxPotentialBlockSizeWithFlags_(gridSize,&
&blockSize,f,dynSharedMemPerBlk,blockSizeLimit,flags) bind(c, &
&name="cuOccupancyMaxPotentialBlockSizeWithFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cuOccupancyMaxPotentialBlockSizeWithFlags_
      type(c_ptr),value :: gridSize
      type(c_ptr),value :: blockSize
      type(c_ptr),value :: f
      integer(c_size_t),value :: dynSharedMemPerBlk
      integer(c_int),value :: blockSizeLimit
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Returns occupancy for a device function.
  !> 
  !>  @param [out] numBlocks        Returned occupancy
  !>  @param [in]  func             Kernel function (hipFunction) for which occupancy is calulated
  !>  @param [in]  blockSize        Block size the kernel is intended to be launched with
  !>  @param [in]  dynSharedMemPerBlk dynamic shared memory usage (in bytes) intended for each block
  interface cuOccupancyMaxActiveBlocksPerMultiprocessor
    function cuOccupancyMaxActiveBlocksPerMultiprocessor_(numBlocks,f,&
&blockSize,dynSharedMemPerBlk) bind(c, &
&name="cuOccupancyMaxActiveBlocksPerMultiprocessor")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cuOccupancyMaxActiveBlocksPerMultiprocessor_
      type(c_ptr),value :: numBlocks
      type(c_ptr),value :: f
      integer(c_int),value :: blockSize
      integer(c_size_t),value :: dynSharedMemPerBlk
    end function

  end interface
  !>  @brief Returns occupancy for a device function.
  !> 
  !>  @param [out] numBlocks        Returned occupancy
  !>  @param [in]  func             Kernel function for which occupancy is calulated
  !>  @param [in]  blockSize        Block size the kernel is intended to be launched with
  !>  @param [in]  dynSharedMemPerBlk dynamic shared memory usage (in bytes) intended for each block
  interface cudaOccupancyMaxActiveBlocksPerMultiprocessor
    function cudaOccupancyMaxActiveBlocksPerMultiprocessor_(numBlocks,&
&f,blockSize,dynSharedMemPerBlk) bind(c, &
&name="cudaOccupancyMaxActiveBlocksPerMultiprocessor")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaOccupancyMaxActiveBlocksPerMultiprocessor_
      type(c_ptr),value :: numBlocks
      type(c_ptr),value :: f
      integer(c_int),value :: blockSize
      integer(c_size_t),value :: dynSharedMemPerBlk
    end function

  end interface
  !>  @brief Returns occupancy for a device function.
  !> 
  !>  @param [out] numBlocks        Returned occupancy
  !>  @param [in]  f                Kernel function for which occupancy is calulated
  !>  @param [in]  blockSize        Block size the kernel is intended to be launched with
  !>  @param [in]  dynSharedMemPerBlk dynamic shared memory usage (in bytes) intended for each block
  !>  @param [in]  flags            Extra flags for occupancy calculation (currently ignored)
  interface cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags
    function &
&cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags_(numBlocks,f,&
&blockSize,dynSharedMemPerBlk,flags) bind(c, &
&name="cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags_
      type(c_ptr),value :: numBlocks
      type(c_ptr),value :: f
      integer(c_int),value :: blockSize
      integer(c_size_t),value :: dynSharedMemPerBlk
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief determine the grid and block sizes to achieves maximum occupancy for a kernel
  !> 
  !>  @param [out] gridSize           minimum grid size for maximum potential occupancy
  !>  @param [out] blockSize          block size for maximum potential occupancy
  !>  @param [in]  f                  kernel function for which occupancy is calulated
  !>  @param [in]  dynSharedMemPerBlk dynamic shared memory usage (in bytes) intended for each block
  !>  @param [in]  blockSizeLimit     the maximum block size for the kernel, use 0 for no limit
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorInvalidValue
  interface cudaOccupancyMaxPotentialBlockSize
    function cudaOccupancyMaxPotentialBlockSize_(gridSize,blockSize,f,&
&dynSharedMemPerBlk,blockSizeLimit) bind(c, &
&name="cudaOccupancyMaxPotentialBlockSize")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaOccupancyMaxPotentialBlockSize_
      type(c_ptr),value :: gridSize
      type(c_ptr),value :: blockSize
      type(c_ptr),value :: f
      integer(c_size_t),value :: dynSharedMemPerBlk
      integer(c_int),value :: blockSizeLimit
    end function

  end interface
  
  interface cudaProfilerStart
    function cudaProfilerStart_() bind(c, name="cudaProfilerStart")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaProfilerStart_
    end function

  end interface
  
  interface cudaProfilerStop
    function cudaProfilerStop_() bind(c, name="cudaProfilerStop")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaProfilerStop_
    end function

  end interface
  !>  @brief Configure a kernel launch.
  !> 
  !>  @param [in] gridDim   grid dimension specified as multiple of blockDim.
  !>  @param [in] blockDim  block dimensions specified in work-items
  !>  @param [in] sharedMem Amount of dynamic shared memory to allocate for this kernel. The
  !>  HIP-Clang compiler provides support for extern shared declarations.
  !>  @param [in] stream    Stream where the kernel should be dispatched.  May be 0, in which case the
  !>  default stream is used with associated synchronization rules.
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue
  !>
  interface cudaConfigureCall
    function cudaConfigureCall_(gridDim,blockDim,sharedMem,stream) &
&bind(c, name="cudaConfigureCall")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaConfigureCall_
      type(dim3),value :: gridDim
      type(dim3),value :: blockDim
      integer(c_size_t),value :: sharedMem
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Set a kernel argument.
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue
  !> 
  !>  @param [in] arg    Pointer the argument in host memory.
  !>  @param [in] size   Size of the argument.
  !>  @param [in] offset Offset of the argument on the argument stack.
  !>
  interface cudaSetupArgument
    function cudaSetupArgument_(arg,mySize,offset) bind(c, &
&name="cudaSetupArgument")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaSetupArgument_
      type(c_ptr),value :: arg
      integer(c_size_t),value :: mySize
      integer(c_size_t),value :: offset
    end function

  end interface
  !>  @brief Launch a kernel.
  !> 
  !>  @param [in] func Kernel to launch.
  !> 
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue
  !>
  interface cudaLaunchByPtr
    function cudaLaunchByPtr_(func) bind(c, name="cudaLaunchByPtr")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaLaunchByPtr_
      type(c_ptr),value :: func
    end function

  end interface
  !>  @brief C compliant kernel launch API
  !> 
  !>  @param [in] function_address - kernel stub function pointer.
  !>  @param [in] numBlocks - number of blocks
  !>  @param [in] dimBlocks - dimension of a block
  !>  @param [in] args - kernel arguments
  !>  @param [in] sharedMemBytes - Amount of dynamic shared memory to allocate for this kernel. The
  !>  HIP-Clang compiler provides support for extern shared declarations.
  !>  @param [in] stream - Stream where the kernel should be dispatched.  May be 0, in which case th
  !>   default stream is used with associated synchronization rules.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, hipInvalidDevice
  !>
  interface cudaLaunchKernel
    function cudaLaunchKernel_(function_address,numBlocks,dimBlocks,&
&args,sharedMemBytes,stream) bind(c, name="cudaLaunchKernel")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaLaunchKernel_
      type(c_ptr),value :: function_address
      type(dim3),value :: numBlocks
      type(dim3),value :: dimBlocks
      type(c_ptr) :: args
      integer(c_size_t),value :: sharedMemBytes
      type(c_ptr),value :: stream
    end function

  end interface
  !>  Copies memory for 2D arrays.
  !> 
  !>  @param pCopy           - Parameters for the memory copy
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  interface cuMemcpy2DUnaligned
    function cuMemcpy2DUnaligned_(pCopy) bind(c, &
&name="cuMemcpy2DUnaligned")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMemcpy2DUnaligned_
      type(c_ptr) :: pCopy
    end function

  end interface
  !>  @brief Launches kernel from the pointer address, with arguments and shared memory on stream.
  !> 
  !>  @param [in] function_address pointer to the Kernel to launch.
  !>  @param [in] numBlocks number of blocks.
  !>  @param [in] dimBlocks dimension of a block.
  !>  @param [in] args pointer to kernel arguments.
  !>  @param [in] sharedMemBytes  Amount of dynamic shared memory to allocate for this kernel.
  !>  HIP-Clang compiler provides support for extern shared declarations.
  !>  @param [in] stream  Stream where the kernel should be dispatched.
  !>  @param [in] startEvent  If non-null, specified event will be updated to track the start time of
  !>  the kernel launch. The event must be created before calling this API.
  !>  @param [in] stopEvent  If non-null, specified event will be updated to track the stop time of
  !>  the kernel launch. The event must be created before calling this API.
  !>  May be 0, in which case the default stream is used with associated synchronization rules.
  !>  @param [in] flags. The value of hipExtAnyOrderLaunch, signifies if kernel can be
  !>  launched in any order.
  !>  @returns hipSuccess, hipInvalidDevice, hipErrorNotInitialized, hipErrorInvalidValue.
  !>
  interface cudaExtLaunchKernel
    function cudaExtLaunchKernel_(function_address,numBlocks,dimBlocks,&
&args,sharedMemBytes,stream,startEvent,stopEvent,flags) bind(c, &
&name="cudaExtLaunchKernel")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaExtLaunchKernel_
      type(c_ptr),value :: function_address
      type(dim3),value :: numBlocks
      type(dim3),value :: dimBlocks
      type(c_ptr) :: args
      integer(c_size_t),value :: sharedMemBytes
      type(c_ptr),value :: stream
      type(c_ptr),value :: startEvent
      type(c_ptr),value :: stopEvent
      integer(c_int),value :: flags
    end function

  end interface
  
  interface cudaBindTexture
    function cudaBindTexture_(offset,tex,devPtr,desc,mySize) bind(c, &
&name="cudaBindTexture")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaBindTexture_
      integer(c_size_t) :: offset
      type(c_ptr) :: tex
      type(c_ptr),value :: devPtr
      type(c_ptr) :: desc
      integer(c_size_t),value :: mySize
    end function

  end interface
  
  interface cudaBindTexture2D
    function cudaBindTexture2D_(offset,tex,devPtr,desc,width,height,&
&pitch) bind(c, name="cudaBindTexture2D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaBindTexture2D_
      integer(c_size_t) :: offset
      type(c_ptr) :: tex
      type(c_ptr),value :: devPtr
      type(c_ptr) :: desc
      integer(c_size_t),value :: width
      integer(c_size_t),value :: height
      integer(c_size_t),value :: pitch
    end function

  end interface
  
  interface cudaBindTextureToArray
    function cudaBindTextureToArray_(tex,array,desc) bind(c, &
&name="cudaBindTextureToArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaBindTextureToArray_
      type(c_ptr) :: tex
      type(c_ptr),value :: array
      type(c_ptr) :: desc
    end function

  end interface
  
  interface cudaGetTextureAlignmentOffset
    function cudaGetTextureAlignmentOffset_(offset,texref) bind(c, &
&name="cudaGetTextureAlignmentOffset")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetTextureAlignmentOffset_
      integer(c_size_t) :: offset
      type(c_ptr) :: texref
    end function

  end interface
  
  interface cudaUnbindTexture
    function cudaUnbindTexture_(tex) bind(c, name="cudaUnbindTexture")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaUnbindTexture_
      type(c_ptr) :: tex
    end function

  end interface
  
  interface cuTexRefGetAddress
    function cuTexRefGetAddress_(dev_ptr,texRef) bind(c, &
&name="cuTexRefGetAddress")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetAddress_
      type(c_ptr) :: dev_ptr
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetAddressMode
    function cuTexRefGetAddressMode_(pam,texRef,dim) bind(c, &
&name="cuTexRefGetAddressMode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetAddressMode_
      type(c_ptr),value :: pam
      type(c_ptr) :: texRef
      integer(c_int),value :: dim
    end function

  end interface
  
  interface cuTexRefGetFilterMode
    function cuTexRefGetFilterMode_(pfm,texRef) bind(c, &
&name="cuTexRefGetFilterMode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetFilterMode_
      type(c_ptr),value :: pfm
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetFlags
    function cuTexRefGetFlags_(pFlags,texRef) bind(c, &
&name="cuTexRefGetFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetFlags_
      type(c_ptr),value :: pFlags
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetFormat
    function cuTexRefGetFormat_(pFormat,pNumChannels,texRef) bind(c, &
&name="cuTexRefGetFormat")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetFormat_
      type(c_ptr),value :: pFormat
      type(c_ptr),value :: pNumChannels
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetMaxAnisotropy
    function cuTexRefGetMaxAnisotropy_(pmaxAnsio,texRef) bind(c, &
&name="cuTexRefGetMaxAnisotropy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetMaxAnisotropy_
      type(c_ptr),value :: pmaxAnsio
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetMipmapFilterMode
    function cuTexRefGetMipmapFilterMode_(pfm,texRef) bind(c, &
&name="cuTexRefGetMipmapFilterMode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetMipmapFilterMode_
      type(c_ptr),value :: pfm
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetMipmapLevelBias
    function cuTexRefGetMipmapLevelBias_(pbias,texRef) bind(c, &
&name="cuTexRefGetMipmapLevelBias")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetMipmapLevelBias_
      type(c_ptr),value :: pbias
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetMipmapLevelClamp
    function cuTexRefGetMipmapLevelClamp_(pminMipmapLevelClamp,&
&pmaxMipmapLevelClamp,texRef) bind(c, &
&name="cuTexRefGetMipmapLevelClamp")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetMipmapLevelClamp_
      type(c_ptr),value :: pminMipmapLevelClamp
      type(c_ptr),value :: pmaxMipmapLevelClamp
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefGetMipMappedArray
    function cuTexRefGetMipMappedArray_(pArray,texRef) bind(c, &
&name="cuTexRefGetMipMappedArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefGetMipMappedArray_
      type(c_ptr) :: pArray
      type(c_ptr) :: texRef
    end function

  end interface
  
  interface cuTexRefSetAddress
    function cuTexRefSetAddress_(ByteOffset,texRef,dptr,bytes) bind(c, &
&name="cuTexRefSetAddress")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetAddress_
      integer(c_size_t) :: ByteOffset
      type(c_ptr) :: texRef
      type(c_ptr),value :: dptr
      integer(c_size_t),value :: bytes
    end function

  end interface
  
  interface cuTexRefSetAddress2D
    function cuTexRefSetAddress2D_(texRef,desc,dptr,Pitch) bind(c, &
&name="cuTexRefSetAddress2D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetAddress2D_
      type(c_ptr) :: texRef
      type(c_ptr) :: desc
      type(c_ptr),value :: dptr
      integer(c_size_t),value :: Pitch
    end function

  end interface
  
  interface cuTexRefSetMaxAnisotropy
    function cuTexRefSetMaxAnisotropy_(texRef,maxAniso) bind(c, &
&name="cuTexRefSetMaxAnisotropy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetMaxAnisotropy_
      type(c_ptr) :: texRef
      integer(c_int),value :: maxAniso
    end function

  end interface
  !>  @}
  interface cudaBindTextureToMipmappedArray
    function cudaBindTextureToMipmappedArray_(tex,mipmappedArray,desc) &
&bind(c, name="cudaBindTextureToMipmappedArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaBindTextureToMipmappedArray_
      type(c_ptr) :: tex
      type(c_ptr),value :: mipmappedArray
      type(c_ptr) :: desc
    end function

  end interface
  
  interface cudaCreateTextureObject
    function cudaCreateTextureObject_(pTexObject,pResDesc,pTexDesc,&
&pResViewDesc) bind(c, name="cudaCreateTextureObject")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaCreateTextureObject_
      type(c_ptr) :: pTexObject
      type(c_ptr) :: pResDesc
      type(c_ptr) :: pTexDesc
      type(c_ptr),value :: pResViewDesc
    end function

  end interface
  
  interface cudaDestroyTextureObject
    function cudaDestroyTextureObject_(textureObject) bind(c, &
&name="cudaDestroyTextureObject")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaDestroyTextureObject_
      type(c_ptr),value :: textureObject
    end function

  end interface
  
  interface cudaGetChannelDesc
    function cudaGetChannelDesc_(desc,array) bind(c, &
&name="cudaGetChannelDesc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetChannelDesc_
      type(c_ptr) :: desc
      type(c_ptr),value :: array
    end function

  end interface
  
  interface cudaGetTextureObjectResourceDesc
    function cudaGetTextureObjectResourceDesc_(pResDesc,textureObject) &
&bind(c, name="cudaGetTextureObjectResourceDesc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetTextureObjectResourceDesc_
      type(c_ptr) :: pResDesc
      type(c_ptr),value :: textureObject
    end function

  end interface
  
  interface cudaGetTextureObjectResourceViewDesc
    function cudaGetTextureObjectResourceViewDesc_(pResViewDesc,&
&textureObject) bind(c, name="cudaGetTextureObjectResourceViewDesc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaGetTextureObjectResourceViewDesc_
      type(c_ptr),value :: pResViewDesc
      type(c_ptr),value :: textureObject
    end function

  end interface
  
  interface cudaGetTextureObjectTextureDesc
    function cudaGetTextureObjectTextureDesc_(pTexDesc,textureObject) &
&bind(c, name="cudaGetTextureObjectTextureDesc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGetTextureObjectTextureDesc_
      type(c_ptr) :: pTexDesc
      type(c_ptr),value :: textureObject
    end function

  end interface
  
  interface cuTexRefSetAddressMode
    function cuTexRefSetAddressMode_(texRef,dim,am) bind(c, &
&name="cuTexRefSetAddressMode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetAddressMode_
      type(c_ptr) :: texRef
      integer(c_int),value :: dim
      integer(kind(cudaAddressModeWrap)),value :: am
    end function

  end interface
  
  interface cuTexRefSetArray
    function cuTexRefSetArray_(tex,array,flags) bind(c, &
&name="cuTexRefSetArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetArray_
      type(c_ptr) :: tex
      type(c_ptr),value :: array
      integer(c_int),value :: flags
    end function

  end interface
  
  interface cuTexRefSetFilterMode
    function cuTexRefSetFilterMode_(texRef,fm) bind(c, &
&name="cuTexRefSetFilterMode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetFilterMode_
      type(c_ptr) :: texRef
      integer(kind(cudaFilterModePoint)),value :: fm
    end function

  end interface
  
  interface cuTexRefSetFlags
    function cuTexRefSetFlags_(texRef,Flags) bind(c, &
&name="cuTexRefSetFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetFlags_
      type(c_ptr) :: texRef
      integer(c_int),value :: Flags
    end function

  end interface
  
  interface cuTexRefSetFormat
    function cuTexRefSetFormat_(texRef,fmt,NumPackedComponents) bind(c,&
& name="cuTexRefSetFormat")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetFormat_
      type(c_ptr) :: texRef
      integer(kind(CU_AD_FORMAT_UNSIGNED_INT8)),value :: fmt
      integer(c_int),value :: NumPackedComponents
    end function

  end interface
  
  interface cuTexObjectCreate
    function cuTexObjectCreate_(pTexObject,pResDesc,pTexDesc,&
&pResViewDesc) bind(c, name="cuTexObjectCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexObjectCreate_
      type(c_ptr) :: pTexObject
      type(c_ptr),value :: pResDesc
      type(c_ptr),value :: pTexDesc
      type(c_ptr),value :: pResViewDesc
    end function

  end interface
  
  interface cuTexObjectDestroy
    function cuTexObjectDestroy_(texObject) bind(c, &
&name="cuTexObjectDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexObjectDestroy_
      type(c_ptr),value :: texObject
    end function

  end interface
  
  interface cuTexObjectGetResourceDesc
    function cuTexObjectGetResourceDesc_(pResDesc,texObject) bind(c, &
&name="cuTexObjectGetResourceDesc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexObjectGetResourceDesc_
      type(c_ptr),value :: pResDesc
      type(c_ptr),value :: texObject
    end function

  end interface
  
  interface cuTexObjectGetResourceViewDesc
    function cuTexObjectGetResourceViewDesc_(pResViewDesc,texObject) &
&bind(c, name="cuTexObjectGetResourceViewDesc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexObjectGetResourceViewDesc_
      type(c_ptr),value :: pResViewDesc
      type(c_ptr),value :: texObject
    end function

  end interface
  
  interface cuTexObjectGetTextureDesc
    function cuTexObjectGetTextureDesc_(pTexDesc,texObject) bind(c, &
&name="cuTexObjectGetTextureDesc")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexObjectGetTextureDesc_
      type(c_ptr),value :: pTexDesc
      type(c_ptr),value :: texObject
    end function

  end interface
  
  interface cuTexRefSetBorderColor
    function cuTexRefSetBorderColor_(texRef,pBorderColor) bind(c, &
&name="cuTexRefSetBorderColor")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetBorderColor_
      type(c_ptr) :: texRef
      type(c_ptr),value :: pBorderColor
    end function

  end interface
  
  interface cuTexRefSetMipmapFilterMode
    function cuTexRefSetMipmapFilterMode_(texRef,fm) bind(c, &
&name="cuTexRefSetMipmapFilterMode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetMipmapFilterMode_
      type(c_ptr) :: texRef
      integer(kind(cudaFilterModePoint)),value :: fm
    end function

  end interface
  
  interface cuTexRefSetMipmapLevelBias
    function cuTexRefSetMipmapLevelBias_(texRef,bias) bind(c, &
&name="cuTexRefSetMipmapLevelBias")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetMipmapLevelBias_
      type(c_ptr) :: texRef
      real(c_float),value :: bias
    end function

  end interface
  
  interface cuTexRefSetMipmapLevelClamp
    function cuTexRefSetMipmapLevelClamp_(texRef,minMipMapLevelClamp,&
&maxMipMapLevelClamp) bind(c, name="cuTexRefSetMipmapLevelClamp")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetMipmapLevelClamp_
      type(c_ptr) :: texRef
      real(c_float),value :: minMipMapLevelClamp
      real(c_float),value :: maxMipMapLevelClamp
    end function

  end interface
  
  interface cuTexRefSetMipmappedArray
    function cuTexRefSetMipmappedArray_(texRef,mipmappedArray,Flags) &
&bind(c, name="cuTexRefSetMipmappedArray")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuTexRefSetMipmappedArray_
      type(c_ptr) :: texRef
      type(c_ptr) :: mipmappedArray
      integer(c_int),value :: Flags
    end function

  end interface
  
  interface cuMipmappedArrayCreate
    function cuMipmappedArrayCreate_(pHandle,pMipmappedArrayDesc,&
&numMipmapLevels) bind(c, name="cuMipmappedArrayCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMipmappedArrayCreate_
      type(c_ptr) :: pHandle
      type(c_ptr) :: pMipmappedArrayDesc
      integer(c_int),value :: numMipmapLevels
    end function

  end interface
  
  interface cuMipmappedArrayDestroy
    function cuMipmappedArrayDestroy_(hMipmappedArray) bind(c, &
&name="cuMipmappedArrayDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMipmappedArrayDestroy_
      type(c_ptr),value :: hMipmappedArray
    end function

  end interface
  
  interface cuMipmappedArrayGetLevel
    function cuMipmappedArrayGetLevel_(pLevelArray,hMipMappedArray,&
&level) bind(c, name="cuMipmappedArrayGetLevel")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cuMipmappedArrayGetLevel_
      type(c_ptr) :: pLevelArray
      type(c_ptr),value :: hMipMappedArray
      integer(c_int),value :: level
    end function

  end interface
  !> 
  !>   @defgroup Callback Callback Activity APIs
  !>   @{
  !>   This section describes the callback/Activity of HIP runtime API.
  interface cudaRegisterApiCallback
    function cudaRegisterApiCallback_(id,fun,arg) bind(c, &
&name="cudaRegisterApiCallback")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaRegisterApiCallback_
      integer(c_int),value :: id
      type(c_ptr),value :: fun
      type(c_ptr),value :: arg
    end function

  end interface
  
  interface cudaRemoveApiCallback
    function cudaRemoveApiCallback_(id) bind(c, &
&name="cudaRemoveApiCallback")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaRemoveApiCallback_
      integer(c_int),value :: id
    end function

  end interface
  
  interface cudaRegisterActivityCallback
    function cudaRegisterActivityCallback_(id,fun,arg) bind(c, &
&name="cudaRegisterActivityCallback")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaRegisterActivityCallback_
      integer(c_int),value :: id
      type(c_ptr),value :: fun
      type(c_ptr),value :: arg
    end function

  end interface
  
  interface cudaRemoveActivityCallback
    function cudaRemoveActivityCallback_(id) bind(c, &
&name="cudaRemoveActivityCallback")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaRemoveActivityCallback_
      integer(c_int),value :: id
    end function

  end interface
  !>  @brief Begins graph capture on a stream.
  !> 
  !>  @param [in] stream - Stream to initiate capture.
  !>  @param [in] mode - Controls the interaction of this capture sequence with other API calls that
  !>  are not safe.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaStreamBeginCapture
    function cudaStreamBeginCapture_(stream,mode) bind(c, &
&name="cudaStreamBeginCapture")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamBeginCapture_
      type(c_ptr),value :: stream
      integer(kind(cudaStreamCaptureModeGlobal)),value :: mode
    end function

  end interface
  !>  @brief Ends capture on a stream, returning the captured graph.
  !> 
  !>  @param [in] stream - Stream to end capture.
  !>  @param [out] pGraph - returns the graph captured.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaStreamEndCapture
    function cudaStreamEndCapture_(stream,pGraph) bind(c, &
&name="cudaStreamEndCapture")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamEndCapture_
      type(c_ptr),value :: stream
      type(c_ptr) :: pGraph
    end function

  end interface
  !>  @brief Get capture status of a stream.
  !> 
  !>  @param [in] stream - Stream under capture.
  !>  @param [out] pCaptureStatus - returns current status of the capture.
  !>  @param [out] pId - unique ID of the capture.
  !> 
  !>  @returns #hipSuccess, #hipErrorStreamCaptureImplicit
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaStreamGetCaptureInfo
    function cudaStreamGetCaptureInfo_(stream,pCaptureStatus,pId) &
&bind(c, name="cudaStreamGetCaptureInfo")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamGetCaptureInfo_
      type(c_ptr),value :: stream
      type(c_ptr),value :: pCaptureStatus
      type(c_ptr),value :: pId
    end function

  end interface
  !>  @brief Get stream's capture state
  !> 
  !>  @param [in] stream - Stream under capture.
  !>  @param [out] captureStatus_out - returns current status of the capture.
  !>  @param [out] id_out - unique ID of the capture.
  !>  @param [in] graph_out - returns the graph being captured into.
  !>  @param [out] dependencies_out - returns pointer to an array of nodes.
  !>  @param [out] numDependencies_out - returns size of the array returned in dependencies_out.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorStreamCaptureImplicit
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaStreamGetCaptureInfo_v2
    function cudaStreamGetCaptureInfo_v2_(stream,captureStatus_out,&
&id_out,graph_out,dependencies_out,numDependencies_out) bind(c, &
&name="cudaStreamGetCaptureInfo_v2")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamGetCaptureInfo_v2_
      type(c_ptr),value :: stream
      type(c_ptr),value :: captureStatus_out
      type(c_ptr),value :: id_out
      type(c_ptr) :: graph_out
      type(c_ptr) :: dependencies_out
      type(c_ptr),value :: numDependencies_out
    end function

  end interface
  !>  @brief Get stream's capture state
  !> 
  !>  @param [in] stream - Stream under capture.
  !>  @param [out] pCaptureStatus - returns current status of the capture.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorStreamCaptureImplicit
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaStreamIsCapturing
    function cudaStreamIsCapturing_(stream,pCaptureStatus) bind(c, &
&name="cudaStreamIsCapturing")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamIsCapturing_
      type(c_ptr),value :: stream
      type(c_ptr),value :: pCaptureStatus
    end function

  end interface
  !>  @brief Update the set of dependencies in a capturing stream
  !> 
  !>  @param [in] stream - Stream under capture.
  !>  @param [in] dependencies - pointer to an array of nodes to Add/Replace.
  !>  @param [in] numDependencies - size of the array in dependencies.
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorIllegalState
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaStreamUpdateCaptureDependencies
    function cudaStreamUpdateCaptureDependencies_(stream,dependencies,&
&numDependencies,flags) bind(c, &
&name="cudaStreamUpdateCaptureDependencies")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaStreamUpdateCaptureDependencies_
      type(c_ptr),value :: stream
      type(c_ptr) :: dependencies
      integer(c_size_t),value :: numDependencies
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Creates a graph
  !> 
  !>  @param [out] pGraph - pointer to graph to create.
  !>  @param [in] flags - flags for graph creation, must be 0.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryAllocation
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphCreate
    function cudaGraphCreate_(pGraph,flags) bind(c, &
&name="cudaGraphCreate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphCreate_
      type(c_ptr) :: pGraph
      integer(c_int),value :: flags
    end function

  end interface
  !>  @brief Destroys a graph
  !> 
  !>  @param [in] graph - instance of graph to destroy.
  !> 
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphDestroy
    function cudaGraphDestroy_(graph) bind(c, name="cudaGraphDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphDestroy_
      type(c_ptr),value :: graph
    end function

  end interface
  !>  @brief Adds dependency edges to a graph.
  !> 
  !>  @param [in] graph - instance of the graph to add dependencies.
  !>  @param [in] from - pointer to the graph nodes with dependenties to add from.
  !>  @param [in] to - pointer to the graph nodes to add dependenties to.
  !>  @param [in] numDependencies - the number of dependencies to add.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphAddDependencies
    function cudaGraphAddDependencies_(graph,from,to,numDependencies) &
&bind(c, name="cudaGraphAddDependencies")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddDependencies_
      type(c_ptr),value :: graph
      type(c_ptr) :: from
      type(c_ptr) :: to
      integer(c_size_t),value :: numDependencies
    end function

  end interface
  !>  @brief Removes dependency edges from a graph.
  !> 
  !>  @param [in] graph - instance of the graph to remove dependencies.
  !>  @param [in] from - Array of nodes that provide the dependencies.
  !>  @param [in] to - Array of dependent nodes.
  !>  @param [in] numDependencies - the number of dependencies to remove.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphRemoveDependencies
    function cudaGraphRemoveDependencies_(graph,from,to,&
&numDependencies) bind(c, name="cudaGraphRemoveDependencies")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphRemoveDependencies_
      type(c_ptr),value :: graph
      type(c_ptr) :: from
      type(c_ptr) :: to
      integer(c_size_t),value :: numDependencies
    end function

  end interface
  !>  @brief Returns a graph's dependency edges.
  !> 
  !>  @param [in] graph - instance of the graph to get the edges from.
  !>  @param [out] from - pointer to the graph nodes to return edge endpoints.
  !>  @param [out] to - pointer to the graph nodes to return edge endpoints.
  !>  @param [out] numEdges - returns number of edges.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  from and to may both be NULL, in which case this function only returns the number of edges in
  !>  numEdges. Otherwise, numEdges entries will be filled in. If numEdges is higher than the actual
  !>  number of edges, the remaining entries in from and to will be set to NULL, and the number of
  !>  edges actually returned will be written to numEdges
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphGetEdges
    function cudaGraphGetEdges_(graph,from,to,numEdges) bind(c, &
&name="cudaGraphGetEdges")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphGetEdges_
      type(c_ptr),value :: graph
      type(c_ptr) :: from
      type(c_ptr) :: to
      type(c_ptr),value :: numEdges
    end function

  end interface
  !>  @brief Returns graph nodes.
  !> 
  !>  @param [in] graph - instance of graph to get the nodes.
  !>  @param [out] nodes - pointer to return the  graph nodes.
  !>  @param [out] numNodes - returns number of graph nodes.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  nodes may be NULL, in which case this function will return the number of nodes in numNodes.
  !>  Otherwise, numNodes entries will be filled in. If numNodes is higher than the actual number of
  !>  nodes, the remaining entries in nodes will be set to NULL, and the number of nodes actually
  !>  obtained will be returned in numNodes.
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphGetNodes
    function cudaGraphGetNodes_(graph,nodes,numNodes) bind(c, &
&name="cudaGraphGetNodes")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphGetNodes_
      type(c_ptr),value :: graph
      type(c_ptr) :: nodes
      type(c_ptr),value :: numNodes
    end function

  end interface
  !>  @brief Returns graph's root nodes.
  !> 
  !>  @param [in] graph - instance of the graph to get the nodes.
  !>  @param [out] pRootNodes - pointer to return the graph's root nodes.
  !>  @param [out] pNumRootNodes - returns the number of graph's root nodes.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  pRootNodes may be NULL, in which case this function will return the number of root nodes in
  !>  pNumRootNodes. Otherwise, pNumRootNodes entries will be filled in. If pNumRootNodes is higher
  !>  than the actual number of root nodes, the remaining entries in pRootNodes will be set to NULL,
  !>  and the number of nodes actually obtained will be returned in pNumRootNodes.
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphGetRootNodes
    function cudaGraphGetRootNodes_(graph,pRootNodes,pNumRootNodes) &
&bind(c, name="cudaGraphGetRootNodes")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphGetRootNodes_
      type(c_ptr),value :: graph
      type(c_ptr) :: pRootNodes
      type(c_ptr),value :: pNumRootNodes
    end function

  end interface
  !>  @brief Returns a node's dependencies.
  !> 
  !>  @param [in] node - graph node to get the dependencies from.
  !>  @param [out] pDependencies - pointer to to return the dependencies.
  !>  @param [out] pNumDependencies -  returns the number of graph node dependencies.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  pDependencies may be NULL, in which case this function will return the number of dependencies in
  !>  pNumDependencies. Otherwise, pNumDependencies entries will be filled in. If pNumDependencies is
  !>  higher than the actual number of dependencies, the remaining entries in pDependencies will be set
  !>  to NULL, and the number of nodes actually obtained will be returned in pNumDependencies.
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphNodeGetDependencies
    function cudaGraphNodeGetDependencies_(node,pDependencies,&
&pNumDependencies) bind(c, name="cudaGraphNodeGetDependencies")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphNodeGetDependencies_
      type(c_ptr),value :: node
      type(c_ptr) :: pDependencies
      type(c_ptr),value :: pNumDependencies
    end function

  end interface
  !>  @brief Returns a node's dependent nodes.
  !> 
  !>  @param [in] node - graph node to get the Dependent nodes from.
  !>  @param [out] pDependentNodes - pointer to return the graph dependent nodes.
  !>  @param [out] pNumDependentNodes - returns the number of graph node dependent nodes.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  DependentNodes may be NULL, in which case this function will return the number of dependent nodes
  !>  in pNumDependentNodes. Otherwise, pNumDependentNodes entries will be filled in. If
  !>  pNumDependentNodes is higher than the actual number of dependent nodes, the remaining entries in
  !>  pDependentNodes will be set to NULL, and the number of nodes actually obtained will be returned
  !>  in pNumDependentNodes.
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphNodeGetDependentNodes
    function cudaGraphNodeGetDependentNodes_(node,pDependentNodes,&
&pNumDependentNodes) bind(c, name="cudaGraphNodeGetDependentNodes")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphNodeGetDependentNodes_
      type(c_ptr),value :: node
      type(c_ptr) :: pDependentNodes
      type(c_ptr),value :: pNumDependentNodes
    end function

  end interface
  !>  @brief Returns a node's type.
  !> 
  !>  @param [in] node - instance of the graph to add dependencies.
  !>  @param [out] pType - pointer to the return the type
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphNodeGetType
    function cudaGraphNodeGetType_(node,pType) bind(c, &
&name="cudaGraphNodeGetType")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphNodeGetType_
      type(c_ptr),value :: node
      type(c_ptr),value :: pType
    end function

  end interface
  !>  @brief Remove a node from the graph.
  !> 
  !>  @param [in] node - graph node to remove
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphDestroyNode
    function cudaGraphDestroyNode_(node) bind(c, &
&name="cudaGraphDestroyNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphDestroyNode_
      type(c_ptr),value :: node
    end function

  end interface
  !>  @brief Clones a graph.
  !> 
  !>  @param [out] pGraphClone - Returns newly created cloned graph.
  !>  @param [in] originalGraph - original graph to clone from.
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorMemoryAllocation
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphClone
    function cudaGraphClone_(pGraphClone,originalGraph) bind(c, &
&name="cudaGraphClone")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphClone_
      type(c_ptr) :: pGraphClone
      type(c_ptr),value :: originalGraph
    end function

  end interface
  !>  @brief Finds a cloned version of a node.
  !> 
  !>  @param [out] pNode - Returns the cloned node.
  !>  @param [in] originalNode - original node handle.
  !>  @param [in] clonedGraph - Cloned graph to query.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphNodeFindInClone
    function cudaGraphNodeFindInClone_(pNode,originalNode,clonedGraph) &
&bind(c, name="cudaGraphNodeFindInClone")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphNodeFindInClone_
      type(c_ptr) :: pNode
      type(c_ptr),value :: originalNode
      type(c_ptr),value :: clonedGraph
    end function

  end interface
  !>  @brief Creates an executable graph from a graph
  !> 
  !>  @param [out] pGraphExec - pointer to instantiated executable graph that is created.
  !>  @param [in] graph - instance of graph to instantiate.
  !>  @param [out] pErrorNode - pointer to error node in case error occured in graph instantiation,
  !>   it could modify the correponding node.
  !>  @param [out] pLogBuffer - pointer to log buffer.
  !>  @param [out] bufferSize - the size of log buffer.
  !> 
  !>  @returns #hipSuccess, #hipErrorOutOfMemory
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphInstantiate
    function cudaGraphInstantiate_(pGraphExec,graph,pErrorNode,&
&pLogBuffer,bufferSize) bind(c, name="cudaGraphInstantiate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphInstantiate_
      type(c_ptr) :: pGraphExec
      type(c_ptr),value :: graph
      type(c_ptr) :: pErrorNode
      type(c_ptr),value :: pLogBuffer
      integer(c_size_t),value :: bufferSize
    end function

  end interface
  !>  @brief Creates an executable graph from a graph.
  !> 
  !>  @param [out] pGraphExec - pointer to instantiated executable graph that is created.
  !>  @param [in] graph - instance of graph to instantiate.
  !>  @param [in] flags - Flags to control instantiation.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  !>
  interface cudaGraphInstantiateWithFlags
    function cudaGraphInstantiateWithFlags_(pGraphExec,graph,flags) &
&bind(c, name="cudaGraphInstantiateWithFlags")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphInstantiateWithFlags_
      type(c_ptr) :: pGraphExec
      type(c_ptr),value :: graph
      integer(c_long_long),value :: flags
    end function

  end interface
  !>  @brief launches an executable graph in a stream
  !> 
  !>  @param [in] graphExec - instance of executable graph to launch.
  !>  @param [in] stream - instance of stream in which to launch executable graph.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphLaunch
    function cudaGraphLaunch_(graphExec,stream) bind(c, &
&name="cudaGraphLaunch")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphLaunch_
      type(c_ptr),value :: graphExec
      type(c_ptr),value :: stream
    end function

  end interface
  !>  @brief Destroys an executable graph
  !> 
  !>  @param [in] pGraphExec - instance of executable graph to destry.
  !> 
  !>  @returns #hipSuccess.
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecDestroy
    function cudaGraphExecDestroy_(graphExec) bind(c, &
&name="cudaGraphExecDestroy")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecDestroy_
      type(c_ptr),value :: graphExec
    end function

  end interface
  !>  @brief Check whether an executable graph can be updated with a graph and perform the update if  *
  !>  possible.
  !> 
  !>  @param [in] hGraphExec - instance of executable graph to update.
  !>  @param [in] hGraph - graph that contains the updated parameters.
  !>  @param [in] hErrorNode_out -  node which caused the permissibility check to forbid the update.
  !>  @param [in] updateResult_out - Whether the graph update was permitted.
  !>  @returns #hipSuccess, #hipErrorGraphExecUpdateFailure
  !> 
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecUpdate
    function cudaGraphExecUpdate_(hGraphExec,hGraph,hErrorNode_out,&
&updateResult_out) bind(c, name="cudaGraphExecUpdate")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecUpdate_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: hGraph
      type(c_ptr) :: hErrorNode_out
      type(c_ptr),value :: updateResult_out
    end function

  end interface
  !>  @brief Creates a kernel execution node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to graph node to create.
  !>  @param [in] graph - instance of graph to add the created node.
  !>  @param [in] pDependencies - pointer to the dependencies on the kernel execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] pNodeParams - pointer to the parameters to the kernel execution node on the GPU.
  !>  @returns #hipSuccess, #hipErrorInvalidValue, #hipErrorInvalidDeviceFunction
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddKernelNode
    function cudaGraphAddKernelNode_(pGraphNode,graph,pDependencies,&
&numDependencies,pNodeParams) bind(c, name="cudaGraphAddKernelNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddKernelNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Gets kernel node's parameters.
  !> 
  !>  @param [in] node - instance of the node to get parameters from.
  !>  @param [out] pNodeParams - pointer to the parameters
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphKernelNodeGetParams
    function cudaGraphKernelNodeGetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphKernelNodeGetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphKernelNodeGetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets a kernel node's parameters.
  !> 
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams -  pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphKernelNodeSetParams
    function cudaGraphKernelNodeSetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphKernelNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphKernelNodeSetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets the parameters for a kernel node in the given graphExec.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams -  pointer to the kernel node parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecKernelNodeSetParams
    function cudaGraphExecKernelNodeSetParams_(hGraphExec,node,&
&pNodeParams) bind(c, name="cudaGraphExecKernelNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecKernelNodeSetParams_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Creates a memcpy node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to graph node to create.
  !>  @param [in] graph - instance of graph to add the created node.
  !>  @param [in] pDependencies -  pointer to the dependencies on the memcpy execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] pCopyParams -  pointer to the parameters for the memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddMemcpyNode
    function cudaGraphAddMemcpyNode_(pGraphNode,graph,pDependencies,&
&numDependencies,pCopyParams) bind(c, name="cudaGraphAddMemcpyNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddMemcpyNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr) :: pCopyParams
    end function

  end interface
  !>  @brief Gets a memcpy node's parameters.
  !> 
  !>  @param [in] node - instance of the node to get parameters from.
  !>  @param [out] pNodeParams - pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphMemcpyNodeGetParams
    function cudaGraphMemcpyNodeGetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphMemcpyNodeGetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphMemcpyNodeGetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets a memcpy node's parameters.
  !> 
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams -  pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphMemcpyNodeSetParams
    function cudaGraphMemcpyNodeSetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphMemcpyNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphMemcpyNodeSetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets the parameters for a memcpy node in the given graphExec.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams -  pointer to the kernel node parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecMemcpyNodeSetParams
    function cudaGraphExecMemcpyNodeSetParams_(hGraphExec,node,&
&pNodeParams) bind(c, name="cudaGraphExecMemcpyNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecMemcpyNodeSetParams_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Creates a 1D memcpy node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to graph node to create.
  !>  @param [in] graph - instance of graph to add the created node.
  !>  @param [in] pDependencies -  pointer to the dependencies on the memcpy execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] dst - pointer to memory address to the destination.
  !>  @param [in] src - pointer to memory address to the source.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddMemcpyNode1D
    function cudaGraphAddMemcpyNode1D_(pGraphNode,graph,pDependencies,&
&numDependencies,dst,src,count,myKind) bind(c, &
&name="cudaGraphAddMemcpyNode1D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddMemcpyNode1D_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: count
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Sets a memcpy node's parameters to perform a 1-dimensional copy.
  !> 
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] dst - pointer to memory address to the destination.
  !>  @param [in] src - pointer to memory address to the source.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphMemcpyNodeSetParams1D
    function cudaGraphMemcpyNodeSetParams1D_(node,dst,src,count,&
&myKind) bind(c, name="cudaGraphMemcpyNodeSetParams1D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphMemcpyNodeSetParams1D_
      type(c_ptr),value :: node
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: count
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Sets the parameters for a memcpy node in the given graphExec to perform a 1-dimensional
  !>  copy.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] dst - pointer to memory address to the destination.
  !>  @param [in] src - pointer to memory address to the source.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecMemcpyNodeSetParams1D
    function cudaGraphExecMemcpyNodeSetParams1D_(hGraphExec,node,dst,&
&src,count,myKind) bind(c, name="cudaGraphExecMemcpyNodeSetParams1D")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecMemcpyNodeSetParams1D_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr),value :: dst
      type(c_ptr),value :: src
      integer(c_size_t),value :: count
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Creates a memcpy node to copy from a symbol on the device and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to graph node to create.
  !>  @param [in] graph - instance of graph to add the created node.
  !>  @param [in] pDependencies -  pointer to the dependencies on the memcpy execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] dst - pointer to memory address to the destination.
  !>  @param [in] symbol - Device symbol address.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] offset - Offset from start of symbol in bytes.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddMemcpyNodeFromSymbol
    function cudaGraphAddMemcpyNodeFromSymbol_(pGraphNode,graph,&
&pDependencies,numDependencies,dst,symbol,count,offset,myKind) bind(c, &
&name="cudaGraphAddMemcpyNodeFromSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddMemcpyNodeFromSymbol_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr),value :: dst
      type(c_ptr),value :: symbol
      integer(c_size_t),value :: count
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Sets a memcpy node's parameters to copy from a symbol on the device.
  !> 
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] dst - pointer to memory address to the destination.
  !>  @param [in] symbol - Device symbol address.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] offset - Offset from start of symbol in bytes.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphMemcpyNodeSetParamsFromSymbol
    function cudaGraphMemcpyNodeSetParamsFromSymbol_(node,dst,symbol,&
&count,offset,myKind) bind(c, &
&name="cudaGraphMemcpyNodeSetParamsFromSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaGraphMemcpyNodeSetParamsFromSymbol_
      type(c_ptr),value :: node
      type(c_ptr),value :: dst
      type(c_ptr),value :: symbol
      integer(c_size_t),value :: count
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Sets the parameters for a memcpy node in the given graphExec to copy from a symbol on the
  !>  * device.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] dst - pointer to memory address to the destination.
  !>  @param [in] symbol - Device symbol address.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] offset - Offset from start of symbol in bytes.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecMemcpyNodeSetParamsFromSymbol
    function cudaGraphExecMemcpyNodeSetParamsFromSymbol_(hGraphExec,&
&node,dst,symbol,count,offset,myKind) bind(c, &
&name="cudaGraphExecMemcpyNodeSetParamsFromSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaGraphExecMemcpyNodeSetParamsFromSymbol_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr),value :: dst
      type(c_ptr),value :: symbol
      integer(c_size_t),value :: count
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Creates a memcpy node to copy to a symbol on the device and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to graph node to create.
  !>  @param [in] graph - instance of graph to add the created node.
  !>  @param [in] pDependencies -  pointer to the dependencies on the memcpy execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] symbol - Device symbol address.
  !>  @param [in] src - pointer to memory address of the src.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] offset - Offset from start of symbol in bytes.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddMemcpyNodeToSymbol
    function cudaGraphAddMemcpyNodeToSymbol_(pGraphNode,graph,&
&pDependencies,numDependencies,symbol,src,count,offset,myKind) bind(c, &
&name="cudaGraphAddMemcpyNodeToSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddMemcpyNodeToSymbol_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr),value :: symbol
      type(c_ptr),value :: src
      integer(c_size_t),value :: count
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Sets a memcpy node's parameters to copy to a symbol on the device.
  !> 
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] symbol - Device symbol address.
  !>  @param [in] src - pointer to memory address of the src.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] offset - Offset from start of symbol in bytes.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphMemcpyNodeSetParamsToSymbol
    function cudaGraphMemcpyNodeSetParamsToSymbol_(node,symbol,src,&
&count,offset,myKind) bind(c, &
&name="cudaGraphMemcpyNodeSetParamsToSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaGraphMemcpyNodeSetParamsToSymbol_
      type(c_ptr),value :: node
      type(c_ptr),value :: symbol
      type(c_ptr),value :: src
      integer(c_size_t),value :: count
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Sets the parameters for a memcpy node in the given graphExec to copy to a symbol on the
  !>  device.
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] symbol - Device symbol address.
  !>  @param [in] src - pointer to memory address of the src.
  !>  @param [in] count - the size of the memory to copy.
  !>  @param [in] offset - Offset from start of symbol in bytes.
  !>  @param [in] kind - the type of memory copy.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecMemcpyNodeSetParamsToSymbol
    function cudaGraphExecMemcpyNodeSetParamsToSymbol_(hGraphExec,node,&
&symbol,src,count,offset,myKind) bind(c, &
&name="cudaGraphExecMemcpyNodeSetParamsToSymbol")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaGraphExecMemcpyNodeSetParamsToSymbol_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr),value :: symbol
      type(c_ptr),value :: src
      integer(c_size_t),value :: count
      integer(c_size_t),value :: offset
      integer(kind(cudaMemcpyHostToHost)),value :: myKind
    end function

  end interface
  !>  @brief Creates a memset node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to the graph node to create.
  !>  @param [in] graph - instance of the graph to add the created node.
  !>  @param [in] pDependencies -  pointer to the dependencies on the memset execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] pMemsetParams -  pointer to the parameters for the memory set.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddMemsetNode
    function cudaGraphAddMemsetNode_(pGraphNode,graph,pDependencies,&
&numDependencies,pMemsetParams) bind(c, name="cudaGraphAddMemsetNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddMemsetNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr) :: pMemsetParams
    end function

  end interface
  !>  @brief Gets a memset node's parameters.
  !> 
  !>  @param [in] node - instane of the node to get parameters from.
  !>  @param [out] pNodeParams - pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphMemsetNodeGetParams
    function cudaGraphMemsetNodeGetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphMemsetNodeGetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphMemsetNodeGetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets a memset node's parameters.
  !> 
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams - pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphMemsetNodeSetParams
    function cudaGraphMemsetNodeSetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphMemsetNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphMemsetNodeSetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets the parameters for a memset node in the given graphExec.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams - pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecMemsetNodeSetParams
    function cudaGraphExecMemsetNodeSetParams_(hGraphExec,node,&
&pNodeParams) bind(c, name="cudaGraphExecMemsetNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecMemsetNodeSetParams_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Creates a host execution node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to the graph node to create.
  !>  @param [in] graph - instance of the graph to add the created node.
  !>  @param [in] pDependencies -  pointer to the dependencies on the memset execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] pNodeParams -pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddHostNode
    function cudaGraphAddHostNode_(pGraphNode,graph,pDependencies,&
&numDependencies,pNodeParams) bind(c, name="cudaGraphAddHostNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddHostNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Returns a host node's parameters.
  !> 
  !>  @param [in] node - instane of the node to get parameters from.
  !>  @param [out] pNodeParams - pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphHostNodeGetParams
    function cudaGraphHostNodeGetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphHostNodeGetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphHostNodeGetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets a host node's parameters.
  !> 
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams - pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphHostNodeSetParams
    function cudaGraphHostNodeSetParams_(node,pNodeParams) bind(c, &
&name="cudaGraphHostNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphHostNodeSetParams_
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Sets the parameters for a host node in the given graphExec.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - instance of the node to set parameters to.
  !>  @param [in] pNodeParams - pointer to the parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecHostNodeSetParams
    function cudaGraphExecHostNodeSetParams_(hGraphExec,node,&
&pNodeParams) bind(c, name="cudaGraphExecHostNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecHostNodeSetParams_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr) :: pNodeParams
    end function

  end interface
  !>  @brief Creates a child graph node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to the graph node to create.
  !>  @param [in] graph - instance of the graph to add the created node.
  !>  @param [in] pDependencies -  pointer to the dependencies on the memset execution node.
  !>  @param [in] numDependencies - the number of the dependencies.
  !>  @param [in] childGraph - the graph to clone into this node
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddChildGraphNode
    function cudaGraphAddChildGraphNode_(pGraphNode,graph,&
&pDependencies,numDependencies,childGraph) bind(c, &
&name="cudaGraphAddChildGraphNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddChildGraphNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr),value :: childGraph
    end function

  end interface
  !>  @brief Gets a handle to the embedded graph of a child graph node.
  !> 
  !>  @param [in] node - instane of the node to get child graph.
  !>  @param [out] pGraph - pointer to get the graph.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphChildGraphNodeGetGraph
    function cudaGraphChildGraphNodeGetGraph_(node,pGraph) bind(c, &
&name="cudaGraphChildGraphNodeGetGraph")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphChildGraphNodeGetGraph_
      type(c_ptr),value :: node
      type(c_ptr) :: pGraph
    end function

  end interface
  !>  @brief Updates node parameters in the child graph node in the given graphExec.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] node - node from the graph which was used to instantiate graphExec.
  !>  @param [in] childGraph - child graph with updated parameters.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecChildGraphNodeSetParams
    function cudaGraphExecChildGraphNodeSetParams_(hGraphExec,node,&
&childGraph) bind(c, name="cudaGraphExecChildGraphNodeSetParams")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaGraphExecChildGraphNodeSetParams_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: node
      type(c_ptr),value :: childGraph
    end function

  end interface
  !>  @brief Creates an empty node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to the graph node to create and add to the graph.
  !>  @param [in] graph - instane of the graph the node is add to.
  !>  @param [in] pDependencies -  pointer to the node dependenties.
  !>  @param [in] numDependencies - the number of dependencies.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddEmptyNode
    function cudaGraphAddEmptyNode_(pGraphNode,graph,pDependencies,&
&numDependencies) bind(c, name="cudaGraphAddEmptyNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddEmptyNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
    end function

  end interface
  !>  @brief Creates an event record node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to the graph node to create and add to the graph.
  !>  @param [in] graph - instane of the graph the node to be added.
  !>  @param [in] pDependencies -  pointer to the node dependenties.
  !>  @param [in] numDependencies - the number of dependencies.
  !>  @param [in] event - Event for the node.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddEventRecordNode
    function cudaGraphAddEventRecordNode_(pGraphNode,graph,&
&pDependencies,numDependencies,event) bind(c, &
&name="cudaGraphAddEventRecordNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddEventRecordNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr),value :: event
    end function

  end interface
  !>  @brief Returns the event associated with an event record node.
  !> 
  !>  @param [in] node -  instane of the node to get event from.
  !>  @param [out] event_out - Pointer to return the event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphEventRecordNodeGetEvent
    function cudaGraphEventRecordNodeGetEvent_(node,event_out) bind(c, &
&name="cudaGraphEventRecordNodeGetEvent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphEventRecordNodeGetEvent_
      type(c_ptr),value :: node
      type(c_ptr) :: event_out
    end function

  end interface
  !>  @brief Sets an event record node's event.
  !> 
  !>  @param [in] node - instane of the node to set event to.
  !>  @param [in] event - pointer to the event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphEventRecordNodeSetEvent
    function cudaGraphEventRecordNodeSetEvent_(node,event) bind(c, &
&name="cudaGraphEventRecordNodeSetEvent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphEventRecordNodeSetEvent_
      type(c_ptr),value :: node
      type(c_ptr),value :: event
    end function

  end interface
  !>  @brief Sets the event for an event record node in the given graphExec.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] hNode - node from the graph which was used to instantiate graphExec.
  !>  @param [in] event - pointer to the event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecEventRecordNodeSetEvent
    function cudaGraphExecEventRecordNodeSetEvent_(hGraphExec,hNode,&
&event) bind(c, name="cudaGraphExecEventRecordNodeSetEvent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: &
&cudaGraphExecEventRecordNodeSetEvent_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: hNode
      type(c_ptr),value :: event
    end function

  end interface
  !>  @brief Creates an event wait node and adds it to a graph.
  !> 
  !>  @param [out] pGraphNode - pointer to the graph node to create and add to the graph.
  !>  @param [in] graph - instane of the graph the node to be added.
  !>  @param [in] pDependencies -  pointer to the node dependenties.
  !>  @param [in] numDependencies - the number of dependencies.
  !>  @param [in] event - Event for the node.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphAddEventWaitNode
    function cudaGraphAddEventWaitNode_(pGraphNode,graph,pDependencies,&
&numDependencies,event) bind(c, name="cudaGraphAddEventWaitNode")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphAddEventWaitNode_
      type(c_ptr) :: pGraphNode
      type(c_ptr),value :: graph
      type(c_ptr) :: pDependencies
      integer(c_size_t),value :: numDependencies
      type(c_ptr),value :: event
    end function

  end interface
  !>  @brief Returns the event associated with an event wait node.
  !> 
  !>  @param [in] node -  instane of the node to get event from.
  !>  @param [out] event_out - Pointer to return the event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphEventWaitNodeGetEvent
    function cudaGraphEventWaitNodeGetEvent_(node,event_out) bind(c, &
&name="cudaGraphEventWaitNodeGetEvent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphEventWaitNodeGetEvent_
      type(c_ptr),value :: node
      type(c_ptr) :: event_out
    end function

  end interface
  !>  @brief Sets an event wait node's event.
  !> 
  !>  @param [in] node - instane of the node to set event to.
  !>  @param [in] event - pointer to the event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphEventWaitNodeSetEvent
    function cudaGraphEventWaitNodeSetEvent_(node,event) bind(c, &
&name="cudaGraphEventWaitNodeSetEvent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphEventWaitNodeSetEvent_
      type(c_ptr),value :: node
      type(c_ptr),value :: event
    end function

  end interface
  !>  @brief Sets the event for an event record node in the given graphExec.
  !> 
  !>  @param [in] hGraphExec - instance of the executable graph with the node.
  !>  @param [in] hNode - node from the graph which was used to instantiate graphExec.
  !>  @param [in] event - pointer to the event.
  !>  @returns #hipSuccess, #hipErrorInvalidValue
  !>  @warning : This API is marked as beta, meaning, while this is feature complete,
  !>  it is still open to changes and may have outstanding issues.
  interface cudaGraphExecEventWaitNodeSetEvent
    function cudaGraphExecEventWaitNodeSetEvent_(hGraphExec,hNode,&
&event) bind(c, name="cudaGraphExecEventWaitNodeSetEvent")
      use iso_c_binding
      use cudafor_enums
      use cudafor_types
      implicit none
      integer(kind(cudaSuccess)) :: cudaGraphExecEventWaitNodeSetEvent_
      type(c_ptr),value :: hGraphExec
      type(c_ptr),value :: hNode
      type(c_ptr),value :: event
    end function

  end interface


  
end module cudafor
