/*
Copyright (c) 2015 - 2022 Advanced Micro Devices, Inc. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#pragma once

#define __HOST_DEVICE__ __host__ __device__


#if __clang__
#define ADDRESS_SPACE_CONSTANT __attribute__((address_space(4)))
#else
#define ADDRESS_SPACE_CONSTANT __attribute__((address_space(2)))
#endif
#define HIP_IMAGE_OBJECT_SIZE_DWORD 12
#define HIP_SAMPLER_OBJECT_SIZE_DWORD 8
#define HIP_SAMPLER_OBJECT_OFFSET_DWORD HIP_IMAGE_OBJECT_SIZE_DWORD
#define HIP_TEXTURE_OBJECT_SIZE_DWORD (HIP_IMAGE_OBJECT_SIZE_DWORD + HIP_SAMPLER_OBJECT_SIZE_DWORD)

template<typename T>
struct __basic_value_type {};

template<> struct __basic_value_type<float>  {typedef float type;};
template<> struct __basic_value_type<float1> {typedef float type;};
template<> struct __basic_value_type<float2> {typedef float type;};
template<> struct __basic_value_type<float4> {typedef float type;};

template<> struct __basic_value_type<int>  {typedef int type;};
template<> struct __basic_value_type<int1> {typedef int type;};
template<> struct __basic_value_type<int2> {typedef int type;};
template<> struct __basic_value_type<int4> {typedef int type;};

template<> struct __basic_value_type<uint>  {typedef uint type;};
template<> struct __basic_value_type<uint1> {typedef uint type;};
template<> struct __basic_value_type<uint2> {typedef uint type;};
template<> struct __basic_value_type<uint4> {typedef uint type;};

template<> struct __basic_value_type<char>  {typedef char type;};
template<> struct __basic_value_type<char1> {typedef char type;};
template<> struct __basic_value_type<char2> {typedef char type;};
template<> struct __basic_value_type<char4> {typedef char type;};

template<> struct __basic_value_type<uchar>  {typedef uchar type;};
template<> struct __basic_value_type<uchar1> {typedef uchar type;};
template<> struct __basic_value_type<uchar2> {typedef uchar type;};
template<> struct __basic_value_type<uchar4> {typedef uchar type;};

template<> struct __basic_value_type<short>  {typedef short type;};
template<> struct __basic_value_type<short1> {typedef short type;};
template<> struct __basic_value_type<short2> {typedef short type;};
template<> struct __basic_value_type<short4> {typedef short type;};

template<> struct __basic_value_type<ushort>  {typedef ushort type;};
template<> struct __basic_value_type<ushort1> {typedef ushort type;};
template<> struct __basic_value_type<ushort2> {typedef ushort type;};
template<> struct __basic_value_type<ushort4> {typedef ushort type;};




template<typename T, typename U>
__HOST_DEVICE__
__forceinline__
typename std::enable_if<
  sizeof(T) / sizeof(typename __basic_value_type<T>::type) == 1 &&
  sizeof(U) / sizeof(typename __basic_value_type<U>::type) >= 1, T>::type
mapElem(const U &u) {
  T t;
  t.x = static_cast<typename __basic_value_type<T>::type>(u.x);
  return t;
}

template<typename T, typename U>
__HOST_DEVICE__
__forceinline__
typename std::enable_if<
  sizeof(T) / sizeof(typename __basic_value_type<T>::type) == 2 &&
  sizeof(U) / sizeof(typename __basic_value_type<U>::type) >= 2, T>::type
mapElem(const U &u) {
  T t;
  t.x = static_cast<typename __basic_value_type<T>::type>(u.x);
  t.y = static_cast<typename __basic_value_type<T>::type>(u.y);
  return t;
}

template<typename T, typename U>
__HOST_DEVICE__
__forceinline__
typename std::enable_if<
  sizeof(T) / sizeof(typename __basic_value_type<T>::type) == 3 &&
  sizeof(U) / sizeof(typename __basic_value_type<U>::type) >= 3, T>::type
mapElem(const U &u) {
  T t;
  t.x = static_cast<typename __basic_value_type<T>::type>(u.x);
  t.y = static_cast<typename __basic_value_type<T>::type>(u.y);
  t.z = static_cast<typename __basic_value_type<T>::type>(u.z);
  return t;
}

template<typename T, typename U>
__HOST_DEVICE__
__forceinline__
typename std::enable_if<
  sizeof(T) / sizeof(typename __basic_value_type<T>::type) == 4 &&
  sizeof(U) / sizeof(typename __basic_value_type<U>::type) >= 4, T>::type
mapElem(const U &u) {
  T t;
  t.x = static_cast<typename __basic_value_type<T>::type>(u.x);
  t.y = static_cast<typename __basic_value_type<T>::type>(u.y);
  t.z = static_cast<typename __basic_value_type<T>::type>(u.z);
  t.w = static_cast<typename __basic_value_type<T>::type>(u.w);
  return t;
}

// template<typename T, typename U>
// __HOST_DEVICE__
// __forceinline__
// typename std::enable_if<
//         std::is_same<T, char>::value ||
//         std::is_same<T, unsigned char>::value ||
//         std::is_same<T, short>::value ||
//         std::is_same<T, unsigned short>::value ||
//         std::is_same<T, int>::value ||
//         std::is_same<T, unsigned int>::value ||
//         std::is_same<T, float>::value, const T>::type
// mapFrom(const U &u) {
//   union {
//     U u;
//     T t;
//   } d = { u };
//   return d.t;
// }

template<typename T, typename U>
__HOST_DEVICE__
__forceinline__
typename std::enable_if<
        (sizeof(T) == sizeof(typename __basic_value_type<T>::type)) ||
        std::is_same<T, int>::value ||
        std::is_same<T, unsigned int>::value ||
        std::is_same<T, float>::value, const T>::type
mapFrom(const U &u) {
  union {
    U u;
    T t;
  } d = { u };
  return d.t;
}

template<typename T, typename U>
__HOST_DEVICE__
__forceinline__
typename std::enable_if<
        (sizeof(T) > sizeof(typename __basic_value_type<T>::type)) && (
        std::is_same<typename __basic_value_type<T>::type, float>::value ||
        std::is_same<typename __basic_value_type<T>::type, char>::value ||
        std::is_same<typename __basic_value_type<T>::type, unsigned char>::value ||
        std::is_same<typename __basic_value_type<T>::type, short>::value ||
        std::is_same<typename __basic_value_type<T>::type, unsigned short>::value ||
        std::is_same<typename __basic_value_type<T>::type, int>::value ||
        std::is_same<typename __basic_value_type<T>::type, unsigned int>::value), const T>::type
mapFrom(const U &u) {
  union {
    U u;
    int4 i4;
    uint4 u4;
    float4 f4;
  } d = { u };
  if(std::is_same<typename __basic_value_type<T>::type, float>::value) {
    return mapElem<T>(d.f4);
  } else if (std::is_same<typename __basic_value_type<T>::type, int>::value) {
    return mapElem<T>(d.i4);
  } else {
    return mapElem<T>(d.u4);
  }
}

/**
 * Surface helper function
 * CUDA is using byte address, need map to pixel address for HIP
 */
static __device__ __forceinline__ int __hipGetPixelAddr(int x, int format, int order) {
    /*
    * use below format index to generate format LUT
      typedef enum {
        HSA_EXT_IMAGE_CHANNEL_TYPE_SNORM_INT8 = 0,
        HSA_EXT_IMAGE_CHANNEL_TYPE_SNORM_INT16 = 1,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNORM_INT8 = 2,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNORM_INT16 = 3,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNORM_INT24 = 4,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNORM_SHORT_555 = 5,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNORM_SHORT_565 = 6,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNORM_SHORT_101010 = 7,
        HSA_EXT_IMAGE_CHANNEL_TYPE_SIGNED_INT8 = 8,
        HSA_EXT_IMAGE_CHANNEL_TYPE_SIGNED_INT16 = 9,
        HSA_EXT_IMAGE_CHANNEL_TYPE_SIGNED_INT32 = 10,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNSIGNED_INT8 = 11,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNSIGNED_INT16 = 12,
        HSA_EXT_IMAGE_CHANNEL_TYPE_UNSIGNED_INT32 = 13,
        HSA_EXT_IMAGE_CHANNEL_TYPE_HALF_FLOAT = 14,
        HSA_EXT_IMAGE_CHANNEL_TYPE_FLOAT = 15
      } hsa_ext_image_channel_type_t;
    */
    static const int FormatLUT[] = { 0, 1, 0, 1, 3, 1, 1, 1, 0, 1, 2, 0, 1, 2, 1, 2 };
    x = FormatLUT[format] == 3 ? x / FormatLUT[format] : x >> FormatLUT[format];

    /*
    * use below order index to generate order LUT
      typedef enum {
        HSA_EXT_IMAGE_CHANNEL_ORDER_A = 0,
        HSA_EXT_IMAGE_CHANNEL_ORDER_R = 1,
        HSA_EXT_IMAGE_CHANNEL_ORDER_RX = 2,
        HSA_EXT_IMAGE_CHANNEL_ORDER_RG = 3,
        HSA_EXT_IMAGE_CHANNEL_ORDER_RGX = 4,
        HSA_EXT_IMAGE_CHANNEL_ORDER_RA = 5,
        HSA_EXT_IMAGE_CHANNEL_ORDER_RGB = 6,
        HSA_EXT_IMAGE_CHANNEL_ORDER_RGBX = 7,
        HSA_EXT_IMAGE_CHANNEL_ORDER_RGBA = 8,
        HSA_EXT_IMAGE_CHANNEL_ORDER_BGRA = 9,
        HSA_EXT_IMAGE_CHANNEL_ORDER_ARGB = 10,
        HSA_EXT_IMAGE_CHANNEL_ORDER_ABGR = 11,
        HSA_EXT_IMAGE_CHANNEL_ORDER_SRGB = 12,
        HSA_EXT_IMAGE_CHANNEL_ORDER_SRGBX = 13,
        HSA_EXT_IMAGE_CHANNEL_ORDER_SRGBA = 14,
        HSA_EXT_IMAGE_CHANNEL_ORDER_SBGRA = 15,
        HSA_EXT_IMAGE_CHANNEL_ORDER_INTENSITY = 16,
        HSA_EXT_IMAGE_CHANNEL_ORDER_LUMINANCE = 17,
        HSA_EXT_IMAGE_CHANNEL_ORDER_DEPTH = 18,
        HSA_EXT_IMAGE_CHANNEL_ORDER_DEPTH_STENCIL = 19
      } hsa_ext_image_channel_order_t;
    */
    static const int OrderLUT[] = { 0, 0, 1, 1, 3, 1, 3, 2, 2, 2, 2, 2, 3, 2, 2, 2, 0, 0, 0, 0 };
    return x = OrderLUT[order] == 3 ? x / OrderLUT[order] : x >> OrderLUT[order];
}
template <
    typename T,
    typename std::enable_if<std::is_scalar<T>::value>::type* = nullptr>
static __device__ __forceinline__ float4 __hipMapToNativeFloat4(const T& t) {
    float4 tmp;
    tmp.x = static_cast<float>(t);
    return tmp;
}

template <
    typename T,
    typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 1>::type* = nullptr>
static __device__ __forceinline__ float4 __hipMapToNativeFloat4(const T& t) {
    float4 tmp;
    tmp.x = static_cast<float>(t.x);
    return tmp;
}

template <
    typename T,
    typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 2>::type* = nullptr>
static __device__ __forceinline__ float4 __hipMapToNativeFloat4(const T& t) {
    float4 tmp;
    tmp.x = static_cast<float>(t.x);
    tmp.y = static_cast<float>(t.y);
    return tmp;
}

template <
    typename T,
    typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 3>::type* = nullptr>
static __device__ __forceinline__ float4 __hipMapToNativeFloat4(const T& t) {
    float4 tmp;
    tmp.x = static_cast<float>(t.x);
    tmp.y = static_cast<float>(t.y);
    tmp.z = static_cast<float>(t.z);
    return tmp;
}

template <
    typename T,
    typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 4>::type* = nullptr>
static __device__ __forceinline__ float4 __hipMapToNativeFloat4(const T& t) {
    float4 tmp;
    tmp.x = static_cast<float>(t.x);
    tmp.y = static_cast<float>(t.y);
    tmp.z = static_cast<float>(t.z);
    tmp.w = static_cast<float>(t.w);
    return tmp;
}

template<typename T>
static __device__ __forceinline__ 
typename std::enable_if<std::is_scalar<T>::value, const T>::type 
__hipMapFromNativeFloat4(const float4& u) {
    T tmp;
    tmp = static_cast<T>(u.x);
    return tmp;
}

template<typename T>
static __device__ __forceinline__
typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 1, const T>::type 
__hipMapFromNativeFloat4(const float4& u) {
    T tmp;
    tmp.x = static_cast<typename T::value_type>(u.x);
    return tmp;
}

template<typename T>
static __device__ __forceinline__
typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 2, const T>::type
__hipMapFromNativeFloat4(const float4& u) {
    T tmp;
    tmp.x = static_cast<typename T::value_type>(u.x);
    tmp.y = static_cast<typename T::value_type>(u.y);
    return tmp;
}

template<typename T>
static __device__ __forceinline__
typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 3, const T>::type
__hipMapFromNativeFloat4(const float4& u) {
    T tmp;
    tmp.x = static_cast<typename T::value_type>(u.x);
    tmp.y = static_cast<typename T::value_type>(u.y);
    tmp.z = static_cast<typename T::value_type>(u.z);
    return tmp;
}

template<typename T>
static __device__ __forceinline__
typename std::enable_if<!std::is_scalar<T>::value && sizeof(T) / sizeof(typename T::value_type) == 4, const T>::type
__hipMapFromNativeFloat4(const float4& u) {
    T tmp;
    tmp.x = static_cast<typename T::value_type>(u.x);
    tmp.y = static_cast<typename T::value_type>(u.y);
    tmp.z = static_cast<typename T::value_type>(u.z);
    tmp.w = static_cast<typename T::value_type>(u.w);
    return tmp;
}
// #include <hip/hip_vector_types.h>

extern "C" {

#define ADDRESS_SPACE_CONSTANT __attribute__((address_space(4)))

__device__ float4 __ockl_image_load_1D(unsigned int ADDRESS_SPACE_CONSTANT*i, int c);

__device__ float4 __ockl_image_load_1Db(unsigned int ADDRESS_SPACE_CONSTANT*i, int c);

__device__ float4 __ockl_image_load_1Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c);

__device__ float4 __ockl_image_load_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c);

__device__ float4 __ockl_image_load_2Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c);

__device__ float4 __ockl_image_load_3D(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c);

__device__ float4 __ockl_image_load_CM(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int f);

__device__ float4 __ockl_image_load_CMa(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int f);

__device__ float4 __ockl_image_load_lod_1D(unsigned int ADDRESS_SPACE_CONSTANT*i, int c, int l);

__device__ float4 __ockl_image_load_lod_1Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int l);

__device__ float4 __ockl_image_load_lod_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int l);

__device__ float4 __ockl_image_load_lod_2Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int l);

__device__ float4 __ockl_image_load_lod_3D(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int l);

__device__ float4 __ockl_image_load_lod_CM(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int f, int l);

__device__ float4 __ockl_image_load_lod_CMa(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int f, int l);

__device__ void __ockl_image_store_1D(unsigned int ADDRESS_SPACE_CONSTANT*i, int c, float4 p);

__device__ void __ockl_image_store_1Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, float4 p);

__device__ void __ockl_image_store_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, float4 p);

__device__ void __ockl_image_store_2Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, float4 p);

__device__ void __ockl_image_store_3D(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, float4 p);

__device__ void __ockl_image_store_CM(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int f, float4 p);

__device__ void __ockl_image_store_CMa(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int f, float4 p);

__device__ void __ockl_image_store_lod_1D(unsigned int ADDRESS_SPACE_CONSTANT*i, int c, int l, float4 p);

__device__ void __ockl_image_store_lod_1Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int l, float4 p);

__device__ void __ockl_image_store_lod_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int l, float4 p);

__device__ void __ockl_image_store_lod_2Da(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int l, float4 p);

__device__ void __ockl_image_store_lod_3D(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int l, float4 p);

__device__ void __ockl_image_store_lod_CM(unsigned int ADDRESS_SPACE_CONSTANT*i, int2 c, int f, int l, float4 p);

__device__ void __ockl_image_store_lod_CMa(unsigned int ADDRESS_SPACE_CONSTANT*i, int4 c, int f, int l, float4 p);

__device__ float4 __ockl_image_sample_1D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float c);

__device__ float4 __ockl_image_sample_1Da(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c);

__device__ float4 __ockl_image_sample_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c);

__device__ float4 __ockl_image_sample_2Da(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c);

__device__ float4 __ockl_image_sample_3D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c);

__device__ float4 __ockl_image_sample_CM(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c);

__device__ float4 __ockl_image_sample_CMa(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c);

__device__ float4 __ockl_image_sample_grad_1D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float c, float dx, float dy);

__device__ float4 __ockl_image_sample_grad_1Da(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c, float dx, float dy);

__device__ float4 __ockl_image_sample_grad_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c, float2 dx, float2 dy);

__device__ float4 __ockl_image_sample_grad_2Da(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c, float2 dx, float2 dy);

__device__ float4 __ockl_image_sample_grad_3D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c, float4 dx, float4 dy);

__device__ float4 __ockl_image_sample_lod_1D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float c, float l);

__device__ float4 __ockl_image_sample_lod_1Da(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c, float l);

__device__ float4 __ockl_image_sample_lod_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c, float l);

__device__ float4 __ockl_image_sample_lod_2Da(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c, float l);

__device__ float4 __ockl_image_sample_lod_3D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c, float l);

__device__ float4 __ockl_image_sample_lod_CM(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c, float l);

__device__ float4 __ockl_image_sample_lod_CMa(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float4 c, float l);

__device__ float4 __ockl_image_gather4r_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c);

__device__ float4 __ockl_image_gather4g_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c);

__device__ float4 __ockl_image_gather4b_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c);

__device__ float4 __ockl_image_gather4a_2D(unsigned int ADDRESS_SPACE_CONSTANT*i, unsigned int ADDRESS_SPACE_CONSTANT*s, float2 c);

__device__ int __ockl_image_channel_data_type_1D(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_1Da(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_1Db(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_2D(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_2Da(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_2Dad(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_2Dd(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_3D(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_CM(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_data_type_CMa(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_1D(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_1Da(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_1Db(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_2D(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_2Da(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_2Dad(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_2Dd(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_3D(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_CM(unsigned int ADDRESS_SPACE_CONSTANT* i);

__device__ int __ockl_image_channel_order_CMa(unsigned int ADDRESS_SPACE_CONSTANT* i);

};