// Copyright (c) 2017-2021 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#ifndef CURAND_COMMON_H_
#define CURAND_COMMON_H_

#define CURAND_2POW16_INV (1.5258789e-05f)
#define CURAND_2POW16_INV_2PI (1.5258789e-05f * 6.2831855f)
#define CURAND_2POW32_INV (2.3283064e-10f)
#define CURAND_2POW32_INV_DOUBLE (2.3283064365386963e-10)
#define CURAND_2POW64_INV_DOUBLE (5.4210108624275221700372640043497e-20)
#define CURAND_2POW32_INV_2PI (2.3283064e-10f * 6.2831855f)
#define CURAND_2POW53_INV_DOUBLE (1.1102230246251565e-16)
#define CURAND_PI  (3.1415926f)
#define CURAND_PI_DOUBLE  (3.1415926535897932)
#define CURAND_2PI (6.2831855f)
#define CURAND_SQRT2 (1.4142135f)
#define CURAND_SQRT2_DOUBLE (1.4142135623730951)

#include <math.h>
#include <stdio.h>
#include <stdint.h>

#ifdef WIN32
#define CURAND_KERNEL __global__ static
#else
#define CURAND_KERNEL __global__
#endif

#ifndef FQUALIFIERS
#define FQUALIFIERS __forceinline__ __device__
#endif // FQUALIFIERS

namespace curand_device {
namespace detail {

FQUALIFIERS
unsigned long long mad_u64_u32(const unsigned int x, const unsigned int y, const unsigned long long z)
{
    return static_cast<unsigned long long>(x) * static_cast<unsigned long long>(y) + z;
}

// This helps access fields of engine's internal state which
// saves floats and doubles generated using the Box–Muller transform
template<typename Engine>
struct engine_boxmuller_helper
{
    static FQUALIFIERS
    bool has_float(const Engine * engine)
    {
        return engine->m_state.boxmuller_float_state != 0;
    }

    static FQUALIFIERS
    float get_float(Engine * engine)
    {
        engine->m_state.boxmuller_float_state = 0;
        return engine->m_state.boxmuller_float;
    }

    static FQUALIFIERS
    void save_float(Engine * engine, float f)
    {
        engine->m_state.boxmuller_float_state = 1;
        engine->m_state.boxmuller_float = f;
    }

    static FQUALIFIERS
    bool has_double(const Engine * engine)
    {
        return engine->m_state.boxmuller_double_state != 0;
    }

    static FQUALIFIERS
    float get_double(Engine * engine)
    {
        engine->m_state.boxmuller_double_state = 0;
        return engine->m_state.boxmuller_double;
    }

    static FQUALIFIERS
    void save_double(Engine * engine, double d)
    {
        engine->m_state.boxmuller_double_state = 1;
        engine->m_state.boxmuller_double = d;
    }
};

} // end namespace detail
} // end namespace curand_device

#endif // CURAND_COMMON_H_
