# coding=utf-8
# Copyright 2020 The HuggingFace Team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from __future__ import annotations

import unittest

from transformers import is_tf_available, is_torch_available
from transformers.testing_utils import DUMMY_UNKNOWN_IDENTIFIER, SMALL_MODEL_IDENTIFIER, is_pt_tf_cross_test, slow


if is_tf_available():
    from transformers import (
        AutoConfig,
        BertConfig,
        GPT2Config,
        T5Config,
        TFAutoModel,
        TFAutoModelForCausalLM,
        TFAutoModelForMaskedLM,
        TFAutoModelForPreTraining,
        TFAutoModelForQuestionAnswering,
        TFAutoModelForSeq2SeqLM,
        TFAutoModelForSequenceClassification,
        TFAutoModelWithLMHead,
        TFBertForMaskedLM,
        TFBertForPreTraining,
        TFBertForQuestionAnswering,
        TFBertForSequenceClassification,
        TFBertModel,
        TFGPT2LMHeadModel,
        TFRobertaForMaskedLM,
        TFT5ForConditionalGeneration,
    )

if is_torch_available():
    from transformers import (
        AutoModel,
        AutoModelForCausalLM,
        AutoModelForMaskedLM,
        AutoModelForPreTraining,
        AutoModelForQuestionAnswering,
        AutoModelForSeq2SeqLM,
        AutoModelForSequenceClassification,
        AutoModelWithLMHead,
        BertForMaskedLM,
        BertForPreTraining,
        BertForQuestionAnswering,
        BertForSequenceClassification,
        BertModel,
        GPT2LMHeadModel,
        RobertaForMaskedLM,
        T5ForConditionalGeneration,
    )


@is_pt_tf_cross_test
class TFPTAutoModelTest(unittest.TestCase):
    @slow
    def test_model_from_pretrained(self):
        #     model_name = 'google-bert/bert-base-uncased'
        for model_name in ["google-bert/bert-base-uncased"]:
            config = AutoConfig.from_pretrained(model_name)
            self.assertIsNotNone(config)
            self.assertIsInstance(config, BertConfig)

            model = TFAutoModel.from_pretrained(model_name, from_pt=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, TFBertModel)

            model = AutoModel.from_pretrained(model_name, from_tf=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, BertModel)

    @slow
    def test_model_for_pretraining_from_pretrained(self):
        #     model_name = 'google-bert/bert-base-uncased'
        for model_name in ["google-bert/bert-base-uncased"]:
            config = AutoConfig.from_pretrained(model_name)
            self.assertIsNotNone(config)
            self.assertIsInstance(config, BertConfig)

            model = TFAutoModelForPreTraining.from_pretrained(model_name, from_pt=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, TFBertForPreTraining)

            model = AutoModelForPreTraining.from_pretrained(model_name, from_tf=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, BertForPreTraining)

    @slow
    def test_model_for_causal_lm(self):
        model_name = "google-bert/bert-base-uncased"
        config = AutoConfig.from_pretrained(model_name)
        self.assertIsNotNone(config)
        self.assertIsInstance(config, GPT2Config)

        model = TFAutoModelForCausalLM.from_pretrained(model_name, from_pt=True)
        model, loading_info = TFAutoModelForCausalLM.from_pretrained(
            model_name, output_loading_info=True, from_pt=True
        )
        self.assertIsNotNone(model)
        self.assertIsInstance(model, TFGPT2LMHeadModel)

        model = AutoModelForCausalLM.from_pretrained(model_name, from_tf=True)
        model, loading_info = AutoModelForCausalLM.from_pretrained(model_name, output_loading_info=True, from_tf=True)
        self.assertIsNotNone(model)
        self.assertIsInstance(model, GPT2LMHeadModel)

    @slow
    def test_lmhead_model_from_pretrained(self):
        model_name = "google-bert/bert-base-uncased"
        config = AutoConfig.from_pretrained(model_name)
        self.assertIsNotNone(config)
        self.assertIsInstance(config, BertConfig)

        model = TFAutoModelWithLMHead.from_pretrained(model_name, from_pt=True)
        self.assertIsNotNone(model)
        self.assertIsInstance(model, TFBertForMaskedLM)

        model = AutoModelWithLMHead.from_pretrained(model_name, from_tf=True)
        self.assertIsNotNone(model)
        self.assertIsInstance(model, BertForMaskedLM)

    @slow
    def test_model_for_masked_lm(self):
        model_name = "google-bert/bert-base-uncased"
        config = AutoConfig.from_pretrained(model_name)
        self.assertIsNotNone(config)
        self.assertIsInstance(config, BertConfig)

        model = TFAutoModelForMaskedLM.from_pretrained(model_name, from_pt=True)
        model, loading_info = TFAutoModelForMaskedLM.from_pretrained(
            model_name, output_loading_info=True, from_pt=True
        )
        self.assertIsNotNone(model)
        self.assertIsInstance(model, TFBertForMaskedLM)

        model = AutoModelForMaskedLM.from_pretrained(model_name, from_tf=True)
        model, loading_info = AutoModelForMaskedLM.from_pretrained(model_name, output_loading_info=True, from_tf=True)
        self.assertIsNotNone(model)
        self.assertIsInstance(model, BertForMaskedLM)

    @slow
    def test_model_for_encoder_decoder_lm(self):
        model_name = "google-bert/bert-base-uncased"
        config = AutoConfig.from_pretrained(model_name)
        self.assertIsNotNone(config)
        self.assertIsInstance(config, T5Config)

        model = TFAutoModelForSeq2SeqLM.from_pretrained(model_name, from_pt=True)
        model, loading_info = TFAutoModelForSeq2SeqLM.from_pretrained(
            model_name, output_loading_info=True, from_pt=True
        )
        self.assertIsNotNone(model)
        self.assertIsInstance(model, TFT5ForConditionalGeneration)

        model = AutoModelForSeq2SeqLM.from_pretrained(model_name, from_tf=True)
        model, loading_info = AutoModelForSeq2SeqLM.from_pretrained(model_name, output_loading_info=True, from_tf=True)
        self.assertIsNotNone(model)
        self.assertIsInstance(model, T5ForConditionalGeneration)

    @slow
    def test_sequence_classification_model_from_pretrained(self):
        #     model_name = 'google-bert/bert-base-uncased'
        for model_name in ["google-bert/bert-base-uncased"]:
            config = AutoConfig.from_pretrained(model_name)
            self.assertIsNotNone(config)
            self.assertIsInstance(config, BertConfig)

            model = TFAutoModelForSequenceClassification.from_pretrained(model_name, from_pt=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, TFBertForSequenceClassification)

            model = AutoModelForSequenceClassification.from_pretrained(model_name, from_tf=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, BertForSequenceClassification)

    @slow
    def test_question_answering_model_from_pretrained(self):
        #     model_name = 'google-bert/bert-base-uncased'
        for model_name in ["google-bert/bert-base-uncased"]:
            config = AutoConfig.from_pretrained(model_name)
            self.assertIsNotNone(config)
            self.assertIsInstance(config, BertConfig)

            model = TFAutoModelForQuestionAnswering.from_pretrained(model_name, from_pt=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, TFBertForQuestionAnswering)

            model = AutoModelForQuestionAnswering.from_pretrained(model_name, from_tf=True)
            self.assertIsNotNone(model)
            self.assertIsInstance(model, BertForQuestionAnswering)

    def test_from_pretrained_identifier(self):
        model = TFAutoModelWithLMHead.from_pretrained(SMALL_MODEL_IDENTIFIER, from_pt=True)
        self.assertIsInstance(model, TFBertForMaskedLM)
        self.assertEqual(model.num_parameters(), 14410)
        self.assertEqual(model.num_parameters(only_trainable=True), 14410)

        model = AutoModelWithLMHead.from_pretrained(SMALL_MODEL_IDENTIFIER, from_tf=True)
        self.assertIsInstance(model, BertForMaskedLM)
        self.assertEqual(model.num_parameters(), 14410)
        self.assertEqual(model.num_parameters(only_trainable=True), 14410)

    def test_from_identifier_from_model_type(self):
        model = TFAutoModelWithLMHead.from_pretrained(DUMMY_UNKNOWN_IDENTIFIER, from_pt=True)
        self.assertIsInstance(model, TFRobertaForMaskedLM)
        self.assertEqual(model.num_parameters(), 14410)
        self.assertEqual(model.num_parameters(only_trainable=True), 14410)

        model = AutoModelWithLMHead.from_pretrained(DUMMY_UNKNOWN_IDENTIFIER, from_tf=True)
        self.assertIsInstance(model, RobertaForMaskedLM)
        self.assertEqual(model.num_parameters(), 14410)
        self.assertEqual(model.num_parameters(only_trainable=True), 14410)
