import gradio as gr
import torch
from transformers.models.speecht5.number_normalizer import EnglishNumberNormalizer
from string import punctuation
import re

from parler_tts import ParlerTTSForConditionalGeneration
from transformers import AutoTokenizer, AutoFeatureExtractor, set_seed

device = "cuda:0" if torch.cuda.is_available() else "cpu"

repo_id = "parler-tts/parler-tts-mini-v1"
repo_id_large = "ylacombe/parler-large-v1-og"

model = ParlerTTSForConditionalGeneration.from_pretrained(repo_id).to(device)
model_large = ParlerTTSForConditionalGeneration.from_pretrained(repo_id_large).to(device)
tokenizer = AutoTokenizer.from_pretrained(repo_id)
feature_extractor = AutoFeatureExtractor.from_pretrained(repo_id)

SAMPLE_RATE = feature_extractor.sampling_rate
SEED = 42

default_text = "All of the data, pre-processing, training code, and weights are released publicly under a permissive license, enabling the community to build on our work and develop their own powerful models."
default_description = "Laura's voice is monotone yet slightly fast in delivery, with a very close recording that almost has no background noise."
examples = [
    [
        "This version introduces speaker consistency across generations, characterized by their name. For example, Jon, Lea, Gary, Jenna, Mike and Laura.",
        "Gary's voice is monotone yet slightly fast in delivery, with a very close recording that has no background noise.",
        None,
    ],
    [
        '''There's 34 speakers. To take advantage of this, simply adapt your text description to specify which speaker to use: "Mike speaks animatedly...".''',
        "Gary speaks slightly animatedly and slightly slowly in delivery, with a very close recording that has no background noise.",
        None
    ],
    [
        "'This is the best time of my life, Bartley,' she said happily.",
        "A female speaker delivers a slightly expressive and animated speech with a moderate speed. The recording features a low-pitch voice and slight background noise, creating a close-sounding audio experience.",
        None,
    ],
    [
        "Montrose also, after having experienced still more variety of good and bad fortune, threw down his arms, and retired out of the kingdom.",
        "A man voice speaks slightly slowly with very noisy background, carrying a low-pitch tone and displaying a touch of expressiveness and animation. The sound is very distant, adding an air of intrigue.",
        None
    ],
    [
        "Once upon a time, in the depth of winter, when the flakes of snow fell like feathers from the clouds, a queen sat sewing at her pal-ace window, which had a carved frame of black wood.",
        "In a very poor recording quality, a female speaker delivers her slightly expressive and animated words with a fast pace. There's high level of background noise and a very distant-sounding reverberation. Her voice is slightly higher pitched than average.",
        None,
    ],
]

number_normalizer = EnglishNumberNormalizer()


def preprocess(text):
    text = number_normalizer(text).strip()
    text = text.replace("-", " ")
    if text[-1] not in punctuation:
        text = f"{text}."

    abbreviations_pattern = r'\b[A-Z][A-Z\.]+\b'

    def separate_abb(chunk):
        chunk = chunk.replace(".", "")
        print(chunk)
        return " ".join(chunk)

    abbreviations = re.findall(abbreviations_pattern, text)
    for abv in abbreviations:
        if abv in text:
            text = text.replace(abv, separate_abb(abv))
    return text


def gen_tts(text, description, use_large=False):
    inputs = tokenizer(description.strip(), return_tensors="pt").to(device)
    prompt = tokenizer(preprocess(text), return_tensors="pt").to(device)

    set_seed(SEED)
    if use_large:
        generation = model_large.generate(
            input_ids=inputs.input_ids, prompt_input_ids=prompt.input_ids, attention_mask=inputs.attention_mask,
            prompt_attention_mask=prompt.attention_mask, do_sample=True, temperature=1.0
        )
    else:
        generation = model.generate(
            input_ids=inputs.input_ids, prompt_input_ids=prompt.input_ids, attention_mask=inputs.attention_mask,
            prompt_attention_mask=prompt.attention_mask, do_sample=True, temperature=1.0
        )
    audio_arr = generation.cpu().numpy().squeeze()

    return SAMPLE_RATE, audio_arr


css = """
        #share-btn-container {
            display: flex;
            padding-left: 0.5rem !important;
            padding-right: 0.5rem !important;
            background-color: #000000;
            justify-content: center;
            align-items: center;
            border-radius: 9999px !important; 
            width: 13rem;
            margin-top: 10px;
            margin-left: auto;
            flex: unset !important;
        }
        #share-btn {
            all: initial;
            color: #ffffff;
            font-weight: 600;
            cursor: pointer;
            font-family: 'IBM Plex Sans', sans-serif;
            margin-left: 0.5rem !important;
            padding-top: 0.25rem !important;
            padding-bottom: 0.25rem !important;
            right:0;
        }
        #share-btn * {
            all: unset !important;
        }
        #share-btn-container div:nth-child(-n+2){
            width: auto !important;
            min-height: 0px !important;
        }
        #share-btn-container .wrap {
            display: none !important;
        }
"""
with gr.Blocks(css=css) as block:
    gr.HTML(
        """
            <div style="text-align: center; max-width: 700px; margin: 0 auto;">
              <div
                style="
                  display: inline-flex; align-items: center; gap: 0.8rem; font-size: 1.75rem;
                "
              >
                <h1 style="font-weight: 900; margin-bottom: 7px; line-height: normal;">
                  Parler-TTS 🗣️
                </h1>
              </div>
            </div>
        """
    )
    gr.HTML(
        f"""
        <p><a href="https://github.com/huggingface/parler-tts"> Parler-TTS</a> 是一个用于高保真文本转语音 (TTS) 模型的训练和推理库。</p> 
        <p>这里展示的模型 Parler-TTS <a href="https://huggingface.co/parler-tts/parler-tts-mini-v1">Mini v1</a> 和 <a href="https://huggingface.co/parler-tts/parler-tts-large-v1">Large v1</a>
        是使用 45000 小时的英语有声读物进行训练的。它可以生成高质量的语音，并且可以使用简单的文本提示来控制其功能（例如性别、背景噪音、语速、音调和混响）。</p>

        <p>默认情况下，Parler-TTS会生成 🎲随机语音。为了确保  🎯 <b>说话者在生成过程中的一致性</b>, 这些模型还针对34位说话者进行了训练，并以姓名为特征（例如 Jon、Lea、Gary、Jenna、Mike、Laura）。</p>
        
        <p>要利用此功能，只需调整文本描述以指定要使用的说话者： `Jon's voice is monotone...`</p>
        """
    )
    with gr.Row():
        with gr.Column():
            input_text = gr.Textbox(label="输入文本", lines=2, value=default_text, elem_id="input_text")
            description = gr.Textbox(label="描述", lines=2, value=default_description,
                                     elem_id="input_description")
            use_large = gr.Checkbox(value=False, label="使用Large模型",
                                    info="使用Parler-TTS Large v1生成效果更好；使用Mini v1生成速度更快")
            run_button = gr.Button("生成音频", variant="primary")
        with gr.Column():
            audio_out = gr.Audio(label="Parler-TTS生成", type="numpy", elem_id="audio_out")

    inputs = [input_text, description, use_large]
    outputs = [audio_out]
    run_button.click(fn=gen_tts, inputs=inputs, outputs=outputs, queue=True)
    gr.Examples(examples=examples, fn=gen_tts, inputs=inputs, outputs=outputs, cache_examples=True)
    gr.HTML(
        """
        <p>提示:
        <ul>
            <li>文本中包含"very clear audio"以生成最高质量的音频, 以及"very noisy audio"以产生高水平的背景噪音</li>
            <li>标点符号可用于控制代际韵律，例如可以使用逗号在语音中添加小停顿</li>
            <li>其余语音特征（性别、语速、音调和混响）可以通过文本直接控制</li>
        </ul>
        </p>

        <p>Parler-TTS可以更快。 在<a href="https://github.com/huggingface/parler-tts/blob/main/INFERENCE.md"> 推理指南</a>给出了一些快速生成的方法，包括SDPA, torch.compile, batching and streaming!</p>
        
        <p> 如果您想了解有关如何训练该模型的更多信息，甚至对其进行微调，请查看 GitHub 上
        <a href="https://github.com/huggingface/parler-tts"> Parler-TTS</a> repository</p>
        
        """
    )

block.queue()
block.launch(server_name='0.0.0.0', share=True)
