import { BaseTelemetryPlugin, IChannelControls, IExtendedConfiguration, NotificationManager, IPlugin, ITelemetryItem, IAppInsightsCore, IProcessTelemetryContext, SendRequestReason } from "@microsoft/1ds-core-js";
import { IPostChannel } from "./DataModels";
/**
 * Class that manages adding events to inbound queues and batching of events
 * into requests.
 */
export default class PostChannel extends BaseTelemetryPlugin implements IChannelControls, IPostChannel {
    identifier: string;
    priority: number;
    version: string;
    _notificationManager: NotificationManager | undefined;
    /** @deprecated This property is not intended to be used directly please let us know if you have taken a dependency on this property as it may be removed in a future release */
    _setTimeoutOverride: typeof setTimeout;
    /** @deprecated This property is not intended to be used directly please let us know if you have taken a dependency on this property as it may be removed in a future release */
    _clearTimeoutOverride: typeof clearTimeout;
    constructor();
    /**
     * Start the queue manager to batch and send events via post.
     * @param config - The core configuration.
     */
    initialize(coreConfig: IExtendedConfiguration, core: IAppInsightsCore, extensions: IPlugin[]): void;
    /**
     * Add an event to the appropriate inbound queue based on its latency.
     * @param ev - The event to be added to the queue.
     * @param itemCtx - This is the context for the current request, ITelemetryPlugin instances
     * can optionally use this to access the current core instance or define / pass additional information
     * to later plugins (vs appending items to the telemetry item)
     */
    processTelemetry(ev: ITelemetryItem, itemCtx?: IProcessTelemetryContext): void;
    /**
     * Sets the event queue limits at runtime (after initialization), if the number of queued events is greater than the
     * eventLimit or autoFlushLimit then a flush() operation will be scheduled.
     * @param eventLimit The number of events that can be kept in memory before the SDK starts to drop events. If the value passed is less than or
     * equal to zero the value will be reset to the default (10,000).
     * @param autoFlushLimit When defined, once this number of events has been queued the system perform a flush() to send the queued events
     * without waiting for the normal schedule timers. Passing undefined, null or a value less than or equal to zero will disable the auto flush.
     */
    setEventQueueLimits(eventLimit: number, autoFlushLimit?: number): void;
    /**
     * Pause the transmission of any requests
     */
    pause(): void;
    /**
     * Resumes transmission of events.
     */
    resume(): void;
    /**
    * Add handler to be executed with request response text.
    */
    addResponseHandler(responseHanlder: (responseText: string) => void): void;
    /**
     * Flush to send data immediately; channel should default to sending data asynchronously
     * @param async - send data asynchronously when true
     * @param callback - if specified, notify caller when send is complete
     */
    flush(async?: boolean, callback?: () => void, sendReason?: SendRequestReason): void;
    /**
     * Set AuthMsaDeviceTicket header
     * @param ticket - Ticket value.
     */
    setMsaAuthTicket(ticket: string): void;
    /**
     * Check if there are any events waiting to be scheduled for sending.
     * @returns True if there are events, false otherwise.
     */
    hasEvents(): boolean;
    /**
     * Load custom transmission profiles. Each profile should have timers for real time, and normal and can
     * optionally specify the immediate latency time in ms (defaults to 0 when not defined). Each profile should
     * make sure that a each normal latency timer is a multiple of the real-time latency and the immediate
     * is smaller than the real-time.
     * Setting the timer value to -1 means that the events for that latency will not be scheduled to be sent.
     * Note that once a latency has been set to not send, all latencies below it will also not be sent. The
     * timers should be in the form of [normal, high, [immediate]].
     * e.g Custom:
     * [10,5] - Sets the normal latency time to 10 seconds and real-time to 5 seconds; Immediate will default to 0ms
     * [10,5,1] - Sets the normal latency time to 10 seconds and real-time to 5 seconds; Immediate will default to 1ms
     * [10,5,0] - Sets the normal latency time to 10 seconds and real-time to 5 seconds; Immediate will default to 0ms
     * [10,5,-1] - Sets the normal latency time to 10 seconds and real-time to 5 seconds; Immediate events will not be
     * scheduled on their own and but they will be included with real-time or normal events as the first events in a batch.
     * This also removes any previously loaded custom profiles.
     * @param profiles - A dictionary containing the transmit profiles.
     */
    _loadTransmitProfiles(profiles: {
        [profileName: string]: number[];
    }): void;
    /**
     * Set the transmit profile to be used. This will change the transmission timers
     * based on the transmit profile.
     * @param profileName - The name of the transmit profile to be used.
     */
    _setTransmitProfile(profileName: string): void;
    /**
     * Backs off transmission. This exponentially increases all the timers.
     */
    _backOffTransmission(): void;
    /**
     * Clears backoff for transmission.
     */
    _clearBackOff(): void;
}
