/**
* Serializer.ts
* @author Abhilash Panwar (abpanwar); Hector Hernandez (hectorh); Nev Wylie (newylie)
* @copyright Microsoft 2018-2020
*/
import { IPerfManagerProvider, IValueSanitizer, FieldValueSanitizerType, FieldValueSanitizerFunc, SendRequestReason, EventSendType } from "@microsoft/1ds-core-js";
import { IPostTransmissionTelemetryItem } from "./DataModels";
import { EventBatch } from "./EventBatch";
export interface ISerializedPayload {
    /**
     * The collection of iKeys included in this payload
     */
    apiKeys: string[];
    /**
     * Serialized payload blob that needs to be sent.
     */
    payloadBlob: string;
    /**
     * Extra events that would not fit into the serialized blob limit
     */
    overflow: EventBatch;
    /**
     * Events that where dropped because they exceeded the individual limit
     */
    sizeExceed: EventBatch[];
    /**
     * Events that where dropped because they could not be serialized
     */
    failedEvts: EventBatch[];
    /**
    * The batches included in this payload
    */
    batches: EventBatch[];
    /**
     * The number of events in the payload
     */
    numEvents: number;
    /**
     * The retry count for this payload
     */
    retryCnt: number;
    /**
     * Was this payload created during a teardown
     */
    isTeardown: boolean;
    /**
     * Is this payload a synchronous payload
     */
    isSync: boolean;
    /**
     * The payload has been constructed for usage via sendBeacon API
     */
    isBeacon: boolean;
    /**
     * The request event sync type
     */
    sendType: EventSendType;
    /**
     * The reason the payload has is being sent
     */
    sendReason?: SendRequestReason;
}
/**
* Class to handle serialization of event and request.
* Currently uses Bond for serialization. Please note that this may be subject to change.
*/
export declare class Serializer {
    constructor(perfManager?: IPerfManagerProvider, valueSanitizer?: IValueSanitizer, stringifyObjects?: boolean, enableCompoundKey?: boolean);
    /**
     * Create a serializer payload package
     * @param retryCnt The retry count for the events in this payload
     * @param isTeardown Is this payload being created as part of a teardown request
     * @param isSync Should this payload be sent as a synchronous request
     * @param useSendBeacon Is this payload going to be sent via sendBeacon() API
     * @param sendReason The reason the payload is being sent
     * @param sendType Identifies how this payload will be sent
     */
    createPayload(retryCnt: number, isTeardown: boolean, isSync: boolean, useSendBeacon: boolean, sendReason: SendRequestReason, sendType: EventSendType): ISerializedPayload;
    /**
     * Append the batched events into the payload
     * @param payload The serialized payload detail object
     * @param theBatch The batched events to append to the payload
     * @param maxEventsPerBatch The maximum number of events to allow in the payload
     * @returns True if the events from the new batch where attempted to be added to the payload otherwise false
     */
    appendPayload(payload: ISerializedPayload, theBatch: EventBatch, maxEventsPerBatch: number): boolean;
    /**
     * Bond serialize the event.
     * @param eventData - The event that needs to be serialized.
     * @returns The serialized json event.
     */
    getEventBlob(eventData: IPostTransmissionTelemetryItem): string;
    /**
     * Does this field value sanitizer handle this path / field combination
     * @param path - The field path
     * @param name - The name of the field
     */
    handleField(path: string, name: string): boolean;
    /**
     * Get the field sanitizer for this type of field based on the field type, value kind and/or event property type
     * @param path - The field path
     * @param name - The name of the field
     * @param theType - The type of field
     * @param theKind - The value kind of the field
     * @param propType - The property type of the field
     */
    getSanitizer(path: string, name: string, theType: FieldValueSanitizerType, theKind?: number, propType?: number): FieldValueSanitizerFunc | null | undefined;
}
