/*
 * Microsoft Application Insights Core Javascript SDK, 2.8.4
 * Copyright (c) Microsoft and contributors. All rights reserved.
 *
 * Microsoft Application Insights Team
 * https://github.com/microsoft/ApplicationInsights-JS#readme
 */

declare namespace ApplicationInsights {
    import { getGlobal } from '@microsoft/applicationinsights-shims';
    import { objCreateFn as objCreate } from '@microsoft/applicationinsights-shims';
    import { strShimFunction as strFunction } from '@microsoft/applicationinsights-shims';
    import { strShimObject as strObject } from '@microsoft/applicationinsights-shims';
    import { strShimPrototype as strPrototype } from '@microsoft/applicationinsights-shims';
    import { strShimUndefined as strUndefined } from '@microsoft/applicationinsights-shims';

    /**
     * Get all of the registered events on the target object, this is primarily used for testing cleanup but may also be used by
     * applications to remove their own events
     * @param target - The EventTarget that has registered events
     * @param eventName - [Optional] The name of the event to return the registered handlers and full name (with namespaces)
     * @param evtNamespace - [Optional] Additional namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace,
     * if the eventName also includes a namespace the namespace(s) are merged into a single namespace
     */
    function __getRegisteredEvents(target: any, eventName?: string, evtNamespace?: string | string[]): _IRegisteredEvents[];

    /**
     * Trys to add an event handler for the specified event to the window, body and document
     * @param eventName {string} - The name of the event
     * @param callback {any} - The callback function that needs to be executed for the given event
     * @param evtNamespace - [Optional] Namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace.
     * @return {boolean} - true if the handler was successfully added
     */
    function addEventHandler(eventName: string, callback: any, evtNamespace?: string | string[] | null): boolean;

    /**
     * Bind the listener to the array of events
     * @param events An string array of event names to bind the listener to
     * @param listener The event callback to call when the event is triggered
     * @param excludeEvents - [Optional] An array of events that should not be hooked (if possible), unless no other events can be.
     * @param evtNamespace - [Optional] Namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace.
     * @returns true - when at least one of the events was registered otherwise false
     */
    function addEventListeners(events: string[], listener: any, excludeEvents?: string[], evtNamespace?: string | string[]): boolean;

    /**
     * Listen to the pagehide and visibility changing to 'hidden' events, because the 'visibilitychange' uses
     * an internal proxy to detect the visibility state you SHOULD use a unique namespace when if you plan to call
     * removePageShowEventListener as the remove ignores the listener argument for the 'visibilitychange' event.
     * @param listener - The event callback to call when a page hide event is triggered
     * @param excludeEvents - [Optional] An array of events that should not be hooked (if possible), unless no other events can be.
     * @param evtNamespace - [Optional] A Namespace to append to the event listeners so they can be uniquely identified and removed
     * based on this namespace. This call also adds an additional unique "pageshow" namespace to the events
     * so that only the matching "removePageHideEventListener" can remove these events.
     * Suggestion: pass as true if you are also calling addPageUnloadEventListener as that also hooks pagehide
     * @returns true - when at least one of the events was registered otherwise false
     */
    function addPageHideEventListener(listener: any, excludeEvents?: string[] | null, evtNamespace?: string | string[] | null): boolean;

    /**
     * Listen to the pageshow and visibility changing to 'visible' events, because the 'visibilitychange' uses
     * an internal proxy to detect the visibility state you SHOULD use a unique namespace when if you plan to call
     * removePageShowEventListener as the remove ignores the listener argument for the 'visibilitychange' event.
     * @param listener - The event callback to call when a page is show event is triggered
     * @param excludeEvents - [Optional] An array of events that should not be hooked (if possible), unless no other events can be.
     * @param evtNamespace - [Optional/Recommended] A Namespace to append to the event listeners so they can be uniquely
     * identified and removed based on this namespace. This call also adds an additional unique "pageshow" namespace to the events
     * so that only the matching "removePageShowEventListener" can remove these events.
     * @returns true - when at least one of the events was registered otherwise false
     */
    function addPageShowEventListener(listener: any, excludeEvents?: string[] | null, evtNamespace?: string | string[] | null): boolean;

    /**
     * Listen to the 'beforeunload', 'unload' and 'pagehide' events which indicates a page unload is occurring,
     * this does NOT listen to the 'visibilitychange' event as while it does indicate that the page is being hidden
     * it does not *necessarily* mean that the page is being completely unloaded, it can mean that the user is
     * just navigating to a different Tab and may come back (without unloading the page). As such you may also
     * need to listen to the 'addPageHideEventListener' and 'addPageShowEventListener' events.
     * @param listener - The event callback to call when a page unload event is triggered
     * @param excludeEvents - [Optional] An array of events that should not be hooked, unless no other events can be.
     * @param evtNamespace - [Optional] Namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace.
     * @returns true - when at least one of the events was registered otherwise false
     */
    function addPageUnloadEventListener(listener: any, excludeEvents?: string[], evtNamespace?: string | string[]): boolean;

    class AppInsightsCore extends BaseCore implements IAppInsightsCore {
        constructor();
        initialize(config: IConfiguration, extensions: IPlugin[], logger?: IDiagnosticLogger, notificationManager?: INotificationManager): void;
        track(telemetryItem: ITelemetryItem): void;
    }

    function areCookiesSupported(logger?: IDiagnosticLogger): any;

    /**
     * Performs the specified action for each element in an array. This helper exists to avoid adding a polyfil for older browsers
     * that do not define Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page checks for presence/absence of the prototype
     * implementation. Note: For consistency this will not use the Array.prototype.xxxx implementation if it exists as this would
     * cause a testing requirement to test with and without the implementations
     * @param callbackfn  A function that accepts up to three arguments. forEach calls the callbackfn function one time for each element in the array. It can return -1 to break out of the loop
     * @param thisArg  [Optional] An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.
     */
    function arrForEach<T = any>(arr: T[], callbackfn: (value: T, index?: number, array?: T[]) => undefined | void | number, thisArg?: any): void;

    /**
     * Returns the index of the first occurrence of a value in an array. This helper exists to avoid adding a polyfil for older browsers
     * that do not define Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page checks for presence/absence of the prototype
     * implementation. Note: For consistency this will not use the Array.prototype.xxxx implementation if it exists as this would
     * cause a testing requirement to test with and without the implementations
     * @param searchElement The value to locate in the array.
     * @param fromIndex The array index at which to begin the search. If fromIndex is omitted, the search starts at index 0.
     */
    function arrIndexOf<T>(arr: T[], searchElement: T, fromIndex?: number): number;

    /**
     * Calls a defined callback function on each element of an array, and returns an array that contains the results. This helper exists
     * to avoid adding a polyfil for older browsers that do not define Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page
     * checks for presence/absence of the prototype implementation. Note: For consistency this will not use the Array.prototype.xxxx
     * implementation if it exists as this would cause a testing requirement to test with and without the implementations
     * @param callbackfn A function that accepts up to three arguments. The map method calls the callbackfn function one time for each element in the array.
     * @param thisArg An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.
     */
    function arrMap<T, R>(arr: T[], callbackfn: (value: T, index?: number, array?: T[]) => R, thisArg?: any): R[];

    /**
     * Calls the specified callback function for all the elements in an array. The return value of the callback function is the accumulated result, and is
     * provided as an argument in the next call to the callback function. This helper exists to avoid adding a polyfil for older browsers that do not define
     * Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page checks for presence/absence of the prototype implementation. Note: For consistency
     * this will not use the Array.prototype.xxxx implementation if it exists as this would cause a testing requirement to test with and without the implementations
     * @param callbackfn A function that accepts up to four arguments. The reduce method calls the callbackfn function one time for each element in the array.
     * @param initialValue If initialValue is specified, it is used as the initial value to start the accumulation. The first call to the callbackfn function provides this value as an argument instead of an array value.
     */
    function arrReduce<T, R>(arr: T[], callbackfn: (previousValue: T | R, currentValue?: T, currentIndex?: number, array?: T[]) => R, initialValue?: R): R;

    /**
     * Binds the specified function to an event, so that the function gets called whenever the event fires on the object
     * @param obj Object to add the event too.
     * @param eventNameWithoutOn String that specifies any of the standard DHTML Events without "on" prefix and optional (dot "." prefixed) namespaces "click" "click.mynamespace".
     * @param handlerRef Pointer that specifies the function to call when event fires
     * @param useCapture [Optional] Defaults to false
     * @returns True if the function was bound successfully to the event, otherwise false
     */
    function attachEvent(obj: any, eventNameWithoutOn: string, handlerRef: any, useCapture?: boolean): boolean;

    class BaseCore implements IAppInsightsCore {
        static defaultConfig: IConfiguration;
        config: IConfiguration;
        logger: IDiagnosticLogger;
        _extensions: IPlugin[];
        isInitialized: () => boolean;
        constructor();
        initialize(config: IConfiguration, extensions: IPlugin[], logger?: IDiagnosticLogger, notificationManager?: INotificationManager): void;
        getTransmissionControls(): IChannelControls[][];
        track(telemetryItem: ITelemetryItem): void;
        getProcessTelContext(): IProcessTelemetryContext;
        getNotifyMgr(): INotificationManager;
        /**
         * Adds a notification listener. The SDK calls methods on the listener when an appropriate notification is raised.
         * The added plugins must raise notifications. If the plugins do not implement the notifications, then no methods will be
         * called.
         * @param {INotificationListener} listener - An INotificationListener object.
         */
        addNotificationListener(listener: INotificationListener): void;
        /**
         * Removes all instances of the listener.
         * @param {INotificationListener} listener - INotificationListener to remove.
         */
        removeNotificationListener(listener: INotificationListener): void;
        /**
         * Get the current cookie manager for this instance
         */
        getCookieMgr(): ICookieMgr;
        /**
         * Set the current cookie manager for this instance
         * @param cookieMgr - The manager, if set to null/undefined will cause the default to be created
         */
        setCookieMgr(cookieMgr: ICookieMgr): void;
        getPerfMgr(): IPerfManager;
        setPerfMgr(perfMgr: IPerfManager): void;
        eventCnt(): number;
        /**
         * Periodically check logger.queue for
         */
        pollInternalLogs(eventName?: string): number;
        /**
         * Periodically check logger.queue for
         */
        stopPollingInternalLogs(): void;
        /**
         * Add a telemetry processor to decorate or drop telemetry events.
         * @param telemetryInitializer - The Telemetry Initializer function
         * @returns - A ITelemetryInitializerHandler to enable the initializer to be removed
         */
        addTelemetryInitializer(telemetryInitializer: TelemetryInitializerFunction): ITelemetryInitializerHandler | void;
        /**
         * Unload and Tear down the SDK and any initialized plugins, after calling this the SDK will be considered
         * to be un-initialized and non-operational, re-initializing the SDK should only be attempted if the previous
         * unload call return `true` stating that all plugins reported that they also unloaded, the recommended
         * approach is to create a new instance and initialize that instance.
         * This is due to possible unexpected side effects caused by plugins not supporting unload / teardown, unable
         * to successfully remove any global references or they may just be completing the unload process asynchronously.
         * @param isAsync - Can the unload be performed asynchronously (default)
         * @param unloadComplete - An optional callback that will be called once the unload has completed
         * @param cbTimeout - An optional timeout to wait for any flush operations to complete before proceeding with the unload. Defaults to 5 seconds.
         */
        unload(isAsync?: boolean, unloadComplete?: (unloadState: ITelemetryUnloadState) => void, cbTimeout?: number): void;
        getPlugin<T extends IPlugin = IPlugin>(pluginIdentifier: string): ILoadedPlugin<T>;
        /**
         * Add a new plugin to the installation
         * @param plugin - The new plugin to add
         * @param replaceExisting - should any existing plugin be replaced, default is false
         * @param doAsync - Should the add be performed asynchronously
         * @param addCb - [Optional] callback to call after the plugin has been added
         */
        addPlugin<T extends IPlugin = ITelemetryPlugin>(plugin: T, replaceExisting?: boolean, doAsync?: boolean, addCb?: (added?: boolean) => void): void;
        /**
         * Returns the unique event namespace that should be used
         */
        evtNamespace(): string;
        /**
         * Add an unload handler that will be called when the SDK is being unloaded
         * @param handler - the handler
         */
        addUnloadCb(handler: UnloadHandler): void;
        /**
         * Flush and send any batched / cached data immediately
         * @param async - send data asynchronously when true (defaults to true)
         * @param callBack - if specified, notify caller when send is complete, the channel should return true to indicate to the caller that it will be called.
         * If the caller doesn't return true the caller should assume that it may never be called.
         * @param sendReason - specify the reason that you are calling "flush" defaults to ManualFlush (1) if not specified
         * @returns - true if the callback will be return after the flush is complete otherwise the caller should assume that any provided callback will never be called
         */
        flush(isAsync?: boolean, callBack?: (flushComplete?: boolean) => void, sendReason?: SendRequestReason): void;
        /**
         * Gets the current distributed trace context for this instance if available
         * @param createNew - Optional flag to create a new instance if one doesn't currently exist, defaults to true
         */
        getTraceCtx(createNew?: boolean): IDistributedTraceContext | null;
        /**
         * Sets the current distributed trace context for this instance if available
         */
        setTraceCtx(newTracectx: IDistributedTraceContext): void;
        protected releaseQueue(): void;
        /**
         * Hook for Core extensions to allow them to update their own configuration before updating all of the plugins.
         * @param updateCtx - The plugin update context
         * @param updateState - The Update State
         * @returns boolean - True means the extension class will call updateState otherwise the Core will
         */
        protected _updateHook?(updateCtx: IProcessTelemetryUpdateContext, updateState: ITelemetryUpdateState): void | boolean;
    }

    /**
     * BaseTelemetryPlugin provides a basic implementation of the ITelemetryPlugin interface so that plugins
     * can avoid implementation the same set of boiler plate code as well as provide a base
     * implementation so that new default implementations can be added without breaking all plugins.
     */
    abstract class BaseTelemetryPlugin implements ITelemetryPlugin {
        identifier: string;
        version?: string;
        /**
         * Holds the core instance that was used during initialization
         */
        core: IAppInsightsCore;
        priority: number;
        /**
         * Call back for telemetry processing before it it is sent
         * @param env - This is the current event being reported
         * @param itemCtx - This is the context for the current request, ITelemetryPlugin instances
         * can optionally use this to access the current core instance or define / pass additional information
         * to later plugins (vs appending items to the telemetry item)
         */
        processNext: (env: ITelemetryItem, itemCtx: IProcessTelemetryContext) => void;
        /**
         * Set next extension for telemetry processing
         */
        setNextPlugin: (next: ITelemetryPlugin | ITelemetryPluginChain) => void;
        /**
         * Returns the current diagnostic logger that can be used to log issues, if no logger is currently
         * assigned a new default one will be created and returned.
         */
        diagLog: (itemCtx?: IProcessTelemetryContext) => IDiagnosticLogger;
        /**
         * Returns whether the plugin has been initialized
         */
        isInitialized: () => boolean;
        /**
         * Helper to return the current IProcessTelemetryContext, if the passed argument exists this just
         * returns that value (helps with minification for callers), otherwise it will return the configured
         * context or a temporary one.
         * @param currentCtx - [Optional] The current execution context
         */
        protected _getTelCtx: (currentCtx?: IProcessTelemetryContext) => IProcessTelemetryContext;
        /**
         * Internal helper to allow setting of the internal initialized setting for inherited instances and unit testing
         */
        protected setInitialized: (isInitialized: boolean) => void;
        /**
         * Teardown / Unload hook to allow implementations to perform some additional unload operations before the BaseTelemetryPlugin
         * finishes it's removal.
         * @param unloadCtx - This is the context that should be used during unloading.
         * @param unloadState - The details / state of the unload process, it holds details like whether it should be unloaded synchronously or asynchronously and the reason for the unload.
         * @param asyncCallback - An optional callback that the plugin must call if it returns true to inform the caller that it has completed any async unload/teardown operations.
         * @returns boolean - true if the plugin has or will call asyncCallback, this allows the plugin to perform any asynchronous operations.
         */
        protected _doTeardown?: (unloadCtx?: IProcessTelemetryUnloadContext, unloadState?: ITelemetryUnloadState, asyncCallback?: () => void) => void | boolean;
        /**
         * Extension hook to allow implementations to perform some additional update operations before the BaseTelemetryPlugin finishes it's removal
         * @param updateCtx - This is the context that should be used during updating.
         * @param updateState - The details / state of the update process, it holds details like the current and previous configuration.
         * @param asyncCallback - An optional callback that the plugin must call if it returns true to inform the caller that it has completed any async update operations.
         * @returns boolean - true if the plugin has or will call asyncCallback, this allows the plugin to perform any asynchronous operations.
         */
        protected _doUpdate?: (updateCtx?: IProcessTelemetryUpdateContext, updateState?: ITelemetryUpdateState, asyncCallback?: () => void) => void | boolean;
        constructor();
        initialize(config: IConfiguration, core: IAppInsightsCore, extensions: IPlugin[], pluginChain?: ITelemetryPluginChain): void;
        /**
         * Tear down the plugin and remove any hooked value, the plugin should be removed so that it is no longer initialized and
         * therefore could be re-initialized after being torn down. The plugin should ensure that once this has been called any further
         * processTelemetry calls are ignored and it just calls the processNext() with the provided context.
         * @param unloadCtx - This is the context that should be used during unloading.
         * @param unloadState - The details / state of the unload process, it holds details like whether it should be unloaded synchronously or asynchronously and the reason for the unload.
         * @returns boolean - true if the plugin has or will call processNext(), this for backward compatibility as previously teardown was synchronous and returned nothing.
         */
        teardown(unloadCtx?: IProcessTelemetryUnloadContext, unloadState?: ITelemetryUnloadState): void | boolean;
        abstract processTelemetry(env: ITelemetryItem, itemCtx?: IProcessTelemetryContext): void;
        /**
         * The the plugin should re-evaluate configuration and update any cached configuration settings.
         * @param updateCtx - This is the context that should be used during updating.
         * @param updateState - The details / state of the update process, it holds details like the current and previous configuration.
         * @returns boolean - true if the plugin has or will call updateCtx.processNext(), this allows the plugin to perform any asynchronous operations.
         */
        update(updateCtx: IProcessTelemetryUpdateContext, updateState: ITelemetryUpdateState): void | boolean;
        /**
         * Add an unload handler that will be called when the SDK is being unloaded
         * @param handler - the handler
         */
        protected _addUnloadCb(handler: UnloadHandler): void;
        /**
         * Add this hook so that it is automatically removed during unloading
         * @param hooks - The single hook or an array of IInstrumentHook objects
         */
        protected _addHook(hooks: IInstrumentHook | IInstrumentHook[]): void;
    }

    /**
     * @deprecated - Use the core.getCookieMgr().isEnabled()
     * Helper method to tell if document.cookie object is available and whether it can be used.
     */
    function canUseCookies(logger: IDiagnosticLogger): any;

    /**
     * Provides a collection of utility functions, included for backward compatibility with previous releases.
     * @deprecated Marking this instance as deprecated in favor of direct usage of the helper functions
     * as direct usage provides better tree-shaking and minification by avoiding the inclusion of the unused items
     * in your resulting code.
     */
    const CoreUtils: ICoreUtils;

    /**
     * Simpler helper to create a dynamic class that implements the interface and populates the values with the defaults.
     * Only instance properties (hasOwnProperty) values are copied from the defaults to the new instance
     * @param defaults Simple helper
     */
    function createClassFromInterface<T>(defaults?: T): new () => T;

    function createCookieMgr(rootConfig?: IConfiguration, logger?: IDiagnosticLogger): ICookieMgr;

    /**
     * Create a 2 index map that maps an enum's key as both the key and value, X["key"] => "key" and X[0] => "keyof 0".
     * @param values - The values to populate on the new object
     * @returns
     */
    function createEnumMap<E, I = keyof E>(values: {
        [key in keyof E]: E[keyof E];
    }): EnumMap<E, I>;

    /**
     * Create an enum style object which has both the key => value and value => key mappings
     * @param values - The values to populate on the new object
     * @returns
     */
    function createEnumStyle<E>(values: {
        [key in keyof E]: E[keyof E];
    }): EnumValue<E>;

    /**
     * Creates a new Telemetry Item context with the current config, core and plugin execution chain
     * @param plugins - The plugin instances that will be executed
     * @param config - The current config
     * @param core - The current core instance
     * @param startAt - Identifies the next plugin to execute, if null there is no "next" plugin and if undefined it should assume the start of the chain
     */
    function createProcessTelemetryContext(telemetryChain: ITelemetryPluginChain | null, config: IConfiguration, core: IAppInsightsCore, startAt?: IPlugin): IProcessTelemetryContext;

    /**
     * Create a new ITraceParent instance using the provided values.
     * @param traceId - The traceId to use, when invalid a new random W3C id will be generated.
     * @param spanId - The parent/span id to use, a new random value will be generated if it is invalid.
     * @param flags - The traceFlags to use, defaults to zero (0) if not supplied or invalid
     * @param version - The version to used, defaults to version "01" if not supplied or invalid.
     * @returns
     */
    function createTraceParent(traceId?: string, spanId?: string, flags?: number, version?: string): ITraceParent;

    function createUniqueNamespace(name: string, includeVersion?: boolean): string;

    function createUnloadHandlerContainer(): {
        add: (handler: UnloadHandler) => void;
        run: (unloadCtx: IProcessTelemetryUnloadContext, unloadState: ITelemetryUnloadState) => void;
    };

    /**
     * Create a 2 index map that maps an enum's key and value to the defined map value, X["key"] => mapValue and X[0] => mapValue.
     * Generic values
     * - E = the const enum type (typeof eRequestHeaders);
     * - V = Identifies the valid values for the keys, this should include both the enum numeric and string key of the type. The
     * resulting "Value" of each entry identifies the valid values withing the assignments.
     * @param values - The values to populate on the new object
     * @returns
     */
    function createValueMap<E, V = E>(values: {
        [key in keyof E]: [E[keyof E], V[keyof V]];
    }): V;

    /**
     * Return the current time via the Date now() function (if available) and falls back to (new Date()).getTime() if now() is unavailable (IE8 or less)
     * https://caniuse.com/#search=Date.now
     */
    function dateNow(): number;

    function deepFreeze<T>(obj: T): T;

    /**
     * @deprecated - Use the core.getCookieMgr().del()
     * Deletes a cookie by setting it's expiration time in the past.
     * @param name - The name of the cookie to delete.
     */
    function deleteCookie(logger: IDiagnosticLogger, name: string): boolean;

    /**
     * Removes an event handler for the specified event
     * @param Object to remove the event from
     * @param eventNameWithoutOn {string} - The name of the event, with optional namespaces or just the namespaces,
     * such as "click", "click.mynamespace" or ".mynamespace"
     * @param handlerRef {any} - The callback function that needs to be removed from the given event, when using a
     * namespace (with or without a qualifying event) this may be null to remove all previously attached event handlers
     * otherwise this will only remove events with this specific handler.
     * @param useCapture [Optional] Defaults to false
     */
    function detachEvent(obj: any, eventNameWithoutOn: string, handlerRef: any, useCapture?: boolean): void;

    class DiagnosticLogger implements IDiagnosticLogger {
        identifier: string;
        /**
         * The internal logging queue
         */
        queue: _InternalLogMessage[];
        constructor(config?: IConfiguration);
        /**
         * When this is true the SDK will throw exceptions to aid in debugging.
         */
        enableDebugExceptions(): boolean;
        /**
         * 0: OFF (default)
         * 1: CRITICAL
         * 2: >= WARNING
         */
        consoleLoggingLevel(): number;
        /**
         * 0: OFF
         * 1: CRITICAL (default)
         * 2: >= WARNING
         */
        telemetryLoggingLevel(): number;
        /**
         * The maximum number of internal messages allowed to be sent per page view
         */
        maxInternalMessageLimit(): number;
        /**
         * This method will throw exceptions in debug mode or attempt to log the error as a console warning.
         * @param severity {LoggingSeverity} - The severity of the log message
         * @param message {_InternalLogMessage} - The log message.
         */
        throwInternal(severity: LoggingSeverity, msgId: _InternalMessageId, msg: string, properties?: Object, isUserAct?: boolean): void;
        /**
         * This will write a warning to the console if possible
         * @param message {string} - The warning message
         */
        warnToConsole(message: string): void;
        /**
         * This will write an error to the console if possible
         * @param message {string} - The warning message
         */
        errorToConsole(message: string): void;
        /**
         * Resets the internal message count
         */
        resetInternalMessageCount(): void;
        /**
         * Logs a message to the internal queue.
         * @param severity {LoggingSeverity} - The severity of the log message
         * @param message {_InternalLogMessage} - The message to log.
         */
        logInternalMessage(severity: LoggingSeverity, message: _InternalLogMessage): void;
    }

    /**
     * @deprecated - Use the core.getCookieMgr().disable()
     * Force the SDK not to store and read any data from cookies.
     */
    function disableCookies(): void;

    /**
     * Helper function to wrap a function with a perf event
     * @param mgrSource - The Performance Manager or a Performance provider source (may be null)
     * @param getSource - The callback to create the source name for the event (if perf monitoring is enabled)
     * @param func - The function to call and measure
     * @param details - A function to return the payload details
     * @param isAsync - Is the event / function being call asynchronously or synchronously
     */
    function doPerf<T>(mgrSource: IPerfManagerProvider | IPerfManager, getSource: () => string, func: (perfEvt?: IPerfEvent) => T, details?: () => any, isAsync?: boolean): T;

    /**
     * Returns string representation of an object suitable for diagnostics logging.
     */
    function dumpObj(object: any): string;

    /**
     * The eEventsDiscardedReason enumeration contains a set of values that specify the reason for discarding an event.
     */
    const enum eEventsDiscardedReason {
        /**
         * Unknown.
         */
        Unknown = 0,
        /**
         * Status set to non-retryable.
         */
        NonRetryableStatus = 1,
        /**
         * The event is invalid.
         */
        InvalidEvent = 2,
        /**
         * The size of the event is too large.
         */
        SizeLimitExceeded = 3,
        /**
         * The server is not accepting events from this instrumentation key.
         */
        KillSwitch = 4,
        /**
         * The event queue is full.
         */
        QueueFull = 5
    }

    const enum _eInternalMessageId {
        BrowserDoesNotSupportLocalStorage = 0,
        BrowserCannotReadLocalStorage = 1,
        BrowserCannotReadSessionStorage = 2,
        BrowserCannotWriteLocalStorage = 3,
        BrowserCannotWriteSessionStorage = 4,
        BrowserFailedRemovalFromLocalStorage = 5,
        BrowserFailedRemovalFromSessionStorage = 6,
        CannotSendEmptyTelemetry = 7,
        ClientPerformanceMathError = 8,
        ErrorParsingAISessionCookie = 9,
        ErrorPVCalc = 10,
        ExceptionWhileLoggingError = 11,
        FailedAddingTelemetryToBuffer = 12,
        FailedMonitorAjaxAbort = 13,
        FailedMonitorAjaxDur = 14,
        FailedMonitorAjaxOpen = 15,
        FailedMonitorAjaxRSC = 16,
        FailedMonitorAjaxSend = 17,
        FailedMonitorAjaxGetCorrelationHeader = 18,
        FailedToAddHandlerForOnBeforeUnload = 19,
        FailedToSendQueuedTelemetry = 20,
        FailedToReportDataLoss = 21,
        FlushFailed = 22,
        MessageLimitPerPVExceeded = 23,
        MissingRequiredFieldSpecification = 24,
        NavigationTimingNotSupported = 25,
        OnError = 26,
        SessionRenewalDateIsZero = 27,
        SenderNotInitialized = 28,
        StartTrackEventFailed = 29,
        StopTrackEventFailed = 30,
        StartTrackFailed = 31,
        StopTrackFailed = 32,
        TelemetrySampledAndNotSent = 33,
        TrackEventFailed = 34,
        TrackExceptionFailed = 35,
        TrackMetricFailed = 36,
        TrackPVFailed = 37,
        TrackPVFailedCalc = 38,
        TrackTraceFailed = 39,
        TransmissionFailed = 40,
        FailedToSetStorageBuffer = 41,
        FailedToRestoreStorageBuffer = 42,
        InvalidBackendResponse = 43,
        FailedToFixDepricatedValues = 44,
        InvalidDurationValue = 45,
        TelemetryEnvelopeInvalid = 46,
        CreateEnvelopeError = 47,
        CannotSerializeObject = 48,
        CannotSerializeObjectNonSerializable = 49,
        CircularReferenceDetected = 50,
        ClearAuthContextFailed = 51,
        ExceptionTruncated = 52,
        IllegalCharsInName = 53,
        ItemNotInArray = 54,
        MaxAjaxPerPVExceeded = 55,
        MessageTruncated = 56,
        NameTooLong = 57,
        SampleRateOutOfRange = 58,
        SetAuthContextFailed = 59,
        SetAuthContextFailedAccountName = 60,
        StringValueTooLong = 61,
        StartCalledMoreThanOnce = 62,
        StopCalledWithoutStart = 63,
        TelemetryInitializerFailed = 64,
        TrackArgumentsNotSpecified = 65,
        UrlTooLong = 66,
        SessionStorageBufferFull = 67,
        CannotAccessCookie = 68,
        IdTooLong = 69,
        InvalidEvent = 70,
        FailedMonitorAjaxSetRequestHeader = 71,
        SendBrowserInfoOnUserInit = 72,
        PluginException = 73,
        NotificationException = 74,
        SnippetScriptLoadFailure = 99,
        InvalidInstrumentationKey = 100,
        CannotParseAiBlobValue = 101,
        InvalidContentBlob = 102,
        TrackPageActionEventFailed = 103,
        FailedAddingCustomDefinedRequestContext = 104,
        InMemoryStorageBufferFull = 105,
        InstrumentationKeyDeprecation = 106
    }

    const enum eLoggingSeverity {
        /**
         * Error will be sent as internal telemetry
         */
        CRITICAL = 1,
        /**
         * Error will NOT be sent as internal telemetry, and will only be shown in browser console
         */
        WARNING = 2
    }

    type EnumMap<E = any, I = E> = {
        readonly [key in keyof E extends string ? keyof E : never]: key extends string ? key : keyof E;
    } & I;

    type EnumValue<E = any> = {
        readonly [key in keyof E]: E[key];
    };

    const EventHelper: IEventHelper;

    /**
     * Removes an event handler for the specified event
     * @param Object to remove the event from
     * @param eventName {string} - The name of the event, with optional namespaces or just the namespaces,
     * such as "click", "click.mynamespace" or ".mynamespace"
     * @param handlerRef {any} - The callback function that needs to be removed from the given event, when using a
     * namespace (with or without a qualifying event) this may be null to remove all previously attached event handlers
     * otherwise this will only remove events with this specific handler.
     * @param evtNamespace - [Optional] Additional namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace,
     * if the eventName also includes a namespace the namespace(s) are merged into a single namespace
     * @param useCapture [Optional] Defaults to false
     */
    function eventOff<T>(target: T, eventName: string, handlerRef: any, evtNamespace?: string | string[] | null, useCapture?: boolean): void;

    /**
     * Binds the specified function to an event, so that the function gets called whenever the event fires on the object
     * @param obj Object to add the event too.
     * @param eventName String that specifies any of the standard DHTML Events without "on" prefix, if may also include an optional (dot "." prefixed)
     * namespaces "click" "click.mynamespace" in addition to specific namespaces.
     * @param handlerRef Pointer that specifies the function to call when event fires
     * @param evtNamespace - [Optional] Additional namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace,
     * if the eventName also includes a namespace the namespace(s) are merged into a single namespace
     * @param useCapture [Optional] Defaults to false
     * @returns True if the function was bound successfully to the event, otherwise false
     */
    function eventOn<T>(target: T, eventName: string, handlerRef: any, evtNamespace?: string | string[] | null, useCapture?: boolean): boolean;

    /**
     * The EventsDiscardedReason enumeration contains a set of values that specify the reason for discarding an event.
     */
    const EventsDiscardedReason: EnumValue<typeof eEventsDiscardedReason>;

    type EventsDiscardedReason = number | eEventsDiscardedReason;

    /**
     * Helper function to fetch the named meta-tag from the page.
     * @param name
     */
    function findMetaTag(name: string): any;

    /**
     * Helper function to fetch the named server timing value from the page response (first navigation event).
     * @param name
     */
    function findNamedServerTiming(name: string): any;

    /**
     * Helper function to fetch the passed traceparent from the page, looking for it as a meta-tag or a Server-Timing header.
     * @returns
     */
    function findW3cTraceParent(): ITraceParent;

    /**
     * Format the ITraceParent value as a string using the supported and know version formats.
     * So even if the passed traceParent is a later version the string value returned from this
     * function will convert it to only the known version formats.
     * This currently only supports version "00" and invalid "ff"
     * @param value - The parsed traceParent value
     * @returns
     */
    function formatTraceParent(value: ITraceParent): string;

    /**
     * generate W3C trace id
     */
    function generateW3CId(): string;

    /**
     * Returns the global console object
     */
    function getConsole(): Console | null;

    /**
     * @deprecated - Use the core.getCookieMgr().get()
     * helper method to access userId and sessionId cookie
     */
    function getCookie(logger: IDiagnosticLogger, name: string): string;

    /**
     * Returns the crypto object if it is present otherwise null.
     * This helper is used to access the crypto object from the current
     * global instance which could be window or globalThis for a web worker
     */
    function getCrypto(): Crypto | null;

    function getDebugExt(config: IConfiguration): IDbgExtension;

    function getDebugListener(config: IConfiguration): INotificationListener;

    /**
     * Returns the global document object if it is present otherwise null.
     * This helper is used to access the document object without causing an exception
     * "Uncaught ReferenceError: document is not defined"
     */
    function getDocument(): Document | null;

    /**
     * Returns the name of object if it's an Error. Otherwise, returns empty string.
     */
    function getExceptionName(object: any): string;

    const enum GetExtCfgMergeType {
        None = 0,
        MergeDefaultOnly = 1,
        MergeDefaultFromRootOrDefault = 2
    }

    /**
     * Get the current global performance manager that will be used with no performance manager is supplied.
     * @returns - The current default manager
     */
    function getGblPerfMgr(): IPerfManager;

    export { getGlobal }

    /**
     * Return the named global object if available, will return null if the object is not available.
     * @param name The globally named object
     */
    function getGlobalInst<T>(name: string): T;

    /**
     * Returns the global history object if it is present otherwise null.
     * This helper is used to access the history object without causing an exception
     * "Uncaught ReferenceError: history is not defined"
     */
    function getHistory(): History | null;

    /**
     * Gets IE version returning the document emulation mode if we are running on IE, or null otherwise
     */
    function getIEVersion(userAgentStr?: string): number;

    /**
     * Returns the global JSON object if it is present otherwise null.
     * This helper is used to access the JSON object without causing an exception
     * "Uncaught ReferenceError: JSON is not defined"
     */
    function getJSON(): JSON | null;

    /**
     * Returns the global location object if it is present otherwise null.
     * This helper is used to access the location object without causing an exception
     * "Uncaught ReferenceError: location is not defined"
     */
    function getLocation(checkForMock?: boolean): Location | null;

    /**
     * Returns the crypto object if it is present otherwise null.
     * This helper is used to access the crypto object from the current
     * global instance which could be window or globalThis for a web worker
     */
    function getMsCrypto(): Crypto | null;

    /**
     * Returns the global navigator object if it is present otherwise null.
     * This helper is used to access the navigator object without causing an exception
     * "Uncaught ReferenceError: navigator is not defined"
     */
    function getNavigator(): Navigator | null;

    /**
     * Returns the performance object if it is present otherwise null.
     * This helper is used to access the performance object from the current
     * global instance which could be window or globalThis for a web worker
     */
    function getPerformance(): Performance | null;

    /**
     * Returns the current value from the target object if not null or undefined otherwise sets the new value and returns it
     * @param target - The target object to return or set the default value
     * @param field - The key for the field to set on the target
     * @param defValue - [Optional] The value to set if not already present, when not provided a empty object will be added
     */
    function getSetValue<T, K extends keyof T>(target: T, field: K, defValue?: T[K]): T[K];

    /**
     * Returns the global window object if it is present otherwise null.
     * This helper is used to access the window object without causing an exception
     * "Uncaught ReferenceError: window is not defined"
     */
    function getWindow(): Window | null;

    /**
     * Checks if document object is available, this is required as we support the API running without a
     * window /document (eg. Node server, electron webworkers) and if we attempt to assign a document
     * object to a local variable or pass as an argument an "Uncaught ReferenceError: document is not defined"
     * exception will be thrown.
     * Defined as a function to support lazy / late binding environments.
     */
    function hasDocument(): boolean;

    /**
     * Checks if history object is available, this is required as we support the API running without a
     * window /document (eg. Node server, electron webworkers) and if we attempt to assign a history
     * object to a local variable or pass as an argument an "Uncaught ReferenceError: history is not defined"
     * exception will be thrown.
     * Defined as a function to support lazy / late binding environments.
     */
    function hasHistory(): boolean;

    /**
     * Checks if JSON object is available, this is required as we support the API running without a
     * window /document (eg. Node server, electron webworkers) and if we attempt to assign a history
     * object to a local variable or pass as an argument an "Uncaught ReferenceError: JSON is not defined"
     * exception will be thrown.
     * Defined as a function to support lazy / late binding environments.
     */
    function hasJSON(): boolean;

    /**
     * Checks if navigator object is available, this is required as we support the API running without a
     * window /document (eg. Node server, electron webworkers) and if we attempt to assign a navigator
     * object to a local variable or pass as an argument an "Uncaught ReferenceError: navigator is not defined"
     * exception will be thrown.
     * Defined as a function to support lazy / late binding environments.
     */
    function hasNavigator(): boolean;

    function hasOwnProperty(obj: any, prop: string): boolean;

    /**
     * Checks if window object is available, this is required as we support the API running without a
     * window /document (eg. Node server, electron webworkers) and if we attempt to assign a window
     * object to a local variable or pass as an argument an "Uncaught ReferenceError: window is not defined"
     * exception will be thrown.
     * Defined as a function to support lazy / late binding environments.
     */
    function hasWindow(): boolean;

    interface IAppInsightsCore extends IPerfManagerProvider {
        config: IConfiguration;
        logger: IDiagnosticLogger;
        /**
         * Returns a value that indicates whether the instance has already been previously initialized.
         */
        isInitialized?: () => boolean;
        initialize(config: IConfiguration, extensions: IPlugin[], logger?: IDiagnosticLogger, notificationManager?: INotificationManager): void;
        getTransmissionControls(): IChannelControls[][];
        track(telemetryItem: ITelemetryItem): void;
        /**
         * Get the current notification manager
         */
        getNotifyMgr(): INotificationManager;
        /**
         * Get the current cookie manager for this instance
         */
        getCookieMgr(): ICookieMgr;
        /**
         * Set the current cookie manager for this instance
         * @param cookieMgr - The manager, if set to null/undefined will cause the default to be created
         */
        setCookieMgr(cookieMgr: ICookieMgr): void;
        /**
         * Adds a notification listener. The SDK calls methods on the listener when an appropriate notification is raised.
         * The added plugins must raise notifications. If the plugins do not implement the notifications, then no methods will be
         * called.
         * @param {INotificationListener} listener - An INotificationListener object.
         */
        addNotificationListener?(listener: INotificationListener): void;
        /**
         * Removes all instances of the listener.
         * @param {INotificationListener} listener - INotificationListener to remove.
         */
        removeNotificationListener?(listener: INotificationListener): void;
        /**
         * Add a telemetry processor to decorate or drop telemetry events.
         * @param telemetryInitializer - The Telemetry Initializer function
         * @returns - A ITelemetryInitializerHandler to enable the initializer to be removed
         */
        addTelemetryInitializer(telemetryInitializer: TelemetryInitializerFunction): ITelemetryInitializerHandler | void;
        pollInternalLogs?(eventName?: string): number;
        stopPollingInternalLogs?(): void;
        /**
         * Return a new instance of the IProcessTelemetryContext for processing events
         */
        getProcessTelContext(): IProcessTelemetryContext;
        /**
         * Unload and Tear down the SDK and any initialized plugins, after calling this the SDK will be considered
         * to be un-initialized and non-operational, re-initializing the SDK should only be attempted if the previous
         * unload call return `true` stating that all plugins reported that they also unloaded, the recommended
         * approach is to create a new instance and initialize that instance.
         * This is due to possible unexpected side effects caused by plugins not supporting unload / teardown, unable
         * to successfully remove any global references or they may just be completing the unload process asynchronously.
         * @param isAsync - Can the unload be performed asynchronously (default)
         * @param unloadComplete - An optional callback that will be called once the unload has completed
         * @param cbTimeout - An optional timeout to wait for any flush operations to complete before proceeding with the unload. Defaults to 5 seconds.
         */
        unload(isAsync?: boolean, unloadComplete?: (unloadState: ITelemetryUnloadState) => void, cbTimeout?: number): void;
        /**
         * Find and return the (first) plugin with the specified identifier if present
         * @param pluginIdentifier
         */
        getPlugin<T extends IPlugin = IPlugin>(pluginIdentifier: string): ILoadedPlugin<T>;
        /**
         * Add a new plugin to the installation
         * @param plugin - The new plugin to add
         * @param replaceExisting - should any existing plugin be replaced, default is false
         * @param doAsync - Should the add be performed asynchronously
         * @param addCb - [Optional] callback to call after the plugin has been added
         */
        addPlugin<T extends IPlugin = ITelemetryPlugin>(plugin: T, replaceExisting?: boolean, doAsync?: boolean, addCb?: (added?: boolean) => void): void;
        /**
         * Returns the unique event namespace that should be used when registering events
         */
        evtNamespace(): string;
        /**
         * Add a handler that will be called when the SDK is being unloaded
         * @param handler - the handler
         */
        addUnloadCb(handler: UnloadHandler): void;
        /**
         * Flush and send any batched / cached data immediately
         * @param async - send data asynchronously when true (defaults to true)
         * @param callBack - if specified, notify caller when send is complete, the channel should return true to indicate to the caller that it will be called.
         * If the caller doesn't return true the caller should assume that it may never be called.
         * @param sendReason - specify the reason that you are calling "flush" defaults to ManualFlush (1) if not specified
         * @param cbTimeout - An optional timeout to wait for any flush operations to complete before proceeding with the unload. Defaults to 5 seconds.
         * @returns - true if the callback will be return after the flush is complete otherwise the caller should assume that any provided callback will never be called
         */
        flush(isAsync?: boolean, callBack?: (flushComplete?: boolean) => void, sendReason?: SendRequestReason, cbTimeout?: number): boolean | void;
        /**
         * Gets the current distributed trace context for this instance if available
         * @param createNew - Optional flag to create a new instance if one doesn't currently exist, defaults to true
         */
        getTraceCtx(createNew?: boolean): IDistributedTraceContext | null;
        /**
         * Sets the current distributed trace context for this instance if available
         */
        setTraceCtx(newTraceCtx: IDistributedTraceContext | null | undefined): void;
    }

    interface IBaseProcessingContext {
        /**
         * The current core instance for the request
         */
        core: () => IAppInsightsCore;
        /**
         * THe current diagnostic logger for the request
         */
        diagLog: () => IDiagnosticLogger;
        /**
         * Gets the current core config instance
         */
        getCfg: () => IConfiguration;
        /**
         * Gets the named extension config
         */
        getExtCfg: <T>(identifier: string, defaultValue?: T | any, mergeDefault?: GetExtCfgMergeType) => T;
        /**
         * Gets the named config from either the named identifier extension or core config if neither exist then the
         * default value is returned
         * @param identifier The named extension identifier
         * @param field The config field name
         * @param defaultValue The default value to return if no defined config exists
         */
        getConfig: (identifier: string, field: string, defaultValue?: number | string | boolean | string[] | RegExp[] | Function) => number | string | boolean | string[] | RegExp[] | Function;
        /**
         * Helper to allow plugins to check and possibly shortcut executing code only
         * required if there is a nextPlugin
         */
        hasNext: () => boolean;
        /**
         * Returns the next configured plugin proxy
         */
        getNext: () => ITelemetryPluginChain;
        /**
         * Helper to set the next plugin proxy
         */
        setNext: (nextCtx: ITelemetryPluginChain) => void;
        /**
         * Synchronously iterate over the context chain running the callback for each plugin, once
         * every plugin has been executed via the callback, any associated onComplete will be called.
         * @param callback - The function call for each plugin in the context chain
         */
        iterate: <T extends ITelemetryPlugin = ITelemetryPlugin>(callback: (plugin: T) => void) => void;
        /**
         * Set the function to call when the current chain has executed all processNext or unloadNext items.
         * @param onComplete - The onComplete to call
         * @param that - The "this" value to use for the onComplete call, if not provided or undefined defaults to the current context
         * @param args - Any additional arguments to pass to the onComplete function
         */
        onComplete: (onComplete: () => void, that?: any, ...args: any[]) => void;
        /**
         * Create a new context using the core and config from the current instance, returns a new instance of the same type
         * @param plugins - The execution order to process the plugins, if null or not supplied
         *                  then the current execution order will be copied.
         * @param startAt - The plugin to start processing from, if missing from the execution
         *                  order then the next plugin will be NOT set.
         */
        createNew: (plugins?: IPlugin[] | ITelemetryPluginChain, startAt?: IPlugin) => IBaseProcessingContext;
    }

    /**
     * Provides data transmission capabilities
     */
    interface IChannelControls extends ITelemetryPlugin {
        /**
         * Pause sending data
         */
        pause(): void;
        /**
         * Resume sending data
         */
        resume(): void;
        /**
         * Tear down the plugin and remove any hooked value, the plugin should be removed so that it is no longer initialized and
         * therefore could be re-initialized after being torn down. The plugin should ensure that once this has been called any further
         * processTelemetry calls are ignored and it just calls the processNext() with the provided context.
         * @param unloadCtx - This is the context that should be used during unloading.
         * @param unloadState - The details / state of the unload process, it holds details like whether it should be unloaded synchronously or asynchronously and the reason for the unload.
         * @returns boolean - true if the plugin has or will call processNext(), this for backward compatibility as previously teardown was synchronous and returned nothing.
         */
        teardown: (unloadCtx?: IProcessTelemetryUnloadContext, unloadState?: ITelemetryUnloadState) => void | boolean;
        /**
         * Flush to send data immediately; channel should default to sending data asynchronously
         * @param async - send data asynchronously when true
         * @param callBack - if specified, notify caller when send is complete, the channel should return true to indicate to the caller that it will be called.
         * If the caller doesn't return true the caller should assume that it may never be called.
         * @param sendReason - specify the reason that you are calling "flush" defaults to ManualFlush (1) if not specified
         * @returns - true if the callback will be return after the flush is complete otherwise the caller should assume that any provided callback will never be called
         */
        flush(async: boolean, callBack?: (flushComplete?: boolean) => void, sendReason?: SendRequestReason): boolean | void;
    }

    /**
     * Configuration provided to SDK core
     */
    interface IConfiguration {
        /**
         * Instrumentation key of resource. Either this or connectionString must be specified.
         */
        instrumentationKey?: string;
        /**
         * Connection string of resource. Either this or instrumentationKey must be specified.
         */
        connectionString?: string;
        /**
         * Polling interval (in ms) for internal logging queue
         */
        diagnosticLogInterval?: number;
        /**
         * Maximum number of iKey transmitted logging telemetry per page view
         */
        maxMessageLimit?: number;
        /**
         * Console logging level. All logs with a severity level higher
         * than the configured level will be printed to console. Otherwise
         * they are suppressed. ie Level 2 will print both CRITICAL and
         * WARNING logs to console, level 1 prints only CRITICAL.
         *
         * Note: Logs sent as telemetry to instrumentation key will also
         * be logged to console if their severity meets the configured loggingConsoleLevel
         *
         * 0: ALL console logging off
         * 1: logs to console: severity >= CRITICAL
         * 2: logs to console: severity >= WARNING
         */
        loggingLevelConsole?: number;
        /**
         * Telemtry logging level to instrumentation key. All logs with a severity
         * level higher than the configured level will sent as telemetry data to
         * the configured instrumentation key.
         *
         * 0: ALL iKey logging off
         * 1: logs to iKey: severity >= CRITICAL
         * 2: logs to iKey: severity >= WARNING
         */
        loggingLevelTelemetry?: number;
        /**
         * If enabled, uncaught exceptions will be thrown to help with debugging
         */
        enableDebugExceptions?: boolean;
        /**
         * Endpoint where telemetry data is sent
         */
        endpointUrl?: string;
        /**
         * Extension configs loaded in SDK
         */
        extensionConfig?: {
            [key: string]: any;
        };
        /**
         * Additional plugins that should be loaded by core at runtime
         */
        extensions?: ITelemetryPlugin[];
        /**
         * Channel queues that is setup by caller in desired order.
         * If channels are provided here, core will ignore any channels that are already setup, example if there is a SKU with an initialized channel
         */
        channels?: IChannelControls[][];
        /**
         * @type {boolean}
         * Flag that disables the Instrumentation Key validation.
         */
        disableInstrumentationKeyValidation?: boolean;
        /**
         * [Optional] When enabled this will create local perfEvents based on sections of the code that have been instrumented
         * to emit perfEvents (via the doPerf()) when this is enabled. This can be used to identify performance issues within
         * the SDK, the way you are using it or optionally your own instrumented code.
         * The provided IPerfManager implementation does NOT send any additional telemetry events to the server it will only fire
         * the new perfEvent() on the INotificationManager which you can listen to.
         * This also does not use the window.performance API, so it will work in environments where this API is not supported.
         */
        enablePerfMgr?: boolean;
        /**
         * [Optional] Callback function that will be called to create a the IPerfManager instance when required and ```enablePerfMgr```
         * is enabled, this enables you to override the default creation of a PerfManager() without needing to ```setPerfMgr()```
         * after initialization.
         */
        createPerfMgr?: (core: IAppInsightsCore, notificationManager: INotificationManager) => IPerfManager;
        /**
         * [Optional] Fire every single performance event not just the top level root performance event. Defaults to false.
         */
        perfEvtsSendAll?: boolean;
        /**
         * [Optional] Identifies the default length used to generate random session and user id's if non currently exists for the user / session.
         * Defaults to 22, previous default value was 5, if you need to keep the previous maximum length you should set this value to 5.
         */
        idLength?: number;
        /**
         * @description Custom cookie domain. This is helpful if you want to share Application Insights cookies across subdomains. It
         * can be set here or as part of the cookieCfg.domain, the cookieCfg takes precedence if both are specified.
         * @type {string}
         * @defaultValue ""
         */
        cookieDomain?: string;
        /**
         * @description Custom cookie path. This is helpful if you want to share Application Insights cookies behind an application
         * gateway. It can be set here or as part of the cookieCfg.domain, the cookieCfg takes precedence if both are specified.
         * @type {string}
         * @defaultValue ""
         */
        cookiePath?: string;
        /**
         * [Optional] A boolean that indicated whether to disable the use of cookies by the SDK. If true, the SDK will not store or
         * read any data from cookies. Cookie usage can be re-enabled after initialization via the core.getCookieMgr().enable().
         */
        disableCookiesUsage?: boolean;
        /**
         * [Optional] A Cookie Manager configuration which includes hooks to allow interception of the get, set and delete cookie
         * operations. If this configuration is specified any specified enabled and domain properties will take precedence over the
         * cookieDomain and disableCookiesUsage values.
         */
        cookieCfg?: ICookieMgrConfig;
        /**
         * [Optional] An array of the page unload events that you would like to be ignored, special note there must be at least one valid unload
         * event hooked, if you list all or the runtime environment only supports a listed "disabled" event it will still be hooked, if required by the SDK.
         * Unload events include "beforeunload", "unload", "visibilitychange" (with 'hidden' state) and "pagehide"
         */
        disablePageUnloadEvents?: string[];
        /**
         * [Optional] An array of page show events that you would like to be ignored, special note there must be at lease one valid show event
         * hooked, if you list all or the runtime environment only supports a listed (disabled) event it will STILL be hooked, if required by the SDK.
         * Page Show events include "pageshow" and "visibilitychange" (with 'visible' state)
         */
        disablePageShowEvents?: string[];
        /**
         * [Optional] A flag for performance optimization to disable attempting to use the Chrome Debug Extension, if disabled and the extension is installed
         * this will not send any notifications.
         */
        disableDbgExt?: boolean;
    }

    interface ICookieMgr {
        /**
         * Enable or Disable the usage of cookies
         */
        setEnabled(value: boolean): void;
        /**
         * Can the system use cookies, if this returns false then all cookie setting and access functions will return nothing
         */
        isEnabled(): boolean;
        /**
         * Set the named cookie with the value and optional domain and optional
         * @param name - The name of the cookie
         * @param value - The value of the cookie (Must already be encoded)
         * @param maxAgeSec - [optional] The maximum number of SECONDS that this cookie should survive
         * @param domain - [optional] The domain to set for the cookie
         * @param path - [optional] Path to set for the cookie, if not supplied will default to "/"
         * @returns - True if the cookie was set otherwise false (Because cookie usage is not enabled or available)
         */
        set(name: string, value: string, maxAgeSec?: number, domain?: string, path?: string): boolean;
        /**
         * Get the value of the named cookie
         * @param name - The name of the cookie
         */
        get(name: string): string;
        /**
         * Delete/Remove the named cookie if cookie support is available and enabled.
         * Note: Not using "delete" as the name because it's a reserved word which would cause issues on older browsers
         * @param name - The name of the cookie
         * @param path - [optional] Path to set for the cookie, if not supplied will default to "/"
         * @returns - True if the cookie was marked for deletion otherwise false (Because cookie usage is not enabled or available)
         */
        del(name: string, path?: string): boolean;
        /**
         * Purge the cookie from the system if cookie support is available, this function ignores the enabled setting of the manager
         * so any cookie will be removed.
         * Note: Not using "delete" as the name because it's a reserved word which would cause issues on older browsers
         * @param name - The name of the cookie
         * @param path - [optional] Path to set for the cookie, if not supplied will default to "/"
         * @returns - True if the cookie was marked for deletion otherwise false (Because cookie usage is not available)
         */
        purge(name: string, path?: string): boolean;
    }

    /**
     * Configuration definition for instance based cookie management configuration
     */
    interface ICookieMgrConfig {
        /**
         * Defaults to true, A boolean that indicates whether the use of cookies by the SDK is enabled by the current instance.
         * If false, the instance of the SDK initialized by this configuration will not store or read any data from cookies
         */
        enabled?: boolean;
        /**
         * Custom cookie domain. This is helpful if you want to share Application Insights cookies across subdomains.
         */
        domain?: string;
        /**
         * Specifies the path to use for the cookie, defaults to '/'
         */
        path?: string;
        /**
         * Hook function to fetch the named cookie value.
         * @param name - The name of the cookie
         */
        getCookie?: (name: string) => string;
        /**
         * Hook function to set the named cookie with the specified value.
         * @param name - The name of the cookie
         * @param value - The value to set for the cookie
         */
        setCookie?: (name: string, value: string) => void;
        /**
         * Hook function to delete the named cookie with the specified value, separated from
         * setCookie to avoid the need to parse the value to determine whether the cookie is being
         * added or removed.
         * @param name - The name of the cookie
         * @param cookieValue - The value to set to expire the cookie
         */
        delCookie?: (name: string, cookieValue: string) => void;
    }

    /**
     * Provides a collection of utility functions, included for backward compatibility with previous releases.
     * @deprecated Marking this interface and instance as deprecated in favor of direct usage of the helper functions
     * as direct usage provides better tree-shaking and minification by avoiding the inclusion of the unused items
     * in your resulting code.
     */
    interface ICoreUtils {
        /**
         * Internal - Do not use directly.
         * @deprecated Direct usage of this property is not recommend
         */
        _canUseCookies: boolean;
        isTypeof: (value: any, theType: string) => boolean;
        isUndefined: (value: any) => boolean;
        isNullOrUndefined: (value: any) => boolean;
        hasOwnProperty: (obj: any, prop: string) => boolean;
        /**
         * Checks if the passed of value is a function.
         * @param {any} value - Value to be checked.
         * @return {boolean} True if the value is a boolean, false otherwise.
         */
        isFunction: (value: any) => value is Function;
        /**
         * Checks if the passed of value is a function.
         * @param {any} value - Value to be checked.
         * @return {boolean} True if the value is a boolean, false otherwise.
         */
        isObject: (value: any) => boolean;
        /**
         * Check if an object is of type Date
         */
        isDate: (obj: any) => obj is Date;
        /**
         * Check if an object is of type Array
         */
        isArray: (obj: any) => boolean;
        /**
         * Check if an object is of type Error
         */
        isError: (obj: any) => obj is Error;
        /**
         * Checks if the type of value is a string.
         * @param {any} value - Value to be checked.
         * @return {boolean} True if the value is a string, false otherwise.
         */
        isString: (value: any) => value is string;
        /**
         * Checks if the type of value is a number.
         * @param {any} value - Value to be checked.
         * @return {boolean} True if the value is a number, false otherwise.
         */
        isNumber: (value: any) => value is number;
        /**
         * Checks if the type of value is a boolean.
         * @param {any} value - Value to be checked.
         * @return {boolean} True if the value is a boolean, false otherwise.
         */
        isBoolean: (value: any) => value is boolean;
        /**
         * Convert a date to I.S.O. format in IE8
         */
        toISOString: (date: Date) => string;
        /**
         * Performs the specified action for each element in an array. This helper exists to avoid adding a polyfil for older browsers
         * that do not define Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page checks for presence/absence of the prototype
         * implementation. Note: For consistency this will not use the Array.prototype.xxxx implementation if it exists as this would
         * cause a testing requirement to test with and without the implementations
         * @param callbackfn  A function that accepts up to three arguments. forEach calls the callbackfn function one time for each element in the array. It can return -1 to break out of the loop
         * @param thisArg  [Optional] An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.
         */
        arrForEach: <T>(arr: T[], callbackfn: (value: T, index?: number, array?: T[]) => void | number, thisArg?: any) => void;
        /**
         * Returns the index of the first occurrence of a value in an array. This helper exists to avoid adding a polyfil for older browsers
         * that do not define Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page checks for presence/absence of the prototype
         * implementation. Note: For consistency this will not use the Array.prototype.xxxx implementation if it exists as this would
         * cause a testing requirement to test with and without the implementations
         * @param searchElement The value to locate in the array.
         * @param fromIndex The array index at which to begin the search. If fromIndex is omitted, the search starts at index 0.
         */
        arrIndexOf: <T>(arr: T[], searchElement: T, fromIndex?: number) => number;
        /**
         * Calls a defined callback function on each element of an array, and returns an array that contains the results. This helper exists
         * to avoid adding a polyfil for older browsers that do not define Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page
         * checks for presence/absence of the prototype implementation. Note: For consistency this will not use the Array.prototype.xxxx
         * implementation if it exists as this would cause a testing requirement to test with and without the implementations
         * @param callbackfn A function that accepts up to three arguments. The map method calls the callbackfn function one time for each element in the array.
         * @param thisArg An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.
         */
        arrMap: <T, R>(arr: T[], callbackfn: (value: T, index?: number, array?: T[]) => R, thisArg?: any) => R[];
        /**
         * Calls the specified callback function for all the elements in an array. The return value of the callback function is the accumulated result, and is
         * provided as an argument in the next call to the callback function. This helper exists to avoid adding a polyfil for older browsers that do not define
         * Array.prototype.xxxx (eg. ES3 only, IE8) just in case any page checks for presence/absence of the prototype implementation. Note: For consistency
         * this will not use the Array.prototype.xxxx implementation if it exists as this would cause a testing requirement to test with and without the implementations
         * @param callbackfn A function that accepts up to four arguments. The reduce method calls the callbackfn function one time for each element in the array.
         * @param initialValue If initialValue is specified, it is used as the initial value to start the accumulation. The first call to the callbackfn function provides this value as an argument instead of an array value.
         */
        arrReduce: <T, R>(arr: T[], callbackfn: (previousValue: T | R, currentValue?: T, currentIndex?: number, array?: T[]) => R, initialValue?: R) => R;
        /**
         * helper method to trim strings (IE8 does not implement String.prototype.trim)
         */
        strTrim: (str: any) => string;
        /**
         * Creates an object that has the specified prototype, and that optionally contains specified properties. This helper exists to avoid adding a polyfil
         * for older browsers that do not define Object.create eg. ES3 only, IE8 just in case any page checks for presence/absence of the prototype implementation.
         * Note: For consistency this will not use the Object.create implementation if it exists as this would cause a testing requirement to test with and without the implementations
         * @param obj Object to use as a prototype. May be null
         */
        objCreate: (obj: object) => any;
        /**
         * Returns the names of the enumerable string properties and methods of an object. This helper exists to avoid adding a polyfil for older browsers
         * that do not define Object.keys eg. ES3 only, IE8 just in case any page checks for presence/absence of the prototype implementation.
         * Note: For consistency this will not use the Object.keys implementation if it exists as this would cause a testing requirement to test with and without the implementations
         * @param obj Object that contains the properties and methods. This can be an object that you created or an existing Document Object Model (DOM) object.
         */
        objKeys: (obj: {}) => string[];
        /**
         * Try to define get/set object property accessors for the target object/prototype, this will provide compatibility with
         * existing API definition when run within an ES5+ container that supports accessors but still enable the code to be loaded
         * and executed in an ES3 container, providing basic IE8 compatibility.
         * @param target The object on which to define the property.
         * @param prop The name of the property to be defined or modified.
         * @param getProp The getter function to wire against the getter.
         * @param setProp The setter function to wire against the setter.
         * @returns True if it was able to create the accessors otherwise false
         */
        objDefineAccessors: <T>(target: any, prop: string, getProp?: () => T, setProp?: (v: T) => void) => boolean;
        /**
         * Trys to add an event handler for the specified event to the window, body and document
         * @param eventName {string} - The name of the event
         * @param callback {any} - The callback function that needs to be executed for the given event
         * @return {boolean} - true if the handler was successfully added
         */
        addEventHandler: (eventName: string, callback: any, evtNamespace?: string | string[]) => boolean;
        /**
         * Return the current time via the Date now() function (if available) and falls back to (new Date()).getTime() if now() is unavailable (IE8 or less)
         * https://caniuse.com/#search=Date.now
         */
        dateNow: () => number;
        /**
         * Identifies whether the current environment appears to be IE
         */
        isIE: () => boolean;
        /**
         * @deprecated - Use the core.getCookieMgr().disable()
         * Force the SDK not to store and read any data from cookies.
         */
        disableCookies: () => void;
        newGuid: () => string;
        /**
         * Return the current value of the Performance Api now() function (if available) and fallback to dateNow() if it is unavailable (IE9 or less)
         * https://caniuse.com/#search=performance.now
         */
        perfNow: () => number;
        /**
         * Generate random base64 id string.
         * The default length is 22 which is 132-bits so almost the same as a GUID but as base64 (the previous default was 5)
         * @param maxLength - Optional value to specify the length of the id to be generated, defaults to 22
         */
        newId: (maxLength?: number) => string;
        /**
         * Generate a random value between 0 and maxValue, max value should be limited to a 32-bit maximum.
         * So maxValue(16) will produce a number from 0..16 (range of 17)
         * @param maxValue
         */
        randomValue: (maxValue: number) => number;
        /**
         * generate a random 32-bit number (0x000000..0xFFFFFFFF) or (-0x80000000..0x7FFFFFFF), defaults un-unsigned.
         * @param signed - True to return a signed 32-bit number (-0x80000000..0x7FFFFFFF) otherwise an unsigned one (0x000000..0xFFFFFFFF)
         */
        random32: (signed?: boolean) => number;
        /**
         * Seed the MWC random number generator with the specified seed or a random value
         * @param value - optional the number to used as the seed, if undefined, null or zero a random value will be chosen
         */
        mwcRandomSeed: (value?: number) => void;
        /**
         * Generate a random 32-bit number between (0x000000..0xFFFFFFFF) or (-0x80000000..0x7FFFFFFF), using MWC (Multiply with carry)
         * instead of Math.random() defaults to un-signed.
         * Used as a replacement random generator for IE to avoid issues with older IE instances.
         * @param signed - True to return a signed 32-bit number (-0x80000000..0x7FFFFFFF) otherwise an unsigned one (0x000000..0xFFFFFFFF)
         */
        mwcRandom32: (signed?: boolean) => number;
        /**
         * generate W3C trace id
         */
        generateW3CId: () => string;
    }

    interface ICustomProperties {
        [key: string]: any;
    }

    interface IDbgExtension {
        isEnabled: () => boolean;
        enable: () => void;
        disable: () => void;
        listener: INotificationListener;
        sendEvt?: (name: string, data: any) => void;
        debugMsg?: (name: string, data: any) => void;
        diagLog?: (name: string, data: any) => void;
    }

    interface IDiagnosticLogger {
        /**
         * When this is true the SDK will throw exceptions to aid in debugging.
         */
        enableDebugExceptions: () => boolean;
        /**
         * 0: OFF
         * 1: only critical (default)
         * 2: critical + info
         */
        consoleLoggingLevel: () => number;
        /**
         * 0: OFF (default)
         * 1: CRITICAL
         * 2: WARNING
         */
        telemetryLoggingLevel: () => number;
        /**
         * The maximum number of internal messages allowed to be sent per page view
         */
        maxInternalMessageLimit: () => number;
        /**
         * The internal logging queue
         */
        queue: _InternalLogMessage[];
        /**
         * This method will throw exceptions in debug mode or attempt to log the error as a console warning.
         * @param severity {LoggingSeverity} - The severity of the log message
         * @param message {_InternalLogMessage} - The log message.
         */
        throwInternal(severity: LoggingSeverity, msgId: _InternalMessageId, msg: string, properties?: Object, isUserAct?: boolean): void;
        /**
         * This will write a warning to the console if possible
         * @param message {string} - The warning message
         */
        warnToConsole(message: string): void;
        /**
         * This will write an error to the console if possible.
         * Provided by the default DiagnosticLogger instance, and internally the SDK will fall back to warnToConsole, however,
         * direct callers MUST check for its existence on the logger as you can provide your own IDiagnosticLogger instance.
         * @param message {string} - The error message
         */
        errorToConsole?(message: string): void;
        /**
         * Resets the internal message count
         */
        resetInternalMessageCount(): void;
        /**
         * Logs a message to the internal queue.
         * @param severity {LoggingSeverity} - The severity of the log message
         * @param message {_InternalLogMessage} - The message to log.
         */
        logInternalMessage?(severity: LoggingSeverity, message: _InternalLogMessage): void;
    }

    interface IDistributedTraceContext {
        /**
         * Returns the current name of the page
         */
        getName(): string;
        /**
         * Sets the current name of the page
         * @param pageName
         */
        setName(pageName: string): void;
        /**
         * Returns the unique identifier for a trace. All requests / spans from the same trace share the same traceId.
         * Must be read from incoming headers or generated according to the W3C TraceContext specification,
         * in a hex representation of 16-byte array. A.k.a. trace-id, TraceID or Distributed TraceID
         */
        getTraceId(): string;
        /**
         * Set the unique identifier for a trace. All requests / spans from the same trace share the same traceId.
         * Must be conform to the W3C TraceContext specification, in a hex representation of 16-byte array.
         * A.k.a. trace-id, TraceID or Distributed TraceID https://www.w3.org/TR/trace-context/#trace-id
         */
        setTraceId(newValue: string): void;
        /**
         * Self-generated 8-bytes identifier of the incoming request. Must be a hex representation of 8-byte array.
         * Also know as the parentId, used to link requests together
         */
        getSpanId(): string;
        /**
         * Self-generated 8-bytes identifier of the incoming request. Must be a hex representation of 8-byte array.
         * Also know as the parentId, used to link requests together
         * https://www.w3.org/TR/trace-context/#parent-id
         */
        setSpanId(newValue: string): void;
        /**
         * An integer representation of the W3C TraceContext trace-flags.
         */
        getTraceFlags(): number | undefined;
        /**
         * https://www.w3.org/TR/trace-context/#trace-flags
         * @param newValue
         */
        setTraceFlags(newValue?: number): void;
    }

    interface IEventHelper {
        /**
         * Binds the specified function to an event, so that the function gets called whenever the event fires on the object
         * @param obj Object to add the event too.
         * @param eventNameWithoutOn String that specifies any of the standard DHTML Events without "on" prefix
         * @param handlerRef Pointer that specifies the function to call when event fires
         * @returns True if the function was bound successfully to the event, otherwise false
         */
        Attach: (obj: any, eventNameWithoutOn: string, handlerRef: any) => boolean;
        /**
         * Binds the specified function to an event, so that the function gets called whenever the event fires on the object
         * @deprecated Use {@link EventHelper#Attach} as we are already in a class call EventHelper the extra "Event" just causes a larger result
         * @param obj Object to add the event too.
         * @param eventNameWithoutOn String that specifies any of the standard DHTML Events without "on" prefix
         * @param handlerRef Pointer that specifies the function to call when event fires
         * @returns True if the function was bound successfully to the event, otherwise false
         */
        AttachEvent: (obj: any, eventNameWithoutOn: string, handlerRef: any) => boolean;
        /**
         * Removes an event handler for the specified event
         * @param eventName {string} - The name of the event
         * @param callback {any} - The callback function that needs to be executed for the given event
         * @return {boolean} - true if the handler was successfully added
         */
        Detach: (obj: any, eventNameWithoutOn: string, handlerRef: any) => void;
        /**
         * Removes an event handler for the specified event
         * @deprecated Use {@link EventHelper#Detach} as we are already in a class call EventHelper the extra "Event" just causes a larger result
         * @param eventName {string} - The name of the event
         * @param callback {any} - The callback function that needs to be executed for the given event
         * @return {boolean} - true if the handler was successfully added
         */
        DetachEvent: (obj: any, eventNameWithoutOn: string, handlerRef: any) => void;
    }

    interface IInstrumentCallDetails {
        name: string;
        inst: any;
        /**
         * This returns an object that the hook function can use to store hook specific
         * context, it it not shared with any other hook instances and is unique for the
         * current call.
         * A hook implementation can use this to pass / share context between different
         * hook callbacks eg. request/response requst/hookErrors etc.
         */
        ctx: () => any;
        /**
         * Allows the hook functions to replace the original arguments
         * @param idx - The argument index (0 based)
         * @param value - The new value for the argument
         */
        set: (idx: number, value: any) => void;
        /**
         * The result of the original method, only populated after the original method has returned
         */
        rslt?: any;
        /**
         * The error (exception) which occurred while executing the original method
         */
        err?: Error;
        /**
         * The Event object from (window.event) at the start of the original call
         */
        evt?: Event;
    }

    /**
     * The holder of the specific instance callback
     */
    interface IInstrumentHook {
        /** Unique Id for this callback on the hooked method */
        id: number;
        /** Holds the callbacks */
        cbks: IInstrumentHooksCallbacks;
        /** Remove this hook from the function */
        rm: () => void;
    }

    interface IInstrumentHooks {
        i: number;
        n: string;
        f: any;
        h: IInstrumentHook[];
    }

    /**
     * The callbacks to call for the instrumented function, you must provide at least the request and/or response callbacks, both are not required.
     * You must always supply the error callback
     */
    interface IInstrumentHooksCallbacks {
        /**
         * [Optional] Namespace details (same as the namespace used for events), useful for debugging and testing to
         * identify the source of the instrumented hooks
         */
        ns?: string | string[];
        /**
         * The hook callback to call before the original function is called
         */
        req?: InstrumentorHooksCallback;
        /**
         * The hook callback to call after the original function was called
         */
        rsp?: InstrumentorHooksCallback;
        /**
         * The callback to call if the hook function causes an exception
         */
        hkErr?: InstrumentorHooksCallback;
        /**
         * The callback to call if the original function causes an exception, even if you
         * supply a callback the original exception will still be thrown
         */
        fnErr?: InstrumentorHooksCallback;
    }

    interface ILoadedPlugin<T extends IPlugin> {
        plugin: T;
        /**
         * Identifies whether the plugin is enabled and can process events. This is slightly different from isInitialized as the plugin may be initialized but disabled
         * via the setEnabled() or it may be a shared plugin which has had it's teardown function called from another instance..
         * @returns boolean = true if the plugin is in a state where it is operational.
         */
        isEnabled: () => boolean;
        /**
         * You can optionally enable / disable a plugin from processing events.
         * Setting enabled to true will not necessarily cause the `isEnabled()` to also return true
         * as the plugin must also have been successfully initialized and not had it's `teardown` method called
         * (unless it's also been re-initialized)
         */
        setEnabled: (isEnabled: boolean) => void;
        remove: (isAsync?: boolean, removeCb?: (removed?: boolean) => void) => void;
    }

    /**
     * Initialize the queue of plugins
     * @param plugins - The array of plugins to initialize and setting of the next plugin
     * @param config The current config for the instance
     * @param core THe current core instance
     * @param extensions The extensions
     */
    function initializePlugins(processContext: IProcessTelemetryContext, extensions: IPlugin[]): void;

    /**
     * An interface used for the notification listener.
     * @interface
     */
    interface INotificationListener {
        /**
         * [Optional] A function called when events are sent.
         * @param {ITelemetryItem[]} events - The array of events that have been sent.
         */
        eventsSent?: (events: ITelemetryItem[]) => void;
        /**
         * [Optional] A function called when events are discarded.
         * @param {ITelemetryItem[]} events - The array of events that have been discarded.
         * @param {number} reason           - The reason for discarding the events. The EventsDiscardedReason
         * constant should be used to check the different values.
         */
        eventsDiscarded?: (events: ITelemetryItem[], reason: number) => void;
        /**
         * [Optional] A function called when the events have been requested to be sent to the sever.
         * @param {number} sendReason - The reason why the event batch is being sent.
         * @param {boolean} isAsync   - A flag which identifies whether the requests are being sent in an async or sync manner.
         */
        eventsSendRequest?: (sendReason: number, isAsync?: boolean) => void;
        /**
         * [Optional] This event is sent if you have enabled perf events, they are primarily used to track internal performance testing and debugging
         * the event can be displayed via the debug plugin extension.
         * @param perfEvent
         */
        perfEvent?: (perfEvent: IPerfEvent) => void;
    }

    /**
     * Class to manage sending notifications to all the listeners.
     */
    interface INotificationManager {
        listeners: INotificationListener[];
        /**
         * Adds a notification listener.
         * @param {INotificationListener} listener - The notification listener to be added.
         */
        addNotificationListener(listener: INotificationListener): void;
        /**
         * Removes all instances of the listener.
         * @param {INotificationListener} listener - AWTNotificationListener to remove.
         */
        removeNotificationListener(listener: INotificationListener): void;
        /**
         * Notification for events sent.
         * @param {ITelemetryItem[]} events - The array of events that have been sent.
         */
        eventsSent(events: ITelemetryItem[]): void;
        /**
         * Notification for events being discarded.
         * @param {ITelemetryItem[]} events - The array of events that have been discarded by the SDK.
         * @param {number} reason           - The reason for which the SDK discarded the events. The EventsDiscardedReason
         * constant should be used to check the different values.
         */
        eventsDiscarded(events: ITelemetryItem[], reason: number): void;
        /**
         * [Optional] A function called when the events have been requested to be sent to the sever.
         * @param {number} sendReason - The reason why the event batch is being sent.
         * @param {boolean} isAsync   - A flag which identifies whether the requests are being sent in an async or sync manner.
         */
        eventsSendRequest?(sendReason: number, isAsync: boolean): void;
        /**
         * [Optional] This event is sent if you have enabled perf events, they are primarily used to track internal performance testing and debugging
         * the event can be displayed via the debug plugin extension.
         * @param perfEvent
         */
        perfEvent?(perfEvent: IPerfEvent): void;
    }

    /**
     * Add an instrumentation hook to the provided named "event" for the target class / object, this doesn't check whether the
     * named "event" is in fact a function and just assigns the instrumentation hook to the target[evtName]
     * @param target - The target object
     * @param evtName - The name of the event
     * @param callbacks - The callbacks to configure and call whenever the function is called
     * @param checkPrototype - If the function doesn't exist on the target should it attempt to hook the prototype function
     */
    function InstrumentEvent(target: any, evtName: string, callbacks: IInstrumentHooksCallbacks, checkPrototype?: boolean): IInstrumentHook;

    /**
     * Intercept the named prototype functions for the target class / object
     * @param target - The target object
     * @param funcName - The function name
     * @param callbacks - The callbacks to configure and call whenever the function is called
     * @param checkPrototype - If the function doesn't exist on the target should it attempt to hook the prototype function
     */
    function InstrumentFunc(target: any, funcName: string, callbacks: IInstrumentHooksCallbacks, checkPrototype?: boolean): IInstrumentHook;

    /**
     * Intercept the named functions for the target class / object
     * @param target - The target object
     * @param funcNames - The function names to intercept and call
     * @param callbacks - The callbacks to configure and call whenever the function is called
     * @param checkPrototype - If the function doesn't exist on the target should it attempt to hook the prototype function
     */
    function InstrumentFuncs(target: any, funcNames: string[], callbacks: IInstrumentHooksCallbacks, checkPrototype?: boolean): IInstrumentHook[];

    /**
     * A callback function that will be called for the wrapped instrumentation function
     * before the original function is executed.
     */
    type InstrumentorHooksCallback = (funcArgs: IInstrumentCallDetails, ...orgArgs: any[]) => void;

    /**
     * Intercept the named prototype functions for the target class / object
     * @param target - The target object
     * @param funcName - The function name
     * @param callbacks - The callbacks to configure and call whenever the function is called
     */
    function InstrumentProto(target: any, funcName: string, callbacks: IInstrumentHooksCallbacks): IInstrumentHook;

    /**
     * Intercept the named prototype functions for the target class / object
     * @param target - The target object
     * @param funcNames - The function names to intercept and call
     * @param callbacks - The callbacks to configure and call whenever the function is called
     */
    function InstrumentProtos(target: any, funcNames: string[], callbacks: IInstrumentHooksCallbacks): IInstrumentHook[];

    class _InternalLogMessage {
        static dataType: string;
        message: string;
        messageId: _InternalMessageId;
        constructor(msgId: _InternalMessageId, msg: string, isUserAct?: boolean, properties?: Object);
    }

    /**
     * Internal message ID. Please create a new one for every conceptually different message. Please keep alphabetically ordered
     */
    const _InternalMessageId: EnumValue<typeof _eInternalMessageId>;

    type _InternalMessageId = number | _eInternalMessageId;

    /**
     * This interface identifies the details of an internal performance event - it does not represent an outgoing reported event
     */
    interface IPerfEvent {
        /**
         * The name of the performance event
         */
        name: string;
        /**
         * The start time of the performance event
         */
        start: number;
        /**
         * The payload (contents) of the perfEvent, may be null or only set after the event has completed depending on
         * the runtime environment.
         */
        payload: any;
        /**
         * Is this occurring from an asynchronous event
         */
        isAsync: boolean;
        /**
         * Identifies the total inclusive time spent for this event, including the time spent for child events,
         * this will be undefined until the event is completed
         */
        time?: number;
        /**
         * Identifies the exclusive time spent in for this event (not including child events),
         * this will be undefined until the event is completed.
         */
        exTime?: number;
        /**
         * The Parent event that was started before this event was created
         */
        parent?: IPerfEvent;
        /**
         * The child perf events that are contained within the total time of this event.
         */
        childEvts?: IPerfEvent[];
        /**
         * Identifies whether this event is a child event of a parent
         */
        isChildEvt: () => boolean;
        /**
         * Get the names additional context associated with this perf event
         */
        getCtx?: (key: string) => any;
        /**
         * Set the named additional context to be associated with this perf event, this will replace any existing value
         */
        setCtx?: (key: string, value: any) => void;
        /**
         * Mark this event as completed, calculating the total execution time.
         */
        complete: () => void;
    }

    /**
     * This defines an internal performance manager for tracking and reporting the internal performance of the SDK -- It does
     * not represent or report any event to the server.
     */
    interface IPerfManager {
        /**
         * Create a new event and start timing, the manager may return null/undefined to indicate that it does not
         * want to monitor this source event.
         * @param src The source name of the event
         * @param payloadDetails - An optional callback function to fetch the payload details for the event.
         * @param isAsync - Is the event occurring from a async event
         */
        create(src: string, payloadDetails?: () => any, isAsync?: boolean): IPerfEvent | null | undefined;
        /**
         * Complete the perfEvent and fire any notifications.
         * @param perfEvent Fire the event which will also complete the passed event
         */
        fire(perfEvent: IPerfEvent): void;
        /**
         * Set an execution context value
         * @param key - The context key name
         * @param value - The value
         */
        setCtx(key: string, value: any): void;
        /**
         * Get the execution context value
         * @param key - The context key
         */
        getCtx(key: string): any;
    }

    /**
     * Identifies an interface to a host that can provide an IPerfManager implementation
     */
    interface IPerfManagerProvider {
        /**
         * Get the current performance manager
         */
        getPerfMgr(): IPerfManager;
        /**
         * Set the current performance manager
         * @param perfMgr The performance manager
         */
        setPerfMgr(perfMgr: IPerfManager): void;
    }

    interface IPlugin {
        /**
         * Initialize plugin loaded by SDK
         * @param config - The config for the plugin to use
         * @param core - The current App Insights core to use for initializing this plugin instance
         * @param extensions - The complete set of extensions to be used for initializing the plugin
         * @param pluginChain - [Optional] specifies the current plugin chain which identifies the
         * set of plugins and the order they should be executed for the current request.
         */
        initialize: (config: IConfiguration, core: IAppInsightsCore, extensions: IPlugin[], pluginChain?: ITelemetryPluginChain) => void;
        /**
         * Returns a value that indicates whether the plugin has already been previously initialized.
         * New plugins should implement this method to avoid being initialized more than once.
         */
        isInitialized?: () => boolean;
        /**
         * Tear down the plugin and remove any hooked value, the plugin should be removed so that it is no longer initialized and
         * therefore could be re-initialized after being torn down. The plugin should ensure that once this has been called any further
         * processTelemetry calls are ignored and it just calls the processNext() with the provided context.
         * @param unloadCtx - This is the context that should be used during unloading.
         * @param unloadState - The details / state of the unload process, it holds details like whether it should be unloaded synchronously or asynchronously and the reason for the unload.
         * @returns boolean - true if the plugin has or will call processNext(), this for backward compatibility as previously teardown was synchronous and returned nothing.
         */
        teardown?: (unloadCtx: IProcessTelemetryUnloadContext, unloadState?: ITelemetryUnloadState) => void | boolean;
        /**
         * Extension name
         */
        readonly identifier: string;
        /**
         * Plugin version (available in data.properties.version in common schema)
         */
        readonly version?: string;
    }

    /**
     * The current context for the current call to processTelemetry(), used to support sharing the same plugin instance
     * between multiple AppInsights instances
     */
    interface IProcessTelemetryContext extends IBaseProcessingContext {
        /**
         * Call back for telemetry processing before it it is sent
         * @param env - This is the current event being reported
         * @returns boolean (true) if there is no more plugins to process otherwise false or undefined (void)
         */
        processNext: (env: ITelemetryItem) => boolean | void;
        /**
         * Create a new context using the core and config from the current instance, returns a new instance of the same type
         * @param plugins - The execution order to process the plugins, if null or not supplied
         *                  then the current execution order will be copied.
         * @param startAt - The plugin to start processing from, if missing from the execution
         *                  order then the next plugin will be NOT set.
         */
        createNew: (plugins?: IPlugin[] | ITelemetryPluginChain, startAt?: IPlugin) => IProcessTelemetryContext;
    }

    /**
     * The current context for the current call to teardown() implementations, used to support when plugins are being removed
     * or the SDK is being unloaded.
     */
    interface IProcessTelemetryUnloadContext extends IBaseProcessingContext {
        /**
         * This Plugin has finished unloading, so unload the next one
         * @param uploadState - The state of the unload process
         * @returns boolean (true) if there is no more plugins to process otherwise false or undefined (void)
         */
        processNext: (unloadState: ITelemetryUnloadState) => boolean | void;
        /**
         * Create a new context using the core and config from the current instance, returns a new instance of the same type
         * @param plugins - The execution order to process the plugins, if null or not supplied
         *                  then the current execution order will be copied.
         * @param startAt - The plugin to start processing from, if missing from the execution
         *                  order then the next plugin will be NOT set.
         */
        createNew: (plugins?: IPlugin[] | ITelemetryPluginChain, startAt?: IPlugin) => IProcessTelemetryUnloadContext;
    }

    /**
     * The current context for the current call to the plugin update() implementations, used to support the notifications
     * for when plugins are added, removed or the configuration was changed.
     */
    interface IProcessTelemetryUpdateContext extends IBaseProcessingContext {
        /**
         * This Plugin has finished unloading, so unload the next one
         * @param updateState - The update State
         * @returns boolean (true) if there is no more plugins to process otherwise false or undefined (void)
         */
        processNext: (updateState: ITelemetryUpdateState) => boolean | void;
        /**
         * Create a new context using the core and config from the current instance, returns a new instance of the same type
         * @param plugins - The execution order to process the plugins, if null or not supplied
         *                  then the current execution order will be copied.
         * @param startAt - The plugin to start processing from, if missing from the execution
         *                  order then the next plugin will be NOT set.
         */
        createNew: (plugins?: IPlugin[] | ITelemetryPluginChain, startAt?: IPlugin) => IProcessTelemetryUpdateContext;
    }

    interface _IRegisteredEvents {
        name: string;
        handler: any;
    }

    /**
     * Check if an object is of type Array with optional generic T, the generic type is not validated
     * and exists to help with TypeScript validation only.
     */
    let isArray: <T = any>(obj: any) => obj is Array<T>;

    /**
     * Checks if HTML5 Beacons are supported in the current environment.
     * @returns True if supported, false otherwise.
     */
    function isBeaconsSupported(): boolean;

    /**
     * Checks if the type of value is a boolean.
     * @param {any} value - Value to be checked.
     * @return {boolean} True if the value is a boolean, false otherwise.
     */
    function isBoolean(value: any): value is boolean;

    /**
     * Check if an object is of type Date
     */
    function isDate(obj: any): obj is Date;

    /**
     * Check if an object is of type Error
     */
    function isError(obj: any): obj is Error;

    /**
     * Checks if the Fetch API is supported in the current environment.
     * @param withKeepAlive - [Optional] If True, check if fetch is available and it supports the keepalive feature, otherwise only check if fetch is supported
     * @returns True if supported, otherwise false
     */
    function isFetchSupported(withKeepAlive?: boolean): boolean;

    function isFunction(value: any): value is Function;

    /**
     * Identifies whether the current environment appears to be IE
     */
    function isIE(): boolean;

    function isNotNullOrUndefined<T>(value: T): value is T;

    function isNotTruthy(value: any): boolean;

    function isNotUndefined<T>(value: T): value is T;

    function isNullOrUndefined(value: any): value is null | undefined;

    /**
     * Checks if the type of value is a number.
     * @param {any} value - Value to be checked.
     * @return {boolean} True if the value is a number, false otherwise.
     */
    function isNumber(value: any): value is number;

    function isObject<T>(value: T): value is T;

    /**
     * Returns whether the environment is reporting that we are running in a React Native Environment
     */
    function isReactNative(): boolean;

    function isSafari(userAgentStr?: string): boolean;

    /**
     * Is the parsed traceParent indicating that the trace is currently sampled.
     * @param value - The parsed traceParent value
     * @returns
     */
    function isSampledFlag(value: ITraceParent): boolean;

    /**
     * Checks if the type of value is a string.
     * @param {any} value - Value to be checked.
     * @return {boolean} True if the value is a string, false otherwise.
     */
    function isString(value: any): value is string;

    /**
     * Checks if the type of value is a Symbol.
     * This only returns a boolean as returning value is Symbol will cause issues for older TypeScript consumers
     * @param {any} value - Value to be checked.
     * @return {boolean} True if the value is a Symbol, false otherwise.
     */
    function isSymbol(value: any): boolean;

    function isTruthy(value: any): boolean;

    function isTypeof(value: any, theType: string): boolean;

    function isUndefined(value: any): value is undefined;

    /**
     * Is the provided W3c span id (aka. parent id) a valid string representation, it must be a 16-character
     * string of lowercase hexadecimal characters, for example, 00f067aa0ba902b7.
     * If all characters are zero (0000000000000000) this is considered an invalid value.
     * @param value - The W3c span id to be validated
     * @returns true if valid otherwise false
     */
    function isValidSpanId(value: string): boolean;

    /**
     * Is the provided W3c Trace Id a valid string representation, it must be a 32-character string
     * of lowercase hexadecimal characters for example, 4bf92f3577b34da6a3ce929d0e0e4736.
     * If all characters as zero (00000000000000000000000000000000) it will be considered an invalid value.
     * @param value - The W3c trace Id to be validated
     * @returns true if valid otherwise false
     */
    function isValidTraceId(value: string): boolean;

    /**
     * Validates that the provided ITraceParent instance conforms to the currently supported specifications
     * @param value
     * @returns
     */
    function isValidTraceParent(value: ITraceParent): boolean;

    /**
     * Checks if XMLHttpRequest is supported
     * @returns True if supported, otherwise false
     */
    function isXhrSupported(): boolean;

    interface ITelemetryInitializerContainer {
        /**
         * Add a telemetry processor to decorate or drop telemetry events.
         * @param telemetryInitializer - The Telemetry Initializer function
         * @returns - A ITelemetryInitializerHandler to enable the initializer to be removed
         */
        addTelemetryInitializer(telemetryInitializer: TelemetryInitializerFunction): ITelemetryInitializerHandler | void;
    }

    interface ITelemetryInitializerHandler {
        remove(): void;
    }

    /**
     * Telemety item supported in Core
     */
    interface ITelemetryItem {
        /**
         * CommonSchema Version of this SDK
         */
        ver?: string;
        /**
         * Unique name of the telemetry item
         */
        name: string;
        /**
         * Timestamp when item was sent
         */
        time?: string;
        /**
         * Identifier of the resource that uniquely identifies which resource data is sent to
         */
        iKey?: string;
        /**
         * System context properties of the telemetry item, example: ip address, city etc
         */
        ext?: {
            [key: string]: any;
        };
        /**
         * System context property extensions that are not global (not in ctx)
         */
        tags?: Tags & Tags[];
        /**
         * Custom data
         */
        data?: ICustomProperties;
        /**
         * Telemetry type used for part B
         */
        baseType?: string;
        /**
         * Based on schema for part B
         */
        baseData?: {
            [key: string]: any;
        };
    }

    /**
     * Configuration provided to SDK core
     */
    interface ITelemetryPlugin extends ITelemetryProcessor, IPlugin {
        /**
         * Set next extension for telemetry processing, this is not optional as plugins should use the
         * processNext() function of the passed IProcessTelemetryContext instead. It is being kept for
         * now for backward compatibility only.
         */
        setNextPlugin?: (next: ITelemetryPlugin | ITelemetryPluginChain) => void;
        /**
         * Priority of the extension
         */
        readonly priority: number;
    }

    /**
     * Configuration provided to SDK core
     */
    interface ITelemetryPluginChain extends ITelemetryProcessor {
        /**
         * Returns the underlying plugin that is being proxied for the processTelemetry call
         */
        getPlugin: () => ITelemetryPlugin;
        /**
         * Returns the next plugin
         */
        getNext: () => ITelemetryPluginChain;
        /**
         * This plugin is being unloaded and should remove any hooked events and cleanup any global/scoped values, after this
         * call the plugin will be removed from the telemetry processing chain and will no longer receive any events..
         * @param unloadCtx - The unload context to use for this call.
         * @param unloadState - The details of the unload operation
         */
        unload?: (unloadCtx: IProcessTelemetryUnloadContext, unloadState: ITelemetryUnloadState) => void;
    }

    interface ITelemetryProcessor {
        /**
         * Call back for telemetry processing before it it is sent
         * @param env - This is the current event being reported
         * @param itemCtx - This is the context for the current request, ITelemetryPlugin instances
         * can optionally use this to access the current core instance or define / pass additional information
         * to later plugins (vs appending items to the telemetry item)
         */
        processTelemetry: (env: ITelemetryItem, itemCtx?: IProcessTelemetryContext) => void;
        /**
         * The the plugin should re-evaluate configuration and update any cached configuration settings or
         * plugins. If implemented this method will be called whenever a plugin is added or removed and if
         * the configuration has bee updated.
         * @param updateCtx - This is the context that should be used during updating.
         * @param updateState - The details / state of the update process, it holds details like the current and previous configuration.
         * @returns boolean - true if the plugin has or will call updateCtx.processNext(), this allows the plugin to perform any asynchronous operations.
         */
        update?: (updateCtx: IProcessTelemetryUpdateContext, updateState: ITelemetryUpdateState) => void | boolean;
    }

    interface ITelemetryUnloadState {
        reason: TelemetryUnloadReason;
        isAsync: boolean;
        flushComplete?: boolean;
    }

    interface ITelemetryUpdateState {
        /**
         * Identifies the reason for the update notification, this is a bitwise numeric value
         */
        reason: TelemetryUpdateReason;
        /**
         * If this is a configuration update this was the previous configuration that was used
         */
        /**
         * If this is a configuration update is the new configuration that is being used
         */
        /**
         * This holds a collection of plugins that have been added (if the reason identifies that one or more plugins have been added)
         */
        added?: IPlugin[];
        /**
         * This holds a collection of plugins that have been removed (if the reason identifies that one or more plugins have been removed)
         */
        removed?: IPlugin[];
    }

    /**
     * This interface represents the components of a W3C traceparent header
     */
    interface ITraceParent {
        /**
         * The version of the definition, this MUST be a string with a length of 2 and only contain lowercase
         * hexadecimal characters. A value of 'ff' is considered to be an invalid version.
         */
        version: string;
        /**
         * This is the ID of the whole trace forest and is used to uniquely identify a distributed trace
         * through a system. It is represented as a 32-character string of lowercase hexadecimal characters,
         * for example, 4bf92f3577b34da6a3ce929d0e0e4736.
         * All characters as zero (00000000000000000000000000000000) is considered an invalid value.
         */
        traceId: string;
        /**
         * This is the ID of the current request as known by the caller (in some tracing systems, this is also
         * known as the parent-id, where a span is the execution of a client request). It is represented as an
         * 16-character string of lowercase hexadecimal characters, for example, 00f067aa0ba902b7.
         * All bytes as zero (0000000000000000) is considered an invalid value.
         */
        spanId: string;
        /**
         * An 8-bit value of flags that controls tracing such as sampling, trace level, etc. These flags are
         * recommendations given by the caller rather than strict rules to follow.
         * As this is a bit field, you cannot interpret flags by decoding the hex value and looking at the resulting
         * number. For example, a flag 00000001 could be encoded as 01 in hex, or 09 in hex if present with the flag
         * 00001000. A common mistake in bit fields is forgetting to mask when interpreting flags.
         */
        traceFlags: number;
    }

    interface IUnloadableComponent {
        /**
         * Teardown / Unload hook to allow implementations to perform some additional unload operations before the BaseTelemetryPlugin
         * finishes it's removal.
         * @param unloadCtx - This is the context that should be used during unloading.
         * @param unloadState - The details / state of the unload process, it holds details like whether it should be unloaded synchronously or asynchronously and the reason for the unload.
         * @param asyncCallback - An optional callback that the plugin must call if it returns true to inform the caller that it has completed any async unload/teardown operations.
         * @returns boolean - true if the plugin has or will call asyncCallback, this allows the plugin to perform any asynchronous operations.
         */
        _doUnload?: (unloadCtx?: IProcessTelemetryUnloadContext, unloadState?: ITelemetryUnloadState, asyncCallback?: () => void) => void | boolean;
    }

    interface IUnloadHandlerContainer {
        add: (handler: UnloadHandler) => void;
        run: (itemCtx: IProcessTelemetryUnloadContext, unloadState: ITelemetryUnloadState) => void;
    }

    /**
     * Helper to support backward compatibility for users that use the legacy cookie handling functions and the use the internal
     * CoreUtils._canUseCookies global flag to enable/disable cookies usage.
     * Note: This has the following deliberate side-effects
     * - Creates the global (legacy) cookie manager if it does not already exist
     * - Attempts to add "listeners" to the CoreUtils._canUseCookies property to support the legacy usage
     * @param config
     * @param logger
     * @returns
     */
    function _legacyCookieMgr(config?: IConfiguration, logger?: IDiagnosticLogger): ICookieMgr;

    const LoggingSeverity: EnumValue<typeof eLoggingSeverity>;

    type LoggingSeverity = number | eLoggingSeverity;

    /**
     * Logs a message to the internal queue.
     * @param logger - The Diagnostic Logger instance to use.
     * @param severity {LoggingSeverity} - The severity of the log message
     * @param message {_InternalLogMessage} - The message to log.
     */
    function _logInternalMessage(logger: IDiagnosticLogger, severity: LoggingSeverity, message: _InternalLogMessage): void;

    function mergeEvtNamespace(theNamespace: string, namespaces?: string | string[] | null): string | string[];

    const MinChannelPriorty: number;

    /**
     * Generate a random 32-bit number between (0x000000..0xFFFFFFFF) or (-0x80000000..0x7FFFFFFF), using MWC (Multiply with carry)
     * instead of Math.random() defaults to un-signed.
     * Used as a replacement random generator for IE to avoid issues with older IE instances.
     * @param signed - True to return a signed 32-bit number (-0x80000000..0x7FFFFFFF) otherwise an unsigned one (0x000000..0xFFFFFFFF)
     */
    function mwcRandom32(signed?: boolean): number;

    /**
     * Seed the MWC random number generator with the specified seed or a random value
     * @param value - optional the number to used as the seed, if undefined, null or zero a random value will be chosen
     */
    function mwcRandomSeed(value?: number): void;

    function newGuid(): string;

    /**
     * Generate random base64 id string.
     * The default length is 22 which is 132-bits so almost the same as a GUID but as base64 (the previous default was 5)
     * @param maxLength - Optional value to specify the length of the id to be generated, defaults to 22
     */
    function newId(maxLength?: number): string;

    /**
     * Validates that the string name conforms to the JS IdentifierName specification and if not
     * normalizes the name so that it would. This method does not identify or change any keywords
     * meaning that if you pass in a known keyword the same value will be returned.
     * This is a simplified version
     * @param name The name to validate
     */
    function normalizeJsName(name: string): string;

    /**
     * Class to manage sending notifications to all the listeners.
     */
    class NotificationManager implements INotificationManager {
        listeners: INotificationListener[];
        constructor(config?: IConfiguration);
        /**
         * Adds a notification listener.
         * @param {INotificationListener} listener - The notification listener to be added.
         */
        addNotificationListener(listener: INotificationListener): void;
        /**
         * Removes all instances of the listener.
         * @param {INotificationListener} listener - AWTNotificationListener to remove.
         */
        removeNotificationListener(listener: INotificationListener): void;
        /**
         * Notification for events sent.
         * @param {ITelemetryItem[]} events - The array of events that have been sent.
         */
        eventsSent(events: ITelemetryItem[]): void;
        /**
         * Notification for events being discarded.
         * @param {ITelemetryItem[]} events - The array of events that have been discarded by the SDK.
         * @param {number} reason           - The reason for which the SDK discarded the events. The EventsDiscardedReason
         * constant should be used to check the different values.
         */
        eventsDiscarded(events: ITelemetryItem[], reason: number): void;
        /**
         * [Optional] A function called when the events have been requested to be sent to the sever.
         * @param {number} sendReason - The reason why the event batch is being sent.
         * @param {boolean} isAsync   - A flag which identifies whether the requests are being sent in an async or sync manner.
         */
        eventsSendRequest?(sendReason: number, isAsync: boolean): void;
        /**
         * [Optional] This event is sent if you have enabled perf events, they are primarily used to track internal performance testing and debugging
         * the event can be displayed via the debug plugin extension.
         * @param perfEvent
         */
        perfEvent?(perfEvent: IPerfEvent): void;
    }

    export { objCreate }

    /**
     * Try to define get/set object property accessors for the target object/prototype, this will provide compatibility with
     * existing API definition when run within an ES5+ container that supports accessors but still enable the code to be loaded
     * and executed in an ES3 container, providing basic IE8 compatibility.
     * @param target The object on which to define the property.
     * @param prop The name of the property to be defined or modified.
     * @param getProp The getter function to wire against the getter.
     * @param setProp The setter function to wire against the setter.
     * @returns True if it was able to create the accessors otherwise false
     */
    function objDefineAccessors<T>(target: any, prop: string, getProp?: () => T, setProp?: (v: T) => void): boolean;

    /**
     * Pass in the objects to merge as arguments, this will only "merge" (extend) properties that are owned by the object.
     * It will NOT merge inherited or non-enumerable properties.
     * @param obj1 - object to merge.  Set this argument to 'true' for a deep extend.
     * @param obj2 - object to merge.
     * @param obj3 - object to merge.
     * @param obj4 - object to merge.
     * @param obj5 - object to merge.
     * @returns The extended first object.
     */
    function objExtend<T2, T3, T4, T5, T6>(deepExtend?: boolean, obj2?: T2, obj3?: T3, obj4?: T4, obj5?: T5, obj6?: T6): T2 & T3 & T4 & T5 & T6;

    function objExtend<T1, T2, T3, T4, T5, T6>(obj1?: T1, obj2?: T2, obj3?: T3, obj4?: T4, obj5?: T5, obj6?: T6): T1 & T2 & T3 & T4 & T5 & T6;

    /**
     * This is a helper function for the equivalent of arForEach(objKeys(target), callbackFn), this is a
     * performance optimization to avoid the creation of a new array for large objects
     * @param target The target object to find and process the keys
     * @param callbackfn The function to call with the details
     */
    function objForEachKey<T = any>(target: T, callbackfn: (name: string, value: T[keyof T]) => void): void;

    const objFreeze: <T>(value: T) => T;

    /**
     * Returns the names of the enumerable string properties and methods of an object. This helper exists to avoid adding a polyfil for older browsers
     * that do not define Object.keys eg. ES3 only, IE8 just in case any page checks for presence/absence of the prototype implementation.
     * Note: For consistency this will not use the Object.keys implementation if it exists as this would cause a testing requirement to test with and without the implementations
     * @param obj Object that contains the properties and methods. This can be an object that you created or an existing Document Object Model (DOM) object.
     */
    function objKeys(obj: {}): string[];

    const objSeal: <T>(value: T) => T;

    function objToString(obj: any): any;

    /**
     * A helper function to assist with JIT performance for objects that have properties added / removed dynamically
     * this is primarily for chromium based browsers and has limited effects on Firefox and none of IE. Only call this
     * function after you have finished "updating" the object, calling this within loops reduces or defeats the benefits.
     * This helps when iterating using for..in, objKeys() and objForEach()
     * @param theObject - The object to be optimized if possible
     */
    function optimizeObject<T>(theObject: T): T;

    /**
     * Attempt to parse the provided string as a W3C TraceParent header value (https://www.w3.org/TR/trace-context/#traceparent-header)
     *
     * @param value
     * @returns
     */
    function parseTraceParent(value: string): ITraceParent;

    class PerfEvent implements IPerfEvent {
        static ParentContextKey: string;
        static ChildrenContextKey: string;
        /**
         * The name of the event
         */
        name: string;
        /**
         * The start time of the event in ms
         */
        start: number;
        /**
         * The payload (contents) of the perfEvent, may be null or only set after the event has completed depending on
         * the runtime environment.
         */
        payload: any;
        /**
         * Is this occurring from an asynchronous event
         */
        isAsync: boolean;
        /**
         * Identifies the total inclusive time spent for this event, including the time spent for child events,
         * this will be undefined until the event is completed
         */
        time?: number;
        /**
         * Identifies the exclusive time spent in for this event (not including child events),
         * this will be undefined until the event is completed.
         */
        exTime?: number;
        /**
         * Identifies whether this event is a child event of a parent
         */
        isChildEvt: () => boolean;
        getCtx?: (key: string) => any | null | undefined;
        setCtx?: (key: string, value: any) => void;
        complete: () => void;
        constructor(name: string, payloadDetails: () => any, isAsync: boolean);
    }

    class PerfManager implements IPerfManager {
        /**
         * General bucket used for execution context set and retrieved via setCtx() and getCtx.
         * Defined as private so it can be visualized via the DebugPlugin
         */
        private ctx;
        constructor(manager?: INotificationManager);
        /**
         * Create a new event and start timing, the manager may return null/undefined to indicate that it does not
         * want to monitor this source event.
         * @param src The source name of the event
         * @param payloadDetails - An optional callback function to fetch the payload details for the event.
         * @param isAsync - Is the event occurring from a async event
         */
        create(src: string, payload?: any, isAsync?: boolean): IPerfEvent | null | undefined;
        /**
         * Complete the perfEvent and fire any notifications.
         * @param perfEvent Fire the event which will also complete the passed event
         */
        fire(perfEvent: IPerfEvent): void;
        /**
         * Set an execution context value
         * @param key - The context key name
         * @param value - The value
         */
        setCtx(key: string, value: any): void;
        /**
         * Get the execution context value
         * @param key - The context key
         */
        getCtx(key: string): any;
    }

    /**
     * Return the current value of the Performance Api now() function (if available) and fallback to dateNow() if it is unavailable (IE9 or less)
     * https://caniuse.com/#search=performance.now
     */
    function perfNow(): number;

    /**
     * This class will be removed!
     * @deprecated use createProcessTelemetryContext() instead
     */
    class ProcessTelemetryContext implements IProcessTelemetryContext {
        /**
         * Gets the current core config instance
         */
        getCfg: () => IConfiguration;
        getExtCfg: <T>(identifier: string, defaultValue?: T | any) => T;
        getConfig: (identifier: string, field: string, defaultValue?: number | string | boolean | string[] | RegExp[] | Function) => number | string | boolean | string[] | RegExp[] | Function;
        /**
         * Returns the IAppInsightsCore instance for the current request
         */
        core: () => IAppInsightsCore;
        /**
         * Returns the current IDiagnosticsLogger for the current request
         */
        diagLog: () => IDiagnosticLogger;
        /**
         * Helper to allow inherited classes to check and possibly shortcut executing code only
         * required if there is a nextPlugin
         */
        hasNext: () => boolean;
        /**
         * Returns the next configured plugin proxy
         */
        getNext: () => ITelemetryPluginChain;
        /**
         * Helper to set the next plugin proxy
         */
        setNext: (nextCtx: ITelemetryPluginChain) => void;
        /**
         * Call back for telemetry processing before it it is sent
         * @param env - This is the current event being reported
         * @param itemCtx - This is the context for the current request, ITelemetryPlugin instances
         * can optionally use this to access the current core instance or define / pass additional information
         * to later plugins (vs appending items to the telemetry item)
         * @returns boolean (true) if there is no more plugins to process otherwise false or undefined (void)
         */
        processNext: (env: ITelemetryItem) => boolean | void;
        /**
         * Synchronously iterate over the context chain running the callback for each plugin, once
         * every plugin has been executed via the callback, any associated onComplete will be called.
         * @param callback - The function call for each plugin in the context chain
         */
        iterate: <T extends ITelemetryPlugin = ITelemetryPlugin>(callback: (plugin: T) => void) => void;
        /**
         * Create a new context using the core and config from the current instance
         * @param plugins - The execution order to process the plugins, if null or not supplied
         *                  then the current execution order will be copied.
         * @param startAt - The plugin to start processing from, if missing from the execution
         *                  order then the next plugin will be NOT set.
         */
        createNew: (plugins?: IPlugin[] | ITelemetryPluginChain, startAt?: IPlugin) => IProcessTelemetryContext;
        /**
         * Set the function to call when the current chain has executed all processNext or unloadNext items.
         */
        onComplete: (onComplete: () => void) => void;
        /**
         * Creates a new Telemetry Item context with the current config, core and plugin execution chain
         * @param plugins - The plugin instances that will be executed
         * @param config - The current config
         * @param core - The current core instance
         */
        constructor(pluginChain: ITelemetryPluginChain, config: IConfiguration, core: IAppInsightsCore, startAt?: IPlugin);
    }

    /**
     * Effectively assigns all enumerable properties (not just own properties) and functions (including inherited prototype) from
     * the source object to the target, it attempts to use proxy getters / setters (if possible) and proxy functions to avoid potential
     * implementation issues by assigning prototype functions as instance ones
     *
     * This method is the primary method used to "update" the snippet proxy with the ultimate implementations.
     *
     * Special ES3 Notes:
     * Updates (setting) of direct property values on the target or indirectly on the source object WILL NOT WORK PROPERLY, updates to the
     * properties of "referenced" object will work (target.context.newValue = 10 => will be reflected in the source.context as it's the
     * same object). ES3 Failures: assigning target.myProp = 3 -> Won't change source.myProp = 3, likewise the reverse would also fail.
     * @param target - The target object to be assigned with the source properties and functions
     * @param source - The source object which will be assigned / called by setting / calling the targets proxies
     * @param chkSet - An optional callback to determine whether a specific property/function should be proxied
     */
    function proxyAssign<T, S>(target: T, source: S, chkSet?: (name: string, isFunc?: boolean, source?: S, target?: T) => boolean): T;

    /**
     * Creates a proxy function on the target which internally will call the source version with all arguments passed to the target method.
     *
     * @param target - The target object to be assigned with the source properties and functions
     * @param name - The function name that will be added on the target
     * @param source - The source object which will be assigned / called by setting / calling the targets proxies
     * @param theFunc - The function name on the source that will be proxied on the target
     * @param overwriteTarget - If `false` this will not replace any pre-existing name otherwise (the default) it will overwrite any existing name
     */
    function proxyFunctionAs<T, S>(target: T, name: string, source: S | (() => S), theFunc: (keyof S), overwriteTarget?: boolean): void;

    /**
     * Creates proxy functions on the target which internally will call the source version with all arguments passed to the target method.
     *
     * @param target - The target object to be assigned with the source properties and functions
     * @param source - The source object which will be assigned / called by setting / calling the targets proxies
     * @param functionsToProxy - An array of function names that will be proxied on the target
     * @param overwriteTarget - If false this will not replace any pre-existing name otherwise (the default) it will overwrite any existing name
     */
    function proxyFunctions<T, S>(target: T, source: S | (() => S), functionsToProxy: (keyof S)[], overwriteTarget?: boolean): T;

    /**
     * generate a random 32-bit number (0x000000..0xFFFFFFFF) or (-0x80000000..0x7FFFFFFF), defaults un-unsigned.
     * @param signed - True to return a signed 32-bit number (-0x80000000..0x7FFFFFFF) otherwise an unsigned one (0x000000..0xFFFFFFFF)
     */
    function random32(signed?: boolean): number;

    /**
     * Generate a random value between 0 and maxValue, max value should be limited to a 32-bit maximum.
     * So maxValue(16) will produce a number from 0..16 (range of 17)
     * @param maxValue
     */
    function randomValue(maxValue: number): number;

    /**
     * Trys to remove event handler(s) for the specified event/namespace to the window, body and document
     * @param eventName {string} - The name of the event, with optional namespaces or just the namespaces,
     * such as "click", "click.mynamespace" or ".mynamespace"
     * @param callback {any} - - The callback function that needs to be removed from the given event, when using a
     * namespace (with or without a qualifying event) this may be null to remove all previously attached event handlers
     * otherwise this will only remove events with this specific handler.
     * @param evtNamespace - [Optional] Namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace.
     */
    function removeEventHandler(eventName: string, callback: any, evtNamespace?: string | string[] | null): void;

    /**
     * Remove the listener from the array of events
     * @param events An string array of event names to bind the listener to
     * @param listener The event callback to call when the event is triggered
     * @param evtNamespace - [Optional] Namespace(s) to append to the event listeners so they can be uniquely identified and removed based on this namespace.
     */
    function removeEventListeners(events: string[], listener: any, evtNamespace?: string | string[]): void;

    /**
     * Removes the pageHide event listeners added by addPageHideEventListener, because the 'visibilitychange' uses
     * an internal proxy to detect the visibility state you SHOULD use a unique namespace when calling addPageHideEventListener
     * as the remove ignores the listener argument for the 'visibilitychange' event.
     * @param listener - The specific listener to remove for the 'pageshow' event only (ignored for 'visibilitychange')
     * @param evtNamespace - The unique namespace used when calling addPageShowEventListener
     */
    function removePageHideEventListener(listener: any, evtNamespace?: string | string[] | null): void;

    /**
     * Removes the pageShow event listeners added by addPageShowEventListener, because the 'visibilitychange' uses
     * an internal proxy to detect the visibility state you SHOULD use a unique namespace when calling addPageShowEventListener
     * as the remove ignores the listener argument for the 'visibilitychange' event.
     * @param listener - The specific listener to remove for the 'pageshow' event only (ignored for 'visibilitychange')
     * @param evtNamespace - The unique namespace used when calling addPageShowEventListener
     */
    function removePageShowEventListener(listener: any, evtNamespace?: string | string[] | null): void;

    /**
     * Remove any matching 'beforeunload', 'unload' and 'pagehide' events that may have been added via addEventListener,
     * addEventListeners, addPageUnloadEventListener or addPageHideEventListener.
     * @param listener - The specific event callback to to be removed
     * @param evtNamespace - [Optional] Namespace(s) uniquely identified and removed based on this namespace.
     * @returns true - when at least one of the events was registered otherwise false
     */
    function removePageUnloadEventListener(listener: any, evtNamespace?: string | string[]): void;

    /**
     * Helper to return the ICookieMgr from the core (if not null/undefined) or a default implementation
     * associated with the configuration or a legacy default.
     * @param core
     * @param config
     * @returns
     */
    function safeGetCookieMgr(core: IAppInsightsCore, config?: IConfiguration): ICookieMgr;

    function safeGetLogger(core: IAppInsightsCore, config?: IConfiguration): IDiagnosticLogger;

    /**
     * The EventsDiscardedReason enumeration contains a set of values that specify the reason for discarding an event.
     */
    const enum SendRequestReason {
        /**
         * No specific reason was specified
         */
        Undefined = 0,
        /**
         * Events are being sent based on the normal event schedule / timer.
         */
        NormalSchedule = 1,
        /**
         * A manual flush request was received
         */
        ManualFlush = 1,
        /**
         * Unload event is being processed
         */
        Unload = 2,
        /**
         * The event(s) being sent are sync events
         */
        SyncEvent = 3,
        /**
         * The Channel was resumed
         */
        Resumed = 4,
        /**
         * The event(s) being sent as a retry
         */
        Retry = 5,
        /**
         * The SDK is unloading
         */
        SdkUnload = 6,
        /**
         * Maximum batch size would be exceeded
         */
        MaxBatchSize = 10,
        /**
         * The Maximum number of events have already been queued
         */
        MaxQueuedEvents = 20
    }

    /**
     * @deprecated - Use the core.getCookieMgr().set()
     * helper method to set userId and sessionId cookie
     */
    function setCookie(logger: IDiagnosticLogger, name: string, value: string, domain?: string): void;

    /**
     * Enable the lookup of test mock objects if requested
     * @param enabled
     */
    function setEnableEnvMocks(enabled: boolean): void;

    /**
     * Set the global performance manager to use when there is no core instance or it has not been initialized yet.
     * @param perfManager - The IPerfManager instance to use when no performance manager is supplied.
     */
    function setGblPerfMgr(perfManager: IPerfManager): void;

    /**
     * Sets the provided value on the target instance using the field name when the provided chk function returns true, the chk
     * function will only be called if the new value is no equal to the original value.
     * @param target - The target object
     * @param field - The key of the target
     * @param value - The value to set
     * @param valChk - [Optional] Callback to check the value that if supplied will be called check if the new value can be set
     * @param srcChk - [Optional] Callback to check to original value that if supplied will be called if the new value should be set (if allowed)
     * @returns The existing or new value, depending what was set
     */
    function setValue<T, K extends keyof T>(target: T, field: K, value: T[K], valChk?: ((value: T[K]) => boolean) | null, srcChk?: ((value: T[K]) => boolean) | null): T[K];

    function sortPlugins<T = IPlugin>(plugins: T[]): T[];

    /**
     * A simple wrapper (for minification support) to check if the value contains the search string.
     * @param value - The string value to check for the existence of the search value
     * @param search - The value search within the value
     */
    function strContains(value: string, search: string): boolean;

    /**
     * The strEndsWith() method determines whether a string ends with the characters of a specified string, returning true or false as appropriate.
     * @param value - The value to check whether it ends with the search value.
     * @param search - The characters to be searched for at the end of the value.
     * @returns true if the given search value is found at the end of the string, otherwise false.
     */
    function strEndsWith(value: string, search: string): boolean;

    export { strFunction }

    export { strObject }

    export { strPrototype }

    /**
     * The strStartsWith() method determines whether a string starts with the characters of the specified string, returning true or false as appropriate.
     * @param value - The value to check whether it ends with the search value.
     * @param checkValue - The characters to be searched for at the start of the value.
     * @returns true if the given search value is found at the start of the string, otherwise false.
     */
    function strStartsWith(value: string, checkValue: string): boolean;

    /**
     * helper method to trim strings (IE8 does not implement String.prototype.trim)
     */
    function strTrim(str: any): string;

    export { strUndefined }

    interface Tags {
        [key: string]: any;
    }

    type TelemetryInitializerFunction = <T extends ITelemetryItem>(item: T) => boolean | void;

    /**
     * The TelemetryUnloadReason enumeration contains the possible reasons for why a plugin is being unloaded / torndown().
     */
    const enum TelemetryUnloadReason {
        /**
         * Teardown has been called without any context.
         */
        ManualTeardown = 0,
        /**
         * Just this plugin is being removed
         */
        PluginUnload = 1,
        /**
         * This instance of the plugin is being removed and replaced
         */
        PluginReplace = 2,
        /**
         * The entire SDK is being unloaded
         */
        SdkUnload = 50
    }

    /**
     * The TelemetryUpdateReason enumeration contains a set of bit-wise values that specify the reason for update request.
     */
    const enum TelemetryUpdateReason {
        /**
         * Unknown.
         */
        Unknown = 0,
        /**
         * The configuration has ben updated or changed
         */
        /**
         * One or more plugins have been added
         */
        PluginAdded = 16,
        /**
         * One or more plugins have been removed
         */
        PluginRemoved = 32
    }

    function throwError(message: string): never;

    /**
     * This is a helper method which will call throwInternal on the passed logger, will throw exceptions in
     * debug mode or attempt to log the error as a console warning. This helper is provided mostly to better
     * support minification as logger.throwInternal() will not compress the publish "throwInternal" used throughout
     * the code.
     * @param logger - The Diagnostic Logger instance to use.
     * @param severity {LoggingSeverity} - The severity of the log message
     * @param message {_InternalLogMessage} - The log message.
     */
    function _throwInternal(logger: IDiagnosticLogger, severity: LoggingSeverity, msgId: _InternalMessageId, msg: string, properties?: Object, isUserAct?: boolean): void;

    /**
     * Convert a date to I.S.O. format in IE8
     */
    function toISOString(date: Date): string;

    function uaDisallowsSameSiteNone(userAgent: string): boolean;

    const Undefined = "undefined";

    /**
     * Teardown / Unload helper to perform teardown/unloading operations for the provided components synchronously or asynchronously, this will call any
     * _doTeardown() or _doUnload() functions on the provided components to allow them to finish removal.
     * @param components - The components you want to unload
     * @param unloadCtx - This is the context that should be used during unloading.
     * @param unloadState - The details / state of the unload process, it holds details like whether it should be unloaded synchronously or asynchronously and the reason for the unload.
     * @param asyncCallback - An optional callback that the plugin must call if it returns true to inform the caller that it has completed any async unload/teardown operations.
     * @returns boolean - true if the plugin has or will call asyncCallback, this allows the plugin to perform any asynchronous operations.
     */
    function unloadComponents(components: any | IUnloadableComponent[], unloadCtx?: IProcessTelemetryUnloadContext, unloadState?: ITelemetryUnloadState, asyncCallback?: () => void): void | boolean;

    type UnloadHandler = (itemCtx: IProcessTelemetryUnloadContext, unloadState: ITelemetryUnloadState) => void;

    function useXDomainRequest(): boolean | undefined;

    /**
     * This is a helper method which will call warnToConsole on the passed logger with the provided message.
     * @param logger - The Diagnostic Logger instance to use.
     * @param message {_InternalLogMessage} - The log message.
     */
    function _warnToConsole(logger: IDiagnosticLogger, message: string): void;

    
}