"use strict";
/*!
 * Copyright 2021 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogSync = void 0;
const entry_1 = require("./entry");
const instrumentation_1 = require("./utils/instrumentation");
const log_common_1 = require("./utils/log-common");
/**
 * A logSync is a named collection of entries in structured log format. In Cloud
 * Logging, structured logs refer to log entries that use the jsonPayload field
 * to add structure to their payloads. In most GCP environments, like GKE and
 * Cloud Functions, structured logs written to process.stdout are automatically
 * picked up and formatted by logging agents.
 *
 * Recommended for Serverless environment logging, especially where async log
 * calls made by the `Log` class can be dropped by the CPU.
 *
 * See {@link https://cloud.google.com/logging/docs/structured-logging|Structured Logging}
 *
 * @class
 *
 * @param {Logging} logging {@link Logging} instance.
 * @param {string} name Name of the logSync.
 * @param {Writable} [transport] transport A custom writable transport stream.
 *     Default: process.stdout.
 *
 * @example
 * ```
 * const {Logging} = require('@google-cloud/logging');
 * const logging = new Logging();
 * const log = logging.logSync('mylog');
 * ```
 */
class LogSync {
    // not projectId, formattedname is expected
    constructor(logging, name, transport) {
        this.formattedName_ = log_common_1.formatLogName(logging.projectId, name);
        this.logging = logging;
        /**
         * @name Log#name
         * @type {string}
         */
        this.name = this.formattedName_.split('/').pop();
        // Default to writing to stdout
        this.transport = transport || process.stdout;
    }
    /**
     * Write a log entry with a severity of "ALERT".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.alert(entry);
     * ```
     */
    alert(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'ALERT'), options);
    }
    /**
     * Write a log entry with a severity of "CRITICAL".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.critical(entry);
     * ```
     */
    critical(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'CRITICAL'), options);
    }
    /**
     * Write a log entry with a severity of "DEBUG".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.debug(entry);
     * ```
     */
    debug(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'DEBUG'), options);
    }
    /**
     * Write a log entry with a severity of "EMERGENCY".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.emergency(entry);
     * ```
     */
    emergency(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'EMERGENCY'), options);
    }
    entry(metadataOrData, data) {
        let metadata;
        if (!data &&
            metadataOrData !== null &&
            Object.prototype.hasOwnProperty.call(metadataOrData, 'httpRequest')) {
            // If user logs entry(metadata.httpRequest)
            metadata = metadataOrData;
            data = {};
        }
        else if (!data) {
            // If user logs entry(message)
            data = metadataOrData;
            metadata = {};
        }
        else {
            // If user logs entry(metadata, message)
            metadata = metadataOrData;
        }
        return this.logging.entry(metadata, data);
    }
    /**
     * Write a log entry with a severity of "ERROR".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.error(entry);
     * ```
     */
    error(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'ERROR'), options);
    }
    /**
     * Write a log entry with a severity of "INFO".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.info(entry);
     * ```
     */
    info(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'INFO'), options);
    }
    /**
     * Write a log entry with a severity of "NOTICE".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.notice(entry);
     * ```
     */
    notice(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'NOTICE'), options);
    }
    /**
     * Write a log entry with a severity of "WARNING".
     *
     * This is a simple wrapper around {@link LogSync#write}. All arguments are
     * the same as documented there.
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     * @example
     * ```
     * const {Logging} = require('@google-cloud/logging');
     * const logging = new Logging();
     * const log = logging.logSync('my-log');
     *
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.warning(entry);
     * ```
     */
    warning(entry, options) {
        this.write(log_common_1.assignSeverityToEntries(entry, 'WARNING'), options);
    }
    /**
     * Write log entries to a custom transport (default: process.stdout).
     *
     * @param {Entry|Entry[]} entry A log entry, or array of entries, to write.
     * @param {?WriteOptions} [options] Write options
     *
     * @example
     * ```
     * const entry = log.entry('gce_instance', {
     *   instance: 'my_instance'
     * });
     *
     * log.write(entry);
     *
     * //-
     * // You may also pass multiple log entries to write.
     * //-
     * const secondEntry = log.entry('compute.googleapis.com', {
     *   user: 'my_username'
     * });
     *
     * log.write([entry, secondEntry]);
     *
     * //-
     * // To save some steps, you can also pass in plain values as your entries.
     * // Note, however, that you must provide a configuration object to specify
     * // the resource.
     * //-
     * const entries = [
     *   {
     *     user: 'my_username'
     *   },
     *   {
     *     home: process.env.HOME
     *   }
     * ];
     *
     * const options = {
     *   resource: 'compute.googleapis.com'
     * };
     *
     * log.write(entries, options);
     *
     * log.write(entries);
     * });
     * ```
     */
    write(entry, opts) {
        var _a;
        const options = opts ? opts : {};
        // We expect projectId and resource to be set before this fn is called...
        let structuredEntries;
        this.formattedName_ = log_common_1.formatLogName(this.logging.projectId, this.name);
        try {
            // Make sure to add instrumentation info
            structuredEntries = instrumentation_1.populateInstrumentationInfo(entry).map(entry => {
                if (!(entry instanceof entry_1.Entry)) {
                    entry = this.entry(entry);
                }
                return entry.toStructuredJSON(this.logging.projectId);
            });
            for (const entry of structuredEntries) {
                entry.logName = this.formattedName_;
                entry.resource =
                    log_common_1.snakecaseKeys((_a = options.resource) === null || _a === void 0 ? void 0 : _a.labels) ||
                        entry.resource ||
                        this.logging.detectedResource;
                entry[entry_1.LABELS_KEY] = options.labels || entry[entry_1.LABELS_KEY];
                this.transport.write(JSON.stringify(entry) + '\n');
            }
        }
        catch (err) {
            // Ignore errors (client libraries do not panic).
        }
    }
}
exports.LogSync = LogSync;
//# sourceMappingURL=log-sync.js.map