/**
* HttpManager.ts
* @author Abhilash Panwar (abpanwar); Hector Hernandez (hectorh); Nev Wylie (newylie)
* @copyright Microsoft 2018-2020
*/
import { IExtendedAppInsightsCore, SendRequestReason, EventSendType } from "@microsoft/1ds-core-js";
import { IXHROverride, IPostChannel, PayloadPreprocessorFunction, PayloadListenerFunction } from "./DataModels";
import { BatchNotificationActions } from "./BatchNotificationActions";
import { EventBatch } from "./EventBatch";
import { IChannelConfiguration } from "./Index";
import { ITimeoutOverrideWrapper } from "./TimeoutOverrideWrapper";
/**
 * Class managing the sending of requests.
 */
export declare class HttpManager {
    sendHook: PayloadPreprocessorFunction | undefined;
    sendListener: PayloadListenerFunction | undefined;
    _responseHandlers: Array<(responseText: string) => void>;
    /**
     * @constructor
     * @param requestQueue   - The queue that contains the requests to be sent.
     */
    constructor(maxEventsPerBatch: number, maxConnections: number, maxRequestRetriesBeforeBackoff: number, actions: BatchNotificationActions, timeoutOverride: ITimeoutOverrideWrapper);
    /**
     * @constructor
     * @param requestQueue   - The queue that contains the requests to be sent.
     * @param endpointUrl   - The collector url to which the requests must be sent.
     * @param postManager   - The post manager that we should add requests back to if needed.
     * @param httpInterface - The http interface that should be used to send HTTP requests.
     * @param channelConfig - The IChannelConfiguration the should be used to get additional configuration
     */
    initialize(endpointUrl: string, core: IExtendedAppInsightsCore, postChannel: IPostChannel, httpInterface: IXHROverride, channelConfig?: IChannelConfiguration): void;
    /**
     * Add query string parameter to url
     * @param name   - Query string name.
     * @param value  - Query string value.
     */
    addQueryStringParameter(name: string, value: string): void;
    /**
     * Add header to request
     * @param name   - Header name.
     * @param value  - Header value.
     */
    addHeader(name: string, value: string): void;
    /**
     * Add the batch of events to the queue for sending
     * @param batch The batch with the events to send
     * @returns True if the http manager has accepted the batch (including if the batch is empty) otherwise false
     */
    addBatch(batch: EventBatch): boolean;
    /**
     * Check if there is an idle connection and we can send a request.
     * @returns True if there is an idle connection, false otherwise.
     */
    canSendRequest(): boolean;
    /**
     * Send requests in the request queue up if there is an idle connection, sending is
     * not pause and clock skew manager allows sending request.
     * @param sendType - Identifies how the batched events should be send, defaults to Batched
     * @param sendReason   - The reason the batch is being sent
     */
    sendQueuedRequests(sendType?: EventSendType, sendReason?: SendRequestReason): void;
    /**
     * Check if there are no active requests being sent.
     * @returns True if idle, false otherwise.
     */
    isCompletelyIdle(): boolean;
    /**
     * Inform the HttpManager that a page unload event was received
     */
    setUnloading(value: boolean): void;
    /**
     * Queue all the remaining requests to be sent. The requests will be
     * sent using HTML5 Beacons if they are available.
     */
    teardown(): void;
    /**
     * Pause the sending of requests. No new requests will be sent.
     */
    pause(): void;
    /**
     * Resume the sending of requests.
     */
    resume(): void;
    /**
     * Sends the batches synchronously to the collector. This api is used to send a batches immediate event.
     *
     * @param batch - The batch of events to be sent.
     * @param sendReason   - The reason the batch is being sent
     * @param sendType - Identifies the sending type to use when sending the batch
     */
    sendSynchronousBatch(batch: EventBatch, sendType?: EventSendType, sendReason?: SendRequestReason): void;
}
