/*!
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { google } from '../protos/protos';
import { CloudLoggingHttpRequest, RawHttpRequest } from './utils/http-request';
export declare const INSERT_ID_KEY = "logging.googleapis.com/insertId";
export declare const LABELS_KEY = "logging.googleapis.com/labels";
export declare const OPERATION_KEY = "logging.googleapis.com/operation";
export declare const SOURCE_LOCATION_KEY = "logging.googleapis.com/sourceLocation";
export declare const SPAN_ID_KEY = "logging.googleapis.com/spanId";
export declare const TRACE_KEY = "logging.googleapis.com/trace";
export declare const TRACE_SAMPLED_KEY = "logging.googleapis.com/trace_sampled";
export declare type Timestamp = google.protobuf.ITimestamp | Date | string;
export declare type LogSeverity = google.logging.type.LogSeverity | string;
export declare type HttpRequest = google.logging.type.IHttpRequest | CloudLoggingHttpRequest | RawHttpRequest;
export declare type LogEntry = Omit<google.logging.v2.ILogEntry, 'timestamp' | 'severity' | 'httpRequest'> & {
    timestamp?: Timestamp | null;
    severity?: LogSeverity | null;
    httpRequest?: HttpRequest | null;
};
export declare type Data = any;
export interface EntryJson {
    timestamp: Timestamp;
    insertId: number;
    jsonPayload?: google.protobuf.IStruct;
    textPayload?: string;
    httpRequest?: google.logging.type.IHttpRequest;
    trace?: string;
    spanId?: string;
    traceSampled?: boolean;
}
export interface StructuredJson {
    message?: string | object;
    httpRequest?: object;
    timestamp?: string;
    [INSERT_ID_KEY]?: string;
    [OPERATION_KEY]?: object;
    [SOURCE_LOCATION_KEY]?: object;
    [LABELS_KEY]?: object;
    [SPAN_ID_KEY]?: string;
    [TRACE_KEY]?: string;
    [TRACE_SAMPLED_KEY]?: boolean;
    logName?: string;
    resource?: object;
}
export interface ToJsonOptions {
    removeCircular?: boolean;
}
/**
 * Create an entry object to define new data to insert into a meta.
 *
 * Note, {@link https://cloud.google.com/logging/quotas|Cloud Logging Quotas and limits}
 * dictates that the maximum log entry size, including all
 * {@link https://cloud.google.com/logging/docs/reference/v2/rest/v2/LogEntry|LogEntry Resource properties},
 * cannot exceed approximately 256 KB.
 *
 * See {@link https://cloud.google.com/logging/docs/reference/v2/rest/v2/LogEntry|LogEntry JSON representation}
 *
 * @class
 *
 * @param {?object} [metadata] See a
 *     [LogEntry
 * Resource](https://cloud.google.com/logging/docs/reference/v2/rest/v2/LogEntry).
 * @param {object|string} data The data to use as the value for this log
 *     entry.
 *
 *     If providing an object, these value types are supported:
 *     - `String`
 *     - `Number`
 *     - `Boolean`
 *     - `Buffer`
 *     - `Object`
 *     - `Array`
 *
 *     Any other types are stringified with `String(value)`.
 *
 * @example
 * ```
 * const {Logging} = require('@google-cloud/logging');
 * const logging = new Logging();
 * const syslog = logging.log('syslog');
 *
 * const metadata = {
 *   resource: {
 *     type: 'gce_instance',
 *     labels: {
 *       zone: 'global',
 *       instance_id: '3'
 *     }
 *   }
 * };
 *
 * const entry = syslog.entry(metadata, {
 *   delegate: 'my_username'
 * });
 *
 * syslog.alert(entry, (err, apiResponse) => {
 *   if (!err) {
 *     // Log entry inserted successfully.
 *   }
 * });
 *
 * //-
 * // You will also receive `Entry` objects when using
 * // Logging#getEntries() and Log#getEntries().
 * //-
 * logging.getEntries((err, entries) => {
 *   if (!err) {
 *     // entries[0].data = The data value from the log entry.
 *   }
 * });
 * ```
 */
declare class Entry {
    metadata: LogEntry;
    data: Data;
    constructor(metadata?: LogEntry, data?: Data);
    /**
     * Serialize an entry to the format the API expects. Read more:
     * https://cloud.google.com/logging/docs/reference/v2/rest/v2/LogEntry
     *
     * @param {object} [options] Configuration object.
     * @param {boolean} [options.removeCircular] Replace circular references in an
     *     object with a string value, `[Circular]`.
     * @param {string} [projectId] GCP Project ID.
     */
    toJSON(options?: ToJsonOptions, projectId?: string): EntryJson;
    /**
     * Serialize an entry to a standard format for any transports, e.g. agents.
     * Read more: https://cloud.google.com/logging/docs/structured-logging
     */
    toStructuredJSON(projectId?: string): StructuredJson;
    /**
     * extractTraceFromHeaders extracts trace and span information from raw HTTP
     * request headers only.
     * @private
     */
    private extractTraceFromHeaders;
    /**
     * Create an Entry object from an API response, such as `entries:list`.
     *
     * @private
     *
     * @param {object} entry An API representation of an entry. See a
     *     {@link https://cloud.google.com/logging/docs/reference/v2/rest/v2/LogEntry| LogEntry}.
     * @returns {Entry}
     */
    static fromApiResponse_(entry: google.logging.v2.LogEntry): Entry;
}
/**
 * Reference to the {@link Entry} class.
 * @name module:@google-cloud/logging.Entry
 * @see Entry
 */
export { Entry };
