"use strict";
/*!
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sink = void 0;
const promisify_1 = require("@google-cloud/promisify");
const extend = require("extend");
/**
 * A sink is an object that lets you to specify a set of log entries to export
 * to a particular destination. Cloud Logging lets you export log entries
 * to destinations including Cloud Storage buckets (for long term log
 * storage), Google BigQuery datasets (for log analysis), Google Pub/Sub (for
 * streaming to other applications).
 *
 * See {@link https://cloud.google.com/logging/docs/basic-concepts#sinks|Introduction to Sinks}
 *
 * @class
 *
 * @param {Logging} logging {@link Logging} instance.
 * @param {string} name Name of the sink.
 *
 * @example
 * ```
 * const {Logging} = require('@google-cloud/logging');
 * const logging = new Logging();
 * const sink = logging.sink('my-sink');
 * ```
 */
class Sink {
    constructor(logging, name) {
        this.logging = logging;
        /**
         * @name Sink#name
         * @type {string}
         */
        this.name = name;
        this.formattedName_ = 'projects/' + logging.projectId + '/sinks/' + name;
    }
    create(config) {
        return this.logging.createSink(this.name, config);
    }
    async delete(gaxOptions) {
        const projectId = await this.logging.auth.getProjectId();
        this.formattedName_ = 'projects/' + projectId + '/sinks/' + this.name;
        const reqOpts = {
            sinkName: this.formattedName_,
        };
        return this.logging.configService.deleteSink(reqOpts, gaxOptions);
    }
    async getMetadata(gaxOptions) {
        const projectId = await this.logging.auth.getProjectId();
        this.formattedName_ = 'projects/' + projectId + '/sinks/' + this.name;
        const reqOpts = {
            sinkName: this.formattedName_,
        };
        [this.metadata] = await this.logging.configService.getSink(reqOpts, gaxOptions);
        return [this.metadata];
    }
    setFilter(filter) {
        return this.setMetadata({
            filter,
        });
    }
    async setMetadata(metadata) {
        const [currentMetadata] = await this.getMetadata();
        const uniqueWriterIdentity = metadata.uniqueWriterIdentity;
        delete metadata.uniqueWriterIdentity;
        let reqOpts = {
            sinkName: this.formattedName_,
            sink: extend({}, currentMetadata, metadata),
        };
        delete reqOpts.sink.gaxOptions;
        // Add user specified uniqueWriterIdentity boolean, if any.
        reqOpts = {
            ...reqOpts,
            ...(uniqueWriterIdentity && { uniqueWriterIdentity }),
        };
        [this.metadata] = await this.logging.configService.updateSink(reqOpts, metadata.gaxOptions);
        return [this.metadata];
    }
}
exports.Sink = Sink;
/*! Developer Documentation
 *
 * All async methods (except for streams) will call a callbakc in the event
 * that a callback is provided.
 */
promisify_1.callbackifyAll(Sink);
//# sourceMappingURL=sink.js.map