/*
 * Copyright (c) 2022-2024, NVIDIA CORPORATION.  All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "tensorrt_llm/common/stringUtils.h"
#include "tensorrt_llm/common/assert.h"

#include <cerrno>
#include <cstdarg>
#include <cstring>
#include <iostream>
#include <string>

namespace tensorrt_llm::common
{

namespace
{
std::string vformat(char const* fmt, va_list args)
{
    va_list args0;
    va_copy(args0, args);
    auto const size = vsnprintf(nullptr, 0, fmt, args0);
    if (size <= 0)
        return "";

    std::string stringBuf(size, char{});
    auto const size2 = std::vsnprintf(&stringBuf[0], size + 1, fmt, args);

    TLLM_CHECK_WITH_INFO(size2 == size, std::string(std::strerror(errno)));

    return stringBuf;
}

} // namespace

std::string fmtstr(char const* format, ...)
{
    va_list args;
    va_start(args, format);
    std::string result = vformat(format, args);
    va_end(args);
    return result;
};

std::unordered_set<std::string> str2set(std::string const& input, char delimiter)
{
    std::unordered_set<std::string> values;
    if (!input.empty())
    {
        std::stringstream valStream(input);
        std::string val;
        while (std::getline(valStream, val, delimiter))
        {
            if (!val.empty())
            {
                values.insert(val);
            }
        }
    }
    return values;
};

} // namespace tensorrt_llm::common
