from __future__ import annotations

import os.path

from .exception import SeekException, SeekTODOException
from typing import List, Union, Tuple, Any, Optional, TextIO
import inspect
import textwrap
import ast
import astpretty


class _Global:
    """
    A global namespace for Seek internal usage
    """
    _current_block_stack = []  # type: List["Block"]  # noqa: F821 (undefined name 'Block')

    @staticmethod
    def current_block() -> "Block":  # noqa: F821 (undefined name 'Block')
        if len(_Global._current_block_stack) == 0:
            raise SeekException("No active current block entered")
        return _Global._current_block_stack[-1]


def _inspect_stack_frame(prev_at: int):
    frame = inspect.currentframe()
    assert frame is not None, "Unimplemented inspect.currentframe()"
    for _ in range(0, prev_at + 1):
        frame = frame.f_back
        if frame is None:
            return None
    return inspect.FrameInfo(frame, *inspect.getframeinfo(frame, 1))


class SrcLoc:
    __slots__ = ["__filename", "__lineno"]

    def __init__(self, filename: str, lineno: int):
        self.__filename = os.path.basename(filename)
        self.__lineno = lineno

    @property
    def filename(self) -> str:
        return self.__filename

    @property
    def lineno(self) -> int:
        return self.__lineno

    def __repr__(self):
        return f"SrcLoc(filename={self.filename!r}, lineno={self.lineno})"

    def __str__(self):
        return f"{self.filename}:{self.lineno}"

    def __eq__(self, other):
        if not isinstance(other, SrcLoc):
            return NotImplemented
        return (self.filename, self.lineno) == (other.filename, other.lineno)

    def __hash__(self):
        return hash((self.filename, self.lineno))

    @staticmethod
    def get_caller_srcloc(stack: int = 1) -> SrcLoc:
        assert stack >= 0, f"Unknown stack depth: {stack}"
        frame = _inspect_stack_frame(stack + 1)  # plus myself
        if frame is None:
            raise SeekException(f"Can't get frame at stack depth {stack} from bottom")

        return SrcLoc(frame.filename, frame.lineno)


def get_caller_assignments(stack: int = 1) -> Optional[Union[str, Tuple, List]]:
    """
    Get the caller line; try to split it and get assignment targets
    """
    assert stack >= 0, f"Unknown stack depth: {stack}"
    frame = _inspect_stack_frame(stack + 1)  # plus myself
    if frame is None:
        raise SeekException(f"Can't get frame at stack depth {stack} from bottom")

    # Seems `code_context` contains just 1 line (at least for Python3)
    # See: https://stackoverflow.com/questions/58720279
    source = "".join(frame.code_context)

    # De-indent this line, if necessary
    source = textwrap.dedent(source)

    try:
        root_node = ast.parse(source, filename=frame.filename, mode="exec")
    except SyntaxError:
        return None

    assert isinstance(root_node, ast.Module)
    body = root_node.body

    assert isinstance(body, list)
    assert len(body) == 1
    body0 = body[0]

    def get_target(target: Any):
        if isinstance(target, ast.Name):
            return target.id
        elif isinstance(target, ast.Attribute) and isinstance(target.value, ast.Name):
            return f"{target.value.id}.{target.attr}"
        elif isinstance(target, ast.Tuple):
            return tuple(get_target(elt) for elt in target.elts)
        elif isinstance(target, ast.List):
            return list(get_target(elt) for elt in target.elts)
        else:
            astpretty.pprint(target)
            raise SeekTODOException(target)

    if isinstance(body0, ast.Assign):  # (simple) assignment
        assign_targets = body0.targets
        assert len(assign_targets) > 0
        if len(assign_targets) > 1:
            # Case like: `a = b = get_caller_assignments(stack=0)`
            raise SeekException("Assignment to multiple targets is not supported")
        return get_target(assign_targets[0])
    elif isinstance(body0, ast.AnnAssign):  # annotated assignment
        return get_target(body0.target)
    else:
        # This is not an assignment
        return None


class IndentedWriter:
    def __init__(self, fp: TextIO, indent: int = 0):
        self._fp = fp
        self._indent = indent

    def __call__(self, s: str = ""):
        s = textwrap.indent(s + "\n", ' ' * self._indent)
        self._fp.write(s)

    def indent(self, num: int = 4):
        class Indenter:
            def __init__(self, writer: IndentedWriter):
                self.writer = writer

            def __enter__(self):
                self.writer._indent += num

            def __exit__(self, exc_type, exc_val, exc_tb):
                self.writer._indent -= num
        return Indenter(self)
