from typing import Set
from abc import ABC
import enum


class PassTag(enum.IntEnum):
    DivideBasicBlock = 0
    OptimizeBasicBlock = 100
    AnalyzeLiveVar = 110
    EliminateDeadCode = 120
    AnnotateClause = 200
    InsertWaitcnt = 300
    ComputeRegisterInterference = 310
    AllocateRegisterRIG = 400
    PrintBasicBlock = 1000


class OptimizerState:
    def __init__(self):
        self.divide_basic_block = None

        # For ComputeRegisterInterferencePass
        self.register_interference_vgpr = None
        self.register_interference_sgpr = None
        self.register_interference_perfect_elimination_ordering_vgpr = None
        self.register_interference_perfect_elimination_ordering_sgpr = None

        # For AllocateRegisterRIGPass
        self.register_allocation_vgpr_by_color = None  # # type: List[List[Gpr]]
        self.register_allocation_sgpr_by_color = None  # # type: List[List[Gpr]]
        self.register_allocation_vgpr_count = None  # # type: int
        self.register_allocation_sgpr_count = None  # # type: int


class BasePass(ABC):
    def __init__(self, /, priority: int):
        """
        Specify smaller integer for higher priority (aka. the passes runs earlier).
        By default, priority is the (integer) value of pass's tag.
        """
        if self.__class__.__name__ == 'BasePass':
            raise NotImplementedError("Can't instantiate BasePass")
        assert isinstance(priority, int)
        self.__priority = priority  # type: int

    @property
    def priority(self) -> int:
        return self.__priority

    def required_tags(self) -> Set[PassTag]:
        raise NotImplementedError("Override required_tags() in derived class")

    def generated_tags(self) -> Set[PassTag]:
        raise NotImplementedError("Override generated_tags() in derived class")

    def invalidated_tags(self) -> Set[PassTag]:
        raise NotImplementedError("Override invalidated_tags() in derived class")

    def reset(self, program):
        raise NotImplementedError("Override reset() in derived class")

    def run(self, program) -> bool:
        """
        Returns true if this pass modifies anything. Returns false otherwise.
        """
        raise NotImplementedError("Override run() in derived class")

    def __repr__(self):  # virtual method
        return self.__class__.__name__
