"""
Copyright 2020 The OneFlow Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""
from typing import List

import oneflow as flow


def affine_grid(theta, size: List[int], align_corners: bool = False):
    """The interface is consistent with PyTorch.    
    The documentation is referenced from: 
    https://pytorch.org/docs/1.10/generated/torch.nn.functional.affine_grid.html.

    Generates a 2D or 3D flow field (sampling grid), given a batch of
    affine matrices :attr:`theta`.

    .. note::
        This function is often used in conjunction with :func:`grid_sample`
        to build `Spatial Transformer Networks`_ .

    Args:
        theta (Tensor): input batch of affine matrices with shape
            (:math:`N, 2, 3`) for 2D or
            (:math:`N, 3, 4`) for 3D
        size (oneflow.Size): the target output image size.
            (:math:`N, C, H, W` for 2D or
            :math:`N, C, D, H, W` for 3D)
            Example: oneflow.Size((32, 3, 24, 24))
        align_corners (bool): if ``True``, consider ``-1`` and ``1``
            to refer to the centers of the corner pixels rather than the image corners.
            Refer to :func:`grid_sample` for a more complete description.
            A grid generated by :func:`affine_grid` should be passed to :func:`grid_sample`
            with the same setting for this option.
            Default: ``False``

    Returns:
        output (Tensor): output Tensor of size (:math:`N, H, W, 2`)

    .. _`Spatial Transformer Networks`:
        https://arxiv.org/abs/1506.02025

    Examples::

        >>> import oneflow as flow
        >>> import numpy as np
        >>> input = flow.tensor(np.arange(1., 7).reshape((1, 2, 3)), dtype=flow.float32)
        >>> output = flow.nn.functional.affine_grid(input, flow.Size([1, 1, 2, 2]), align_corners=True)
        >>> output
        tensor([[[[ 0., -3.],
                  [ 2.,  5.]],
        <BLANKLINE>
                 [[ 4.,  7.],
                  [ 6., 15.]]]], dtype=oneflow.float32)
    """
    y = flow._C.affine_grid(theta, size=size, align_corners=align_corners)
    return y


if __name__ == "__main__":
    import doctest

    doctest.testmod(raise_on_error=True)
