import { app, BrowserWindow, autoUpdater, dialog } from 'electron'
import { spawn, exec } from 'child_process'
import * as path from 'path'
import * as fs from 'fs'

require('@electron/remote/main').initialize()

// This allows TypeScript to pick up the magic constants that's auto-generated by Forge's Webpack
// plugin that tells the Electron app where to look for the Webpack-bundled app code (depending on
// whether you're running in development or production).
declare const MAIN_WINDOW_WEBPACK_ENTRY: string
declare const MAIN_WINDOW_PRELOAD_WEBPACK_ENTRY: string

// Handle creating/removing shortcuts on Windows when installing/uninstalling.
if (require('electron-squirrel-startup')) {
  app.quit()
}

const createWindow = (): void => {
  // Create the browser window.
  // Create the browser window.
  const mainWindow = new BrowserWindow({
    width: 800,
    height: 600,
    minWidth: 400,
    minHeight: 300,
    titleBarStyle: 'hiddenInset',
    webPreferences: {
      preload: MAIN_WINDOW_PRELOAD_WEBPACK_ENTRY,
      nodeIntegration: true,
      contextIsolation: false,
    },
  })

  require('@electron/remote/main').enable(mainWindow.webContents)

  // and load the index.html of the app.
  mainWindow.loadURL(MAIN_WINDOW_WEBPACK_ENTRY)
}

const ollama = path.join(process.resourcesPath, 'ollama')

// if the app is packaged then run the server
if (app.isPackaged) {
  // Start the executable
  console.log(`Starting server`)
  const proc = spawn(ollama, ['serve'])
  proc.stdout.on('data', data => {
    console.log(`server: ${data}`)
  })
  proc.stderr.on('data', data => {
    console.error(`server: ${data}`)
  })

  process.on('exit', () => {
    proc.kill()
  })
}

function server() {
  const binary = app.isPackaged
    ? path.join(process.resourcesPath, 'ollama')
    : path.resolve(__dirname, '..', '..', 'ollama')

  console.log(`Starting server`)
  const proc = spawn(binary, ['serve'])
  proc.stdout.on('data', data => {
    console.log(`server: ${data}`)
  })
  proc.stderr.on('data', data => {
    console.error(`server: ${data}`)
  })

  process.on('exit', () => {
    proc.kill()
  })
}

function installCLI() {
  const symlinkPath = '/usr/local/bin/ollama'

  if (fs.existsSync(symlinkPath) && fs.readlinkSync(symlinkPath) === ollama) {
    return
  }

  dialog
    .showMessageBox({
      type: 'info',
      title: 'Ollama CLI installation',
      message: 'To install the Ollama CLI, we need to ask you for administrator privileges.',
      buttons: ['OK'],
    })
    .then(result => {
      if (result.response === 0) {
        let command = `
    do shell script "ln -F -s ${ollama} /usr/local/bin/ollama" with administrator privileges
    `
        exec(`osascript -e '${command}'`, (error: Error | null, stdout: string, stderr: string) => {
          if (error) {
            console.error(`exec error: ${error}`)
            return
          }
          console.log(`stdout: ${stdout}`)
          console.error(`stderr: ${stderr}`)
        })
      }
    })
}

// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.on('ready', () => {
  createWindow()

  if (app.isPackaged) {
    installCLI()
  }
})

// Quit when all windows are closed, except on macOS. There, it's common
// for applications and their menu bar to stay active until the user quits
// explicitly with Cmd + Q.
app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit()
  }
})

app.on('activate', () => {
  // On OS X it's common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (BrowserWindow.getAllWindows().length === 0) {
    createWindow()
  }
})

// In this file you can include the rest of your app's specific main process
// code. You can also put them in separate files and import them here.
autoUpdater.setFeedURL({
  url: `https://ollama.ai/api/update?os=${process.platform}&arch=${process.arch}&version=${app.getVersion()}`,
})

autoUpdater.checkForUpdates()
setInterval(() => {
  autoUpdater.checkForUpdates()
}, 60000)

autoUpdater.on('error', e => {
  console.error('update check failed', e)
})

autoUpdater.on('update-downloaded', (event, releaseNotes, releaseName) => {
  dialog
    .showMessageBox({
      type: 'info',
      buttons: ['Restart Now', 'Later'],
      title: 'New update available',
      message: process.platform === 'win32' ? releaseNotes : releaseName,
      detail: 'A new version of Ollama is available. Restart to apply the update.',
    })
    .then(returnValue => {
      if (returnValue.response === 0) autoUpdater.quitAndInstall()
    })
})
