import os
import re
import sys

from setuptools import find_packages, setup

import subprocess
from typing import Optional, Union
from pathlib import Path
import torch
import shutil

pwd = os.path.dirname(__file__)
version_file = 'lmdeploy/version.py'


def readme():
    with open(os.path.join(pwd, 'README.md'), encoding='utf-8') as f:
        content = f.read()
    return content


def get_sha(pytorch_root: Union[str, Path]) -> str:
    try:
        return subprocess.check_output(['git', 'rev-parse', 'HEAD'], cwd=pytorch_root).decode('ascii').strip()
    except Exception:
        return 'Unknown'


def get_abi():
    try:
        command = "echo '#include <string>' | gcc -x c++ -E -dM - | fgrep _GLIBCXX_USE_CXX11_ABI" 
        result = subprocess.run(command, shell=True, capture_output=True, text=True) 
        output = result.stdout.strip() 
        abi = "abi" + output.split(" ")[-1]
        return abi
    except Exception:
        return 'abiUnknown'


def get_version_add(sha: Optional[str] = None) -> str:
    version=''
    lmdeploy_root = os.path.dirname(os.path.abspath(__file__))
    add_version_path = os.path.join(os.path.join(lmdeploy_root, "lmdeploy"), "version.py")
    if sha != 'Unknown':
        if sha is None:
            sha = get_sha(lmdeploy_root)
        version = 'git' + sha[:7]

    # abi
    version += "." + get_abi()

    # dtk version
    if os.getenv("ROCM_PATH"):
        rocm_path = os.getenv('ROCM_PATH', "")
        rocm_version_path = os.path.join(rocm_path, '.info', "rocm_version")
        with open(rocm_version_path, 'r',encoding='utf-8') as file:
            lines = file.readlines()
        rocm_version=lines[0][:-2].replace(".", "")
        version += ".dtk" + rocm_version
    
    # torch version
    version += ".torch" + torch.__version__[:5]

    lines=[]
    with open(add_version_path, 'r',encoding='utf-8') as file:
        lines = file.readlines()
    lines[2] = "__dcu_version__ = '0.2.6+das1.1.{}'\n".format(version)
    with open(add_version_path, encoding="utf-8",mode="w") as file:
        file.writelines(lines)
    file.close()

def get_version():
    get_version_add()
    version_file = 'lmdeploy/version.py'
    with open(version_file, encoding='utf-8') as f:
        exec(compile(f.read(), version_file, 'exec'))
    return locals()['__dcu_version__']


def check_ext_modules():
    if os.path.exists(os.path.join(pwd, 'lmdeploy', 'lib')):
        return True
    return False


def get_cuda_pkgs():
    arg_name = '--cuda='
    arg_value = None
    for arg in sys.argv[1:]:
        if arg.startswith(arg_name):
            arg_value = arg[len(arg_name):]
            sys.argv.remove(arg)
            break

    cuda_pkgs = []
    if arg_value == '11':
        cuda_pkgs = [
            'nvidia-nccl-cu11', 'nvidia-cuda-runtime-cu11',
            'nvidia-cublas-cu11'
        ]
    elif arg_value == '12':
        cuda_pkgs = [
            'nvidia-nccl-cu12', 'nvidia-cuda-runtime-cu12',
            'nvidia-cublas-cu12'
        ]
    return cuda_pkgs


cuda_pkgs = get_cuda_pkgs()


def parse_requirements(fname='requirements.txt', with_version=True):
    """Parse the package dependencies listed in a file but strips specific
    versioning information.

    Args:
        fname (str): path to the file
        with_version (bool, default=False): if True include version specs

    Returns:
        List[str]: list of requirements items

    CommandLine:
        python -c "import setup; print(setup.parse_requirements())"
    """
    require_fpath = fname

    def parse_line(line):
        """Parse information from a line in a requirements text file."""
        if line.startswith('-r '):
            # Allow specifying requirements in other files
            target = line.split(' ')[1]
            for info in parse_require_file(target):
                yield info
        else:
            info = {'line': line}
            if line.startswith('-e '):
                info['package'] = line.split('#egg=')[1]
            elif '@git+' in line:
                info['package'] = line
            else:
                # Remove versioning from the package
                pat = '(' + '|'.join(['>=', '==', '>']) + ')'
                parts = re.split(pat, line, maxsplit=1)
                parts = [p.strip() for p in parts]

                info['package'] = parts[0]
                if len(parts) > 1:
                    op, rest = parts[1:]
                    if ';' in rest:
                        # Handle platform specific dependencies
                        # http://setuptools.readthedocs.io/en/latest/setuptools.html#declaring-platform-specific-dependencies
                        version, platform_deps = map(str.strip,
                                                     rest.split(';'))
                        info['platform_deps'] = platform_deps
                    else:
                        version = rest  # NOQA
                    info['version'] = (op, version)
            yield info

    def parse_require_file(fpath):
        with open(fpath, 'r') as f:
            for line in f.readlines():
                line = line.strip()
                if line and not line.startswith('#'):
                    for info in parse_line(line):
                        yield info

    def gen_packages_items():
        if os.path.exists(require_fpath):
            for info in parse_require_file(require_fpath):
                parts = [info['package']]
                if with_version and 'version' in info:
                    parts.extend(info['version'])
                if not sys.version.startswith('3.4'):
                    # apparently package_deps are broken in 3.4
                    platform_deps = info.get('platform_deps')
                    if platform_deps is not None:
                        parts.append(';' + platform_deps)
                item = ''.join(parts)
                yield item

    packages = list(gen_packages_items())
    packages += cuda_pkgs
    return packages

def copy_ck_so():
    lmdeploy_root = os.path.dirname(os.path.abspath(__file__))
    so_path = os.path.join(os.path.join(lmdeploy_root, "3rdparty","composable_kernel"), "libgemm_multiB_int4.so")
    # dtk version
    target_path=os.path.join(lmdeploy_root, "lmdeploy","lib")
    if os.path.exists(target_path):
        shutil.copy(so_path, target_path)
    elif os.getenv("ROCM_PATH"):
        rocm_path = os.getenv('ROCM_PATH', "")    
        rocm_so_path = os.path.join(rocm_path, 'lib')
        print("rocm_so_path:",rocm_so_path)
        shutil.copy(so_path, rocm_so_path)
    else:
        shutil.copy(so_path, "usr/local/lib")

if __name__ == '__main__':
    lmdeploy_package_data = ['lmdeploy/bin/llama_gemm']
    copy_ck_so()
    setup(
        name='lmdeploy',
        version=get_version(),
        description='A toolset for compressing, deploying and serving LLM',
        long_description=readme(),
        long_description_content_type='text/markdown',
        author='OpenMMLab',
        author_email='openmmlab@gmail.com',
        packages=find_packages(exclude=()),
        
        package_data={
            'lmdeploy': lmdeploy_package_data,
        },
        include_package_data=True,
        setup_requires=parse_requirements('requirements/build.txt'),
        tests_require=parse_requirements('requirements/test.txt'),
        install_requires=parse_requirements('requirements/runtime.txt'),
        extras_require={
            'all': parse_requirements('requirements.txt'),
            'lite': parse_requirements('requirements/lite.txt'),
            'serve': parse_requirements('requirements/serve.txt')
        },
        has_ext_modules=check_ext_modules,
        classifiers=[
            'Programming Language :: Python :: 3.8',
            'Programming Language :: Python :: 3.9',
            'Programming Language :: Python :: 3.10',
            'Programming Language :: Python :: 3.11',
            'Intended Audience :: Developers',
            'Intended Audience :: Education',
            'Intended Audience :: Science/Research',
        ],
        entry_points={'console_scripts': ['lmdeploy = lmdeploy.cli:run']},
    )