/***************************************************************************************************
 * Copyright (c) 2023 - 2025 Hygon Information Technology Co., Ltd. All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 **************************************************************************************************/
/*! \file
    \brief Unit tests for thread-level GEMM
*/

#include "../../common/hytlass_unit_test.h"

#include "hytlass/aligned_buffer.h"

#include "hytlass/gemm/warp/mma_simt.h"
#include "hytlass/gemm/warp/mma_simt_policy.h"

#include "hytlass/epilogue/thread/linear_combination.h"    
#include "hytlass/epilogue/threadblock/default_epilogue_simt.h"

#include "hytlass/util/host_tensor.h"
#include "hytlass/util/tensor_view_io.h"
#include "hytlass/util/reference/host/tensor_fill.h"

#include "testbed.h"

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Real-valued single precision tests
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementOutput = float;
  using ElementAccumulator = float;
  using ElementCompute = float;
  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<32, 64, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;
  using Element = float;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_32x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementOutput = float;
  using ElementAccumulator = float;
  using ElementCompute = float;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;
  using Element = float;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_64x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementOutput = float;
  using ElementAccumulator = float;
  using ElementCompute = float;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<64, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;
  using Element = float;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_128x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementOutput = float;
  using ElementAccumulator = float;
  using ElementCompute = float;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;
  using Element = float;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Real-valued double precision tests
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f64_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = double;
  using ElementOutput = double;
  using ElementAccumulator = double;
  using ElementCompute = double;
  
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 64, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<2, 2, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f64_32x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = double;
  using ElementOutput = double;
  using ElementAccumulator = double;
  using ElementCompute = double;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<2, 2, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f64_64x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = double;
  using ElementOutput = double;
  using ElementAccumulator = double;
  using ElementCompute = double;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<64, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<2, 2, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f64_128x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = double;
  using ElementOutput = double;
  using ElementAccumulator = double;
  using ElementCompute = double;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<2, 2, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Complex-valued single-precision
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_complex_f32_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::complex<float>;
  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 64, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;
  
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<2, 2, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_complex_f32_32x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::complex<float>;
  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<2, 2, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_complex_f32_128x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::complex<float>;
  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<2, 2, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}


/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Complex-valued double-precision
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_complex_f64_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::complex<double>;
  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 64, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<1, 1, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_complex_f64_32x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::complex<double>;
  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<1, 1, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_complex_f64_128x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::complex<double>;
  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;
  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;

  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<1, 1, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Real-valued half precision tests
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f16_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::half_t;
  using ElementOutput = hytlass::half_t;
  using ElementAccumulator = hytlass::half_t;
  using ElementCompute = hytlass::half_t;

  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<32, 64, 8>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 8>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f16_64x64_64x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::half_t;
  using ElementOutput = hytlass::half_t;
  using ElementAccumulator = hytlass::half_t;
  using ElementCompute = hytlass::half_t;

  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<64, 64, 8>;
  using WarpShape = hytlass::gemm::GemmShape<64, 64, 8>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<8, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f16_64x128_64x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::half_t;
  using ElementOutput = hytlass::half_t;
  using ElementAccumulator = hytlass::half_t;
  using ElementCompute = hytlass::half_t;

  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<64, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<64, 64, 8>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<8, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f16_128x128_64x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::half_t;
  using ElementOutput = hytlass::half_t;
  using ElementAccumulator = hytlass::half_t;
  using ElementCompute = hytlass::half_t;

  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<128, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<64, 64, 8>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<8, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f16_128x256_64x128x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::half_t;
  using ElementOutput = hytlass::half_t;
  using ElementAccumulator = hytlass::half_t;
  using ElementCompute = hytlass::half_t;

  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<128, 256, 8>;
  using WarpShape = hytlass::gemm::GemmShape<64, 128, 8>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<8, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f16_256x128_128x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using Element = hytlass::half_t;
  using ElementOutput = hytlass::half_t;
  using ElementAccumulator = hytlass::half_t;
  using ElementCompute = hytlass::half_t;

  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<256, 128, 8>;
  using WarpShape = hytlass::gemm::GemmShape<128, 64, 8>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using ElementOutput = Element;
  using ElementAccumulator = Element;
  using ElementCompute = Element;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    Element,
    LayoutA,
    Element,
    LayoutB,
    Element,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<8, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Real-valued Integer  tests
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i32_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;
  
  using Shape = hytlass::gemm::GemmShape<32, 64, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
      hytlass::MatrixShape<4, 16>,
      hytlass::layout::RowMajorInterleaved<2>,
      hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i32_32x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i32_64x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<64, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i32_128x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i32_128x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 64, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Real-valued Integer - single-precision float output
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_i32_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = float;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 64, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_i32_32x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = float;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_i32_64x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = float;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<64, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_i32_128x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = float;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_f32_i32_128x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = float;
  using ElementAccumulator = int;
  using ElementCompute = float;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 64, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Real-valued Integer tests - mixed-precision with clamping
//
/////////////////////////////////////////////////////////////////////////////////////////////////

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i8_i32_32x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int8_t;
  using ElementAccumulator = int;
  using ElementCompute = int;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 64, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i8_i32_32x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int8_t;
  using ElementAccumulator = int;
  using ElementCompute = int;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<32, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i8_i32_64x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int8_t;
  using ElementAccumulator = int;
  using ElementCompute = int;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<64, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i8_i32_128x128_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int8_t;
  using ElementAccumulator = int;
  using ElementCompute = int;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 128, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

TEST(GFX906_Epilogue_threadblock_epilogue, simt_i8_i32_128x64_32x64x8) {

  //
  // Define the warp-level matrix multiply
  //

  using ElementA = int8_t;
  using ElementB = int8_t;
  using ElementC = int;
  using ElementOutput = int8_t;
  using ElementAccumulator = int;
  using ElementCompute = int;

  int const kElementsPerAccess = 1;

  using Shape = hytlass::gemm::GemmShape<128, 64, 32>;
  using WarpShape = hytlass::gemm::GemmShape<32, 64, 32>;
  using ElementC = ElementAccumulator;
  using LayoutA = hytlass::layout::ColumnMajor;
  using LayoutB = hytlass::layout::RowMajor;
  using LayoutC = hytlass::layout::RowMajor;

  using WarpMmaSimt = hytlass::gemm::warp::MmaSimt<
    WarpShape,
    ElementA,
    LayoutA,
    ElementB,
    LayoutB,
    ElementC,
    LayoutC,
    hytlass::gemm::warp::MmaSimtPolicy<
    hytlass::MatrixShape<4, 16>,
    hytlass::layout::RowMajorInterleaved<2>,
    hytlass::gemm::GemmShape<4, 4, 1>
    >
  >;

  //
  // Output operator
  //

  using OutputOp = hytlass::epilogue::thread::LinearCombination<
    ElementOutput,
    kElementsPerAccess,
    ElementAccumulator,
    ElementCompute
  >;

  //
  // Define the epilogue
  //

  using Epilogue = typename hytlass::epilogue::threadblock::DefaultEpilogueSimt<
    Shape,
    WarpMmaSimt,
    OutputOp,
    kElementsPerAccess
  >::Epilogue;

  //
  // Instantiate epilogue
  //

  EpilogueTestbed<Epilogue> testbed;

  bool passed = testbed.run_all();

  EXPECT_TRUE(passed);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
