#!/bin/bash
#set -x

. ./tools/utils.sh
. ./tools/pkg_check.sh
. ./tools/dcuopn_check.sh
. ./tools/sys_info.sh
. ./tools/sys_log.sh
. ./tools/pcie_check.sh
. ./tools/kernel_check.sh
. ./tools/sme_check.sh
. ./tools/log_analyze.sh


# 默认参数配置
DEFAULT_OUTPUT_DIR="system_info_$(date +%Y%m%d_%H%M%S)"
DEFAULT_KEYWORD="hydcu"
DEFAULT_LOG_AGE=24  # 小时
DEFAULT_LOG_SIZE_LIMIT=10  # 单位：MB
QUIET_MODE=0
DEBUG_MODE=0

DEVICE_NAME=""
DEVICE_ID=""


# 显示帮助信息
show_help() {
    echo "Usage: $0 [OPTIONS]"
    echo "系统诊断脚本 - 收集系统信息并检测驱动问题"
    echo
    echo "选项："
    echo "  -o DIR      指定输出目录 (默认: 自动生成)"
    echo "  -k KEYWORD  设置检测关键字 (默认: $DEFAULT_KEYWORD)"
    echo "  -t HOURS    收集日志的时间范围(小时) (默认: 24)"
    echo "  -s SIZE     日志文件大小限制(MB) (默认: 10)"
    echo "  -q          静默模式(仅输出错误)"
    echo "  -d          调试模式"
    echo "  -h          显示此帮助信息"
    echo
    echo "示例："
    echo "  $0 -o /tmp/logs -k buserr -t 48"
}

# 解析参数
while getopts "o:k:t:s:qdh" opt; do
    case $opt in
        o) CUSTOM_OUTPUT_DIR="$OPTARG" ;;
        k) KEYWORD="$OPTARG" ;;
        t) LOG_AGE="$OPTARG" ;;
        s) LOG_SIZE_LIMIT="$OPTARG" ;;
        q) QUIET_MODE=1 ;;
        d) DEBUG_MODE=1; set -x ;;
        h) show_help; exit 0 ;;
        \?) echo "无效选项: -$OPTARG" >&2; exit 1 ;;
        :) echo "选项 -$OPTARG 需要参数" >&2; exit 1 ;;
    esac
done

# 设置默认值
: ${OUTPUT_DIR:=${CUSTOM_OUTPUT_DIR:-$DEFAULT_OUTPUT_DIR}}
: ${KEYWORD:=$DEFAULT_KEYWORD}
: ${LOG_AGE:=$DEFAULT_LOG_AGE}
: ${LOG_SIZE_LIMIT:=$DEFAULT_LOG_SIZE_LIMIT}


# 初始化目录
mkdir -p "$OUTPUT_DIR" || exit 1


# 主流程
main() {

    hline
    echo -e '################ 日志收集 ##################'
	hline
    get_dcu
    pkg_check
    collect_system_info
    collect_logs
    #analyze_errors
	get_pcie_info
    
    echo -e '\n###### 日志分析 #######'
    hline
    head_normal "分析pcie信息"
    pcie_check $OUTPUT_DIR/pcie_info.log
    head_normal "分析sme信息"
    sme_check $OUTPUT_DIR/dmesg.log
    head_normal "分析驱动安装位置"
    kernel_check
    
	./tools/driver_load_check.sh > $OUTPUT_DIR/driver_status.log
    ./tools/board_check.sh > $OUTPUT_DIR/board_check.log
	product_name=`dmidecode -s system-product-name`
	if [ "$product_name" != "X785-H30" ]; then
    	./tools/pcie_speed_check.sh > $OUTPUT_DIR/pcie_speek_check.log
    fi
	local status=$?
    
    打包结果
    log "打包诊断数据..."
    tar -czf "${OUTPUT_DIR}.tar.gz" "$OUTPUT_DIR" 2>/dev/null
    rm -rf "$OUTPUT_DIR"
    
    log "诊断文件已保存为：${OUTPUT_DIR}.tar.gz"
    return $status
}

# 执行主程序
main
exit $?

